/****************************************************************************

 spell.h
 Copyright 1992 by GO Corporation, All Rights Reserved

 Spelling Checking

 $Revision:   1.6  $
   $Author:   ghullend  $
     $Date:   28 Feb 1992 14:12:18  $


 See Also proof.h, pdict.h
*/
#ifndef SPELL_INCLUDED
#define SPELL_INCLUDED

/*
	Debug Flags:
	
		/DS0001	Low-level debug messages; LOTS of output
		/DS0002	mid-level debug messages
		/DS0004	high-level debugs - general information
		/DS8000 disable dictionary
*/

#ifndef GO_INCLUDED
#include <go.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				   Common Definitions										   *	
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
	maxSpellList is the most bytes a list of spelling corrections can use.
	maxSpellXlateChoices is the dictionary alphabet size
*/

#define maxSpellList			128
#define maxSpellXlateChoices	30	

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				   Common typedefs										   *	
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef struct SPELL_LIST {
	U16	count;				// Number of strings in the list
	CHAR	words[maxSpellList];			// List of concatenated strings
} SPELL_LIST, * P_SPELL_LIST;

typedef struct SPELL_XLATE {
	U16			index;		// Offset within bank
	U8				bits;			// Nibble and bank indicator
	U8				character;	// Out: Character at that location
} SPELL_XLATE, *P_SPELL_XLATE;

typedef struct SPELL_DICT_LIST {
	P_CHAR		pName;		// name of dictionary (e.g. English)
	P_CHAR		pPath;		// path to dictionary (e.g. \\boot\dicts\webf77k)
	U16			bankCount;	// Number of 16K banks the lex is divided into
	P_UNKNOWN	pLangHeader;// Pointer to language specific info
} SPELL_DICT_LIST, *P_SPELL_DICT_LIST;

/* Definitions of different types of word capitalization */

Enum16(SPELL_CASE) {
	spellCommonCase,	// all letters are in lower case
	spellProperCase,	// The First Letter Of Each Word Is Capitalized
	spellUpperCase,	// ALL LETTERS ARE CAPITALIZED
	spellSpecialCase,	// tHere IS a StRANge Mix of cAPitALizATion
};

/*
	This structure controls the rules for converting words from upper to
	lower case.  The defaults work for converting text starting with the
	beginning of a sentence.  If you are starting in the middle of a sentence,
	set firstWord to false.  If the text is not in a sentence, set sentence to
	false.  If you want all the words to have an initial capital letter, set
	minCase to spellProperCase.
*/

typedef struct SPELL_CASE_CONTEXT {
	SPELL_CASE	minCase;	  	// lowest case allowed for output dictionary words
	SPELL_CASE	unkCase;		// case for non-dictionary words
	BOOLEAN		sentence;		// do end-of-sentence processing
	BOOLEAN		dictionary;		// use the dictionary for capitalization info
	BOOLEAN 	allCapsWriter;	// user writes all caps only

	BOOLEAN		firstWord;	// In/Out: This word is first in a sentence
} SPELL_CASE_CONTEXT, * P_SPELL_CASE_CONTEXT;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				   Functions											   *	
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 SpellDictSelect	returns STATUS
	Sets the active dictionary to the language specified.

	dictCode is an index into spellDictList; -1 means deselect.  Currently, only
	American English can be selected, and its code is 0.

*/

STATUS EXPORTED SpellDictSelect(
	S16	dictCode
);

/****************************************************************************
 SpellSetOptionsX	returns void
	Turns the dictionary on or off.

	Pass it true to turn the dictionary on, false to turn it off.

*/

void EXPORTED SpellSetOptionsX(BOOLEAN mode);

/****************************************************************************
 SpellGetOptionsX	returns BOOLEAN
	Returns current dictionary status.
	
	True means spelling is on; false means it's off.
*/

BOOLEAN EXPORTED SpellGetOptionsX(void);


/****************************************************************************
 SpellCheck	returns BOOLEAN
	Checks if a word is in the dictionary or not.

	Argument may contain punctuation but should not contain spaces.  This
	is designed so higher-level software can parse a line of text into
	space-delimited tokens and pass those tokens (with no further
	processing) to this routine.
*/

BOOLEAN EXPORTED SpellCheck(P_CHAR pWord);

/****************************************************************************
 SpellCorrect	returns STATUS
	Finds all the corrections for a word and adds them to a SPELL_LIST
	structure.

	This also takes a space-delimited token, as described above, strips
	off the punctuation, and puts it back on the correction candidates.
	Note that the count field in the SPELL_LIST structure must be
	initialized to zero, unless you are deliberately adding to an
	already-existing list.  This routine avoids adding duplicates to the
	list if it already had some words in it.

*/

STATUS EXPORTED SpellCorrect(
	P_CHAR pWord,				// Word to be corrected
	P_SPELL_LIST pSpellList,	// Out: List to add the word to
	BOOLEAN phonetic				// Perform phonetic correction?
);

/****************************************************************************
 SpellCorrectWord	returns U16
	Finds the first correction for a word.  Returns 0 if none found, else 1.

	The word is a space-delimited token, as described above.  In this
	case, "first" means "first in alphabetical order," this routine is
	not suitable for most applications.
*/

U16 EXPORTED SpellCorrectWord(
	P_CHAR pWord,	 		// Word to be corrected
	P_CHAR pCorrectWord	// Out: place to put the correction
);

/****************************************************************************
 SpellAddToDict	returns STATUS
	Add a word to thePersonalDictionary.

 The prefered way to add words to the current personal dictionary.  As usual, 
 it takes space-delimited tokens and strips off extraneous punctuation.
*/

STATUS EXPORTED SpellAddToDict(
	P_CHAR pWord
);


/****************************************************************************
 SpellAddToAnyDict	returns STATUS
	Add a word to any one of the personal dictionaries.

 The prefered way to add words to a personal dictionary other than the
 current one. It takes a pdict object (clsPDict) that specifies the
 personal dictionary to add to, and space-delimited tokens. It strips 
 off extraneous punctuation.
*/

STATUS EXPORTED SpellAddToAnyDict(
	OBJECT pDict,
	P_CHAR pWord
);


/****************************************************************************
 SpellWordSetCase	returns STATUS
	Convert all-upper-case input into a reasonable mix of upper and lower
	case using dictionary information and other lexical clues.

	Call SpellWordSetCase the first time with pWord == pNull to
	initialize the context structure.  Then pass it the words to be
	converted (in order) with the same context structure each time.  It
	converts each word in place.  To modify the default behavior, change
	the appropriate context parameters (see the definition of the
	SPELL_CASE_CONTEXT structure).

	Defaults
		minCase:		SpellCommonCase
		unkCase:		SpellCommonCase
		sentence:		true
		dictionary:		true
		allCapsWriter:	false
		firstWord:		true
*/

STATUS EXPORTED SpellWordSetCase(
	P_CHAR	pWord,
	P_SPELL_CASE_CONTEXT pSpellCaseContext
);

/****************************************************************************
 SpellLineSetCase	returns STATUS
	Convert all-upper-case input into a reasonable mix of upper and lower
	case using dictionary information and other lexical clues.

	Identical to SpellWordSetCase, except it expects the input to be
	a line of text, which it splits into tokens as required.
*/

STATUS EXPORTED SpellLineSetCase(
	P_CHAR	pLine,
	P_SPELL_CASE_CONTEXT pSpellCaseContext
);

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				   Miscellaneous											   *	
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Address of the list of legal dictionaries */

extern const SPELL_DICT_LIST spellDictList[];

#endif
