/****************************************************************************
 File: pdict.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.6  $
   $Author:   ghullend  $
     $Date:   28 Feb 1992 13:50:34  $

 This file contains the Personal Dictionary Class API.
 This class contains methods that maintain an ordered ASCII list of
 words and can produce a compressed list (called the template), which
 is specially organized for use with handwriting translation software.

 clsPDict inherits from clsObject.
 thePersonalDictionary is a well known instance of clsPDict.

 The word list maintained by thePersonalDictionary is used by default 
 whenever spelling-assisted handwriting translation is performed.

 See Also spell.h
****************************************************************************/
#ifndef PDICT_INCLUDED
#define PDICT_INCLUDED

#ifndef FS_INCLUDED
#include <uuid.h>
#include <fs.h>
#endif

#ifndef INSTLMGR_INCLUDED
#include <instlmgr.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				   Common typedefs										   *	
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****  Personal Dictionary Metrics  ****/

/*
 This structure is used in conjunction with msgPDictGetMetrics to get
 two very important parameters of a personal dictionary: the number of words
 in it and a pointer to the compressed template.  The word count is useful
 for a variety of things, but the compressed template is valuable because
 it can be used directly in the pTemplate field of a translator object
 (see xlate.h)
*/

typedef struct PDICT_METRICS {
	U16			wordCount;  // number of words in the personal dictionary (RO)
	P_UNKNOWN	pXTemplate;	// pointer to compressed template
} PDICT_METRICS, * P_PDICT_METRICS;

/****  Personal Dictionary New Structs  ****/

typedef struct PDICT_NEW_ONLY {
	IM_HANDLE			handle;		// if objNull then use current pdict.
	U32	spare;
} PDICT_NEW_ONLY, * P_PDICT_NEW_ONLY;

typedef struct PDICT_NEW {
	OBJECT_NEW_ONLY		object;
	PDICT_NEW_ONLY		pdict;
} PDICT_NEW, * P_PDICT_NEW;

/****  Miscellaneous  ****/
/*
 This structure is used for converting a word index into a word and
 vice versa.  (That is, for example, to get word #5 from the PDict or to
 find out which word number in the PDict "PenPoint" is.)
*/

typedef struct PDICT_NUM_WORD {
	U16			number;
	P_CHAR		pWord;
} PDICT_NUM_WORD, * P_PDICT_NUM_WORD;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				   Messages												   *	
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgPDictGetMetrics takes P_PDICT_METRICS, returns STATUS
	Gets a copy of the personal dictionary metrics structure.

	This is mainly useful to find out how many words are in the
	personal dictionary.
*/

#define msgPDictGetMetrics						MakeMsg(clsPDict,1)

/****************************************************************************
 msgPDictEnumerateWords takes PP_CHAR, returns STATUS
	Fills a list of pointers to strings with pointers to all the words in
	the personal dictionary.

	The pArgs must be the address of the base of an array of pointers to
	be filled in.  This array must have an entry for every word in the
	personal dictionary plus one for the final null (get the metrics to
	find out how many words are in the PDict.  The words will be in ASCII
	collating sequence, and because the pointers all point to an internal
	data structure, no memory is allocated.  N.B. you must treat this
	structure as strictly read-only!
*/

#define msgPDictEnumerateWords				MakeMsg(clsPDict,2)

/****************************************************************************
 msgPDictAddWord takes P_PDICT_NUM_WORD, returns STATUS
	Adds a word to the personal dictionary.
		

	The routine SpellAddWord(), defined in spell.h, is a better way for
	most clients to add words to the Personal Dictionary, since it has a
	simpler API, strips excess punctuation, checks for duplicates, etc.

	msgPDictAddWord adds the string from the PDICT_NUM_WORD structure,
	computes the zero-based offset of the new word in the personal
	dictionary, and passes back that offset in the number component of
	the PDICT_NUM_WORD structure.

	Although the ASCII representation of the Personal Dictionary is
	updated immediately, the compressed template is not rebuilt until the
	next time msgPDictUpdateTemplate is called.  Handwriting Translation
	does this automatically when it needs the template, but spelling does
	not.

*/

#define msgPDictAddWord							MakeMsg(clsPDict,3)

/****************************************************************************
 msgPDictDeleteWord takes P_PDICT_NUM_WORD, returns STATUS
	Deletes a word from the personal dictionary.

	The reverse of msgPDictAddWord, this message removes the word from
	the personal dictionary and passes back the zero-based offset of the
	location where it formerly was.

	Like msgPDictAddWord, this only affects the ASCII representation of
	the Personal Dictionary.  The next handwriting translation operation
	will rebuild the template, but if you need it built before that (for
	example, to change the behavior of spelling), send
	msgPDictUpdateTemplate.
*/

#define msgPDictDeleteWord						MakeMsg(clsPDict,4)

/****************************************************************************
 msgPDictNumToWord takes P_PDICT_NUM_WORD, returns STATUS
	Locates a word in the personal dictionary by index number, passing
	back the word at that offset.

	Words are indexed in ASCII collating sequence from zero.
*/

#define msgPDictNumToWord						MakeMsg(clsPDict,5)

/****************************************************************************
 msgPDictFindWord takes P_CHAR, returns STATUS
	Checks if a word is in the personal dictionary.

	stsOK means it was found; stsFailed means it was not.
		
*/

#define msgPDictFindWord						MakeMsg(clsPDict,6)

/****************************************************************************
 msgPDictDeleteNum takes P_PDICT_NUM_WORD, returns STATUS
	Locates a word in the personal dictionary by index number and deletes
	the word at that offset.

	Words are indexed in ASCII collating sequence from zero. The number
	of the  word to delete is the number field from the PDICT_NUM_WORD
	struct; the actual word deleted is passed back in pWord. (This MUST
	be set to point to something by the caller!  Max size is
	maxWordLength+1.  Setting pWord to Nil(P_CHAR) passes nothing back.)
*/

#define msgPDictDeleteNum						MakeMsg(clsPDict,7)

/****************************************************************************
 msgPDictWordToNum takes P_PDICT_NUM_WORD, returns STATUS
	Given a word, computes its offset within the personal dictionary.

	Words are counted from zero in ASCII collating sequence.
		
*/

#define msgPDictWordToNum						MakeMsg(clsPDict,8)

/****************************************************************************
 msgPDictUpdateTemplate takes PP_UNKNOWN, returns STATUS
	Recomputes the compressed template from the word list and updates the
	pointer.

	When the ASCII form of the personal dictionary is modified, the
	compressed template is not automatically modified.  Since compression
	can be time consuming, this is deferred until it is absolutely
	necessary.  This routine is called by Handwriting Translation at the
	beginning of every translation.
	
	If the current template is not out of date, this just copies old value into
	the argument.
*/

#define msgPDictUpdateTemplate				MakeMsg(clsPDict,9)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				   Miscellaneous												   *	
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
	Base of the template of thePersonalDictionary.  Handwriting
	Translation needs to be able to get at this very quickly, so it's
	stored as an exported global variable to allow it to avoid the
	ObjectCall.
*/

extern P_UNKNOWN PASCAL pPDictBase;

/*
	Quick Help Tag for the Personal Dictionary Application (tool)
*/

#define hlpPDAppBackground	MakeTag(clsPDApp,1)

#endif	// PDICT_INCLUDED
