/****************************************************************************
 File: mfilter.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.23  $
   $Author:   cmeyer  $
     $Date:   18 Mar 1992 16:58:00  $

 This file contains the API for clsModalFilter.

 clsModalFilter inherits from clsObject.
 Modal filters implement window-relative input modality.

 Modal filters are useful for making a window tree behave in a modal fashion:
 the user must interact with the windows in the tree (and make it go away)
 before they can use other windows in the application (or system).

 Here is an example of how to set up a modal filter object:
//{
	MODAL_FILTER_NEW	mfn;

	// Create it.
	ObjCallWarn(msgNewDefaults, clsModalFilter, &mfn);
	mfn.modalFilter.flags = <appropriate flags>;
	mfn.modalFilter.process = OSThisProcess();
	mfn.modalFilter.subTreeRoot = <root of window tree to make modal>;
	ObjCallRet(msgNew, clsModalFilter, &mfn, s);

	// Activate it.
	ObjCallRet(msgModalFilterActivate, mfn.object.uid, pNull, s);

	// Tell input system about it.
	StsRet(InputFilterAdd( \
        mfn.object.uid, inputAllRealEventsFlags, 0, <priority>), s);
//}

 See input.h for a discussion of filter priorities and tips on choosing
 a priority.
****************************************************************************/

/**** Debugging Flags ****/
/*
 The clsModalFilter debugging flag is 'K'.  Defined values are:

    flag10 (0x0400):    general
*/

#ifndef MFILTER_INCLUDED
#define MFILTER_INCLUDED

												#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
												#endif
												#ifndef OSTYPES_INCLUDED
#include <ostypes.h>
												#endif
												#ifndef WIN_INCLUDED
#include <win.h>
												#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Common #defines and typedefs					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT MODAL_FILTER, *P_MODAL_FILTER;

// Flags
#define mfSystemModal		flag0
#define mfAutoDismiss		flag1
#define mfDefaultFlags		mfAutoDismiss

typedef struct MODAL_FILTER_METRICS {
    U16			flags;
	OS_TASK_ID	process; 		// app process for filter; ignored for mfSystemModal
	WIN		  	subTreeRoot;	// window tree to which to give events
	U32			spare;			// reserved
} MODAL_FILTER_METRICS,  *P_MODAL_FILTER_METRICS;


/****************************************************************************
 msgNew			takes P_MODAL_FILTER_NEW, returns STATUS
	category: class message
	Creates a modal filter.

 The fields you commonly set are:
	pArgs->modalFilter.flags:		appropriate flags
	pArgs->modalFilter.process:		process owning the window tree
	pArgs->modalFilter.subTreeRoot:	root of window tree for which to filter

 A filter is active after msgNew, and becomes deactivated only after
 it has dismissed its window.
*/

typedef MODAL_FILTER_METRICS	MODAL_FILTER_NEW_ONLY, *P_MODAL_FILTER_NEW_ONLY;

#define modalFilterNewFields	\
	objectNewFields     		\
	MODAL_FILTER_NEW_ONLY		modalFilter;
        
typedef struct MODAL_FILTER_NEW {
	modalFilterNewFields
} MODAL_FILTER_NEW, *P_MODAL_FILTER_NEW;


/****************************************************************************
 msgNewDefaults			takes P_MODAL_FILTER_NEW, returns STATUS
	category: class message
	Initializes the MODAL_FILTER_NEW structure to default values.

 Zeroes out pArgs->modalFilter and sets:
//{
	pArgs->modalFilter.flags = mfDefaultFlags;
//}
*/


/****************************************************************************
 msgModalFilterGetFlags				takes P_U16, returns STATUS
	Passes back the receiver's flags.
*/
#define msgModalFilterGetFlags		MakeMsg(clsModalFilter, 1)


/****************************************************************************
 msgModalFilterSetFlags				takes P_U16, returns STATUS
	Sets the receiver's flags.
*/
#define msgModalFilterSetFlags		MakeMsg(clsModalFilter, 2)


/****************************************************************************
 msgModalFilterActivate				takes nothing, returns STATUS
	Activates the filter.

 A filter is active after msgNew, and becomes deactivated only after
 it has dismissed its window.
*/
#define msgModalFilterActivate		MakeMsg(clsModalFilter, 3)


/****************************************************************************
 msgModalFilterDeactivate			takes nothing, returns STATUS
	Deactivates the filter.

 A filter is active after msgNew, and becomes deactivated only after
 it has dismissed its window.
*/
#define msgModalFilterDeactivate	MakeMsg(clsModalFilter, 4)


/****************************************************************************
 msgModalFilterDismissWin			takes nothing, returns STATUS
	category: third-party notification
	Sent to the subTreeRoot if the win should be dismissed.
*/
#define msgModalFilterDismissWin	MakeMsg(clsModalFilter, 5)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from Other Classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgInputEvent 					takes P_INPUT_EVENT, returns STATUS
	Notification of an input event. 

 If the filter is inactive, or the input event's devCode is not of clsPen,
 or the evfGrabTracker flag is set in pArgs->flags, or there's a grabber
 object present (InputGetGrab), then the filter just returns stsInputContinue.

 Next, if the pArgs->destination is not a valid object, the filter returns
 stsInputTerminate.

 If, at this point, the mfSystemModal flag is clear and the process of
 the pArgs->destination doesn't match MODAL_FILTER_METRICS.process, the
 filter does the following:
//{
	if mfAutoDismiss is on
		if the pArgs->devCode is msgPenDown
			self-send msgModalFilterDeactivate
			send msgModalFilterDismissWin to MODAL_FILTER_METRICS.subTreeRoot
				(and if that returns an error status, top and flash subTreeRoot)
			return stsInputTerminate.
		otherwise return stsInputContinue.
	otherwise return stsInputContinue.
//}
 Now, if pArgs->destination is within subTreeRoot, return stsInputSkipTo4.
 (See input.h)

 Next, if the subTreeRoot is not a valid object, return stsFailed.

 Next, if mfAutoDismiss is on and pArgs->devCode is msgPenDown:
//{
	self-send msgModalFilterDeactivate
	send msgModalFilterDismissWin to MODAL_FILTER_METRICS.subTreeRoot
		(and if that returns an error status, top and flash subTreeRoot)
	return stsInputTerminate.
//}
 Finally, if pArgs->devCode is msgPenDown, the filter tops the subTreeRoot,
 flashes it, and returns stsInputTerminate.

 Return Value
	stsInputContinue
	stsInputTerminate
	stsInputSkipTo4
	stsFailed

 See Also
	msgWinInsert:	used by a filter to top the subTreeRoot
	msgBorderFlash:	used by a filter to flash the subTreeRoot
*/


#endif
