/****************************************************************************
 link.h

 Copyright 1992, Go Corporation, All rights reserved

 Link layer API definition.

 $Revision:   1.7  $
   $Author:   bldmstr  $
     $Date:   20 Feb 1992 02:39:18  $

 This file contains the interface definition for link layer protocols.

 1. Link layer protocols must sub-class clsLink.

 2. clsLink sub-classes clsService.

****************************************************************************/
#ifndef LINK_INCLUDED
#define LINK_INCLUDED


typedef struct
{
	U16						addrSize;   // size of address pointed to
	U8							addr[8];		// address 
} ADDRESS, *P_ADDRESS;


/*
The PROTOCOL_ADDRESS structure contains all the addressing information 
needed below the transport level.  Unspecified addresses have null pointers.
*/
typedef struct {
	ADDRESS					linkAddr;		/* link address */
} PROTOCOL_ADDRESS, *   P_PROTOCOL_ADDRESS;


/*
The PROTOCOL_INFO structures in the transmit and receive descriptors holds the following information.
*/

typedef struct {
	U16						linkType;			/* Type of link packet */
	PROTOCOL_ADDRESS		src;
	PROTOCOL_ADDRESS		dest;
} PROTOCOL_INFO;

#define sizeRxBuf 608

typedef struct RXBUFDESC {
	struct RXBUFDESC		*link;  	        /* link to next buffer descriptor */
	U16						rxBlockLen;		/* Length of  remaining data in frame */
	U8							*pRxBlock;		/* Pointer to remaining data in frame */
	U16						packetSize;		/* Original packet size */
	PROTOCOL_INFO			info;
	U32						physAddr;       /* physical address of immedData */
	U8							immedData[sizeRxBuf];	 /* Contains raw packet  */
} RX_DESC, *P_RX_DESC;


typedef struct {
	U16						blockLen;
	U8							*pBlock;
} BLOCK, *P_BLOCK;


#define lnkMaxBlocks 8
#define sizeTxImmedData 32

typedef struct {
	PROTOCOL_INFO			info;
	U16						txBlocks;			/* Count of data blocks, max is 8 */
	BLOCK						txBlockTab[lnkMaxBlocks];
	U16						usedImmedData;	/* Amount of ImmedData thats been used */
	U8							immedData[sizeTxImmedData];
} TX_DESC, *P_TX_DESC;
        
#define stsNoTxBuffer				MakeStatus(clsLink, 1)
#define stsNoRxBuffer				MakeStatus(clsLink, 2)
#define stsTxCollisionOrDefer  	    MakeStatus(clsLink, 3)
#define stsTxTimeout  	            MakeStatus(clsLink, 4)

// A power cycle has happened, the link should be closed and reinitialized
#define stsLinkPowerCycle         MakeStatus(clsLink, 5)

// The link cable is not connected.
#define stsLinkNotConnected       MakeStatus(clsLink, 6)


typedef U16			LINK_PROTOCOL_TYPE;

typedef enum
{
    linkMulticast   = flag0,        // multicast transmit and receive
    linkBroadcast   = flag1,		// broadcast transmit and receive 
    linkPromiscuous = flag2,	    // promiscuous receive mode 
    linkLoopback    = flag3		    // loopback of transmit to receive
} LINK_SERVICES;


typedef struct
{
    U16         tableSize;
    U8          linkAddress[2];
}  *P_BROADCAST_ADDR,  *P_MULTICAST_ADDR;


typedef struct
{
    U16                 tableSize; 	    // size of link Attributes table
    U8                  typeName[32];	// ASCIIZ name of LINK type: LocalTalk, Ethernet
				                        // ASYNC, SDLC, etc.
    U16                 linkAddrLen;	// length in bytes of link addresses
    U8                  linkAddr[16];	// current link address of local station
    U32                 linkSpeed;		// link communication speed in bits per second
    U16                 maxDataSize;	// maximum amount of data that will fit in a link frame
    U16                 maxFrameSize;   // maximum size of a link frame (including link header) 
    U16                 numBuffers;		// total number of available link buffers for this device 
    LINK_SERVICES       linkServices;	// LINK services supported
    ADDRESS             broadcast;	    // broadcast address
    P_MULTICAST_ADDR    pMulticastTable;    // pointer to multicast address table
    /* add additional fields here */
} LINK_ATTRIBUTES,  *P_LINK_ATTRIBUTES;


typedef enum
{
    linkOperational,
    linkHardwareFailure,
    linkConfigurationFailure,
    linkHardwareNotInstalled
} LINK_OPERATING_STATUS;


typedef struct
{
    U16                     tableSize;		/* length of table */
    LINK_OPERATING_STATUS   linkStatus;	
    U16                     freeBuffers;	/* current number of free rx buffers */
    U32                     framesRx;		/* number of good frames rx */
    U32                     bytesRx;		/* total number of bytes rx */
    U32                     crcErrors;		/* number of frames rx with CRC errors */
    U32                     noBufferRx;		/* number of rx frames missed because no rx buffer available */
    U32                     bufferOverflowRx;   /* rx frame too large to fit in receive buffer */
    U32                     frameErrorRx;		/* total rx frame errors */
    U32                     MulticastFramesReceived;    /* number of multicast frames received */
    U32                     broadcastFramesReceived;    /* number of broadcast frames received */
    U32                     framesTx;		/* number of frames tx */
    U32                     bytesTx;		/* total number of bytes tx */
    U32                     txTimeout;		/* tx frame timeout errors */
    U32                     txHWError;		/* tx frame HW error */
/* additional specific status info goes here */
} LINK_STATUS,  *P_LINK_STATUS;


typedef void (EXPORTED * PF_PROTOCOL_HANDLER)(P_RX_DESC);


/****************************************************************************
structure of the link header 
*/
#pragma pack(1)     // byte boundary packing for protocol headers
typedef struct 
{
    U8          destLinkAddr;
    U8          srcLinkAddr;
    U8          typeLink;
} LINK_HEADER,  *P_LINK_HEADER;
#pragma pack()	 // back to command line stuff

#define maxRxFrameSize sizeRxBuf


typedef struct TX_FRAME
{
    struct TX_FRAME  *   link;
    BOOLEAN                 sent;
    U16                     length;
    U32                     physAddr;
    unsigned char           buf[maxRxFrameSize];
} TX_FRAME,  *P_TX_FRAME;


#define lnkMaxShortFrameSize 10


typedef struct SHORT_TX_FRAME
{
struct SHORT_TX_FRAME  * link;
    BOOLEAN                 sent;
    U16                     length;
    U32                     physAddr;
    unsigned char           buf[lnkMaxShortFrameSize];
} SHORT_TX_FRAME,  *P_SHORT_TX_FRAME;




/****************************************************************************
 msgLINKInstallProtocol		takes P_INSTALL_PROTOCOL, returns STATUS

 Install a link layer protocol handler to receive frames.
*/
#define  msgLINKInstallProtocol  MakeMsg( clsLink, 1 )

typedef struct INSTALL_PROTOCOL{
	LINK_PROTOCOL_TYPE		linkProtocolType;
	PF_PROTOCOL_HANDLER		pNewHandler;
} INSTALL_PROTOCOL, * P_INSTALL_PROTOCOL;


/****************************************************************************
 msgLINKRemoveProtocol		takes P_REMOVE_PROTOCOL, returns STATUS

 Remove a link layer protocol handler.
*/
#define  msgLINKRemoveProtocol  MakeMsg( clsLink, 2 )

typedef struct REMOVE_PROTOCOL{
	LINK_PROTOCOL_TYPE		linkProtocolType;
} REMOVE_PROTOCOL, * P_REMOVE_PROTOCOL;


/****************************************************************************
 msgLINKTransmit		takes P_LINK_TRANSMIT, returns STATUS

 Transmit a packet.
*/
#define  msgLINKTransmit  MakeMsg( clsLink, 5 )

typedef struct LINK_TRANSMIT {
	P_TX_DESC					pTD;
} LINK_TRANSMIT, * P_LINK_TRANSMIT;


/****************************************************************************
 msgLINKBufferReturn		takes P_BUFFER_RETURN, returns STATUS

 Return receive buffer to the link layer.
*/
#define  msgLINKBufferReturn  MakeMsg( clsLink, 6 )

typedef struct BUFFER_RETURN {
	P_RX_DESC					pRD;
} BUFFER_RETURN, * P_BUFFER_RETURN;


/****************************************************************************
 msgLINKAttributesGet		takes P_ATTRIBUTES_GET, returns STATUS

 Obtain the link layer attributes.
*/
#define  msgLINKAttributesGet  MakeMsg( clsLink, 7 )

typedef struct ATTRIBUTES_GET {
	P_LINK_ATTRIBUTES			pAttributes;
} ATTRIBUTES_GET, * P_ATTRIBUTES_GET;


/****************************************************************************
 msgLINKStatusGet		takes P_STATUS_GET, returns STATUS

 Obtain the link layer statistics.
*/
#define  msgLINKStatusGet  MakeMsg( clsLink, 8 )

typedef struct STATUS_GET {
	P_LINK_STATUS				pStatus;
} STATUS_GET, * P_STATUS_GET;


/****************************************************************************
 msgLINKAddressAcquire		takes P_ADDRESS_ACQUIRE, returns STATUS

 Acquire the link layer address.
*/
#define  msgLINKAddressAcquire  MakeMsg( clsLink, 9 )

typedef struct ADDRESS_ACQUIRE {
	U16							linkAddrLen;	// length in bytes of link addresses
	U8								linkAddr[16];	// current link address of local station
	BOOLEAN						server;			// acquire a server address
} ADDRESS_ACQUIRE, * P_ADDRESS_ACQUIRE;



#endif
