/****************************************************************************
 File: iobxsvc.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.7  $
	 $Date:   30 Mar 1992 10:59:42  $

 This file contains the API definition for clsIOBXService.
 clsIOBXService inherits from clsService.

****************************************************************************/

#ifndef IOBXSVC_INCLUDED
#define IOBXSVC_INCLUDED

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif
#ifndef GO_INCLUDED
#include <go.h>
#endif
#ifndef SERVICE_INCLUDED
#include <service.h>
#endif
#ifndef AUXNBMGR_INCLUDED
#include <auxnbmgr.h>
#endif

/****  1. Introduction  ****/

/*
 clsIOBXService implements most of the behavior of its two subclasses:
 clsOBXService (Outbox service class) and clsINBXService (Inbox
 service class).  While its subclasses deal with either the system
 Inbox or the system Outbox, clsIOBXService allows a service to access
 both the Inbox and the Outbox at the same time.  For details about
 the two subclasses of clsIOBXService, see inbxsvc.h and obxsvc.h.
*/

/****  2. Choosing the Appropriate Superclass for Your Service  ****/

/*
 An Outbox service is assigned a section in the system Outbox.	Thus,
 if a service's primary function is to send data out of a PenPoint
 computer, it should probably be a subclass of clsOBXService.  A good
 example for this type of services is a printer device driver.	A very
 important behavior for an Outbox service is to hold the output data
 until the physical device is available.  This "deferred output"
 feature allows any documents in an Outbox section to be sent only
 when the conditions are right for the output operation to commence.
 This is implemented as a series of messages associated with
 msgIOBXSvcPollDocumens, which basically "polls" the Outbox section
 looking for documents to be sent out.	By default, all Outbox
 services inherit such auto polling behavior.  (See the IOBXSVC_NEW
 structure defined in this API for inhibiting this behavior.)

 Similary, an Inbox service is associated with a section in the system
 Inbox and concerns itself with transfering data into a PenPoint
 computer.	For example, the device driver for an optical scanner
 should probably be a subclass of clsINBXService.  However, the notion
 of "deferred input" may not apply to most types of Inbox services.
 Therefore an Inbox service by default does not "poll" the documents
 in its Inbox section.	When "deferred input" does make sense, as in
 the case of a stock quote service periodically downloading the latest
 stock prices from a host computer, the auto polling behavior can be
 easily enabled through the newArgs.

 Some services may need to transfer data both into and out of the
 PenPoint computer.	 (E.g., an electronic mail service.)  There are
 several alternatives to deal with this situation.	First, such
 services can still subclass from either clsINBXService or
 clsOBXService and avoid the complexity of dealing with two separte
 sections in the system Inbox and Outbox.  Second, the input and
 output operations can be divided into two services, one inheriting
 from clsINBXService and one inheriting from clsOBXService.	 Third,
 the service can inherit directly from clsIOBXService and deal with
 both an Inbox section and an Outbox section at the same time.	Both
 sections will have the same name as the service itself, and enabling
 one of them will automatically enable the other.
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****  Inbox/Outbox Service Status Codes  ****/

/*
 The inbox/outbox section associated with the service is empty.
 This status is returned by msgIOBXSvcNextDocument.
*/
#define stsIOBXSvcSectionEmpty			MakeStatus(clsIOBXService, 101)

/*
 The outbox section associated with the service is not empty,
 but none of the document is ready for output.
 This status is returned by msgIOBXSvcNextDocument.
*/
#define stsIOBXSvcDocNotReady			MakeStatus(clsIOBXService, 102)

/****  Outbox Service Standard Dialog Codes	 ****/
#define tagOBXSvcDocumentExists			MakeDialogTag(clsOBXService, 0)
#define tagOBXSvcOutputPending			MakeDialogTag(clsOBXService, 1)

/****  Inbox Service Standard Dialog Codes	****/
#define tagINBXSvcDocumentExists		MakeDialogTag(clsINBXService, 0)
#define tagINBXSvcInputPending			MakeDialogTag(clsINBXService, 1)

/****  Filesystem Attributes  ****/

/*
 The state of a document in the inbox/outbox. 
*/
#define iobxsvcAttrDocState			 FSMakeFix32Attr(clsIOBXService, 1)
Enum32(IOBXSVC_ATTR_DOC_STATE) {
	iobxsvcDocNotScheduled		= 0, // Document hasn't been scheduled
									 // Same as no attribute.
	iobxsvcDocOutputInProgress	= 1, // Output started, not finished yet
	iobxsvcDocUserCancelled		= 2, // Cancelled by user
	iobxsvcDocError				= 3, // Unable to finish due to errors
	iobxsvcDocInputInProgress	= 4, // Input started, not finished yet
	iobxsvcDocReserved5			= 5, // Reserved for future expansion
	iobxsvcDocReserved6			= 6, // Reserved for future expansion
	iobxsvcDocReserved7			= 7, // Reserved for future expansion
	iobxsvcDocReserved8			= 8, // Reserved for future expansion
	iobxsvcDocReserved9			= 9, // Reserved for future expansion
	iobxsvcDocReserved10		= 10, // Reserved for future expansion
	iobxsvcDocReserved11		= 11, // Reserved for future expansion
	iobxsvcDocReserved12		= 12, // Reserved for future expansion
	iobxsvcDocReserved13		= 13, // Reserved for future expansion
	iobxsvcDocReserved14		= 14, // Reserved for future expansion
	iobxsvcDocReserved15		= 15 // Reserved for future expansion
};

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Class Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNewDefaults				takes P_IOBXSVC_NEW, returns STATUS
	category: class message
	Initializes the P_IOBXSVC_NEW structure to default values.

	Zeroes out pArgs->iobxsvc.

*/

typedef struct IOBXSVC_SECTION_METRICS {

	BOOLEAN		autoPoll;		// True if svc should poll documents when
								// it's both enabled and connected.
	CLASS		sectionClass;	// Section Class.  Must be clsNBToc or
								// a subclass of it, or objNull for none.
	U32			reserved[2];	// Reserved.

} IOBXSVC_SECTION_METRICS, *P_IOBXSVC_SECTION_METRICS;

typedef struct IOBXSVC_NEW_ONLY {

	IOBXSVC_SECTION_METRICS in;		// Inbox section spec
	IOBXSVC_SECTION_METRICS out;	// Outbox section spec
	U32		reserved[3];

} IOBXSVC_NEW_ONLY, *P_IOBXSVC_NEW_ONLY;

#define ioSvcNewFields		\
	serviceNewFields		\
	IOBXSVC_NEW_ONLY		 iobxsvc;

typedef struct IOBXSVC_NEW {
	ioSvcNewFields
} IOBXSVC_NEW, *P_IOBXSVC_NEW;

/****************************************************************************
 msgNew						takes P_IOBXSVC_NEW, returns STATUS
	category: class message
	Creates a new inbox/outbox service object.

*/

/****************************************************************************
 msgIOBXSvcSwitchIcon	takes nothing, returns STATUS
	category: class message
	Toggles the inbox or outbox icon (to empty or filled) if neccessary.

 Check the content of the inbox or outbox notebook.	 For outbox, show
 the "filled" icon if any document is found.  For inbox, show the
 "filled" icon if there is at least one document that has not been
 opened.
*/
#define msgIOBXSvcSwitchIcon				MakeMsg(clsIOBXService, 1)

/****************************************************************************
 msgIOBXDocGetService	takes P_IOBX_DOC_GET_SERVICE, returns STATUS
	category: class message
	Gets the service name.

 Get the name of the service associated with an inbox/outbox document.
 If the document has not been placed into an inbox/outbox section,
 stsFailed is returned.

 Note that the document must be at the top level within an
 inbox/outbox section.	That is, if the document is embedded in
 another document, stsFailed will be returned even if its embeddor is
 within an inbox/outbox section.
*/ 
#define msgIOBXDocGetService				MakeMsg(clsIOBXService, 2)

typedef struct IOBX_DOC_GET_SERVICE {

	OBJECT			document;				// In: document uid
	CHAR			svcName[nameBufLength]; // Out: service name

} IOBX_DOC_GET_SERVICE, *P_IOBX_DOC_GET_SERVICE;

/****************************************************************************
 msgIOBXDocInIOBox	takes P_IOBX_DOC_IN_IOBOX, returns STATUS
	category: class message
	Checks if a document is in a section in the Inbox/Outbox notebook.

*/
#define msgIOBXDocInIOBox					MakeMsg(clsIOBXService, 3)

typedef struct IOBX_DOC_IN_IOBOX {

	ANM_AUX_NOTEBOOK	notebook;		// In: Which notebook?
	UUID				uuid;			// In: document uuid
	CLASS				svcClass;		// In: service class to check for

} IOBX_DOC_IN_IOBOX, *P_IOBX_DOC_IN_IOBOX;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				Messages Sent to an Outbox Service Instance				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgIOBXSvcMoveInDoc	takes P_IOBXSVC_MOVE_COPY_DOC, returns STATUS
	Moves a document into the outbox section.

 Superclass behavior is to move the document located at pArgs->source
 into the input/output queue associated with the inbox/outbox service.
 For example, set pArgs->sequence to 1 to move the document to the
 top of the queue.	Set it to maxU16 to move the document to the
 bottom of the queue.

 After the document is moved (or copied) to the input/output queue, it
 is considered to be in a state ready for input/output, even though
 the service may not be connected at the time.	Client should not
 alter the document in any way once it has been moved to the
 input/output queue.

 Subclasses can provide their own behavior if they wish.  Remember to
 use the class message msgIOBXSvcSwitchIcon to change the inbox/outbox
 icon.
*/
#define msgIOBXSvcMoveInDoc					MakeMsg(clsIOBXService, 4)

typedef struct IOBXSVC_MOVE_COPY_DOC {

	ANM_AUX_NOTEBOOK	notebook;	// In: Which notebook?
	FS_LOCATOR			source;		// In: Location of source document.
	U16					sequence;	// In: Sequence number to move/copy
									//	   in front of.
} IOBXSVC_MOVE_COPY_DOC, *P_IOBXSVC_MOVE_COPY_DOC;

/****************************************************************************
 msgIOBXSvcCopyInDoc	takes P_IOBXSVC_MOVE_COPY_DOC, returns STATUS
	Copies a document into the Inbox/Outbox section.

 Same as msgIOBXSvcMoveInDoc, except that the document is copied
 to the input/output queue.
*/
#define msgIOBXSvcCopyInDoc					MakeMsg(clsIOBXService, 5)

/****************************************************************************
 msgIOBXSvcGetTempDir	takes P_OBJECT, returns STATUS
	Passes back a handle for a temporary directory.

 This message is provided for clients who may want to prepare their
 input/output document before moving it into the input/output queue.
 The handle of an "official" temporary directory is passed back and it
 can be used as temporary storage for documents, data, etc.	 Clients
 are responsible for deleting temporary files they created when done.
 This temporary directory will be flushed after a warm boot.
*/
#define msgIOBXSvcGetTempDir				MakeMsg(clsIOBXService, 6)

/****************************************************************************
 msgIOBXSvcPollDocuments		takes nothing, returns STATUS
	Polls all documents waiting for input/output.

 This message tells the inbox/outbox service to look through its queue
 and initiate the input/output process for the first document ready to
 do so.	 The service will first make sure that it is enabled and is
 connected to the designated input/output port.	 If these conditions
 are met, it will then self-send msgIOBXSvcNextDocument to locate the
 next document ready for input/output.

 If msgIOBXSvcNextDocument returns stsOK, indicating that a document
 is ready, this message proceeds to self-send msgIOBXSvcLockDocument
 to lock the document, and finally msgIOBXSvcIOStart to initiate the
 input/output process.

 If msgIOBXSvcNextDocument returns stsOBXSvcDocNotReady, indicating
 that the section is not empty but none of the documents are ready
 for input/output, this message self-sends msgIOBXSvcScheduleDocument
 to schedule the document passed back in pArgs at a later time.

 Subclasses normally do not process this message.

 See Also
	msgIOBXSvcNextDocument
	msgIOBXSvcLockDocument
	msgIOBXSvcUnlockDocument
	msgIOBXSvcIOStart
	msgIOBXSvcScheduleDocument
*/
#define msgIOBXSvcPollDocuments				MakeMsg(clsIOBXService, 7)

/****************************************************************************
  msgIOBXSvcNextDocument		takes P_IOBXSVC_DOCUMENT, returns STATUS
	 category: self-sent
	 Passes back the next document ready for input/output.

 Superclass behavior is to start from the top of the queue and locate
 the first document ready for input/output.	 If one is found,
 information about the document is passed back in pArgs.  The same
 pArgs will be passed to messages msgIOBXSvcLockDocument and
 msgIOBXSvcIOStart.	 By default, a document is ready for input/output
 when it is closed.	 If the document is open, it will receive
 msgIOBXDocIOStartOK and it should return stsOK to indicate that it is
 ready for input/output.

 Subclasses can provide their own behavior if they wish.  Return
 stsOBXSvcSectionEmpty to give the superclass an opportunity to
 change the inbox/outbox icon from filled to empty.	 Or refresh the
 look of the icon by sending msgIOBXSvcSwitchIcon to the service
 class.

 Return Value
	stsOK:					A document is ready for input/output.
	stsOBXSvcSectionEmpty:	The input/output queue is empty.
	stsOBXSvcDocNotReady:	No document in the input/output queue is ready.
	Service-Specific Error Returns.

 See Also
	msgIOBXSvcPollDocuments
*/
#define msgIOBXSvcNextDocument				MakeMsg(clsIOBXService, 8)

typedef struct IOBXSVC_DOCUMENT {

	OBJECT			uid;					// uid of the doc
	OBJECT			dir;					// app dir of the doc
	OBJECT			docClass;				// class of the doc
	U16				sequence;				// sequence of the doc
	CHAR			pName[nameBufLength];	// name of this doc
	P_UNKNOWN		pDocData;				// subclass's private data

} IOBXSVC_DOCUMENT, *P_IOBXSVC_DOCUMENT;

/****************************************************************************
  msgIOBXSvcLockDocument		takes P_IOBXSVC_DOCUMENT, returns STATUS
	 category: self-sent
	 Locks the document in preparation for input/output.

 This message is a place holder for subclasses that may require
 additional preparatory work to be performed on a document before it
 is ready for input/output.	 For example, a document may have to be
 "locked" so that it can not be opened during the input/output
 process.  This message may be used for other purposes as well.	 For
 example, an inbox/outbox service may decide to store a light-weight
 "shadow" document (e.g., a report designator for a database
 application) in the input/output queue until it is chosen for
 input/output.	The service then handles this message by converting
 the shadow document to a real one (e.g., the actual report).

 The superclass behavior for this message is to stamp the document
 directory with the filesystem attribute iobxsvcDocIOInProgress.  This
 stamp will prevent any gestures over the document from being
 processed.	 This means that once a document is locked for
 input/output it can not be deleted, renamed, etc. via gestures.

 See Also
	msgIOBXSvcUnlockDocument
*/
#define msgIOBXSvcLockDocument				MakeMsg(clsIOBXService, 9)

/****************************************************************************
  msgIOBXSvcUnlockDocument		takes P_IOBXSVC_DOCUMENT, returns STATUS
	 category: self-sent
	 Unlocks a document that was previously locked.

 This message is a place holder for subclasses that may require
 additional "cleanup" work to be performed on a document before it
 is put back to the input/output queue.

 The superclass behavior for this message is to remove the
 iobxsvcDocIOInProgress stamp on the document directory.

 See Also
	msgIOBXSvcLockDocument
*/
#define msgIOBXSvcUnlockDocument			MakeMsg(clsIOBXService, 10)

/****************************************************************************
  msgIOBXSvcScheduleDocument		takes P_IOBXSVC_DOCUMENT, returns STATUS
	 category: self-sent
	 Schedules a document that is not ready for input/output

 This message is sent when msgIOBXSvcNextDocument locates a document in
 the input/output queue but the document is not ready for input/output.

 Subclasses should provide their own behavior.  The default behavior is
 to release the ownership of the target service (i.e., become disabled),
 with the expectation that the user must manually schedule the document
 later on (by re-enabling the section.)

 See Also
	msgIOBXSvcNextDocument
*/
#define msgIOBXSvcScheduleDocument			MakeMsg(clsIOBXService, 11)

/****************************************************************************
  msgIOBXSvcIOStart		takes P_IOBXSVC_DOCUMENT, returns STATUS
	 category: self-sent
	 Starts the input/output process for a document in the input/output queue.

 Superclass behavior is to activate the inbox/outbox document if it isn't
 already active, and then send msgIOBXDocIOStart to the
 document instance.

 Subclasses can provide their own behavior if they wish.
*/
#define msgIOBXSvcIOStart					MakeMsg(clsIOBXService, 12)

/****************************************************************************
  msgIOBXSvcIOCancel		takes nothing, returns STATUS
	 Cancels the input/output process.

 This message is sent to the service when the caller wishes to
 cancel any input/output operation in progress.	 The service responds
 to this message by sending msgIOBXDocOutuptCancel to an active
 inbox/outbox document.	 After the document is cancelled, the service
 will post an error note to the user if there are other documents
 waiting to be processed.  The user then decides whether or not
 the service should proceed to send the remaining documents.

 Subclasses do not normally process this message.
*/
#define msgIOBXSvcIOCancel					MakeMsg(clsIOBXService, 13)

/****************************************************************************
  msgIOBXSvcIOCleanUp		takes P_IOBX_DOC_OUTPUT_DONE, returns STATUS
	 category: self-post.
	 Cleans up after the current input/output is done.

 This message is posted to self as a result of the service receiving
 msgIOBXDocIODone, which is sent by the inbox/outbox document when it
 finishes the input/output operation.  The inbox/outbox document will
 be either deleted or marked as specified in pArgs, and when
 everything is properly cleaned up the service will post
 msgIOBXSvcPollDocuments to self to see if anything else is waiting
 for input/output.

 Subclasses do not normally process this message.

 See Also
	msgIOBXDocIODone
*/
#define msgIOBXSvcIOCleanUp					MakeMsg(clsIOBXService, 14)

Enum32(IOBX_DOC_EXIT_BEHAVIOR) {	// What to do after a document 
									// is processed

	iobxDocExitDoNothing		= 0,
	iobxDocExitDelete			= 1,
	iobxDocExitMarkAsFailed		= 2,
	iobxDocExitMarkAsCancelled	= 3

};

typedef struct IOBX_DOC_OUTPUT_DONE {

	IOBX_DOC_EXIT_BEHAVIOR	behavior;	// exit behavior
	P_UNKNOWN				pDocData;	// Unused: document specific data

} IOBX_DOC_OUTPUT_DONE, *P_IOBX_DOC_OUTPUT_DONE;

/****************************************************************************
 msgIOBXSvcStateChanged		takes OBJECT, returns STATUS
	 category: observer notification.
	 Tells observers that the service state just changed.

 Informs observers that the state of a service has just changed.  pArgs is
 the UID of the service.
*/
#define msgIOBXSvcStateChanged				MakeMsg(clsIOBXService, 15)

/****************************************************************************
 msgIOBXSvcQueryState		takes P_IOBXSVC_QUERY_STATE, returns STATUS
	 Passes back the state of the service.

*/
#define msgIOBXSvcQueryState				MakeMsg(clsIOBXService, 16)

typedef struct {
	BOOLEAN		enabled;				// is the service enabled?
	CHAR		status[nameBufLength];	// text describing the status of
										// the service.
	CHAR		docName[nameBufLength]; // document being processed
	P_UNKNOWN	pStateData;				// subclass's private data
} IOBXSVC_QUERY_STATE, *P_IOBXSVC_QUERY_STATE;

/****************************************************************************
 msgIOBXSvcGetEnabled		takes P_BOOLEAN, returns STATUS
	Gets the enabled state of the service.

 Subclasses can override this message and redefine the notion of
 "enabled."	 The default behavior of the superclass is to equate
 "enabled" with the ownership of the target service (i.e., input/output
 device).  That is, the service is "enabled" when it owns the target
 service.  By appending to or replacing the default behavior, a
 subclass can define additional conditions which must be met before a
 service is considered enabled.
*/
#define msgIOBXSvcGetEnabled				MakeMsg(clsIOBXService, 17)

/****************************************************************************
 msgIOBXSvcSetEnabled		takes BOOLEAN, returns STATUS
	Sets the enabled state of the service.

 This message is sent to the service in response to service
 notification messages msgSvcOwnerAcquired and msgSvcOwnerReleased.
 Subclasses can provide their own behavior and thereby redefine the
 notion of "enabled" for the service.  If they do, they must pass this
 message up to the ancestor so that observers of the inbox/outbox service
 will be properly notified.
*/
#define msgIOBXSvcSetEnabled				MakeMsg(clsIOBXService, 18)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Inbox/Outbox Document Messages					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgIOBXDocIOStartOK			takes nothing, returns STATUS
	Asks the inbox/outbox document if it is OK to start the input/output process

 When an inbox/outbox service finds an opened document in the inbox/outbox
 section, it sends this message to the document instance, asking
 whether it's OK to start the input/output operation while the document
 remains open.	When the document receives this message, it should
 return stsOK to give the service permission to begin the input/output
 process.  An error status, including stsNotUnderstood, is taken to
 mean that the document instance vetos the request and the service
 will not start the input/output process.
*/
#define msgIOBXDocIOStartOK					MakeMsg(clsIOBXService, 19)

/****************************************************************************
 msgIOBXDocIOStart			takes nothing, returns STATUS
	Tells an inbox/outbox document to start the input/output process.

 This message is sent by the inbox/outbox service to a document.  The
 document should respond to this message by starting the input/output
 process.
*/
#define msgIOBXDocIOStart					MakeMsg(clsIOBXService, 20)

/****************************************************************************
 msgIOBXDocIODone		takes P_IOBX_DOC_OUTPUT_DONE, returns STATUS
	category: client responsibility
	Tells the inbox/outbox service that input/output is finished.

 When the input/output process is finished, the inbox/outbox document in charge of
 the input/output should send this message to the inbox/outbox service.	 This
 message must be sent even if the input/output process has been aborted.
 The pArgs for this message tells the inbox/outbox service what to do with
 the inbox/outbox document.	 If obxDocExitDelete is specified, the document
 will be removed from the inbox/outbox.	 In all other cases the document
 will be unlocked and left in the inbox/outbox.	 If either
 obxDocExitMarkAsCancelled or obxDocExitMarkAsFailed are specified,
 the name of the document will be altered to provide visual indication
 for the user that the input/output process has not completed successfully.

 See Also
	msgIOBXDocGetService
	msgIOBXDocIOCancel
	msgIOBXDocIOCleanup
*/
#define msgIOBXDocIODone					MakeMsg(clsIOBXService, 21)

/****************************************************************************
 msgIOBXDocIOCancel			takes nothing, returns STATUS
	Tells an inbox/outbox document to cancel the input/output process.

 This message is used by the inbox/outbox service to inform a document that
 it should cancel the input/output process.	 The document should handle this
 message by terminating its input/output operation and then sending
 msgIOBXDocIODone to the service with pArgs->behavior set to
 obxDocExistMarkAsCancelled.

*/
#define msgIOBXDocIOCancel					MakeMsg(clsIOBXService, 22)

/****************************************************************************
 msgIOBXDocStatusChanged	takes P_IOBX_DOC_STATUS_CHANGED, returns STATUS
	category: client responsibility
	Tells the inbox/outbox service that the document status is changed.

 This message is sent by the inbox/outbox document to the service whenever
 its status has just changed.  This status is displayed on Status
 column for the inbox/outbox section, in the Inbox/Outbox notebook.
*/
#define msgIOBXDocStatusChanged				MakeMsg(clsIOBXService, 23)

typedef struct IOBX_DOC_STATUS_CHANGED {

	CHAR		status[nameBufLength];	// Text describing document state
	P_UNKNOWN	pDocData;				// Unused: document-specific data

} IOBX_DOC_STATUS_CHANGED, *P_IOBX_DOC_STATUS_CHANGED;

#endif	// IOBXSVC_INCLUDED



