/****************************************************************************
 
 File: import.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.26  $
 $Author:   DYEE  $
 $Date:   24 Feb 1992 20:14:04  $

 This file contains the API definition for clsImport.

 clsImport inherits from clsObject.
 clsImport is the abstract class defining the API for importing foreign
 files from external disks into notebook documents.

****************************************************************************/

/**** Overview ****/
/*
 The import protocol is triggered when the TOC receives msgSelMoveSelection
 or msgSelCopySelection the TOC, and the source of the move/copy includes
 clsFileSystem as an item in the list returned by msgXferList, then the TOC
 initiates the import protocol.  (See xfer.h and sel.h for information
 on PenPoint's move/copy protocol and selection management.)

 The import protocol sends msgImportQuery, as a class message, to each 
 installed application class to determine the set of applications that can
 import the file.

 Once every installed application has been queried, clsApp will put up
 an import dialog box.  An instance of the application is created on
 the destination and msgImport is sent.  If the import succeeds, the
 importer should return stsOK.  If an error occurs and the user has not
 been notified of the failure, the importer should return stsImportFailed.
 If an error occurs and the user has been notified, the importer should
 return stsImportFailedUserNotified.
*/

/**** How to Be an Importing Application ****/
/*
 Any application that wants to import must handle msgImportQuery and
 msgImport.

 The import protocol sends msgImportQuery as a class message.
 (See clsmgr.h for more general information about class messages.)
 For your app to receive a class message you must have an entry
 something like this in your application class's method table:
//{
	MSG_INFO myAppMethods [] = {
		...
    	msgImportQuery, "MyAppImportQuery", objClassMessage,
		...
		0
	};
//}

 The 'ImportQueryHandler' method can look at the contents or the name of 
 the imported file to determine if that file can be imported by the app.  
 If the app can import the file, the 'ImportQueryHandler' method sets 
 the pArgs->canImport boolean to true (the default is false) and returns 
 stsOK.  The TOC will then add the application's name to the list of
 possible import destinations for the import dialog.  
*/

#ifndef IMPORT_INCLUDED
#define IMPORT_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef UID_INCLUDED
#include <uid.h>
#endif

#ifndef FS_INCLUDED
#include <fs.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/**** Status codes ****/
/* 
 * Importing applications should re stsImportFailedUserNotified if the
 * importer detected an error during the importation and notified the user of
 * the error.  This allows the importer to give a more detailed error message
 * to the user.
*/
#define stsImportFailed                 MakeStatus(clsImport, 1)
#define stsImportFailedUserNotified     MakeStatus(clsImport, 2)
#define stsImportInvalidFormat          MakeStatus(clsImport, 3)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgImportQuery     takes P_IMPORT_QUERY, returns STATUS
	category: client responsibility
    Queries each app class to see if it is capable of importing the file.

 This message is sent by the browser to each application class.
 The applicatin should pass back pArgs->canImport set to true if
 it can import the file.  pArgs->suitabilityRating is the relative
 rating of how suitable the application is to importing the file.
 This rating determines the ordering within the list of applications
 in the import dialog box displayed by PenPoint.
*/
#define msgImportQuery                      MakeMsg(clsImport,1)

typedef struct {
    FILE_HANDLE         file;                       // Open file handle to imported file.
    TAG                 fileType;                   // File type if it exists.
    CHAR                fileName[nameBufLength];    // Source file name.
    BOOLEAN             canImport;                  // Out: TRUE if app can import the file.
                                                    // Default setting on entry is false.
    U16                 suitabilityRating;          // Out:   0 - lowest 
                                                    //       50 - average (default)
                                                    //      100 - highest
    U8                  spare[64];                  // Spare: reserved.
} IMPORT_QUERY, *P_IMPORT_QUERY;

/****************************************************************************
 msgImport  takes P_IMPORT_DOC, returns STATUS
	category: client responsibility
    Initiates the import.

 This message is sent by clsApp to a newly created instance of the
 destination application.  The application should import the data from
 the file and return stsOK.  If this message returns an error status
 the newly created app instance will be deleted.
*/
#define msgImport                           MakeMsg(clsImport,2)

typedef struct {
    FILE_HANDLE file;                       // Open file handle to file.
    TAG         fileType;                   // File type if exists.
    U8          fileName[nameBufLength];    // Source file name.
    U32         sequence;                   // Sequence number for dest.
    DIR_HANDLE  destHandle;                 // Dir handle to dest section.
} IMPORT_DOC, *P_IMPORT_DOC;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Miscellaneous							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/**** Help tags ****/
/*
 * These are help tags on various pieces of the standard export dialog box.
*/
#define hlpImportSheet                  MakeTag(clsImport, 50)
#define hlpImportName                   MakeTag(clsImport, 51)
#define hlpImportNewName                MakeTag(clsImport, 52)
#define hlpImportChoice                 MakeTag(clsImport, 53)

#endif // IMPORT_INCLUDED


