/****************************************************************************
 dvintr.h

 Copyright 1992, GO Corporation,   All Rights Reserved

 The public definition of the interrupt controller device interface.

 $Revision:   1.7  $
   $Author:   thall  $
     $Date:   24 Mar 1992 15:10:38  $
****************************************************************************/
#ifndef DVINTR_INCLUDED
#define DVINTR_INCLUDED

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 This section contains definitions specific to the interrupt controller
 device. The definitions contained here are data structures that are
 necessary for the MIL Interrupt Controller specific functions along
 with standard MIL device data structures with additional fields
 specific to the interrupt controller device.
*/

/**** Interrupt Controller Device Specific Function Numbers ****/

/*
 The following definitions are the device specific functions supported
 by the interrupt controller device. The function numbers start immediately
 after the last function number used by the functions that are common to
 all MIL devices. These function numbers correspond to the Function
 Transfer Table (FTT) entries.

*/

#define milIntrGetDevParameters	milLastCommonFunction+1
#define milIntrSetMask			milLastCommonFunction+2
#define milIntrSetBase			milLastCommonFunction+3
#define milIntrPending			milLastCommonFunction+4
#define milIntrEndOfInterrupt	milLastCommonFunction+5

#define fnCountIntr				fnCountBase+5


/**** Interrupt Controller Device Function Transfer Table ****/
/*
 This structure is the declaration of the Interrupt Controller
 Device Function Transfer Table. It includes one entry for each
 of the MIL driver functions common to all MIL devices along with
 one entry for each of the device specific functions for the MIL
 Interrupt Contrller device. Each entry contains fields that define
 the size of the request block for that function, the staging
 requirements of the function, and the address of the actual function. 
*/

typedef struct MIL_FTT_INTR {
	funcTransTableCommonFields
	MIL_FUNCTION_DESC functionDesc[fnCountIntr];
} MIL_FTT_INTR, *P_MIL_FTT_INTR;




/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 		   	Interrupt Controller Device Specific Functions	  			   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 The following function definitions are the interrupt controller device
 specific functions. Access to these functions is provided by the FTT
 defined above. All of the functions defined below will be passed 3
 pointers.

	pCommonData:
		This is a pointer to the common data area.

	pRB:
		This is a pointer to the function request block.

	pDeviceBlock:
		This is a pointer to the device block for the device

 All of the functions return STATUS. Any STATUS other than staging 
 statuses will cause the request to terminate. The staging statuses
 will result in the specified staging operation to be performed.
*/


/****************************************************************************
 milIntrGetDevParameters		returns STATUS
 	Provides interrupt controller device specific information.

 This function is a single stage function that provides device
 specific information about the interrupt controller device.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milIntrGetDevParameters funtion request block contains a data 
 structure that defines the interrupt controller device specific
 parameters. 

	intrFlags:
		This output field contains configuation, mode, and/or status 
		information bits.

	intrMask:
		This output field indicates the state of all the interrupts.
		each bit represents an interrupt. A value of 1 indicates the
		corresponding interrupt is enabled.

	intrValid:
		This output field is a mask that indicates the supported
		hardware interrupts. This field should be the logical
		"OR" of all device interruptLevels fields.

*/
		
typedef struct MIL_INTR_DEVICE_PARAMETERS {
	U16 intrFlags;
	U32	intrMask;
	U32 intrValid;
	U16 reservedU16[2];
	U32 reservedU32[2];
} MIL_INTR_DEVICE_PARAMETERS, *P_MIL_INTR_DEVICE_PARAMETERS;

/*
 intrFlags

 The following values are the bit values defined for the intrFlags
 field in the device parameters.

	milIntrEdgeTriggered:
		This bit indicates the interrupt controller is edge 
		triggered. 
*/	

#define milIntrEdgeTriggered	flag0

/*
 rbPublicFieldsIntrGetDevParameters

 This define merges the device specific data structure with the standard
 MIL device parameters reserved fields.

*/

#define rbPublicFieldsIntrGetDevParameters \
	MIL_INTR_DEVICE_PARAMETERS intrDevParms; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milIntrGetDevParameters Request Block

*/

typedef struct MIL_RB_PUB_INTR_GET_DEV_PARAMETERS {
	milRequestBlockCommonFields
	rbPublicFieldsIntrGetDevParameters
} MIL_RB_PUB_INTR_GET_DEV_PARAMETERS, *P_MIL_RB_PUB_INTR_GET_DEV_PARAMETERS;


/****************************************************************************
 milIntrSetMask		returns STATUS
 	Controls mask bits for specified interrupt level.

 This function is a single stage function that provides control
 over interrupt enable mask. A call to this  function can either
 enable or disable one or more interrupt levels.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milIntrSetMask funtion request block contains the following
 fields.

	enable:
		This input field is a boolean that indicates the action to
		be performed on the indicated interrupt levels. If TRUE
		the interrupts will be enabled.

	mask:
		This input field contains 1 bit for each interrupt. It is
		used to indicate which interrupts are to be affected by
		this function.

*/

#define rbPublicFieldsIntrSetMask \
	BOOLEAN enable; \
	U32 mask; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milIntrSetMask Request Block

*/

typedef struct MIL_RB_PUB_INTR_SET_MASK {
	milRequestBlockCommonFields
	rbPublicFieldsIntrSetMask
} MIL_RB_PUB_INTR_SET_MASK, *P_MIL_RB_PUB_INTR_SET_MASK;


/****************************************************************************
 milIntrSetBase		returns STATUS
 	Sets the base of the interrupt vectors.

 This function is a single stage function that is specific to the 80386
 family of processors. It allows the base of the interrupt vectors in the
 Interrupt Descriptor Table (IDT) to be set in the interrupt controller.
 When an interrupt occurs the interrupt-response logic will use this
 base to select an entry in the Interrupt Descriptor Table.
 
*/

/**** Device Specific Request Block Fields ****/
/*

 The milIntrSetBase funtion request block contains the following
 fields.

	interruptBase:
		This input field is a interrupt number that will be used
		as the first interrupt in the interrupt controller. 

*/

#define rbPublicFieldsIntrSetBase \
	U16	interruptBase; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milIntrSetBase Request Block

*/

typedef struct MIL_RB_PUB_INTR_SET_BASE {
	milRequestBlockCommonFields
	rbPublicFieldsIntrSetBase
} MIL_RB_PUB_INTR_SET_BASE, *P_MIL_RB_PUB_INTR_SET_BASE;


/****************************************************************************
 milIntrPending		returns STATUS
 	Indicates if an interrupt is pending.

 This function is a single stage function that returns a boolean value that
 indicates if the specified interrupt level is pending.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milIntrPending funtion request block contains the following
 fields.

	interruptLevel:
		This input field is a interrupt number that specifies the 
		interrupt to be checked.

	pending:
		This field is an output field that indicates if the interrupt
		specified in interruptLevel is pending. If this field is returned
		as a TRUE value the interrupt is pending.

*/

#define rbPublicFieldsIntrPending \
	U16 interruptLevel; \
	BOOLEAN pending; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];
/*
 
 milIntrPending Request Block

*/

typedef struct MIL_RB_PUB_INTR_PENDING {
	milRequestBlockCommonFields
	rbPublicFieldsIntrPending
} MIL_RB_PUB_INTR_PENDING, *P_MIL_RB_PUB_INTR_PENDING;


/****************************************************************************
 milIntrEndOfInterrupt		returns STATUS
 	Clears the pending interrupt.

 This function is a single stage function that signals that a given 
 interrupt level has been serviced. 

*/

/**** Device Specific Request Block Fields ****/
/*

 The milIntrEndOfInterrupt funtion request block contains the following
 fields.

	interruptLevel:
		This input field is a interrupt number that specifies the 
		interrupt to be cleared. If it is set to milIntrNonSpecificLevel
		(defined below) the highest priority interrupt level pending
		will be cleared.

*/

#define milIntrNonSpecificLevel 32

#define rbPublicFieldsIntrEndOfInterrupt \
	U16 interruptLevel; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milIntrEndOfInterrupt Request Block

*/

typedef struct MIL_RB_PUB_INTR_END_OF_INTERRUPT {
	milRequestBlockCommonFields
	rbPublicFieldsIntrEndOfInterrupt
} MIL_RB_PUB_INTR_END_OF_INTERRUPT, *P_MIL_RB_PUB_INTR_END_OF_INTERRUPT;

#endif
