/****************************************************************************
 dvblock.h

 Copyright 1992, GO Corporation,   All Rights Reserved

 The public definition of the generic block device interface. All other
 block device types are a superset of this functionality.

 $Revision:   1.15  $
   $Author:   thall  $
     $Date:   17 Mar 1992 11:35:12  $
****************************************************************************/

#ifndef DVBLOCK_INCLUDED
#define DVBLOCK_INCLUDED

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Block Device Specific Function Numbers ****/

/*
 The following definitions are the device specific functions supported
 by the block devices. The function numbers start immediately after the
 last function number used by the functions that are common to all
 MIL devices. These function numbers correspond to the Function Transfer
 Table (FTT) entries.
*/

#define milBlockGetDevParameters	milLastCommonFunction+1
#define milBlockRead				milLastCommonFunction+2
#define milBlockWrite				milLastCommonFunction+3
#define milBlockSetMediumParameters	milLastCommonFunction+4
#define milBlockGetMediumParameters	milLastCommonFunction+5
#define milBlockGetStatus			milLastCommonFunction+6
#define milBlockEjectMedium			milLastCommonFunction+7
#define milBlockGetStandardFormat   milLastCommonFunction+8
#define milBlockFormatMedium		milLastCommonFunction+9
#define milBlockFormatTrack			milLastCommonFunction+10

#define milLastBlockFunction		milBlockFormatTrack

#define fnCountBlock				fnCountBase+10

/**** Block Device Function Transfer Table ****/
/*
 This structure is the declaration of the Block Device Function
 Transfer Table. It includes one entry for each of the MIL driver
 functions common to all MIL devices along with one entry for
 each of the device specific functions for MIL block devices.
 Each entry contains fields that define the size of the request
 block for that function, the staging requirements of the function,
 and the address of the actual function. 
*/

typedef struct MIL_FTT_BLOCK {
	funcTransTableCommonFields						// common functions
	MIL_FUNCTION_DESC functionDesc[fnCountBlock];	// block device specific
} MIL_FTT_BLOCK, *P_MIL_FTT_BLOCK;


/****   Block Device Related Resources Stored in MIL.RES   ****/
/*

 These are mil.res resources used by block devices.

	pcBootSectorTemplateResId: 
		This is the resource ID for the boot sector for block devices.

	pcPartitionSectorTemplateResId: 
		This is the resource ID for the partitioned device boot sector.

*/

#define pcBootSectorTemplateResId		MakeWknResId (clsMILBlockDevice, 0)
#define pcPartitionSectorTemplateResId	MakeWknResId (clsMILBlockDevice, 1)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 					Block Device Specific Functions	   	 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****  Introduction  ****/
/*
 The following function definitions are block device specific functions. 
 Access to these functions is provided by the FTT defined above. All of
 the functions defined below will be passed 3 pointers.

	pCommonData:
		This is a pointer to the common data area.

	pRB:
		This is a pointer to the function request block.

	pDeviceBlock:
		This is a pointer to the device block for the device

 All of the functions return STATUS. Any STATUS other than staging 
 statuses will cause the request to terminate. The staging statuses
 will result in the specified staging operation to be performed.
*/
 

/****************************************************************************
 milBlockGetDevParameters		returns STATUS
 	Provides device and unit specific information.

 This function provides device and unit specific information about the 
 requested unit. The request block for this function contains two data
 structures specific to block devices. One of the data structures will
 be filled with device specific information common to all units attached
 to this block device. The other data structure will be filled with
 information specific to the specified unit.

*/ 
 

/**** Device Specific Information ****/
/*

 This data structure contains information specific to block devices but
 common to all units for this block device.

*/

typedef struct MIL_BLOCK_DEVICE_PARAMETERS {
	U32 blockDevFlags;		   // block device flags (see defines below) 
	U16 reservedU16[2];		   // reserved for future use
	U32 reservedU32[2];		   // reserved for future ues
} MIL_BLOCK_DEVICE_PARAMETERS, *P_MIL_BLOCK_DEVICE_PARAMETERS;

/*
	blockDevFlags:
		No block device specific flags are currently defined.
*/

/**** Unit Specific Information ****/
/*
 
 This data structure contains information specific to a unit of
 a block device.
 
 	blockUnitFlags: 
		The blockUnitFlags describe the characteristics 
	 	of the specified unit. Each bit in this field defines a specific
		characteristic. See the	definitions for these flag bits below for
		more information.
 
 	numStandardMediumFormats:
		This field specifies the number of media
		layouts supported for a device. For example, 3.5 inch floppies 
		support 2 standard medium formats, 720K and 1.44M.
 
	maxBlockSize:
		This field identifies the maximum size allowed for a
		logical block on the device. For devices that support different 
		standard formats that have different block sizes this field should 
		be set to the number of bytes in the largest of thoes supported 
		block sizes.
 
	retries:
		This field indicates the maximum number of retries this
		device will perform if a recoverable error is detected when 
		performing an operation.
 
 	resourceId:
		This field is reserved for future use.
 
*/

typedef struct MIL_BLOCK_UNIT_PARAMETERS {
	U32 blockUnitFlags;				// bits define unit characteristics			
	U16 numStandardMediumFormats;	// number of supported media layouts
	U16 maxBlockSize;				// maximum logical block size supported
	U16 retries;					// # retires used by the driver
	U32 resourceID;					// future use 
	U16	bootDeviceIndex;			// matches boot struct bootDeviceIndex
	U16 reservedU16[2];				// reserved for future use
	U32 reservedU32[2];				// reserved for future use
} MIL_BLOCK_UNIT_PARAMETERS, *P_MIL_BLOCK_UNIT_PARAMETERS;

/*
 blockUnitFlags

 The meaning of the blockUnitFlags are as follows:

 	milRemovableMedia:
		unit supports removable media.

 	milEjectableMedia:
		unit supports "Eject Media" function.

 	milReadOnlyMedia:
		write and format operations are not supported.

	milUsesPartitionTable:
		media has a partition table.

 	milMediumChangeDetectable:
		media change and removal is detectable.

 	milWriteProtectDetectable:
		can detect write protect w/o operation error.

 	milAsyncBatteryStatusNotify:
		supports low battery event on attach func.

 	milAsyncMediumChangeNotify:
		supports media change event on attach func.	

 	milFormatMediumSupported:
		unit supports media format function.

 	milFormatTrackSupported:
		unit support track format function.	

 	milBlkTheSwapUnit:
		unit is the system swap unit.

*/					

#define milRemovableMedia				flag0	// removable media
#define milEjectableMedia				flag1	// supports hardware eject
#define milReadOnlyMedia				flag2	// read only media
#define milUsesPartitionTable			flag3	// has partition table
#define milMediumChangeDetectable		flag4	// has changline support
#define milWriteProtectDetectable		flag5	// w/p detected w/o operation
#define milAsyncBatteryStatusNotify		flag6	// has battery state events
#define milAsyncMediumChangeNotify		flag7	// has media change events
#define milFormatMediumSupported		flag8	// supports format media
#define milFormatTrackSupported			flag9	// supports format track
#define milBlkTheSwapUnit				flag10  // is the swap unit
#define	milBlkBootDevIndexValid			flag11	// boot index is valid

/****  Request Block Definition  ****/
/*

 The following section defines the default request block for
 milBlockGetDevParameters. Several defines define statements
 are used so a specific device can define a request block
 that includes more fields AFTER the default fields. This allows a
 block device to add private areas in the request block for function
 specific work variables. NOTE, THE DEFAULT FIELDS MUST COME FIRST.

*/

/*
 blockDeviceParameterFields

 This define merges the device specific data structure with the unit
 specific data structure.

*/

#define blockDeviceParametersFields	\
	MIL_BLOCK_DEVICE_PARAMETERS blockDevParms; \
	MIL_BLOCK_UNIT_PARAMETERS blockUnitParms;

/*
 rbPublicFieldsBlockGetDevParameters

 This defininition creates a name that can be used to specify
 all the PUBLIC fields in the default request block. 

*/
#define rbPublicFieldsBlockGetDevParameters \
	blockDeviceParametersFields \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*

 milBlockGetDevParameters Request Block

*/

typedef struct MIL_RB_PUB_BLOCK_GET_DEV_PARAMETERS {
	milRequestBlockCommonFields				// common request block fields
	rbPublicFieldsBlockGetDevParameters		// function specific fields
} MIL_RB_PUB_BLOCK_GET_DEV_PARAMETERS, 
	*P_MIL_RB_PUB_BLOCK_GET_DEV_PARAMETERS;


/****************************************************************************
 milBlockRead		returns STATUS
 	This function reads blocks from a block device.

 This funcion performs a block device read operation. The request block
 contains the logical starting block number, the block count, and
 a pointer to the destination buffer.  

 The following section defines the default request block for milBlockRead.
 Define statements are used to define the public fields. These can
 be used to create custom request block for block devices that need
 function specific private data variables. 

 NOTE, the public fields common to disk read and write must
 be at the same location in a request block. This allows common setup
 code inside the MIL implementations. Read/Write-specific fields can
 be added after the common fields BUT ONLY IF THEY ARE THE SAME SIZE.
 This allows the private fields to also be declared at the same offsets.

*/

/*
 rbPublicFieldsBlockReadWrite

 This definition groups all public fields together that are common
 to the read write functions.

*/

#define rbPublicFieldsBlockReadWrite \
	P_U8 pBuffer;       \
	U32 startBlock;     \
	U32 blockCount;     \
	U16 hardwareError;	

/*
 rbPublicFieldsBlockRead

 This definition groups the read/write common fields together with
 the fileds that are specific to read.

 	readFlags:
		These flags indicate how the read should be performed.
		see the definitions below.

*/

#define rbPublicFieldsBlockRead  \
	rbPublicFieldsBlockReadWrite \
	U16 readFlags;               \
	U16 funcReservedU16[2]; 	 \
	U32 funcReservedU32[2];

/*
 readFlags

 	milVerifyOnly:
		This flag indicates the designated blocks are to be
		read but no data is stored in memory. This is used to make sure an
		area on the block device is readable.

*/

#define milVerifyOnly	flag0		// perform verify only read

/*

 milBlockRead Request Block

*/

typedef struct MIL_RB_PUB_BLOCK_READ {
	milRequestBlockCommonFields
	rbPublicFieldsBlockRead
} MIL_RB_PUB_BLOCK_READ, *P_MIL_RB_PUB_BLOCK_READ;


/****************************************************************************
 milBlockWrite		returns STATUS
 	This function write blocks to a block device.

 This funcion performs a block device write operation. The request block
 contains the logical starting block number, the block count, and
 a pointer to the source buffer.  

 The following section defines the default request block for milBlockWrite.
 It uses names defined above in the milBlockRead section

 NOTE, the public fields common to disk read and write must
 be at the same location in a request block. This allows common setup
 code inside the MIL implementations. Read/Write-specific fields can
 be added after the common fields BUT ONLY IF THEY ARE THE SAME SIZE.
 This allows the private fields to also be declared at the same offsets.

*/

/*
 rbPublicFieldsBlockWrite

 This definition groups the read/write common fields together with
 the fields that are specific to write.

 	writeFlags:
		These flags indicate how the write should be performed.
 		see the definitions below.

*/

#define rbPublicFieldsBlockWrite \
	rbPublicFieldsBlockReadWrite \
	U16 writeFlags; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 writeFlags

 	milWriteAndVerify:
		This flag indicates the designated blocks are to be
 		written and then verified to make sure the data was written correctly.

*/

#define milWriteAndVerify	flag0

/*

 milBlockWrite Request Block

*/

typedef struct MIL_RB_PUB_BLOCK_WRITE {
	milRequestBlockCommonFields
	rbPublicFieldsBlockWrite
} MIL_RB_PUB_BLOCK_WRITE, *P_MIL_RB_PUB_BLOCK_WRITE;

/****************************************************************************
 milBlockSetMediumParameters		Returns STATUS
	Selects one of the supported formats as the current format.

 This function selects the caller to select one of the supported formats as 
 the current format for the indicated unit. 

 This function is used to set the current format prior to a 
 milBlockFormatMedium or milBlockFormatTrack function call.

 This function also allows the operating system several options in determing
 the format of new media. When a driver for a unit that supports removable
 media and multiple media formats detects a new media has been inserted it
 may not be able to determine the media format prior to a read or write
 operation. If this is the case, the only sector that the operating system
 should attempt to access is sector 0. When the system reads sector 0 the
 driver should make every reasonable attempt to identify the format by
 trying as many of the standard formats supported by the driver as is
 reasonable. The operating system can verify the drivers media format
 selection if there is media format information stored in sector 0 of the
 media. If the operating system determines the format is incorrect it can
 override the drivers selection by performing this call.

 The operating system can also control the attempted media formats when
 a format is unknown by setting the media format waith this function and
 attempting an operation. If the operation fails, the system can attempt
 another format. Using this function to set the format disables the drivers
 automatic attempts to determine the format.

*/

/*
 rbPublicFieldsBlockSetMediumParameters

 This defines the public fields for milBlockSetMediumParameters.
 
 	formatIndex:
		This field is a number in the range of 0 to the
		number of supported formats -1. The value identifies an entry
		in the drivers internal table of supported formats. Information
		concerning these formats can be accessed by performing the 
		milBlockGetStandardFormat function call.

*/

#define rbPublicFieldsBlockSetMediumParameters \
	U16 formatIndex; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*

 milBlockSetMediumParameters Request Block

*/

typedef struct MIL_RB_PUB_BLOCK_SET_MEDIUM_PARAMETERS {
	milRequestBlockCommonFields
	rbPublicFieldsBlockSetMediumParameters
} MIL_RB_PUB_BLOCK_SET_MEDIUM_PARAMETERS, 
	*P_MIL_RB_PUB_BLOCK_SET_MEDIUM_PARAMETERS;



/****************************************************************************
 milBlockGetMediumParameters	Returns STATUS
	This function returns the current medium format if known.

 This function returns a data structure that describes the current 
 medium format. If the current format is already known or the driver can
 determine the format it returns the format format information. If the
 format is not known and the driver can not determine the format it will
 return stsMILUnknownMediaError. 

*/
/*

 MEDIUM_FORMAT

 This structure defines the medium format. 

 	formatIndex:
		Index of the current format in the drivers internal
    	table of supported formats.

 	numBlocks:
		This field specifies the total number of blocks.

	blockSize:
		This field specifies the block size in bytes.

	blocksPerTrack:
		This field indicates the number of blocks per track.

	numHeads:
		This field indicates the number of heads in the format.

	numTracks:
		This field indicates the total number of tracks.

	dataTransferRate:
		This field contains data transfer rate information.

	formatGapLength:
		This field indicates the intersector gap length.

	formatFillByte:
		This field indicates the byte value to use for format.

	readWriteGapLength:
		This filed is the gap to be used on read or write.

	recommendedInterleave:
		This is the recommended sector interleave.

	optimumCacheSize:
		This field is the recommended cache size.

	minFormatTrackBufferSize:
		This is the requested buffer size for
		a format track operation.

	resourceId:
		This is a resource ID that is associated with this format.

*/

typedef struct MEDIUM_FORMAT {
	U16	formatIndex;				// standard format index
	U32 numBlocks; 					// total number of blocks
	U16 blockSize; 					// block size in bytes
	U16 blocksPerTrack; 			// number of blocks in a track
	U16 numHeads;					// number of read/write heads
	U16	numTracks; 					// total number of tracks
	U16 dataTransferRate;			// data transfer rate
	U16 formatGapLength;			// gap length for format operation
	U16 formatFillByte;				// fill byte for format
	U16	readWriteGapLength;			// gap length for read/write
	U16 recommendedInterleave;		// recommended sector interleave
	U16 optimumCacheSize;			// recommended cache size
	U16 minFormatTrackBufferSize; 	// requested format buffer size
	U32 resourceId;					// future use 		  
} MEDIUM_FORMAT, *P_MEDIUM_FORMAT;

/* 
 rbPublicFieldsBlockGetMediumParameters

 This is the definition of the public fields defined in the
 milBlockGetMediumParameters request block.

*/

#define rbPublicFieldsBlockGetMediumParameters \
	MEDIUM_FORMAT	mediumFormat; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 milBlockGetMediumParameters Request Block

*/

typedef struct MIL_RB_PUB_BLOCK_GET_MEDIUM_PARAMETERS {
	milRequestBlockCommonFields
	rbPublicFieldsBlockGetMediumParameters
} MIL_RB_PUB_BLOCK_GET_MEDIUM_PARAMETERS, 
	*P_MIL_RB_PUB_BLOCK_GET_MEDIUM_PARAMETERS;


/****************************************************************************
 milBlockGetCurrentStatus	Returns STATUS
 	This function returns the current status of the indicated unit.

 This function returns the current status of the media detection and the 
 media battery level if applicable.

*/

/*
 rbPublicFieldsBlockGetStatus

 This defines the public fields that are part of the
 milBlockGetCurrentStatus request block.

 currentStatus:
 	This field defines the current status. Each bit 
 	represents a piece of status information. The definitions of the
 	bits are below.

*/

#define rbPublicFieldsBlockGetStatus \
	U32 currentStatus; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 currentStatus bit definitions

	milUnitPresent:
		This bit indicates the unit is connected.

	milMediumPresent:
		This bit indicates medium is in the unit.

	milMediumChanged:
		This bit indicates the medium may have been changed.

	milMediumWriteProtected:
		This bit indicates write protect is on.

	milMediumBatteryBelowWarn:
		This bit indicates medium battery is low.

	milMediumBatteryFail:
		This bit indicates the medium battery has failed.

*/

#define milUnitPresent					flag0
#define	milMediumPresent				flag1	
#define milMediumChanged				flag2	
#define milMediumWriteProtected  	    flag3   
#define milMediumBatteryBelowWarn		flag4  	
#define milMediumBatteryFail			flag5	

/*
 milBlockGetCurrentStatus Request Block

*/

typedef struct MIL_RB_PUB_BLOCK_GET_STATUS {
	milRequestBlockCommonFields
	rbPublicFieldsBlockGetStatus
} MIL_RB_PUB_BLOCK_GET_STATUS, 
	*P_MIL_RB_PUB_BLOCK_GET_STATUS;


/****************************************************************************
 milBlockEjectMedium 	Returns STATUS
	This function causes the unit to eject the medium.

 This function is a request for the unit to eject the medium. If this
 feature is not supported a stsMILNotSupported error will be returned.

*/

/*
 rbPublicFieldsBlockEjectMedium

 This defines the public fields that are in the milBlockEjectMedium
 request block.

 	hardwareError:
		This field allows the driver to return a hardware
 		specific error code if an error occurs during the eject operation.

*/

#define rbPublicFieldsBlockEjectMedium \
	U16 hardwareError; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 milBlockEjectMedium Request Block

*/

typedef struct MIL_RB_PUB_BLOCK_EJECT_MEDIUM {
	milRequestBlockCommonFields
	rbPublicFieldsBlockEjectMedium
} MIL_RB_PUB_BLOCK_EJECT_MEDIUM, *P_MIL_RB_PUB_BLOCK_EJECT_MEDIUM;



/****************************************************************************
 milBlockGetStandardFormat	Returns STATUS
	This function returns information about a supported standard format.

 This function is called to find out information about the standard formats.
 A call provides the index for the standard format requested. The driver
 returns the information concerning that format from the internal
 standard format table. If the format index is out of range, the driver
 returns stsMILInvalidParameter.	

*/

/*
 rbPublicFieldsGetStandardFormat

 This definition defines the public fields in the milBlockGetStandardFormat
 request block.

	formatIndex:
		This is a number in the range of 0 to the number of
		supported standard formats - 1. 

	mediumFormat:
		This field is a structure that will receive the standard
		format information. This structure was defined for a previous function
		above.

*/

#define rbPublicFieldsBlockGetStandardFormat \
	U16				formatIndex; \
	MEDIUM_FORMAT	mediumFormat; \
	U16 			funcReservedU16[2]; \
	U32 			funcReservedU32[2];

/*
 milBlockGetStandardFormat Request Block

*/

typedef struct MIL_RB_PUB_BLOCK_GET_STANDARD_FORMAT {
	milRequestBlockCommonFields
	rbPublicFieldsBlockGetStandardFormat
} MIL_RB_PUB_BLOCK_GET_STANDARD_FORMAT,
	*P_MIL_RB_PUB_BLOCK_GET_STANDARD_FORMAT;

												
/****************************************************************************
 milBlockFormatMedium 	Returns STATUS
	This function performs a full medium format.

 This function, if supported, formats the entire medium. It is assumed
 that a current medium format has already been establish by calling the
 milBlockSetMediumParameters function.

*/

/*
 rbPublicFieldsBlockFormatMedium

 This definition defines the public fields in the milBlockFormatMedium
 request block.

	interleave:
		This field allows the user to specify the interleave
		to use for the format operation. If this field is set to 0 the
		recommended interleave for the current format will be used.

	hardwareError:
		This field allows the driver to return a hardware
		specific error code if an error occurs during the format.

*/

#define rbPublicFieldsBlockFormatMedium \
	U16 	interleave; \
	U16 	hardwareError; \
	U16 	funcReservedU16[2]; \
	U32 	funcReservedU32[2];

/*
 milBlockFormatMedium Request Block

*/

typedef struct MIL_RB_PUB_BLOCK_FORMAT_MEDIUM {
	milRequestBlockCommonFields
	rbPublicFieldsBlockFormatMedium
} MIL_RB_PUB_BLOCK_FORMAT_MEDIUM, *P_MIL_RB_PUB_BLOCK_FORMAT_MEDIUM;

/****************************************************************************
 milBlockFormatTrack	Returns STATUS
	This function formats the specified track.
 
 This function is called for format a track on the specified unit. The 
 driver uses the starting block along with the current format blocks
 per track value to determine the track number to be formatted. The
 track is formatted with the specfied interleave. If the interleave is
 0 the current format recommended interleave will be used. If the 
 minFormatTrackBuffer size field is not zero the caller will provide
 a pointer to a buffer that the driver can use as a storage area if
 needed.

*/

/*
 rbPublicFieldsBlockFormatTrack

 This definition defines the public fields for the milBlockFormatTrack
 request block.

	pBuffer:
		This field will be a pointer to a buffer if the 
		minFormatTrackBufferSize in the current format description is not 0.

	interleave:
		This field is the requested interleave. The recommended
		interleave for the current format will be used if this value is 0.

	startBlock:
		This field is the starting point of the track to be
		formatted. The current format blocks per track and number of heads
		is used to calculate the track number.

	hardwareError:
		This field allows the driver to return a hardware
		specific error code if an error occurs during the format.

*/

#define rbPublicFieldsBlockFormatTrack \	
	P_U8 			pBuffer; \
	U16 			interleave; \
	U32 			startBlock; \
	U16 			hardwareError; \
	U16 			funcReservedU16[2]; \
	U32 			funcReservedU32[2];

/*
 milBlockFormatTrack Request Block

*/

typedef struct MIL_RB_PUB_BLOCK_FORMAT_TRACK {
	milRequestBlockCommonFields
	rbPublicFieldsBlockFormatTrack
} MIL_RB_PUB_BLOCK_FORMAT_TRACK, *P_MIL_RB_PUB_BLOCK_FORMAT_TRACK;

#endif
