/****************************************************************************
 File: ccitt.h

 (C) Copyright 1992 by GO Corporation. All Rights Reserved.

 $Revision:   1.6  $
   $Author:   cleu  $
     $Date:   24 Feb 1992 12:02:34  $

 CCITT Fax Group 3, one-dimensional data encoding and decoding routines.
 The functions described in this file are contained within CCITT.LIB.
****************************************************************************/
#ifndef CCITT_INCLUDED
#define CCITT_INCLUDED

typedef struct ENCODE31			// Structure for use by function CcittEncode31
{
	U16			pixCnt; 		// In:	How many pixels in the scanline.
	BOOLEAN		photoNegative;	// In:	Input bitmap's palette:
								//		true:  0 = white, 1 = black.
								//		false: 1 = white, 0 = black.
	P_U8		pScanLine;		// In:	Scanline data to encode.
								//		Note: A scanline must be
								//				a multiple of words.
	BOOLEAN		writeEol;		// In:	EOL code is to be written at
								//		the beginning of the scanline.
	BOOLEAN		writeRtc;		// In:	6 EOLs are to be written at
								//		the end of the scan line.
	P_U8		pOutBuf;		// In:	Starting byte at which to put data.
								//		The buffer size must accomodate a
								//		worst case encoding for one scanline.
								//		2*pixCnt, +2 w/EOL, +9 w/RTC.
	U16			inBitPos;		// In:	Bit # in pOutBuf to start encoding
								//		Bit 0 = MSB, Bit 7 = LSB
	P_U8		pOutLast;		// Out: Last byte where data was put
	U16			byteUsed;		// Out: Number of bytes used for encoding,
								//		including the last partial byte.
	U16			outBitPos;		// out: Bit # in pOutLast where last bit
								//		was put + 1.
} ENCODE31, *P_ENCODE31;

#define ccittDecodeToPacked		0	// Decode to Packed bitmap.
#define ccittDecodeToRunLen		1	// Decode to sample image operator Run-Length.
#define ccittDecodeToGroup3_1D	2	// Decode to Group 3 1-Dimension fax encoding.

typedef struct DECODE31			// Structure for use by fuction CcittDecode31
{
	S16			format; 		// in:	ccittDecodeToPacked,RunLen, or Group3_1D.
	S16			pixCnt; 		// in:	How many pixels comprise a scanline.
	BOOLEAN		readEolRtc;		// in:	EOL or RTC string is to be read
								//		at the end of each scanline.
	BOOLEAN		photoNegative;	// in:	Output palette:
								//		true:  0 = white, 1 = black
								//		false: 1 = white, 0 = black.
	BOOLEAN		newLine;		// in/out:	Must be set to true at the
								//			start of each scanline and left
								//			alone for remainder of scanline.
	P_U8		pInBuf; 		// in:	Input buffer:
								//		Starting byte of data to decode.
	U16			inBitPos;		// in:	Bit # in pInBuf to start decoding.
								//		Bit 0 = MSB, Bit 7 = LSB
	S16			inBufSz;		// in:	# of data bytes within input buffer.
	P_U8		pOutBuf;		// in:	Output buffer:
								//		This field should be initialized once
								//		at the beginning of each scanline and
								//		left alone for the rest of the line.
								//		The size of the output buffer must
								//		accomodate for the worst case decoding
								//		pixCnt for decodeToRunLength,
								//		2*((pixCnt+15)/16) for ccittDecodeToPacked,
								//		((9*pixCnt)/16)+2 for ccittDecodeToGroup3_1D.
	S16			decodedSz;		// out: The number of bytes of decoded output
								//		placed into *pOutBuf.
	BOOLEAN		done;			// out: A complete scanline has been decoded.
	BOOLEAN		rtcRead;		// out:	RTC detected (6 consecutive EOLs).
	P_U8		pInLast;		// out: Points to last data byte within
								//		*pInBuf that was decoded.
	S16			lastBitPos;		// out:	Next bit # within *pInLast byte
								//		that will be decoded.
	S16			outBitPos;		// private:	Bit # within pOutBuf at which next
								//		bit of decoded data will be placed.
	BOOLEAN		curIs0;			// private: Last run was zero bits/pixels.
	S16			nDecoded;		// private: # of scanline pixels decoded.
	S16			nEolRead;		// private: # of EOLs read with scanline.
    BOOLEAN		resyncToNextEol;// private: Resync to next EOL - data error.
	S16			adjacentZeros;	// private: Consecutive zero bit run count.

} DECODE31, *P_DECODE31;

/****************************************************************************
 CcittEncode31 returns nothing
    Encode one scanline of a packed bitmap into fax group 3 T.4 1-D format.
*/
void EXPORTED CcittEncode31 (
   P_ENCODE31 pEncode );

/****************************************************************************
 CcittDecode31 returns BOOLEAN
    Decode one scanline worth of fax group 3 T.4 1-D image data.

    Output can be either the packed bitmap format, sampled image operator
	run length encoded format, or Group 3 1 dimensional image format without
	EOLs. Function returns true if successful, false if the input data
    are not valid fax data. The interface to this function is such that
    multiple calls may be needed to decode a complete scanline. As such,
    intermediate states are kept in the interface structure. Fields labeled
    with private are not to be molested by the caller.

    Example of decoding a TIFF CCITT/3 image (where there is no EOL or RTC
    bits and the number of scanlines is known a priori, using a decoded
    format of our run length format:

//{
    decode.format		= ccittDecodeToRunLen;
    decode.pixCnt		= 1024;
    decode.readEolRtc	= false;

	for (all scanlines)
	{
		decode.inBitPos	= 0;
		decode.pOutBuf	= whatever;
		decode.pInBuf	= whatever;
		decode.inBufSz	= whatever;
		decode.newLine	= true;

		while(true)
		{
			if (!CcittDecode31(&decode))
				break;		// ------ the input data is screwed up.

			if (decode.done)
				break;		// ------ done decoding current scanline.

			//	Supply new bits for next call.  Note that there may be
			//	partial bits left undecoded within the last decoded byte.
			//	The next call to decode must start with any undecoded bits.
			/	If you buffer the source bits, then copy all undecoded bits
			//	into the new buffer. The pInLast and lastBitPos fields tell
			//	you the amount left undecoded.

			decode.pInBuf   = pInLast;				//	Or your new buffer.
			decode.inBufSz  = whatever;				//	# of bytes w/in buffer.
			decode.inBitPos = decode.lastBitPos;	//	Assuming that you copy
													//	*decode.pInLast to new
		}											//	buffer.

		//	Done decoding a scanline.

	}
//}

    Example of decoding a raw fax input where there is EOLs and RTC
    bits and the number of scanlines is not known a priori, using a
    decoded format of packed bit output:

//{
	decode.format		= ccittDecodeToPacked;
    decode.inBitPos		= 0;
    decode.pInBuf		= whatever;
    decode.inBufSz		= whatever;
    decode.readEolRtc	= true;
    decode.rtcRead		= false;

    while (!decode.rtcRead)
	{
		decode.newLine	= true;
		decode.pOutBuf	= whatever;
		decode.pixCnt	= whatever;		// # of pixels of packed data
										//	*pOutBuf can accomodate.
		while(true)
		{
			if (!CcittDecode31(&decode))
				break;		// ------ the input data is screwed up.

			if (decode.done)
				break;		// ------ done decoding current scanline.

			//	Supply new bits for next call.  Note that there may be
			//	partial bits left undecoded within the last decoded byte.
			//	The next call to decode must start with any undecoded bits.
			/	If you buffer the source bits, then copy all undecoded bits
			//	into the new buffer. The pInLast and lastBitPos fields tell
			//	you the amount left undecoded.

			decode.pInBuf   = pInLast;				//	Or your new buffer.
			decode.inBufSz  = whatever;				//	# of bytes w/in buffer.
			decode.inBitPos = decode.lastBitPos;	//	Assuming that you copy
													//	*decode.pInLast to new
		}											//	buffer.

		//	Done decoding a scanline.

	}
//}
*/
BOOLEAN EXPORTED CcittDecode31 (
   P_DECODE31 pDecode );

#endif	// CCITT_INCLUDED
