/*#######################################################################
#									#
#			     NOTICE:					#
#									#
#    The source code contained in this listing is a proprietary trade	#
#    secret of DIGITAL RESEARCH, INC., Pacific Grove, California and	#
#    is also subject to copyright protection as an unpublished work	#
#    pursuant to Section 104(a) of Title 17 of the United States Code.	#
#    Unauthorized copying, adaptation, distribution, use or display is	#
#    prohitibed by law and may be subject to civil and criminal		#
#    penalties.								#
#									#
#######################################################################*/

/*#######################################################################
#									#
#  Filename:  PICODE02.C						#
#									#
#   Purpose:  Primitive routine library run as application support	#
#	      routines.							#
#									#
#  Routines:  22 primitive library routines:				#
#	      1). main()						#
#	      2). ULONG PFatEntryOP();					#
#	      3). ULONG GetBufCurrentEntry();				#
#	      4). ULONG PFlushFatEntryData();				#
#	      5). ULONG PDataEntryOP();					#
#	      6). ULONG GetMask();					#
#	      7). ULONG RootDirEntryOP();				#
#	      8). ULONG SubDirEntryOP();				#
#	      9). ULONG GetSectorOffset();				#
#	     10). ULONG PDirEntryOP();					#
#	     11). ULONG PConvertDiskAddr();				#
#	     12). ULONG PGetFatSizeInSectors();				#
#	     13). ULONG FatEntryOPInit();				#
#	     14). ULONG PInitFatCache();				#
#	     15). UWORD PGetFatSize();					#
#	     16). void	GetClusterSize();				#
#	     17). void	ReadWriteOP();					#
#	     18). void	PMakeOSBootRec();				#
#	     19). void	GetCacheCurrentEntry();				#
#	     20). void	ReadManip();					#
#	     21). void	WriteManip();					#
#	     22). ULONG PFlushDirEntryData();				#
#									#
#  Maintenance Log							#
#									#
#  Version No.	 Date	Initials   SPR/APR No.	 Change/Comments	#
#  -----------	 ----	---------  -----------	 ---------------	#
#    1.0       09/18/87    YS	      A702	   CREATE.		#
#    1.1       10/02/87    YS	      A702	   MODIFY.		#
#    1.2       11/09/87    WeD	      N/A	 Fixed 12-bit fat probs #
#						 & write w/no modify	#
#    1.3       11/11/87    WeD	      N/A	 Fixed OSBootRec ovrflw #
#    1.4       11/12/87    WeD	      N/A	 Fixed fat flush; added #
#					    dir flush; fixed subdir bug #
#    1.5       11/13/87    WeD	      N/A	 Fixed MakeOSBootRec	#
#    1.6       11/18/87    WeD	      N/A	 12-bit resrvd clusters #
#						 return w/0xF000 set	#
#    1.7       11/23/87    WeD	      N/A	 Fixed 64k _fill_char() #
#    1.8       12/02/87    WeD	      N/A	 MakeOSBootRec again.	#
#######################################################################*/

/*#######################################################################
#									#
#  Included Files:							#
#									#
#######################################################################*/

#include <stdlib.h>
#include "PiGlobal.h"
#include "PiStruct.h"
#include "PiProtos.h"

/*#######################################################################
#									#
#  Static Variables							#
#									#
#######################################################################*/

static UBYTE FatEntryBit;    /* Indicates the FAT entry is 12-bit or 16*/

static UWORD FatSizeInSector;/* Number sectors in a FAT table.	       */

static ULONG SectorSize;     /* The number of bytes in one sector.     */

static UWORD ClusterMask;    /* Used to mask ClusterIndex to get the   */
			     /* FAT table entry offset based on the    */
			     /* sector which is high 8 bits in the     */
			     /* ClusterIndex.			       */

static UWORD ShiftCount;    /* Used to shift ClusterIndex to right to */
			     /* the SectorOffset.		       */

static UWORD *CurrentEntry=0;/* The entry positionin the cache or drive*/
			     /* buffer.				       */

static ULONG CurrentSector   /* Used to store the SectorOffset on the  */
	     = -1;	     /* current buffer.			       */

static LONG TotalByteAlloc;  /* Total number of byte which needs to    */
			     /* be allocated.			       */

static ULONG SectorOffset;   /* Sector offset on the disk.	       */

static UWORD SectorCount;    /* The number of sectors needs to be read */
			     /* from or written to the disk.	       */

static UWORD SectorBoundFlag;/* Indicates if is the odd sector boundry */
			     /* or the even sector bountry.	       */

static UWORD Initialized     /* Indicates if this run is first time    */
	     = UNINIT;
static UBYTE FATChangedFlag  /* Keep track if our FAT buffer is modifed*/
	     = FALSE;
static ULONG ErrorCode;      /* Data will be returned to the caller.   */

UWORD *PCache1 = NULL;	     /* First Cache for FAT manipulation.      */

UWORD *PCache2 = NULL;	     /* Second Cache for FAT manipulation.     */

UWORD *PDriveBuffer;	     /* Buffer for disk FAT manipulation.      */

static ULONG EntryOffset;    /* Entry offset on cluster base in PDirOP */
			     /* and on sector base in PFatOP.	       */

static UWORD LastClusterIndex/* Used to determine if using cache or    */
	     = -1;	     /* reading from the disk.		       */

static UWORD DirClusterIndex /* The current cluster in a dir chain     */
	     = 0;
PDirEntryStruct *Buffer;     /* Pointer of the cache buffer (four      */
			     /* clusters size).			       */

static UWORD BootUp = BOOTUP;/* Indicates if it is bootup running      */

static UBYTE DirChangedFlag  /* Indicate if dir buffer needs writing.  */
	     = FALSE;

static ULONG EntryMask;      /* Max number of the entris in one cluster*/

static ULONG LastSectorOffset = 0;

static UWORD LastSectorCount = 0;

/*#######################################################################
#									#
#  External Functions and location.					#
#									#
#######################################################################*/

/* These functions are located in the file "PICODE01.C" */

extern UWORD PGetFatSectors();
extern UWORD PGetSectorSize();
extern ULONG PReadLogicalSector();
extern ULONG PWriteLogicalSector();
extern UWORD PGetClusterSize();
extern ULONG PConvertClusterToSector();
extern ULONG PRootDirLogicalSector();
extern ULONG PGetRootDirEntries();

extern void  lbase();

/*#######################################################################
#									#
# Function Name: PFATENTRYOP()						#
#									#
#  Return Value: ULONG: 0 for success, else error code.			#
#									#
#   Environment: Run as an application support routine.			#
#									#
#     Arguments: There are five arguments passed through this function. #
#		 1). UWORD OperatID	-- indicates using caches or	#
#					   disk buffer.			#
#		 2). UWORD ClusterIndex -- Used to index the FAT table. #
#		 3). UWORD Read		-- Indicates Read if PREAD_OP,	#
#					   or write if PWRITE_OP.	#
#		 4). UWORD *Value	-- Used to store or get the	#
#					   value in the entry slot.	#
#		 5). PDiskIDStruct *DiskID:Used to identify the disk.	#
#					   If the DiskID is NULL	#
#					   use FAT cache: else disk.	#
#									#
#   Description: Fat Table reading and writing manipulation.		#
#									#
#######################################################################*/

ULONG PFatEntryOP( UWORD OperatID, UWORD ClusterIndex, UWORD Read,
	UWORD *Value, PDiskIDStruct *DiskID )
{
   /*								       */
   /*		      Part One: Initialization			       */
   /*								       */

   if (Initialized == UNINIT)
   {  /* First time running */
      if ((ErrorCode = FatEntryOPInit(DiskID)) != SUCCESS)
      {
	 return(ErrorCode);
      }
      Initialized = INIT;
   }

   /*								       */
   /*		      Part Two: Get Current Entry		       */
   /*								       */
   if (OperatID == BUFFER)
   {  /* Get DriveBuffer CurrentEntry */
      if ((ErrorCode = GetBufCurrentEntry(ClusterIndex, DiskID)) != SUCCESS)
      {
	 return(ErrorCode);
      }
      if (Read == PWRITE_OP)
      {
	 FATChangedFlag = TRUE;
      }
   }
   else /* Get cache CurrentEntry */
   {
      GetCacheCurrentEntry(ClusterIndex);
   }


   /*								       */
   /*		     Part Three: Manipulation			       */
   /*								       */

   if (Read == PREAD_OP)
   {  /* Read manipulation */

      ReadManip(Value);
   }
   else if (Read == PWRITE_OP)
   {  /* Writing Manipulation */

      WriteManip(Value);
   }
   else /* Unknown data */
   {
      return(BUG4);
   }
   return(SUCCESS);
}

/*#######################################################################
#									#
# Function Name: FatEntryOPInit()					#
#									#
#  Return Value: 0 for SUCCESS else Error Code.				#
#									#
#   Environment: Run as an support routine.				#
#									#
#     Arguments: PdiskIDStruct DiskID;					#
#									#
#   Description: Allocates memory for the caches and disk buffer and	#
#		 zero out them.						#
#									#
#######################################################################*/

ULONG FatEntryOPInit( PDiskIDStruct *DiskID )
{

   /* Get FAT table size in sectors. */

   FatSizeInSector = PGetFatSectors();

   /*  Get the bit size of the cluster (12-bit or 16-bit) */

   FatEntryBit = (DiskID -> MediaDesBlock).TypeOfFileSystem;

   /* Get sector size in bytes */

   SectorSize = PGetSectorSize();

   /* Set ClusterMask and ShiftCount */

   if (FatEntryBit == BIT16)
   {
      ClusterMask =BIT16MASK;
      ShiftCount = BIT16SHIFT;
   }
   else
   {  /* FatEntryBit is 12 bit */

      ClusterMask = BIT12MASK;
      ShiftCount = BIT12SHIFT;
   }

   /* Alocates the memory */

   /* Get one or two cache buffers */

   if ((ErrorCode = PInitFatCache()) != SUCCESS)
   {
      return(ErrorCode);
   }

   /* Get a buffer for disk manipulation */

   TotalByteAlloc = SectorSize * NUM_SECTOR_INBUF;

   PDriveBuffer = (UWORD *) malloc((UWORD)TotalByteAlloc);

   TotalByteAlloc = (TotalByteAlloc >> 1) - 1;

   _fill_char( PDriveBuffer, TotalByteAlloc, 0 );

   return(SUCCESS);
}

/*#######################################################################
#									#
# Function Name: PInitFatCache()					#
#									#
#  Return Value: None							#
#									#
#   Environment: Run as an support routine.				#
#									#
#     Arguments: None							#
#									#
#   Description: Allocates the cache memory and zero out it.		#
#									#
#									#
#######################################################################*/

ULONG PInitFatCache( void )
{
   ULONG MemoryParaBlock[3];

   /* Check if alocate one cache or two caches */

   TotalByteAlloc = (ULONG) FatSizeInSector * SectorSize;

   if (TotalByteAlloc > MAX_SEG_SIZE)
   {
      TotalByteAlloc -= MAX_SEG_SIZE;

      MemoryParaBlock[0] = 0;
      MemoryParaBlock[1] = TotalByteAlloc;
      MemoryParaBlock[2] = TotalByteAlloc;

      if ((ErrorCode = s_malloc(1, MemoryParaBlock)) != SUCCESS)
      {
	 return(ErrorCode);
      }

      PCache2 = (UWORD *) MemoryParaBlock[0];

      /* Zero out the cache2; */
      _fill_char( PCache2, TotalByteAlloc, 0 );

      TotalByteAlloc = MAX_SEG_SIZE;
   }

   MemoryParaBlock[0] = 0;
   MemoryParaBlock[1] = TotalByteAlloc;
   MemoryParaBlock[2] = TotalByteAlloc;

   if ((ErrorCode = s_malloc(1, MemoryParaBlock)) != SUCCESS)
   {
      return(ErrorCode);
   }

   PCache1 = (UWORD *) MemoryParaBlock[0];

   /* Zero out the cache1 */
   if( TotalByteAlloc == MAX_SEG_SIZE )
   {
      PCache1[0] = 0;
      _fill_char( PCache1+1, MAX_SEG_SIZE-2, 0 );
   }
   else
      _fill_char( PCache1, TotalByteAlloc, 0 );

   return(SUCCESS);
}

/*#######################################################################
#									#
# Function Name: GetBufCurrentEntry()					#
#									#
#  Return Value: 0 for success, else error code.			#
#									#
#   Environment: Run as an support routine.				#
#									#
#     Arguments: ClusterIndex:	Used to index the FAT table.		#
#		 DiskID      : See above.				#
#									#
#   Description: Get the disk buffer current entry index .		#
#									#
#######################################################################*/

ULONG GetBufCurrentEntry( UWORD ClusterIndex, PDiskIDStruct *DiskID )
{
   /* Get the SectorBoundFlag if FatEntryBit is 12 */

   if (FatEntryBit == BIT12)
   {
      SectorBoundFlag = ClusterIndex & EVEN_ODD;

      /* if the FAT is 12-bit, then the ClusterIndex will be BYTE Index */

      ClusterIndex += (ClusterIndex >> 1);
   }


   /* Get SectorOffset */

   SectorOffset = (ClusterIndex >> ShiftCount) + 1;

   /* Check if use drive buffer or read from the disk */
   if (SectorOffset != CurrentSector)
   {  /* The user requires a FAT entry that is not in the current sectors. */
      /* Check if we need to write the buffer back to the disk, then read  */
      /* another two sectors from the disk.				   */

      if (FATChangedFlag)
      {
	 if( (ErrorCode = PWriteLogicalSector(CurrentSector,NUM_SECTOR_INBUF,
			  (UBYTE *) PDriveBuffer, DiskID)) != SUCCESS)
	 {
	    return(ErrorCode);
	 }
	 FATChangedFlag = FALSE;
      }
      if ((ErrorCode = PReadLogicalSector(SectorOffset, NUM_SECTOR_INBUF,
			(UBYTE *) PDriveBuffer, DiskID)) != SUCCESS)
      {
	 return(ErrorCode);
      }
   }

   CurrentSector = SectorOffset;

   /*  Get entry offset index */

   EntryOffset = ClusterIndex & ClusterMask;

   if (FatEntryBit == BIT16)
   {
      CurrentEntry = PDriveBuffer + EntryOffset;
   }
   else
   {
      CurrentEntry = (UWORD *) ( (UBYTE *) PDriveBuffer + EntryOffset);
   }

   return(SUCCESS);
}

/*#######################################################################
#									#
# Function Name: GetCacheCurrentEntry()					#
#									#
#  Return Value: None							#
#									#
#   Environment: Run as an support routine.				#
#									#
#     Arguments: UWORD ClusterIndex;					#
#									#
#   Description: Get cache current entry index.				#
#									#
#######################################################################*/

void GetCacheCurrentEntry( UWORD ClusterIndex )
{
   if (FatEntryBit == BIT12)
   {
      SectorBoundFlag = ClusterIndex & EVEN_ODD;
      ClusterIndex += ClusterIndex >> 1;
   }

   EntryOffset = ClusterIndex;

   if ( ClusterIndex >= MAX_SEG_SIZE/2 )
   {
      CurrentEntry = PCache2 + (ClusterIndex & NEXTSEG);
   }
   else if (FatEntryBit == BIT12)
   {
      CurrentEntry = (UWORD *) ( (UBYTE *) PCache1 + ClusterIndex);
   }
   else /* BIT16 */
   {
      CurrentEntry = PCache1 + ClusterIndex;
   }
}

/*#######################################################################
#									#
# Function Name: WriteManip()						#
#									#
#  Return Value: 0 for SUCCESS, else error code.			#
#									#
#   Environment: Run as an support routine.				#
#									#
#     Arguments: UWORD *Value;						#
#									#
#   Description: Fat table writing manipulation.			#
#									#
#######################################################################*/

void WriteManip( UWORD *Value )
{
   if (FatEntryBit == BIT16)
   {
      *CurrentEntry = *Value;
   }
   else /* FatEntryBit == BIT12 */
   {
      /* Check if the value needs to be shifted or masked */

      if (SectorBoundFlag == EVEN)
      {  /* needs to be masked */

	 *CurrentEntry = (*CurrentEntry & 0xF000) | (*Value & 0x0FFF);
      }
      else /* SectorEntry == ODD */
      {
	 *CurrentEntry = (*CurrentEntry & 0x000F) | (*Value << 4);
      }
   }
}

/*#######################################################################
#									#
# Function Name: ReadManip()						#
#									#
#  Return Value: None							#
#									#
#   Environment: run as an support routine				#
#									#
#     Arguments: UWORD *Value;						#
#									#
#   Description: Fat table reading manipulation.			#
#									#
#######################################################################*/

void ReadManip( UWORD *Value )
{
   if (FatEntryBit == BIT16)
   {
      *Value = *CurrentEntry;
   }
   else
   {
      if (SectorBoundFlag == EVEN)
      {  /* data needs to be masked */

	 *Value = *CurrentEntry & VALUE_MASK;
      }
      else /* SectorBountFlag == ODD  */
      {
	 *Value = *CurrentEntry >> 4;
      }

      if ( (*Value & 0x0FF0) == 0x0FF0 )
      {
	 *Value |= 0xF000;
      }
   }
}

/*#######################################################################
#									#
# Function Name: PFlushFatEntryData()					#
#									#
#  Return Value: 0 for SUCCESS, else Error Code.			#
#									#
#   Environment: Run as an support routine.				#
#									#
#     Arguments: UWORD FlushIndor, PDiskIDStruct *DiskID		#
#									#
#   Description:							#
#									#
#######################################################################*/

ULONG PFlushFatEntryData( UWORD FlushIndor, PDiskIDStruct *DiskID )
{
   /* Check if to flush the cache or buffer */

   if (FlushIndor == BUFFER)
   {  /* Flush the PDriveBuffer */

      if (FATChangedFlag)
      {
	 if ( (ErrorCode = PWriteLogicalSector(CurrentSector,NUM_SECTOR_INBUF,
			   (UBYTE *) PDriveBuffer, DiskID)) != SUCCESS )
	 {
	    return( ErrorCode );
	 }
	 FATChangedFlag = FALSE;
      }
   }
   else /* FlushIndor == CACHE */
   {
      if (PCache2)
      {  /* Flush two caches */              

	 SectorOffset = MAX_SEG_SIZE / PGetSectorSize();
         SectorCount = FatSizeInSector - SectorOffset;

         if ((ErrorCode = PWriteLogicalSector(SectorOffset+1,
              SectorCount, (UBYTE *) PCache2, DiskID)) != SUCCESS)
         {
            return(ErrorCode);
         }
         SectorCount = SectorOffset;
      }                                                  
      else /* flush one cache */
      {                                                        
         SectorCount = FatSizeInSector;
      }                                                       

      if ((ErrorCode = PWriteLogicalSector( 1L,
           SectorCount, (UBYTE *) PCache1, DiskID)) != SUCCESS)
      {
         return(ErrorCode);
      }
   }
   return(SUCCESS);
}

/*#######################################################################
#									#
# Function Name:PDataEntryOP.						#
#									#
#  Return Value: 0 for success, else error code.			#
#									#
#   Environment: run as an support routine.				#
#									#
#     Arguments: Five arguments:					#
#									#
#		 1). UWORD ClusterIndex ---				#
#		     Logical address of a cluster.			#
#									#
#		 2). UWORD Read ---					#
#		     This argument indicates if the operation is READ	#
#		     or WRITE.						#
#		     PREAD_OP:	0					#
#		     PWRITE_OP: 1					#
#									#
#		 3). UBYTE *Buffer ---					#
#		     32-bit pointer points a buffer where data will be	#
#		     read to or written from.				#
#									#
#		 4). ULONG BytesCount ---				#
#		     Indicates how namy bytes you want to read or write.#
#									#
#		 5). PDiskIDStruct DiskID.				#
#									#
#   Description: Manipulating the Data Area on the diak.		#
#									#
#######################################################################*/

ULONG PDataEntryOP( UWORD ClusterIndex, UWORD Read, UBYTE *Buffer,
	ULONG BytesCount, PDiskIDStruct *DiskID )
{
   /*								       */
   /*		      First Part: Get Sector Offset		       */
   /*								       */

   SectorOffset = PConvertClusterToSector(ClusterIndex);

   /*								       */
   /*	       Second Part: Convert BytesCount to SectorCount	       */
   /*								       */

   SectorCount = (UWORD) (BytesCount / PGetSectorSize());
   /*								       */
   /*		    Third Part: Read or Write Operation		       */
   /*								       */

   if (Read == PREAD_OP)
   {
      ErrorCode = PReadLogicalSector(SectorOffset,SectorCount,Buffer,DiskID);
   }
   else if (Read == PWRITE_OP)
   {
      ErrorCode=PWriteLogicalSector(SectorOffset,SectorCount,Buffer,DiskID);
   }
   else
   {
      ErrorCode = BUG2;
   }
   return(ErrorCode);
}

/*#######################################################################
#									#
# Function Name: PDirEntryOP						#
#									#
#  Return Value: 0 if success, else error code.				#
#									#
#   Environment: Run as  an application support routine.		#
#									#
#     Arguments: Six arguments pass through this function.		#
#		 1). UWORD Root ---					#
#		     This parameter indicates if the directory entry,	#
#		     which will be modified, is Root Directory or not.	#
#		     It has two values:					#
#		     ROOTDIR (0) -- Is Root Directory.			#
#		     SUBDIR  (1) -- Not Root Directory.			#
#									#
#		 2). UWORD ClusterIndex ---				#
#		     Case 1: If the Root indicates the directory is the #
#			     Root Directory: Ignore this parameter.	#
#		     Case 2: If the Root indicates the directory is not #
#			     the Root Directory: This parameter is the	#
#			     Logical Cluster Index.			#
#									#
#		 3). ULONG EntryIndex ---				#
#		     In both cases, this will be the Entry Number that	#
#		     will be modified.					#
#									#
#		 4). UWORD Read ---					#
#		     This parameter indicates if the operation is READ	#
#		     or WRITE.						#
#		     PREAD_OP  0					#
#		     PWRITE_OP 1					#
#									#
#		 5). DirEntryStruct *Ptr_DirEntry ---			#
#		     32-bit DirEntry Pointer which points to the	#
#		     Directory Entry.					#
#									#
#		 6). PDiskIDStruct *DiskID   ---			#
#		     32-bit DiskID pointer. The disk ID field in the	#
#		     structure is used in the special call to do the	#
#		     write.						#
#									#
#   Description: Dir entry reading and writing manipulation.		#
#									#
#######################################################################*/

ULONG PDirEntryOP( UWORD Root, UWORD ClusterIndex, ULONG EntryIndex, UWORD Read,
	PDirEntryStruct *DirEntry, PDiskIDStruct *DiskID )
{
   if (BootUp == BOOTUP)
   {
      /* Allocates the memory for the first time running */

      Buffer = (PDirEntryStruct *)
	       malloc((UWORD) (PGetRootDirEntries() * sizeof(PDirEntryStruct)));
      if( Buffer == NULL )
      {
	 return( FAILURE );
      }
      BootUp = NOT_BOOTUP;
   }
   switch(Root)
   {
      case ROOTDIR:
	 if ((ErrorCode = RootDirEntryOP(EntryIndex, Read, DirEntry, DiskID))
								 != SUCCESS)
	 {
	    return(ErrorCode);
	 }
	 break;
      case SUBDIR:
	 if ((ErrorCode = SubDirEntryOP(ClusterIndex, EntryIndex, Read,
					DirEntry, DiskID)) != SUCCESS)
	 {
	    return(ErrorCode);
	 }
	 break;
   }
}

/*#######################################################################
#									#
# Function Name: RootDirEntryOP						#
#									#
#  Return Value: 0 if success, else error code.				#
#									#
#   Environment: Run as  an application support routine.		#
#									#
#     Arguments: Four arguments pass through this function.		#
#		 1). ULONG EntryIndex ---				#
#									#
#		 2). UWORD Read ---					#
#		     This parameter indicates if the operation is READ	#
#		     or WRITE.						#
#		     PREAD_OP  0					#
#		     PWRITE_OP 1					#
#									#
#		 3). DirEntryStruct *Ptr_DirEntry ---			#
#		     32-bit DirEntry Pointer which points to the	#
#		     Directory Entry.					#
#									#
#		 4). PDiskIDStruct *DiskID   ---			#
#		     32-bit DiskID pointer. The disk ID field in the	#
#		     structure is used in the special call to do the	#
#		     write.						#
#									#
#   Description: Root Directories reading and writing manipulation.	#
#									#
#######################################################################*/

ULONG RootDirEntryOP( ULONG EntryIndex, UWORD Read, PDirEntryStruct *DirEntry,
	PDiskIDStruct *DiskID )
{
   /* Get SectorOffset and SectorCount */

   SectorOffset = PRootDirLogicalSector();
   SectorCount =  (sizeof(PDirEntryStruct) * PGetRootDirEntries())
		  /PGetSectorSize();

   if (LastClusterIndex != ROOTDIR)
   {
      if ( DirChangedFlag )
      {
	 if ((ErrorCode = PWriteLogicalSector(LastSectorOffset, LastSectorCount,
					 (UBYTE *) Buffer, DiskID)) != SUCCESS)
	 {
	    return(ErrorCode);
	 }
	 DirChangedFlag = FALSE;
      }

      /* Read the RootDir Area into the buffer */
      if ((ErrorCode = PReadLogicalSector(SectorOffset, SectorCount,
				      (UBYTE *) Buffer, DiskID)) != SUCCESS)
      {
	 return(ErrorCode);
      }
      LastSectorOffset = SectorOffset;
      LastSectorCount = SectorCount;
      LastClusterIndex = ROOTDIR;
   }

   EntryOffset = EntryIndex;

   /* Read and Write operation */

   ReadWriteOP(Read, DirEntry);

   return(SUCCESS);
}

/*#######################################################################
#									#
# Function Name: SubDirEntryOP						#
#									#
#  Return Value: 0 if success, else error code.				#
#									#
#   Environment: Run as  an application support routine.		#
#									#
#     Arguments: Five arguments pass through this function.		#
#		 1). UWORD ClusterIndex					#
#		 2). ULONG EntryIndex ---				#
#		 3). UWORD Read ---					#
#		     This parameter indicates if the operation is READ	#
#		     or WRITE.						#
#		     PREAD_OP  0					#
#		     PWRITE_OP 1					#
#									#
#		 4). DirEntryStruct *Ptr_DirEntry ---			#
#		     32-bit DirEntry Pointer which points to the	#
#		     Directory Entry.					#
#									#
#		 5). PDiskIDStruct *DiskID   ---			#
#		     32-bit DiskID pointer. The disk ID field in the	#
#		     structure is used in the special call to do the	#
#		     write.						#
#									#
#   Description: Subdirectories reading and writing manipulation.	#
#									#
#									#
#######################################################################*/

ULONG SubDirEntryOP( UWORD ClusterIndex, ULONG EntryIndex, UWORD Read,
	PDirEntryStruct *DirEntry, PDiskIDStruct *DiskID )
{
   EntryMask = GetMask();
   /*								       */
   /*		    First part: Getting the SectorOffset & SectorCount */
   /*								       */

   if ((ErrorCode = GetSectorOffset(ClusterIndex,EntryIndex,DiskID))
							      != SUCCESS)
   {
      return(ErrorCode);
   }
   /*								       */
   /*		      Second part: Check current buffer state	       */
   /*								       */

   if (LastClusterIndex != DirClusterIndex)
   {
      if ( DirChangedFlag )
      {
	 if ((ErrorCode = PWriteLogicalSector(LastSectorOffset, LastSectorCount,
					 (UBYTE *) Buffer, DiskID)) != SUCCESS)
	 {
	    return(ErrorCode);
	 }
	 DirChangedFlag = FALSE;
      }

      if ((ErrorCode = PReadLogicalSector(SectorOffset, SectorCount,
				      (UBYTE *) Buffer, DiskID)) != SUCCESS)
      {
	 return(ErrorCode);
      }
      LastSectorOffset = SectorOffset;
      LastSectorCount = SectorCount;
      LastClusterIndex = DirClusterIndex;
   }

   /*								       */
   /*		   Third Part: Read or write operation		       */
   /*								       */

   ReadWriteOP(Read, DirEntry);

   return(SUCCESS);
}

/*#######################################################################
#									#
# Function Name: GetSectorOffset					#
#									#
#  Return Value: 0 for success, else error code.			#
#									#
#   Environment: Run as an support routine.				#
#									#
#     Arguments: UWORD ClusterIndex;					#
#		 ULONG EntryIndex;					#
#		 PDiskIDStruct *DiskID;					#
#									#
#   Description: Get the sector offset address.				#
#									#
#######################################################################*/

ULONG GetSectorOffset( UWORD ClusterIndex, ULONG EntryIndex,
	PDiskIDStruct *DiskID )
{

   DirClusterIndex = ClusterIndex;

   while ( EntryIndex > EntryMask )
   {
      if ((ErrorCode = PFatEntryOP(BUFFER, DirClusterIndex, PREAD_OP,
				    &DirClusterIndex, DiskID)) !=SUCCESS)
      {
	 return(ErrorCode);
      }
      EntryIndex -= EntryMask+1;
   }

   EntryOffset = EntryIndex;

   SectorOffset = PConvertClusterToSector(DirClusterIndex);
   SectorCount = PGetClusterSize();

   return(SUCCESS);
}

/*#######################################################################
#									#
# Function Name: ReadWriteOP						#
#									#
#  Return Value: NULL.							#
#									#
#   Environment: Run as suppout routine.				#
#									#
#     Arguments: Read, DirEntry.					#
#									#
#   Description: Reading and writing manipulation of the directory.	#
#									#
#######################################################################*/


void ReadWriteOP( UWORD Read, PDirEntryStruct *DirEntry )
{
   if (Read == PREAD_OP)
   {
      _move((UBYTE *) (Buffer + EntryOffset), (UBYTE *) DirEntry,
	     sizeof(PDirEntryStruct));
   }
   else if (Read == PWRITE_OP)
   {
      _move((UBYTE *) DirEntry, (UBYTE *) (Buffer + EntryOffset),
	     sizeof(PDirEntryStruct));
      DirChangedFlag = TRUE;
   }
}

/*#######################################################################
#									#
# Function Name: PFlushDirEntryData()					#
#									#
#  Return Value: 0 for SUCCESS, else Error Code.			#
#									#
#   Environment: Runs as an support routine.				#
#									#
#     Arguments: PDiskIDStruct *DiskID					#
#									#
#   Description: If the current directory entry buffer has been		#
#		 modifed, flush it out to disk.				#
#									#
#######################################################################*/

ULONG PFlushDirEntryData( PDiskIDStruct *DiskID )
{
   if ( DirChangedFlag )
   {
      if ( (ErrorCode = PWriteLogicalSector( LastSectorOffset, LastSectorCount,
					(UBYTE *) Buffer, DiskID )) != SUCCESS )
      {
	 return( ErrorCode );
      }
      DirChangedFlag = FALSE;
   }
   return( SUCCESS );
}

/*#######################################################################
#									#
# Function Name: GetMask.						#
#									#
#  Return Value: Max number of entries in one cluster.			#
#									#
#   Environment: Run as an support routine.				#
#									#
#     Arguments: None							#
#									#
#   Description: Get max number of entries in an cluster.		#
#									#
#######################################################################*/


ULONG GetMask( void )
{
   return(PGetClusterSize() * PGetSectorSize() / ENTRY_SIZE - 1);
}

/*#######################################################################
#									#
# Function Name: PConvertDiskAddr()					#
#									#
#  Return Value: ULONG A physical disk sector offset			#
#									#
#   Environment: Run as an application routine.				#
#									#
#     Arguments: DiskAddress *DiskAddr;					#
#		   -- Disk address which will be converted.		#
#		 UBYTE SectorsPerTrack;					#
#		   -- Number of sectors per track.			#
#		 UBYTE NumberOfHeads;					#
#		   -- Number of heads on the disk.			#
#									#
#   Description: Convert a head, sector, cylinder address to a physical #
#		 disk sector offset address.				#
#									#
#######################################################################*/

ULONG PConvertDiskAddr( DiskAddress *DiskAddr, UBYTE SectorsPerTrack,
	UBYTE NumberOfHeads )
{
   ULONG LSectorOffset;

   LSectorOffset = ((ULONG)DiskAddr->Cylinder * SectorsPerTrack * NumberOfHeads)
		   + ((ULONG)DiskAddr->Head * SectorsPerTrack)
		   + (DiskAddr->Sector - 1);
   return(LSectorOffset);
}

/*#######################################################################
#									#
# Function Name: GetClusterSize()					#
#									#
#  Return Value: NONE							#
#									#
#   Environment: Run as a support routine				#
#									#
#     Arguments: ULONG PartSizeInBytes;					#
#		   --  Partition size in bytes.				#
#		 UWORD *FatBitNum;					#
#		   -- Pointer to the number bits of FAT.		#
#		 UWORD *ClusterSize;					#
#		   -- Pointer to the cluster size in sectors.		#
#									#
#   Description: Get the cluster size and FAT bit number.		#
#									#
#######################################################################*/

void  GetClusterSize( ULONG PartSizeInBytes, UWORD *FatBitNum,
	UWORD *ClusterSize )
{

   if (PartSizeInBytes <= FIXDISK)
   {  /* Smaller than 10 Meg Bytes */

      /* Cluster size is 8 and bits of the FAT is 12 */

      *ClusterSize = EIGHT_SECTORS_PER_CLUSTER;
      *FatBitNum   = BIT12;
   }
   else if (PartSizeInBytes <= MEG_32)
   {  /* 10Meg < PartitionSize <= 32Meg */

      /* Cluster isze is 4 and bits of the FAT is 16 */

      *ClusterSize = FOUR_SECTORS_PER_CLUSTER;
      *FatBitNum   = BIT_16;
   }
   else if (PartSizeInBytes <= MEG_64)
   {  /* 32Meg < PartitionSize <= 64Meg */

      /* Cluster isze is 4 and bits of the FAT is 16 Large */

      *ClusterSize = FOUR_SECTORS_PER_CLUSTER;
      *FatBitNum   = BIT_16L;
   }
   else if (PartSizeInBytes <= MEG_128)
   {  /*  64 MegBytes < PartitionSize <= 128 MegBytes */

      /* Cluster isze is 8 and bits of the FAT is 16 */

      *ClusterSize = EIGHT_SECTORS_PER_CLUSTER;
      *FatBitNum   = BIT_16L;
   }
   else if (PartSizeInBytes <= MEG_512)
   {  /* 128 MegBytes < Partition size <= 512 MegBytes */

      /* Cluster isze is 8 and bits of the FAT is 16 large  */

      *ClusterSize = EIGHT_SECTORS_PER_CLUSTER;
      *FatBitNum   = BIT_16L;
   }
   else /* Partition size > 256 Meg */
   {
      /* Cluster size is 16 and bits of the FAT is 16 large */

      *ClusterSize = SIXTEEN_SECTORS_PER_CLUSTER;
      *FatBitNum   = BIT_16L;
   }
}

/*#######################################################################
#									#
# Function Name: PGetFatSize()						#
#									#
#  Return Value: UWORD FAT Table size in sectors.			#
#									#
#   Environment: Run as a support routine.				#
#									#
#     Arguments: ULONG PartSizeInBytes;					#
#		   -- Partition size in bytes starts from the OSBootRec #
#		      to the end of the partition.			#
#		 UWORD ClusterSize;					#
#		   -- Cluster size in sectors .				#
#		 UWORD FatBitNum;					#
#		   -- Number bits of the FAT.				#
#		 PDiskIDStruct DiskID;					#
#									#
#   Description: Get the FAT table size in sectors.			#
#									#
#######################################################################*/

UWORD PGetFatSize( ULONG PartSizeInSectors, UWORD ClusterSize, UWORD FatBitNum,
	PDiskIDStruct *DiskID )
{
   ULONG DataAreaSizeInSectors;  /* The size of the data area in that  */
				 /* partition in sectors.	       */
   ULONG RootDirSize;		 /* Root directory size in bytes       */
   UWORD OSBootRecSize;		 /* Master boot record size in bytes   */
   UWORD SectorSize;		 /* Sector size in bytes.	       */
   UBYTE NumFat;		 /* The number of the FAT Tables.      */
   ULONG FatSizeInSectors;

   /* Get thesector size in bytes */

   SectorSize	= DiskID->MediaDesBlock.SectorSize;

   /* Get the number of the FATs ( usually is 2) */

   NumFat	= DiskID->MediaDesBlock.NumberFats;

   /* Get the root directory size in sectors */

   RootDirSize	= (DiskID->MediaDesBlock.NumberRootDirEntries * ENTRY_SIZE)
		  / SectorSize;

   /* The OS Boot Record start at the track boundery and size is one sector */

   OSBootRecSize = OSBOOT_REC_SIZE;

   /*									 */
   /* First level calculation --- on the size of both FATs and data area */
   /*									 */

   DataAreaSizeInSectors = PartSizeInSectors - RootDirSize - OSBootRecSize;


   FatSizeInSectors = PGetFatSizeInSectors(DataAreaSizeInSectors, FatBitNum,
					  ClusterSize, SectorSize);


   /* Get the real file data area size (without FAT tables in) */

   DataAreaSizeInSectors = DataAreaSizeInSectors - (FatSizeInSectors * NumFat);

   /*									*/
   /* Second level calculation --- on the size of the data area only	*/
   /*									*/


   FatSizeInSectors = PGetFatSizeInSectors(DataAreaSizeInSectors, FatBitNum,
					  ClusterSize, SectorSize);


   return(FatSizeInSectors);
}

/*#######################################################################
#									#
# Function Name: PGetFatSizeInSectors()					#
#									#
#  Return Value: ULONG -- The number of sectors in the FAT.		#
#									#
#   Environment: Run as a support routine.				#
#									#
#     Arguments: ULONG DataAreaSizeInSectors;				#
#		   -- File data area size in sectors.			#
#		 UWORD FatBitNum;					#
#		   -- Number bits of the FAT .				#
#		 UWORD ClusterSize;					#
#		   -- Cluster size in sectors.				#
#		 ULONG SectorSize;					#
#		   -- Sector size in bytes.				#
#									#
#   Description: Calculates how many sectors in the FAT Table.		#
#									#
#######################################################################*/

ULONG PGetFatSizeInSectors( ULONG DataAreaSizeInSectors, UWORD FatBitNum,
	UWORD ClusterSize, UWORD SectorSize )
{
   ULONG FatSizeInBytes;
   ULONG FatSizeInSectors;
   ULONG FatBackUp;	  /* Used to hold the FatSizeInBytes for   */
			  /* calculation.			   */

   FatSizeInBytes = DataAreaSizeInSectors / ClusterSize;
   /* Save FatSizeInBytes in FatBackUp */

   FatBackUp = FatSizeInBytes;

   /* Check if using 1.5 bytes or 2 bytes to represent a FAT entry */

   if (FatBitNum == BIT12)
   {  /* Three bytes to represent a FAT entry */

      FatBackUp = FatSizeInBytes / 2;

      /* Round the division result */

      if ((FatSizeInBytes & 1) != 0)
      {
	 FatBackUp += 1;
      }
   }

   FatSizeInBytes = FatSizeInBytes + FatBackUp;

   /* Append the head entry in the FAT table */

   if (FatBitNum == BIT12)
   {
      FatSizeInBytes = FatSizeInBytes + 3;
   }
   else /* FatBitNum == BIT_16 */
   {
      FatSizeInBytes = FatSizeInBytes + 4;
   }

   /* Get the number of sectors in this FAT */

   FatSizeInSectors = FatSizeInBytes / SectorSize;

   /* Round the division result */

   if (FatSizeInBytes % SectorSize)
   {
      FatSizeInSectors += 1;
   }

   return(FatSizeInSectors);
}

/*#######################################################################
#									#
# Function Name: PMakeOSBootRec()					#
#									#
#  Return Value: NULL							#
#									#
#   Environment: Run as an application support routine.			#
#									#
#     Arguments: ULONG	PartStartSectorOffset;				#
#		 ULONG	PartEndSectorOffset;				#
#		 ULONG	       SectorOverRide;				#
#		   -- Cluster size in sectors over ride flag.		#
#		      0: using system cluster size.			#
#		      NON-0: user defined cluster size.			#
#		 POSBootRec    *OSBootRec;				#
#		   -- Pointer to the OS Boot Record.			#
#		 PDiskIDStruct *DiskID;					#
#									#
#   Description: This function loads information to the OS Boot Record. #
#									#
#######################################################################*/

void PMakeOSBootRec( ULONG PartStartSectorOffset, ULONG PartEndSectorOffset,
	UWORD SectorOverride, POSBootRec *OSBootRec, PDiskIDStruct *DiskID )
{
   UWORD ClusterSize;		/* How many sectors per cluster.      */
   UWORD FatBitNum;		/* 12 bit FAT or 16 bit FAT.	      */
   ULONG PartitionSizeInBytes;	/* Partition length in bytes.	      */
   ULONG PartitionSizeInSectors;/* Partition length in sectors	      */


   PartitionSizeInSectors = PartEndSectorOffset - PartStartSectorOffset + 1;

   PartitionSizeInBytes = PartitionSizeInSectors
			  * DiskID->MediaDesBlock.SectorSize;

   /* Load the Jump Code */

   _move( lbase, OSBootRec, sizeof(POSBootRec) );

   /* Load the bytes per sector */

   OSBootRec->BPB.BytesPerSector = DiskID->MediaDesBlock.SectorSize;

   /* Load the number of sectors per cluster */

   if (SectorOverride == DEFAULT_CLUSTER_SIZE)
   {  /* System defined cluster size */

      GetClusterSize(PartitionSizeInBytes, &FatBitNum, &ClusterSize);
   }
   else /* User defined cluster size */
   {
      ClusterSize = SectorOverride;
      FatBitNum   = DiskID->MediaDesBlock.TypeOfFileSystem;
   }
   OSBootRec->BPB.SectorsPerCluster = ClusterSize;

   /* Load the number of Sectors occupied by OS */

   OSBootRec->BPB.OSResSectors = SECTOR_ONE;

   /* Load the number of the FAT tables in the partition */

   OSBootRec->BPB.NumFatsInPart = DiskID->MediaDesBlock.NumberFats;

   /* Load the number of root directory entries */

   OSBootRec->BPB.NumRootDirEntries=DiskID->MediaDesBlock.NumberRootDirEntries;

   /* Load the the partition size in sectors */

   if (PartitionSizeInSectors <= SEGMENT_SIZE)
   {  /* Partition size is larger than a sgement size (0xFFFF) */

      OSBootRec->BPB.NumSectorsInPart= PartitionSizeInSectors;
   }
   else /* Partition size in sectors is larger than one segment (0xFFFF)*/
   {
      OSBootRec->BPB.ExtenNumSecInPart= PartitionSizeInSectors;
      OSBootRec->BPB.NumSectorsInPart= 0;
   }

   /* Load the Media Descriptor */

   OSBootRec->BPB.MediaDescripByte = DiskID->MediaDesBlock.FatID;

   /* Load the Number of sectors per FAT */

   OSBootRec->BPB.FatSizeInSectors =
	      PGetFatSize(PartitionSizeInSectors, ClusterSize, FatBitNum, DiskID);

   /* Load the number of sectors per track */

   OSBootRec->BPB.SectorsPerTrack = DiskID->MediaDesBlock.SectorsPerTrack;

   /* Load the number of heads on the disk */

   OSBootRec->BPB.NumHeadsOnDisk = DiskID->MediaDesBlock.NumberOfHeads;

   /* Load the number of sectors preceding the partition ( the number */
   /* of sectors before the current OS Boot Record ).		      */

   OSBootRec->BPB.NumPrecedSectors = PartStartSectorOffset;

   /* Load the the first physical sector address of the data area */


   OSBootRec->BPB.FirstDataSector = PartStartSectorOffset +
	   ( SECTOR_ONE		   /* OSBootRecord uses one sector*/
	     + (OSBootRec->BPB.NumFatsInPart * OSBootRec->BPB.FatSizeInSectors)
	     + ((OSBootRec->BPB.NumRootDirEntries * ENTRY_SIZE)
		/ OSBootRec->BPB.BytesPerSector));

   /* Load the signature */

   OSBootRec->Signature = SIGNATURE;

}

/*#######################################################################
#									#
#			     NOTICE:					#
#									#
#    The source code contained in this listing is a proprietary trade	#
#    secret of DIGITAL RESEARCH, INC., Pacific Grove, California and	#
#    is also subject to copyright protection as an unpublished work	#
#    pursuant to Section 104(a) of Title 17 of the United States Code.	#
#    Unauthorized copying, adaptation, distribution, use or display is	#
#    prohitibed by law and may be subject to civil and criminal		#
#    penalties.								#
#									#
#######################################################################*/

