/*#######################################################################
#                                                                       #
#                            NOTICE:                                    #
#                                                                       #
#    The source code contained in this listing is a proprietary trade   #
#    secret of DIGITAL RESEARCH, INC., Pacific Grove, California and    #
#    is also subject to copyright protection as an unpublished work     #
#    pursuant to Section 104(a) of Title 17 of the United States Code.  #
#    Unauthorized copying, adaptation, distribution, use or display is  #
#    prohitibed by law and may be subject to civil and criminal         #
#    penalties.                                                         #
#                                                                       #
#######################################################################*/

/*#######################################################################
#                                                                       #
#  Filename: FDISK.C                                                    # 
#                                                                       # 
#   Purpose: A set of routines which performance the FDISK function.    #
#                                                                       #
#  Routines:                                                            # 
#      1). void main();                                                 #  
#      2). ULONG LevelOneOption();                                      #
#      3). ULONG CreatePartition();                                     #
#      4). ULONG DeletePartition();                                     #
#      5). ULONG DisplayPartition();                                    #
#      6). ULONG ActivatePartition();                                   #
#      7). void  ReverseConvertDiskAddr();                              #
#      8). ULONG CreateAll32Meg();                                      #
#      9). ULONG CreateUserSizePart();                                  #
#     10). ULONG CreateWholeDiskPart();                                 #
#     11). ULONG PartitionCreation();                                   #
#     12). ULONG FindDiskGaps();                                        #
#     13). ULONG BuildGapEntry();                                       #
#     14). void  LoadAddrFromMBR();                                     #
#     15). ULONG CheckDisk();                                           #
#     16). ULONG GetSectorCount();                                      #
#     17). ULONG GetTrackCount();                                       #
#     18). ULONG ConvertCylinderSizeToMeg();                            #
#     19). ULONG ConvertDiskAddrFromMBR();                              #
#     20). void  ZeroMBR();                                             #
#     21). void  AssignAddr();                                          #
#     22). void  GetFatRootDirAreaSize();                               #
#     23). void  ModifyMBR();                                           #
#     24). void  LoadAddr();                                            #
#     25). void  GetFatClusterSize();                                   #
#     26). ULONG MakeExtOSBootRec();                                    #
#     27). UWORD MBRBitOwnerType();                                     #
#     28). UWORD MBRNonOwnerType();                                     #
#     29). UWORD MBRExtOwnerType();                                     #
#     30). void  ZeroMBRSlot();                                         #
#     31). ULONG FDiskScreen();                                         #
#     32). void  Screen1();                                             #
#     33). void  Screen2();                                             #
#     34). void  Screen31();                                            #
#     35). void  Screen32();                                            #
#     36). void  Screen40();                                            #
#     37). ULONG Screen41();                                            #
#     38). void  Screen42();                                            #
#     39). void  Screen43();                                            #
#     40). void  Screen44();                                            #
#     41). ULONG Screen51();                                            #
#     42). void  Screen52();                                            #
#     43). void  Screen53();                                            #
#     44). void  Screen54();                                            #
#     45). void  Screen55();                                            #
#     46). ULONG Screen61();                                            #
#     47). void  Screen62();                                            #
#     48). void  Screen63();                                            #
#     49). void  Screen64();                                            #
#     50). void  Screen65();                                            #
#     51). ULONG Screen7();                                             #
#     52). ULONG ViewDisplay();                                         #
#     53). UWORD CylinderAddrConvert();                                 #
#     54). void  InitialLine();                                         #
#     55). void  IncreaseLine();                                        #
#     56). int   MBRUnkOwnerType();                                     #
#                                                                       #  
#  Maintenance Log                                                      #
#                                                                       #
#  Version No.   Date   Initials   SPR/APR No.   Change/Comments        #
#  -----------   ----   ---------  -----------   ---------------        #
#    1.0       09/30/87    YS         A702         CREATE.              # 
#    1.1       10/01/87    YS         A702         CREATE.              #  
#    1.2       10/02/87    YS                      CREATE.              #
#    1.3       10/04/87    YS                      CREATE.              #
#    1.4       10/05/87    YS                      CREATE.              #
#    1.5       10/06/87    YS                      CREATE.              #
#    1.6       10/07/87    YS                      CREATE.              #
#    1.7       10/08/87    YS                      CREATE.              #
#    1.8       10/09/87    YS                      MODIFY.              #
#    1.9       10/11/87    YS                      MODIFY.              #
#    2.0       10/12/87    YS                      MODIFY.              #
#    2.1       10/28/87    YS                      CORRECT.             #
#    2.2       10/29/87    YS                      CORRECT.             #
#    2.3       10/30/87    YS                      CORRECT.             #
#    2.4       11/09/87    YS                      LINKING.             #
#    2.5       11/10/87    YS                      COMBINING FILES      #
#    2.6        1/25/88    jmb                     Put text in FDMSGS.C #
#    2.7       02/20/88    ldt                     IBM switch enabled.  #
#                                                                       #
#######################################################################*/

/*#######################################################################
#                                                                       #
#  Included Files:                                                      #
#                                                                       #
#######################################################################*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "portab.h"
#include "PIStruct.h"
#include "PIGlobal.h"
#include "piprotos.h"

/*#######################################################################
#                                                                       #
#  Static Variables                                                     #
#                                                                       #
#######################################################################*/

static ULONG ErrorCode;          /* used to hold the error code returned from */
                          /* the calling function.                     */
static UWORD Drive1PartNum;
static UBYTE PhyDrvName1[LEN_DRIVENAME];
static UBYTE PhyDrvName2[LEN_DRIVENAME];
static WORD  TotalCylinder;

static ULONG RtnCode = 1; /* Return code from PWritePhysicalSector()   */
                          /* which is used to check if any change on   */
                          /* the disk. 	If the value is not equal to 1 */
                          /* diskplay the message to the user.         */

static POption FDArgOpt0 = { ' ', FALSE, -1L };
static POption FDArgOpt1 = { 'C', FALSE, -1L };
static POption FDArgOpt2 = { 'A', FALSE, -1L };
static POption FDArgOpt3 = { 'D', FALSE, -1L };
static POption FDArgOpt4 = { 'I', FALSE, -1L };
static POption FDArgOpt5 = { 'N', FALSE, -1L };
static POption FDArgOpt6 = { 'S', FALSE, -1L };
static POption FDArgOpt7 = { 'E', FALSE, -1L };
static POption FDArgOpt8 = { '?', FALSE, -1L };
static POption FDArgOpt9 = { 'W', FALSE, -1L };


static PCommandLineOptions FDParseTable = 
       {
#if IBM_FDISK       
          10,
#else   
          9,
#endif
          &FDArgOpt0, &FDArgOpt1, &FDArgOpt2,
          &FDArgOpt3, &FDArgOpt4, &FDArgOpt5,
          &FDArgOpt6, &FDArgOpt7, &FDArgOpt8,
#if IBM_FDISK
          &FDArgOpt9,
#endif
       };

MasterBootRec *MBR, *LastMBR, *FirstMBR; 

/*#######################################################################
#                                                                       #
#  External Functions and location.                                     #
#                                                                       #
#######################################################################*/

/**************************************************************/
/* Following extern functions are allocated in the PICODE01.C */
/**************************************************************/

extern ULONG PDriveLock();
extern ULONG PUnlockDrive();
extern ULONG PGetDefaultDrive();
extern ULONG PCheckDiskArea();
extern ULONG PReadPhysicalSector();
extern ULONG PWritePhysicalSector();

/*******************************************************/
/* Following functions are allocated in the PICODE02.C */
/*******************************************************/

extern ULONG PConvertDiskAddr();
extern UWORD PGetFatSize();
extern void  GetClusterSize();
extern void  PMakeOSBootRec();

/*****************************************************/
/* Following variable is allocated in the PIGLOBAL.C */
/*****************************************************/

extern POSBootRec GOSBootRec;

/******************************************************/
/* Following function is allocated in the  HDBOOT.A86 */
/******************************************************/

extern void  hdboot(); /* Master Boot Record Code */

/*******************************************************/
/* Following functions are allocates in the PICODE03.C */
/*******************************************************/

extern int   PDisplay();
extern void  PClearDisplay();
extern void  PEraseDisplayToEnd();
extern WORD  PConfirm();
extern void  PClearRectangle();
extern void  PEraseLineToEnd();
extern int   PGetNum();
extern int   PGetString();
extern ULONG PParseOptions();
extern ULONG PInitScreenMenu();
extern ULONG PUnInitScreenMenu();

/**************************************************************/
/* Following extern variables are allocated in the FDSCREEN.C */
/**************************************************************/

extern PDisplayObject PartitionNumObj;
extern PDisplayObject DataObj;
extern PDisplayObject PartActiveObj;
extern PDisplayObject PartInactiveObj;
extern PDisplayObject PartExtendedObj;
extern PDisplayObject CylinderSizeObj;
extern PDisplayObject DOSPartObj;
extern PDisplayObject FlexOSPartObj;
extern PDisplayObject BannerObj;
extern PDisplayObject ChosenDiskObj;
extern PDisplayObject OptionListObj;
extern PDisplayObject ChooseObj;
extern PDisplayObject ChoiceOneObj;
extern PDisplayObject ChoiceTwoObj;
extern PDisplayObject ChoiceThreeObj;
extern PDisplayObject ChoiceFourObj;
extern PDisplayObject ChoiceFiveObj;
extern PDisplayObject EnterChoiceObj;
extern PDisplayObject EscapeFdiskObj;
extern PDisplayObject CreatePartitionObj;
extern PDisplayObject NoSpaceObj;
extern PDisplayObject CausePartitionObj;
extern PDisplayObject SelectOneObj;
extern PDisplayObject SelectTwoObj;
extern PDisplayObject WarningObj;
extern PDisplayObject EnterSelectionObj;
extern PDisplayObject EscapeToMainMenuObj;
extern PDisplayObject CreateAllObj;
extern PDisplayObject SpaceLeftObj;
extern PDisplayObject MakePartitionsObj;
extern PDisplayObject RemainingObj;
extern PDisplayObject EscapeToCreateMenuObj;
extern PDisplayObject CurrentPartitionsObj;
extern PDisplayObject HeadingObj;
extern PDisplayObject UnderLineObj;
extern PDisplayObject ViewPartitionsObj;
extern PDisplayObject DefaultCylinderObj;
extern PDisplayObject EnterCylinderObj;
extern PDisplayObject DefaultSizeObj;
extern PDisplayObject EnterSizeObj;
extern PDisplayObject MegaSizeObj;
extern PDisplayObject MakePartitionObj;
extern PDisplayObject ChangePartitionObj;
extern PDisplayObject EnterNewPartitionObj;
extern PDisplayObject ActivatePartitionObj;
extern PDisplayObject DeletePartitionObj;
extern PDisplayObject DeleteNumberObj;
extern PDisplayObject DeleteVerifyObj;
extern PDisplayObject DisplayPartitionObj;
extern PDisplayObject No32GapObj;
extern PDisplayObject IllegalDataObj;
extern PDisplayObject PartIsActiveObj;
extern PDisplayObject DeleteOrderErrorObj;
extern PDisplayObject EscapePosObj;
extern PDisplayObject AllPartDeletedObj;
extern PDisplayObject PrimePartNotExistObj;
extern PDisplayObject MegDataObj;
extern PDisplayObject DiskUnPartObj;
extern PDisplayObject UnknownPartObj;
extern PDisplayObject DriverErrorObj;

/*******************************************************************/
/* Following extern strings are defined and allocated in FDMSGS.C */
/*******************************************************************/
extern BYTE fdmsg01[]; /*\nIllegal Drive Name -- The Removable Drive Can Not Been Partitioned.\n*/
extern BYTE fdmsg02[]; /*Changes Have Been Made. Reboot system with CTRL-ALT-DEL\n*/
extern BYTE fdmsg03[]; /*..........\n*/
extern BYTE fdmsg04[]; /*\nIllegal O Option.\n*/
extern BYTE fdmsg05[]; /*\nIllegal Option.\n*/
extern BYTE fdmsg06[]; /*\nThe Specified Hard Disk Is Unusable\n*/
extern BYTE fdmsg07[]; /*\nThe First Master Boot Record Is Not Readable.\n*/
extern BYTE fdmsg08[]; /*\nThere Is No Space For Creating Partition.\n*/
extern BYTE fdmsg09[]; /*\nIllegal FDISK Options.\n*/
extern BYTE fdmsg10[]; /*\nThere Is Not Enough Space For Creating 32 Meg Partition.\n*/
extern BYTE fdmsg11[]; /*\nActive Partition Already Exists.\n*/
extern BYTE fdmsg12[]; /*\nIllegal Cylinder Starting Address.\n*/
extern BYTE fdmsg13[]; /*\nIllegal Cylinder Size.\n*/
extern BYTE fdmsg14[]; /*\nIllegal Option Data.\n*/
extern BYTE fdmsg15[]; /*\nThe Specified Partition Can Not Found.\n*/
extern BYTE fdmsg16[]; /*\nNot Enough Working Memory.\n*/
extern BYTE fdmsg17[]; /*\nMust Delete Logical Partitions Before First Extended Partition\n*/
extern BYTE fdmsg18[]; /*\nOnly DOS or FlexOS Partition Can Been Activated.\n*/
extern BYTE fdmsg19[]; /*\nPrimary Partition Does Not Exist.\n*/
extern BYTE fdmsg20[]; /*\nMaster Boot Record Is Fully Used.\n*/
extern BYTE fdmsg21[]; /*\nDisk Is Unpartitioned \n*/
extern BYTE fdmsg22[]; /*\nThis FDISK Only Can Delete DOS Or FlexOS Partition.\n*/
extern BYTE fdmsg23[]; /*\nDriver Reading Or Writing Error == %lx, ErrorCode*/

/* on line help messages */
extern BYTE hpmsg[];

/*#######################################################################
#                                                                       #
# Function Name: main()                                                 # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as an application.                                 # 
#                                                                       #  
#     Arguments: argc -- Total number of arguments                      #
#                argv -- Pointer to argument string                     #
#                                                                       #
#   Description: This routine performs the FDISK utility which includes #
#                the following four functions:                          # 
#                1). Create partition.                                  #
#                2). Delete partition.                                  #
#                3). Activate partition.                                #
#                4). Display partition.                                 #
#                                                                       #
#######################################################################*/

void main( WORD  argc, UBYTE *argv[] )
{
   PDiskIDStruct DiskID;          /* Disk ID table                     */
   UWORD Menu1Driven;             /* Indicates if using screen 1 or not*/
   UWORD SectorSize;
   ULONG SectorOffset;
   UWORD SectorCount;
   WORD  BitIndex, ExtIndex, Index;

   /*                                                                  */ 
   /*                Part One -- Parsing input string                  */
   /*                                                                  */

   PInitScreenMenu();

   if ((ErrorCode = PParseOptions(argc, argv, &FDParseTable)) != SUCCESS)
   {
      ErrorHandler(BADOPTION);
      OnLineHelp();
      PUnInitScreenMenu();
      exit(1);
   }
   
   /*                                                                  */ 
   /*              Part Two ---- Check the option line                 */ 
   /*                                                                  */ 

   /* Check if user wants to jump over the screen 1 */

   if (
       (FDParseTable.ArgpPtr[C_OPTION])->Found ||  /* C option searched */
       (FDParseTable.ArgpPtr[A_OPTION])->Found ||  /* A option searched */
       (FDParseTable.ArgpPtr[D_OPTION])->Found ||  /* D option searched */
       (FDParseTable.ArgpPtr[I_OPTION])->Found     /* I option searched */
      )
   {  /* Option C, D, A, I in the command line */

      /* Turn off the Menu. */

      Menu1Driven = OFF;
   }
   else if (FDParseTable.ArgpPtr[H_OPTION]->Found)
   {
      OnLineHelp();
      PUnInitScreenMenu();
      exit(1);
   }
   else if (argc > 1 )
   {
      ErrorHandler(BADOPTION);
      OnLineHelp();
      PUnInitScreenMenu();
      exit(1);
   }
   else /* there are not option in the command line */
   {
      Menu1Driven = ON;
   }

   if (Menu1Driven == OFF)
   {   
      if (FDParseTable.ArgpPtr[DRIVE_NAME]->Found)
      {  /* Using user's drive name */

         strcpy(&(DiskID.DriveName), (char *) (FDParseTable.ArgpPtr[0]->Result));
      }
      else /* User didn't enter the drive name */
      {
         if ((ErrorCode = PGetDefaultDrive(DiskID.DriveName)) != SUCCESS)
         {
            ErrorHandler(ErrorCode);
            PUnInitScreenMenu();
            exit(1);
         }
      }
   }
   else
   {
      DiskID.DriveName[0] = 'h';
      DiskID.DriveName[1] = 'd';
      DiskID.DriveName[2] = '0';
      DiskID.DriveName[3] = ':';
      DiskID.DriveName[4] = '\0';
   }

   /*                                                                  */
   /*               Part Three -- Lock the drive                       */ 
   /*                                                                  */ 

   if ((ErrorCode = PDriveLock(&DiskID, PPHYSICAL_LOCK, PREAD_WRITE)) 
                                                          != SUCCESS)
   {
      ErrorHandler(ErrorCode);
      PUnInitScreenMenu();
      exit(1);
   }

   if (Menu1Driven == OFF)
   {
      /* Check if user attends to partition floppy diskette */

      if (DiskID.DriveType != HARD_DISK_TYPE)
      {
         printf(fdmsg01);		/* Illegal Drive Name -- The Removable Drive */
         PUnlockDrive();		/* Can Not Been Partitioned. */
         PUnInitScreenMenu();
         exit(1);
      }
   }

   if (DiskID.MediaDesBlock.PhysicalDriveNumber == 0)
   {
      DiskID.DriveName[0] = '1';
   }
   else 
   {
      DiskID.DriveName[0] = '2';
   }

   DiskID.DriveName[1] = ':';
   DiskID.DriveName[2] = '\0';

   
   /* allocating the memory for MBR */
 
   SectorSize = DiskID.MediaDesBlock.SectorSize;

   if ((MBR = (MasterBootRec *) malloc(SectorSize)) == NULL      ||
       (LastMBR = (MasterBootRec *) malloc(SectorSize)) == NULL  ||
       (FirstMBR = (MasterBootRec *) malloc(SectorSize)) == NULL )
   {
      ErrorHandler((ULONG) NOTENOUGHMEMORY);
      PUnlockDrive();
      PUnInitScreenMenu();
      exit(1);
            
   }

   DiskID.MediaDesBlock.NumberFats = NUMBER_FATS;
   DiskID.MediaDesBlock.NumberRootDirEntries = NUMBER_ROOTDIR_ENTRY;
   DiskID.MediaDesBlock.NumberOfCylinders -= 1;
   TotalCylinder = DiskID.MediaDesBlock.NumberOfCylinders;

   /*                                                                  */ 
   /*           Part Four -- Option switch                             */ 
   /*                                                                  */ 

   /* Get the number of partitions on the first physical driver */

   if (Menu1Driven == ON)
   {
      Drive1PartNum = 0;
      SectorOffset = SECTOR_ZERO;
      SectorCount  = SECTOR_ONE;

      if ((ErrorCode = PReadPhysicalSector(SectorOffset, SectorCount, 
                                 (BYTE *) MBR, &DiskID)) != SUCCESS)
      {
         ErrorHandler(ErrorCode);
         PUnlockDrive();
         PUnInitScreenMenu();
         exit(1);
      }

      BitIndex = MBRBitOwnerType(MBR);
      ExtIndex = MBRExtOwnerType(MBR);

      if ((BitIndex != NOT_FOUND) || (ExtIndex != NOT_FOUND))
      {
         if (BitIndex != NOT_FOUND)
         {
            for(Index = 0; Index < 3; Index++)
            {
               if ((MBR->PartTable[Index].OwnerType == BIT12)  ||
                   (MBR->PartTable[Index].OwnerType == BIT_16) ||
                   (MBR->PartTable[Index].OwnerType == BIT_16L))
               {
                  Drive1PartNum += 1;
               }
            }
         }
      
         while (ExtIndex != NOT_FOUND)
         {
            SectorOffset = ConvertDiskAddrFromMBR(
                        &(MBR->PartTable[ExtIndex].StartAddr), &DiskID);

            if ((ErrorCode = PReadPhysicalSector(SectorOffset, SectorCount,
                                      (BYTE *) MBR, &DiskID)) != SUCCESS)
            {
               ErrorHandler(ErrorCode);
               PUnlockDrive();
               PUnInitScreenMenu();
               exit(1);
            }

            BitIndex = MBRBitOwnerType(MBR);
            ExtIndex = MBRExtOwnerType(MBR);
      
            if (BitIndex != NOT_FOUND)
            {
               Drive1PartNum += 1;
            }
         }
      }

      /* Get the first physical driver name */

      PhyDrvName1[0] = 'h';
      PhyDrvName1[1] = 'd';
      PhyDrvName1[2] = '0';
      PhyDrvName1[3] = ':';
      PhyDrvName1[4] = '\0';

      /* Get the second physical driver name */
   
      PhyDrvName2[0] = 'h';
      PhyDrvName2[1] = 'd';

      if (Drive1PartNum == 0)
      {  /* No partition in the first physical driver */

         PhyDrvName2[2] = '1';
         PhyDrvName2[3] = ':';
         PhyDrvName2[4] = '\0';
      }
      else if ((Drive1PartNum) > 0 && (Drive1PartNum < 10))
      {
         PhyDrvName2[2] = 0x30 + Drive1PartNum;
         PhyDrvName2[3] = ':';
         PhyDrvName2[4] = '\0';
      }
      else if ((Drive1PartNum >= 10) && (Drive1PartNum < 20))
      {
         PhyDrvName2[2] = '1';
         PhyDrvName2[3] = 0x30 + (Drive1PartNum - 10);
         PhyDrvName2[4] = ':';
         PhyDrvName2[5] = '\0';
      }
      else
      {
         PhyDrvName2[2] = '2';
         PhyDrvName2[3] = 0x30 + (Drive1PartNum - 20);
         PhyDrvName2[4] = ':';
         PhyDrvName2[5] = '\0';
      }
   }
   
   if ((ErrorCode = LevelOneOption(Menu1Driven,&DiskID,&FDParseTable)) 
                                                              != SUCCESS)
   {
      ErrorHandler(ErrorCode);
      PUnlockDrive();
      PUnInitScreenMenu();
      exit(1);
   }

   /*                                                                  */ 
   /*              Part Five -- Unlock the drive                       */ 
   /*                                                                  */ 

   if ((ErrorCode = PUnlockDrive()) != SUCCESS)
   {
      ErrorHandler(ErrorCode);
      PUnInitScreenMenu();
      exit(1);
   }
      
   PUnInitScreenMenu();
   
   if (ErrorCode == SUCCESS)
   {
      PClearDisplay();

      if (RtnCode != 1)
      {  /* The disk has been changed */

         printf(fdmsg02); 	/*"Changes Have Been Made. Reboot system with CTRL-ALT-DEL\n"*/
         printf(fdmsg03);	/*"..........\n"*/

         while(1)
         {
         }
      }
   }
}

/*#######################################################################
#                                                                       #
# Function Name:  LevelOneOption()                                      # 
#                                                                       # 
#  Return Value:  0 for SUCCESS else Error Code.                        #
#                                                                       #
#   Environment:  Run as an support routine.                            # 
#                                                                       #  
#     Arguments:  UWORD MenuDriven;                                     #
#                       Indiscates using screen driven or command line  #
#                       options.                                        #
#                 PDIskIDStruct *DiskID;                                #
#                 PCommandLineOptions *ParseTable;                      #
#                                                                       #
#   Description:  Get option from the screen or the command line.       # 
#                 Then go to the switch(Option)_case block.             # 
#                                                                       # 
#######################################################################*/


ULONG LevelOneOption(UWORD MenuDriven, PDiskIDStruct *DiskID,
	PCommandLineOptions *ParseTable)
{
   int   Option;                  /* Used to hold the menu option      */
   UWORD Menu2Driven;             /* Using creation menu 2             */
   UBYTE DriveName[LEN_DRIVENAME];

   do
   {
      if (MenuDriven == ON)
      {  
         /* Display screen and get the option from the screen. */

         FDiskScreen(1, NULLPTR, DiskID->DriveName, NULLPTR, DiskID);
         PDisplay( &EnterChoiceObj );
         Option = PGetNum(1, 5, 4);
         if (Option == FAILURE)
         {
            Option = DONE;
         }
         while (Option == NEXT)
         {
            if ((ErrorCode = PUnlockDrive()) != SUCCESS)
            {
               return(ErrorCode);
            }

            strcpy(DriveName, DiskID->DriveName);

            if (DriveName[0] == '1')
            {  /* attend to change to the first physical driver */

               strcpy(DiskID->DriveName, PhyDrvName2);
            }
            else
            {  /* Attend to chande to the second physical driver */
               strcpy(DiskID->DriveName, PhyDrvName1);
            }

            if ((ErrorCode = PDriveLock(DiskID, PPHYSICAL_LOCK, PREAD_WRITE)) 
                                                                  != SUCCESS)
            {
               /* the second hard disk does not exist */
               /* relock the original physical disk.  */

               if (DriveName[0] == '1')
               {  
                  strcpy(DiskID->DriveName, PhyDrvName1);
               }
               else
               {  
                  strcpy(DiskID->DriveName, PhyDrvName2);
               }
               
               if ((ErrorCode = PDriveLock(DiskID, PPHYSICAL_LOCK, PREAD_WRITE)) 
                                                                    != SUCCESS)
               {
                  return(ErrorCode);
               }
               strcpy(DiskID->DriveName, DriveName);
            }
            else /* Drive lock successfully */
            {
               strcpy(DiskID->DriveName, DriveName);

               if (DiskID->MediaDesBlock.PhysicalDriveNumber == 0)
               {
                  DiskID->DriveName[0] = '1';
               }
               else
               {
                  DiskID->DriveName[0] = '2';
               }
            }

            DiskID->MediaDesBlock.NumberOfCylinders -= 1;
            DiskID->MediaDesBlock.NumberFats = NUMBER_FATS;
            DiskID->MediaDesBlock.NumberRootDirEntries = NUMBER_ROOTDIR_ENTRY;
            TotalCylinder = DiskID->MediaDesBlock.NumberOfCylinders;

            PDisplay( &ChosenDiskObj, DiskID->DriveName );
            PDisplay( &EnterChoiceObj );
            Option = PGetNum(1, 5,4);
            if (Option == FAILURE)
            {
               Option = DONE;
            }
         }
      }
      else /* MenuDriven == OFF */
      {
         /* get the option from the command line */

         if ( (ParseTable->ArgpPtr[C_OPTION])->Found )
         {
            Option = CREATE;
         }
         else if ( (ParseTable->ArgpPtr[A_OPTION])->Found )
         {
            Option = ACTIVATE;
         }
         else if ( (ParseTable->ArgpPtr[D_OPTION])->Found )
         {
            Option = DELETE;
         }
         else  /* ( (ParseTable->ArgpPtr[I_OPTION])->Found ) */
         {
            Option = DISPLAY;
         }
      }

      Menu2Driven = MenuDriven;
      
      switch(Option)
      {
         case CREATE:
            if ( ParseTable->ArgpPtr[C_OPTION]->Found)
            {
	       if (!ParseTable->ArgpPtr[N_OPTION]->Found &&
	           !ParseTable->ArgpPtr[S_OPTION]->Found &&
	           !ParseTable->ArgpPtr[E_OPTION]->Found &&
                   !ParseTable->ArgpPtr[W_OPTION]->Found )
                {  /* No N, S, E and W options are found, then turn on the screen */

                   Menu2Driven = ON;
                }
                else if((ParseTable->ArgpPtr[N_OPTION]->Found  &&
                         ParseTable->ArgpPtr[S_OPTION]->Found) ||
                         ParseTable->ArgpPtr[E_OPTION]->Found  ||
                         ParseTable->ArgpPtr[W_OPTION]->Found  )
                {  /* N, S, E and W options in the command line */

                   Menu2Driven = OFF;
                }
                else /* Bad options combination */
                {
                   return(BADOPTION);
                }
            } 
            while ((ErrorCode = CreatePartition(Menu2Driven, DiskID, ParseTable))
	                                                           != SUCCESS)
            {
               if (Menu2Driven == ON)
               {
                  PDisplay( &DriverErrorObj );
                  PGetString( NULLPTR, 0, 0);
               }
               else 
               {
                  return(ErrorCode);
               }
            }
            break;
         case DELETE:
            if (ParseTable->ArgpPtr[D_OPTION]->Found)
            {		
	       if( !ParseTable->ArgpPtr[N_OPTION]->Found ) 
               {
	           Menu2Driven = ON;
	       }
               else if ( ParseTable->ArgpPtr[N_OPTION]->Found ) 
               {
	          Menu2Driven = OFF;
               }
	       else
	       {
                  return(BADOPTION);
	       }
	    }
	    while ((ErrorCode = DeletePartition(Menu2Driven, DiskID, ParseTable))
                                                                     != SUCCESS)
            {
               if (Menu2Driven == ON)
               {
                  PDisplay( &DriverErrorObj );
                  PGetString( NULLPTR, 0, 0);
               }
               else 
               {
                  return(ErrorCode);
               }
            }
            break;
         case ACTIVATE:
            if (ParseTable->ArgpPtr[A_OPTION]->Found)
            {		
	       if ( !ParseTable->ArgpPtr[N_OPTION]->Found ) 
               {
	          Menu2Driven = ON;
               }		   		    
	       else if ( ParseTable->ArgpPtr[N_OPTION]->Found ) 
               {
	          Menu2Driven = OFF;
               }
	       else 
	       {
		  return(BADOPTION);
               }
            }				   		   
	    while ((ErrorCode = ActivatePartition(Menu2Driven, ParseTable, 
	                                                   DiskID))!= SUCCESS)
            {
               if (Menu2Driven == ON)
               {
                  PDisplay( &DriverErrorObj );
                  PGetString( NULLPTR, 0, 0);
               }
               else 
               {
                  return(ErrorCode);
               }
            }
            break;
         case DISPLAY:
            if ((ErrorCode = DisplayPartition(DiskID)) != SUCCESS)
            {
               return(ErrorCode);
            }
            break;
         case DONE:  /* = ESC */
            MenuDriven = OFF; /* Turn off the screen switch */
            break;
         default:
            if (MenuDriven == ON)
            {
               break;
            }
            else 
            {
               return(BUG11);
            }
      }
   }                    /* if using option on the command line or DONE then */
   while(MenuDriven);   /* exit, else screen driven, back to the main menu  */
   return(SUCCESS);
}

/*#######################################################################
#                                                                       #
# Function Name: CreatPartition()                                       # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as support routine.                                # 
#                                                                       #  
#     Arguments: PDiskIDStruct *DiskID;                                 #
#                PCommandLineOptions *ParseTable;                       #
#                                                                       #
#   Description: Get options from command line or screen, then goto the #
#                specified creating function.                           # 
#                                                                       #
#######################################################################*/

ULONG CreatePartition(UWORD MenuDriven, PDiskIDStruct *DiskID, 
	PCommandLineOptions *ParseTable)
{
   int  Option;               /* Used to hold the menu option      */
   int  InMsg;                /* Message come from the screen.     */

   do
   {
      /* Get the option from either screen or command line */

      if (MenuDriven == ON)
      {  
         /* Display Screen 2 and get the option */

         FDiskScreen(2, &InMsg, DiskID->DriveName, NULLPTR, DiskID);
    
         Option = InMsg;
      }
      else /* MenuDriven == OFF */
      {
         /* get the option from the command line */

         if (
             ParseTable->ArgpPtr[N_OPTION]->Found && 
             ParseTable->ArgpPtr[S_OPTION]->Found 
            )
         {  /* O, N, S options on the command line */
           
            /* Check for the legal option data */

            if (ParseTable->ArgpPtr[S_OPTION]->Result < 0 ||
                ParseTable->ArgpPtr[N_OPTION]->Result <= 0)
            {   /* Starting cylinder address can be 0 bu not negative */
                /* Size in cylinders can not be zero or less          */

               return(ILLEGALOPTIONDATA2);
            }

            Option = USERSIZE;
         }
         else if ( ParseTable->ArgpPtr[E_OPTION]->Found )
         { /* E option on the command line */

            Option = ALL32MEG;
         }
         else /* W option on the command line */
         {
            Option = RANDOMSIZE;
         }
      }

      switch(Option)
      {
         case ALL32MEG: /* Create ALL 32 Meg partitions */
            if ((ErrorCode = CreateAll32Meg(MenuDriven, DiskID)) != SUCCESS)
            {
               return(ErrorCode);
            }
            break;
         case USERSIZE: /* Create user specified size partition */
            while ((ErrorCode=CreateUserSizePart(MenuDriven,DiskID,ParseTable))
                                                                    != SUCCESS)
            {
               if (MenuDriven == ON)
               {
                  PDisplay( &DriverErrorObj );
                  PGetString( NULLPTR, 0, 0);
               }
               else 
               {
                  return(ErrorCode);
               }
            }
            break;
         case RANDOMSIZE: /* Create whole disk as one partition */
            if ((ErrorCode = CreateWholeDiskPart(DiskID))!=SUCCESS)
            {
               return(ErrorCode);
            }
            break;
         case DONE:
            MenuDriven = OFF;
            break;
         default:
            return(BUG12);
      }
   }                   /* if command line option than exit, else */
   while(MenuDriven); /* go back to display screen 2 again      */
   return(SUCCESS);
}

/*#######################################################################
#                                                                       #
# Function Name: CreateAll32Meg()                                       # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: UWORD MenuDriven;                                      #
#                      Indicates using screen driven or command line    #
#                      options.                                         #
#                PDiskIDStruct *DiskID;                                 #
#   Description: This function creates all 32 MegBytes partitions on    #
#                on the disk.                                           #
#                                                                       # 
#######################################################################*/

ULONG CreateAll32Meg(UWORD MenuDriven, PDiskIDStruct *DiskID)
{
   GapEntry *DiskGapList;
   GapEntry *CurrentGapEntry; /* The current gap entry in the list     */
   ULONG Num_32Meg = 1;       /* The number of the 32 meg gaps.        */
   ULONG Num_Cyl_Left;        /* The number of the cylinders left.     */
   ULONG SectorsPerCyl;       /* The number of sectors per cylinder.   */
   int   InMsg;               /* Message come from the screen.         */
   int   OutMsg[2];           /* Message goes to the screen.           */
   ULONG SectorSize;          /* The number of the bytes per sector.   */
   ULONG SectorsPerTrack;     /* Number of sectors per track.          */
   ULONG SectorsIn32Meg;      /* The number of sectors in 32 MB        */
   ULONG CylindersPer32Meg;   /* used to calculate the number of the   */
                              /* cylinders in 32Meg partition.         */
   UWORD First_Loop;          /* Indicates this loop is first loop     */
   ULONG Dummy;

   /* Get the sector size in bytes */

   SectorSize = (ULONG) DiskID->MediaDesBlock.SectorSize;

   /* Get SectorsPerTrack */

   SectorsPerTrack = DiskID->MediaDesBlock.SectorsPerTrack;

   /* Calculate how many sectors in a 32-MB */

   SectorsIn32Meg = (ULONG) (32L * (MEG_1 / SectorSize));
   
   /* Get the number of sectors per cylinder */

   SectorsPerCyl = DiskID->MediaDesBlock.NumberOfHeads * SectorsPerTrack;

   /* Get the number of cylinders in a 32Meg */
   
   CylindersPer32Meg = SectorsIn32Meg / SectorsPerCyl;
   
   /*                                                                      */
   /*                   Algorithm for 32 Meg Creation:                     */
   /*                                                                      */
   /*  LOOP_1 Until there is no any 32 Mb Gap on the disk.                 */
   /*  {                                                                   */
   /*     Find whole gaps on the disk.                                     */
   /*                                                                      */
   /*     LOOP_1_1 Find how 32 Mb gaps on the disk.                        */
   /*     {                                                                */
   /*        Put the number of 32 Mb gap on the disk and then number of    */
   /*        cylinders left on the disk after creation.                    */
   /*        LOOP_1 Exit:                                                  */
   /*        1). If this is first loop in the LOOP_1 and                   */
   /*            the number of 32 Mb gap is 0 then                         */
   /*            {                                                         */
   /*              display the message and return.                         */
   /*            }                                                         */
   /*        2). Else return(SUCCESS).                                     */
   /*     }                                                                */
   /*                                                                      */
   /*     LOOP_1_2 Find a 32 Mb gap in the gap link list until no          */
   /*              any 32 Mb gap in the list.                              */
   /*     {                                                                */
   /*        Creat the found 32 Mb gap as a partition.                     */
   /*     }                                                                */
   /*   }                                                                  */

   First_Loop = 0;

   while (1)
   {  /* LOOP_1 untill there is no any 32 Meg gap on the disk */

      if ((DiskGapList = (GapEntry *) malloc(sizeof(GapEntry))) == NULL )
      {
         return(NOTENOUGHMEMORY);
      }

      DiskGapList->NextGapEntry = NULL;
            
      /*                                                                  */
      /*              Part 1 -- Finding the empty gaps                    */
      /*                                                                  */
    
      if ((ErrorCode = FindDiskGaps(DiskID, DiskGapList)) != SUCCESS)
      {
         return(ErrorCode);
      }

      /*                                                                */
      /*         Part 2 -- Getting the number of the 32 meg gaps        */
      /*                   Calculating the sum of sylinder left         */
      /*                                                                */

      /* Initialize the sum variables */

      Num_32Meg = 0;                                        
      Num_Cyl_Left = 0;
      CurrentGapEntry = DiskGapList->NextGapEntry;

      while(CurrentGapEntry != NULL)
      {  /* LOOP_1_1 until there is no 32 Mb gap in the gap link list */
 
         Dummy = CurrentGapEntry->EndAddr.Cylinder 
	         - CurrentGapEntry->OSBootStartAddr.Cylinder 
		 + 1L;

         if (Dummy >= CylindersPer32Meg)
         {  /* Found the 32 Mb gap in the gap link list */

            /* Get the number of 32 Meg in that gap */

            Num_32Meg +=(Dummy / CylindersPer32Meg);

            /* Get the number of cylinders left after 32 Meg creation */

            Num_Cyl_Left +=(Dummy % CylindersPer32Meg);
         }
         else /* Gap size is smaller the 32 meg */
         {
            Num_Cyl_Left += CurrentGapEntry->GapSizeInSectors/SectorsPerCyl;
         }

         CurrentGapEntry = CurrentGapEntry->NextGapEntry;

      }  /* LOOP_1_1 */

      /* Check if there is no 32 meg gap on the disk       */
      /* If uses the screen, display the no 32 Meg message */
      /* then wait for user enter the ESC key, else return */
      /* error message.                                    */

      if (Num_32Meg == 0 && First_Loop == 0)
      {  /* There is no any 32 MB gap on the disk before creation */

         if(MenuDriven == ON)
         {  /* Using screen driven then display the */
            /* error message to the screen          */

            FDiskScreen(31, NULLPTR, NULLPTR, NULLPTR, DiskID);

            /* Back to the previous screen */

            return(SUCCESS);
         }
         else /* Command line */
         {
            return(NO32MEGPARTITION);
         }
      }
      else if (Num_32Meg == 0 && First_Loop != 0 )
      {  /* creation is successfully done */

         return(SUCCESS);
      }
   
      /*                                                                */
      /*            Part 4 -- Conforming with user for creation         */
      /*                                                                */

      /* The number of 32 Meg is not 0 */

      if (MenuDriven == ON && First_Loop == 0)
      {
         OutMsg[0] = Num_32Meg;
         OutMsg[1] = Num_Cyl_Left;
         
         /* Display the screen 3 */

         FDiskScreen(32, &InMsg, DiskID->DriveName, OutMsg, DiskID);
 
         /* Check user's responce */

         if (InMsg == FALSE || InMsg == DONE)
         {  /* User press the NO key or ESC key */

            return(SUCCESS);
         }

         First_Loop = 1;
      }

      /*                                                                */
      /*            Part 5 -- Creating 32 meg partition                 */
      /*                                                                */

      CurrentGapEntry = DiskGapList->NextGapEntry;

      while(CurrentGapEntry != NULL)
      {  /* LOOP_1_2 utill there is no 32Me gap in the gap link list */

         Dummy = CurrentGapEntry->EndAddr.Cylinder 
	         - CurrentGapEntry->OSBootStartAddr.Cylinder 
		 + 1L;
	 
         if (Dummy >= CylindersPer32Meg)
         {  /* Found the 32 Mb gap in the gap link list */

            /* Modify the gap size for consistence of the usage of */
            /* PartitionCreation routine                           */

            CurrentGapEntry->GapSizeInSectors = CylindersPer32Meg 
	                                        * SectorsPerCyl;

            CurrentGapEntry->EndAddr.Cylinder = 
                                    (CurrentGapEntry->OSBootStartAddr.Cylinder
                                    + CylindersPer32Meg) - 1; 

            CurrentGapEntry->EndAddr.Head = DiskID->MediaDesBlock.NumberOfHeads
                                            - 1;

            CurrentGapEntry->EndAddr.Sector = 
                                          DiskID->MediaDesBlock.SectorsPerTrack;
            
            /* Partition creation */

            if ((ErrorCode=PartitionCreation(CurrentGapEntry,DiskID))!=SUCCESS)
            {
               return(ErrorCode);
            }
         }

         CurrentGapEntry = CurrentGapEntry->NextGapEntry;

      }  /* LOOP_1_2 */

      First_Loop = 1;
      free(DiskGapList);

   }  /* LOOP_1*/
}

/*#######################################################################
#                                                                       #
# Function Name: FindDiskGaps()                                         # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as An application support routine.                 # 
#                                                                       #  
#     Arguments: PDiskIDStruct *DiskID;                                 #
#                GapEntry *GapListHead;                                 #
#                         Head of the gap link list.                    #
#                                                                       #
#   Description: This function looks for all the gaps on the disk, and  #
#                link them together.                                    #
#                                                                       #
#######################################################################*/

ULONG FindDiskGaps(PDiskIDStruct *DiskID, GapEntry *GapLinkList )
{
   DiskAddress FullDiskSize;      /* Used to hold the disk size in     */
                                  /* Cylinders, Heads, and Sectors.    */
   LONG  LastMBRAddr;             /* Used to hold the last MBR sector  */
                                  /* offset address.                   */
   ULONG SectorOffset;            /* sector offset address for I/O.    */
   UWORD SectorCount;             /* Number of sector read or write.   */
   UWORD SectorSize;              /* Number of bytes per sector.       */
   UBYTE SectorPerTrack;          /* Number of sectors per track.      */
   ULONG TotalSectorsOnDisk;      /* Length of the disk in sectors.    */
   UBYTE NumberOfHeads;           /* Total number of heads on the disk.*/
   UWORD NumberOfCylinders;       /* Total number of cylinders on disk.*/
   ULONG CurrentPartEndAddr;      /* The end address of a current      */
                                  /* partition in sector format.       */
   ULONG ExtPartStartAddr;        /* The starting address of a extended*/
                                  /* partition in sector format.       */
   WORD BitIndex, UnkIndex;
   WORD ExtIndex, NonIndex, LastIndex;
   GapEntry *CurrentGapEntry, *LastGapEntry;
   ULONG GapStartAddr, GapEndAddr;

   DiskAddress DiskAddr;

   int  IndexArray[NUM_PARTITION_ENTRY];
   int  NumOfPart, Index;

   DiskAddress GapStart1, GapStart2;

   /*                                                                  */
   /*             Part 1 -- Get the total size of the disk             */
   /*                                                                  */

   CurrentGapEntry = GapLinkList;
   
   /* Get the sector size in bytes */

   SectorSize = DiskID->MediaDesBlock.SectorSize;
   
   /* Get the number of sectors per track */

   FullDiskSize.Sector = SectorPerTrack 
                       = DiskID->MediaDesBlock.SectorsPerTrack;

   /* Get the total number of the heads on the disk */

   FullDiskSize.Head = NumberOfHeads 
                     = DiskID->MediaDesBlock.NumberOfHeads;

   /* Get the total number of cylinders on the disk */

   FullDiskSize.Cylinder = NumberOfCylinders 
                         = DiskID->MediaDesBlock.NumberOfCylinders;

   /* Calculates the total number sector on the disk */

   TotalSectorsOnDisk = ((ULONG) SectorPerTrack 
                        * (ULONG) NumberOfHeads 
			* (ULONG) NumberOfCylinders)
			- 1;

   /*                                                                  */
   /*            Part 2 -- Check the first MBR                         */
   /*                                                                  */

   /* Read the first Master Boot Record */
   
   SectorOffset = SECTOR_ZERO;
   SectorCount  = SECTOR_ONE;
   LastMBRAddr = -2;

   if ((ErrorCode = PReadPhysicalSector(SectorOffset, SectorCount,
                                      (UBYTE *) FirstMBR, DiskID)) != SUCCESS)
   {
      return(ErrorCode);
   }

   /* Check if there is no partition on the disk */

   BitIndex = MBRBitOwnerType(FirstMBR);
   ExtIndex = MBRExtOwnerType(FirstMBR);
   NonIndex = MBRNonOwnerType(FirstMBR);
   UnkIndex = MBRUnkOwnerType(FirstMBR);

   if (FirstMBR->Signature != SIGNATURE)
   {  /* The disk is unpartitioned */
  
   /********************************************************************/
   /*                                                                  */
   /*                CASE 1 -- Unpartitioned Disk                      */
   /*                                                                  */
 
      LastMBRAddr = -1;

      if ((ErrorCode = BuildGapEntry(FirstMBR, CurrentGapEntry, FullDiskSize,
                                  LastMBRAddr, CASE1, DiskID)) != SUCCESS)
      {
         return(ErrorCode);
      }
      return(SUCCESS);
             
   }
   else if (ExtIndex == NOT_FOUND)
   {  /* No extended partition in the first MBR */

      ReOrderMBR(IndexArray, &NumOfPart, FirstMBR, DiskID);

      if (NumOfPart == 3)
      {  /* No room for partition creation in the first MBR */
  
         LastIndex = IndexArray[NumOfPart - 1];

         CurrentPartEndAddr = ConvertDiskAddrFromMBR(
                           &(FirstMBR->PartTable[LastIndex].EndAddr), DiskID);

         if (CurrentPartEndAddr < TotalSectorsOnDisk)
         {   /* There is gap from the end of the first partition to end disk*/
          
            CurrentGapEntry->NextGapEntry = (GapEntry *) malloc(sizeof(GapEntry));
            LastGapEntry = CurrentGapEntry;
            CurrentGapEntry = CurrentGapEntry->NextGapEntry;
            CurrentGapEntry->LastMBRAddr = 0;

            LoadAddrFromMBR(&(DiskAddr), &(FirstMBR->PartTable[LastIndex].EndAddr));
            AssignAddr(&DiskAddr, &(CurrentGapEntry->StartAddr), STARTADDR, DiskID);
            AssignAddr(&FullDiskSize, &(CurrentGapEntry->EndAddr), ENDADDR, DiskID);

            if ((ErrorCode = CheckDisk(CurrentGapEntry, DiskID)) != SUCCESS)
            {
               free(CurrentGapEntry);
               LastGapEntry->NextGapEntry = NULL;
               CurrentGapEntry = LastGapEntry;
            }
            else
            {
               GapStartAddr = PConvertDiskAddr(
                                       &(CurrentGapEntry->OSBootStartAddr),
                                       FullDiskSize.Sector, FullDiskSize.Head);
               GapEndAddr = PConvertDiskAddr(
                                       &(CurrentGapEntry->EndAddr),
                                       FullDiskSize.Sector, FullDiskSize.Head);

               CurrentGapEntry->GapSizeInSectors = GapEndAddr - GapStartAddr + 1;
               CurrentGapEntry->NextGapEntry = NULL;
            }
         }
         return(SUCCESS);
      }

      if (NonIndex != NOT_FOUND)
      {  /* Room for DOS or FlexOS Partition creation */

         Index = 0;

         GapStart1.Sector   = SECTOR_ONE;
         GapStart1.Cylinder = 0;
         GapStart1.Head     = 0;
         
         while(Index < NumOfPart)
         {
            LoadAddrFromMBR(&GapStart2, &(FirstMBR->PartTable[IndexArray[Index]].StartAddr));

            if (GapStart1.Cylinder < GapStart2.Cylinder) 
            {
               LastGapEntry = CurrentGapEntry;
               CurrentGapEntry->NextGapEntry = (GapEntry *) malloc(sizeof(GapEntry));
               CurrentGapEntry = CurrentGapEntry->NextGapEntry;

               AssignAddr(&(GapStart1), &(CurrentGapEntry->StartAddr), ASSIGN, DiskID);
 
               AssignAddr(&(GapStart2), &(CurrentGapEntry->EndAddr), ASSIGN, DiskID);
           
               CurrentGapEntry->EndAddr.Cylinder -= 1; 

               CurrentGapEntry->EndAddr.Sector = SectorPerTrack;

               CurrentGapEntry->EndAddr.Head = NumberOfHeads - 1;

               CurrentGapEntry->LastMBRAddr = -2;

               /* Check if the disk gap area can be found */

               if ((ErrorCode = CheckDisk(CurrentGapEntry, DiskID)) != SUCCESS)
               {  /* No sylinder is good in that gap disk area */

                  /* Free that gap entry from memory */

                  free(CurrentGapEntry);
                  LastGapEntry->NextGapEntry = NULL;
                  CurrentGapEntry = LastGapEntry;
               }
               else
               {
           
                  GapStartAddr = PConvertDiskAddr(&(CurrentGapEntry->OSBootStartAddr),
                                 FullDiskSize.Sector, FullDiskSize.Head);

                  GapEndAddr = PConvertDiskAddr(&(CurrentGapEntry->EndAddr),
                                 FullDiskSize.Sector, FullDiskSize.Head);

                  CurrentGapEntry->GapSizeInSectors = GapEndAddr 
                                                      - GapStartAddr 
                                                      + 1;
               } 
            }

            LoadAddrFromMBR(&GapStart1, &(FirstMBR->PartTable[IndexArray[Index]].EndAddr));
            GapStart1.Sector = SECTOR_ONE;
            GapStart1.Head   = 0;
            GapStart1.Cylinder += 1;

            Index += 1;
         }

         Index -= 1;

         CurrentPartEndAddr = ConvertDiskAddrFromMBR(
                     &(FirstMBR->PartTable[IndexArray[Index]].EndAddr), DiskID);

         if (CurrentPartEndAddr < TotalSectorsOnDisk)
         {

   /********************************************************************/
   /*                                                                  */
   /*            There is gap from the end of the current partition    */
   /*                         to the end of the disk                   */
   /*                                                                  */
            CurrentGapEntry->NextGapEntry = (GapEntry *) malloc(sizeof(GapEntry));
            LastGapEntry = CurrentGapEntry;
            CurrentGapEntry = CurrentGapEntry->NextGapEntry;

            CurrentGapEntry->LastMBRAddr = -2;

            LoadAddrFromMBR(&(DiskAddr), 
                            &(FirstMBR->PartTable[IndexArray[Index]].EndAddr));

            AssignAddr(&(DiskAddr),&(CurrentGapEntry->StartAddr),STARTADDR,DiskID);

            AssignAddr(&FullDiskSize, &(CurrentGapEntry->EndAddr), ENDADDR, DiskID);
   
            /* Check if the disk gap area can be found */

            if ((ErrorCode = CheckDisk(CurrentGapEntry, DiskID)) != SUCCESS)
            {  /* No sylinder is good in that gap disk area */

               /* Free that gap entry from memory */

               free(CurrentGapEntry);
               LastGapEntry->NextGapEntry = NULL;
               CurrentGapEntry = LastGapEntry;
            }
            else
            {
           
               GapStartAddr = PConvertDiskAddr(&(CurrentGapEntry->OSBootStartAddr),
                           FullDiskSize.Sector, FullDiskSize.Head);

               GapEndAddr = PConvertDiskAddr(&(CurrentGapEntry->EndAddr),
                              FullDiskSize.Sector, FullDiskSize.Head);

               CurrentGapEntry->GapSizeInSectors = GapEndAddr - GapStartAddr + 1;

               CurrentGapEntry->NextGapEntry = NULL;
            } 
         }
         return(SUCCESS);
      }
   }
   else if (ExtIndex != NOT_FOUND)
   {  /* Extended partition is found */

      ReOrderMBR(IndexArray, &NumOfPart, FirstMBR, DiskID);

      if ( NumOfPart < 3)
      {
         if ((BitIndex == NOT_FOUND) && (UnkIndex == NOT_FOUND))
         {
  
   /********************************************************************/
   /*                                                                  */
   /*            CASE 2 -- Partition Starts From The Beginning         */
   /*                      To A Extended Partition.                    */
   /*                                                                  */

            if ((ErrorCode = BuildGapEntry(FirstMBR, CurrentGapEntry, FullDiskSize,
                                      LastMBRAddr, CASE2, DiskID)) != SUCCESS)
            {
               return(ErrorCode);
            }

            if (CurrentGapEntry->NextGapEntry != NULL)
            {
               CurrentGapEntry = CurrentGapEntry->NextGapEntry;
            }
         }
         else if (NonIndex != NOT_FOUND)
         {  /* MBR Slot for partition creation */

            Index = 0;

            GapStart1.Sector   = SECTOR_ONE;
            GapStart1.Cylinder = 0;
            GapStart1.Head     = 0;

            while(Index < NumOfPart)
            {

               LoadAddrFromMBR(&GapStart2, &(FirstMBR->PartTable[IndexArray[Index]].StartAddr));

               if (GapStart1.Cylinder < GapStart2.Cylinder) 
               {
                  LastGapEntry = CurrentGapEntry;
                  CurrentGapEntry->NextGapEntry = (GapEntry *) malloc(sizeof(GapEntry));
                  CurrentGapEntry = CurrentGapEntry->NextGapEntry;

                  AssignAddr(&(GapStart1), &(CurrentGapEntry->StartAddr), ASSIGN, DiskID);
 
                  AssignAddr(&(GapStart2), &(CurrentGapEntry->EndAddr), ASSIGN, DiskID);
           
                  CurrentGapEntry->EndAddr.Cylinder -= 1; 

                  CurrentGapEntry->EndAddr.Sector = SectorPerTrack;

                  CurrentGapEntry->EndAddr.Head = NumberOfHeads - 1;

                  CurrentGapEntry->LastMBRAddr = -2;

                  /* Check if the disk gap area can be found */

                  if ((ErrorCode = CheckDisk(CurrentGapEntry, DiskID)) != SUCCESS)
                  {  /* No sylinder is good in that gap disk area */

                     /* Free that gap entry from memory */

                     free(CurrentGapEntry);
                     LastGapEntry->NextGapEntry = NULL;
                     CurrentGapEntry = LastGapEntry;
                  }
                  else
                  {
           
                     GapStartAddr = PConvertDiskAddr(&(CurrentGapEntry->OSBootStartAddr),
                                    FullDiskSize.Sector, FullDiskSize.Head);

                     GapEndAddr = PConvertDiskAddr(&(CurrentGapEntry->EndAddr),
                                    FullDiskSize.Sector, FullDiskSize.Head);

                     CurrentGapEntry->GapSizeInSectors = GapEndAddr 
                                                         - GapStartAddr 
                                                         + 1;
                     CurrentGapEntry->NextGapEntry = NULL;
                  } 
               }

               LoadAddrFromMBR(&GapStart1, &(FirstMBR->PartTable[IndexArray[Index]].EndAddr));
               GapStart1.Sector = SECTOR_ONE;
               GapStart1.Head   = 0;
               GapStart1.Cylinder += 1;

               Index += 1;
            }

            Index -= 1;
            CurrentPartEndAddr = ConvertDiskAddrFromMBR(
                        &(FirstMBR->PartTable[IndexArray[Index]].EndAddr), DiskID);

            ExtPartStartAddr = ConvertDiskAddrFromMBR(
                     &(FirstMBR->PartTable[ExtIndex].StartAddr), DiskID);

            if (CurrentPartEndAddr < (ExtPartStartAddr - 1))
            {

   /********************************************************************/
   /*                                                                  */
   /*            There is gap from the end of the first partition      */
   /*                         to the end of the disk                   */
   /*                                                                  */
               CurrentGapEntry->NextGapEntry = (GapEntry *) malloc(sizeof(GapEntry));
               LastGapEntry = CurrentGapEntry;
               CurrentGapEntry = CurrentGapEntry->NextGapEntry;

               CurrentGapEntry->LastMBRAddr = -2;

               LoadAddrFromMBR(&(DiskAddr), &(FirstMBR->PartTable[IndexArray[Index]].EndAddr));

               AssignAddr(&(DiskAddr),&(CurrentGapEntry->StartAddr),STARTADDR,DiskID);

               LoadAddrFromMBR(&(DiskAddr), &(FirstMBR->PartTable[ExtIndex].StartAddr));
 
               DiskAddr.Cylinder -= 1;
               DiskAddr.Sector = DiskID->MediaDesBlock.SectorsPerTrack;
               DiskAddr.Head = DiskID->MediaDesBlock.NumberOfHeads - 1;
 
               AssignAddr(&(DiskAddr), &(CurrentGapEntry->EndAddr), ASSIGN, DiskID);
   
               /* Check if the disk gap area can be found */

               if ((ErrorCode = CheckDisk(CurrentGapEntry, DiskID)) != SUCCESS)
               {  /* No sylinder is good in that gap disk area */

                  /* Free that gap entry from memory */

                  free(CurrentGapEntry);
                  LastGapEntry->NextGapEntry = NULL;
                  CurrentGapEntry = LastGapEntry;
               }
               else
               {
           
                  GapStartAddr = PConvertDiskAddr(&(CurrentGapEntry->OSBootStartAddr),
                                 FullDiskSize.Sector, FullDiskSize.Head);

                  GapEndAddr = PConvertDiskAddr(&(CurrentGapEntry->EndAddr),
                                 FullDiskSize.Sector, FullDiskSize.Head);

                  CurrentGapEntry->GapSizeInSectors = GapEndAddr - GapStartAddr + 1;

                  CurrentGapEntry->NextGapEntry = NULL;
               } 
            }
         }
      }
   }
   /*                                                                  */
   /*           Part 3 -- Loop to find gaps between partitions         */
   /*                                                                  */

   SectorOffset = ConvertDiskAddrFromMBR(
                        &(FirstMBR->PartTable[ExtIndex].StartAddr), DiskID);

   if ((ErrorCode = PReadPhysicalSector(SectorOffset, SectorCount,
                                       (UBYTE *) MBR, DiskID)) != SUCCESS)
   {
      return(ErrorCode);
   }

   BitIndex = MBRBitOwnerType(MBR);
   ExtIndex = MBRExtOwnerType(MBR);
    
   if ((BitIndex == NOT_FOUND ) && (ExtIndex != NOT_FOUND))
   {
      LastGapEntry = CurrentGapEntry;
      CurrentGapEntry->NextGapEntry = (GapEntry *) malloc(sizeof(GapEntry));
      CurrentGapEntry = CurrentGapEntry->NextGapEntry;
       
      LoadAddrFromMBR(&(CurrentGapEntry->EndAddr), 
                      &(MBR->PartTable[ExtIndex].StartAddr));
      CurrentGapEntry->EndAddr.Cylinder -= 1;
      CurrentGapEntry->EndAddr.Sector = DiskID->MediaDesBlock.SectorsPerTrack;
      CurrentGapEntry->EndAddr.Head = DiskID->MediaDesBlock.NumberOfHeads - 1;

      ExtIndex = MBRExtOwnerType(FirstMBR);
      LoadAddrFromMBR(&(CurrentGapEntry->StartAddr), 
                      &(FirstMBR->PartTable[ExtIndex].StartAddr));
      		      
      /* Check if the disk gap area can be found */

      if ((ErrorCode = CheckDisk(CurrentGapEntry, DiskID)) != SUCCESS)
      {  /* No sylinder is good in that gap disk area */

         /* Free that gap entry from memory */

         free(CurrentGapEntry);
         LastGapEntry->NextGapEntry = NULL;
         CurrentGapEntry = LastGapEntry;
      }
      else
      {
         /* Get the gap zise in sectors */
 
         GapStartAddr = PConvertDiskAddr(&(CurrentGapEntry->OSBootStartAddr),
                              FullDiskSize.Sector, FullDiskSize.Head);

         GapEndAddr = PConvertDiskAddr(&(CurrentGapEntry->EndAddr),
                              FullDiskSize.Sector, FullDiskSize.Head);

         CurrentGapEntry->GapSizeInSectors = GapEndAddr - GapStartAddr + 1;
  
         CurrentGapEntry->LastMBRAddr = 0;

         CurrentGapEntry->NextGapEntry = NULL;
      }    
      LastMBRAddr = SectorOffset;
   
      ExtIndex = MBRExtOwnerType(MBR);
      SectorOffset = ConvertDiskAddrFromMBR(
                       &(MBR->PartTable[ExtIndex].StartAddr), DiskID);

      if ((ErrorCode = PReadPhysicalSector(SectorOffset, SECTOR_ONE,
                            (UBYTE *) MBR, DiskID)) != SUCCESS)
      {
         return(ErrorCode);
      }
   }
   	 				       
   while(1)
   {
      LastMBRAddr = SectorOffset;
   
      BitIndex = MBRBitOwnerType(MBR); 

      /* Get the partiton end sector offset address */

      CurrentPartEndAddr = ConvertDiskAddrFromMBR(
                            &(MBR->PartTable[BitIndex].EndAddr), DiskID);

      if ((ExtIndex = MBRExtOwnerType(MBR)) != NOT_FOUND)
      {  /* One partition and one extended partition */
        

         /* Get the extended partition starting sector offset */

         ExtPartStartAddr=ConvertDiskAddrFromMBR(
                                &(MBR->PartTable[ExtIndex].StartAddr), DiskID);

         if (CurrentPartEndAddr != (ExtPartStartAddr - 1))
         {  

   /************************************************************************/
   /*                                                                      */
   /*                   CASE 3 -- Gap Between Two Partitions               */
   /*                                                                      */

            if ((ErrorCode = BuildGapEntry(MBR, CurrentGapEntry, FullDiskSize,
                                LastMBRAddr, CASE3, DiskID)) != SUCCESS)
            {
               return(ErrorCode);
            }
            
	    if (CurrentGapEntry->NextGapEntry != NULL)
	    {
	       CurrentGapEntry = CurrentGapEntry->NextGapEntry;
	    }  

	 }

         /* then read Extended MBR */

         SectorOffset = ExtPartStartAddr;

         if ((ErrorCode = PReadPhysicalSector(SectorOffset, SectorCount,
                                       (UBYTE *) MBR, DiskID)) != SUCCESS)
         {
            return(ErrorCode);
         }
      }
      else /* No more extended partition */
      { 
         if (CurrentPartEndAddr != TotalSectorsOnDisk)
         {  /* There is a gap between the end of current partiton and    */
            /* the end of the disk.                                      */
   
   /**********************************************************************/
   /*                                                                    */
   /*               CASE 4 -- Gap From Somewhere To End Of Disk          */
   /*                                                                    */

            if ((ErrorCode = BuildGapEntry(MBR, CurrentGapEntry, FullDiskSize,
                                 LastMBRAddr, CASE4, DiskID)) != SUCCESS)
            {
               return(ErrorCode);
            }
         }
   /*******************************************************************/
   /*                                                                 */
   /*            CASE 5 -- No Gap Between Last Partition And          */
   /*                                End Of The Disk                  */
   /*                                                                 */

         return(SUCCESS);
      }
   }
}

/*#######################################################################
#                                                                       #
# Function Name: ReverseConvertDiskAddr()                               # 
#                                                                       # 
#  Return Value: None.                                                  #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: ULONG SectorOffset;                                    #
#                DiskAddress *DiskAddr;                                 #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: This function converts disk address from sectoroffset  #
#                to head, sector sylinder.                              #
#                                                                       # 
#######################################################################*/

void ReverseConvertDiskAddr( ULONG SectorOffset, DiskAddress *DiskAddr,
	PDiskIDStruct *DiskID)
{
   ULONG Rest;

   DiskAddr->Cylinder = SectorOffset 
                        / (ULONG) (DiskID->MediaDesBlock.SectorsPerTrack 
                                   * DiskID->MediaDesBlock.NumberOfHeads);
   Rest = SectorOffset % ((ULONG) (DiskID->MediaDesBlock.SectorsPerTrack
                                   * DiskID->MediaDesBlock.NumberOfHeads));

   DiskAddr->Head = Rest / (ULONG) DiskID->MediaDesBlock.SectorsPerTrack;

   DiskAddr->Sector = Rest % (ULONG) DiskID->MediaDesBlock.SectorsPerTrack;

   DiskAddr->Sector += 1;
}

/*#######################################################################
#                                                                       #
# Function Name: ReOrderMBR()                                           # 
#                                                                       # 
#  Return Value: None.                                                  #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: int   IndexArray[];                                    #
#                int   *NumOfPart;                                      #
#                MasterBootRec *MBR;                                    #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: This function sorts the partition slot index           #
#                according who is located first.                        #
#                                                                       # 
#######################################################################*/

void ReOrderMBR( int  IndexArray[], int  *NumOfPart,  MasterBootRec *MBR,
	PDiskIDStruct *DiskID )
{
   int   Index, Index1, Index2;
   ULONG SectorOffset1, SectorOffset2;

   Index = Index1 = 0;

   while (Index < NUM_PARTITION_ENTRY)
   {
      if (MBR->PartTable[Index].OwnerType != 0 &&
          MBR->PartTable[Index].OwnerType != 05)
      {
         IndexArray[Index1] = Index;
         Index1 += 1;
      }
      Index += 1;
   }
   
   *NumOfPart = Index1;

   Index = Index1 = 0;

   while (Index < (*NumOfPart - 1))
   {
      Index1 = Index + 1;
      while(Index1 < *NumOfPart)
      {
         SectorOffset1 = ConvertDiskAddrFromMBR(
                      &(MBR->PartTable[IndexArray[Index]].StartAddr), DiskID );
         SectorOffset2 = ConvertDiskAddrFromMBR(
                      &(MBR->PartTable[IndexArray[Index1]].StartAddr), DiskID );
         if (SectorOffset1 > SectorOffset2)
         {  
            Index2 = IndexArray[Index];
            IndexArray[Index] = IndexArray[Index1];
            IndexArray[Index1] = Index2;
         }
         Index1 += 1;
      }
      Index += 1;
   }
}

/*#######################################################################
#                                                                       #
# Function Name: BuildGapEntry()                                        # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: MasterBootRec *MBR;                                    #
#                  -- Pointer to the MBR.                               #
#                GapEntry *CurrentGapEntry;                             #
#                  -- Pointer to the current gap entry.                 #
#                DiskAddress FullDiskSize;                              #
#                  -- Total disk size in head, sector, and cylinder     #
#                     format.                                           #
#                ULONG LastMBRAddr;                                     #
#                  -- Last MBR starting address sector offset.          #
#                UWORD CaseNum;                                         #
#                  -- Indicates what kind gap it is.                    #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: This function fills up the disk gap which is found by  #
#                FindDiskGap() with different data according to the     #
#                parameter CaseNum.                                     #
#                                                                       # 
#######################################################################*/

ULONG 
BuildGapEntry( MasterBootRec *MBR, GapEntry *GapListLastEntry,
	DiskAddress FullDiskSize, LONG LastMBRAddr,  UWORD CaseNum,
	PDiskIDStruct *DiskID )
{
   GapEntry *LastGapEntry, *CurrentGapEntry;
                                  /* Used to hold a pointer for freeing*/
                                  /* a gap entry if that whole gap disk*/
                                  /* is bad.                           */
   ULONG GapStartAddr;            /* Gap Starting sector offset address*/
   ULONG GapEndAddr;              /* Gap end sector offset address     */
   DiskAddress DiskAddr;
   WORD ExtIndex, BitIndex;
   
   /* Allocate a Gap Entry */

   if ((GapListLastEntry->NextGapEntry = 
                              (GapEntry *) malloc(sizeof(GapEntry))) == NULL)
   {
      return(NOTENOUGHMEMORY);
   }

   LastGapEntry = GapListLastEntry;
   CurrentGapEntry = GapListLastEntry->NextGapEntry;

   CurrentGapEntry->LastMBRAddr = LastMBRAddr;
    
   BitIndex = MBRBitOwnerType(MBR);
   ExtIndex = MBRExtOwnerType(MBR);

   switch(CaseNum)
   {
      case CASE1:  /*  There is no any partition on the disk */

         /* Assign 1 to sector, 0 to cylinder, and 0 to head */

         AssignAddr(NULLPTR, &(CurrentGapEntry->StartAddr), BEGINNING, DiskID);

         /* Assign Full size to sector, head, and cylinder */

         AssignAddr(&FullDiskSize, &(CurrentGapEntry->EndAddr), ENDADDR, DiskID);

         break;
      case CASE2:    /* gap between the beginning of the disk and somewhere */
                   
         /* Assign 1 to sector, 0 to cylinder, and 0 to head */

         AssignAddr(NULLPTR, &(CurrentGapEntry->StartAddr), BEGINNING, DiskID);

         /* Assign Full size to sector and head, and current cylinder number */
         /* to the cylinder address.                                         */

         /* Note: the MBR must start on the cylinder boundery.   */

         LoadAddrFromMBR(&(DiskAddr), &(MBR->PartTable[ExtIndex].StartAddr));

         AssignAddr(&(DiskAddr), &(CurrentGapEntry->EndAddr), ENDADDR, DiskID);

         break;

      case CASE3:  /* gap between two partitions */

         /* Assign 1 to sector, 0 to head, and cylinder address */

         LoadAddrFromMBR(&(DiskAddr), &(MBR->PartTable[BitIndex].EndAddr));

         AssignAddr(&(DiskAddr),&(CurrentGapEntry->StartAddr),STARTADDR,DiskID);

         /* Assign 17 to sector, 3 to head , and cylinder address */

         LoadAddrFromMBR(&(DiskAddr), &(MBR->PartTable[ExtIndex].StartAddr));

         AssignAddr(&(DiskAddr), &(CurrentGapEntry->EndAddr), ENDADDR, DiskID);

         break;

      case CASE4:  /* Gap between somewhere and the end of the disk */

         /* Assign 1 to sector, 0 to head, and cylinder address */

         LoadAddrFromMBR(&(DiskAddr), &(MBR->PartTable[BitIndex].EndAddr));

         AssignAddr(&(DiskAddr),&(CurrentGapEntry->StartAddr),STARTADDR,DiskID);

         /* Assign 17 to sector, 3 to head, and full size cylinder address */

         AssignAddr(&FullDiskSize, &(CurrentGapEntry->EndAddr), ENDADDR, DiskID);

         break;
   }

   /* Check if the disk gap area can be found */

   if ((ErrorCode = CheckDisk(CurrentGapEntry, DiskID)) != SUCCESS)
   {  /* No sylinder is good in that gap disk area */

      /* Free that gap entry from memory */

      free(CurrentGapEntry);
      LastGapEntry->NextGapEntry = NULL;
      return(SUCCESS);
   }

   /* Get the gap zise in sectors */
 
   GapStartAddr = PConvertDiskAddr(&(CurrentGapEntry->OSBootStartAddr),
                        FullDiskSize.Sector, FullDiskSize.Head);

   GapEndAddr = PConvertDiskAddr(&(CurrentGapEntry->EndAddr),
                        FullDiskSize.Sector, FullDiskSize.Head);

   CurrentGapEntry->GapSizeInSectors = GapEndAddr - GapStartAddr + 1;

   CurrentGapEntry->NextGapEntry = NULL;
 
   return(SUCCESS);

}   

/*#######################################################################
#                                                                       #
# Function Name: AssignAddr()                                           # 
#                                                                       # 
#  Return Value: NONE                                                   #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: DiskAddress SourceAddr;                                #
#                DiskAddress DestAddr;                                  #
#                UWORD SourceType;                                      #
#                  -- Indicates what kind assignment it is.             #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: Assign the disk address in head, cylinder, and         #
#                sector format from the source address to the target    #
#                adddress.                                              #
#                                                                       # 
#######################################################################*/

void AssignAddr( DiskAddress *SourceAddr, DiskAddress *DestAddr,
	UWORD SourceType, PDiskIDStruct *DiskID )
{

   switch(SourceType)
   {
      case BEGINNING:
         DestAddr->Head = HEAD_ZERO;
         DestAddr->Cylinder = CYLINDER_ZERO;
         DestAddr->Sector = SECTOR_ONE;
         break;

      case STARTADDR:
         DestAddr->Head = HEAD_ZERO;
         DestAddr->Cylinder = SourceAddr->Cylinder + 1;
         DestAddr->Sector = SECTOR_ONE;
         break;
         
      case ENDADDR:
         DestAddr->Head = DiskID->MediaDesBlock.NumberOfHeads - 1;
         DestAddr->Cylinder = SourceAddr->Cylinder - 1;
         DestAddr->Sector = DiskID->MediaDesBlock.SectorsPerTrack;
         break;

      case LAST:
      case ASSIGN:
         DestAddr->Head = SourceAddr->Head;
         DestAddr->Cylinder = SourceAddr->Cylinder;
         DestAddr->Sector = SourceAddr->Sector;
         break;
   }
}

/*#######################################################################
#                                                                       #
# Function Name: CheckDisk()                                            # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as a suppout routine.                              # 
#                                                                       #  
#     Arguments: GapEntry *CurrentGapEntry;                             #
#                  -- Pointer to the current gap entry.                 #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: Check the disk area which includes MBR, OS Boot Record #
#                two FAT tables, and root directories.                  #
#                                                                       # 
#######################################################################*/

ULONG CheckDisk( GapEntry *CurrentGapEntry, PDiskIDStruct *DiskID )
{
   ULONG SectorOffset;    /* Starting sector number of the checking area*/ 
   UWORD SectorCount;     /* Number of the sectors will be read & write */
   ULONG CylinderCount;   /* Used to hold the number of cylinders in a  */
                          /* gap.                                       */
   UBYTE HeadNum;         /* Used to hold the number of the heads.      */
   UWORD TrackSize;       /* Used to hold number of sectors per track.  */
   DiskAddress DiskAddr;  /* Used to hold the disk offset address in the*/
                          /* cylinder, head, and sector format.         */
   ULONG TrackInitCount;  /* Used to store the number of tracks in the  */
                          /* OS Boot Record, FATs, and Root Directory   */
                          /* area.                                      */
   ULONG TrackCount;      /* Counter used to control the CheckDiskArea  */
                          /* Loop.                                      */
   
   /* Get the total head number */
   
   HeadNum = DiskID->MediaDesBlock.NumberOfHeads;

   /* Get the track size in sectors */
   
   TrackSize = DiskID->MediaDesBlock.SectorsPerTrack;
   
   SectorCount = SECTOR_ONE;

   /* Initialize DiskAddr */

   AssignAddr(&(CurrentGapEntry->StartAddr), &DiskAddr, ASSIGN, DiskID);

   do 
   {
      /* Check the MBR area */

      SectorOffset = PConvertDiskAddr(&DiskAddr, TrackSize, HeadNum);

      if ((ErrorCode = PCheckDiskArea(SectorOffset, (long)SectorCount, DiskID))
                                                                    != SUCCESS)
      {  /* Bad Sector found */

         /* Check for if it is first MBR or last cylinder of that gap */
         /* NOTE: The MBR starts on the cylinder boundery.            */

         if (CurrentGapEntry->StartAddr.Cylinder == 0 ||
             DiskAddr.Cylinder == CurrentGapEntry->EndAddr.Cylinder)
         {  
            /* First MBR, then whole disk can not be used. Or there is no */
            /* more cylinder can be used as MBR beginning address.        */

            return(ErrorCode);
         }
         else /* Not the first MBR and still have more cylinders can be used */
         {
            /* Increase the cylinder counter */

            DiskAddr.Cylinder += 1;
         }
      }
      else /* Return code is SUCCESS */
      {
         break;
      }
   } while(DiskAddr.Cylinder <= CurrentGapEntry->EndAddr.Cylinder);

   /* send the updated disk address back to the current gap entry */

   AssignAddr(&DiskAddr, &(CurrentGapEntry->StartAddr), ASSIGN, DiskID);

   DiskAddr.Head += 1;

   AssignAddr(&DiskAddr, &(CurrentGapEntry->OSBootStartAddr), ASSIGN, DiskID);

   /* Get the number of cylinder in that gap */

   CylinderCount = CurrentGapEntry->EndAddr.Cylinder
                   - CurrentGapEntry->StartAddr.Cylinder + 1;
   
   /* Get the Sector Count for the disk checking:                */
   /* SectorCount = OSBootRecSize + FatSize * 2 + RootDirSize.   */

   SectorCount = GetSectorCount(CurrentGapEntry, DiskID);

   /* Get the number of tracks in the SectorCount */
   
   TrackInitCount = GetTrackCount(SectorCount, DiskID);

   /* Make disk address point to the current OSBoot Record starting address */

   while (CylinderCount != 0L)
   {  /* Loop until whole cylinders in that gap is run out */

      /* Back up the beginning address of the OSBoot Record */

      AssignAddr(&DiskAddr, &(CurrentGapEntry->OSBootStartAddr), ASSIGN, DiskID);

      /* Renew the TrackCount for next searching */

      TrackCount = TrackInitCount;

      while(TrackCount != 0L)
      {  /* Loop untill find the good FATs and Root Dir Area */

         SectorOffset = PConvertDiskAddr(&DiskAddr, TrackSize, HeadNum);
         
	 if (DiskAddr.Head == (HeadNum - 1))
         {  /* This Track is last one in the current cylinder */
            
            /* Renew the DiskAddr */

            DiskAddr.Head = HEAD_ZERO;
            DiskAddr.Cylinder +=1;
            CylinderCount -= 1;
         }
         else /* DiskAddr.Head < (NeadNum - 1) */
         {    /* This track is in the current cylinder */

            DiskAddr.Head += 1;
         }

	 if ((ErrorCode = PCheckDiskArea(SectorOffset, (long)TrackSize, DiskID))
	                                                       != SUCCESS)
         {  /* There is bad sector in this track               */
            /* Next track is beginning track of the whole area */
            /* This is just because the OS Boot Record must be */
            /* on the track boundary, and this area must be    */
            /* reserved continuously.                          */
            
            break;
         }
         else /* ErrorCode == SUCCESS */
         {  
            TrackCount -= 1;
         }
      }  /* Back to look for next track */
      
      /* Check if found good area */

      if (ErrorCode == SUCCESS && TrackCount == 0)
      {  /* Found */

         return(SUCCESS);
      }

   }  /* Back to look for next cylinder */

   return(BADDISK);
}   

/*#######################################################################
#                                                                       #
# Function Name: GetSectorCount()                                       # 
#                                                                       # 
#  Return Value: ULONG -- Sector Count                                  #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: GapEntry *CurrentGapEntry;                             #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: Get the sector number of the OS Boot Record, FATs,     #
#                and Root Directory.                                    #
#                                                                       # 
#######################################################################*/

UWORD GetSectorCount( GapEntry *CurrentGapEntry, PDiskIDStruct *DiskID )
{
   UWORD FatSize;         /* Number of sectors in a FAT table.          */
   UBYTE NumFat;          /* Number of the FATs in the partition.       */
   UWORD OSBootRecSize;   /* OS Boot Record size in sectors.            */
   UWORD RootDirSize;     /* Root Directory size in sectors.            */

   GetFatRootDirAreaSize(&OSBootRecSize, &FatSize, &RootDirSize, 
                                           CurrentGapEntry, DiskID);

   NumFat = DiskID->MediaDesBlock.NumberFats;
   return(OSBootRecSize + (FatSize * NumFat) + RootDirSize);
}

/*#######################################################################
#                                                                       #
# Function Name: GetTrackCount()                                        # 
#                                                                       # 
#  Return Value: ULONG -- The number of tracks in the file manager area #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: ULONG SectorCount;                                     #
#                      --  Number of sectors will be check in           #
#                          CheckDiskArea();                             #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: Convert sector count to track count.                   #
#                                                                       # 
#######################################################################*/

ULONG GetTrackCount( UWORD SectorCount, PDiskIDStruct *DiskID )
{
   ULONG TrackSize;       /* Used to hold number of sectors per track.  */
   ULONG TrackCount;      /* Counter used to control the CheckDiskArea  */
                          /* Loop.                                      */

   /* Get the track size in sector */
   
   TrackSize = DiskID->MediaDesBlock.SectorsPerTrack;

   TrackCount = SectorCount / TrackSize;

   /* Round the division result */

   if ((ULONG) SectorCount % TrackSize != 0)
   {
      TrackCount += 1;
   }
   return(TrackCount);
}

/*#######################################################################
#                                                                       #
# Function Name: GetFatRootDirAreaSize()                                # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: ULONG *OSBootRecSize;                                  #
#                  -- OS Boot Record size in sectors.                   #
#                UWORD *FatSize;                                        #
#                  -- Pointer to the FAT table size in sectors.         #
#                ULONG *RootDirSize;                                    #
#                  -- Pointer to the Root Directory size in sector.     #
#                GapEntry *CurrentGapEntry;                             #
#                  -- Pointer to the current gap entry.                 #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: Get the size of the OS Boot Record, FAT Table, and     #
#                Root directories.                                      #
#                                                                       # 
#######################################################################*/

void GetFatRootDirAreaSize( UWORD *OSBootRecSize, UWORD *FatSize, 
	UWORD *RootDirSize, GapEntry *CurrentGapEntry, PDiskIDStruct *DiskID )
{
   UWORD ClusterSize;     /* The number of sectors per cluster */
   UWORD FatBitNum;       /* The bit number of the FAT         */
   ULONG GapSize;         /* The gap size in sectors.          */

   /* Set OSBoot  Record size -- always one Track */

   *OSBootRecSize = OSBOOT_REC_SIZE;

   /* Get Root Directory size in sectors */

   *RootDirSize = (DiskID->MediaDesBlock.NumberRootDirEntries * ENTRY_SIZE)
                  / DiskID->MediaDesBlock.SectorSize;

   /* Get the gap size in sectors. Physical sector offset is 0-base */

   GapSize = PConvertDiskAddr(&(CurrentGapEntry->EndAddr), 
                             DiskID->MediaDesBlock.SectorsPerTrack,
                             DiskID->MediaDesBlock.NumberOfHeads)
             - PConvertDiskAddr(&(CurrentGapEntry->OSBootStartAddr),
                             DiskID->MediaDesBlock.SectorsPerTrack,
                             DiskID->MediaDesBlock.NumberOfHeads)
             + 1;

   /* Get the FAT size in sectors, cluster size in sectors, and */
   /* FAT table bit nember -- 16 bit or 12 bit                  */

   GetFatClusterSize(FatSize, &ClusterSize, &GapSize, &FatBitNum, DiskID);
}

/*#######################################################################
#                                                                       #
# Function Name: GetFatClusterSize()                                    # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: UWORD *FatSize;                                        #
#                  -- Pointer to the FAT table size in sectors.         #
#                UWORD *ClusterSize;                                    #
#                  -- Pointer to the cluster size in sectors.           #
#                ULONG *PartSize;                                       #
#                  -- Pointer to the partition size in sectors.         #
#                UWORD *FatBitNum;                                      #
#                  -- Pointer to the number bits of the FAT.            #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: Get three things:                                      #
#                1). FAT Table bit number -- 12 bit or 16 bit           # 
#                2). Cluster size in sectors.                           # 
#                3). FAT Table size in sectors.                         # 
#                                                                       #
#######################################################################*/

void GetFatClusterSize( UWORD *FatSize, UWORD *ClusterSize, ULONG *PartSize,
	UWORD *FatBitNum, PDiskIDStruct *DiskID )
{
   ULONG PartSizeInBytes; /* Total partition size in bytes.  */
      
   /* Get partition size in bytes */

   PartSizeInBytes = (*PartSize) * DiskID->MediaDesBlock.SectorSize;

   /*                                                                    */
   /* Get the FAT table bit number -- 12 bit or 16 bit, and cluster size */
   /*                                                                    */

   GetClusterSize(PartSizeInBytes, FatBitNum, ClusterSize);

   /*                                                                    */
   /*                     Get the FAT table size                         */
   /*                                                                    */

   *FatSize = PGetFatSize(*PartSize, *ClusterSize, *FatBitNum, DiskID);      
}

/*#######################################################################
#                                                                       #
# Function Name: PartitionCreation()                                    # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: GapEntry *GapTable;                                    #
#                PDiskIDStruct DiskID;                                  #
#                                                                       #
#   Description: This function creates the individual partition.        #
#                The partition information is passed by the GapTalbe.   # 
#                                                                       # 
#######################################################################*/

ULONG PartitionCreation( GapEntry *GapTable, PDiskIDStruct *DiskID )
{
   ULONG PartitionSize;            /* Partition size in sectors        */
   LONG  ReturnAddr;
   ULONG MBRAddress;               /* MBR starting address.            */
   ULONG OSBootStartAddr;          /* OS Boot Record starting addresss */
   ULONG PartEndAddr;
   WellDoneDiskAddr CurrentMBRAddr;/* Extended MBR starting address    */
   WORD NonIndex;

   /* Get the Partition size in sectors : Starts from the OSBootRec    */
   /* to the end of the partition.                                     */

   PartitionSize = GapTable->GapSizeInSectors;
   
   /* Read the current MBR */

   if ((GapTable->LastMBRAddr == -1) || (GapTable->LastMBRAddr == -2))
   {
      MBRAddress = SECTOR_ZERO;
   }
   else
   {
      MBRAddress = PConvertDiskAddr(&(GapTable->StartAddr),
                                DiskID->MediaDesBlock.SectorsPerTrack,
                                DiskID->MediaDesBlock.NumberOfHeads);
   }
   OSBootStartAddr = PConvertDiskAddr(&(GapTable->OSBootStartAddr),
                                     DiskID->MediaDesBlock.SectorsPerTrack,
                                     DiskID->MediaDesBlock.NumberOfHeads);

   PartEndAddr = PConvertDiskAddr(&(GapTable->EndAddr),
                                     DiskID->MediaDesBlock.SectorsPerTrack,
                                     DiskID->MediaDesBlock.NumberOfHeads);

   if ((ErrorCode = PReadPhysicalSector(MBRAddress, SECTOR_ONE,
                            (UBYTE *) MBR, DiskID)) != SUCCESS)
   {
      return(ErrorCode);
   }

   /* Find the empty slot in the MBR, at this time, it will be first one */

   if (GapTable->LastMBRAddr == -1)
   { 
      if (MBR->Signature != SIGNATURE)
      {
         ZeroMBR((UBYTE *) MBR, DiskID->MediaDesBlock.SectorSize);

         /* Load the Master Boot Record Code */

         _move(hdboot, MBR->BootCode, 0x1BF);

      }
   }
   else if (GapTable->LastMBRAddr != -2) 
   { 
      ZeroMBR((UBYTE *) MBR, DiskID->MediaDesBlock.SectorSize);
   }     
      
   NonIndex = MBRNonOwnerType(MBR);
   
   /* Make the MBR and OS Boot Record */

   ModifyMBR(MBR, &(GapTable->StartAddr), &(GapTable->OSBootStartAddr),
                   &(GapTable->EndAddr), NonIndex, GapTable->LastMBRAddr,
                   DiskID);

   PMakeOSBootRec(OSBootStartAddr, PartEndAddr, DEFAULT_CLUSTER_SIZE, 
                  &GOSBootRec, DiskID);

   if ((GapTable->LastMBRAddr == -1) || (GapTable->LastMBRAddr == -2))
   {  /* This Gap is in the first MBR */
      
      /* Writing order is 1) OSBootRecord, 2) MBR */

      /* Write the OS Boot Record back to the disk */

      if ((RtnCode = PWritePhysicalSector(OSBootStartAddr, SECTOR_ONE,
                                   (UBYTE *) &GOSBootRec, DiskID)) != SUCCESS)
      {
         return(RtnCode);
      }

      /* Write the MBR back to the disk */

      if ((RtnCode = PWritePhysicalSector(MBRAddress, SECTOR_ONE,
                                   (UBYTE *) MBR, DiskID)) != SUCCESS)
      {
         return(RtnCode);
      }
   }
   else /* The gap is in the extended MBR */
   {
      /* Read the last MBR */

      if ((ErrorCode = PReadPhysicalSector(GapTable->LastMBRAddr,SECTOR_ONE,
                                    (UBYTE *) LastMBR, DiskID)) != SUCCESS)
      {
         return(ErrorCode);
      }
      
      /* Convert disk address from the DiskAddress to WellDonDiskAddrss */

      LoadAddr(&CurrentMBRAddr, &(GapTable->StartAddr));

      /* Rebuild the MBR's chain. */

      MakeExtOSBootRec(LastMBR, MBR, &CurrentMBRAddr, GapTable->LastMBRAddr,
                       &ReturnAddr, DiskID);

      /* Write OSBoot Record, Current MBR and last MBR back to the disk */

      /* Write the OS Boot Record back to the disk */

      if ((RtnCode = PWritePhysicalSector(OSBootStartAddr, SECTOR_ONE,
                                   (UBYTE *) &GOSBootRec, DiskID)) != SUCCESS)
      {
         return(RtnCode);
      }

      /* Write the current MBR back to the disk */

      if ((RtnCode = PWritePhysicalSector(MBRAddress, SECTOR_ONE,
                                   (UBYTE *) MBR, DiskID)) != SUCCESS)
      {
         return(RtnCode);
      }

      /* Write the last MBR back to the disk */
      if (ReturnAddr < 0)
      {
         MBRAddress = GapTable->LastMBRAddr;
      }
      else
      {
         MBRAddress = ReturnAddr;
      }	 

      if ((RtnCode = PWritePhysicalSector(MBRAddress, SECTOR_ONE,
                                   (UBYTE *) LastMBR, DiskID)) != SUCCESS)
      {
         return(RtnCode);
      }
   }
   return(SUCCESS); 
}

/*#######################################################################
#                                                                       #
# Function Name: LoadAddrFromMBR()                                      # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: DiskAddr         *SourceAddr;                          #
#                  -- Pointer to the unformated disk address.           #
#                WellDoneDiskAddr *TargetAddr;                          #
#                  -- Pointer to the Formated disk address.             #
#                                                                       #
#   Description: Convert a disk address from WellDoneDiskAddressto      #
#                DiskAddress.                                           #
#                                                                       #
#######################################################################*/

void LoadAddrFromMBR( DiskAddress *TargetAddr, WellDoneDiskAddr *SourceAddr )
{
   UWORD UpperByteCylinder;     /* Upper byte cylinder address of the */
                                /* parttion.                          */
   UBYTE LowerByteCylinder;     /* Lower byte cylinder address of the */
                                /* partition.                         */

   LowerByteCylinder = SourceAddr->Cylinder;
   UpperByteCylinder = (UWORD) SourceAddr->Sector & 0xC0;
   UpperByteCylinder = UpperByteCylinder << 2;

   TargetAddr->Head = SourceAddr->Head;
   TargetAddr->Sector = SourceAddr->Sector & 0x3F;
   TargetAddr->Cylinder = (UWORD) LowerByteCylinder | UpperByteCylinder;
}

/*#######################################################################
#                                                                       #
# Function Name: LoadAddr()                                             # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: WellDoneDiskAddr *TargetAddr;                          #
#                  -- Pointer to the Formated disk address.             #
#                DiskAddr         *SourceAddr;                          #
#                  -- Pointer to the unformated disk address.           #
#                                                                       #
#   Description: Convert a disk address from the DiskAddre to           #
#                WellDoneDiskAddress. (Combine sector and cylindr ).    # 
#                                                                       #
#######################################################################*/

void LoadAddr( WellDoneDiskAddr *TargetAddr, DiskAddress *SourceAddr )
{
   UWORD UpperByteCylinder;     /* Upper byte cylinder address of the */
                                /* parttion.                          */
   UBYTE LowerByteCylinder;     /* Lower byte cylinder address of the */
                                /* partition.                         */

   LowerByteCylinder = (UBYTE) SourceAddr->Cylinder & LOWER_BYTE;
   UpperByteCylinder = SourceAddr->Cylinder & UPPER_BYTE;
   UpperByteCylinder = UpperByteCylinder >> 2;

   TargetAddr->Head = SourceAddr->Head;
   TargetAddr->Sector = SourceAddr->Sector | (UBYTE) UpperByteCylinder;
   TargetAddr->Cylinder = LowerByteCylinder;
}

/*#######################################################################
#                                                                       #
# Function Name:  ModifyMBR()                                           # 
#                                                                       #
#  Return Value:  NULL.                                                 #
#                                                                       #
#   Environment:  Run as an support routine.                            # 
#                                                                       #  
#     Arguments: MasterBootRec *MBR;                                    #
#                  -- Pointer to MBR.                                   #
#                DiskAddress   *PartStartAddr;                          #
#                  -- OS Boot Record Starting address.                  #
#                DiskAddrss    *PartEndAddr;                            #
#                  -- Partition end address.                            #
#                UWORD BitIndex;                                        #
#                  -- Indicates which slot will be modified.            #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: This function writes the information to the MBR.       #
#                                                                       # 
#######################################################################*/

void ModifyMBR( MasterBootRec *MBR, DiskAddress *PartStartAddr,
	DiskAddress *OSBootStartAddr, DiskAddress *PartEndAddr, WORD BitIndex,
        ULONG Mode, PDiskIDStruct *DiskID )
{
   UWORD ClusterSize;           /* How many sectors per cluster.      */
   UWORD FatBitNum;             /* 12 bit FAT or 16 bit FAT.          */
   ULONG PartitionSizeInBytes;  /* Partition length in bytes.         */
   ULONG PartitionSizeInSectors;/* Partition length in sectors        */
   ULONG PartStartSectorOffset; /* Partition starting sector address  */
   ULONG OSBootStartSectorOffset;
   ULONG PartEndSectorOffset;   /* Partition end sector address.      */
  
   /* Get the partition length in bytes */

   PartStartSectorOffset = PConvertDiskAddr(PartStartAddr, 
                                       DiskID->MediaDesBlock.SectorsPerTrack,
                                       DiskID->MediaDesBlock.NumberOfHeads);
				       
   OSBootStartSectorOffset = PConvertDiskAddr(OSBootStartAddr, 
                                       DiskID->MediaDesBlock.SectorsPerTrack,
                                       DiskID->MediaDesBlock.NumberOfHeads);

   PartEndSectorOffset = PConvertDiskAddr(PartEndAddr, 
                                       DiskID->MediaDesBlock.SectorsPerTrack,
                                       DiskID->MediaDesBlock.NumberOfHeads);

   PartitionSizeInSectors = (PartEndSectorOffset - OSBootStartSectorOffset + 1);

   
   PartitionSizeInBytes = PartitionSizeInSectors
                          * DiskID->MediaDesBlock.SectorSize;

   /* Set MBR Boot Indicator */

   MBR->PartTable[BitIndex].BootIndor = INACTIVE;

   /* Load the partition beginning address */

   LoadAddr(&(MBR->PartTable[BitIndex].StartAddr), OSBootStartAddr);

   /* Load the owner type */

   GetClusterSize(PartitionSizeInBytes, &FatBitNum, &ClusterSize);

   MBR->PartTable[BitIndex].OwnerType = FatBitNum;

   /* Load the number of hidden sectors before partiton     */
   /* Usually this data is sector per track unless if there */
   /* are some bad sectors between MBR and OS Boot Record.  */

   if (Mode != -1 && Mode != -2)
   {  /* EBR -- Distance from the beginning of the partition to the */
      /*        beginning of the OS boot recort                     */

      MBR->PartTable[BitIndex].SectorNumBeforePart = 
                      OSBootStartSectorOffset - PartStartSectorOffset;
   }
   else
   {
      MBR->PartTable[BitIndex].SectorNumBeforePart = OSBootStartSectorOffset; 
   }
    
   /* Load the partition end address */

   LoadAddr(&(MBR->PartTable[BitIndex].EndAddr), PartEndAddr);

   /* Load the prtition length in sectors */

   MBR->PartTable[BitIndex].PartSize = PartitionSizeInSectors;

   /* Write the signature */

   MBR->Signature = SIGNATURE;

   DiskID->MediaDesBlock.FatID = HARD_DISK;
}

/*#######################################################################
#                                                                       #
# Function Name: MakeExtOSBootRec()                                     # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments: MasterBootRec *MBR, ExtMBR;                            #
#                WellDoneDiskAddr *ExtMBRAddr;                          #
#                   -- Pointer to the extended MBR Formated address.    #
#                ULONG MBRSectorOffset;                                 #
#                   -- 0 indicates the MBR is first MBR on the disk.    #
#                      Else not.                                        #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: Make OS Boot Record and link MBRs together.            #
#                                                                       # 
#######################################################################*/

ULONG MakeExtOSBootRec( MasterBootRec *MBR, MasterBootRec *ExtMBR,
	WellDoneDiskAddr *ExtMBRAddr, LONG MBRSectorOffset, LONG *ReturnAddr,
	PDiskIDStruct    *DiskID ) 
{
   WORD ExtExtIndex;
   WORD ExtBitIndex;
   WORD ExtNonIndex;
   
   WORD NonIndex;
   WORD BitIndex;
   WORD ExtIndex;
   
   WORD FirstExtIndex;
   ULONG FirstExtMBRAddr;
   DiskAddress FullSizeAddr;
   ULONG FullDiskSizeInSector;

   *ReturnAddr = -1;
   
   FullSizeAddr.Sector = DiskID->MediaDesBlock.SectorsPerTrack;
   FullSizeAddr.Cylinder = DiskID->MediaDesBlock.NumberOfCylinders - 1;
   FullSizeAddr.Head = DiskID->MediaDesBlock.NumberOfHeads - 1;
   
   FullDiskSizeInSector = PConvertDiskAddr( &FullSizeAddr,  
                          DiskID->MediaDesBlock.SectorsPerTrack,
			  DiskID->MediaDesBlock.NumberOfHeads) + 1;
			  
   ExtNonIndex = MBRNonOwnerType(ExtMBR);
   ExtBitIndex = MBRBitOwnerType(ExtMBR);
   ExtExtIndex = MBRExtOwnerType(ExtMBR);
   
   ExtIndex = MBRExtOwnerType(MBR);
   NonIndex = MBRNonOwnerType(MBR);
   BitIndex = MBRBitOwnerType(MBR);
   
   if (MBRSectorOffset == 0)
   {  /* ExtMBR is the first ext MBR */
   
      /* Link the first extended MBR to the main MBR */
      
      if (ExtIndex == NOT_FOUND)
      {
         MBR->PartTable[NonIndex].StartAddr.Head      =  ExtMBRAddr->Head;
         MBR->PartTable[NonIndex].StartAddr.Sector    =  ExtMBRAddr->Sector;
         MBR->PartTable[NonIndex].StartAddr.Cylinder  =  ExtMBRAddr->Cylinder;
   
         /* The distance form the beginning of the disk to the end of the */
         /* logical partition.                                            */

         FirstExtMBRAddr = ConvertDiskAddrFromMBR(
                             &(MBR->PartTable[NonIndex].StartAddr), DiskID);

         MBR->PartTable[NonIndex].SectorNumBeforePart = FirstExtMBRAddr;

         MBR->PartTable[NonIndex].PartSize = 
                            FullDiskSizeInSector - FirstExtMBRAddr;
   
         LoadAddr(&(MBR->PartTable[NonIndex].EndAddr), &FullSizeAddr); 

         MBR->PartTable[NonIndex].OwnerType = 05;

         return(SUCCESS);
      }
      else       
      {  /* ExtIndex is found */
      
         if ((ExtMBRAddr->Cylinder != MBR->PartTable[ExtIndex].StartAddr.Cylinder) ||
	     (ExtMBRAddr->Sector != MBR->PartTable[ExtIndex].StartAddr.Sector))
	 {   /* The creating partition is in the middle of the first */
	     /* extended partition which has been deleted.           */
	    
	    FirstExtMBRAddr = ConvertDiskAddrFromMBR(
	                      &(MBR->PartTable[ExtIndex].StartAddr), DiskID);
			      
            if ((ErrorCode = PReadPhysicalSector(FirstExtMBRAddr, SECTOR_ONE,
	                      (UBYTE *) MBR, DiskID)) != SUCCESS)
            {
	       return(ErrorCode);
	    }
            *ReturnAddr =FirstExtMBRAddr;
         }
	 else
	 {  /* The creating partition is the same location as before */
	 
	    return(SUCCESS);
	 }
      }	 
   }
	    	     
   if (ExtIndex != NOT_FOUND)
   {		     
      _move(&(MBR->PartTable[ExtIndex]), &(ExtMBR->PartTable[ExtNonIndex]),
                                                   sizeof(PartitionTable) );
   }   	       			      
   else
   {
      ExtIndex = NonIndex;
   }

   /* Get the first MBR and the first extended MBR sector offset */

   if ((ErrorCode = PReadPhysicalSector((long)SECTOR_ZERO, SECTOR_ONE,
                                      (UBYTE *) FirstMBR, DiskID)) != SUCCESS)
   {
      return(ErrorCode);
   }

   FirstExtIndex = MBRExtOwnerType(FirstMBR);

   FirstExtMBRAddr = FirstMBR->PartTable[FirstExtIndex].SectorNumBeforePart;
      
   /* Link the extended MBR to the Last MBR */
   
   MBR->PartTable[ExtIndex].StartAddr.Head      =  ExtMBRAddr->Head;
   MBR->PartTable[ExtIndex].StartAddr.Sector    =  ExtMBRAddr->Sector;
   MBR->PartTable[ExtIndex].StartAddr.Cylinder  =  ExtMBRAddr->Cylinder;

   MBR->PartTable[ExtIndex].OwnerType = 05;

   MBR->PartTable[ExtIndex].EndAddr.Head      =  
                            ExtMBR->PartTable[ExtBitIndex].EndAddr.Head;
   MBR->PartTable[ExtIndex].EndAddr.Sector    =  
                            ExtMBR->PartTable[ExtBitIndex].EndAddr.Sector;
   MBR->PartTable[ExtIndex].EndAddr.Cylinder  =  
                            ExtMBR->PartTable[ExtBitIndex].EndAddr.Cylinder;
      	    
   MBR->PartTable[ExtIndex].SectorNumBeforePart = 
              ConvertDiskAddrFromMBR(ExtMBRAddr, DiskID) - FirstExtMBRAddr;

   MBR->PartTable[ExtIndex].PartSize = 
                       ExtMBR->PartTable[ExtBitIndex].PartSize
                       + ExtMBR->PartTable[ExtBitIndex].SectorNumBeforePart;
   return(SUCCESS);
}   			      			      
	   
/*#######################################################################
#                                                                       #
# Function Name: CreateUserSizePart()                                   # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: UWORD         MenuDriven;                              #
#                  -- Indicates using screen driven or command line     #
#                     options.                                          #
#                PDiskIDStruct *DiskID;                                 #
#                PParseStruct *ParseTable;                              #
#                                                                       #
#   Description: Creates user defined size partition.                   #
#                                                                       #
#######################################################################*/

ULONG CreateUserSizePart( UWORD MenuDriven, PDiskIDStruct *DiskID, 
	PCommandLineOptions *ParseTable )
{
   GapEntry *DiskGapList;     /* The head of the disk gap link list.   */
   GapEntry *CurrentGapEntry; /* The current gap entry in the list     */
   int      InMsg[2];         /* Message come from the screen.         */
   ULONG    UserSizeInSector; /* User defined size in sectors.         */
   ULONG    UserSizeInCylinder; /* User defined size in cylinders.     */
   ULONG    UserSizeInMeg;    /* User defined size in MegBytes.        */
   ULONG    StartCylinder;    /* User defined starting cylinder address*/
   UWORD    Found;            /* indicates if found the user required  */
                              /* Gap.                                  */
   int	    MadeAtLeastOne[1];/*LDT*/
   ULONG    GapStartSectorOffset, GapEndSectorOffset;
   ULONG    MegSize;
   PLocation Line;
   
   Line.Row = 9;
   Line.Column = 1;

   /*                                                                  */
   /*              Part 1 -- Allocate a head of the link list          */
   /*                                                                  */
   DiskGapList = (GapEntry *) malloc(sizeof(GapEntry));
   MadeAtLeastOne[0] = 0; /*LDT*/
    
   do
   {
      while(1)
      {

         DiskGapList->NextGapEntry = NULL;

         /*                                                                  */
         /*              Part 2 -- Find gaps on the disk                     */
         /*                                                                  */
    
         if ((ErrorCode = FindDiskGaps(DiskID, DiskGapList)) != SUCCESS)
         {
            return(ErrorCode);
         }

         if (DiskGapList->NextGapEntry != NULL)
         {
/*LDT*/     if (MenuDriven == ON)
            {
		MadeAtLeastOne[0] = 1;
                FDiskScreen(40, NULLPTR, DiskID->DriveName, NULLPTR, NULLPTR);
            }
   
            InMsg[0] = (int) DiskGapList->NextGapEntry->StartAddr.Cylinder;
            InMsg[1] = (int) (DiskGapList->NextGapEntry->EndAddr.Cylinder
                             - DiskGapList->NextGapEntry->StartAddr.Cylinder
	                     + 1);
            MegSize = ((ULONG) InMsg[1] 
                     * (ULONG) DiskID->MediaDesBlock.NumberOfHeads
                     * (ULONG) DiskID->MediaDesBlock.SectorsPerTrack
		     * (ULONG) DiskID->MediaDesBlock.SectorSize)
		     / MEG_1;
         }
         else
         {
            if (MenuDriven == ON)
	    {
/*LDT*/        FDiskScreen(44, MadeAtLeastOne, NULLPTR, NULLPTR, NULLPTR);
   	       free(DiskGapList);
	       return(SUCCESS);
            }
            else 
            {
               return(NOGAP);
            }
         }
			  
         /*                                                                  */
         /*              Part 3 -- Display the screen 4                      */
         /*                                                                  */

         if (MenuDriven == OFF)
         {  /* Using the command line option */
      
            StartCylinder = ParseTable->ArgpPtr[S_OPTION]->Result;
            UserSizeInCylinder = ParseTable->ArgpPtr[N_OPTION]->Result;
         }
         else /* Using the screen */
         { 
            /* Get the drive name */ 

            if ((ErrorCode = FDiskScreen(41, InMsg, NULLPTR, NULLPTR,
                                                     DiskID)) != SUCCESS)
            {
               return(ErrorCode);
            }

         /*                                                                  */
         /*          Part 4 -- Get the starting address and size             */
         /*                                                                  */
            if (InMsg[0] == FAILURE)
            {  /* ESC from user */
 
               free(DiskGapList);
               return(SUCCESS);
            }

            /* Check the data legalization */

            StartCylinder = InMsg[0];
            UserSizeInCylinder = InMsg[1];
         }

         /* Convert the partitin size in cylinders to siz in MEgByts */

         UserSizeInSector = (DiskID->MediaDesBlock.SectorsPerTrack
                             * DiskID->MediaDesBlock.NumberOfHeads
                             * UserSizeInCylinder)
                             - DiskID->MediaDesBlock.SectorsPerTrack;

         UserSizeInMeg = ConvertCylinderSizeToMeg(UserSizeInCylinder, DiskID);

         /*                                                                  */
         /*      Part 5 -- Check if there is gap match the user requirement  */
         /*                                                                  */
   
         CurrentGapEntry = DiskGapList->NextGapEntry;

         Found = OFF;

         while (CurrentGapEntry != NULL)
         {
            if (CurrentGapEntry->StartAddr.Cylinder == StartCylinder &&
                CurrentGapEntry->GapSizeInSectors >= UserSizeInSector  )
            {   /* There is a gap match the user requirement */

               Found = ON;
               break;
            }
            else if (CurrentGapEntry->EndAddr.Cylinder > StartCylinder &&
                     CurrentGapEntry->StartAddr.Cylinder < StartCylinder )
            {
               CurrentGapEntry->StartAddr.Cylinder = StartCylinder;
	 
               CurrentGapEntry->OSBootStartAddr.Cylinder = StartCylinder;
	 
               GapStartSectorOffset = PConvertDiskAddr(
	                                &(CurrentGapEntry->OSBootStartAddr),
                                        DiskID->MediaDesBlock.SectorsPerTrack,
                                        DiskID->MediaDesBlock.NumberOfHeads);
         
               GapEndSectorOffset = PConvertDiskAddr(
	                                &(CurrentGapEntry->EndAddr),
                                        DiskID->MediaDesBlock.SectorsPerTrack,
                                        DiskID->MediaDesBlock.NumberOfHeads);

              CurrentGapEntry->GapSizeInSectors = GapEndSectorOffset 
                                                  - GapStartSectorOffset 
                                                  + 1;
           }
           else
           {
              CurrentGapEntry = CurrentGapEntry->NextGapEntry;
           }
        }

        if (Found == OFF)
        {   /* not found a gap */

           if (MenuDriven == ON)
           {
              FDiskScreen(43, NULLPTR, NULLPTR, NULLPTR, DiskID);
              Line.Row = 22;
              Line.Column = 1;
	      PEraseLineToEnd(Line);
	      break;
           }
           else /* User uses the command line options */
           {
              return(NOSUCHGAPEXIST);
           }
        }
  
        if (MenuDriven == ON)
        {    

            /* display th size in MegBytes on the screen  */

            FDiskScreen(42, &InMsg[0], NULLPTR, NULLPTR, DiskID);
            if (InMsg[0] == FALSE)
            {
	       free(DiskGapList);
               return(SUCCESS);
            }
            else if (InMsg[0] == ESC)
            {
               break;
            }	 
        }
    
        /*                                                                  */
        /*                       Part 6 --  creation                        */
        /*                                                                  */

        /* Partition Creation */

        /* modify the gap entry */

        CurrentGapEntry->EndAddr.Head = DiskID->MediaDesBlock.NumberOfHeads-1;

        CurrentGapEntry->EndAddr.Cylinder = CurrentGapEntry->StartAddr.Cylinder
                                             + UserSizeInCylinder - 1;

        CurrentGapEntry->EndAddr.Sector = DiskID->MediaDesBlock.SectorsPerTrack;

        CurrentGapEntry->GapSizeInSectors = UserSizeInSector;

        /* Create that gap as partition */

        if ((ErrorCode=PartitionCreation(CurrentGapEntry,DiskID))!=SUCCESS)
        { 
           return(ErrorCode);
        }
        break;
      }
      PClearRectangle();
   
   } while(MenuDriven);
   return(SUCCESS);
}

/*#######################################################################
#                                                                       #
# Function Name: ConvertCylinderSizeToMeg();                            # 
#                                                                       # 
#  Return Value:  ULONG -- MegBytes of cylinder size                    #
#                                                                       #
#   Environment:  Run as a support routine.                             # 
#                                                                       #  
#     Arguments:  ULONG CylinderSize;                                   #
#                   -- Number of the cylinders.                         #
#                 PDiskIDStruct *PDiskID;                               #
#                                                                       #
#   Description:  Convert the size in cylinders into MegBytes.          #
#                                                                       # 
#######################################################################*/

ULONG ConvertCylinderSizeToMeg( ULONG CylinderSize, PDiskIDStruct *DiskID )
{
   ULONG MegSize;

   MegSize = (
               CylinderSize
                * (ULONG) DiskID->MediaDesBlock.SectorSize
                * (ULONG) DiskID->MediaDesBlock.SectorsPerTrack
                * (ULONG) DiskID->MediaDesBlock.NumberOfHeads
             ) / MEG_1;
   return(MegSize + 1);
}

/*#######################################################################
#                                                                       #
# Function Name: CreateWholeDiskPart()                                  # 
#                                                                       # 
#  Return Value: 0 for SUCCESS, else Error Code.                        #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: Creates the first gap as a partition.                  #
#                                                                       #
#######################################################################*/

ULONG CreateWholeDiskPart( PDiskIDStruct *DiskID )
{
   GapEntry *CurrentGapEntry; /* The current gap entry in the list     */

   CurrentGapEntry = (GapEntry *) malloc(sizeof(GapEntry));

   CurrentGapEntry->StartAddr.Sector   = 1;
   CurrentGapEntry->StartAddr.Cylinder = 0;
   CurrentGapEntry->StartAddr.Head     = 0;

   CurrentGapEntry->EndAddr.Sector  = DiskID->MediaDesBlock.SectorsPerTrack;
   CurrentGapEntry->EndAddr.Cylinder=DiskID->MediaDesBlock.NumberOfCylinders-1;
   CurrentGapEntry->EndAddr.Head    = DiskID->MediaDesBlock.NumberOfHeads - 1;

   CurrentGapEntry->LastMBRAddr = -1;

   if ((ErrorCode = CheckDisk(CurrentGapEntry, DiskID)) != SUCCESS)
   {
      return(BADDISK);
   }

   CurrentGapEntry->GapSizeInSectors = 
                    PConvertDiskAddr(&(CurrentGapEntry->EndAddr),
                                     DiskID->MediaDesBlock.SectorsPerTrack,
                                     DiskID->MediaDesBlock.NumberOfHeads)
                    -PConvertDiskAddr(&(CurrentGapEntry->OSBootStartAddr),
                                     DiskID->MediaDesBlock.SectorsPerTrack,
                                     DiskID->MediaDesBlock.NumberOfHeads)
                    + 1;
   CurrentGapEntry->NextGapEntry = NULL;

   if ((ErrorCode = PReadPhysicalSector((LONG) SECTOR_ZERO, SECTOR_ONE, 
                                        (UBYTE *) MBR, DiskID)) != SUCCESS)
   {
      return(ErrorCode);
   }
   
   ZeroMBR((UBYTE *) MBR, DiskID->MediaDesBlock.SectorSize);

   if ((RtnCode = PWritePhysicalSector((LONG) SECTOR_ZERO, SECTOR_ONE,
                                         (UBYTE *) MBR, DiskID)) != SUCCESS)
   {
      return(RtnCode);
   }

   if ((ErrorCode=PartitionCreation(CurrentGapEntry,DiskID))!=SUCCESS)
   {
      return(ErrorCode);
   }

   return(SUCCESS);
}

/*#######################################################################
#                                                                       #
# Function Name: DeletePartition()                                      # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments:  UWORD MenuDriven;                                     #
#                   -- Indicates using screen driven or command line    #
#                      options.                                         #
#                 PDiskIDStruct *DiskID;                                #
#                 PCommandLineOptions *ParseTable;                      #
#                                                                       #
#   Description:  Delete user required partition.                       #
#                                                                       # 
#######################################################################*/

ULONG DeletePartition( UWORD MenuDriven, PDiskIDStruct *DiskID,
	PCommandLineOptions *ParseTable )
{
   ULONG SectorOffset;        /* Sector offset for reading and writing */
   ULONG LastMBRSectorOffset; /* Last MBR Sector offset for I/O        */
   UWORD SectorCount;         /* Sectors will be read or written.      */
   int   PartNum;             /* Partition number will be deleted.     */
   ULONG ChaseCount;          /* counter for chase the MBR chain.      */
   int   InMsg;               /* Message comes from the screen.        */
   WORD  ExtIndex, LastExtIndex, FirstExtIndex;
   WORD  BitIndex, LastBitIndex, FirstBitIndex;
   UWORD Finished;
   UWORD Result;

   WORD  IndexArray[4], NumOfPart, Index1, Index2;

   if (MenuDriven == ON)
   {
      /* Get the drive name */

      if ((ErrorCode = FDiskScreen(61, &InMsg, DiskID->DriveName, NULLPTR,
                                                       DiskID)) != ErrorCode)
      {
         return(ErrorCode);
      }
      if (InMsg == ESC)
      {
         return(SUCCESS);
      }	 
   }      

   if ((ErrorCode = PReadPhysicalSector((LONG) SECTOR_ZERO, SECTOR_ONE,
                                      (UBYTE *) FirstMBR, DiskID)) != SUCCESS)
   {
      return(ErrorCode);
   }

   if ((MBRBitOwnerType(FirstMBR) == NOT_FOUND) &&
       (MBRExtOwnerType(FirstMBR) == NOT_FOUND) )
   {
      if (MenuDriven == ON)
      {
         FDiskScreen(65, NULLPTR, NULLPTR, NULLPTR, NULLPTR);
         return(SUCCESS);
      }
      else 
      {
         return(NOSUCHPARTITIONEXIST);
      }
   }
         
   Index1 = 0;
   Index2 = 0;

   while(Index1 < NUM_PARTITION_ENTRY)
   {
      if (FirstMBR->PartTable[Index1].OwnerType != 0)
      {
         Index2 += 1;
      }
      Index1 += 1;
   }

   if (Index2 == 0)
   {  /* No Partition on the disk */

      if (MenuDriven == ON)
      {
         PDisplay(&DiskUnPartObj);
         PDisplay(&EscapePosObj);
         PGetString(NULLPTR, 0, 0);
         return(SUCCESS);
      }
      else 
      {
         return(NOPARTONDISK);
      }
   }

   do
   {
      /* Chase the MBR chain */

      SectorOffset = SECTOR_ZERO;
      SectorCount  = SECTOR_ONE;
      ChaseCount = 1;
      LastMBRSectorOffset = 0;
      Finished = FALSE; 
      InMsg = TRUE;

      while(1)
      {
         /* Get the partition number which will be deleted */
   
         if (MenuDriven == ON)
         {
            PDisplay( &DeleteNumberObj );
            PartNum = PGetNum(1, 99, -1);

            if (PartNum == FAILURE)
            {
               return(SUCCESS);
            }
         }
         else 
         {
            PartNum = ParseTable->ArgpPtr[N_OPTION]->Result;
 
            /* Check for partition number legalization */

            if (PartNum <= 0)
            {
               return(NOSUCHPARTITIONEXIST);
            }
         } 

         if ((ErrorCode = PReadPhysicalSector((LONG) SECTOR_ZERO, SECTOR_ONE,
                                      (UBYTE *) FirstMBR, DiskID)) != SUCCESS)
         {
            return(ErrorCode);
         }

         FirstBitIndex = MBRBitOwnerType(FirstMBR);
         FirstExtIndex = MBRExtOwnerType(FirstMBR);

         /* Get the number of the partitions in the first MBR (Index2) */

         Index1 = 0;
         Index2 = 0;

         while(Index1 < NUM_PARTITION_ENTRY)
         {
            if (FirstMBR->PartTable[Index1].OwnerType != 0 &&
                FirstMBR->PartTable[Index1].OwnerType != 5 )
            {
               Index2 += 1;
            }
            Index1 += 1;
         }

         /*******************************************************************/
         /*                                                                 */
         /*                         Check First MBR                         */
         /*                                                                 */

         if (PartNum <= Index2)
         {  /* There is at least one partition in the first MBR and         */
            /* User wants to delete the partition which may be in the first MBR */

            ReOrderMBR(IndexArray, &NumOfPart, FirstMBR, DiskID);

            PartNum = IndexArray[PartNum - 1];

            if (MenuDriven == ON)
            {
               /* Confirm with user */

               FDiskScreen(62, &InMsg, NULLPTR, NULLPTR, DiskID);
               if (InMsg == ESC)
               {
                  return(SUCCESS);
               }
               else if (InMsg == FALSE)
               {
                  break;
               }
            }

            if ((Index2 == 1) && (FirstExtIndex == NOT_FOUND))
            {   /* Only one partition on the disk */
 
                ZeroMBR((UBYTE *) FirstMBR, DiskID->MediaDesBlock.SectorSize);
                Finished = TRUE;
            }
            else /* There is at least one foreign parition on the first MBR */
            {
                ZeroMBRSlot(FirstMBR, PartNum);
            }
            if ((RtnCode = PWritePhysicalSector((LONG) SECTOR_ZERO,
                       SECTOR_ONE,(UBYTE *) FirstMBR, DiskID)) != SUCCESS)
            {
               return(RtnCode);
            }
            break;
         }
         else if (FirstExtIndex == NOT_FOUND)
         {  /* The deleting partition is not in the first MBR */
            /* and Extended partition is not found            */

            if (MenuDriven == ON)
            {
               FDiskScreen(63, NULLPTR, NULLPTR, NULLPTR, NULLPTR);
               break;
            }
            else 
            {
               return(NOSUCHPARTITIONEXIST);
            }
         }
         else /* Extended partition is found */
         {
            SectorOffset = ConvertDiskAddrFromMBR(
                      &(FirstMBR->PartTable[FirstExtIndex].StartAddr), DiskID);

            if ((ErrorCode = PReadPhysicalSector(SectorOffset, SectorCount,
                                       (UBYTE *) LastMBR, DiskID)) != SUCCESS)
            {
               return(ErrorCode);
            }

            LastBitIndex = MBRBitOwnerType(LastMBR);
            LastExtIndex = MBRExtOwnerType(LastMBR);

            if (LastBitIndex != NOT_FOUND)
            {
               Index2 += 1;
            }
         }

         /*******************************************************************/
         /*                                                                 */
         /*                         Check Second MBR                        */
         /*                                                                 */

         if (PartNum == Index2)
         {  /* There is DOS or FlexOS partition in the second MBR */
            /* And user wants to delete this partition            */

            if (MenuDriven == ON)
            {
               /* Confirm with user */
               FDiskScreen(62, &InMsg, NULLPTR, NULLPTR, DiskID);
               if (InMsg == ESC)
               {
                  return(SUCCESS);
               }
               else if (InMsg == FALSE)
               {
                  break;
               }
            }

            if (LastExtIndex == NOT_FOUND)
            {  /* No Extended partition in the current MBR */

               /* Zero out the second MBR */

               ZeroMBR((UBYTE *) LastMBR, DiskID->MediaDesBlock.SectorSize);

               if((RtnCode = PWritePhysicalSector(SectorOffset,
    	                   SectorCount, (UBYTE *) LastMBR, DiskID)) != SUCCESS)
               {
                  return(RtnCode);
               }

               if (Index2 == 1)
               {  /* First MBR only has one extended partition */

                  ZeroMBR ((UBYTE *) FirstMBR, DiskID->MediaDesBlock.SectorSize);
                  Finished = TRUE;
               }
               else
               {
                  ZeroMBRSlot(FirstMBR, FirstExtIndex);
               }

               if((RtnCode = PWritePhysicalSector((long)SECTOR_ZERO,
                	   SECTOR_ONE, (UBYTE *) FirstMBR, DiskID)) != SUCCESS)
               {
	          return(RtnCode);
               }
            }
            else /* Extended partition is found in the second MBR */
            {    /* Zero out the partition slot in the second MBR */
                     
               ZeroMBRSlot(LastMBR, LastBitIndex);

               if((RtnCode = PWritePhysicalSector((long)SectorOffset,
			   SectorCount, (UBYTE *) LastMBR, DiskID)) != SUCCESS)
	       {
	          return(RtnCode);
               }
            }
            break;
         }
         else /* The deleting partition is not in the second MBR */
         {
            if (LastExtIndex == NOT_FOUND)
            {  /* Extended partition is not found in the second MBR */

               if (MenuDriven == ON)
               {
                  FDiskScreen(63, NULLPTR, NULLPTR, NULLPTR, NULLPTR);
                  break;
               }
               else 
               {
                  return(NOSUCHPARTITIONEXIST);
               }
            }
            else /* Extended partition is found in the second MBR */
            {
               LastMBRSectorOffset = SectorOffset;
               SectorOffset = ConvertDiskAddrFromMBR(
                        &(LastMBR->PartTable[LastExtIndex].StartAddr), DiskID);
 
               if ((ErrorCode = PReadPhysicalSector(SectorOffset, 
                             SectorCount,(UBYTE *) MBR, DiskID)) != SUCCESS)
               {
                  return(ErrorCode);
               }

               BitIndex = MBRBitOwnerType(MBR);
               ExtIndex = MBRExtOwnerType(MBR);
                  
               Index2 += 1;
            }
         }
 
         /*******************************************************************/
         /*                                                                 */
         /*                         Check Third MBR                         */
         /*                                                                 */

         if (PartNum == Index2)
         {  /* Deleting third partition */

            if (MenuDriven == ON)
            {
               /* Confirm with user */
               FDiskScreen(62, &InMsg, NULLPTR, NULLPTR, DiskID);
               if (InMsg == ESC)
               {
                  return(SUCCESS);
               }
               else if (InMsg == FALSE)
               {
                  break;
               }
            }
                     
            if (ExtIndex == NOT_FOUND)
            {  /* No Extended partition found in current MBR */
                                    
               if (LastBitIndex == NOT_FOUND)
               {  /* No partition found in the second MBR */
                  /* Zero out the second MBR */

                  ZeroMBR((UBYTE *) LastMBR, DiskID->MediaDesBlock.SectorSize);

                  if (Index2 == 1)
                  {  /* Only extended partition in the first MBR */

                     ZeroMBR((UBYTE *) FirstMBR, DiskID->MediaDesBlock.SectorSize);
                     Finished = TRUE;
                  }
                  else /* More than one partition in the first MBR */
                  {    /* Zero out the extended slot in the 1th MBR */

                     ZeroMBRSlot(FirstMBR, FirstExtIndex);
                  }			 

                  if((RtnCode = PWritePhysicalSector((long)SECTOR_ZERO,
		        SECTOR_ONE, (UBYTE *) FirstMBR, DiskID)) != SUCCESS)
                  {
                     return(RtnCode);
                  }
               }	
               else /* Partition found in the second MBR */
               {    /* Zero out the extended slot in the seconf MBR */

                  ZeroMBRSlot(LastMBR, LastExtIndex);
               }
                        		  
               if ((RtnCode = PWritePhysicalSector(LastMBRSectorOffset,
	               SectorCount, (UBYTE *) LastMBR, DiskID)) != SUCCESS)
               {
                  return(RtnCode);
               }
            }
            else /* Extended partition found in the current MBR */
            {   
               _move(&(MBR->PartTable[ExtIndex]), 
                     &(LastMBR->PartTable[LastExtIndex]), 
                     sizeof(PartitionTable));
                  
               if ((RtnCode = PWritePhysicalSector(LastMBRSectorOffset,
	                SECTOR_ONE, (UBYTE *) LastMBR, DiskID)) != SUCCESS)
               {
                  return(RtnCode);
               }
            }
               
            ZeroMBR((UBYTE *) MBR, DiskID->MediaDesBlock.SectorSize);
                     
            if ((RtnCode = PWritePhysicalSector(SectorOffset, 
                        SectorCount, (UBYTE *) MBR, DiskID)) != SUCCESS)
            {
               return(RtnCode);
            }
            break;
         }
         else if (ExtIndex == NOT_FOUND)
         {  /* The deleting partition is not in the third MBR */

            if (MenuDriven == ON)
            {
               FDiskScreen(63, NULLPTR, NULLPTR, NULLPTR, NULLPTR);
               break;
            }
            else 
            {
               return(NOSUCHPARTITIONEXIST);
            }
         }

         /*******************************************************************/
         /*                                                                 */
         /*                         Check Rest MBR                          */
         /*                                                                 */
       
         while(1)
         {
            LastMBRSectorOffset = SectorOffset;
            LastExtIndex = ExtIndex;
            _move(MBR, LastMBR, DiskID->MediaDesBlock.SectorSize);

            SectorOffset = ConvertDiskAddrFromMBR(
                                &(MBR->PartTable[ExtIndex].StartAddr), DiskID);
   
            if ((ErrorCode = PReadPhysicalSector(SectorOffset,
                                SectorCount, (UBYTE *) MBR, DiskID)) != SUCCESS)
            {
               return(ErrorCode);
            }

            BitIndex = MBRBitOwnerType(MBR);
            ExtIndex = MBRExtOwnerType(MBR);
            Index2 += 1;

            if (PartNum == Index2)
            {
               if (MenuDriven == ON)
               {
                  /* Confirm with user for the deletion */

                  FDiskScreen(62, &InMsg, NULLPTR, NULLPTR, DiskID);

                  if (InMsg == ESC)
                  {  /* User cancle the deletion */
   
                     return(SUCCESS);
                  }
                  else if (InMsg == FALSE)
                  {
                     break;
                  }
               }

               if (ExtIndex != NOT_FOUND)
      	       {
                  _move(&(MBR->PartTable[ExtIndex]), 
	                &(LastMBR->PartTable[LastExtIndex]), 
                        sizeof(PartitionTable));

                  /* Write the last MBR back to the disk */
               }
	       else
	       {
	          ZeroMBRSlot(LastMBR, LastExtIndex);
	       }
               if((RtnCode = PWritePhysicalSector(LastMBRSectorOffset,
                            SectorCount, (UBYTE *) LastMBR, DiskID)) != SUCCESS)
               {
                  return(RtnCode);
               }
	       
	       ZeroMBR((UBYTE *) MBR, DiskID->MediaDesBlock.SectorSize);
	       
    	       if ((RtnCode = PWritePhysicalSector(SectorOffset, 
                                SectorCount, (UBYTE *) MBR, DiskID)) != SUCCESS)
               {
	          return(RtnCode);
               }
               break;
            }
            else /* ChaseCount != PartNum */
            {
               /* Check if this is end of MBR chain */
  
               if (ExtIndex == NOT_FOUND)
               {  /* End of the MBR chain */

                  /* Partition searching failed */
                  if (MenuDriven == ON)
                  {
                     FDiskScreen(63, NULLPTR, NULLPTR, NULLPTR, NULLPTR);
                     break;
                  }
                  return(NOSUCHPARTITIONEXIST);
               }
            }
         }
         break;
      }

      if (Finished == TRUE)
      {
         if (MenuDriven == ON)
         {
            PClearRectangle();
            FDiskScreen(65, NULLPTR, NULLPTR, NULLPTR, NULLPTR);
         }
         return(SUCCESS);
      }
      else
      {                
         if (MenuDriven == ON)
         {	    
            PClearRectangle();
	               
            ViewDisplay(&Result, ALLMBR, DiskID);
	    
            if (Result == ESC)
            {
               return(SUCCESS);
            }
         }
         else
         {
            return(SUCCESS);
         }
      }	    
   } while(MenuDriven);   
   return(SUCCESS);
}

/*#######################################################################
#                                                                       #
# Function Name: ZeroMBR()                                              # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments: UBYTE *MBRBuffer;                                      #
#                  -- Pointer to the MBR Buffer.                        #
#                UWORD BytesCount;                                      #
#                  -- Buffer length in bytes.                           #
#   Description: Zero out the MBR buffer.                               #
#                                                                       # 
#######################################################################*/

void ZeroMBR( UBYTE *MBRBuffer, UWORD BytesCount )
{
   WORD Index = 0;

   while (Index < BytesCount)
   {
      *(MBRBuffer + Index) = 0;
      Index += 1;
   }
}

/*#######################################################################
#                                                                       #
# Function Name: ZeroMBRSlot()                                          # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments: MasterBootRec *MBR;                                    #
#                                                                       #
#   Description: Zero out the partiton slot                             #
#                                                                       # 
#######################################################################*/

void ZeroMBRSlot( MasterBootRec *MBR, WORD Index )
{
   MBR->PartTable[Index].OwnerType           = 0;
   MBR->PartTable[Index].BootIndor           = 0;
   MBR->PartTable[Index].StartAddr.Sector    = 0;
   MBR->PartTable[Index].StartAddr.Head      = 0;
   MBR->PartTable[Index].StartAddr.Cylinder  = 0;
   MBR->PartTable[Index].EndAddr.Sector      = 0;
   MBR->PartTable[Index].EndAddr.Head        = 0;
   MBR->PartTable[Index].EndAddr.Cylinder    = 0;
   MBR->PartTable[Index].PartSize            = 0;
   MBR->PartTable[Index].SectorNumBeforePart = 0;
}

/*#######################################################################
#                                                                       #
# Function Name: ConvertDiskAddrFromMBR()                               # 
#                                                                       # 
#  Return Value: ULONG Sector Offset of MBR.                            #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments: WellDoneDiskAddr *MBRAddr;                             #
#                PDiskIDStruct *DiskID;                                 #
#   Description: Convert MBR Address in head, sector, cylinder into     #
#                sectorOffset.                                          # 
#######################################################################*/

ULONG ConvertDiskAddrFromMBR( WellDoneDiskAddr *MBRAddr, PDiskIDStruct *DiskID )
{
   DiskAddress DiskAddr;

   DiskAddr.Head = MBRAddr->Head;
   DiskAddr.Sector = MBRAddr->Sector & 0x3F;
   DiskAddr.Cylinder = 0;
   DiskAddr.Cylinder = (UWORD) (MBRAddr->Sector & 0xC0);
   DiskAddr.Cylinder = DiskAddr.Cylinder << 2;
   DiskAddr.Cylinder = DiskAddr.Cylinder | (UWORD) MBRAddr->Cylinder;
   return(PConvertDiskAddr(&DiskAddr, DiskID->MediaDesBlock.SectorsPerTrack,
                          DiskID->MediaDesBlock.NumberOfHeads));
}

/*#######################################################################
#                                                                       #
# Function Name: ActivatePartition()                                    # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: UWORD MenuDriven;                                      #
#                  -- Indicates using screen driven or command line     #
#                     options.                                          #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: Activates the user required partition.                 #
#                                                                       # 
#######################################################################*/

ULONG ActivatePartition( UWORD MenuDriven, PCommandLineOptions *ParseTable,
	PDiskIDStruct *DiskID )
{
   int   InMsg;                /* Message come from the screen        */
   ULONG PartitionNum; 
   WORD  BitIndex;
   int   Index, Found;

   int IndexArray[4], NumOfPart;

   if ((ErrorCode = PReadPhysicalSector((long)SECTOR_ZERO, SECTOR_ONE,
                                  (UBYTE *) MBR, DiskID)) != SUCCESS)
   {
      return(ErrorCode);
   }

   if (MenuDriven == ON)
   {
      /* FDiskScreen(51) check the Input value is in the range of the */
      /* current partition number.                                    */

      if ((ErrorCode = FDiskScreen(51, &InMsg, DiskID->DriveName, NULLPTR, 
                                                         DiskID)) != SUCCESS)
      {
         return(ErrorCode);
      }
      if (InMsg == ESC)
      {
         return(SUCCESS);
      }
   }

   BitIndex = MBRBitOwnerType(MBR);

   /* check if ODS or FlesOS partition in the first MBR */

   if (BitIndex == NOT_FOUND)
   {
      if (MenuDriven == ON)
      {
         FDiskScreen(55, NULLPTR, NULLPTR, NULLPTR, NULLPTR);
         return(SUCCESS);
      }
      else
      {
         return(NOPRIMEPART);
      }
   }

   if (MenuDriven == ON)
   {            
      while(1)
      {
         PDisplay( &EnterNewPartitionObj );
         InMsg = PGetNum(1, 3, 1);
      
         /* Check for ESC */
 
         if (InMsg == FAILURE)
         {
            return(SUCCESS);
         }

         if ((InMsg <= NUM_PARTITION_ENTRY) &&
             (MBR->PartTable[InMsg - 1].OwnerType != BIT12)  &&
             (MBR->PartTable[InMsg - 1].OwnerType != BIT_16) && 
             (MBR->PartTable[InMsg - 1].OwnerType != BIT_16L) )
         {
            FDiskScreen(54, NULLPTR, NULLPTR, NULLPTR, NULLPTR);
         }
         else
         {
            break;
         }
      }            
      PartitionNum = InMsg;
   }
   else
   {
      PartitionNum = ParseTable->ArgpPtr[N_OPTION]->Result;

      if ((PartitionNum <= NUM_PARTITION_ENTRY)   &&
          (MBR->PartTable[PartitionNum - 1].OwnerType != BIT12)  &&
          (MBR->PartTable[PartitionNum - 1].OwnerType != BIT_16) &&
          (MBR->PartTable[PartitionNum - 1].OwnerType != BIT_16L))
      {
         return(ILLEGALACTOPTION);
      }
   }

   ReOrderMBR(IndexArray, &NumOfPart, MBR, DiskID);
 
   Index = 0;
   Found = OFF;

   while (Index < NumOfPart)
   {
      if (MBR->PartTable[IndexArray[Index]].BootIndor == ACTIVE)
      {
         if (MenuDriven == ON)
         {
            PartInactiveObj.Position.Row = 9 + Index;
            PDisplay( &PartInactiveObj );
         }
         Found = ON;
         break;
      }
      else
      {
         Index += 1;
      }
   }

   if (MenuDriven == ON)
   {
      PartActiveObj.Position.Row = 9 + (PartitionNum - 1);
      
      PDisplay(&PartActiveObj);
   }      

   if (Found == ON)
   {
      MBR->PartTable[Index].BootIndor = INACTIVE;
   }

   MBR->PartTable[IndexArray[PartitionNum -1]].BootIndor = ACTIVE;

   /* Write that MBR back to the disk */
        
   if ((RtnCode = PWritePhysicalSector((long)SECTOR_ZERO, SECTOR_ONE,
                                  (UBYTE *) MBR, DiskID)) != SUCCESS)
   {
      return(RtnCode);
   }
}

/*#######################################################################
#                                                                       #
# Function Name: DisplayPartition()                                     # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: Display partition information on the disk.             #
#                                                                       # 
#######################################################################*/

ULONG DisplayPartition( PDiskIDStruct *DiskID )
{
   /* Get the Drive name */

   if ((ErrorCode = FDiskScreen(7,NULLPTR,DiskID->DriveName, NULLPTR, DiskID))
                                                                != SUCCESS)
   {
      return(ErrorCode);
   }
   return(SUCCESS);
}

/*#######################################################################
#                                                                       #
# Function Name: MBRBitOwnerType()                                      # 
#                                                                       # 
#  Return Value: -1 for Not found else MBE slot index                   #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: MasterBootRec *MBR;                                    #
#                                                                       #
#   Description: Searching MBR for 12 Bit or 16 Bit Slot.               #
#                                                                       # 
#######################################################################*/

WORD MBRBitOwnerType( MasterBootRec *MBR )
{
   WORD BitIndex;

   BitIndex = 0;

   while (BitIndex < MAX_MBR_SLOTS)
   {
      if (MBR->PartTable[BitIndex].OwnerType == BIT12 ||
          MBR->PartTable[BitIndex].OwnerType == BIT_16||
          MBR->PartTable[BitIndex].OwnerType == BIT_16L)
      {
         return(BitIndex);
      }
      BitIndex += 1;
   }
   return(NOT_FOUND);
}

/*#######################################################################
#                                                                       #
# Function Name: MBRExtOwnerType()                                      # 
#                                                                       # 
#  Return Value: -1 For Nor Found else Found slot index.                #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: MasterBootRec *MBR;                                    #
#                                                                       #
#   Description: Searching MBR for Extended owner type.                 #
#                                                                       # 
#######################################################################*/

WORD MBRExtOwnerType( MasterBootRec *MBR )
{
   WORD ExtIndex;

   ExtIndex = 0;

   while (ExtIndex < MAX_MBR_SLOTS)
   {
      if (MBR->PartTable[ExtIndex].OwnerType == EXTPARTITION)
      {
         return(ExtIndex);
      }
      ExtIndex += 1;
   }
   return(NOT_FOUND);
}

/*#######################################################################
#                                                                       #
# Function Name: MBRNonOwnerType()                                      # 
#                                                                       # 
#  Return Value: -1 for not found else found slot's index               #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: MasterBootRec *MBR;                                    #
#                                                                       #
#   Description: Searching MBR for Empty slot.                          #
#                                                                       # 
#######################################################################*/

WORD MBRNonOwnerType( MasterBootRec *MBR )
{
   WORD NonIndex;

   NonIndex = 0;

   while (NonIndex < MAX_MBR_SLOTS)
   {
      if (MBR->PartTable[NonIndex].OwnerType == 0)
      {
         return(NonIndex);
      }
      NonIndex += 1;
   }
   return(NOT_FOUND);
}

/*#######################################################################
#                                                                       #
# Function Name: MBRUnkOwnerType()                                      # 
#                                                                       # 
#  Return Value: -1 for not found else found slot's index               #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: MasterBootRec *MBR;                                    #
#                                                                       #
#   Description: Searching MBR for foreign partition slot.              #
#                                                                       # 
#######################################################################*/

WORD MBRUnkOwnerType( MasterBootRec *MBR )
{
   WORD UnkIndex;

   UnkIndex = 0;

   while (UnkIndex < MAX_MBR_SLOTS)
   {
      if (MBR->PartTable[UnkIndex].OwnerType != BIT12 &&
          MBR->PartTable[UnkIndex].OwnerType != BIT_16 &&
          MBR->PartTable[UnkIndex].OwnerType != BIT_16L &&
          MBR->PartTable[UnkIndex].OwnerType != EXTPARTITION &&
          MBR->PartTable[UnkIndex].OwnerType != 0 )
      {
         return(UnkIndex);
      }
      UnkIndex += 1;
   }
   return(NOT_FOUND);
}

/*#######################################################################
#                                                                       #
# Function Name: LastUnkPartSlots()                                     # 
#                                                                       # 
#  Return Value: -1 for not found else found slot's index               #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: MasterBootRec *MBR;                                    #
#                                                                       #
#   Description: Searching MBR for last foreign partition slot.         #
#                                                                       # 
#######################################################################*/

WORD LastUnkPartSlots( MasterBootRec *MBR )
{
   WORD UnkIndex, Index;

   Index = MBRUnkOwnerType(MBR);

   if (Index == NOT_FOUND)
   {
      return(NOT_FOUND);
   }

   UnkIndex = Index + 1;

   while (UnkIndex < MAX_MBR_SLOTS)
   {
      if (MBR->PartTable[UnkIndex].OwnerType != BIT12 &&
          MBR->PartTable[UnkIndex].OwnerType != BIT_16 &&
          MBR->PartTable[UnkIndex].OwnerType != BIT_16L &&
          MBR->PartTable[UnkIndex].OwnerType != EXTPARTITION &&
          MBR->PartTable[UnkIndex].OwnerType != 0 )
      {
         if (MBR->PartTable[UnkIndex].StartAddr.Sector >
             MBR->PartTable[Index].StartAddr.Sector)
         {
            Index = UnkIndex;
         }
         else if (MBR->PartTable[UnkIndex].StartAddr.Sector ==
                  MBR->PartTable[Index].StartAddr.Sector)
         {
            if (MBR->PartTable[UnkIndex].StartAddr.Cylinder >
                MBR->PartTable[Index].StartAddr.Cylinder)
            {
               Index = UnkIndex;
            }
         }
      }
      UnkIndex += 1;
   }
   return(Index);
}

/*#######################################################################
#                                                                       #
# Function Name: ErrorHandler()                                         # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as a support routine.                              # 
#                                                                       #  
#     Arguments: ULONG ErrorCode;                                       #
#                                                                       #
#   Description: Print out the error code on the screen.                #
#                                                                       # 
#######################################################################*/

void  ErrorHandler( ULONG ErrorCode )
{
   switch(ErrorCode)
   {
      case BUG11:
         printf(fdmsg04);	/*"\nIllegal O Option.\n"*/
	 break;
      case BUG12:
         printf(fdmsg05);	/*\nIllegal Option.\n*/
	 break;	 
      case BADDISK:
         printf(fdmsg06);	/*\nThe Specified Hard Disk Is Unusable\n"*/
         break;
      case BADMBR:
         printf(fdmsg07);	/*\nThe First Master Boot Record Is Not Readable.\n*/
	 break;
      case NOGAP:
         printf(fdmsg08);	/*\nThere Is No Space For Creating Partition.\n*/
	 break;
      case BADOPTION:
         printf(fdmsg09);	/*\nIllegal FDISK Options.\n*/
         break;
      case NO32MEGPARTITION:
         printf(fdmsg10);	/*\nThere Is Not Enough Space For Creating 32 Meg Partition.\n*/
	 break;
      case PARTALREADYACTIVE:
         printf(fdmsg11);	/*\nActive Partition Already Exists.\n*/
	 break;
      case ILLEGALOPTIONDATA1:
         printf(fdmsg12);	/*\nIllegal Cylinder Starting Address.\n*/
	 break;
      case ILLEGALOPTIONDATA2:
         printf(fdmsg13);	/*\nIllegal Cylinder Size.\n*/
	 break;
      case NOSUCHGAPEXIST:
         printf(fdmsg14);	/*\nIllegal Option Data.\n*/
	 break;
      case NOSUCHPARTITIONEXIST:
         printf(fdmsg15);	/*\nThe Specified Partition Can Not Found.\n*/
	 break;
      case NOTENOUGHMEMORY:
         printf(fdmsg16);	/*\nNot Enough Working Memory.\n*/
	 break;
      case DELETEORDERERROR:
         printf(fdmsg17);	/*\nMust Delete Logical Partitions*/
	 break;			/*Before First Extended Partition\n*/
      case ILLEGALACTOPTION:
         printf(fdmsg18);	/*\nOnly DOS or FlexOS Partition Can Been Activated.*/
	 break;
      case NOPRIMEPART:
         printf(fdmsg19);	/*\nPrimary Partition Does Not Exist.\n*/
	 break;
      case NOEMPTYSLOTS:
         printf(fdmsg20);	/*\nMaster Boot Record Is Fully Used.\n*/
         break;
      case NOPARTONDISK:
         printf(fdmsg21);	/*\nDisk Is Unpartitioned \n*/
         break;
      case NOFOREIGNDEL:
         printf(fdmsg22);	/*\nThis FDISK Only Can Delete DOS Or FlexOS Partition.\n*/
	 break;
      default:
         printf(fdmsg23, ErrorCode);
	 break;
   }
}   

/*#######################################################################
#                                                                       #
# Function Name: FDiskScreen()                                          # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments: UWORD ScreenNum;                                       #
#                  -- Indicates which part screen needs to be display.  #
#                ULONG *InMsg[];                                        #
#                  -- Pointer to the Message come from the caller.      #
#                ULONG *OutMsg[];                                       #
#                  -- Poitor to the message which come from screen and  #
#                     will be returned back to the caller.              #
#                                                                       #
#   Description: display screen 1 - 7.                                  #
#                                                                       # 
#######################################################################*/

ULONG FDiskScreen( UWORD ScreenNum, int InMsg[], UBYTE *DriveName,
	int OutMsg[], PDiskIDStruct *DiskID )
{
   switch (ScreenNum)
   {
      case 1:   /* Display Screen 1 */
         Screen1(DriveName);
         break;
      case 2:   /* Display Screen 2 */
         Screen2(InMsg, DriveName);
         break;
      case 31:  /* Display the Error Message " NO 32 Meg Gap On The Disk" */
         Screen31();
         break;
      case 32:  /* Display the Screen 3 and confirmation */
         Screen32(InMsg, DriveName, OutMsg);
         break;
      case 40:
         Screen40(DriveName);
	 break;
      case 41:  /* Display Screen 4 and get input data from user */
         if ((ErrorCode = Screen41(InMsg, DiskID)) != SUCCESS)
         {
            return(ErrorCode);
         }
         break;
      case 42:  /* Display the partition size in MegBytes */
         Screen42(InMsg);
         break;
      case 43:  /* Display illegal data message */
         Screen43(); 
         break;
      case 44:
         Screen44(InMsg);
	 break;
      case 51:  /* Display Screen 5 and get the partition number */
         if ((ErrorCode = Screen51(InMsg, DriveName, DiskID)) != SUCCESS)
         {
            return(ErrorCode);
         }
         break;
      case 52:  /* Confirm for activation */
         Screen52(InMsg);
         break;
      case 53:  /*  Partition is already active */
         Screen53();
	 break;	 
      case 54:
         Screen54();
	 break;
      case 55:
         Screen55();
	 break;
      case 61:  /* Display Screen 6 */
         if ((ErrorCode = Screen61(InMsg, DriveName, DiskID)) != SUCCESS)
         {
            return(ErrorCode);
         }
         break;
      case 62:  /* Confirm for deletion */
         Screen62(InMsg);
         break;
      case 63:  /* Invalid deleting partition number */
         Screen63();
	 break;	 
      case 64:  /* Display deletion order error */
         Screen64();
	 break;
      case 65:
         Screen65();
	 break;
      case 7:   /* Display Screen 7 */
         if (( ErrorCode = Screen7(DriveName, DiskID)) != SUCCESS)
         {
            return(ErrorCode);
         }
         break;
   }
   return(SUCCESS);
}

/*#######################################################################
#                                                                       #
# Function Name: Screen1()                                              # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                ULONG *InMsg[];                                        #
#                  -- Pointer to the Message come from the caller.      #
#                UBYTE *DriveName;                                      #
#                  -- Poitor of the drive name string.                  #
#                                                                       #
#   Description: Display Screen 1.                                      #
#                                                                       # 
#######################################################################*/

void Screen1( UBYTE *DriveName )
{
   PClearDisplay();
   PDisplay( &BannerObj );

   PDisplay( &ChosenDiskObj, DriveName);
   PDisplay( &OptionListObj );
   PDisplay( &ChooseObj );
   PDisplay( &ChoiceOneObj );
   PDisplay( &ChoiceTwoObj );
   PDisplay( &ChoiceThreeObj );
   PDisplay( &ChoiceFourObj );
   PDisplay( &ChoiceFiveObj );
   PDisplay( &EscapeFdiskObj );
}

/*#######################################################################
#                                                                       #
# Function Name: Screen2()                                              # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                ULONG *InMsg[];                                        #
#                  -- Pointer to the Message come from the caller.      #
#                UBYTE *DriveName;                                      #
#                  -- Poitor of the drive name string                   #
#                                                                       #
#   Description:  Display Screen 2.                                     #
#                                                                       # 
#######################################################################*/

void  Screen2( int InMsg[], UBYTE *DriveName )
{
   PClearDisplay();
   PDisplay( &BannerObj );
   
   PDisplay( &CreatePartitionObj, DriveName);
   PDisplay( &ChooseObj );
   PDisplay( &SelectOneObj );
   PDisplay( &SelectTwoObj );
   PDisplay( &EscapeToMainMenuObj );
   PDisplay( &EnterSelectionObj );
   if ((InMsg[0] = PGetNum(1, 2, 2)) == FAILURE)
   {
      InMsg[0] = ESC;
   }
}

/*#######################################################################
#                                                                       #
# Function Name: Screen31()                                             # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments: NONE.                                                  #
#                                                                       #
#   Description: Dispaly no 32 Meg Gap on the disk.                     #
#                                                                       # 
#######################################################################*/

void Screen31( void )
{
   PDisplay( &No32GapObj );
   PDisplay( &EscapePosObj );
   PGetString(NULLPTR, 0, 0);
}

/*#######################################################################
#                                                                       #
# Function Name: Screen32()                                             # 
#                                                                       # 
#  Return Value: NULL.                                                  #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                ULONG *InMsg[];                                        #
#                  -- Pointer to the Message come from the caller.      #
#                UBYTE *DriveName;                                      #
#                  -- Pointer of the drive name string.                 3
#                ULONG *OutMsg[];                                       #
#                  -- Poitor to the message which come from screen and  #
#                     will be returned back to the caller.              #
#                                                                       #
#   Description: Display screen 3.                                      #
#                                                                       # 
#######################################################################*/

void Screen32( int InMsg[], UBYTE *DriveName, int OutMsg[] )
{
   PClearDisplay();
   PDisplay( &BannerObj );
	
   PDisplay( &CreateAllObj, DriveName);
   PDisplay( &SpaceLeftObj, OutMsg[0] );
   PDisplay( &WarningObj );
   PDisplay( &RemainingObj, OutMsg[1] );
   PDisplay( &EscapeToCreateMenuObj );
   InMsg[0] = PConfirm( &MakePartitionsObj, TRUE );
}

/*#######################################################################
#                                                                       #
# Function Name: Screen40()                                             # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                UBYTE *DriveName;                                      #
#                  -- Pointer to the Drive Name String.                 #
#                                                                       #
#   Description: Display First Part Of Screen 4.                        #
#                                                                       # 
#######################################################################*/

void Screen40( UBYTE *DriveName )
{
   PClearDisplay();
   PDisplay( &BannerObj );
	
   PDisplay( &CausePartitionObj, DriveName);
   PDisplay( &CurrentPartitionsObj );
   PDisplay( &HeadingObj );
   PDisplay( &UnderLineObj );
   PDisplay( &CylinderSizeObj, TotalCylinder );
   PDisplay( &EscapeToCreateMenuObj );
}

/*#######################################################################
#                                                                       #
# Function Name: Screen41()                                             # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                ULONG *InMsg[];                                        #
#                  -- Pointer to the Message come from the caller.      #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: Display The Second Part Of Screen 4.                   #
#                                                                       # 
#######################################################################*/

ULONG Screen41( int InMsg[], PDiskIDStruct *DiskID )
{
   UWORD Result;
   float MegSize;
   
   PDisplay( &DefaultCylinderObj, InMsg[0] );
   PDisplay( &DefaultSizeObj, InMsg[1] );
   
   MegSize = ((float) InMsg[1]
              * (float) DiskID->MediaDesBlock.SectorSize
	      * (float) DiskID->MediaDesBlock.SectorsPerTrack
	      * (float) DiskID->MediaDesBlock.NumberOfHeads)
	      / MEG_1;
   
   PDisplay( &MegaSizeObj, MegSize );
           
   if ((ErrorCode = ViewDisplay(&Result, ALLMBR,  DiskID)) != SUCCESS)
   {
      return(ErrorCode);
   }

   if (Result == ESC)
   {
      InMsg[0] = FAILURE;
      return(SUCCESS);
   }

   PDisplay( &EnterCylinderObj );
   if ((InMsg[0] = PGetNum(0, 10000, InMsg[0] )) == FAILURE)
   {
      return(SUCCESS);
   }
   
   PDisplay( &EnterSizeObj );
   InMsg[1] = PGetNum(1, 10000, InMsg[1]);
   
   if (InMsg[1] == FAILURE)
   {
      InMsg[0] = FAILURE;
      return(SUCCESS);
   }

   MegSize = ((float) InMsg[1]
              * (float) DiskID->MediaDesBlock.SectorSize
	      * (float) DiskID->MediaDesBlock.SectorsPerTrack
	      * (float) DiskID->MediaDesBlock.NumberOfHeads)
	      / MEG_1;
   
   PDisplay( &MegaSizeObj, MegSize );
   
   return(SUCCESS);
}

/*#######################################################################
#                                                                       #
# Function Name: Screen42()                                             # 
#                                                                       # 
#  Return Value: NULL.                                                  #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                ULONG *InMsg[];                                        #
#                  -- Poitor to the message which come back screen and  #
#                     will be returned back to the caller.              #
#                                                                       #
#   Description: Confirm with user for creation.                        # 
#                                                                       #
#######################################################################*/

void  Screen42( int InMsg[] )
{
   PLocation Line;
   
   InMsg[0] = PConfirm( &MakePartitionObj, TRUE );
   Line.Row = 22;
   Line.Column = 1;
   PEraseLineToEnd(Line);

}

/*#######################################################################
#                                                                       #
# Function Name: Screen43()                                             # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:  NONE.                                                 #
#                                                                       #
#   Description:  Display illegal data error message.                   #
#                                                                       # 
#######################################################################*/

void Screen43( void )
{
   PLocation Line;
   
   Line.Row = 22;
   Line.Column = 1;
   PEraseDisplayToEnd(Line);
   PDisplay( &IllegalDataObj );
   PGetString(NULLPTR, 0, 0);
}

/*#######################################################################
#                                                                       #
# Function Name: Screen44()                                             # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:  UBYTE *DriveName;                                     #
#                       The pointer of the drive name string.           #
#                 PDiskIDStruct *DiskID;                                #
#                                                                       #
#   Description:  Display No Space error message.                       #
#                                                                       # 
#######################################################################*/

ULONG Screen44( int InMsg[] )
{

   if ( InMsg[0] == 1 )
   {
         return(SUCCESS);
   }
   else						 /*LDT*/
   {
      PDisplay( &NoSpaceObj );
      PGetString(NULLPTR, 0, 0);
   }
   return(SUCCESS);
}   

/*#######################################################################
#                                                                       #
# Function Name: Screen51()                                             # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                ULONG *InMsg[];                                        #
#                  -- Pointer to the Message come from the caller.      #
#                UBYTE *DriveName;                                      #
#                  -- The pointer of the drive name string.             #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: Display Scree 5.                                       #
#                                                                       # 
#######################################################################*/

ULONG Screen51( int InMsg[], UBYTE *DriveName, PDiskIDStruct *DiskID )
{
   UWORD Result;
      
   PClearDisplay();
   PDisplay( &BannerObj );
	
   PDisplay( &ChangePartitionObj, DriveName);
   PDisplay( &CurrentPartitionsObj );
   PDisplay( &HeadingObj );
   PDisplay( &UnderLineObj );
   PDisplay( &EscapeToMainMenuObj );
   PDisplay( &CylinderSizeObj, TotalCylinder );
   PDisplay( &EnterNewPartitionObj );

   if ((ErrorCode = ViewDisplay(&Result, FOURMBR, DiskID)) != SUCCESS)
   {
      return(ErrorCode);
   }
    
   if (Result == ESC)
   {
      InMsg[0] = ESC;
   }    
   return(SUCCESS);
}

/*#######################################################################
#                                                                       #
# Function Name: Screen52()                                             # 
#                                                                       # 
#  Return Value: NULL.                                                  #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                ULONG *InMsg[];                                        #
#                  -- Pointer to the Message come from the caller.      #
#                                                                       #
#   Description: Confirm with user for the activation.                  #
#                                                                       # 
#######################################################################*/

void Screen52( int InMsg[] )
{
   InMsg[0] = PConfirm( &ActivatePartitionObj, TRUE );

}

/*#######################################################################
#                                                                       #
# Function Name: Screen53()                                             # 
#                                                                       # 
#  Return Value: NULL.                                                  #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                                                                       #
#   Description: Partiton is already active.                            #
#                                                                       # 
#######################################################################*/

void Screen53( void )
{
   PLocation Line;
   
   PDisplay(&PartIsActiveObj);
   PGetString(NULLPTR, 0, 0);
   Line.Row = 22;
   Line.Column = 1;
   PEraseLineToEnd(Line);

}

/*#######################################################################
#                                                                       #
# Function Name: Screen54()                                             # 
#                                                                       # 
#  Return Value: NULL.                                                  #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                                                                       #
#   Description: Illegal Partition Number.                              #
#                                                                       # 
#######################################################################*/

void Screen54( void )
{
/*   
   PLocation Line;

   PDisplay(&IllegalPartNumObj);
   PGetString(NULLPTR, 0, 0);
   Line.Row = 22;
   Line.Column = 1;
   PEraseLineToEnd(Line);

*/
   /* No message, just beep and return... */
   printf("\7\7");

}

/*#######################################################################
#                                                                       #
# Function Name: Screen55()                                             # 
#                                                                       # 
#  Return Value: NULL.                                                  #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                                                                       #
#   Description: Partiton is not exist.                                 #
#                                                                       # 
#######################################################################*/

void Screen55( void )
{
   PDisplay(&PrimePartNotExistObj);
   PGetString(NULLPTR, 0, 0);
}

/*#######################################################################
#                                                                       #
# Function Name: Screen61()                                             # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                ULONG *InMsg[];                                        #
#                  -- Pointer to the Message come from the caller.      #
#                UBYTE *DriveName;                                      #
#                  -- The pointer of the drive name string.             #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: Display Screen 6.                                      #
#                                                                       # 
#######################################################################*/

ULONG Screen61( int InMsg[], UBYTE *DriveName, PDiskIDStruct *DiskID )
{
   UWORD Result;
      
   PClearDisplay();
   PDisplay( &BannerObj );
	
   PDisplay( &DeletePartitionObj, DriveName);
   PDisplay( &CurrentPartitionsObj );
   PDisplay( &HeadingObj );
   PDisplay( &UnderLineObj );
   PDisplay( &WarningObj );
   PDisplay( &EscapeToMainMenuObj );
   PDisplay( &CylinderSizeObj, TotalCylinder );
   PDisplay( &DeleteNumberObj );
   
   if ((ErrorCode= ViewDisplay(&Result, ALLMBR,  DiskID)) != SUCCESS)
   {
      return(ErrorCode);
   }
   if (Result == ESC)
   {
      InMsg[0] = ESC;
   }   
   return(SUCCESS);
}

/*#######################################################################
#                                                                       #
# Function Name: Screen62()                                             # 
#                                                                       # 
#  Return Value: NULL.                                                  #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                ULONG *InMsg[];                                        #
#                  -- Pointer to the Message come from the caller.      #
#   Description: Confirm with user for deletion.                        #
#                                                                       # 
#######################################################################*/

void Screen62( int InMsg[] )
{
   PLocation Line;

   InMsg[0] = PConfirm( &DeleteVerifyObj, FALSE );

   Line.Row = 22;
   Line.Column = 1;
   PEraseLineToEnd(Line);

}

/*#######################################################################
#                                                                       #
# Function Name: Screen63()                                             # 
#                                                                       # 
#  Return Value: NULL.                                                  #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                                                                       #
#   Description: Deleting partition is not exist.                       #
#                                                                       # 
#######################################################################*/

void Screen63( void )
{
/*
   PLocation Line;
   
   PDisplay(&PartNotExistObj );
   PGetString(NULLPTR, 0, 0);

   Line.Row = 22;
   Line.Column = 1;
   PEraseLineToEnd(Line);
*/
   /* No message, just beep and return... */
   printf("\7\7");
}

/*#######################################################################
#                                                                       #
# Function Name: Screen64()                                             # 
#                                                                       # 
#  Return Value: NULL.                                                  #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                                                                       #
#   Description: Display deletion order error.                          #
#                                                                       # 
#######################################################################*/

void Screen64( void )
{
   PLocation Line;
   
   PDisplay(&DeleteOrderErrorObj );
   PGetString(NULLPTR, 0, 0);
   
   Line.Row = 22;
   Line.Column = 1;
   PEraseLineToEnd(Line);

}

/*#######################################################################
#                                                                       #
# Function Name: Screen65()                                             # 
#                                                                       # 
#  Return Value: NULL.                                                  #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                                                                       #
#   Description: Display all partitons has been deleted.                #
#                                                                       # 
#######################################################################*/

void Screen65( void )
{
    PDisplay( &AllPartDeletedObj );
    PGetString(NULLPTR, 0, 0);
}    
   
/*#######################################################################
#                                                                       #
# Function Name: Screen7()                                              # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments:                                                        #
#                UBYTE *DriveName;                                      #
#                  -- The poiter of the drive name string.              #
#                PDiskIDStruct *DiskID;                                 #
#                                                                       #
#   Description: Display Screen 7.                                      #
#                                                                       # 
#######################################################################*/

ULONG Screen7( UBYTE *DriveName, PDiskIDStruct *DiskID )
{
   UWORD Result;
      
   PClearDisplay();
   PDisplay( &BannerObj );
	
   PDisplay( &DisplayPartitionObj, DriveName );
   PDisplay( &HeadingObj );
   PDisplay( &UnderLineObj );
   PDisplay( &EscapeToMainMenuObj );
   PDisplay( &CylinderSizeObj, TotalCylinder );

   if ((ErrorCode = ViewDisplay(&Result, ALLMBR, DiskID)) != SUCCESS)
   {
      return(ErrorCode);
   }

   if (Result == SUCCESS)
   {
      PDisplay( &EscapePosObj );
      PGetString(NULLPTR, 0, 0);
   }
   return(SUCCESS);
}

/*#######################################################################
#                                                                       #
# Function Name: ViewDisplay()                                          # 
#                                                                       # 
#  Return Value: 0 for SUCCESS else Error Code.                         #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments: ULONG *Result;                                         #
#                                                                       #
#   Description: Display the partition information.                     #
#                                                                       # 
#######################################################################*/

ULONG ViewDisplay( UWORD *Result, WORD  Flag, PDiskIDStruct *DiskID )
{
   UWORD  PartitionCount;   /* Partition number will be displayed     */
   ULONG  SectorOffset;     /* Sector offset for I/O.                 */
   UWORD  SectorCount;      /* Sector Counter for I/O.                */
   UWORD  StartingCylinder; /* Partition starting cylinder.           */
   UWORD  EndCylinder;      /* Partition end cylinder address.        */
   UWORD  TotalCylinders;   /* Partition size in cylinder.            */
   float  TotalMegs;        /* Partition size in MegBytes.            */
   UWORD  Count;            /* Counter for only display 4 partitions  */
                            /* at a time.                             */
   ULONG  SectorsIn32Megs;  /* The number of sectors in 32 MegBytes.  */
   WORD   Reply, FirstRun;
   WORD  BitIndex, ExtIndex, Index;
   PLocation Line;

   int IndexArray[4], NumOfPart;

   /* Initialize the SectorOffset and SectorCount */

    
   SectorOffset = SECTOR_ZERO;
   SectorCount  = SECTOR_ONE;

   PartitionCount = 1;
   FirstRun = 1;

   /* Set the boundery oh the rectangle */

   while(1)
   {
      /* Initialize the diplay retangle position */

      InitialLine();

      Count = 1;      

      /* Every time user requires to see more partition, display 4 partitions */

      while (Count <= 4)
      {
         /* Chase the partition chain until no more partition on the disk */
         
         while(1)
         {  

            DataObj.Position.Column = 30;

            /* Read the MBR */

            if ((ErrorCode = PReadPhysicalSector(SectorOffset, SectorCount,
                                         (UBYTE *) MBR, DiskID)) != SUCCESS)
            {
               return(ErrorCode);
            }

            ExtIndex = MBRExtOwnerType(MBR);
 
            if (FirstRun == 1)
            {
               FirstRun = 0;

               Index = 0;

               ReOrderMBR(IndexArray, &NumOfPart, MBR, DiskID);

               if (NumOfPart != 0)
               {         
                  while(Index < NumOfPart)
                  {
                     DataObj.Position.Column = 30;

                     /* Display the Partition nember */

                     PDisplay( &PartitionNumObj, PartitionCount);

                     /* Display the System indicator */

                     if (MBR->PartTable[IndexArray[Index]].BootIndor == ACTIVE)
                     {  /* This partition is active */
   
                        PDisplay( &PartActiveObj );
                     }
                     else 
                     {  /* This partition is inactive */
 
                        PDisplay( &PartInactiveObj );
                     }
   
                     if (MBR->PartTable[IndexArray[Index]].OwnerType != BIT12 &&
                         MBR->PartTable[IndexArray[Index]].OwnerType != BIT_16 &&
                         MBR->PartTable[IndexArray[Index]].OwnerType != BIT_16L )
                     {
                        PDisplay(&UnknownPartObj);
                     }
                     else
                     {
                        /* Get the number of sectors in a 32 Meg Bytes */
  
                        SectorsIn32Megs = 32L 
   	                     * (MEG_1 / (ULONG) DiskID->MediaDesBlock.SectorSize);

                        /* Display the OS name */

                        if (MBR->PartTable[IndexArray[Index]].PartSize > SectorsIn32Megs)
                        {  /* FlexOS Operating system */

                           PDisplay( &FlexOSPartObj );
                        }
                        else 
                        {  /* DOS Operating system */
   
                           PDisplay( &DOSPartObj );
                        }
                     }
                     /* Display the starting cylinder address */
         
                     StartingCylinder = CylinderAddrConvert(
                                        &(MBR->PartTable[IndexArray[Index]].StartAddr) );

                     PDisplay( &DataObj, StartingCylinder);

                     /* Increase the column munber */

                     DataObj.Position.Column += 15;

                     /* Diplay the end cylinder address */

                     EndCylinder=CylinderAddrConvert(
                                           &(MBR->PartTable[IndexArray[Index]].EndAddr));
 
                     PDisplay( &DataObj, EndCylinder);
   
                     /* Increase the column munber */

                     DataObj.Position.Column += 15;

                     /* Display the partition size in cylinders */

                     TotalCylinders = EndCylinder - StartingCylinder + 1;

                     PDisplay( &DataObj, TotalCylinders);

                     /* Increase the column munber */

                     DataObj.Position.Column += 11;

                     MegDataObj.Position.Row = DataObj.Position.Row;
        	     MegDataObj.Position.Column = DataObj.Position.Column;

                     /* Display the partition size in Meg Bytes */

                     TotalMegs = ( (float) TotalCylinders
                            * (float) DiskID->MediaDesBlock.SectorSize
	                    * (float) DiskID->MediaDesBlock.SectorsPerTrack
	                    * (float) DiskID->MediaDesBlock.NumberOfHeads)
	                   / MEG_1;

                     PDisplay( &MegDataObj, TotalMegs);
                     Index += 1;
                     IncreaseLine();
                     Count += 1;
                     PartitionCount += 1;
                  }
               }
            
               /* if come from ActivatePartition, only display the first MBR */

               if (Flag == FOURMBR)
               {
           	  *Result = SUCCESS;
   	          return(SUCCESS);
               }

               if (ExtIndex == NOT_FOUND)
               {
                  *Result = SUCCESS;
                  return(SUCCESS);
               }
               else
               {
                  SectorOffset = ConvertDiskAddrFromMBR(
                                 &(MBR->PartTable[ExtIndex].StartAddr), DiskID);
                  
                  if ((ErrorCode = PReadPhysicalSector(SectorOffset, 
                               SectorCount, (UBYTE *) MBR, DiskID)) != SUCCESS)
                  {
                     return(ErrorCode);
                  }
               }
            }
            
            BitIndex = MBRBitOwnerType(MBR);
            ExtIndex = MBRExtOwnerType(MBR);

            DataObj.Position.Column = 30;

            if (BitIndex == NOT_FOUND)
            {  /* Ther is no partition in this MBR */

               if (ExtIndex == NOT_FOUND)
               {  /* Also no extended partition, that means no more partition */

                  Line.Row = 14;
	          Line.Column = 1;
	          PEraseLineToEnd(Line);
                  *Result = SUCCESS;
                  return(SUCCESS);
               }
               else 
               {  /* There is a extended partition */

                  /* Get Extended MBR Sector offset */
                 
                  SectorOffset = ConvertDiskAddrFromMBR(
                                 &(MBR->PartTable[ExtIndex].StartAddr), DiskID);
      
                  /* Go back to read next MBR */
               }
            }
            else 
	    {
	       if (ExtIndex != NOT_FOUND)
               {  /* There is a partition in the current MBR */

                  /* Get the next MBR sector offset ready */

                  SectorOffset = ConvertDiskAddrFromMBR(
                              &(MBR->PartTable[ExtIndex].StartAddr), DiskID);
                  break;
               }
	       else
	       {
	          break;
	       }
	    }
         }

         /* Display the Partition nember */

         PDisplay( &PartitionNumObj, PartitionCount);

         /* Display the System indicator */

         PDisplay( &PartExtendedObj );

         /* Get the number of sectors in a 32 Meg Bytes */

         SectorsIn32Megs = 32L 
	                  * (MEG_1 / (ULONG) DiskID->MediaDesBlock.SectorSize);

         /* Display the OS name */

         if (MBR->PartTable[BitIndex].PartSize > SectorsIn32Megs)
         {  /* FlexOS Operating system */

            PDisplay( &FlexOSPartObj );
         }
         else 
         {  /* DOS Operating system */

            PDisplay( &DOSPartObj );
         }

         /* Display the starting cylinder address */
         
         StartingCylinder = CylinderAddrConvert(
                           &(MBR->PartTable[BitIndex].StartAddr) );

         PDisplay( &DataObj, StartingCylinder);

         /* Increase the column munber */

         DataObj.Position.Column += 15;

         /* Diplay the end cylinder address */

         EndCylinder=CylinderAddrConvert(&(MBR->PartTable[BitIndex].EndAddr));

         PDisplay( &DataObj, EndCylinder);
   
         /* Increase the column munber */

         DataObj.Position.Column += 15;

         /* Display the partition size in cylinders */

         TotalCylinders = EndCylinder - StartingCylinder + 1;

         PDisplay( &DataObj, TotalCylinders);

         /* Increase the column munber */

         DataObj.Position.Column += 11;

         MegDataObj.Position.Row = DataObj.Position.Row;
	 MegDataObj.Position.Column = DataObj.Position.Column;
	 
         /* Display the partition size in Meg Bytes */

         TotalMegs = ( (float) TotalCylinders
	             * (float) DiskID->MediaDesBlock.SectorSize
		     * (float) DiskID->MediaDesBlock.SectorsPerTrack
		     * (float) DiskID->MediaDesBlock.NumberOfHeads)
		     / MEG_1;

         PDisplay( &MegDataObj, TotalMegs);
         
	 if (ExtIndex == NOT_FOUND)
	 {
            Line.Row = 14;
      	    Line.Column = 1;
 	    PEraseLineToEnd(Line);
            *Result = SUCCESS;
	    return(SUCCESS);
	 }

         Count += 1;
         PartitionCount += 1;
         
         /* Increse the display line */

         IncreaseLine();
      }

      /* Question user for displaying more partitions */

      Reply = PConfirm( &ViewPartitionsObj, TRUE );

      if ( Reply == FALSE || Reply == ESC )
      {
         Line.Row = 14;
	 Line.Column = 1;
	 PEraseLineToEnd(Line);

         *Result = Reply;
         break;
      }

      /* Clear rectangle */

      PClearRectangle();
   }
   return(SUCCESS);
}

/*#######################################################################
#                                                                       #
# Function Name: InitialLine()                                          # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments: NONE                                                   #
#                                                                       #
#   Description:                                                        #
#                                                                       # 
#######################################################################*/

void InitialLine( void )
{
   DataObj.Position.Row         = 
   PartitionNumObj.Position.Row = 
   PartActiveObj.Position.Row   = 
   UnknownPartObj.Position.Row  =
   PartExtendedObj.Position.Row  =
   PartInactiveObj.Position.Row = 
   DOSPartObj.Position.Row      = 
   FlexOSPartObj.Position.Row   = 9;
}


/*#######################################################################
#                                                                       #
# Function Name: IncreaseLine ()                                        # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments: NONE                                                   #
#                                                                       #
#   Description:                                                        #
#                                                                       # 
#######################################################################*/
 
void IncreaseLine( void )
{
   DataObj.Position.Row         += 1;
   PartitionNumObj.Position.Row += 1;
   PartActiveObj.Position.Row   += 1;
   UnknownPartObj.Position.Row  += 1;
   PartInactiveObj.Position.Row += 1;
   PartExtendedObj.Position.Row += 1;
   DOSPartObj.Position.Row      += 1;
   FlexOSPartObj.Position.Row   += 1;
}

/*#######################################################################
#                                                                       #
# Function Name: CylinderAddrConvert()                                  # 
#                                                                       # 
#  Return Value: UWORD cylinder address after converting.               #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments: WellDoneDiskAddr *Diskaddr                             #
#                                                                       #
#   Description: Convert WellDoneDiskAddr to DiskAddress;               #
#                (Get the full length cylinder value ).                 # 
#######################################################################*/

UWORD CylinderAddrConvert( WellDoneDiskAddr *DiskAddr )
{
   UWORD Sector;
   UWORD Cylinder;

   Sector = (UWORD) DiskAddr->Sector;
   Cylinder = (UWORD) DiskAddr->Cylinder;

   /* Shift high 2 bits in the sector lower byte to the higher byte */

   Sector = Sector << 02;

   /* Keep those two bits only in Sector */

   Sector = Sector & 0x300;

   /* Get the full length cylinder value */

   Cylinder = Cylinder | Sector;
     
   return(Cylinder);
}

/*#######################################################################
#                                                                       #
# Function Name: OnLineHelp()                                           # 
#                                                                       # 
#  Return Value: NULL                                                   #
#                                                                       #
#   Environment: Run as an application support routine.                 # 
#                                                                       #  
#     Arguments: 				                        #
#                                                                       #
#   Description: Display the help screen of command line option.        #
#                					                # 
#######################################################################*/

void OnLineHelp(void)
{
   printf(hpmsg);
}

/*#######################################################################
#                                                                       #
#                            NOTICE:                                    #
#                                                                       #
#    The source code contained in this listing is a proprietary trade   #
#    secret of DIGITAL RESEARCH, INC., Pacific Grove, California and    #
#    is also subject to copyright protection as an unpublished work     #
#    pursuant to Section 104(a) of Title 17 of the United States Code.  #
#    Unauthorized copying, adaptation, distribution, use or display is  #
#    prohitibed by law and may be subject to civil and criminal         #
#    penalties.                                                         #
#                                                                       #
#######################################################################*/
