  .page
  .sbttl	'remote hard 5 disk reads ands writes'
  .prntx	'made it to remote hard 5 rd/wr'

; +++++++++++++++++++++++++++++++++++++++++++++++++++++
; +						      +
; +	DMS-15	Remote 5 Inch Hard Disk Driver	      +
; +						      +
; +	last modified>	03oct83	joel wittenberg	      +
; +						      +
; +++++++++++++++++++++++++++++++++++++++++++++++++++++



;	++	CP/M mapping functions
;		---- ------- ---------



;		Buffering Scheme
;		--------- ------
;
;	N  write buffers	- (N * 256) bytes
;	N  write descriptors	- (N *   5) bytes
;	01 temp  buffer		-      256  bytes
;	01 read  buffer		-     1024  bytes
;	01 read  descriptor	-        2  bytes
;	01 free  buf list 	- (N *   3) bytes
;
;
;
; --		Read Buffer Descriptor
;		---- ------ ----------
;
;	01 entry consisting of:
;
;		16 bits => logical address
;
;  InReadBuf:
;
;	entry	    16 bits
;	-----	    -------
;	 00	<logical address>
;
;		InReadBuf marks a 1024-byte block
;	as present in the (single) read buffer.
;
;
; --		Write Buffer Descriptor Array
;		-----------------------------
;
;	N entries each consisting of:
;
;		16 bits => logical address
;		 8 bits => record map
;		16 bits => data buffer address
;
;  DescBase:
;
;	entry	 16 bits  08 bits   16 bits
;	-----	 -------  -------   -------
;	00	<log adr><rec map><data adr>
;	01	<log adr><rec map><data adr>
;		. . .
;		. . .
;		. . .
;	N-2	<log adr><rec map><data adr>
;	N-1	<log adr><rec map><data adr>
;
;		Logical address marks a 256-byte block
;	as present in a write buffer.
;
;		Record map is a bit-mapped byte with
;	the 2 least significant bits marking 128-byte
;	records within the 256-byte block as dirty.
;	The next two least significant bits mark these
;	same 128-byte recs as present in the 256-byte
;	block. The high	nibble is not currently used.
;
;	xxxx  0101
;   msb       |||| lsb
;	      ||||
;	      ||||----	first 	record is dirty
;	      |||-----	second	record is clean
;	      ||------	first	record is present
;	      |-------	second	record is absent
;
;		Data address is a pointer to the actual
;	data buffer. It is needed because because there
;	is no  correlation between  descriptor array
;	position and data array position.
;
;
; --		Data Buffer Array
;		---- ------ -----
;
;  FreeBufList:
;
;	entry	 8 bits	     16 bits
;	-----	 ------      -------
;	 00	<in use><data buffer adr>
;	 01	<in use><data buffer adr>
;		. . .
;		. . .
;		. . .
;	 N-2	<in use><data buffer adr>
;	 N-1	<in use><data buffer adr>
;
;		The list of available data buffers -
;	when a buffer is used the InUse byte associated
;	with its data buf adr in the FreeBufList is set
;	to 0ffh, and when the buffer is reclaimed the
;	InUse byte is set to 00h.
;
;
; --		Least Recently Used Algorithm
;		----- -------- ---- ---------
;
;		LRU priority is determined by position
;	in the descriptor array; the closer to the
;	array base, the higher the priority. Any access
;	of a buffer moves that buffer descriptor to the
;	head of the descriptor array. When all buffers
;	are in use and a new one is needed then the
;	last buffer in the descriptor array is flushed
;	(if dirty), descriptors #0 - (N-1) are moved
;	down one place in the array, and the new desc-
;	riptor is placed at the head of the array.
;

;-------
;
;		Index Register Offsets
;		----- -------- -------

; 	Index register => .BufDesc(nth,LogAdr)

LogLow		==	0	; logical record adr
LogHigh		==	1
RecMap		==	2	; dirty and/or present 
DataLow		==	3	; data buffer adr
DataHigh	==	4


;		Record Map Bits
;		------ --- ----

AnyDirty	==	3	; any recs dirty mask
FirstDirty	==	0	; first record dirty
SecDirty	==	1	; second record dirty
HaveFirst	==	2	; first record present
HaveSec		==	3	; second record present
HaveAny		==	0ch	; any recs present mask
HaveBoth	==	0ch	; both recs present


;		Miscellaneous Equates
;		------------- -------

NumDesc		==	8	; number of wr desc's
LenBufDesc	==	5	; each desc is 5 bytes
FirmRec		==	01	; firmware sector
AllocRec	==	79h	; first alloc record

Free		==	00h	; buffer available
InUse		==	0ffh	; buffer in use


;		Mapping Functions
;		------- ---------

;
;------
; entry>	a  = disk
;		hl = dma adr
;		de = track
;		b  = volume
;		c  = sec
; exit>		data xfer'd (128 bytes)
HardW:
	sta	hd5dsk
	shld	hd5dma
	sded	hd5trk
	sbcd	hd5sec
;			-- fall through to BufWrite
;-------
; entry>     none
; exit>      data written to buffer
;	     a = 0 if successful
Hd5write:
BufWrite:
	push	x		; some programs use Ix
	call	ConvWrite	; cp/m desc -> log adr
	call	CkWriteBufs	; is it in a wr buffer
	cnz	MakeNewDesc	; make a new desc if
;				; one does not exist
	call	NewPriority	; this desc -> 1st pos
	lda	ActSec	; ActSec reflects 256 byte secs
	mov	b,a
	lda	Hd5sec	; Hd5sec reflects 128 byte recs
	sub	b
;
	push	psw
	inr	a
	mov	b,a		; b = 1 or 2
	slar	a
	slar	a		; a = 4 or 8
	ora	b		; a = 5 or 0ah
	ora	RecMap(x)	; a = 5,0ah,0dh,0eh,0fh
	mov	RecMap(x),a	; set record mask
	pop	psw		; a = 0 or 1
;
	rrc			; 0 => 0, 1 => 128
	mov	e,a
	mvi	d,0	; de = byte offset (0 or 128)
	xra	a	; ensure carry is clear
;
	mov	l,DataLow(x)
	mov	h,DataHigh(x)	; hl = .buf(cur base)
	dad	d		; hl = .buf(data)
	lded	hd5dma
	xchg
	call	BufSwap
;
	mvi	a,0ffh	; at least one buf is dirty
	sta	ActDirty
;
	pop	x	; restore Ix register
	xra	a	; return success to cp/m
	ret
;
;-------
; entry>	a  = disk
;		hl = dma adr
;		de = track
;		b  = volume
;		c  = sec
; exit>		data xfer'd (1024 bytes)
Hard1:
	sta	hd5dsk
	shld	hd5dma
	sded	hd5trk
	sbcd	hd5sec
;
	call	BufRd1k		; get the data
	lxi	h,Rd1kBuf	; hl = .source
	lded	hd5dma		; de = .dest
	lxi	b,1024		; bc = length
	ldir
	xra	a		; ret success to cpm
	ret
;
;-------
; entry>	a  = disk
;		hl = dma adr
;		de = track
;		b  = volume
;		c  = sec
; exit>		data xfer'd (128 bytes)
HardR:
	sta	hd5dsk
	shld	hd5dma
	sded	hd5trk
	sbcd	hd5sec
;			-- fall through to RdHrd5
;-------
; entry>     none
; exit>      data read from write or read buffer
;	     a = 0 if successful
Hd5read:
RdHrd5:
	call	BufRd1k
	call	GetRdDisp	; de = byte offset
	lxi	h,Rd1kBuf
	dad	d		; hl = .data xfer
	lded	hd5dma		; de = .user buf
	call	BufSwap
	xra	a		; ret succes to cpm
	ret
;
;-------
; entry>	none
; exit>		reads 1k of fresh data into Rd1kbuf
BufRd1k:
	push	x		; some programs use Ix
	call	CkReadBuf
	cnz	Phys1kRead	; ensure rd buf has blk
;
	call	CkSequence	; set sequence data
	call	CkWriteBufs	; search for 1st 2 recs
	jrnz	..2nd
;				; copy write > read buf
	lxi	h,Rd1kBuf + (0*SecSize)
	call	FillBufs
;
..2nd:
	call	CkNextWrBlock	; search for 2nd 2 recs
	jrnz	..3rd
;				; copy write > read buf
	lxi	h,Rd1kBuf + (1*SecSize)
	call	FillBufs
;
..3rd:
	call	CkNextWrBlock	; search for 3rd 2 recs
	jrnz	..4th
;				; copy write > read buf
	lxi	h,Rd1kBuf + (2*SecSize)
	call	FillBufs
;
..4th:
	call	CkNextWrBlock	; search for 4th 2 recs
	jrnz	..done
;				; copy write > read buf
	lxi	h,Rd1kBuf + (3*SecSize)
	call	FillBufs
..done:
	pop	x		; restore Ix
	ret
;
;-------
; entry>	none
; exit>		none
ConvWrite:
	mvi	c,0feh
;			-- fall through to Conv2log
;-------
; entry>	c = mask byte
; exit>		none
Conv2Log:
	push	b
	lxi	h,hd5dsk
	lxi	d,ActDsk
	lxi	b,5
	ldir
;
	lda	hd5sec
	dcr	a
	pop	b
	ana	c
	inr	a
	sta	ActSec
	call	DoLogAdr
	shld	ReqLogAdr
	ret
;
;-------
; entry>	none
; exit>		none
CkNextWrBlock:
	lhld	ReqLogAdr
	inx	h		; incr the logical adr
	shld	ReqLogAdr
;
;		- fall into	CkWriteBufs
;-------
; entry>	none
; exit>		 a = 0 if match
;		   = 0ffh if no match
;		Ix = .BufDesc(current)
CkWriteBufs:
	mvi	b,NumDesc
	lxi	x,DescBase	; Ix = .BufDesc(start)
	lded	ReqLogAdr	; de = LogAdr(search)
..comp:
	mov	l,LogLow(x)
	mov	h,LogHigh(x)	; hl = LogAdr(test)
	xra	a
	dsbc	d
	rz			; description match
;
	inx	x
	inx	x
	inx	x
	inx	x
	inx	x
	djnz	..comp
;
..fail:
	ori	0ffh
	ret
;
;-------
; entry>	none
; exit>		z set if buffer matches
CkReadBuf:
	mvi	c,0f8h
	call	Conv2log	; cp/m desc -> log adr
	lhld	InReadBuf	; compare buf contents
	lded	ReqLogAdr	; against read request
	xra	a
	dsbc	d
	ret
;
;-------
; entry>	none
; exit>		Ix = .Desc(current)
MakeNewDesc:
	lhld	CurLastDesc	; hl = .Desc(LRU)
	lxi	d,LenBufDesc
	dad	d		; hl = .Desc(available)
	push	h
;
	lxi	d,EndDesc	; de = .Desc(illegal)
	xra	a
	dsbc	d		; ck for end of array
	pop	h
	jrc	..setLast
;
	call	FlushLast	; flush last descriptor
	lxi	h,LastDesc
;
..setLast:
	shld	CurLastDesc	; hl = .Desc(current)
	push	h
	pop	x		; Ix = .desc(cur)
	lded	ReqLogAdr	; de = .block(request)
	mov	LogLow(x),e
	mov	LogHigh(x),d	; Request => Logical
	mvi	RecMap(x),0	; nothing dirty yet
	call	GetNewBuf	; de = .DataBuf(avail)
	mov	DataLow(x),e	; adr of data buf into
	mov	DataHigh(x),d	; the descriptor array
	ret
;
;-------
; entry>	Ix = .Desc(current)
; exit>		Ix = .Desc(new current)
;		- moves current desc into
;		first desc array position
NewPriority:
	push	x
	pop	h		;  hl = .Desc(cur)
	push	h		; TOS = .Desc(cur)
	lxi	d,SaveDesc
	lxi	b,LenBufDesc
	ldir			; save the cur desc
;
	pop	h		;  hl = .Desc(cur)
	push	h		; TOS = .Desc(current)
	lxi	d,DescBase
	xra	a
	dsbc	d
	pop	d		; de = .Desc(cur)
	rz			; Desc(cur) = DescBase
;
	mov	c,l
	mov	b,h		; bc = len to move

	dcx	d		;  de = .Desc(source)
;
	lxi	h,LenBufDesc
	dad	d		; hl = .Desc(dest)
	xchg
	lddr
;
	lxi	d,DescBase	; de = .Desc(first)
	push	d
	pop	x		; Ix = .Desc(new cur)
	lxi	h,SaveDesc
	lxi	b,LenBufDesc
	ldir			; move the new desc in
	ret
;
;-------
; entry>	hl = .destination
;		Ix = .Desc(current)
; exit>		Ix = .Desc(current)
;		Write buf copied to Read buf and
;		Read buf copied to write buf if needed
FillBufs:
	mov	a,RecMap(x)
	ani	HaveAny
	rz
;
	mov	e,DataLow(x)	
	mov	d,DataHigh(x)	; de = .DataSource
;
	bit	HaveFirst,RecMap(x)
	jrnz	..Rec2
;				; first record absent
	push	h		; save destination adr
	push	d		; save source adr
	lxi	b,RecSize	; PreRead since read
	ldir			; buf has data
	set	HaveFirst,RecMap(x)
	pop	d		; restore source adr
	pop	h		; restore dest adr
;
	lxi	b,RecSize
	dad	b		; inr dest adr by 1 rec
	xchg
	dad	b		; inr source by 1 rec
	ldir			; move 1 record
	ret
;
;---
..Rec2:
	bit	HaveSec,RecMap(x)
	lxi	b,SecSize
	jrnz	..xfer
;
	push	h		; save destination adr
	push	d		; save source adr
	lxi	b,RecSize
	dad	b		; inr dest adr by 1 rec
	xchg
	dad	b		; inr source by 1 rec
	xchg
	ldir			; PreRead data
	set	HaveSec,RecMap(x)
	pop	d		; restore source adr
	pop	h		; restore dest adr
;
	lxi	b,RecSize
..xfer:
	xchg
	ldir
	ret
;
;------
; entry>	none
; exit>		de = .read data 128-byte block
GetRdDisp:
	lxi	h,ActSec; reflects 8 record boundary
	lda	hd5sec	; reflects cpm record
	sub	m
	mov	d,a
	mvi	e,0
	jmp	divDEby2
;
;-------
; entry>	none
; exit>		Ix = .InReadBuf
Phys1kRead:
	lxi	x,InReadBuf	; set Ix = .InReadBuf
	lhld	ReqLogAdr	; mark requested block
	shld	InReadBuf	; as present in rd buf
	lxi	h,Rd1kBuf
	call	Read1k		; read 1k into Rd1kBuf
	ret
;
;-------
; entry>	none
; exit>		de = .DataBuffer(new)
; used>		all
GetNewBuf:
	lxi	h,FreeBufList
	mvi	b,NumDesc
;
..search:
	mov	a,m		; InUse (0ffh) - in use
	ora	a		; Free  ( 00h) - avail
	jrz	..found
;
	inx	h		; hl = .bufPtr(low)
	inx	h		;    = .bufPtr(high)
	inx	h		;    = .usage
	djnz	..search	;  b = count of descs
;
	call	FlushLast	; flush last buf
	jmpr	GetNewBuf	; buf free now
;
..found:
	mvi	m,InUse		; mark buf not free
	inx	h		; hl = .dataBuf
	mov	e,m
	inx	h
	mov	d,m
	ret
;
;------
; Mark this request as sequential only if
; it is in the next 1k data block
; entry>	none
; exit>		none
CkSequence:
	call	GetLast		; de = user's last req
	lxi	h,4
	dad	d
	xchg			; de = last + 4
	lhld	ReqLogAdr	; hl = user's new req
	xra	a
	dsbc	d		; hl = new - (last +4)
	jm	..setFalse
;				; new - old is >= 4
	lxi	d,4
	xra	a
	dsbc	d		; hl = new -(old +4) -4
	jm	..setSeq
;
..setFalse:
	mvi	a,0ffh		; new -old is <4 or >7
..setSeq:
	sta	InSequence	; set sequential flag
;
;		- fall thru to setLast and make
;		- this request the last request
;
;------
; entry>	none
; exit>		none
SetLast:
	call	CurSeq
	lded	ReqLogAdr	; hl = .SeqList(cur)
	mov	m,e
	inx	h
	mov	m,d
	ret
;
;-------
; entry>	none
; exit>		de = LogAdr(curUsr,last)
GetLast:
	call	CurSeq
	mov	e,m
	inx	h
	mov	d,m
	ret
;
;-------
; entry>	none
; exit>		hl = .SequentialList(current user)
CurSeq:
	lxi	d,SeqList
	lhld	MastUsr
	mvi	h,0
	dad	h		; hl = CurUser * 2
	dad	d		; hl = .SeqList(CurUsr)
	ret
;
;-------
; entry>	Ix = .Desc(to flush)
; exit>		Ix = .Desc(flushed)
FlushOnce:
	mov	a,RecMap(x)
	ani	AnyDirty
	rz			; nothing dirty
;
	lhld	InReadBuf	; log adr of read data
	mov	e,LogLow(x)
	mov	d,LogHigh(x)	; log adr of write data
	mvi	b,1024/SecSize	; 4 adrs to check
;
..ckRead:
	push	b
	push	h
	xra	a
	dsbc	d
	pop	h
	pop	b
	jrz	..match
;
	inx	h		; inr log adr
	djnz	..ckRead	; 4 sectors in Read buf
	jmpr	..flush		; not in read buf
;
..match:
	lxi	h,Rd1kBuf	; base of read buf
	lxi	d,SecSize	; size of write buf
	mvi	a,1024/SecSize
	sub	b		; a = displacement, 0-3
	jrz	..fill
;
..disp:
	dad	d		; add offset to rd buf
	dcr	a
	jrnz	..disp
;
..fill:
	call	FillBufs	; update read buf
;
..flush:
	mov	a,RecMap(x)
	ani	HaveAny
	cpi	HaveBoth
	jrz	..all		; both records in buf
;
	push	psw
	lxi	h,TempBuf
	call	Read256		; rd sec into temp buf
	mov	e,DataLow(x)
	mov	d,DataHigh(x)
	lxi	b,RecSize
	pop	psw
;
	bit	HaveFirst,a	; ck for first record
	jrnz	..second	; if 1st rec present
;
	lxi	h,TempBuf
	ldir			; mov 1st rec to wr buf
	set	HaveFirst,RecMap(x)
	jmpr	..all
;
..second:
	lxi	h,TempBuf+RecSize
	xchg			; de = .TmpBuf(2nd rec)
	dad	b		; hl = .WrBuf(2nd rec)
	xchg			; hl = .TmpBuf(2nd rec)
	ldir			; de = .WrBuf(2nd rec)
	set	HaveSec,RecMap(x)
;
..all:
	mov	l,DataLow(x)
	mov	h,DataHigh(x)
	res	FirstDirty,RecMap(x)
	res	SecDirty,RecMap(x)
	call	Write256	; write to disk
	ret
;
;-------
;    flush all the dirty buffers
; entry>	none
; exit>		none
; used>		a,de,hl,Ix
LruFlush:
	lda	ActDirty
	ora	a
	rz			; no bufs are dirty
;
	lixd	CurLastDesc
..CkDirty:
	mov	a,RecMap(x)
	ani	AnyDirty
	jnz	FlushOnce	; this one is dirty
;
	push	x
	pop	h		; hl = .last desc cked
	lxi	d,LenBufDesc
	xra	a
	dsbc	d
	push	h		; hl = .desc to check
	pop	x		; Ix = next desc to ck
	lxi	d,DescBase	; de = .desc(first)
	xra	a
	dsbc	d
	jrnc	..CkDirty
;
	sta	ActDirty	; no buffers are dirty
	ret
;
;-------
; entry>	none
; exit>		none
FlushLast:
	push	x		; save Ix
	lxi	x,LastDesc
	call	FlushOnce
	mvi	RecMap(x),0	; no records in buf
	mvi	a,NumDesc	;  a = number of descs
	mov	c,DataLow(x)
	mov	b,DataHigh(x)	; bc = .dataBuf
	lxi	h,FreeBufList	; hl = .freeBuf(0,flag)
	pop	x		; restore Ix
..comp:
	inx	h		; hl = .freeBuf(i)
	push	psw		; save loop count
	mov	e,m
	inx	h
	mov	d,m		; de = .dataBuf(i)
	xchg			; hl = .dataBuf(i)
	xra	a		; de = .freeBuf(i,high)
	dsbc	b
	xchg			; hl = .freeBuf(i,high)
	jrz	..found
;
	inx	h		; hl = .freeBuf(i,flag)
	pop	psw
	dcr	a
	jrnz	..comp
;
	jmp	xRetryErr	; should never get here
;
..found:
	pop	psw		; clean up stack
	dcx	h
	dcx	h
	mvi	m,free
	ret
;
;-------
; entry>	A  = logical record
; exit> 	hl = physical sector address
DoLogAdr:
	push	PSW		; save logical sector
	call	SetCurPtrs	; set Trk offset, max
	lxi	d,RecPerTrk	; ** MUST be same as
				; ** HD DPB rec/track
	pop	psw		; here's the sector
	call	CalcLogAdr	; ret hl = logical adr
	ret
;
;-------
; entry>	none
; exit> 	CurTrkOff, CurMaxTrk set
; used>		de,hl
SetCurPtrs:
	lda	ActDsk
	mov	e,a		; unit #
	mvi	d,0
	push	d		; need for CurMaxTrk
	lxi	h,TrkOffTable	; hl = trk offset table
	dad	d
	dad	d		; word table
	mov	e,m
	inx	h
	mov	d,m		; de = # of offset trks
	sded	CurTrkOff
;
	pop	d		; de = unit#
	lxi	h,MaxTable	; hl = Max Track table
	dad	d
	dad	d
	mov	e,m
	inx	h
	mov	d,m		; de = cur max track
	sded	CurMaxTrk	
	ret
;
;-------
; entry>	DE =	sec/trk (from dpb)
;		A  =	log sector no.
; exit> 	HL =	log unit adr for XEBEC cont
;
; multiply (hd5trk+CurTrkOff) by 2 for
; (log base 2 of DE) times -
; (ie. mult HL by DE, DE must = some power of 2)
; then add the hd5sec-1 and divide by
; [(bytes/sec)/(bytes/record)]	 (currently = 2)
; log adr =
;   [(((trk# + CurTrkOff) X (sec/trk)) + sec#-1) / 2]
; 
; Also the size of the disk requires 17 bits,
; thus 'bit17' flags the overflow
;
CalcLogAdr:
	push	PSW
	xra	A
	sta	bit17	; reset the overflow flag
	lhld	actTRK
	lbcd	CurTrkOff
	dad	b	; add in unit trk offset
;
	push	h	; save logical track
	lbcd	CurMaxTrk
	ora	a	; subtract Max track
	dsbc	b	; from logical track
	pop	h	; restore log track
	jrnc	..TrkErr; jmp if log trk => max trk
;
..loop:
	call	divDEby2; sectors/track
	jrc	..endMULT
;
	dad	H
	jrnc	..loop
;
	lda	bit17	; current value
	adi	80h
	sta	bit17	; store overflow flag
	jrnc	..loop	; not more than 17 bits used
;
			; can't use more than 17 bits
..trkERR:
	pop	h		; ActSec is on stack
;
	lda	AutoRd		; if this is an auto
	ora	a		; preRead then just
	rnz			; return error status
;
	lxi	h,BadTrack
	call	prtMSG		; print error msg
	lhld	ActTrk
	mov	a,h		; print hi Track
	call	PrtByt
	mov	a,l		; print lo Track
	jmpr	..BadMap
;
..secERR:
	lxi	h,BadSect	; print error msg
	call	PrtMsg
	lda	ActSec
;
..BadMap:
	call	PrtByt
	mvi	c,'h'		; tell user it's in hex
	call	ConOut
	call	xHaveErr	; and handle error
	lda	ActSec
	jmpr	CalcLogAdr
;
..endMULT:
	pop	psw	; here's the logical sector
	dcr	A	; xebec sec# start at 0
	jm	..secERR
;
	mov	C,A
	mvi	B,0
	dad	B	; (bytes/sec)/(bytes/rec)
	call	divHLby2
..fudge:
	lda	bit17
	add	H	; add overflow (we know this
	mov	H,A	; addition WON'T overflow)
	xra	a
	sta	AutoRd	; clear preRead flag
	ret
;
;-------
; entry>	hl = data buffer address
;		Ix = .InReadBuf
; exit>		 z set if xfer good
Read256:
	mvi	b,1
	jmpr	JustRead
;
;-------
; entry>	hl = data buffer address
;		Ix = .InReadBuf
; exit>		 z set if xfer good
Read1k:
	mvi	b,1024/SecSize
;			-- fall through to JustRead
;
;-------
; entry>	hl = data buffer address
;		Ix = .InReadBuf
;		 b = number of sectors to read
; exit>		 z set if xfer good
JustRead:
	shld	CurDataBuf	; for error handling
	call	SetCmdAdr	; set cmd block adr
	call	SetCount	; ret hl = xfer adr
				;     de = xfer len
	push	d		; save data xfer len
;
	mvi	a,xRead		; send the read request
	call	CmdHard5
;				; hl = .data transfer
	pop	b		; bc = data xfer length
	call	xRcvData
	call	xGetRetStatus
	lxi	h,Hd5busy
	mvi	m,0		; disk not busy
	rz
;
	mvi	m,0ffh		; disk still busy
	call	xHaveErr	; tell user about error
	lhld	CurDataBuf	; retrieve data adr
	lda	CountFld	; and #secs to read
	mov	b,a
	lda	xErrBuf		; check for corrected
	ani	03fh		; data CRC error
	cpi	FixedData
	jrnz	JustRead	; retry function
;
; We had an automatically corrected data CRC error on a
; multiple sector read - the command was aborted after
; sending the corrected data

	mov	h,LogHigh(x)
	mov	l,LogLow(x)	; hl = LUN
	lded	xErrBuf +2	; get last LUN
	mov	a,d		; invert new LUN
	mov	d,e
	mov	e,a
	inx	d		; and increment
	push	d		; might need for new rd
	xchg			; hl = error LUN +1
	xra	a		; de = first LUN req
	dsbc	d		; hl = #secs read
	mov	b,l		; #secs is < 256
	lxi	d,SecSize
	lhld	CurDataBuf
	lda	CountFld
..inxParams:
	dad	d
	dcr	a
	djnz	..inxParams
;
	pop	d		; de = new LUN
	ora	a		; check for at least
	rz			; sector left to read
;
	mov	LogHigh(x),d
	mov	LogLow(x),e	; Logical(x) = new LUN
	mov	b,a
	jmpr	JustRead
;
;-------
; entry>	hl = data buffer address
		I  .Desc(current)
; exit>		 z set if xfer good
Write256:
	mvi	b,1
;			-- fall through to JustWrite
;
;-------
; entry>	hl = data buffer address
		I  .Desc(current)
;		 b = number of sectors to write
; exit>		 z set if xfer good
JustWrite:
	shld	CurDataBuf	; for error handling
	call	SetCmdAdr	; set cmd blk xfer adr
	call	SetCount	; ret hl = xfer adr
				;     de = xfer len
	push	d		; save data xfer len
;
	mvi	a,xWrite	; send the wr request
	call	CmdHard5
;				; hl = data xfer adr
	pop	b		; bc = data xfer length
	call	xSendData
	call	xGetRetStatus
	lxi	h,Hd5busy
	mvi	m,0		; disk not busy
	rz
;
	mvi	m,0ffh		; disk busy now
	call	xHaveErr	; tell user about error
	lhld	CurDataBuf	; notice that since we
	lda	CountFld	; only do 1 sector
	mov	b,a		; writes we don't check
	jmpr	JustWrite	; for FixedData status
;
;------
; entry>	Ix = .Desc(current)
;		hl = .dataBuf (not used)
; exit>		Ix = .Desc(current)
;		hl = .dataBuf
SetCmdAdr:
	mvi	a,0ffh
	sta	Hd5busy		; disk is busy now
	call	SetLogUnitNo	; get volume right
	mov	a,LogHigh(x)
	sta	MidAdr		; store adr in cmd blk
	mov	a,LogLow(x)
	sta	LowAdr
	ret
;
;-------
; entry>	hl = data buffer address
;		 b = number of sectors to write
; exit>		hl = data buffer address
;		de = data transfer length
SetCount:
	mov	a,b
	sta	CountFld
	push	h	; hl = .data buffer
;
	lxi	h,0
	lxi	d,SecSize
..dad:
	dad	d
	djnz	..dad
	pop	d	; de = data buffer adr
	xchg		; hl = data buffer adr
	ret		; de = len of data transfer
;
;-------
; entry>	none
; exit> 	none
; used>		a
SetLogUnitNo:
	lda	ActVol
	ani	1		; LUN always < 2
	rrc			; HighAdr always 0
	rrc
	rrc
	sta	HighAdr
	ret
;
;------
; entry>	hl = source
;		de = dest
; exit> 	-- data xferred --
BufSwap:
	lxi	B,RecSize	; (128)
	ldir
	ret
;
;-------
; Get hard disk subsystem status
; entry>	bc = block address
;		on return block has 8 byte hard disk
;		command status followed by 128 byte
;		volume information block
; exit>		none
; used>		all
HDstat:
; Request thru local user

	sbcd	HSTadr	; Store block addr
	mvi	a,hdstNET
	sta	netCOM
	jmp	netREQ	; return from there
;
;-------
; entry>	bc = dma adr
; exit>		status in data block
MMhdstat:

; Read the volume information from the Xebec
; "firmware" on Volume 0, Track 0, record 1.

	push	B	; save block address
	pop	H
	mvi	B,5
..fill:
	mvi	M,0
	inx	H
	djnz	..fill
;
	mvi	M,'X'	; say it's XEBEC disk

	inx	H
	mvi	M,0	; byte 6 zeroed out
;
	push	H
	call	xInErrBytes	; rets err status
	call	xGetRetStatus
	pop	H
	inx	H
	mov	M,A
	rnz		; don't read if disk is down
;
	inx	H	; hl = .user dma
	push	h
	mvi	c,FirmRec
	call	PhysZero
	call	BufRd1k	; read firmware
	lxi	h,Rd1kBuf
	pop	d	; de = .user dma
	jmp	BufSwap
;
;-------
;	Set Hd5dsk, Trk, Vol and Sec to aim at physical
;	vol0, disk0, track0, and record passed in C
; entry>	c = record
; exit>		none
; used>		hl,a
PhysZero:
	xra	a
	sta	hd5vol	; volume 0
	sta	Hd5dsk	; unit 0
	lxi	h,0
	shld	hd5trk	; track 0
	mov	a,c
	sta	hd5sec	; set sector
	ret
;
;-------
; read alloc table and assign directly
; entry>	hl = adr of partition name and psw
; exit> 	b  = unit#
;		c  = size
;		d  = volume
;		e  = control
do5assign:
	push	h		; save string adr
	mvi	c,AllocRec
	call	PhysZero	; point to alloc table
	call	BufRd1k		; read alloc table
	lxi	h,Rd1kBuf+1	; hl = .table(name)
	pop	d		; de = .string(name)
	mvi	b,MaxAllocEntry
..testASN:
	push	h	; alloc name adr
	push	d	; asn string adr
	call	testNAME
	pop	d	; restore string adr
	pop	h	; restore alloc adr
	jrz	..match
;
	push	d
	lxi	d,LenAllocEntry
	dad	d	; hl = new alloc adr
	pop	d
	djnz	..testASN	; test 64 entries
;
..fail:
	lxi	b,0ffffh	; return failure
	lxi	d,0ffffh
	jmpr	..status

..match:
	mvi	a,MaxAllocEntry
	sub	b
	mov	b,a	; b  = unit number
	dcx	h	; hl = adr of size
	mov	c,m	; c  = size byte
	lxi	d,LenAllocEntry-1
	dad	d	; adr of control
	mov	e,m	; e  = control byte
	mvi	d,0	; d  = volume
..status:
	sbcd	HARDstat
	sded	HARDstat+2
	ret
;
;-------
; entry>	hl = alloc string adr
;		de = assign string adr
; exit>		 z = set if match
testNAME:
	mvi	c,LenAssnStr
..test:
	ldax	d
	cmp	m	; do chars match
	jrz	..next
;
	mov	a,c	  ; no match> so is it
	cpi	LenAssnStr; first char of string
	jrz	..fail
;
	ldax	d	; not first char so
	ora	a	; match if byte is zero
	jrz	..next
;
..fail:
	ori	0ffh
	ret
;
..next:
	inx	h	; check rest of string
	inx	d
	dcr	c
	jrnz	..test
	ret
  .page
;
;-------
;
;		Error Messages
;		----- --------

BadTrack:	.asciz	[cr][lf]'*** bad cpmTRK> '
BadSect:	.asciz	[cr][lf]'*** bad cpmSEC> '
xFatalErr:	.asciz	[cr][lf]'*** Fatal'
xNotFatalErr:	.asciz	[cr][lf]'***'
xRestofErr:	.asciz	' Hard5 error at '
xCmdMsg:	.asciz	[cr][lf]'command>'
xStatusMsg:	.asciz	[cr][lf]'status> '


;		XEBEC command block  
;		----- ------- -----

xOPcode:	.byte	0
HighAdr:	.byte	0
MidAdr: 	.byte	0
LowAdr: 	.byte	0
CountFld:	.blkb	1	; numSecs to transfer
ctl$fld:	.byte	7	; 15 uSec buf'd step

;
;		Data Storage
;		------------

Hd5busy:	.byte	0	; init to not busy
bit17:		.byte	0	; overflow flag
ActDirty:	.byte	0	; init to buffer clean
CurTrkOff: 	.word	0	; trk incr for cur unit
CurMaxTrk:	.word	0	; Last legal track +1
CurDataBuf:	.word	80h	; last hard5 DMA

InReadBuf:	.word	0
ReqLogAdr:	.word	0

CurLastDesc:	.word	DescBase - LenBufDesc

FreeBufList:
		.byte	free		; not in use
		.word	BufBase
		.byte	free		; not in use
		.word	BufBase + 100h
		.byte	free		; not in use
		.word	BufBase + 200h
		.byte	free		; not in use
		.word	BufBase + 300h
		.byte	free		; not in use
		.word	BufBase + 400h
		.byte	free		; not in use
		.word	BufBase + 500h
		.byte	free		; not in use
		.word	BufBase + 600h
		.byte	free		; not in use
		.word	BufBase + 700h

DescBase:
	.word	0		; logical adr
	.byte	0		; record map
	.word	BufBase	+ 0	; data adr

	.word	0		; logical adr
	.byte	0		; record map
	.word	BufBase	+ 100h	; data adr

	.word	0		; logical adr
	.byte	0		; record map
	.word	BufBase	+ 200h	; data adr

	.word	0		; logical adr
	.byte	0		; record map
	.word	BufBase	+ 300h	; data adr

	.word	0		; logical adr
	.byte	0		; record map
	.word	BufBase	+ 400h	; data adr

	.word	0		; logical adr
	.byte	0		; record map
	.word	BufBase	+ 500h	; data adr

	.word	0		; logical adr
	.byte	0		; record map
	.word	BufBase	+ 600h	; data adr

LastDesc	==	.
	.word	0		; logical adr
	.byte	0		; record map
	.word	BufBase	+ 700h	; data adr
EndDesc		==	.	; illegal

SaveDesc:	.blkb	LenBufDesc

;++++++++++++++++++++++++++++++++++++++++++++++++++++++
;+						      +
;+	end Hard 5 buffered Master Module	      +
;+						      +
;++++++++++++++++++++++++++++++++++++++++++++++++++++++
