      {***************************************************************}
      {                                                               }
      { Program name              : SEMA4                             }
      {                                                               }
      { Author                    : DVH                               }
      {                                                               }
      { Date of creation          : October 12, 1983                  }
      {                           : adapted for MSDOS 2-28-84 (BRK)   }
      {                           :  for IBM/MICROSOFT PASCAL         }
      {                                                               }
      { Library used              : DRIVEC2.OBJ                       }
      {                                                               }
      { Description               : This program is designed to       }
      {                             illustrate the use of the         }
      {                             "semaphores" on the CORVUS drive  }
      {                             This program can :                }
      {                                1. Lock (set) and report the   }
      {                                   previous state of a         }
      {                                   specific semaphore.         }
      {                                2. Unlock (clear) and report   }
      {                                   the previous state of a     }
      {                                   specific semaphore.         }
      {                                3. Clear the semaphore table   }
      {                                   of all semaphores set.      }
      {                             The major use of this program is  }
      {                             as an example of the protocol     }
      {                             of the commands needed to send to }
      {                             the firmware to control the       }
      {                             semaphores.                       }
      {                             Because this program gets the     }
      {                             semaphore key from the keyboard,  }
      {                             it can only lock or unlock        }
      {                             semaphores with alpha-numeric     }
      {                             names.  However, the eight byte   }
      {                             semaphore name can consist of any }
      {                             combination of eight bit chars.   }
      {                                                               }
      {***************************************************************}

      program SEMA4 (INPUT,OUTPUT);

      const

         version = '[1.0]';

         beep = 7;
         esc  = 27;

         longstrmax = 536;

         SemNotSet = 0;       { the prior state was unlocked  }
         SemWasSet = 128;     { the prior state of this semaphore was locked }
         SemFull   = 253;     { semaphore table is full...(32 active sema4's) }
         DiskErr   = 254;
         { *Note : negative function return values indicate error conditions }

         normal = 0;

      type

         kludge = (kludge1,kludge2);

         longstr = record
                   len : integer;
                   case kludge of
                      kludge1 : (str : packed array [1..longstrmax] of char);
                      kludge2 : (int : packed array [1..longstrmax] of byte);
                   end;

         semkeys = packed array [0..7] of char;
         semkeylist = packed array [0..31] of semkeys;

      var

         rc  : integer;         { return code }
         xcv : longstr;         { command buffer }
         CRTinfile:    FILE OF CHAR;

         { DRIVEC2.OBJ }
         function INITIO : integer; extern;
         procedure CDSEND (var cmd : longstr); extern;
         procedure CDRECV (var cmd : longstr); extern;


{###################### SEMAPHORE CALLS EXAMPLES START #######################}

      {---------------------------------------------------------------}
      { Routine name              : SEMINIT                           }
      { Author                    : DVH                               }
      { Date of creation          : October 12, 1983                  }
      { Date of last modification : October 12, 1983                  }
      { Input                     : none                              }
      { Output                    : returns 0 if semaphore init ok    }
      {                             returns error code otherwise      }
      { Description               : Semaphore init (rev b drives only)}
      {                             send 5 bytes                      }
      {                             byte #                            }
      {                                0 -  5                         }
      {                                1 - 26 (decimal)               }
      {                                    10 (hex)                   }
      {                                2 - 16 (decimal)               }
      {                                    10 (hex)                   }
      {                                3 -  0                         }
      {                                4 -  0                         }
      {                             receive 1 byte                    }
      {                                1 - disk error                 }
      {---------------------------------------------------------------}

      function SEMINIT : integer;

      begin { SEMINIT }

      { build the command }
      xcv.len    := 5;  { command length }
      xcv.int[1] := 26;
      xcv.int[2] := 16;
      xcv.int[3] := 0;  { don't care about the rest of the bytes }
      xcv.int[4] := 0;
      xcv.int[5] := 0;

      { call assembly language routine to send the command ...}
      { ... and receive a return code in byte 1 }
      CDSEND (xcv);
      CDRECV (xcv);

      { a return code < 128 means no disk error }
      if (xcv.int[1] < 128) then
         SEMINIT := normal
      else
         SEMINIT := ORD(xcv.int[1]); { return disk error }

      end; { SEMINIT }

      {---------------------------------------------------------------}
      { Routine name              : SEMLOCK                           }
      { Author                    : DVH                               }
      { Input                     : the semaphore key to be locked    }
      { Output                    : hex  decimal                      }
      {                               0    0  stat ret: sem was clear }
      {                              80  128  stat ret: sem was set   }
      {                              FD  253  semaphore table is full }
      {                              FE  254  err.during semaphore r/w}
      { Description               : Semaphore lock                    }
      {                             send 10 bytes                     }
      {                             byte #                            }
      {                                0 - 10 (decimal)               }
      {                                     A (hex)                   }
      {                                1 - 11 (decimal)               }
      {                                     B (hex)                   }
      {                                2 -  1                         }
      {                                3 - semaphore name             }
      {                                 thru                          }
      {                               10                              }
      {                             receive 12 bytes                  }
      {                                1 - disk error                 }
      {                                2 - semaphore status           }
      {                                3 - unused                     }
      {                                 thru                          }
      {                               12                              }
      {---------------------------------------------------------------}

      function SEMLOCK (var key : lstring) : integer;

      var
        i,len : integer;

      begin { SEMLOCK }

      { build command }
      xcv.len    := 10;
      xcv.int[1] := 11;
      xcv.int[2] := 1;

      len := ORD(key.len);

      { build the semaphore key to be locked }
      { take up to 8 bytes only and ignore the rest }
      { if there is less than 8 bytes, fill in with blanks }
      for i := 1 to 8 do
        if (i <= len) then
          xcv.str[i+2] := key[i]
        else
          xcv.str[i+2] := ' ';

      { call assembly language routine to send the command ...}
      { ... and receive return code }
      CDSEND (xcv);
      CDRECV (xcv);

      { a return code < 128 in byte 1 means no disk error }
      if (xcv.int[1] < 128) then
        SEMLOCK := ORD(xcv.int[2])           { return the semaphore status }
      else
        SEMLOCK := ORD(-xcv.int[1])     {convert disk errors to negatives}

      end; { SEMLOCK }

      {---------------------------------------------------------------}
      { Routine name              : SEMUNLOCK                         }
      { Author                    : DVH                               }
      { Input                     : semaphore key to be unlocked      }
      { Output                    : hex  decimal                      }
      {                               0    0  stat ret: sem was clear }
      {                              80  128  stat ret: sem was set   }
      {                              FD  253  semaphore table is full }
      {                              FE  254  err.during semaphore r/w}
      { Description               : Semaphore unlock                  }
      {                             send 10 bytes                     }
      {                             byte #                            }
      {                                0 - 10 (decimal)               }
      {                                     A (hex)                   }
      {                                1 - 11 (decimal)               }
      {                                     B (hex)                   }
      {                                2 - 17 (decimal)               }
      {                                    11 (hex)                   }
      {                                3 - semaphore name             }
      {                                 thru                          }
      {                               10                              }
      {                             receive 12 bytes                  }
      {                                1 - disk error                 }
      {                                2 - semaphore status           }
      {                                3 - unused                     }
      {                                 thru                          }
      {                               12                              }
      {---------------------------------------------------------------}

      function SEMUNLOCK (var key : lstring) : integer;

      var
        i,len : integer;

      begin { SEMUNLOCK }

      { build command }
      xcv.len    := 10;
      xcv.int[1] := 11;
      xcv.int[2] := 17;

      len := ORD(key.len);

      { build the semaphore key to be unlocked }
      { take up to 8 bytes only and ignore the rest }
      { if there is less than 8 bytes, fill in with blanks }
      for i := 1 to 8 do
        if (i <= len) then
          xcv.str[i+2] := key[i]
        else
          xcv.str[i+2] := ' ';

      { call assembly language routine to send the command ...}
      { ... and receive return code }
      CDSEND (xcv);
      CDRECV (xcv);

      { a return code < 128 in byte 1 means no disk error }
      if (xcv.int[1] < 128) then
        SEMUNLOCK := ORD(xcv.int[2])      { return semaphore status }
      else
        SEMUNLOCK := ORD(-xcv.int[1])          {convert disk errors to negatives}

      end; { SEMUNLOCK }

{######################## SEMAPHORE CALLS EXAMPLES END ########################}

      {-----------------------------------------------------------------}
      { routinename             :CRTIOinit                              }
      {                                                                 }
      { description             :Initializes the console input device.  }
      {-----------------------------------------------------------------}

      procedure CRTIOinit;

      begin   { CRTIOinit }

      assign(CRTinfile,'USER');
      reset(CRTinfile);

      end;    { CRTIOinit }


      {---------------------------------------------------------------}
      { function name             : INKEY                             }
      { Author                    : DO                                }
      { Input                     : keyboard char.                    }
      { Output                    : keyboard char. in upper case      }
      { Description               : gets char. without echo           }
      {---------------------------------------------------------------}

      function  INKEY: char;

      var
        Key:  char;

      begin   { INKEY }

      Key := ' ';
      repeat
        GET(CRTinfile);
      until (CRTinfile^ = CHR(0));      { wait for "no char." condition }

      repeat
        GET(CRTinfile);
      until (CRTinfile^ <> CHR(0));

      Key := CRTinfile^;

      if (Key = chr(13)) then           {        CR => SP               }
        Key := ' '
      else
        if (Key = CHR(27)) then         {       ESC => '!'              }
          Key := '!'
        else
          if (Key IN ['a'..'z']) then   { lowercase => uppercase        }
            Key := CHR(ORD(Key) - 32);
      INKEY := Key;

      end;    { INKEY }

      {---------------------------------------------------------------}
      { Routine name              : PRINT_MENU                        }
      { Author                    : DVH                               }
      { Input                     : none                              }
      { Output                    : none                              }
      { Description               : Prints the main menu              }
      {---------------------------------------------------------------}

      procedure PRINT_MENU;

      begin { PRINT_MENU }

      WRITELN (output);
      WRITELN (output,'Corvus Semaphore Utility ',version);
      WRITELN (output,'Main Menu');
      WRITELN (output,' --------------------------------------------');
      WRITELN (output);
      WRITELN (output,'   ? - Print this menu');
      WRITELN (output);
      WRITELN (output,'   L - Lock');
      WRITELN (output);
      WRITELN (output,'   U - Unlock');
      WRITELN (output);
      WRITELN (output,'   C - Clear');
      WRITELN (output);
      WRITELN (output,'   H - Help');
      WRITELN (output);
      WRITELN (output,'   E - Exit');
      WRITELN (output);
      WRITELN (output,'--------------------------------------------');

      end; { PRINT_MENU }

      {---------------------------------------------------------------}
      { Routine name              : DISPATCH                          }
      { Author                    : DVH                               }
      { Input                     : none                              }
      { Output                    : none                              }
      { Description               : Dispatches to appropriate routines}
      {---------------------------------------------------------------}

      procedure DISPATCH;

      var
         action : char;
         xit    : boolean;
         name   : lstring(80);
         res    : integer;

      {---------------------------------------------------------------}
      { Routine name              : CLEAR                             }
      { Author                    : DVH                               }
      { Input                     : none                              }
      { Output                    : none                              }
      { Description               : Ask user once more if user really }
      {                             wants to clear semaphore table.   }
      {                             If yes, call SEMINIT to initialize}
      {                             semaphore table.                  }
      {---------------------------------------------------------------}

      procedure CLEAR;

      var
         response : lstring(80);

      begin { CLEAR }

      WRITELN (output);
      WRITE (output,'Do you really want to clear the semaphore table',
                        ' (Y/N) ? ');
      repeat
         READLN (response);
      until (response[1] = 'Y') or
            (response[1] = 'y') or
            (response[1] = 'N') or
            (response[1] = 'n');
      if (response[1] = 'Y') or (response[1] = 'y') then begin
         WRITELN (output);  
         res := SEMINIT;
         if (res >= DiskErr) then
            WRITELN (output,'Disk ERROR!!! (#',res,')', CHR(beep))
         else
            WRITELN (output,'Semaphore table has been cleared')
         end;

      end; { CLEAR }

      {---------------------------------------------------------------}
      { Routine name              : LOCK                              }
      { Author                    : DVH                               }
      { Input                     : none                              }
      { Output                    : none                              }
      { Description               : Gets the semaphore key to be      }
      {                             locked and call SEMLOCK.          }
      {---------------------------------------------------------------}

      procedure LOCK;

      begin { LOCK }

      WRITELN (output);
      WRITE (output,'Lock which semaphore (1-8 ASCII chars.) ? ');
      READLN (name);
      WRITELN (output);
      if (name[1] <> CHR(esc)) then begin
         res := SEMLOCK (name);
         case res of
           SemNotSet : WRITELN (output,'Semaphore ', name:8,
                         ' was previously UNLOCKED, and is now   LOCKED');
           SemWasSet : WRITELN (output,'Semaphore ', name:8,
                         ' was previously   LOCKED, and is now   LOCKED');
           SemFull   : WRITELN (output,' ERROR table full!!!', CHR(beep));
           DiskErr   : WRITELN (output,' Fatal DISK ERROR!!!', CHR(beep));
           otherwise WRITELN (output,' Unknow ERROR!!!', CHR(beep));
           end;
         end;
      end; { LOCK }

      {---------------------------------------------------------------}
      { Routine name              : UNLOCK                            }
      { Author                    : DVH                               }
      { Input                     : none                              }
      { Output                    : none                              }
      { Description               : Gets the key to be unlocked and   }
      {                             calls SEMUNLOCK.                  }
      {---------------------------------------------------------------}

      procedure UNLOCK;

      begin { UNLOCK }

      WRITELN (output);
      WRITE (output,'Unlock which semaphore (1-8 ASCII chars.) ? ');
      READLN (name);
      WRITELN (output);
      if (name[1] <> CHR(esc)) then begin
         res := SEMUNLOCK (name);
         case res of
            SemNotSet : WRITELN (output,'Semaphore ', name:8,
                          ' was previously UNLOCKED, and is now UNLOCKED');
            SemWasSet : WRITELN (output,'Semaphore ', name:8,
                          ' was previously   LOCKED, and is now UNLOCKED');
            SemFull   : WRITELN (output,' ERROR table full!!!', CHR(beep));
            DiskErr   : WRITELN (output,' Fatal DISK ERROR!!!', CHR(beep));
            otherwise WRITELN (output,' Unknow ERROR!!!', CHR(beep));
            end; { case }
         end;
      end; { UNLOCK }

      {---------------------------------------------------------------}
      { Routine name              : HELP                              }
      { Author                    : DVH                               }
      { Input                     : none                              }
      { Output                    : none                              }
      { Description               : Prints the help message           }
      {---------------------------------------------------------------}

      procedure HELP;

      begin { HELP }

      WRITELN (output);
      WRITELN (output,' This program is designed to serve as an');
      WRITELN (output,' example of how to access the semaphores');
      WRITELN (output,' supported by the CORVUS drive.  These');
      WRITELN (output,' semaphores (software switches) are');
      WRITELN (output,' maintained by the CORVUS controller and');
      WRITELN (output,' saved on a hidden area of the disc.  The CORVUS');
      WRITELN (output,' drive firmware supports 32 binary semaphores');
      WRITELN (output,' - each associated with a user selected');
      WRITELN (output,' 8 byte name (key).  This feature was');
      WRITELN (output,' implemented to provide a way for applications');
      WRITELN (output,' programs to "safely" control simultaneous');
      WRITELN (output,' file access by two or more users on the');
      WRITELN (output,' CORVUS CONSTELLATION.  However, you may');
      WRITELN (output,' find other uses for them.  For instance,');
      WRITELN (output,' they could also be used for password');
      WRITELN (output,' access control of various user programs.');
      WRITELN (output);
      WRITELN (output,' -------------- WARNING --------------');
      WRITELN (output,' Use this program with caution on a system');
      WRITELN (output,' that is already using semaphores.  If used');
      WRITELN (output,' indiscriminately, you may set or clear');
      WRITELN (output,' semaphores in use by currently running');
      WRITELN (output,' programs.  This would confuse their file');
      WRITELN (output,' or record protection scheme and could');
      WRITELN (output,' lead to either loss of data or system');
      WRITELN (output,' lockup (or both).');
      WRITELN (output);
      WRITELN (output);
      end; { HELP }

      begin { DISPATCH }

      FILLC (ADR name, 8, ' ');

      repeat

        WRITELN (output);
        WRITE   (output,'Please select an option (?,L,U,C,H,E) : ');

        xit := false;
        repeat
          action := INKEY;
          if (action in ['L','U','C','H','E','?']) then
            xit := true
          else
            WRITE (output,CHR(beep));
        until xit or
              (action in ['L','U','C','H','E','?']);
        WRITELN (output,action);

        case action of
          'L': LOCK;
          'U': UNLOCK;
          'C': CLEAR;
          'H': HELP;
          '?': PRINT_MENU;
          'E': {exit};
        end; {case}

      until (action = 'E');

      end; { DISPATCH }

      {---------------------------------------------------------------}
      {                                                               }
      { MAIN BODY                                                     }
      {                                                               }
      {---------------------------------------------------------------}

      begin { MAIN BODY }

      rc := INITIO{*};
      if (rc = 0 ) then begin
        WRITELN (output);
        WRITELN (output,'Copyright 1983  Corvus Systems, Inc.');
        WRITELN (output,'        All Rights Reserved');
        CRTIOinit{*};
        PRINT_MENU{*};
        DISPATCH{*};
        end
      else
        WRITELN (output,CHR(beep), 'Driver not found');


      end. { MAIN BODY }











