/* vi: set ts=4 sw=4 : <- modeline
 * @(#) Copyright 1989, The Wollongong Group, All rights reserved
 */

#ident "@(#)slipd.c (TWG)      1.3      14:12:05 - 89/07/31"

#include <stdio.h>
#include <errno.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/termio.h>
#include <sys/stropts.h>
#include <sys/signal.h>

#define SLIPCONFFL	"/usr/etc/slip.cf"
#define SLIPPID		"/usr/etc/slip.pid"
#define O_NOCTTY    0x800
#define NUMSLIPS    128
#define LINESIZE    128
#define DEVNMSIZE   40
#define MTUSIZE     1024
#define CTLSIZE     256

#define	END	0300     /* indicates end of packet */
#define	ESC	0333     /* indicates byte stuffing */
#define	ESC_END	0334 /* ESC ESC_END means END data byte */
#define	ESC_ESC	0335 /* ESC ESC_ESC means ESC data byte */

struct slconf {
	int        pid;
	int        baud;
	char       sl_drv[ DEVNMSIZE ];
	char       sl_tty[ DEVNMSIZE ];
};

struct slconf slip[ NUMSLIPS ];
FILE          *cffl = NULL;
char          *progname;
int           numslips = 0;
int           confline = 0;
int           net_fd;
int           tty_fd;

/* forward procedure declarations */

void    doit();
void    endslconf();
void    restart_child();
void    setslconf();
void    start_child();

int
main(argc, argv)
	int   argc;
	char  *argv[];
{
	int           i;
	struct slconf *slconf;
	int           stat_loc;
	int           pid;

	progname = argv[0];

	if (argc > 1)
		setslconf(argv[1]);
	else
		setslconf((char *) NULL);

	if (cffl == NULL) {
		perror ("open cffile");
		fprintf(stderr, "usage: %s [cffile]\n", progname );
		exit(-1);
	}

	for (slconf = &slip[0]; readconfline( slconf ) != -1; ++slconf) {
		++numslips;
	}

	endslconf();

#ifndef DEBUG
	if ((pid = fork()) < 0) {
		perror("fork");
		exit(-1);
	}
	if ( pid > 0 ) {  /* Parent */
		FILE * fp;

		if ((fp = fopen(SLIPPID, "w")) == NULL) {
			perror("fopen");
			exit(-1);
		}
		fprintf(fp, "%d\n", pid );
		(void) fclose(fp);
		exit( 0 );

	}

	/* Code below runs as demon */
#endif

	signal(SIGCLD,SIG_IGN); /* This ensures that no Zombies hang around */

	slconf = &slip[0];
	for ( i = numslips + 1; --i; ++slconf ) {
		start_child( slconf );
	}

	for (;;) {

		pid = wait( &stat_loc );
		if ( pid >= 0 ) {  /* death of child */
			restart_child( pid );
		}
	}
}

void
start_child( slconf )
	struct slconf *slconf;
{

	slconf->pid = 0;

	if ( (tty_fd = open(slconf->sl_tty, O_RDWR|O_NDELAY)) <= 0) {
		perror(slconf->sl_drv);
		return;
	}

	if ( (net_fd = open(slconf->sl_drv, O_RDWR|O_NDELAY)) <= 0) {
		perror(slconf->sl_drv);
		return;
	}

	if ((slconf->pid = fork()) < 0) {
		perror("fork");
		exit(-1);
	}

	if (slconf->pid > 0) {     /* Parent */
		close(net_fd);
		close(tty_fd);
		return;
	} 

	/* code below is the daemon for one SLIP line */

	(void) open("/dev/console", O_NOCTTY );
	(void) dup2(0, 1);
	(void) dup2(0, 2);

	doit( slconf );

}

void
restart_child( pid )
	int        pid;
{
	struct slconf *slconf;
	int           i;

	slconf = &slip[0];
	for (i = numslips + 1; --i; ++slconf ) {
		if ( slconf->pid == pid ) {
			start_child( slconf );
			break;
		}
	}
}

void
setslconf(filename)
char *filename;
{

	if (filename == NULL)
		filename = SLIPCONFFL;

	if (cffl == NULL)
		cffl = fopen(filename, "r" );
	else
		rewind(cffl);
}

void
endslconf()
{
	if (cffl) {
		fclose(cffl);
		cffl = NULL;
	}
}

readconfline( slconf )
	struct slconf *slconf;
{
	char *cp;
	char *p;
	int  i;
	char line[ LINESIZE ];

again:
	if ((p = fgets(line, LINESIZE, cffl)) == NULL) {
		return(-1);
	}

	confline++;

#ifdef	DEBUG
	{
		printf("---- next line ----\n");
		printf("-> drv = <%s>, tty = <%s>, baud = <%d>\n", 
			slconf->sl_drv, slconf->sl_tty, slconf->baud);
		fflush(stdout);
	}
#endif

	if (*p == '#')
		goto again;

	cp = p;

	for (i = 1; i < 4; i++) {
		while (*cp == ' ' || *cp == '\t')
			cp++;
		if (*cp == '\n' || *cp == '#') goto again;
		p = cp;
		while (*cp != ' ' && *cp != '\t' && *cp != '\n') {
			cp++;
		}
		*cp = '\0';
		cp++;

		switch(i) {

		case 1:
			(void) strncpy( slconf->sl_drv, p, DEVNMSIZE );
			break;
		case 2:
			(void) strncpy( slconf->sl_tty, p, DEVNMSIZE );
			break;
		case 3:
			slconf->baud = atoi(p);
			break;
		default:
			printf("%s: too many columns in cf file\n", progname);
		}

	}
	return(0);
}

/*--------------------------------------------------------------------------
 * Code above this line runs in the parent daemon (1 per system)
 * CODE below this line runs in the child 
 * One child per slip line
 *-----------------------------------------------------------------------*/

void
sig_poll(sig)
	int sig;
{
	int cc, flags;
	struct strbuf ctlbuf;
	struct strbuf databuf;
	char          cbuf[CTLSIZE];
	char          dbuf[MTUSIZE];

	sighold(SIGPOLL);

	ctlbuf.buf = cbuf;
	ctlbuf.maxlen = CTLSIZE;

	databuf.buf = dbuf;
	databuf.maxlen = MTUSIZE;
	/*
	 * Read from the FD.
	 */

	for (;;) {
		flags = 0;
		if ((cc = getmsg(net_fd, &ctlbuf, &databuf, &flags)) >= 0){
#ifdef XDEBUG
			printf("slipd: 0x%x bytes from net\n", databuf.len);
#endif
			slip_send(databuf.buf, databuf.len, tty_fd);
		}
		if (cc < 0 && errno == EAGAIN)
			break;
		else if (cc < 0 && errno == EINTR)
			continue;
		else if (cc == 0 || (cc < 0 && errno == ENXIO)){
			exit( 0 );
			/* NOTREACHED */
		}
	}

	sigrelse(SIGPOLL);
}

void
doit(slconf )
	struct slconf *slconf;
{
	int           flags;
	int           ret;
	struct strbuf putbuf;
	struct termio ottyb;           /* stuff for term info */
	char          pbuf[MTUSIZE];

#ifndef	DEBUG
	setpgrp();
#endif

	putbuf.buf = pbuf;
	putbuf.maxlen = MTUSIZE;

	signal(SIGINT,  SIG_IGN); /* Ignore Keyboard signals */
	signal(SIGQUIT, SIG_IGN);

	ioctl(tty_fd, TCGETA, &ottyb);

	set_line(tty_fd, slconf->baud );

	ioctl(net_fd, I_SETSIG, S_INPUT);

	for (;;) {
		if (ret = slip_recv(putbuf.buf, putbuf.maxlen, tty_fd) ) {
#ifdef XDEBUG
			printf("read 0x%x bytes from tty\n", ret);
#endif
			putbuf.len = ret;
			flags = 0;
			putmsg(net_fd, NULL, &putbuf, flags);
		}
	}

}

int
slip_send(ip, ilen, tty)
register unsigned char *ip;
register int           ilen;
         int           *tty;
{
	register int           c;
	register unsigned char *oc;
	         unsigned char outbuf[ 2*MTUSIZE ];

#define wc( x ) *oc++ = (x)

#ifdef XDEBUG
	printf("slip_send(ip = 0x%x, len = 0x%x, tty = 0x%x)\n", ip, ilen, tty);
#endif

	oc = &outbuf[0];

	/* send an initial END to flush any data due to line noise */
	wc( END );

	for ( ++ilen; --ilen; ++ip) {
		switch(*ip) {
	
		case END:
#ifdef XDEBUG
			printf("<END>");   
#endif
			wc( ESC );
			wc( ESC_END );
			break;
		
		case ESC:
#ifdef XDEBUG
			printf("<ESC>");
#endif
			wc( ESC );
			wc( ESC_ESC );
			break;

		default:
			c = *ip & 0xFF;
#ifdef XDEBUG
			printf("<%d>", c); 
#endif
			wc( c );
		}
	}
	wc( END );

	write( tty, outbuf, oc - &outbuf[0] );
}

int
slip_recv(ip, len, tty) 
register unsigned char *ip;
register int len;
register int tty;
{
	int received = 0;
	int c;

#ifdef	RDEBUG
	printf("slip_recv(ip = 0x%x, len = 0x%x, tty = 0x%x)\n", ip, len, tty);
#endif
	while (1) {
		if ( read(tty, &c, 1) < 0) {
#ifdef	RDEBUG
			perror("slip_recv");
			printf("slip_recv: EOF\n");
#endif
			continue;
		}

		c &= 0xFF;

		switch (c) {

		case END:
#ifdef	RDEBUG
			printf("slip_recv: got END\n");
#endif
			if (received)
				return(received);
			else
				break;
		case ESC:
#ifdef	RDEBUG
			printf("slip_recv: got ESC\n");
#endif
			read(tty, &c, 1);
			switch (c) {

			case ESC_END:
				c = END;
				break;
			case ESC_ESC:
				c = ESC;
				break;
			}
		/*FALLTHROUGH*/
		default:
			if (received < len) {
				*ip++ = c;
				received++;
#ifdef	RDEBUG
				printf("<%d>", c & 0xFF);
#endif
			} else {
#ifdef	RDEBUG
			printf(
			"slip_recv: pkt too long, received = 0x%x\n", received);
#endif
			}
		}
	}
}

set_line(tty, baud)
register int tty;
         int baud;
{
	struct termio lv;
	int baudmk;

	if (ioctl(tty, TCGETA, &lv) != 0)
		return;

	switch (baud) {
		case 50:
			baudmk = B50;
			break;
		case 75:
			baudmk = B75;
			break;
		case 110:
			baudmk = B110;
			break;
		case 134:
			baudmk = B134;
			break;
		case 150:
			baudmk = B150;
			break;
		case 200:
			baudmk = B200;
			break;
		case 300:
			baudmk = B300;
			break;
		case 600:
			baudmk = B600;
			break;
		case 1200:
			baudmk = B1200;
			break;
		case 1800:
			baudmk = B1800;
			break;
		case 2400:
			baudmk = B2400;
			break;
		case 4800:
			baudmk = B4800;
			break;
		case 9600:
			baudmk = B9600;
			break;
		case 19200:
			baudmk = B19200;
			break;
		default:
			printf("slipd: invalid baud setting\n");
			exit(-1);
	}

	lv.c_iflag = lv.c_oflag = lv.c_lflag = (ushort) 0;
	lv.c_iflag = (IGNPAR | IGNBRK);
	lv.c_cc[VMIN] = 1;
	lv.c_cc[VTIME] = -1;
	lv.c_cflag = baudmk | CLOCAL | CS8 | CREAD;

	ioctl(tty, TCSETAW, &lv);
}

