/*
 * @(#) Copyright 1986.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)route.c (TWG)  1.4     89/07/30 "

/*
 * Copyright (c) 1983 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 */

#include <sys/types.h>
#include <sys/param.h>
#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <sys/stropts.h>
#include <sys/stream.h>

#include <sys/inet.h>
#include <sys/socket.h>
#include <sys/if.h>
#include <sys/in.h>
#include <sys/in_var.h>
#include <sys/ip.h>
#include <sys/inetioctl.h>
#include <sys/route.h>
#include <netdb.h>

#ifdef XNS
#include <netns/ns.h>
#endif


static struct	rtentry route;
static int	ip_fd;
static int	forcehost, forcenet, doflush, nflag;
static struct	sockaddr_in sin = { AF_INET };
static void	usage();
extern struct hostent *_gethtbyname();
extern struct hostent *_gethtbyaddr();

static struct	strioctl ioc;

struct	in_addr inet_makeaddr();
char	*malloc();


static void
usage()
{
	printf("usage: route [-nf] [ <cmd> [ net | host ] <dest> <gateway> [<metric>] ]\n");
	printf("       <cmd> = add | delete | change\n");
}


main(argc, argv)
	int argc;
	char *argv[];
{

	if (argc < 2) {
		usage();
		exit(1);
	}

	if ((ip_fd = open(DEV_IP, 2)) < 0) {
		perror("route: ip open");
		exit(1);
	}

	argc--, argv++;
	for (; argc >  0 && argv[0][0] == '-'; argc--, argv++) {
		for (argv[0]++; *argv[0]; argv[0]++)
			switch (*argv[0]) {
			case 'f':
				doflush++;
				break;
			case 'n':
				nflag++;
				break;
			}
	}

	if (doflush) {
		printf("Flushing routing tables:\n");
		flush_routes();
	}

	if (argc > 0) {
		if (strcmp(*argv, "add") == 0)
			newroute(argc, argv);
		else if (strncmp(*argv, "del", 3) == 0)
			newroute(argc, argv);
		else if (strcmp(*argv, "change") == 0)
			changeroute(argc-1, argv+1);
		else
			usage();
	}
	exit(0);

}	/* end of main() */


/*
 * Go through the route tables and flush out all entries
 * Purge all entries in the routing tables not
 * associated with network interfaces.
 */
flush_routes()
{
	struct rtentry	route;
	char		*routename(), 
			*netname();
	int		count = 0;

	while (1)
	{
		/* Get next entry */
		route.rt_hash = count++;
		ioc.ic_cmd = IPIOC_GETROUTE;
		ioc.ic_timout = 0;
		ioc.ic_len = sizeof route;
		ioc.ic_dp  = (char *)&route;
		if (ioctl(ip_fd, I_STR, &ioc ) < 0) {
			if (errno != EADDRNOTAVAIL )
				perror( "netstat: print_route" );
			return;		
		}

		/* Skip entries not in use */
		if ((route.rt_flags & RTF_INUSE) == 0)
			continue;

		/* Skip all interface routes */
		if (route.rt_flags & RTF_GATEWAY)
		{
			struct sockaddr_in sin;
			sin.sin_family = AF_INET;

			sin.sin_addr = route.rt_dst;
			printf("%-20.20s ", (route.rt_flags & RTF_HOST) ?
				routename(&sin) : netname(&sin));

			sin.sin_addr = route.rt_gateway;
			printf("%-20.20s ", routename(&sin));
#ifdef DEBUG
		fprintf(stderr,"\nroute.rt_dst = %x\n", route.rt_dst);
		fprintf(stderr,"route.rt_gateway = %x\n", route.rt_gateway);
		fprintf(stderr,"route.rt_flags = %x\n", route.rt_flags);
		fprintf(stderr,"route.rt_net = %x\n", route.rt_net);
		fprintf(stderr,"route.rt_ipb = %x\n", route.rt_ipb);
		fprintf(stderr,"route.rt_next = %x\n", route.rt_next);
		fprintf(stderr,"route.rt_use = %x\n", route.rt_use);
#endif /* DEBUG */

			ioc.ic_cmd = IPIOC_DELETEROUTE;
			ioc.ic_timout = 60;
			ioc.ic_len = sizeof (struct rtentry);
			ioc.ic_dp = (char *)&route;
			if (ioctl(ip_fd, I_STR, &ioc) < 0)
				error("delete");
			else
				printf("done\n");
		}
	}
}

char *
routename(sa)
	struct sockaddr *sa;
{
	register char *cp;
	static char line[50];
	struct hostent *hp;
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN	64
#endif
	static char domain[MAXHOSTNAMELEN + 1];
	static int first = 1;
	char *index();

#ifdef XNS
	char *ns_print();
#endif

	if (first) {
		first = 0;
		if (gethostname(domain, MAXHOSTNAMELEN) == 0 &&
		    (cp = index(domain, '.')))
			(void) strcpy(domain, cp + 1);
		else
			domain[0] = 0;
	}

	switch (sa->sa_family) {

	case AF_INET:
	    {	struct in_addr in;
		in = ((struct sockaddr_in *)sa)->sin_addr;

		cp = 0;
		if (in.s_addr == INADDR_ANY)
			cp = "default";
		if (cp == 0 && !nflag) {
			if (doflush)
				hp = _gethtbyaddr(&in, sizeof (struct in_addr),
				AF_INET);
			else
				hp = gethostbyaddr(&in, sizeof (struct in_addr),
				AF_INET);
			if (hp) {
				if ((cp = index(hp->h_name, '.')) &&
				    !strcmp(cp + 1, domain))
					*cp = 0;
				cp = hp->h_name;
			}
		}
		if (cp)
			strcpy(line, cp);
		else {
#define C(x)	((x) & 0xff)
			in.s_addr = ntohl(in.s_addr);
			sprintf(line, "%u.%u.%u.%u", C(in.s_addr >> 24),
			   C(in.s_addr >> 16), C(in.s_addr >> 8), C(in.s_addr));
		}
		break;
	    }

#ifdef XNS
	case AF_NS:
		return (ns_print((struct sockaddr_ns *)sa));
#endif

	default:
	    {	u_short *s = (u_short *)sa->sa_data;

		sprintf(line, "af %d: %x %x %x %x %x %x %x", sa->sa_family,
			s[0], s[1], s[2], s[3], s[4], s[5], s[6]);
		break;
	    }

	}	/* End of Switch */

	return (line);

}	/* routename() */


/*
 * Return the name of the network whose address is given.
 * The address is assumed to be that of a net or subnet, not a host.
 */
char *
netname(sa)
	struct sockaddr *sa;
{
	char *cp = 0;
	static char line[50];
	struct netent *np = 0;
	u_long net, mask;
	register i;
	int subnetshift;

	switch (sa->sa_family) {

	case AF_INET:
	    {	struct in_addr in;
		in = ((struct sockaddr_in *)sa)->sin_addr;

		in.s_addr = ntohl(in.s_addr);
		if (in.s_addr == 0)
			cp = "default";
		else if (!nflag) {
			if (IN_CLASSA(i)) {
				mask = IN_CLASSA_NET;
				subnetshift = 8;
			} else if (IN_CLASSB(i)) {
				mask = IN_CLASSB_NET;
				subnetshift = 8;
			} else {
				mask = IN_CLASSC_NET;
				subnetshift = 4;
			}
			/*
			 * If there are more bits than the standard mask
			 * would suggest, subnets must be in use.
			 * Guess at the subnet mask, assuming reasonable
			 * width subnet fields.
			 */
			while (in.s_addr &~ mask)
				mask = (long)mask >> subnetshift;
			net = in.s_addr & mask;
			while ((mask & 1) == 0)
				mask >>= 1, net >>= 1;
			np = getnetbyaddr(net, AF_INET);
			if (np)
				cp = np->n_name;
		}
		if (cp)
			strcpy(line, cp);
		else if ((in.s_addr & 0xffffff) == 0)
			sprintf(line, "%u", C(in.s_addr >> 24));
		else if ((in.s_addr & 0xffff) == 0)
			sprintf(line, "%u.%u", C(in.s_addr >> 24),
			    C(in.s_addr >> 16));
		else if ((in.s_addr & 0xff) == 0)
			sprintf(line, "%u.%u.%u", C(in.s_addr >> 24),
			    C(in.s_addr >> 16), C(in.s_addr >> 8));
		else
			sprintf(line, "%u.%u.%u.%u", C(in.s_addr >> 24),
			    C(in.s_addr >> 16), C(in.s_addr >> 8),
			    C(in.s_addr));
		break;
	    }

#ifdef XNS
	case AF_NS:
		return (ns_print((struct sockaddr_ns *)sa));
		break;
#endif

	default:
	    {	u_short *s = (u_short *)sa->sa_data;

		sprintf(line, "af %d: %x %x %x %x %x %x %x", sa->sa_family,
			s[0], s[1], s[2], s[3], s[4], s[5], s[6]);
		break;
	    }
	}
	return (line);

}	/* End of netname() */


newroute(argc, argv)
	int argc;
	char *argv[];
{
#ifdef BSD
	struct sockaddr_in *sin;
#else
	struct sockaddr_in sin;
#endif
	char *cmd, *dest, *gateway;
	int ishost, metric = 0, ret, attempts, oerrno;
	struct hostent *hp;
	extern int errno;

	cmd = argv[0];

	if ((strcmp(argv[1], "host")) == 0) {
		forcehost++;
		argc--, argv++;
	} else if ((strcmp(argv[1], "net")) == 0) {
		forcenet++;
		argc--, argv++;
	}

	if (*cmd == 'a') {
	    if (argc < 4) {
		printf("usage: %s destination gateway metric [tos]\n", cmd);
		printf("(metric of 0 if gateway is this host)\n");
		return;
	    }
	    metric = atoi(argv[3]);
	    if (argc > 4)
		route.rt_tos = atoi(argv[4]);
	    else
		route.rt_tos = 0;
	} else {
	    if (argc < 3) {
		printf("usage: %s destination gateway [tos]\n", cmd);
		return;
	    }
	    if (argc > 3)
		route.rt_tos = atoi(argv[3]);
	    else
		route.rt_tos = 0;
	}

#ifdef BSD
	sin = (struct sockaddr_in *)&route.rt_dst;
	ishost = getaddr(argv[1], &route.rt_dst, &hp, &dest, forcenet);
#else
	ishost = getaddr(argv[1], &sin, &hp, &dest, forcenet);
	route.rt_dst = sin.sin_addr;
#endif

	if (forcehost)
		ishost = 1;

	if (forcenet)
		ishost = 0;

#ifdef BSD
	sin = (struct sockaddr_in *)&route.rt_gateway;
	(void) getaddr(argv[2], &route.rt_gateway, &hp, &gateway, 0);
#else
	(void) getaddr(argv[2], &sin, &hp, &gateway, 0);
	route.rt_gateway = sin.sin_addr;
#endif
	route.rt_flags = RTF_UP;
	if (ishost)
		route.rt_flags |= RTF_HOST;
	if (metric > 0)
		route.rt_flags |= RTF_GATEWAY;
	route.rt_protocol = RTR_LOCAL;

	/* get ready for I_STR ioctl */
	ioc.ic_cmd = *cmd == 'a' ? IPIOC_ADDROUTE : IPIOC_DELETEROUTE;
	ioc.ic_timout = 60;
	ioc.ic_len = sizeof (struct rtentry);
	ioc.ic_dp = (char *)&route;

	for (attempts = 1; ; attempts++) {
		errno = 0;

#ifdef DEBUG
fprintf(stderr,"\nroute.rt_dst = %x\n", route.rt_dst);
fprintf(stderr,"route.rt_gateway = %x\n", route.rt_gateway);
fprintf(stderr,"route.rt_flags = %x\n", route.rt_flags);
fprintf(stderr,"route.rt_net = %x\n", route.rt_net);
fprintf(stderr,"route.rt_ipb = %x\n", route.rt_ipb);
fprintf(stderr,"route.rt_next = %x\n", route.rt_next);
fprintf(stderr,"route.rt_use = %x\n", route.rt_use);
#endif /* DEBUG */
		if ((ret = ioctl(ip_fd, I_STR, &ioc)) == 0)
			break;

		if (errno != ENETUNREACH && errno != ESRCH)
			break;

		if (hp && hp->h_addr_list[1]) {
			hp->h_addr_list++;
#ifdef BSD
			bcopy(hp->h_addr_list[0], (caddr_t)&sin->sin_addr,
			    hp->h_length);
#else
			bcopy(hp->h_addr_list[0], (caddr_t)&sin.sin_addr,
			    hp->h_length);
#endif
		} else
			break;
	}

	oerrno = errno;
	printf("%s %s %s: gateway %s tos %d", cmd,
		ishost? "host" : "net",
		dest, gateway, route.rt_tos);

	if (attempts > 1 && ret == 0)
#ifdef BSD
	    printf(" (%s)",
		inet_ntoa(((struct sockaddr_in *)&route.rt_gateway)->sin_addr));
#else
	    printf(" (%s)",
		inet_ntoa(route.rt_gateway));
#endif

	if (ret == 0)
		printf("\n");
	else {
		printf(": ");
		fflush(stdout);
		errno = oerrno;
		error(0);

	}

}	/* End of newroute() */


changeroute(argc, argv)
	int argc;
	char *argv[];
{
	printf("not supported\n");
}


error(cmd)
	char *cmd;
{

	if (errno == ESRCH)
		fprintf(stderr, "not in table\n");
	else if (errno == EBUSY)
		fprintf(stderr, "entry in use\n");
	else if (errno == ENOBUFS)
		fprintf(stderr, "routing table overflow\n");
	else
		perror(cmd);

}	/* End of error() */


char *
savestr(s)
	char *s;
{
	char *sav;

	sav = malloc(strlen(s) + 1);
	if (sav == NULL) {
		fprintf("route: out of memory\n");
		exit(1);
	}
	strcpy(sav, s);
	return (sav);

}	/* End of savestr() */


/*
 * Interpret an argument as a network address of some kind,
 * returning 1 if a host address, 0 if a network address.
 */
getaddr(s, sin, hpp, name, isnet)
	char *s;
	struct sockaddr_in *sin;
	struct hostent **hpp;
	char **name;
	int isnet;
{
	struct hostent *hp;
	struct netent *np;
	u_long val;

	*hpp = 0;
	if (strcmp(s, "default") == 0) {
		sin->sin_family = AF_INET;
		sin->sin_addr = inet_makeaddr(0, INADDR_ANY);
		*name = "default";
		return(0);
	}
	sin->sin_family = AF_INET;
	if (isnet == 0) {
		val = inet_addr(s);
		if (val != -1) {
			sin->sin_addr.s_addr = val;
			*name = s;
			return(inet_lnaof(sin->sin_addr) != INADDR_ANY);
		}
	}
	val = inet_network(s);
	if (val != -1) {
		sin->sin_addr = inet_makeaddr(val, INADDR_ANY);
		*name = s;
		return(0);
	}
	np = getnetbyname(s);
	if (np) {
		sin->sin_family = np->n_addrtype;
		sin->sin_addr = inet_makeaddr(np->n_net, INADDR_ANY);
		*name = savestr(np->n_name);
		return(0);
	}
	if (doflush)
		hp = _gethtbyname(s);
	else
		hp = gethostbyname(s);
	if (hp) {
		*hpp = hp;
		sin->sin_family = hp->h_addrtype;
		bcopy(hp->h_addr, &sin->sin_addr, hp->h_length);
		*name = savestr(hp->h_name);
		return(1);
	}
	fprintf(stderr, "%s: bad value\n", s);
	exit(1);

}	/* End of getaddr() */


#ifdef XNS
short ns_nullh[] = {0,0,0};
short ns_bh[] = {-1,-1,-1};

char *
ns_print(sns)
struct sockaddr_ns *sns;
{
	struct ns_addr work;
	union { union ns_net net_e; u_long long_e; } net;
	u_short port;
	static char mybuf[50], cport[10], chost[25];
	char *host = "";
	register char *p; register u_char *q; u_char *q_lim;

	work = sns->sns_addr;
	port = ntohs(work.x_port);
	work.x_port = 0;
	net.net_e  = work.x_net;
	if (ns_nullhost(work) && net.long_e == 0) {
		if (port ) {
			sprintf(mybuf, "*.%xH", port);
			upHex(mybuf);
		} else
			sprintf(mybuf, "*.*");
		return (mybuf);
	}

	if (bcmp(ns_bh, work.x_host.c_host, 6) == 0) { 
		host = "any";
	} else if (bcmp(ns_nullh, work.x_host.c_host, 6) == 0) {
		host = "*";
	} else {
		q = work.x_host.c_host;
		sprintf(chost, "%02x%02x%02x%02x%02x%02xH",
			q[0], q[1], q[2], q[3], q[4], q[5]);
		for (p = chost; *p == '0' && p < chost + 12; p++);
		host = p;
	}
	if (port)
		sprintf(cport, ".%xH", htons(port));
	else
		*cport = 0;

	sprintf(mybuf,"%xH.%s%s", ntohl(net.long_e), host, cport);
	upHex(mybuf);
	return(mybuf);

}	/* End of ns_print() */

upHex(p0)
char *p0;
{
	register char *p = p0;
	for (; *p; p++) switch (*p) {

	case 'a': case 'b': case 'c': case 'd': case 'e': case 'f':
		*p += ('A' - 'a');
	}
}

#endif /* XNS */

