/*
 * @(#) Copyright 1989, The Wollongong Group, All rights reserved.
 */

#ident "@(#)ping.c (TWG)  1.3     89/07/11 "

/*
 *			P I N G . C
 *
 * Using the InterNet Control Message Protocol (ICMP) "ECHO" facility,
 * measure round-trip-delays and packet loss across network paths.
 *
 * Author -
 *	Mike Muuss
 *	U. S. Army Ballistic Research Laboratory
 *	December, 1983
 * Modified at Uc Berkeley
 *
 * Status -
 *	Public Domain.  Distribution Unlimited.
 *
 * Bugs -
 *	More statistics could always be gathered.
 *	This program has to run SUID to ROOT to access the ICMP socket.
 *
 * @(#)ping.c	4.5 (Berkeley) 4/14/86
 */

#ifndef BSD
#include <sys/types.h>
#include <sys/stream.h>
#include <sys/times.h>
#include <signal.h>
#include <sys/inet.h>
#endif

#include <stdio.h>
#include <errno.h>
#include <sys/time.h>

#include <sys/param.h>
#include <sys/socket.h>
#include <sys/file.h>

#ifdef BSD
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#else
#include <sys/in.h>
#include <sys/ip.h>
#include <sys/ip_icmp.h>
#endif

#include <netdb.h>

#define	MAXWAIT		10	/* max time to wait for response, sec. */
#define	MAXPACKET	4096	/* max packet size */

#ifndef BSD
#define	MAXBUF		MAXPACKET+8	/* real max packet size */
#endif

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN	64
#endif

static int	verbose;

#ifdef BSD
u_char	packet[MAXPACKET];
#else
static u_char	packet[MAXBUF];
#endif

static int	options;
extern	int errno;

static int s;			/* Socket file descriptor */
static struct hostent *hp;	/* Pointer to host info */
static struct timezone tz;	/* leftover */

static struct sockaddr whereto;	/* Who to ping */
static int datalen;		/* How much data */

static char usage[] = "Usage:  ping [-r] [-v] host [packetsize] [count]\n";

static char *hostname;
static char hnamebuf[MAXHOSTNAMELEN];

static int npackets;
static int ntransmitted = 0;	/* sequence # for outbound packets = #sent */
static int ident;

static int nreceived = 0;	/* # of packets we got back */
static int timing = 0;
static int tmin = 999999999;
static int tmax = 0;
static int tsum = 0;		/* sum of all times, for doing average */
static int finish(), catcher();
#ifndef BSD
static char buffer[BUFSIZ];
static u_short in_cksum();
static int gettimeofday();
#endif

/*
 * 			M A I N
 */
main(argc, argv)
char *argv[];
{
	struct sockaddr_in from;
	char **av = argv;
	struct sockaddr_in *to = (struct sockaddr_in *) &whereto;
	int on = 1;
	struct protoent *proto;

	argc--, av++;
	while (argc > 0 && *av[0] == '-') {
		while (*++av[0]) switch (*av[0]) {
#ifdef NOTDEF
			case 'd':
				options |= SO_DEBUG;
				break;
#endif
			case 'r':
				options |= SO_DONTROUTE;
				break;
			case 'v':
				verbose++;
				break;
			default:
				printf(usage);
				exit(1);
		}
		argc--, av++;
	}
	if( argc < 1)  {
		printf(usage);
		exit(1);
	}

	bzero( (char *)&whereto, sizeof(struct sockaddr) );
	to->sin_family = AF_INET;
	to->sin_addr.s_addr = inet_addr(av[0]);
	if (to->sin_addr.s_addr != -1) {
		strcpy(hnamebuf, av[0]);
		hostname = hnamebuf;
	} else {
		hp = gethostbyname(av[0]);
		if (hp) {
			to->sin_family = hp->h_addrtype;
			bcopy(hp->h_addr, (caddr_t)&to->sin_addr, hp->h_length);
			hostname = hp->h_name;
		} else {
			printf("%s: unknown host %s\n", argv[0], av[0]);
			exit(1);
		}
	}

	if( argc >= 2 )
		datalen = atoi( av[1] );
	else
		datalen = 64-8;
	if (datalen > MAXPACKET) {
		fprintf(stderr, "ping: packet size too large\n");
		exit(1);
	}
	if (datalen >= sizeof(struct timeval))
		timing = 1;
	if (argc > 2)
		npackets = atoi(av[2]);

	ident = getpid() & 0xFFFF;

	if ((proto = getprotobyname("icmp")) == NULL) {
		fprintf(stderr, "icmp: unknown protocol\n");
		exit(10);
	}
	if ((s = socket(AF_INET, SOCK_RAW, proto->p_proto)) < 0) {
		perror("ping: socket");
		exit(5);
	}
#ifdef NOTDEF
	if (options & SO_DEBUG)
		setsockopt(s, SOL_SOCKET, SO_DEBUG, &on, sizeof(on));
#endif
	if (options & SO_DONTROUTE)
		setsockopt(s, SOL_SOCKET, SO_DONTROUTE, &on, sizeof(on));

	printf("PING %s: %d data bytes\n", hostname, datalen );

#ifdef BSD
	setlinebuf( stdout );
#else
	setvbuf(stdout, buffer, _IOLBF, BUFSIZ);
#endif

	signal( SIGINT, finish );
	signal(SIGALRM, catcher);

	catcher();	/* start things going */

	for (;;) {
		int len = sizeof (packet);
		int fromlen = sizeof (from);
		int in, cc;

		/* Select added for SOMOD compat */
		in = 1 << s;
		if (select(s + 1, &in, NULL, NULL, 0) < 0) {
			if (errno == EINTR) {
				continue;
			}
			perror("ping: select");
			continue;
		}
		if (in == 0) {
			continue;
		}

		if ((cc = recvfrom(s, packet, len, 0, &from, &fromlen)) < 0) {
			if( errno == EINTR ) {
				continue;
			}
			perror("ping: recvfrom");
			continue;
		}
		pr_pack( packet, cc, &from );
		if (npackets && nreceived >= npackets)
			finish();
	}
	/*NOTREACHED*/
}

/*
 * 			C A T C H E R
 * 
 * This routine causes another PING to be transmitted, and then
 * schedules another SIGALRM for 1 second from now.
 * 
 * Bug -
 * 	Our sense of time will slowly skew (ie, packets will not be launched
 * 	exactly at 1-second intervals).  This does not affect the quality
 *	of the delay and loss statistics.
 */
static
catcher()
{
	int waittime;

#ifndef BSD
	signal(SIGALRM, catcher);
#endif

	pinger();
	if (npackets == 0 || ntransmitted < npackets) {
		alarm(1);
	} else {
		if (nreceived) {
			waittime = 2 * tmax / 1000;
			if (waittime == 0)
				waittime = 1;
		} else
			waittime = MAXWAIT;

		signal(SIGALRM, finish);
		alarm(waittime);
	}
}

/*
 * 			P I N G E R
 * 
 * Compose and transmit an ICMP ECHO REQUEST packet.  The IP packet
 * will be added on by the kernel.  The ID field is our UNIX process ID,
 * and the sequence number is an ascending integer.  The first 8 bytes
 * of the data portion are used to hold a UNIX "timeval" struct in VAX
 * byte-order, to compute the round-trip time.
 */
static
pinger()
{
#ifdef BSD
	static u_char outpack[MAXPACKET];
#else
	static u_char outpack[MAXBUF];
#endif
	register struct icmp *icp = (struct icmp *) outpack;
	int i, cc;
	register struct timeval *tp = (struct timeval *) &outpack[8];
	register u_char *datap = &outpack[8+sizeof(struct timeval)];

	icp->icmp_type = ICMP_ECHO;
	icp->icmp_code = 0;
	icp->icmp_cksum = 0;
	icp->icmp_seq = ntransmitted++;
	icp->icmp_id = ident;		/* ID */

	cc = datalen+8;			/* skips ICMP portion */

	if (timing)
		gettimeofday( tp, &tz );

	for( i=8; i<datalen; i++)	/* skip 8 for time */
		*datap++ = i;

	/* Compute ICMP checksum here */
	icp->icmp_cksum = in_cksum( icp, cc );

	/* cc = sendto(s, msg, len, flags, to, tolen) */
	i = sendto( s, outpack, cc, 0, &whereto, sizeof(struct sockaddr) );

	if( i < 0 || i != cc )  {

		if( i<0 )
			perror("sendto");

		printf("ping: wrote %s %d chars, ret=%d\n", hostname, cc, i );
		fflush(stdout);
	}
}

/*
 * 			P R _ T Y P E
 *
 * Convert an ICMP "type" field to a printable string.
 */
static char *
pr_type( t )
register int t;
{
	static char *ttab[] = {
		"Echo Reply",
		"ICMP 1",
		"ICMP 2",
		"Dest Unreachable",
		"Source Quence",
		"Redirect",
		"ICMP 6",
		"ICMP 7",
		"Echo",
		"ICMP 9",
		"ICMP 10",
		"Time Exceeded",
		"Parameter Problem",
		"Timestamp",
		"Timestamp Reply",
		"Info Request",
		"Info Reply",
		"Mask Request",
		"Mask Reply"
	};

	if( t < 0 || t > 18 )
		return("OUT-OF-RANGE");

	return(ttab[t]);
}

/*
 *			P R _ P A C K
 *
 * Print out the packet, if it came from us.  This logic is necessary
 * because ALL readers of the ICMP socket get a copy of ALL ICMP packets
 * which arrive ('tis only fair).  This permits multiple copies of this
 * program to be run without having intermingled output (or statistics!).
 */
static
pr_pack( buf, cc, from )
char *buf;
int cc;
struct sockaddr_in *from;
{
	struct ip *ip;
	register struct icmp *icp;
	register long *lp = (long *) packet;
	register int i;
	struct timeval tv;
	struct timeval *tp;
	int hlen, triptime;
	char *inet_ntoa();

	from->sin_addr.s_addr = ntohl( from->sin_addr.s_addr );
	gettimeofday( &tv, &tz );

	ip = (struct ip *) buf;
	hlen = ip->ip_hl << 2;
	if (cc < hlen + ICMP_MINLEN) {
		if (verbose)
			printf("packet too short (%d bytes) from %s\n", cc,
				inet_ntoa(ntohl(from->sin_addr.s_addr)));
		return;
	}
	cc -= hlen;
	icp = (struct icmp *)(buf + hlen);
	if( icp->icmp_type != ICMP_ECHOREPLY )  {
		if (verbose) {
			printf("%d bytes from %s: ", cc,
				inet_ntoa(ntohl(from->sin_addr.s_addr)));
			printf("icmp_type=%d (%s)\n",
				icp->icmp_type, pr_type(icp->icmp_type) );
			for( i=0; i<12; i++)
			    printf("x%2.2x: x%8.8x\n", i*sizeof(long), *lp++ );
			printf("icmp_code=%d\n", icp->icmp_code );
		}
		return;
	}
	if( icp->icmp_id != ident )
		return;			/* 'Twas not our ECHO */

	tp = (struct timeval *)&icp->icmp_data[0];
	printf("%d bytes from %s: ", cc,
		inet_ntoa(ntohl(from->sin_addr.s_addr)));
	printf("icmp_seq=%d. ", icp->icmp_seq );
	if (timing) {
		tvsub( &tv, tp );
#ifdef BSD
		triptime = tv.tv_sec*1000+(tv.tv_usec/1000);
		printf("time=%d. ms\n", triptime );
#else
		triptime = tv.tv_sec*HZ + tv.tv_usec;
		printf("time=%d  %dth of sec\n", triptime, HZ);
#endif
		tsum += triptime;
		if( triptime < tmin )
			tmin = triptime;
		if( triptime > tmax )
			tmax = triptime;
	} else
		putchar('\n');
	nreceived++;
}


/*
 *			I N _ C K S U M
 *
 * Checksum routine for Internet Protocol family headers (C Version)
 *
 */
#ifndef BSD
static u_short
#endif
in_cksum(addr, len)
u_short *addr;
int len;
{
	register int nleft = len;
	register u_short *w = addr;
	register u_short answer;
	register int sum = 0;

	/*
	 *  Our algorithm is simple, using a 32 bit accumulator (sum),
	 *  we add sequential 16 bit words to it, and at the end, fold
	 *  back all the carry bits from the top 16 bits into the lower
	 *  16 bits.
	 */
	while( nleft > 1 )  {
		sum += *w++;
		nleft -= 2;
	}

	/* mop up an odd byte, if necessary */
	if( nleft == 1 )
#if defined(vax) || defined(i386)
		sum += *(u_char *)w;
#else
		sum += ((*(u_char *)w) << 8);
#endif

	/*
	 * add back carry outs from top 16 bits to low 16 bits
	 */
	sum = (sum >> 16) + (sum & 0xffff);	/* add hi 16 to low 16 */
	sum += (sum >> 16);			/* add carry */
	answer = ~sum;				/* truncate to 16 bits */
	return (answer);
}

/*
 * 			T V S U B
 * 
 * Subtract 2 timeval structs:  out = out - in.
 * 
 * Out is assumed to be >= in.
 */
static
tvsub( out, in )
register struct timeval *out, *in;
{
	if( (out->tv_usec -= in->tv_usec) < 0 )   {
		out->tv_sec--;
#ifdef BSD
		out->tv_usec += 1000000;
#else
		out->tv_usec += HZ;
#endif
	}
	out->tv_sec -= in->tv_sec;
}

/*
 *			F I N I S H
 *
 * Print out statistics, and give up.
 * Heavily buffered STDIO is used here, so that all the statistics
 * will be written with 1 sys-write call.  This is nice when more
 * than one copy of the program is running on a terminal;  it prevents
 * the statistics output from becomming intermingled.
 */
static
finish()
{

#ifndef BSD
	signal(SIGALRM, SIG_IGN);
	signal( SIGINT, SIG_IGN);
#endif
	printf("\n----%s PING Statistics----\n", hostname );
	printf("%d packets transmitted, ", ntransmitted );
	printf("%d packets received, ", nreceived );
	if (ntransmitted)
	    printf("%d%% packet loss",
		(int) (((ntransmitted-nreceived)*100) / ntransmitted ) );
	printf("\n");
	if (nreceived && timing)
#ifdef BSD
	    printf("round-trip (ms)  min/avg/max = %d/%d/%d\n",
#else
	    printf("round-trip (%dth of sec)  min/avg/max = %d/%d/%d\n",
		HZ,
#endif
		tmin,
		tsum / nreceived,
		tmax );
	fflush(stdout);
	exit(0);
}


#ifndef BSD

static
gettimeofday(tp, tzp)
	struct timeval *tp;
	struct timezone *tzp;
{
	struct tms tms;
	long secs_hz;
	extern long times();
	extern long _daylight, _timezone;

	secs_hz = times(&tms);
	tp->tv_sec  = secs_hz / HZ;	/* seconds */
	tp->tv_usec = secs_hz % HZ;	/* HZ-th of sec. */

	if (tzp == (struct timezone *)0)
		return 0;
	(void) _tzset();	/* Set up the external variables */
	tzp->tz_minuteswest = _timezone/60;
	tzp->tz_dsttime = _daylight;
	return 0;
}
#endif /* BSD */
