/*
 * @(#) Copyright 1986.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)arpbypass.c (TWG)  1.2     89/08/01 "

#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stream.h>
#include <sys/ip.h>
#include <sys/lihdr.h>
#include <sys/inet.h>
#include <sys/socket.h>
#include <sys/inetioctl.h>
#include <sys/stropts.h>
#include <sys/arp.h>
#include <sys/route.h>
#include <sys/in.h>
#include <errno.h>
#include <ctype.h>
#include <netdb.h>
#include <macros.h>

static struct	arptab arpioc;
static struct	strioctl strioc;
static int	arpfd;		/* open file descriptor for /dev/arp */
static char	*atf[] =
{
	"{INCOMPLETE}",
	"{INCOMPLETE}",
	"{COM}",
	"{COM}",
	"{PERM}",
	"{PERM}",
	"{COM,PERM}",
	"{COM,PERM}",
	"{PUBL}",
	"{PUBL}",
	"{COM,PUBL}",
	"{COM,PUBL}",
	"{PERM,PUBL}",
	"{PERM,PUBL}",
	"{COM,PERM,PUBL}",
	"{COM,PERM,PUBL}",
};

#define ARPDEV		"/dev/arp"
#define ARPCTLDEV	"/dev/arp0"

static int options = -1;
static char *cmd = NULL;
static char *_inet_addr = NULL;
static char *ether_addr = NULL;

main(argc, argv)
	int argc;
	char *argv[];
{
	char *lower();
	int c;
	extern char *optarg;
	extern int optind;
	int errflg = 0;

	/* check number of arguments */
	if (argc < 2)
		errflg++;
	
	arpioc.at_type = DL_ETHER;	/* default value */

#ifdef NCR
	while ((c = getopt(argc, argv, "f:a")) != -1)
#else
	while ((c = getopt(argc, argv, "f:t:a")) != -1)
#endif /* NCR */
		switch (c) {

		case 'f':
			options = atoi(optarg);
			break;

#ifndef NCR
		case 't':
			arpioc.at_type = atoi(optarg);
			break;
#endif /* NCR */

		case 'a':
			dump( );
			return;

		case '?':
			errflg++;	
			break;

		}

	if (errflg) {
		usage(1, NULL);
		exit(2);
	}


	if (optind < argc)
		cmd = lower(argv[optind++]);

	if (optind < argc)
		_inet_addr = argv[optind++];

	if (optind < argc)
		ether_addr = argv[optind++];

	if (optind < argc) {
		usage(1, NULL);
		exit(2);
	}

	if (strcmp(cmd,"set") && strcmp(cmd,"get") && strncmp(cmd,"del",3)) {
		(void)fprintf(stderr, "arpbypass: invalid command\n");
		exit(1);
	}

	/* open the arp control device for read/write */
	if ((arpfd = open(ARPCTLDEV, O_RDWR)) < 0) {
		perror("arpbypass: open failed for ARPCTLDEV");
		exit(1);
	}

	newroute();

	exit(0);
}


/*
 * Dump the entire arp table
 */
dump( )
{
	int		arptab_size = -1;
	int		i, sz, *val;
	struct arptab	*at, *atp;
	struct hostent	*hp;
	char 		*host;
	int 		bynumber = 0;
	struct strioctl strioctl;

	/* open the arp control device for read/write */
	if ((arpfd = open(ARPCTLDEV, O_RDWR)) < 0) {
		perror("arpbypass: open failed for ARPCTLDEV");
		exit(1);
	}
	/* get number of arp table entries */
	strioctl.ic_cmd    = ARP_GETTABLEN;
	strioctl.ic_timout = 0;		/* default timeout = 15 seconds */
	strioctl.ic_dp     = (char *) &arptab_size;
	strioctl.ic_len    = sizeof arptab_size;
	if (ioctl(arpfd, I_STR, &strioctl ) < 0 ) {
		perror("arpbypass: cannot get arptab size");
		exit(1);
	}
	if ( arptab_size <= 0 || arptab_size > 1000) {
		fprintf(stderr, "arpbypass: invalid arptab size (%d)\n", 
			arptab_size);
		exit(1);
	}
	sz = arptab_size * sizeof (struct arptab);
	at = (struct arptab *)malloc(sz);
	if (at == NULL) {
		fprintf(stderr, "arpbypass: can't get memory for arptab\n");
		exit(1);
	}
#define APSZ	64
	for (atp = at, i = 0; i < arptab_size; i += APSZ)
	{
		/* get entire arp table  	*/
		val = (int *)&atp[i];
		val[0] = i; val[1] = min(APSZ, arptab_size - i);
		strioctl.ic_cmd    = ARP_GETARPTAB;
		strioctl.ic_timout = 0;		/* default = 15 seconds */
		strioctl.ic_dp     = (char *)&atp[i];
		strioctl.ic_len    = sizeof (struct arptab) * APSZ;
		if (ioctl(arpfd, I_STR, &strioctl ) < 0 ) {
			perror("arpbypass: error reading arptab");
			exit(1);
		}
	}
	close(arpfd);
	for (; arptab_size-- > 0; at++) {
		if (at->at_in == 0 || at->at_flags == 0)
			continue;
		if (bynumber == 0)
			hp = gethostbyaddr((caddr_t)&at->at_in,
			    sizeof at->at_in, AF_INET);
		else
			hp = 0;
		if (hp)
			host = hp->h_name;
		else {
			host = "?";
			if (errno == TRY_AGAIN)
				bynumber = 1;
		}
		printf("%s (%s) at ", host, inet_ntoa(at->at_in));
		if (at->at_flags & ATF_COM)
			ether_print(at->at_mac);
		else
			printf("(incomplete)");
		if (at->at_flags & ATF_PERM) printf(" permanent");
		if (at->at_flags & ATF_PUBL) printf(" published");
		if (at->at_type == DL_ETHER)
			printf(" ethernet");
		else
			printf(" csmacd");
		printf("\n");
	}
}

ether_print(cp)
	u_char *cp;
{
	printf("%x:%x:%x:%x:%x:%x", cp[0], cp[1], cp[2], cp[3], cp[4], cp[5]);
}

newroute()
{

	if (!strcmp(cmd, "set")) {
		if ((_inet_addr == NULL) || (ether_addr == NULL)) {
			usage(2, cmd);
			return;
		}
	} else if (_inet_addr == NULL) {
		usage(3, cmd);
		return;
	}

	/*
	 *	Get the host Internet address
	 */
	getaddr(_inet_addr, &arpioc.at_in);
	arpioc.at_type = DL_ETHER;

	if(!strcmp(cmd,"set")) {
		switch (options) {
			case 0:
				arpioc.at_flags = ATF_COM;
				break;
			case 1:
				arpioc.at_flags = ATF_COM|ATF_PERM;
				break;
			case 2:
				arpioc.at_flags = ATF_COM|ATF_PUBL;
				break;
			case 3:
				arpioc.at_flags = ATF_COM|ATF_PUBL|ATF_PERM;
				break;
			default:
				arpioc.at_flags = ATF_COM|ATF_PERM;
				break;
		}

		/*
		 *	Get the host Ethernet address
		 */
		if(geteaddr(ether_addr, arpioc.at_mac) < 0) {
			fprintf(stderr,"\n%s: bad format\n", ether_addr);
			exit(1);
		}
	}
 
	/*
	 *	Send an ioctl(2) to /dev/arp to handle command
	 */
	switch(*cmd) {

	case 's':
		strioc.ic_cmd = ARP_RESOLVE;
		break;

	case 'g':
		strioc.ic_cmd = ARP_GET;
		break;

	case 'd':
		strioc.ic_cmd = ARP_DELETE;
		break;

	}
		
	strioc.ic_timout = 60;
	strioc.ic_len = sizeof(struct arptab);
	strioc.ic_dp = (char *)&arpioc;

	if (ioctl(arpfd, I_STR, &strioc) < 0) {
		error(cmd);
		exit(1);
	}

	/*
	 *	Print results
	 */
	if(!strcmp(cmd,"set") && arpioc.at_flags == 0)
		arpioc.at_flags = 2; /* for print statement to look good */
	printf("\n%s inter %s: ether 0x%x.0x%x.0x%x.0x%x.0x%x.0x%x %s\n",
		 cmd,
		 inet_ntoa((struct in_addr *)arpioc.at_in),
		 arpioc.at_mac[0]&0xff,
		 arpioc.at_mac[1]&0xff, arpioc.at_mac[2]&0xff,
		 arpioc.at_mac[3]&0xff,
		 arpioc.at_mac[4]&0xff, arpioc.at_mac[5]&0xff,
		 atf[arpioc.at_flags]);
}

error(cmd)
	char *cmd;
{
	extern int errno;

	switch(errno) {

		case ENOENT:
			fprintf(stderr, "%s: not in ARP table\n",cmd);
			break;

		case EPERM:
			fprintf(stderr,"%s: must have SYSPRV\n",cmd);
			break;

		case ENETUNREACH:
			fprintf(stderr,"%s: no interface for internet address\n",
				cmd);
			break;

		case EADDRNOTAVAIL:
			fprintf(stderr,"%s: No room in ARP table, try later\n",
				cmd);
			break;

		default:
			perror(cmd);
			break;

	}
	exit(1);
}

getaddr(s, inaddr)
	char *s;
	ulong *inaddr;		/* ptr to at_in field */
{
	register struct hostent *hp;

	if (hp = gethostbyname(s)) {
		bcopy((caddr_t)hp->h_addr,(caddr_t)inaddr,hp->h_length);
		return;
	}

	*inaddr = inet_addr(s);
	if (*inaddr == -1) {
		fprintf(stderr, "%s: bad value\n", s);
		exit(1);
	}

	return;
}


/*
 * Ethernet address interpretation routine.
 * The value returned is in network order.
 */
geteaddr(cp, eaddr)
	register char *cp;
	unchar *eaddr;		/* ptr to at_mac field */
{
	register ulong val, base, n, i;
	register char c;
	ulong parts[6], *pp = parts;

again:
	/*
	 * Collect number up to ``.''.
	 * Values are specified as for C:
	 * 0x=hex, 0=octal, other=decimal.
	 */
	val = 0; base = 10;
	if (*cp == '0')
		base = 8, cp++;
	if (*cp == 'x' || *cp == 'X')
		base = 16, cp++;
	while (c = *cp) {
		if (isdigit(c)) {
			val = (val * base) + (c - '0');
			cp++;
			continue;
		}
		if (base == 16 && isxdigit(c)) {
			val = (val << 4) + (c + 10 - (islower(c) ? 'a' : 'A'));
			cp++;
			continue;
		}
		break;
	}
	if (*cp == '.') {
		/*
		 * Ethernet format:
		 *	a.b.c.d.e.f
		 *	a.b.c.d
		 *	a.b
		 */
		if (pp >= parts + 6)
			return (-1);
		*pp++ = val, cp++;
		goto again;
	}

	/*
	 * Check for trailing characters.
	 */
	if (*cp && !isspace(*cp))
		return (-1);
	*pp++ = val;

	/*
	 * Concoct the address according to
	 * the number of parts specified.
	 */
	n = pp - parts;
	switch (n) {

	case 2:				/* a.b -- 24.24 bits */
		eaddr[0] = (parts[0]&0x00ff0000)>>16;
		eaddr[1] = (parts[0]&0x0000ff00)>>8;
		eaddr[2] = (parts[0]&0x000000ff);
		eaddr[3] = (parts[1]&0x00ff0000)>>16;
		eaddr[4] = (parts[1]&0x0000ff00)>>8;
		eaddr[5] = (parts[1]&0x000000ff);
		break;

	case 4:				/* a.b.c.d -- 24.8.8.8 bits */
		eaddr[0] = (parts[0]&0x00ff0000)>>16;
		eaddr[1] = (parts[0]&0x0000ff00)>>8;
		eaddr[2] = (parts[0]&0x000000ff);
		for(i=3; i<6; i++)
			eaddr[i] = (parts[i-2]&0x000000ff);
		break;

	case 6:				/* a.b.c.d.e.f -- 8.8.8.8.8.8 bits */
		for(i=0; i<6; i++)
			eaddr[i] = (parts[i]&0x000000ff);
		break;

	default:
		return (-1);
	}

	return 0;
}

char *
lower(str)
	char *str;
{
	register char *cp = str;

	while (*cp) {
		if (isupper(*cp))
			*cp = tolower(*cp);
		cp++;
	}

	return (str);
}

usage(num, command)
	int num;
	char *command;
{

    switch (num) {

	case 1:

#ifdef NCR
	(void)fprintf(stderr, "usage: arpbypass [-f flags] cmd internet-addr [ethernet-addr]\n");
#else
	(void)fprintf(stderr, "usage: arpbypass [-f flags] [-t type] cmd internet-addr [ethernet-addr]\n");
#endif /* NCR */
	(void)fprintf(stderr, "usage: arpbypass -a \n");
	break;

    	case 2:

#ifdef NCR
	(void)fprintf(stderr, "usage: arpbypass [-f flags] %s internet-addr ethernet-addr\n", command);
#else
	(void)fprintf(stderr, "usage: arpbypass [-f flags] [-t type] %s internet-addr ethernet-addr\n", command);
#endif /* NCR */
	break;

    	case 3:

	(void)fprintf(stderr, "usage: arpbypass %s internet-addr\n", command);
	break;

    }

}
