/*
 * @(#) Copyright 1988.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)udp.c (TWG)  1.14     89/09/18 "

/*
 * udp tli multiplexer
 */

#include "sys/param.h"
#include "sys/types.h"
#include "sys/inline.h"
#include "sys/errno.h"
#include "sys/stream.h"
#include "sys/stropts.h"
#include "sys/tihdr.h"
#include "sys/tiuser.h"
#include "sys/strlog.h"
#ifndef	XENIX
#include "sys/fs/s5dir.h"
#else
#include "sys/dir.h"
#endif
#ifdef u3b2
#include "sys/psw.h"
#include "sys/pcb.h"
#include "sys/immu.h"
#include "sys/region.h"
#include "sys/proc.h"
#endif
#include "sys/signal.h"
#include "sys/user.h"
#include "sys/debug.h"

#include "sys/inet.h"
#include "sys/socket.h"
#include "sys/if.h"
#include "sys/in.h"
#include "sys/in_var.h"
#include "sys/ip.h"
#include "sys/ip_var.h"
#include "sys/udp.h"
#include "sys/somod.h"
#include "sys/inetioctl.h"

extern int udpcksum;
struct udpd *udpd_head = (struct udpd *)NULL;
struct udpstat udpstat;
struct udpb udpb;	/* Bottom Queue structure	*/
extern struct udpd udpd[];
extern int udpd_cnt;
extern void bcopy();

/***********************
 *  Table of Contents  *
 **********************/

static	int	udpopen();	/* top QUEUE function */
static	int	udpclose();	/* top QUEUE function */
static	void	_udpclose();
int		udpisrv();	/* top QUEUE function */
int		udpoput();	/* top QUEUE function */
int		udposrv();	/* top QUEUE function */
void		udp_output();
void		udpmuxnotify();
int		udpbiput();	/* bottom QUEUE function */
void		udpmuxack();
int		udpbisrv();	/* bottom QUEUE function */
void		udp_input();
long		udp_bind();
int		udp_getaddr();
int		udp_sockopt();
int		udp_ipoptions();

/***** End of TOC *****/



static struct module_info 	udp_rinfo = {	/* top Read info */
UDP_ID, "udp", 0, INFPSZ, 65535, 65535 };


static struct module_info 	udp_winfo = {	/* top Write info */
UDP_ID, "udp", 0, INFPSZ, 65535, 65535 };


static struct module_info 	udpb_rinfo = {	/* Bottom Read info */
UDP_ID, "udp", 0, INFPSZ, 65535, 65535 };


static struct module_info 	udpb_winfo = {	/* Bottom Write info */
UDP_ID, "udp", 0, INFPSZ, 0, 0 };


static struct qinit 		udprinit = { 	/* top Read QUEUE */
NULL, udpisrv, udpopen, udpclose, NULL, &udp_rinfo, NULL };


static struct qinit 		udpwinit = { 	/* top Write QUEUE */
udpoput, udposrv, NULL, NULL, NULL, &udp_winfo, NULL };


static struct qinit 		udpbrinit = { 	/* Bottom Read QUEUE */
udpbiput, udpbisrv, NULL, NULL, NULL, &udpb_rinfo, NULL };


static struct qinit 		udpbwinit = { 	/* Bottom Write QUEUE */
NULL, NULL, NULL, NULL, NULL, &udpb_winfo, NULL };


struct streamtab 		udpinfo = {	/* link to cdevsw table */ 
&udprinit, &udpwinit, &udpbrinit, &udpbwinit };




/*
 * may have to open control channel first
 * if dev==CLONEOPEN, then select any free channel to open
 */

static int
udpopen(q, dev, flag, sflag)
	register queue_t *q;
	dev_t dev;
{
	register struct udpd *udp;
	int mdev = minor(dev);

	ASSERT(q);

	mdev = minor(dev);
	if (sflag == CLONEOPEN) {
		/*  Look for first available minor device */
		for (udp = &udpd[0]; udp < &udpd[udpd_cnt]; udp++) {
		        if (udp->ud_rdq == (queue_t *)NULL)
				break;
		}
		/*  Set and test validity of selected minor device */
		if ((mdev = (udp - &udpd[0])) == 0 || mdev >= udpd_cnt) {
			u.u_error = ENODEV;
			return OPENFAIL;
		}

	/* NOT a CLONEOPEN */
	} else {
		/*  Requested minor dev not already open */
		if (mdev < 0 || mdev >= udpd_cnt) {
			u.u_error = ENODEV;
			return OPENFAIL;
		}
		/* Allocate a minor device and continue */
		udp = &udpd[mdev];
	}

	/* Check if requested minor dev already open */
	if (udp == (struct udpd *)q->q_ptr) {
		ASSERT(udp->ud_dev == mdev);
		return mdev;
	}

	/* initialize data structure */
	udp->ud_rdq = q;
	udp->ud_dev = mdev;
	udp->ud_tstate = TS_UNBND;
	udp->ud_flags = u.u_uid ? 0 : UD_SUPERUSER;
	if (udpcksum)
		udp->ud_flags |= (UD_CKSUM_IN | UD_CKSUM_OUT);
	udp->ud_ipopt = NULL;		/* pointer to outgoing IP options */
	udp->ud_ipoptin = NULL;		/* pointer to incoming IP options */
	udp->ud_tos = 0;		/* default Type-of-Service */
	udp->ud_ttl = UDP_TTL;		/* default Time-to-Live */
	udp->ud_next = udpd_head;
	udpd_head = udp;

	q->q_ptr = (caddr_t)udp;
	WR(q)->q_ptr = (caddr_t)udp;
	noenable(WR(q));
	STRlog(UDP_ID, mdev, DPRI_LO, SL_TRACE, "udpopen: open chan %x\n",
		mdev, 0);
	return mdev;

}	/*  End of udpopen()  */

/*
 * if closing control channel, close all other chans and clean up mux
 */

static int
udpclose(q)
	queue_t *q;
{
	register struct udpd *udp, *p;
	register struct udpb *ubp = (struct udpb *)&udpb;

	ASSERT(q && q->q_ptr);

	udp = (struct udpd *)q->q_ptr;
	if (UDPCHAN(udp) == UDPCTLCHAN) {
		for (p = &udpd[0]; p < &udpd[udpd_cnt]; p++)
			if (p != udp && p->ud_rdq)
				_udpclose(p);
		if (ubp->ub_wrq) {
			ubp->ub_mstate = IPIF_CLOSED;
			udpmuxnotify(ubp);
			ubp->ub_wrq = NULL;
			if (ubp->ub_iocblk) {
				/* stream head timeout, don't care iocnak */
				freeb(ubp->ub_iocblk);
				ubp->ub_iocblk = NULL;
			}
		}
	}
	_udpclose(udp);

}	/*  End of udpclose()  */

/*
 * no need to send "disconnect" to user
 */
static void
_udpclose(udp)
     struct udpd *udp;
{
    register struct udpd **p;

    STRlog(UDP_ID, UDPCHAN(udp), DPRI_LO, SL_TRACE,
	   "udpclose: %x\n", UDPCHAN(udp), 0);

    for (p = &udpd_head; *p; p = &((*p)->ud_next)) {
	if (*p == udp) {
	    *p = udp->ud_next;
	    break;
	    }
	}
    if (udp->ud_ipopt)
	freeb(udp->ud_ipopt);
    if (udp->ud_ipoptin)
	freeb(udp->ud_ipoptin);
    if (udp->ud_iocblk)
	freemsg(udp->ud_iocblk);
    bzero((char *)udp, sizeof(struct udpd));

}	/*  End of _udpclose()  */

/*
 * called from udp_input()
 * if canput, just pass the data upward
 * else do nothing (mesg is left in the current q)
 */
udpisrv(q)
	register queue_t *q;
{
	register struct udpd *udp;
	register mblk_t *bp;
	register int chan;

	udp = (struct udpd *)q->q_ptr;
	ASSERT(udp);
	if (udp->ud_tstate != TS_IDLE) {
		chan = UDPCHAN(udp);
		flushq(q, 1);
		STRlog(UDP_ID, chan, DPRI_MED, SL_te,
			"udpisrv: Not idle, chan %x, state %x\n", 
			chan, udp->ud_tstate);
		return;
	}


	while (canput(q->q_next) && (bp = getq(q))) {
		putnext(q, bp);
	}

}	/*  End of udpisrv()  */

static struct T_info_ack UDP_info_ack =
{
	T_INFO_ACK,		/* type */
	INFPSZ,			/* TSDU size */
	-2,			/* ETSDU size */
	0,			/* connect data size */
	0,			/* disconnect data size */
	UDP_ADDRLEN,		/* prot address size */
	0,			/* option size */
	65535,			/* TIDU size */
	T_CLTS,			/* orderly release support */
	TS_IDLE,		/* default current state */
};


#define fill_in_msg( msg, source, count ) \
/*	register mblk_t	*msg;		/* first block of target message */ \
/*	char		*source;	/* source data structure */	\
/*	int		count;		/* bytes to copy from source */ \
{	bcopy( (char *)(source), (msg)->b_rptr, count ); \
	(msg)->b_wptr = (msg)->b_rptr + count ; }

/*
 * udpoput - 
 *
 *	This is the put procedure called by the higher level modules (Closer to
 *	the user).  This routine determines the type of buffer passed to it.
 *	The buffer types passed are typically of type M_PROTO or M_PCPROTO.  If
 *	the protocol message is any other than a unit data request
 *	T_UNITDATA_REQ, it is processed immedeatly.  There are only two ioctl's
 *	recognized by this routine I_LINK is used to link a driver beneath UDP.
 *	this would typically be IP. Only one driver is currently allowed to be
 *	linked. This is because of the fact that UDP has intrinsic knowledge
 *	that IP exists beneath it.  I_UNLINK ioctl is used to unlink a driver
 *	from beneath this module.  Further ioctl's that are not recognized by
 *	this module are not passed down to the driver beneath it. This would be
 *	a breach in security.  Protocol messages of the type T_UNITDATA_REQ are
 *	put on the queue via the putq routine.
 */
udpoput(q, bp)
    register queue_t *q;
    register mblk_t *bp;
{
    register union T_primitives *Tp;
    register struct udpd *udp;
    register unchar Ttype;
    register mblk_t *bp1;
    struct iocblk *ip;
    struct udpb *ubp;
    int cmd;
    queue_t *bq;
    struct linkblk *lp;
    struct T_info_ack *Ti;
    
    ASSERT(q && q->q_ptr && bp);
    
    udp = (struct udpd *)q->q_ptr;
    
    if (udp->ud_iocblk)	{		/* previous must have timed out */
	freemsg(udp->ud_iocblk);
	udp->ud_iocblk = NULL;
	}
    
    switch(MTYPE(bp)) {
	
      case M_PROTO:
      case M_PCPROTO:
	Tp = (union T_primitives *)bp->b_rptr;
	Ttype = (unchar)TTYPE(Tp);
	
	switch(Ttype) {
	    
	  case T_INFO_REQ:
	    if (!REUSEABLE(bp, sizeof(struct T_info_ack))) {
		freemsg(bp);
		if (!(bp = allocb(sizeof(*Ti), BPRI_MED)))
		    goto fatal;
		}
	    
	    MTYPE(bp) = M_PCPROTO;
	    Ti = (struct T_info_ack *)bp->b_wptr;
	    *Ti = UDP_info_ack;
	    Ti->CURRENT_state = udp->ud_tstate;
	    bp->b_wptr += sizeof(struct T_info_ack);
	    putnext((RD(q)), bp);
	    break;
	    
	  case T_UNITDATA_REQ:
	    if (udp->ud_tstate == TS_IDLE) {
		putq(q, bp);
		qenable(q);
		} else
		    freemsg(bp);
	    break;
	    
	  case T_BIND_REQ:
	    {
	    struct T_bind_req *Tr;
	    struct T_bind_ack *Ta;
	    long retcode;
	    struct tsap ta, *ap1;
	    
	    if (udp->ud_tstate != TS_UNBND)
		goto fatal;
	    
	    Tr = (struct T_bind_req *)Tp;
	    
	    if (BLEN(bp) < Tr->ADDR_offset + Tr->ADDR_length) {
		retcode = TBADADDR;
		goto nonfatal;
		}
	    
	    switch (Tr->ADDR_length) {
		
	      case UDP_ADDRLEN:
	      case BINDLEN:
		/*
		 * Copy: because data not aligned.
		 * Ignores that last eight bytes.
		 */
		bcopy((char *)(bp->b_rptr+Tr->ADDR_offset),
		      (char *)&ta, BINDLEN);
		break;
		
	      case 0:
		ta.ta_addr = INADDR_ANY;
		ta.ta_port = 0;
		break;
		
	      default:
		retcode = TBADADDR;
		goto nonfatal;
		}
	    
	    if (retcode = udp_bind(&ta, udp->ud_flags))
		goto nonfatal;
	    
	    /* success, return T_bind_ack */
	    if (!REUSEABLE(bp, sizeof(*Ta)+ UDP_ADDRLEN)) {
		freemsg(bp);
		if (!(bp = allocb(sizeof(*Ta)+ UDP_ADDRLEN,
				  BPRI_LO)))
		    goto fatal;
		}
	    
	    Ta = (struct T_bind_ack *)bp->b_rptr;
	    Ta->PRIM_type = T_BIND_ACK;
	    Ta->ADDR_length = BINDLEN;
	    Ta->ADDR_offset = sizeof(struct T_bind_ack);
	    Ta->CONIND_number = 0;
	    ap1 = (struct tsap *)(bp->b_rptr + Ta->ADDR_offset);
	    ap1->ta_family = udp->ud_la.ta_family = AF_INET;
	    ap1->ta_port = udp->ud_la.ta_port = ta.ta_port;
	    ap1->ta_addr = udp->ud_la.ta_addr = ta.ta_addr;
	    MTYPE(bp) = M_PCPROTO;
	    bp->b_wptr = bp->b_rptr + sizeof(struct T_bind_ack) 
		+ BINDLEN;
	    /* zero out the last 8 bytes in sockaddr_in */
	    if (Tr->ADDR_length == UDP_ADDRLEN) {
		Ta->ADDR_length = UDP_ADDRLEN;
		bzero((char *)bp->b_wptr, 8);
		bp->b_wptr += 8;
		}
	    udp->ud_tstate = TS_IDLE;
	    putnext((RD(q)), bp);
	    break;
	    
	  nonfatal:
	    STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
		   "udpoput: bind nonfatal retcode 0x%x", retcode);
	    if (up_error_ack(q, bp, T_BIND_REQ, retcode))
		goto fatal;
	    break;
	    }
	    
	  case T_UNBIND_REQ:
	    if (upflush(q) == -1 || udp->ud_tstate != TS_IDLE) {
	      fatal:
		if (bp1 = allocb(1, BPRI_HI)) {
		    MTYPE(bp1) = M_ERROR;
		    *bp1->b_wptr++ = EPROTO;
		    putnext((RD(q)), bp1);
		    }
		udp->ud_tstate = TS_UNBND;
		udp->ud_flags = 0;
		freemsg(bp);
		STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te, 
		       "udp fatal unbind %x\n",
		       udp->ud_tstate, 0);
		break;
		}
	    udp->ud_tstate = TS_UNBND;
	    up_ok_ack(q, bp, T_UNBIND_REQ);
	    break;
	    
	  default:
	    STRlog(UDP_ID, UDPCHAN(udp), DPRI_LO, SL_te,
		   "udpoput: unknown PROTO %x\n", TTYPE(Tp), 0);
	    freemsg(bp);
	    break;
	    }
	break;
	
      case M_IOCTL:
	ip = (struct iocblk *)bp->b_rptr;
	cmd = ip->ioc_cmd;
	
	ubp = (struct udpb *)&udpb;
	switch(cmd) {
	    
	  case I_LINK:
	    /*
	     * there is only one bottom q, i.e. for ip
	     */
	    if ((udp->ud_flags & UD_SUPERUSER) == 0){
		ip->ioc_error = EPERM;
		goto nack;
		}
	    lp = (struct linkblk *)bp->b_cont->b_rptr;
	    if (ubp->ub_wrq) {
		STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
		       "udpoput: link to ip again\n", 0, 0);
	      nack:
		if (ip->ioc_error == 0)
		    ip->ioc_error = EINVAL;
		ip->ioc_count = 0;
		MTYPE(bp) = M_IOCNAK;
		iocreply(q, bp);
		break;
		}
	    if (ubp->ub_iocblk) {
		STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
		       "udp bottom busy..\n", 0, 0);
		ip->ioc_error = EAGAIN;
		goto nack;
		}
	    bq = lp->l_qbot;
	    bq->q_ptr = (RD(bq))->q_ptr = (caddr_t)ubp;
	    ubp->ub_wrq = bq;
	    ubp->ub_qtop = /* lp->l_qtop; */ q;
	    ubp->ub_cookie = lp->l_index;
	    ubp->ub_mstate = IPIF_LINKING;
	    /*
	     * keep the M_IOCTL block and change to M_IOCACK
	     * when the M_CTL is acked
	     * actually the only thing needed to be saved is
	     * ioc_id. Saving the block to save time (don't need
	     * freemsg here and allocb later).
	     * Remember to free it when the chan is closing
	     */
	    ubp->ub_iocblk = bp;
	    udpmuxnotify(ubp);
	    break;
	    
	  case I_UNLINK:
	    if (ubp->ub_wrq == NULL) {
		STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
		       "udpoput: ioctl unlink what?\n", 0, 0);
		goto nack;
		}
	    ubp->ub_mstate = IPIF_UNLINKING;
	    ubp->ub_iocblk = bp;
	    udpmuxnotify(ubp);
	    ubp->ub_wrq = NULL;
	    break;
	    
	  case UDPIOC_GETMYSTATE:
	    
	    /* find out what TLI state we are */
	    
	    if ((bp->b_cont == (mblk_t *)NULL) ||
		(ip->ioc_count < sizeof(char))) {
		ip->ioc_error = EINVAL;
		goto nack;
		}
	    
	    bp1 = bp->b_cont;
	    
	    bp1->b_wptr = bp->b_rptr + sizeof (char);
	    *(mtod (bp1, char *)) = udp->ud_tstate;
	    ip->ioc_count = sizeof (char);
	  ack:
	    ip->ioc_error = 0;
	    MTYPE(bp) = M_IOCACK;
	    qreply(q, bp);
	    return;
	    
	  case UDPIOC_GETSOCKNAME:
	  case UDPIOC_GETPEERNAME:
	    {
	    /*
	     * Return address bound to.
	     */
	    struct tsap *tp;
	    
	    if ((bp->b_cont == (mblk_t *)NULL) ||
		(ip->ioc_count < UDP_ADDRLEN))  {
		ip->ioc_error = EINVAL;
		goto nack;
		}
	    bp1 = bp->b_cont;
	    tp = (struct tsap *)bp1->b_rptr;
	    if (ip->ioc_cmd == UDPIOC_GETSOCKNAME)
		*tp = udp->ud_la;
	    else
		*tp = udp->ud_fa;
	    tp->ta_family = AF_INET;
	    bp1->b_wptr = bp1->b_rptr + BINDLEN;
	    bzero((char *)bp1->b_wptr, 8);
	    bp1->b_wptr += 8;
	    ip->ioc_count = UDP_ADDRLEN;
	    goto ack;
	    }
	    
	  case UDPIOC_CONNECT:
	    {
	    /* 
	     * Connect to a specified address. Both address
	     * and port must be specified. Socket library 
	     * connect() will insure that we have a local 
	     * address.
	     */
	    struct tsap *tp;
	    queue_t	*q1;
	    ulong taddr;
	    
	    if (bp->b_cont == (mblk_t *)NULL ||
		ip->ioc_count < BINDLEN) 
		goto nack;
	    bp1 = bp->b_cont;
	    tp = (struct tsap *)bp1->b_rptr;
	    
	    if (udp->ud_flags & UD_CONNECT) {
		if (tp->ta_addr == 0 && tp->ta_port == 0) {
		    /* We are disconnecting. */ 
		    bzero((char *)&udp->ud_fa, 
			  sizeof(udp->ud_fa));
		    udp->ud_lipaddr = 0;
		    udp->ud_flags &= ~UD_CONNECT;
		    ip->ioc_count = 0;
		    goto ack;
		    } else {
			ip->ioc_error = EISCONN;
			goto nack;
			}
		}
	    if (tp->ta_port == 0) {
		ip->ioc_error = EADDRNOTAVAIL;
		goto nack;
		}
	    udp->ud_faddr = tp->ta_addr; 
	    udp->ud_fport = tp->ta_port; 
	    q1 = ((struct udpb *)&udpb)->ub_wrq;
	    udp->ud_iocid = ip->ioc_id;
	    udp->ud_iocblk = bp;

	    /* Get foreign address */
	    if (udp->ud_faddr == INADDR_ANY ||
		udp->ud_faddr == INADDR_BROADCAST) {
		if (!udp_getaddr(udp, q1, IPIF_NGETHISADDR)) {
		    /* Could not establish 
		       destination address. */
		    udp->ud_iocblk = NULL;
		    udp->ud_iocid = 0;
		    ip->ioc_error = TSYSERR;
		    STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
			   "udpoput: udp_getaddr failed");
		    goto nack;
		    }
		}
	    
	    /* Get local address */
	    if (!udp_getaddr(udp, q1, IPIF_NGETMYADDR)) {
		/*
		 * Could not establish my address.
		 */
		ip->ioc_error = TSYSERR;
		udp->ud_iocblk = NULL;
		udp->ud_iocid = 0;
		STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
		       "udpoput: udp_getaddr failed");
		goto nack;
		}
	    return;			/* will be acked on upstream side */
	    }
	    
	  case UDPIOC_DO_CKSUM_IN:
	    udp->ud_flags |= UD_CKSUM_IN;
	    ip->ioc_count = 0;
	    goto ack;	
	    
	  case UDPIOC_DO_CKSUM_OUT:
	    udp->ud_flags |= UD_CKSUM_OUT;
	    ip->ioc_count = 0;
	    goto ack;	
	    
	  case UDPIOC_DONT_CKSUM_IN:
	    udp->ud_flags &= ~UD_CKSUM_IN;
	    ip->ioc_count = 0;
	    goto ack;	
	    
	  case UDPIOC_DONT_CKSUM_OUT:
	    udp->ud_flags &= ~UD_CKSUM_OUT;
	    ip->ioc_count = 0;
	    goto ack;	
	    
	  case UDPIOC_SOCKOPT:
	    /* handle all socket options */
	    if (ip->ioc_error = udp_sockopt(ip, udp, bp))
		goto nack;
	    else
		goto ack;

	  case UDPIOC_SETTOS:	/* Set Type of service */
	    ip->ioc_error = 0;
	    if (ip->ioc_count != sizeof(unsigned char))
		goto nack;
	    udp->ud_tos = *(unsigned char *)bp->b_cont->b_rptr;
	    goto ack;

	  case UDPIOC_GETTOS:	/* Get Type of service */
	    ip->ioc_error = 0;
	    if (ip->ioc_count != sizeof(unsigned char))
		goto nack;
	    *(unsigned char *)bp->b_cont->b_rptr = udp->ud_tos;
	    goto ack;

	  case UDPIOC_SETTTL:	/* Set Time to live */
	    ip->ioc_error = 0;
	    if (ip->ioc_count != sizeof(unsigned char))
		goto nack;
	    udp->ud_ttl = *(unsigned char *)bp->b_cont->b_rptr;
	    goto ack;

	  case UDPIOC_GETTTL:	/* Get Time to live */
	    ip->ioc_error = 0;
	    if (ip->ioc_count != sizeof(unsigned char))
		goto nack;
	    *(unsigned char *)bp->b_cont->b_rptr = udp->ud_ttl;
	    goto ack;

	  case UDPIOC_GETFAMILY:	/* Get address family. */
	    ip->ioc_error = 0;
	    if (ip->ioc_count != sizeof(ushort))
		goto nack;
	    *(ushort *)bp->b_cont->b_rptr = AF_INET;
	    goto ack;
	    
	  case UDPIOC_GETUDPD:
	    {
	    if ((bp1 = bp->b_cont) == NULL
		|| ip->ioc_count < sizeof(struct udpd) 
		|| ! pullupmsg(bp1, sizeof(struct udpd) ) ) {
		ip->ioc_error = EINVAL;
		goto nack;
		}
	    udp = ((struct udpd *)bp1->b_rptr)->ud_next;
	    /* if user passed in an invalid udpd address, then
	     * return the udpd at the list head 
	     */
	    if (udp <  &udpd[ 0 ]
		||  udp >= &udpd[ udpd_cnt ]
		||  ((char *)udp - (char *)&udpd[0]) % sizeof *udp ) {
		udp = udpd_head;
		if (udp == (struct udpd *)0) {
		    ip->ioc_error = EADDRNOTAVAIL;
		    goto nack; 
		    }
		}
	    ip->ioc_count = sizeof(struct udpd);
	    fill_in_msg( bp1, udp, sizeof(struct udpd) );
	    goto ack;
	    }
	    
	  case UDPIOC_GETUDPSTAT:
	    {
	    if ((bp1 = bp->b_cont) == NULL
		|| ip->ioc_count < sizeof udpstat
		|| ! pullupmsg(bp1, sizeof udpstat ) ) {
		ip->ioc_error = EINVAL;
		goto nack;
		}
	    ip->ioc_count = sizeof udpstat;
	    fill_in_msg( bp1, &udpstat, sizeof udpstat );
	    goto ack;
	    }
	    
	  default:
	    /* unrecognised IOCTL - just nack it */
	    ip->ioc_error = EINVAL;
	    goto nack;
	    }
	break;
	
	
      case M_FLUSH:
	endflush(q, bp);
	break;
	
      case M_DATA:
	/* 
	 * if we are connected build a T_unitdata_req,
	 * and queue it for output, else reject it.
	 */
	if (udp->ud_flags & UD_CONNECT) {
		register struct T_unitdata_req *Tp;
		register mblk_t *mp;
		register int size;

		size = sizeof(*Tp) + UDP_ADDRLEN;
		if ((mp = allocb(size, BPRI_MED)) == NULL) {
			freemsg(bp);
			break;
		}	
		MTYPE(mp) = M_PROTO;
		Tp = (struct T_unitdata_req *)mp->b_rptr;
		Tp->PRIM_type = T_UNITDATA_REQ;
		Tp->DEST_length = UDP_ADDRLEN;
		Tp->DEST_offset = sizeof(*Tp);
		Tp->OPT_length = 0;
		Tp->OPT_offset = 0;
		bcopy((caddr_t)&udp->ud_fa, 
		      (caddr_t)Tp + Tp->DEST_offset,
		      sizeof(udp->ud_fa));
		mp->b_wptr = mp->b_rptr + size;
		mp->b_cont = bp;
		putq(q, mp);
		qenable(q);
		break;
	}
	/* fall thru, and free it */
      default:
	STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
	       "udpoput: unknown bp type %x\n", MTYPE(bp), 0);
	freemsg(bp);
	break;
	}
    
}	/*  End of udpoput()  */

/*
 *  udposrv -
 *	This is the service routine for the top end write
 *	queue of UDP. The only messages that are put on
 *	this queue are T_UNITDATA_REQ. These are taken
 *	off the queue and passed to the routine udp_output.
 */

/*
 * all M_PROTOs with T_UNITDATA_REQ
 */
udposrv(q)
	register queue_t *q;
{
	register struct udpd *udp;
	register mblk_t *bp;

	udp = (struct udpd *)q->q_ptr;
	ASSERT(udp);
	while (bp = getq(q)) {
		(void) udp_output(udp, bp);
	}

}	/*  End of udposrv()  */

/*
 *  udp_output -
 *	This routine processes the data that has to be sent out.
 *	The routine in order to calculate the checksum of the
 *	data packet has to know its own internet address. This
 *	is obtained from IP, every time data is sent to
 *	a different destination that one the previous time.
 *	This routine allocates a udpipheader, and fills
 *	in the various fields. It then performs the checksum, and
 *	invokes the put procedure of the module downstream (IP).
 */

/*
 * Send out Data to IP.
 * If we are sending to the same destinations
 * as the last time, then we don't have to
 * know our own source address.
 * otherwise, we have to find our source address
 * from IP for the purpose of checksum calculation.
 */
void
udp_output(udp, bp)
     struct udpd *udp;
     mblk_t *bp;
{
    register struct udpiphdr *ui;
    register struct T_unitdata_req *Tp;
    register queue_t *q;	/* bottom write q */
    register mblk_t *hbp, *dbp, *abp;
    register ushort len;
    struct tsap ta;
    long retcode;
    int ipflags;
    extern ushort in_cksum();

    STRlog(UDP_ID, 0, DPRI_MED, SL_te,"udp_output: Entry\n", 0, 0);
    Tp = (struct T_unitdata_req *)bp->b_rptr;
    ASSERT(Tp->PRIM_type == T_UNITDATA_REQ);

    q = ((struct udpb *)&udpb)->ub_wrq;	/* only 1 bottom q */
    if (q == NULL || q->q_next == NULL || q->q_next->q_flag & QFULL) {
	STRlog(UDP_ID, 0, DPRI_MED, SL_te,
	       "udp_output: ip not linked/full, drop packet\n", 0, 0);
	goto drop;
	}
    if ((BLEN(bp) < (Tp->DEST_offset + Tp->DEST_length))
	|| (Tp->DEST_offset < sizeof(*Tp))
	|| (Tp->DEST_length < BINDLEN)) {
	retcode = TBADADDR;
	STRlog(UDP_ID, 0, DPRI_MED, SL_te,"udp_output: Bad addr.", 0, 0);
	goto nonfatal;
	}
    if((dbp = bp->b_cont) == NULL) {
	retcode = TNODATA;
	STRlog(UDP_ID, 0, DPRI_MED, SL_te,"udp_output: No Data.", 0, 0);
	goto nonfatal;
	}
    if ((hbp = allocb(sizeof(struct udpiphdr), BPRI_LO)) == NULL) {
	retcode = TSYSERR;
	STRlog(UDP_ID, 0, DPRI_MED, SL_te,"udp_output: allocb failed.", 0, 0);
	goto nonfatal;
	}
    /* can't use pointer arithemetic: struct not aligned */
    bcopy((char *)(bp->b_rptr+Tp->DEST_offset), (char *)&ta, BINDLEN);
    freeb(bp);
    bp = (mblk_t *) NULL;
    len = (ushort)msgdsize(dbp);

    ui = (struct udpiphdr *)hbp->b_wptr;
    hbp->b_wptr += sizeof(struct udpiphdr);
    hbp->b_cont = dbp;
    ui->ui_next = ui->ui_prev = 0;
    ui->ui_x1 = 0;
    ui->ui_pr = IPPROTO_UDP;
    ui->ui_len = ui->ui_ulen = htons(len + sizeof(struct udphdr));

    if ((udp->ud_flags & UD_CONNECT) == 0) {
	udp->ud_faddr = ta.ta_addr;
	udp->ud_fport = ta.ta_port;
	}


    if (udp->ud_faddr == INADDR_ANY || udp->ud_faddr == INADDR_BROADCAST) {

	/*
	 * If the destination is INADDR_ANY, get the
	 * primary local address. If it is INADDR_BROADCAST
	 * get the broadcast address for that interface.
	 */
		
	udp_getaddr(udp, q, IPIF_NGETHISADDR);
	if (udp->ud_faddr == 0){
	    /*
	     * Could not establish destination address.
	     */
	    retcode = TBADADDR;
	    freemsg(hbp);
	    STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
		   "udp_output: no dest addr for %x\n",ta.ta_addr);
	    goto nonfatal;
	    }
	}
		
    if ((udp->ud_flags & UD_CONNECT) &&
	(ta.ta_addr != udp->ud_faddr || ta.ta_port != udp->ud_fport)) {
	retcode = TSYSERR;	/* errno = EISCONN; */
	STRlog(UDP_ID, 0, DPRI_MED, SL_te,
	       "udp_output: sending to other than connected peer. Dropped\n");
	goto nonfatal;
	}

    /*
     * If retransmitting to the same destination as
     * one previouly used, then reuse old address.
     * If to a new destination then get my source address
     * from IP.
     */
    udp_getaddr(udp, q, IPIF_NGETMYADDR);
    if (udp->ud_lipaddr == 0){
	/*
	 * Could not establish my address.
	 */
	retcode = TBADADDR;
	freemsg(hbp);
	STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
	       "udp_output: no ip src addr for %x\n", ta.ta_addr,0);
	goto nonfatal;
	}

    ui->ui_dst = udp->ud_faddr;
    ui->ui_dport = udp->ud_fport;
    ui->ui_src = udp->ud_lipaddr;
    ui->ui_sport = udp->ud_sport;
    ui->ui_sum = 0;
    if (udp->ud_flags & UD_CKSUM_OUT) {
	ui->ui_sum = in_cksum(hbp, (int)(len+sizeof(struct udpiphdr)));
	if (ui->ui_sum == 0)
	    ui->ui_sum = -1;
	}

    /* checksum is done, convert src and dst back host format */
    ((struct ip *)ui)->ip_len = len + sizeof(struct udpiphdr);
    ((struct ip *)ui)->ip_hl = IPHLEN >> 2;
    ((struct ip *)ui)->ip_ttl = udp->ud_ttl;
    ((struct ip *)ui)->ip_tos = udp->ud_tos;

    /*
     * See if we have any "arguments" for IP.
     * Only ip options and ip flags at present.
     */
    ipflags = (int)(udp->ud_flags & (IP_ROUTETOIF | IP_ALLOWBROADCAST));
    if (udp->ud_ipopt || ipflags) {
	/*
	 * Set up "arguments" for IP in a message block of type M_PROTO.
	 * The continuation pointer will point to the real data.
	 */
	struct IP_args *ipargs;
	int buflen;
	if (udp->ud_ipopt)
	    buflen = 64;
	else
	    buflen = 16;

	if ((abp = allocb(buflen, BPRI_LO)) == NULL) {
	    retcode = TSYSERR;
	    freemsg(hbp);
	    STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
		   "udp_output: allocb failed for IP arguments");
	    goto nonfatal;
	    }

	MTYPE(abp) = M_PROTO;

	if (ipflags) {
	    ipargs = (struct IP_args *)abp->b_wptr;
	    ipargs->IP_type = IPFLAGS;
	    ipargs->IP_length = sizeof(int);
	    abp->b_wptr += sizeof(struct IP_args);
	    bcopy((caddr_t)&ipflags, abp->b_wptr, sizeof(int));
	    abp->b_wptr += sizeof(int);
	    }

	if (udp->ud_ipopt) {
	    int size = BLEN(udp->ud_ipopt);

	    ipargs = (struct IP_args *)abp->b_wptr;
	    ipargs->IP_type = IPOPTS;
	    ipargs->IP_length = size;
	    abp->b_wptr += sizeof(struct IP_args);
	    bcopy(udp->ud_ipopt->b_rptr, abp->b_wptr, size);
	    abp->b_wptr += size;
	    }

	ipargs = (struct IP_args *)abp->b_wptr;
	ipargs->IP_type = ipargs->IP_length = IPENDLIST;
	abp->b_wptr += sizeof(struct IP_args);

	abp->b_cont = hbp;
		
	} else
	    abp = hbp;

    udpstat.udpOutDatagrams++;
    putnext(q, abp);
    return;
    
  nonfatal:
    if (hbp = allocb(sizeof(struct T_uderror_ind), BPRI_HI)) {
	struct T_uderror_ind *Tp1;

	Tp1 = (struct T_uderror_ind *)hbp->b_wptr;
	Tp1->PRIM_type = T_UDERROR_IND;
	Tp1->DEST_length = Tp->DEST_length;
	Tp1->DEST_offset = Tp->DEST_offset;
	Tp1->OPT_length = Tp->OPT_length;
	Tp1->OPT_offset = Tp->OPT_offset;
	Tp1->ERROR_type = retcode;
	hbp->b_wptr += sizeof(struct T_uderror_ind);
	MTYPE(hbp) = M_PROTO;
	STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
	       "udp_output: Non fatal error %d\n", retcode, 0);
	putnext(udp->ud_rdq, hbp);
	}
  drop:	/* drop the incoming message */
    if(bp)
	freemsg(bp);

}	/*  End of udp_output()  */

/*
 * send "ip_if" to lower module telling you are udp
 */
void
udpmuxnotify(ubp)
	register struct udpb *ubp;
{
	register queue_t *bq;
	register mblk_t *bp;
	register struct ip_if *mp;

	ASSERT(ubp && ubp->ub_wrq && ubp->ub_qtop && ubp->ub_iocblk);
	bq = ubp->ub_wrq;
	if ((bp = allocb(sizeof(struct ip_if), BPRI_MED)) == NULL) {
		/* send M_IOCNAK upstream */
		if (bp = ubp->ub_iocblk) {
			MTYPE(bp) = M_IOCNAK;
			ubp->ub_iocblk = NULL;
			qreply(ubp->ub_qtop, bp);
		}
		return;
	}

	MTYPE(bp) = M_CTL;
	mp = (struct ip_if *)bp->b_wptr;
	mp->ipf_type = ubp->ub_mstate;
	mp->ipf_id = bq->q_qinfo->qi_minfo->mi_idnum;
	bp->b_wptr += sizeof(struct ip_if);
	putnext(bq, bp);

}	/*  End of udpmuxnotify()  */

/*
 * got data from ip
 */
udpbiput(q, bp)
    register queue_t *q;
    register mblk_t *bp;
{
    register struct udpb *ubp;
    register struct ip_if *mp;
    register mblk_t *bpo;
    register struct udpd *udp;

    ASSERT(q && bp);
    ubp = (struct udpb *)q->q_ptr;
    ASSERT(ubp);

    switch (MTYPE(bp)) {

      case M_PROTO:
	bpo = bp;
	bp = unlinkb(bp);
	freeb(bpo);		/* XXXX - save options later */
	/* fall thru... */
	
      case M_DATA:
	putq(q, bp);
	return;

      case M_CTL:
	/*
	 * this is the ack for I_LINK/I_UNLINK from bottom module
	 */
	mp = (struct ip_if *)bp->b_rptr;
	switch(mp->ipf_type) {
		    
	  case IPIF_NGETMYADDR:
          {
	    struct iocblk *iocp;

	    udp = (struct udpd *)mp->ipf_priv;
	    if ((udp->ud_iocblk != NULL) &&
	        ((iocp = (struct iocblk *)udp->ud_iocblk->b_rptr) != NULL))
            {
		if (udp->ud_iocid != mp->ipf_iocid) {
		    STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
			   "udpiput: invalid getmyaddr id");
		    break;
		    }
		MTYPE(udp->ud_iocblk) = M_IOCNAK;
		if (!(udp->ud_lipaddr = mp->ipf_addr)) {
		    STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
			   "udpiput: invalid getmyaddr address");
		    iocp->ioc_error = EINVAL;
		    goto reply;
		    }
		udp->ud_flags |= UD_CONNECT;
		iocp->ioc_error = 0;
		MTYPE(udp->ud_iocblk) = M_IOCACK;
	      reply:
		iocp->ioc_count = 0;
		putnext(udp->ud_rdq, udp->ud_iocblk);
		udp->ud_iocblk = NULL;
	    } else {
		mblk_t *bp1;

		udp->ud_lipaddr = mp->ipf_addr;
		STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
		       "udpiput: getmyaddr address %x", udp->ud_lipaddr);
		}
	    break;
	    }
	  case IPIF_NGETHISADDR:
	    udp = (struct udpd *)mp->ipf_priv;
/*	    if (udp->ud_iocid != mp->ipf_iocid) {
		STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
		       "udpiput: invalid gethisaddr id");
		break;
		}*/
	    if (!(udp->ud_faddr = mp->ipf_addr)) {
		STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
		       "udpiput: invalid gethisaddr address");
		break;
		}
	    STRlog(UDP_ID, UDPCHAN(udp), DPRI_MED, SL_te,
		   "udpiput: gethisaddr address %x", udp->ud_faddr);
	    break;

	  case IPIF_LINKED:
	    ubp->ub_botid = mp->ipf_id;
	    /* fall thru */

	  case IPIF_UNLINKED:
	    udpmuxack(ubp, mp->ipf_type);
	    break;

	  default:
	    STRlog(UDP_ID, 0, DPRI_MED, SL_te,
		   "udpbiput: unkown M_CTL type %x\n", mp->ipf_type, 0);
	    }
	break;

      default:
	STRlog(UDP_ID, 0, DPRI_MED, SL_te,
	       "udpbiput: unkown message type %x\n", MTYPE(bp), 0);
	break;
	}
    freemsg(bp);

}	/*  End of udpbiput()  */


/*
 * send M_IOCACK/M_IOCNAK upstream
 */
void
udpmuxack(ubp, muxtype)
	register struct udpb *ubp;
	char muxtype;
{
	register mblk_t *bp;
	register struct iocblk *ioc;

	bp = ubp->ub_iocblk;
	ASSERT(bp && ubp->ub_qtop);
	ubp->ub_iocblk = NULL;
	if ((muxtype == IPIF_LINKED && ubp->ub_mstate != IPIF_LINKING) ||
	    (muxtype == IPIF_UNLINKED && ubp->ub_mstate != IPIF_UNLINKING)) {
		ubp->ub_mstate = IPIF_UNLINKED;
		MTYPE(bp) = M_IOCNAK;
	} else {
		ubp->ub_mstate = muxtype;
		MTYPE(bp) = M_IOCACK;
	}
	ioc = (struct iocblk *)bp->b_rptr;
	ioc->ioc_count = ioc->ioc_error = 0;
	qreply(ubp->ub_qtop, bp);	/* top control queue */

}	/*  End of udpmuxack()  */

/*
 *  udpbisrv -
 *	This is the service routine for the bottom end read queue.
 *	Messages are basically data that is to be passed to the
 *	user. This routine takes the messages off the queue,
 *	and invokes the udp_input routine.
 */
udpbisrv(q)
	register queue_t *q;
{
	register mblk_t *bp;

	while (bp = getq(q)) { /* must all be M_DATA */
		ASSERT(MTYPE(bp) == M_DATA);
		udp_input(bp);
	}
}

void
udp_input(bp)
	register mblk_t *bp;
{
	register struct udpiphdr *ui;
	register struct udpd *udp;
	register mblk_t *bp1;
	register struct T_unitdata_ind *Tp;
	mblk_t *mopt = NULL;			/* ip option block */
	int len, ulen, errcode;
	struct tsap ta;
	extern ushort in_cksum();

#define	UILOG(x)	{errcode = x; goto err;}

	/*
	 * pull IP and UDP header together in first block
	 */
	ui = (struct udpiphdr *)bp->b_rptr;
	if (BLEN(bp) < sizeof(struct udpiphdr)) {
		if (pullupmsg(bp, sizeof(struct udpiphdr)) == 0) {
			udpstat.udps_hdrops++;
			udpstat.udpInErrors++;
			UILOG(UDPE_HDR);
		}
		ui = (struct udpiphdr *)bp->b_rptr;
	}

	/*  Do the checksum later as desired by each end point.  */

	/*
	 * convert UDP protocol specific fields to host format.
	 */

	ta.ta_addr = ui->ui_dst;
	ta.ta_port = ui->ui_dport;

	for (udp = udpd_head; udp ; udp = udp->ud_next) {
		if (udp->ud_rdq == NULL)
			continue;
		if (udp->ud_sport != ta.ta_port)
			continue;
		/*
		 * Receive from the connected peer only.
		 */
		if (udp->ud_flags & UD_CONNECT) {
		    if (udp->ud_faddr != ui->ui_src ||
		        udp->ud_fport != ui->ui_sport) {
			    STRLOG(UDP_ID, 0, DPRI_MED, SL_TRACE,
			    "udp_input: dropping:connected-peer=%x.%x",
			    ntohl(udp->ud_faddr), ntohs(udp->ud_fport));
			    STRLOG(UDP_ID, 0, DPRI_MED, SL_TRACE,
			    "udp_input: came from %x.%x\n",
			     ntohl(ui->ui_src), ntohs(ui->ui_sport));
			    continue;
		    }
		}

		/*
		 * Check for an exact match or  
		 * INADDR_ANY local address.
		 */
		if (ta.ta_addr != udp->ud_src 
			&& udp->ud_src != INADDR_ANY)
			continue;
		if (udp->ud_rdq->q_flag & QFULL) {
			udpstat.udps_qfull++;
			udpstat.udpInErrors++;
			UILOG(UDPE_FULL);
		} else if (udp->ud_tstate != TS_IDLE) {
			/*
			 * Is this a problem..
			 */
			udpstat.udps_busy++;
			udpstat.udpInErrors++;
			UILOG(UDPE_IDLE);
		}

		/*
		 * checksum extended UDP header and data (if desired).
		 */
		ulen = ((struct ip *)ui)->ip_len; /* ulen: udp header + data */

		/*
		 * the spec says the len should be ulen + 12 (pseudo header) ???
		 * but we use the whole ip header (20 bytes) here for checksum
		 */
		len = sizeof(struct ip) + ulen;
		if ((udp->ud_flags & UD_CKSUM_IN) && ui->ui_sum){
			ui->ui_next = ui->ui_prev = 0;
			ui->ui_x1 = 0;
			ui->ui_len = htons((ushort)ulen);
			if (in_cksum(bp, (int)len)) {
				udpstat.udps_badsum++;
				udpstat.udpInErrors++;
				UILOG(UDPE_SUM);
			}
		}

		/* drop udp & ip header */
		bp->b_rptr += sizeof(struct udpiphdr);

#define LEN	sizeof(struct T_unitdata_ind) + UDP_ADDRLEN

		if ((bp1 = allocb(LEN, BPRI_LO)) == NULL)
			UILOG(UDPE_ALLOCB);
		MTYPE(bp1) = M_PROTO;
		Tp = (struct T_unitdata_ind *)bp1->b_wptr;
		Tp->PRIM_type = T_UNITDATA_IND;
		Tp->SRC_length = UDP_ADDRLEN;
		Tp->SRC_offset = sizeof(struct T_unitdata_ind);
		Tp->OPT_length = 0;
		Tp->OPT_offset = sizeof(struct T_unitdata_ind);
		ta.ta_family = AF_INET;
		ta.ta_addr = ui->ui_src;
		ta.ta_port = ui->ui_sport;
		bcopy((char *)&ta, (char *)(bp1->b_wptr +
			sizeof(struct T_unitdata_ind)), BINDLEN);
		bp1->b_wptr = bp1->b_rptr + sizeof(struct T_unitdata_ind) +
			      BINDLEN;
		bzero((char *)bp1->b_wptr, 8);
		bp1->b_wptr += 8;
		bp1->b_cont = bp;
#undef LEN
		/*
	 	 * store the incoming IP options away
	 	 */
		if (udp->ud_ipoptin) {
			freeb(udp->ud_ipoptin);
			udp->ud_ipoptin = NULL;
		}
		if (mopt) {
			udp->ud_ipoptin = mopt;
			mopt = NULL;
		}

		STRLOG(UDP_ID, UDPCHAN(udp), DPRI_LO, SL_TRACE,
		  "udp_input: sent up from %x.%x\n", ta.ta_addr, ta.ta_port);
		udpstat.udpInDatagrams++;
		putq(udp->ud_rdq, bp1);
		return;
	}
	udpstat.udpNoPorts++;
	UILOG(UDPE_LOOKUP);
err:
	STRlog(UDP_ID, 0, DPRI_MED, SL_te,
		"udp_input: log code %x\n", errcode, 0);

	/* drop the messages */
	if (mopt)
		freeb(mopt);
	freemsg(bp);
	return;

}	/*  End of udp_input()  */

ushort udpport = UDPPORT_RESERVED;		/* XXX */
/*
 * Bind to a unused port number.
 * if none specified.
 */

/*
 *  udp_bind -
 *	This routine is used to find the next available port number
 *	to which a user can bind. If the user requested a port (non zero),
 *	the routine just checks to see it there is already a
 *	stream bound to that port. If there is one, then an error is returned.
 *	If the user requests a port which is less than UDP_PORTRESERVED,
 *	and is not the super user, then error is returned. If no any port
 *	is requested, this routine just picks any available port number,
 *	that is not in use.
 */
long
udp_bind(ta, flags)
	struct tsap *ta;
	ushort flags;
{
	register struct udpd *up;
	register int nxt;
	register int port;

	port = ntohs(ta->ta_port);

	/*
	 * Find an acceptable port.
	 * If user is root then allow lower than
	 * UDPPORT_RESERVED.
	 */
	if (port && (port < UDPPORT_RESERVED) && ((flags & UD_SUPERUSER) == 0))
		return TACCES;

	if (++udpport == 0xffff)
		udpport = UDPPORT_RESERVED;

	nxt = htons(udpport);
	port = htons(port);
	for (up = udpd_head; up ; up = up->ud_next) {
		/*
		 * Find the requested address.
		 */
		if (port && up->ud_sport == port && up->ud_laddr == ta->ta_addr)
			if (!(flags & SO_REUSEADDR))
				return TNOADDR;

		if ((port == 0) && (nxt == up->ud_sport)) {
		    if (++udpport == 0xffff)
			udpport = UDPPORT_RESERVED;
		    nxt = htons(udpport);
		    }
	}

	if (port)
		ta->ta_port = port;
	else
		ta->ta_port = nxt;

	return 0;

}	/*  End of udp_bind()  */

#undef UILOG

udp_getaddr(udp,q,type)
     struct udpd *udp;
     queue_t *q;
     int type;
{
    register mblk_t *bp;
    register struct ip_if *tpf;

    if ((bp = allocb( sizeof(struct ip_if), BPRI_MED))==NULL)
	return(0);
    tpf = (struct ip_if *)bp->b_wptr;
    bp->b_wptr += sizeof(struct ip_if);
    MTYPE(bp) = M_CTL;
    tpf->ipf_type = type;
    tpf->ipf_addr = udp->ud_faddr;
    tpf->ipf_iocid = udp->ud_iocid;
    tpf->ipf_priv = (char *)udp;
    tpf->ipf_tos = udp->ud_tos;
    putnext(q,bp);
    return(1);

}	/*  End of udp_getaddr()  */


/*
 *	udp_sockopt() allows users to set or get various "socket" options.
 *
 *	The beginning of bp->cont is expected to contain a winsopt 
 *	structure which comprises the following members:
 *
 *		1. level - this is the protocol level
 *		2. name  - this is the option name
 *		3. flags - this indicates the action to take (i.e. get or set)
 */
udp_sockopt(ip, udp, bp)
	register struct iocblk *ip;
	register struct udpd *udp;
	register mblk_t *bp;
{
	register mblk_t *bp1 = bp->b_cont;
	register struct winsopt *winsopt;
	int *tmp;
	int ans = EINVAL;
	int len;
	int ret;

	if (bp1 == NULL || BLEN(bp1) < sizeof(struct winsopt))
		goto nonfatal;

	winsopt = (struct winsopt *)bp1->b_rptr;

	/*
	 * Let's see if the option level is really for us. We'll
	 * accept SOCKET level and UDP level options.   IP level
	 * options will be sent down to IP using a M_CTL message.
	 */
	if (winsopt->level == IPPROTO_IP) {
		if ((ans = udp_ipoptions(bp, udp, winsopt->flags)) == 0) {
			bp1 = bp->b_cont;
			if (bp1)
				ip->ioc_count = BLEN(bp1);
			else
				ip->ioc_count = 0;
			return ans;
		}
		goto nonfatal;
	} else if (winsopt->level != IPPROTO_UDP &&
						winsopt->level != SOL_SOCKET) {
		/* reject option */
		ans = ENOPROTOOPT;
		goto nonfatal;
	}

	/* Only integers expected */
	if ((len = BLEN(bp1) - sizeof(struct winsopt)) != sizeof(int))
		goto nonfatal;

	if (winsopt->flags == T_NEGOTIATE) {
		/* set option */
		tmp = (int *)(bp1->b_rptr + sizeof(struct winsopt));

		switch (winsopt->name) {

		case SO_REUSEADDR:
		case SO_DONTROUTE:
		case SO_BROADCAST:
			/*
			 * Turn option on or off.
			 */

			if (*tmp)
			    udp->ud_flags |= winsopt->name;  /* set flag   */
			else
			    udp->ud_flags &= ~winsopt->name; /* reset flag */

			break;

		default:
			ans = ENOPROTOOPT;
			goto nonfatal;
		}

	} else if (winsopt->flags == T_DEFAULT) {
		/* get option */
		switch (winsopt->name) {

		case SO_REUSEADDR:
		case SO_DONTROUTE:
		case SO_BROADCAST:
			/*
			 * Return option setting.
			 */
			ret = (int)(winsopt->name & udp->ud_flags);
			tmp = &ret;
			break;

		default:
			ans = ENOPROTOOPT;
			goto nonfatal;
		}

	} else
		/* unknown request */
		goto nonfatal;

	bcopy((caddr_t)tmp, (caddr_t)bp1->b_rptr, len);
	ip->ioc_count = len;
	bp1->b_wptr = bp1->b_rptr + len;
	return 0;

nonfatal:
	STRlog(UDP_ID, -1, DPRI_LO, SL_te,
			"udp_sockopt: nonfatal error %0x\n", ans);
	return ans;

}	/*  End of udp_sockopt()  */


/*
 * Options meant for IP - send them down.
 */
udp_ipoptions(bp, udp, flags)
	register mblk_t *bp;
	register struct udpd *udp;
	register int flags;
{
	register mblk_t *ifbp;
	register mblk_t *bp1 = bp->b_cont;
	ipfcmd *iftype;
	int ans = 0;
	extern char udpopt_out; /* see /etc/master.d/udp */

	if ((ifbp = allocb(sizeof(ipfcmd), BPRI_MED)) == NULL) {
		STRlog(UDP_ID, -1, DPRI_HI, SL_te,
			"udp_sendipopts: out of buffer\n");
		return -1;
	}

	bp->b_cont = NULL;

	/*
	 * ifbp's continuation will tell IP what to do with the sent options.
	 * It contains the winsopt structure and data as sent down by the user.
	 */
	ifbp->b_cont = bp1;

	/*
	 * bp1's continuation pointer will point to the ip options message
	 * block. This could be NULL if there are currently no options set.
	 *
	 * If udpopt_out is set then get the outgoing IP options
	 * else get the incoming IP options.
	 */
	if ((udpopt_out) || flags == T_NEGOTIATE)
		bp1->b_cont = udp->ud_ipopt; /* set (or get) outgoing opts. */
	else
		bp1->b_cont = udp->ud_ipoptin; /* get incoming options */

	iftype = (ipfcmd *)ifbp->b_wptr;
	*iftype = IPIF_OPTIONS;
	ifbp->b_wptr += sizeof(ipfcmd);

	MTYPE(ifbp) = M_CTL;
	putnext(udpb.ub_wrq, ifbp); /* send message to IP */

	/* iftype indicates success or failure on return */
	iftype = (ipfcmd *)ifbp->b_rptr;
	if (*iftype == 0) {
		if (flags == T_NEGOTIATE) {
			/* setting - update udp's option pointer */
			if (!udp->ud_ipopt)
				udp->ud_ipopt = ifbp->b_cont;
		} else {
			/* getting - update b_cont for user */
			bp->b_cont = ifbp->b_cont;
		}
	} else {
		STRlog(UDP_ID, -1, DPRI_HI, SL_te,
			"udp_ipoptions: error 0x%x from ip", *iftype);
		ans = (int)*iftype;
	}

	freeb(ifbp);
	return ans;

}	/*  End of udp_ipoptions()  */

