/*
 * @(#) Copyright 1988.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)tcp_tv.c (TWG)  1.5     89/08/09 "

#include "sys/param.h"
#include "sys/types.h"
#ifndef XENIX
#include "sys/inline.h"
#endif
#include "sys/stream.h"
#include "sys/stropts.h"
#include "sys/tihdr.h"
#include "sys/tiuser.h"
#include "sys/strlog.h"
#include "sys/debug.h"
#ifdef XENIX
#include "sys/assert.h"
#endif
#include "sys/inet.h"
#include "sys/socket.h"
#include "sys/if.h"
#include "sys/in.h"
#include "sys/in_var.h"
#include "sys/ip.h"
#include "sys/tcp.h"


/*
 * input events to the tcp finite state machine
 */

/************************
 *  Table of Contents   *
 ***********************/

mblk_t		*tv_popen();
int		tv_abort();
int		tv_close();
int 		tv_aopen();
int		tv_template();
int 		tv_check_addr();

/***** End of TOC *****/

extern	int	tcp_maxrcvwin;
extern struct tcpstat tcpstat;
/*
 * tv_popen - passive open
 *	alloc a tcpcb and points it back to 'tdp'.
 *	return the block just allocated, or NULL if failed
 *	note: tcp enters LISTEN state
 */
mblk_t *
tv_popen(tdp)
     register struct tcpd *tdp;
{
    register struct tcpcb *tp;
    register mblk_t *bp;

    if (bp = allocb(sizeof(struct tcpcb), BPRI_MED)) {
	tp = (struct tcpcb *)bp->b_wptr;
	bzero(tp, sizeof(struct tcpcb));
	tp->t_tdp = tdp;
	tp->t_rsq.b_next = tp->t_rsq.b_prev = (struct qhead *)&tp->t_rsq;
	tp->t_conq.b_next = tp->t_conq.b_prev =	(struct qhead *)&tp->t_conq;
	tp->t_state = TV_LISTEN;
	tp->t_laddr = tp->t_faddr = INADDR_ANY;
	tp->t_maxseg = 1460;	/* XXX, should consult next module */
	tp->snd_cwnd = t_rhiw(tdp);
	tp->t_srtt = TCPTV_SRTTBASE;
	tp->t_rttvar = TCPTV_SRTTDFLT << 2;
	TCPT_RANGESET(tp->t_rxtcur, 
		      ((TCPTV_SRTTBASE >> 2) + (TCPTV_SRTTDFLT << 2)) >> 1,
		      TCPTV_MIN, TCPTV_REXMTMAX);
	bp->b_wptr += sizeof(struct tcpcb);
	}
    return(bp);

}	/*  End of tv_popen()  */

/*
 * tv_abort - abort the connection (as opposed to orderly release)
 *	tp is the tcpcb
 *	CLOSED : return -1;
 *	LISTEN/CLOSING/LAST_ACK/TIME_WAIT : enter CLOSED, return 0;
 *	SYN_RCVD/ESTABLISHED/FIN_W1/FIN_W2/CLOSE_WAIT : send RST, flush
 *		the send/recv queues, enter CLOSED, return 0;
 */
int
tv_abort(tp)

	register struct tcpcb *tp;
{
	switch(tp->t_state) {

	case TV_SYN_RCVD:
	case TV_ESTABLISHED:
	case TV_FIN_W1:
	case TV_FIN_W2:
	case TV_CLOSE_WAIT:
		/*
		 * send RST out to remote
		 * maybe we should call tw_send directly
		 */
		tp->t_state = TV_CLOSED;
		tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;	/* PDF TIMERS */
		(void) tcp_output(tp);
		break;

	case TV_LISTEN:
	case TV_CLOSING:
	case TV_LAST_ACK:
	case TV_TIME_WAIT:
		tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;	/* PDF TIMERS */
		tp->t_state = TV_CLOSED;
		break;

	default:
		tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;	/* PDF TIMERS */
		return(-1);
	}

	return(0);

}	/*  End of tv_abort()  */

/*
 * tv_close - local user orderly releases, tell remote
 *	tp is the tcpcb
 *	CLOSED/FIN_W1/FIN_W2/CLOSING/LAST_ACK/TIME_WAIT/
 *		LISTEN/SYN_SENT: return -1;
 *	SYN_RCVD/ESATAB: send all pending data, then send FIN,
 *			enter FIN_WAIT_1, return 0;
 *	CLOSE_WAIT: send all pending data, then send FIN,
 *			enter LAST_ACK, return 0;
 */
int
tv_close(tp)

	register struct tcpcb *tp;
{
	STRLOG(TCP_ID, TCPCHAN(tp->t_tdp), DPRI_MED, SL_TRACE,
		"tv_close: incoming state = %x\n", tp->t_state);

	switch(tp->t_state) {

	case TV_SYN_RCVD:
	case TV_ESTABLISHED:
		tp->t_state = TV_FIN_W1;
		break;

	case TV_CLOSE_WAIT:
		tp->t_state = TV_LAST_ACK;
		tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;	/* PDF TIMERS */
		break;

	case TV_LISTEN:
	case TV_SYN_SENT:
	case TV_CLOSED:
		/*
		 * Dump the TCPCB. (Caller Will do it ??).
		 */
	default:
		tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;	/* PDF TIMERS */
		tp->t_state = TV_CLOSED;
		return -1;

	case TV_FIN_W1:
	case TV_FIN_W2:
		tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;	/* PDF TIMERS */
		return 0;	/* This is OK (Page 60) */
	}

	STRLOG(TCP_ID, TCPCHAN(tp->t_tdp), DPRI_MED, SL_TRACE,
		"tv_close: outgoing state = %x\n", tp->t_state);

	(void) tcp_output(tp);		/* send pending output and send FIN */
	return 0;

}	/*  End of tv_close()  */

/*
 * tv_aopen - active open
 *	tp points to tcpcb, ap is the foreign address you want to connect
 *	send SYN, enter SYN_SENT
 *	return 0 if o.k. -1 if failed
 */
int
tv_aopen(tp, ap)

	register struct tcpcb *tp;
	register struct tsap *ap;
{
	tp->t_fport = ap->ta_port;
	tp->t_faddr = ap->ta_addr;
	/*
	 * Make sure that this will not
	 * Cause a duplicate session to be created. (Remsh might do it).
	 */
	if (tv_check_addr (tp))
		return TBADADDR;
	if (tv_template(tp))
		return TBUFOVFLW;
	tcpstat.tcps_connattempt++;
	tp->t_state = TV_SYN_SENT;
	tp->t_timer[TCPT_KEEP] = TCPTV_KEEP_INIT;
	tp->iss = tcp_iss;
	tcp_iss += TCP_ISSINCR/2;
	tcp_sendseqinit(tp);
	tw_send(tp, NULL, NULL, tp->snd_nxt, tp->rcv_nxt, TH_SYN);
	tcpstat.tcpActiveOpens++;
	return(0);

}	/*  End of tv_aopen()  */

/*
 * tv_template - internal utility
 *	alloc template, init send seq #, set timer, calls tcp_output
 *	returns 0 if o.k. -1 if failed
 */
int
tv_template(tp)

	register struct tcpcb *tp;
{
	register mblk_t *bp;
	register struct tcpiphdr *ti;

	ASSERT(tp && tp->t_tdp);

	if ((bp = allocb(TCPIPHLEN, BPRI_MED)) == NULL) {
		STRlog(TCP_ID, TCPCHAN(tp->t_tdp), DPRI_LO, SL_te,
			"tv_template: template alloc failed\n");
		return -1;
	}
	/*
	 * initialize template and make it ready to be used by keep-alive
	 * this template will also be used (e.g. in tcp_output) and modified
	 * (e.g. the length, seq, ack, checksum, etc.)
	 */

	ti = (struct tcpiphdr *)bp->b_rptr;
	ti->ti_next = ti->ti_prev = 0;
	ti->ti_x1 = 0;
	ti->ti_pr = IPPROTO_TCP;
	ti->ti_len = htons(TCPHLEN);
	ti->ti_src = tp->t_laddr;
	ti->ti_dst = tp->t_faddr;
	ti->ti_sport = tp->t_lport;
	ti->ti_dport = tp->t_fport;
	ti->ti_seq = 0;
	ti->ti_ack = 0;
	ti->ti_x2 = 0;
	ti->ti_off = 5;
	ti->ti_flags = 0;
	ti->ti_win = 0;
	ti->ti_sum = 0;
	ti->ti_urp = 0;
	bp->b_wptr += TCPIPHLEN;
	tp->t_template = bp;
	return(0);

}	/*  End of tv_template()  */


int
tv_check_addr(tp)

	register struct tcpcb *tp;
{
	/*
	 * We have to look up the TCPCB structures
	 * If there would be a duplicate, then 
	 * signal error.
	 */
	register mblk_t *bp;
	register struct tcpcb *tp1;

	for (bp = ((mblk_t *)&tcpcb)->b_next; bp != (mblk_t *)&tcpcb;
		bp = bp->b_next)
	{
		tp1 = mtod (bp, struct tcpcb *);
		if (tp1 == tp)
			continue;
		if ((tp1->t_fport == tp->t_fport) &&
		    (tp1->t_lport == tp->t_lport) &&
		    (tp1->t_laddr == tp->t_laddr) &&
		    (tp1->t_faddr == tp->t_faddr))
			return -1;
	}
	return 0;

}	/*  End of tv_check_addr()  */
