/* LINTLIBRARY */
/*
 * @(#) Copyright 1988.  The Wollongong Group, Inc.  All Rights Reserved.
 *
 * Fixes: ID	SPR#	Description
 *  	  nw2	4696	rlogin to multiple IP interfaces("Connection reset").
 */

#ident "@(#)tcp_tu.c (TWG)  1.8     89/08/17 "

#include "sys/param.h"
#include "sys/types.h"
#include "sys/inline.h"
#include "sys/stream.h"
#include "sys/stropts.h"
#include "sys/errno.h"
#include "sys/tihdr.h"
#include "sys/tiuser.h"
#include "sys/strlog.h"
#include "sys/debug.h"
#include "sys/inet.h"
#include "sys/socket.h"
#include "sys/if.h"
#include "sys/in.h"
#include "sys/in_var.h"
#include "sys/ip.h"
#include "sys/tcp.h"


#define	z	UNREACH

unchar tufsm_tbl[TE_NOEVENTS][TS_NOSTATES] = {

/* 0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 */
/* U  W  W  I  W  W  W  W  W  D  W  W  W  W  W  W  W */
/* N  A  A  D  A  A  C  R  A  A  I  R  A  A  A  A  A */
/* B  C  C  L  C  C  O  E  C  T  N  E  C  C  C  C  C */
/* N  K  K  E  K  K  N  S  K  A  D  Q  K  K  K  K  K */
/* D  :  :     :  :  :  :  :  :  :  :  :  :  :  :  : */
/*    B  U     O  C  C  C  C  X  O  O  D  D  D  D  D */
/*    R  R     P  R  R  I  R  F  R  R  R  R  R  R  R */
/*    E  E     T  E  E  N  E  E  D  D  E  E  E  E  E */
/*    Q  Q     R  Q  Q  D  S  R  R  R  Q  Q  Q  Q  Q */
/*             E                 E  E  6  7  9  1  1 */
/*             Q                 L  L           0  1 */

  {5, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z}, /* BIND_REQ	0 */
  {z, z, z, 6, z, z, z, z, z, z, z, z, z, z, z, z, z}, /* UNBIND_REQ	1 */
  {z, z, z, 7, z, z, z, z, z, z, z, z, z, z, z, z, z}, /* OPTMGMT_REQ	2 */
  {z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z}, /* BIND_ACK	3 */
  {z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z}, /* OPTMGMT_ACK	4 */
  {z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z}, /* ERROR_ACK	5 */
  {z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z}, /* OK_ACK1	6 */
  {z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z}, /* OK_ACK2	7 */
  {z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z}, /* OK_ACK3	8 */
  {z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z}, /* OK_ACK4	9 */
  {z, z, z, 1, z, z, z, z, z, z, z, z, z, z, z, z, z}, /* TE_CONN_REQ	10 */
  {z, z, z, z, z, z, z, 2, z, z, z, z, z, z, z, z, z}, /* CONN_RES	11 */
  {z, z, z, z, z, z, 3, 3, z, 3, 3, 3, z, z, z, z, z}, /* DISCON_REQ	12 */
  {z, z, z, z, z, z, z, z, z, 4, z, 4, z, z, z, z, z}, /* DATA_REQ	13 */
  {z, z, z, z, z, z, z, z, z, 9, z, 9, z, z, z, z, z}, /* EXDATA_REQ	14 */
  {z, z, z, z, z, z, z, z, z, 8, z, 8, z, z, z, z, z}, /* ORDREL_REQ	15 */
};

#undef z

/***********************
 *  Table of Contents  *
 **********************/

int		tufsm();
unchar 		tu_nonfatal();
unchar		tu_fatal(); 
int		upflush();
int		up_ok_ack();
int		up_error_ack();
void		tu_cleancb();
void		tu_getinfo();
unchar		tu_unbind(); 
unchar		tu_bind(); 
unchar		tu_optmgmt();
unchar		tu_disconreq(); 
unchar 		tu_connreq(); 
unchar		tu_connres(); 
unchar		tu_ordrel();
unchar		tu_data();
unchar		tu_exdata();
long		tcp_bind();

/****** End of TOC *****/

extern mblk_t *tcp_pcbcache;

unchar (*tufsm_action[])() = {
	tu_nonfatal,		/* 0 */
	tu_connreq,		/* 1 */
	tu_connres,		/* 2 */
	tu_disconreq,		/* 3 */
	tu_data,		/* 4 */
	tu_bind,		/* 5 */
	tu_unbind,		/* 6 */
	tu_optmgmt,		/* 7 */
	tu_ordrel,		/* 8 */
	tu_exdata,		/* 9 */
	tu_fatal,		/* 10 */
};

/*
 * from T_primitives to the events of the fsm
 */
#define NUPRIM	T_ORDREL_REQ+1	/* # of primitives from user */
static unchar prim_to_event[NUPRIM] = {
	TE_CONN_REQ,		/* 0. T_CONN_REQ	*/
	TE_CONN_RES,		/* 1. T_CONN_RES	*/
	TE_DISCON_REQ,		/* 2. T_DISCON_REQ	*/
	TE_DATA_REQ,		/* 3. T_DATA_REQ	*/
	TE_EXDATA_REQ,		/* 4. T_EXDATA_REQ	*/
	UNREACH,		/* 5. T_INFO_REQ	*/
	TE_BIND_REQ,		/* 6. T_BIND_REQ	*/
	TE_UNBIND_REQ,		/* 7. T_UNBIND_REQ	*/
	UNREACH,		/* 8. T_UNINTDATA_REQ	*/
	TE_OPTMGMT_REQ,		/* 9. T_OPTMGMT_REQ	*/
	TE_ORDREL_REQ,		/* 10. T_ORDREL_REQ	*/
};

/*
 * tufsm - tcp tli part of finite state machine
 *	input: q (downward q) and bp (with T_primitives)
 *	state is changed in each action routine (e.g. tu_bind)
 *	return 0 if o.k., -1 if failed
 */
int
tufsm(q, bp)

	queue_t *q;
	register mblk_t *bp;
{
	register struct tcpd *tdp;
	register unchar Ttype, actindex;
	register union T_primitives *Tp;

	tdp = (struct tcpd *)q->q_ptr;
	Tp = (union T_primitives *)bp->b_rptr;
	Ttype = (unchar)TTYPE(Tp);
	actindex = tufsm_tbl[prim_to_event[Ttype]][tdp->td_tstate];
	if (actindex == UNREACH)
		goto fsmerr;

	if ((*tufsm_action[actindex])(bp, q) == UNREACH)
		goto fsmerr;
	
	STRLOG(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_TRACE,
		"tufsm: Ttype %x, final_state %x", Ttype, tdp->td_tstate);
	return(0);

fsmerr:
	tdp->td_tstate = TS_UNBND;
	if (up_error_ack(q, bp, (long)Ttype, TOUTSTATE) < 0) {
	    tu_fatal(tdp);
	} else {
	    tu_cleancb(tdp);
	    }
	return(-1);

}	/*  End of tufsm()  */

/*
 * nonfatal error
 * TO BE CONTINUED
 */
unchar
tu_nonfatal(tdp)

	register struct tcpd *tdp;
{

}	/*  End of tu_nonfatal()  */

/*
 * fatal error
 * send TOUTSTATE upstream, reset tli state, and cleanup tcpcb
 */
unchar
tu_fatal(tdp)

	register struct tcpd *tdp;
{
	register queue_t *q;

	STRlog(TCP_ID, TCPCHAN(tdp), DPRI_HI, SL_te,
		"tu_fatal: state %x", tdp->td_tstate);
	q = tdp->td_rdq;
	tdp->td_flags |= TD_ERROR;

	tu_cleancb(tdp);
	tdp->td_tstate = TS_UNBND;
	/*
	 * tell user: expect user to close the stream
	 */
	if (putctl1(q->q_next, M_ERROR, EPROTO) == 0)
		return(UNREACH);

	/*
	 * abort all connections or half connections, free tcpcb
	 */
	return(UNREACH);

}	/*  End of tu_fatal()  */
/*
 * clean tcpcb associated with tdp
 */
void
tu_cleancb(tdp)
	struct tcpd *tdp;

{
	register struct tcpcb *tp;
	register mblk_t *bp, *bp1, *bp2, *head;
	extern void tw_clean();

	ASSERT(tdp);

	if ((bp = tdp->td_tcpcb) == NULL)
		return;
	tp = (struct tcpcb *)bp->b_rptr;
	/*
	 * discard stuff in conq and tell peer we are aborting
	 */
	TRACE(1<<15,"tu_cleancb..",0);
	head = (mblk_t *)&tp->t_conq;
	bp1 = head->b_next;
	while (bp1 != head) {
		bp2 = bp1;
		bp1 = bp1->b_next;
		STRLOG(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_TRACE,
			"tu_cleancb: aborting conq\n");
		tv_abort((struct tcpcb *)bp2->b_rptr);
		(void) tw_clean (mtod (bp2, struct tcpcb *));
		c_deq(bp2);
		freeb(bp2);
	}
	/*
	 * aborting the listener channel: set to TV_CLOSED
	 * however, send RST is meaningless because it has no peer??
	 */
	tv_abort(tp);
	/*
	 * discard all stuff in reassembly queue and template
	 */
	tw_clean(tp);
#ifdef notdef
	head = (mblk_t *)&tcpcb;
	for (bp1 = head->b_next; bp1 != head; bp1 = bp1->b_next) {
		if (bp1 == bp) {
			c_deq(bp1);
			goto free;
			}
	}
	STRlog(TCP_ID, TCPCHAN(tdp), DPRI_MED, SL_te,
		"tu_cleancb: not on tcpcb\n");
#else
	if (bp == tcp_pcbcache)
	    tcp_pcbcache = (mblk_t *)NULL;
	c_deq(bp);
#endif /* notdef */
#ifdef notdef
free:
#endif /* notdef */
	freeb(bp);
	tdp->td_tcpcb = NULL;
	tdp->td_flags = 0;
	wakeup(tdp);
	return;

}	/*  End of tu_cleancb()  */

/*
 * the following size limits (for getinfo) are randomly chosen
 */
static struct T_info_ack TCP_info_ack =
{
	T_INFO_ACK,		/* type */
	0,			/* TSDU size */
	-2,			/* ETSDU size (nw see tiu_data, tu_exdata)*/
	0,			/* connect data size */
	0,			/* disconnect data size */
	TCP_ADDRLEN, 		/* prot address size */
	0,			/* option size */
	4096,			/* TIDU size */
	T_COTS_ORD,		/* orderly release support */
	TS_IDLE,		/* default current state */
};

/*
 * tu_getinfo
 * the user can ask for information in any tli state
 * if sane, send T_info_ack upstream
 */
void
tu_getinfo(bp, q)

	register mblk_t *bp;
	register queue_t *q;
{
	register struct T_info_ack *p;

	ASSERT(bp && q);

	if (!REUSEABLE(bp, sizeof(struct T_info_ack))) {
		freemsg(bp);
		if ((bp = allocb(sizeof(struct T_info_ack), BPRI_MED)) == NULL)
			return;
		MTYPE(bp) = M_PCPROTO;
	} else bp->b_wptr = bp->b_rptr;

	p = (struct T_info_ack *)bp->b_wptr;

	*p = TCP_info_ack;
	p->CURRENT_state = ((struct tcpd *)q->q_ptr)->td_tstate;
	bp->b_wptr += sizeof(struct T_info_ack);
	qreply(q, bp);
	return;

}	/*  End of tu_getinfo()  */

/*
 * tu_unbind
 * unbind the address, free tcpcb
 * if sane, return TE_OK_ACK1; else UNREACH
 */
unchar
tu_unbind(bp, q)

	register mblk_t *bp;
	register queue_t *q;
{
	register struct tcpd *tdp;
	register struct tcpcb *tp;

	ASSERT(bp && q);
	tdp = (struct tcpd *)q->q_ptr;
	ASSERT(tdp && tdp->td_tcpcb);

	tp = (struct tcpcb *)tdp->td_tcpcb->b_rptr;
	STRLOG(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_TRACE,
		"tu_unbind: lport = %x", ntohs(tp->t_lport));

	tu_cleancb(tdp);
	tdp->td_flags = 0;	/* User can't rebind to priv port */
	/* send M_FLUSH upstream */
	if (upflush(q) == -1)
		goto error;

	tdp->td_tstate = TS_UNBND;
	if (up_ok_ack(q, bp, T_UNBIND_REQ) == 0)
	    return(TE_OK_ACK1);
error:
	/*
	 * fatal error: caller will freemsg(bp) and do "strlog"
	 */
	return(UNREACH);

}	/*  End of tu_unbind()  */

/*
 * tu_bind
 * attach a tcp control block to the tcpd
 * if sane, return TE_BIND_ACK; else TE_ERROR_ACK
 * (for non-fatal) or UNREACH (fatal)
 */
unchar
tu_bind(bp, q)

	mblk_t *bp;
	queue_t *q;
{
	register struct tcpd *tdp;
	register struct T_bind_req *Tp;
	register struct T_bind_ack *Tp1;
	register mblk_t *bp1 = NULL;
	register struct tcpcb *tp;
	long connumber;
	long retcode, tcp_bind();
	struct tsap ta, *ap1;
	extern int tcp_maxconind;

	ASSERT(bp && q);
	tdp = (struct tcpd *)q->q_ptr;
	ASSERT(tdp && tdp->td_tcpcb == NULL);

	Tp = (struct T_bind_req *)bp->b_rptr;
	/*
	 * check the sanity of the request message
	 * assume primitive and ADDR in one block
	 */
	if (BLEN(bp) < (Tp->ADDR_offset + Tp->ADDR_length)) {
		retcode = TBADADDR;	/* used to be TBADREQ */
		goto nonfatal;
	}

	connumber = min(tcp_maxconind, Tp->CONIND_number);

	/*
	 * which address to be bound?
	 */
	switch(Tp->ADDR_length) {

	case TCP_ADDRLEN:
	case BINDLEN:
		/*
		 * may have to check user id if binding well-known address
		 */
		/* ADDR_offset may not be aligned */
		bcopy((char *)(bp->b_rptr+Tp->ADDR_offset),
			 (char *)&ta, BINDLEN);
		break;

	case 0:
		ta.ta_addr = INADDR_ANY;
		ta.ta_port = 0;
		break;

	default:
		retcode = TBADADDR;
		goto nonfatal;
	}

	/*
	 * going to bind 'pt' to this stream
	 * attach a tcpcb to the channel if none is there yet
	 * initialize the protocol parameters, e.g. seq# etc.
	 */

	if ((bp1 = tv_popen(tdp)) == NULL)
		goto fatal;		/* can't alloc tcpcb */
	tdp->td_tcpcb = bp1;

	if (!REUSEABLE(bp, TCP_BINDACKLEN)) {
		freemsg(bp);
		if ((bp = allocb(TCP_BINDACKLEN, BPRI_MED)) == NULL)
			goto fatal;
	} else bp->b_wptr = bp->b_rptr;

	if ((retcode = tcp_bind(&ta, tdp)) != 0)
		goto nonfatal;

	tp = (struct tcpcb *)bp1->b_rptr;
	/*
	 * now commit to success return
	 * return T_bind_ack
	 */
	Tp1 = (struct T_bind_ack *)bp->b_wptr;
	Tp1->PRIM_type = T_BIND_ACK;
	Tp1->ADDR_length = BINDLEN;
	Tp1->ADDR_offset = sizeof(struct T_bind_ack);
	Tp1->CONIND_number = connumber;
	ap1 = (struct tsap *)(bp->b_rptr + Tp1->ADDR_offset);
	ap1->ta_port = tp->t_lport = ta.ta_port;
	ap1->ta_addr = tp->t_laddr = ta.ta_addr;
	tp->t_maxcon = connumber;
	c_enq(bp1, &tcpcb);
	MTYPE(bp) = M_PCPROTO;
	bp->b_wptr += sizeof(struct T_bind_ack) + BINDLEN;
	if (Tp->ADDR_length == TCP_ADDRLEN) {
		Tp1->ADDR_length = TCP_ADDRLEN;
		bzero((char *)bp->b_wptr, 8);
		bp->b_wptr += 8;
	}
	STRLOG(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_TRACE,
		"tu_bind: addr %x.%d", ntohl(tp->t_laddr), ntohs(tp->t_lport));
	tdp->td_tstate = TS_IDLE;
	qreply(q, bp);
	return(TE_BIND_ACK);

nonfatal:
	if (up_error_ack(q, bp, T_BIND_REQ, retcode))
		goto fatal;
	if (tdp->td_tcpcb) {
		freeb(tdp->td_tcpcb);
		tdp->td_tcpcb = NULL;
	}
	STRlog(TCP_ID, TCPCHAN(tdp), DPRI_MED, SL_te,
		"tu_bind: nonfatal %x\n", retcode);
	return(TE_ERROR_ACK);

fatal:
	return(UNREACH);

}	/*  End of tu_bind()  */

/*
 * tu_optmgmt
 * what are options??? TO BE CONTINUED
 * send T_ERROR_ACK up and return TE_ERROR_ACK
 * if fatal, return UNREACH
 */
unchar
tu_optmgmt(bp, q)

	mblk_t *bp;
	queue_t *q;
{
	register struct tcpd *tdp;
	ASSERT(bp && q);

	tdp = (struct tcpd *)q->q_ptr;
	STRLOG(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_TRACE, 
		"tu_optmgmt: we are here\n");
	if (up_error_ack(q, bp, T_OPTMGMT_REQ, TACCES))
		return(UNREACH);
	else
		return(TE_ERROR_ACK);

}	/*  End of tu_optmgmt */

/*
 * tu_disconreq
 * bp is the T_discon_req (M_PROTO) followed (via b_cont) by M_DATAs
 * SEQ_number identifies which conn_req is rejected; -1 indicates
 * the outgoing connection to be dropped
 * currently no user data is needed/supported
 * if sane, send T_OK_ACK up and return TE_OK_ACK*
 * else send T_ERROR_ACK up and return TE_ERROR_ACK
 */
unchar
tu_disconreq(bp, q)

	mblk_t *bp;
	queue_t *q;
{
	register struct tcpd *tdp;
	register struct tcpcb *tp;
	register struct tcpcb *tp1;
	register mblk_t *bp1, *conbp;
	int flush = 0;
	int to_idle = 1;
	long seq;

	ASSERT(bp && q);
	tdp = (struct tcpd *)q->q_ptr;
	ASSERT(tdp);

	tp = (struct tcpcb *)tdp->td_tcpcb->b_rptr;
	ASSERT(tp);
	seq = ((struct T_discon_req *)bp->b_rptr)->SEQ_number;
	STRLOG(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_TRACE,
		"tu_disconreq: minor device = %d \n", TCPCHAN(tdp));
	switch(tdp->td_tstate) {

	case TS_DATA_XFER:
	case TS_WREQ_ORDREL:
	case TS_WIND_ORDREL:
		flush = 1;
		/* fall thru */

	case TS_WCON_CREQ:
		/* disconnecting existing connection */
		if (seq != -1)
			goto nonfatal;		/* can't match seq# */
		/*
		 * Should I Clean TCPCB if tv_close
		 * Returns error.
		 * Temporary fix until tirdwr module close is 
		 * not fixed.
		 */
		(void) tv_close(tp); /* This should be tv_abort */
		break;

	case TS_WRES_CIND:
		/* rejecting a pending request from remote */
		conbp = (mblk_t *)&tp->t_conq;
		for (bp1 = conbp->b_next; bp1 != conbp;) {
			register mblk_t *bp2 = bp1;

			tp1 = (struct tcpcb *)bp1->b_rptr;
			bp1 = bp1->b_next;
			if (tp1->t_conid == seq) {
				(void) tv_abort(tp1);
				(void) tw_clean (tp1);
				c_deq(bp2);
				freeb(bp2);
				if (--tp->t_cqlen > 0)
					to_idle = 0;
				goto ok;
			}
		}
		goto nonfatal;

	default:	/* out of sync */
		goto fatal;
	}

ok:
	if (to_idle)
	    tdp->td_tstate = TS_IDLE;
	if (up_ok_ack(q, bp, T_DISCON_REQ) == 0) {
		/* send M_FLUSH upstream */
		if (flush && upflush(q) == -1)
			goto fatal;
		STRLOG(TCP_ID, TCPCHAN(tdp), DPRI_MED, SL_TRACE,
			"tu_disconreq: seq =  %x\n", seq);
		return(TE_OK_ACK1);
	}

fatal:
	STRlog(TCP_ID, TCPCHAN(tdp), DPRI_MED, SL_te, "tu_disconreq: fatal");
	return(UNREACH);

nonfatal:
	/* doesn't change to idle, stay in the same state */
	if (up_error_ack(q, bp, T_DISCON_REQ, TBADSEQ))
		goto fatal;
	STRlog(TCP_ID, TCPCHAN(tdp), DPRI_MED, SL_te,
		"tu_disconreq: nonfatal %x\n", TBADSEQ);
	return(TE_ERROR_ACK);

}	/*  End of tu_disconreq()  */

/*
 * tu_connreq
 * bp is the T_conn_req (M_PROTO) followed (via b_cont) by M_DATAs
 * currently no user data is needed/supported
 *
 * if sane, send T_OK_ACK up and return TE_OK_ACK1
 * else send T_ERROR_ACK up and return TE_ERROR_ACK
 */
unchar
tu_connreq(bp, q)
     mblk_t *bp;
     queue_t *q;
{
    register struct tcpd *tdp;
    register struct T_conn_req *Tp;
    register struct tcpcb *tp;
    struct tsap ta;
    register long retcode;
    extern void tcp_getmyaddr();

    ASSERT(bp && q);
    tdp = (struct tcpd *)q->q_ptr;

    if (tdp->td_tcpcb == NULL){
	/*
	 * Should bind the user to any
	 * Odd address.
	 */
	retcode = TBADADDR;
	goto nonfatal;
	}

    tp = (struct tcpcb *)tdp->td_tcpcb->b_rptr;
    Tp = (struct T_conn_req *)bp->b_rptr;
    /*
     * check the sanity of the request message
     * assume primitive and opt/dest all in one block
     * ignore the option here (OPT_length and OPT_offset)
     */
    /* ASSERT(bp->b_cont == NULL); XXX: send/rcv bug */
    if ((BLEN(bp) < (Tp->DEST_offset + Tp->DEST_length))
	|| (Tp->DEST_offset < sizeof(*Tp))
	|| (Tp->DEST_length < BINDLEN)) {
	retcode = TBADADDR;
	goto nonfatal;
	}

    bcopy((char *)(bp->b_rptr+Tp->DEST_offset), (char *)&ta, BINDLEN);
    STRLOG(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_TRACE,
	   "tu_conreq: tv_aopen on %x.%d", ntohl(ta.ta_addr), ntohs(ta.ta_port));

    /*
     * If the foreign address is INADDR_ANY or INADDR_BROADCAST
     *	set the foreign address to the primary local 
     * 	source address or the primary broadcast address.
     *	Send a M_CTL msg to ip to get the appropriate
     * 	address.
     */
    tp->t_faddr = ta.ta_addr;
    if (ta.ta_addr == INADDR_ANY ||
	ta.ta_addr == INADDR_BROADCAST) 
	tcp_gethisaddr(tp);

    /*
     * Get the source address here from IP.
     */
    tp->t_laddr = 0;					/*nw2*/
    tcp_getmyaddr(tp);
    if (tp->t_laddr == 0){
	/*
	 * Network Unreachable
	 */
	STRlog(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_te,
	       "tu_connreq: Could not get my own address(%x)",ntohl(tp->t_faddr));
	tp->t_faddr = 0;
	retcode = TBADADDR;
	goto nonfatal;
	}

    if (retcode = tv_aopen(tp, &ta))
	goto nonfatal;

    tdp->td_tstate = TS_WCON_CREQ;
    if (up_ok_ack(q, bp, T_CONN_REQ) == 0)
	return(TE_OK_ACK1);

  fatal:
    STRlog(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_te,
	   "tu_con: fatal\n");
    return(UNREACH);

  nonfatal:
    if (up_error_ack(q, bp, T_CONN_REQ, retcode))
	goto fatal;
    STRlog(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_te,
	   "tu_connreq: nonfatal %x\n", retcode);
    return TE_ERROR_ACK;

}	/*  End of tu_connreq()  */

/*
 * tu_connres
 * called when T_conn_res is received from user to accept a conn
 * indication tdp is the listening stream, rtdp is the response 
 * stream where the connection shall be established,
 * bp points to T_conn_res, currently no user data associated with it
 * T_conn_res.OPT_length (!= 0) is not supported
 *
 * if sane, send T_OK_ACK up and return TE_OK_ACK1
 * else send T_ERROR_ACK up and return TE_ERROR_ACK
 */
unchar
tu_connres(bp, q)
     mblk_t *bp;
     queue_t *q;
{
    struct T_conn_res *Tp;
    queue_t *rq;
    register struct tcpd *tdp;
    register struct tcpd *rtdp;
    register struct tcpcb *tp;
    struct tcpcb *rtp;
    register struct tcpcb *tp1;
    register mblk_t *bp1, *conbp, *bpo;
    long seq, code;
    extern void tw_drop_floater();

    ASSERT(bp && q);
    tdp = (struct tcpd *)q->q_ptr;
    ASSERT(tdp && tdp->td_tcpcb);
    tp = (struct tcpcb *)tdp->td_tcpcb->b_rptr;
    STRlog (TCP_ID, TCPCHAN (rtdp), DPRI_MED, SL_te, "tu_connres: entering");

    Tp = (struct T_conn_res *)bp->b_rptr;
    if (Tp->OPT_length) {
	code = TBADOPT;
	goto nonfatal;
	}
    if ((rq = (queue_t *)(Tp->QUEUE_ptr)) == NULL) {
badf:
	code = TBADF;
	goto nonfatal;
	}
    /* rq should have been opened and bound before */
    rtdp = (struct tcpd *)rq->q_ptr;
	if (rtdp == NULL || rtdp->td_tcpcb == NULL)
	    goto badf;
    
    rtp = (struct tcpcb *)rtdp->td_tcpcb->b_rptr;
    seq = Tp->SEQ_number;
    conbp = (mblk_t *)&tp->t_conq;

    if (rtdp == tdp) {
	mblk_t *to_free;
	/** accepting on the listening chan **/
	
	if (tp->t_cqlen != 1)
	    goto badf;
	bp1 = conbp->b_next;
	tp1 = (struct tcpcb *)bp1->b_rptr;
	if (tp1->t_conid != seq) {
	    code = TBADSEQ;
	    goto nonfatal;
	    }
	/* discard listening tcpcb, replace with floating one */
	tp1->t_conq.b_next = tp1->t_conq.b_prev =
	    (struct qhead *)&tp1->t_conq;
	to_free = tdp->td_tcpcb;
	if (to_free == tcp_pcbcache)
	    tcp_pcbcache = (mblk_t *)NULL;
	c_deq(to_free);
	c_enq(bp1, &tcpcb);
	(void) tw_clean (mtod (to_free, struct tcpcb *));
	(void) freeb (to_free);
	rtdp->td_tcpcb = bp1;
	goto ok;
	}

    
    /*
     * check if this seq # in the t_conq
     */
    for (bp1 = conbp->b_next; bp1 != conbp; bp1 = bp1->b_next) {
	tp1 = (struct tcpcb *)bp1->b_rptr;
	if (tp1->t_conid == seq) {
	    /*
	     * match: the original bound address is meaningless
	     * we are overwriting it here
	     * replace the accepting tcpcb with the floating one
	     */
	    tp1->t_maxcon = rtp->t_maxcon;
	    tp1->t_tdp = rtdp;
	    tu_cleancb(rtdp);	/* delete the old one */
	    rtdp->td_tcpcb = bp1;	/* replace with floating one */
	    c_deq(bp1);
	    c_enq(bp1, &tcpcb);
	    goto ok;
	    }
	}
    code = TBADSEQ;
    goto nonfatal;

ok:	
    if ((rtp = mtod (bp1, struct tcpcb *))->t_state == TV_SYN_RCVD) {
	/*
	 * We will now respond with a SYN ACK for the
	 * SYN that we received. But don't enable the write queue.
	 * This way when we get an ACK for our SYN we can
	 * do it. This will make the use think that he is set up.
	 * Also fire up the timers (RETRANSMIT). The keep alive
	 * timer will be started when we get an ACK for this SYN.
	 */
	/* new guy on the block, clear tcp_pcbcache  MM */
	tcp_pcbcache = (mblk_t *)NULL;
        STRlog (TCP_ID, TCPCHAN (rtdp), DPRI_MED, SL_te, 
	   "tu_connres: sending back the SYN/ACK");
	tw_send (rtp, NULL, NULL, rtp->snd_nxt, 
		 rtp->rcv_nxt, TH_ACK|TH_SYN);
	rtp->t_timer[TCPT_REXMT] = tp->t_rxtcur;
	noenable(WR(rq));
    } else {
	LABEL (tu_cres_bad_state);		
	STRlog (TCP_ID, TCPCHAN (rtdp), DPRI_MED, SL_te,
		"tu_connres: not in SYN_RCVD state (%x)\n",rtp->t_state);
	/*
	 * Drop this floating TCPCB.
	 */
	if (rtp != tp)
	    tw_drop_floater (bp1);
	goto nonfatal;
	}

    rtdp->td_tstate = TS_DATA_XFER;		/* pass_conn goes here */
    code = TE_OK_ACK4;
    if (tdp == rtdp) {
	rtp->t_cqlen = 0;
	code = TE_OK_ACK2;
	tdp->td_tstate = TS_DATA_XFER;
    } else {
	if ((bpo = allocb(sizeof(struct stroptions), BPRI_MED))) {
	    struct stroptions *optp = (struct stroptions *)bpo->b_wptr;
	    bpo->b_wptr += sizeof(struct stroptions);
	    bpo->b_datap->db_type = M_SETOPTS;
	    optp->so_flags = SO_LOWAT;
	    optp->so_lowat = STRLOW;
	    putnext(rtdp->td_rdq, bpo);
	    }
	if (--tp->t_cqlen == 0) {
	    code = TE_OK_ACK3;
	    tdp->td_tstate = TS_IDLE;
	    if ((bpo = allocb(sizeof(struct stroptions), BPRI_MED))) {
		struct stroptions *optp = (struct stroptions *)bpo->b_wptr;
		bpo->b_wptr += sizeof(struct stroptions);
		bpo->b_datap->db_type = M_SETOPTS;
		optp->so_flags = SO_LOWAT;
		optp->so_lowat = 0;
		qreply(q, bpo);
		}
	    }
	}
    if (up_ok_ack(q, bp, T_CONN_RES))
	goto fatal;
    /* else OK_ACK4, stay in same state WRES_CIND */
    STRLOG(TCP_ID, TCPCHAN(tdp), DPRI_MED, SL_TRACE,
	   "tu_connres: seq %x, q %x, state %x\n", seq, q, tdp->td_tstate);
    return(code);

nonfatal:
    if (up_error_ack(q, bp, T_CONN_RES, code))
	goto fatal;
    STRlog(TCP_ID, TCPCHAN(tdp), DPRI_MED, SL_te,
	   "tu_connres: nonfatal %x\n", code);
    return(TE_ERROR_ACK);

fatal:
    return(UNREACH);

}	/*  End of tu_connres()  */

/*
 * tu_ordrel
 * go to next state and tell the protocol to engage closing
 */
unchar
tu_ordrel(bp, q)

	register mblk_t *bp;
	register queue_t *q;
{
	register struct tcpcb *tp;
	register struct tcpd *tdp;

	ASSERT(bp && q);

	tdp = (struct tcpd *)q->q_ptr;
	tp = (struct tcpcb *)tdp->td_tcpcb->b_rptr;

	if (bp)
		freemsg(bp);

	STRLOG(TCP_ID, TCPCHAN(tdp), DPRI_MED, SL_TRACE,
		"tu_ordrel: incoming state = %x\n", tdp->td_tstate);

	if (tv_close(tp) == -1)
		return(UNREACH);

	switch (tdp->td_tstate) {

	case TS_DATA_XFER:
		tdp->td_tstate = TS_WIND_ORDREL;
		break;

	case TS_WREQ_ORDREL:
		tdp->td_tstate = TS_IDLE;
		break;

	default:
		return(UNREACH);
	}

	STRLOG(TCP_ID, TCPCHAN(tdp), DPRI_MED, SL_TRACE,
		"tu_ordrel: outgoing state = %x\n", tdp->td_tstate);

	return TE_OK_ACK1;

}	/*  End of tu_ordrel()  */

/*
 * tu_data
 * call putq which (tcposrv) calls tcp_output
 * throw away M_PROTO (with T_DATA_REQ and MORE flag) since tcp
 * can't support MORE (tcp is a byte stream module)
 * stay in the same state
 * can be deleted --- currently no one calls this routine
 */
unchar
tu_data(bp, q)

	register mblk_t *bp;
	register queue_t *q;
{
	STRlog(TCP_ID, -1, DPRI_MED, SL_te, "tu_data\n");
	freemsg(bp);
	return(UNREACH);

}	/* End of tu_data()  */

/*
 * User Wants to send Expedited Data.
 * So Let us fire it up.
 */
unchar
tu_exdata(bp, q)

	register mblk_t *bp;
	register queue_t *q;
{
	mblk_t *bp1;
	register struct tcpd *tdp;
	register struct tcpcb *tp;

	tdp = (struct tcpd *)q->q_ptr;
	tp = (struct tcpcb *)tdp->td_tcpcb->b_rptr;

	/**
	 ** set the urgent pointer to 
	 ** 	unacknowledged + window size + 1
	 ** the expedited data is sent along with normal
	 ** data in the byte stream
	 **/

	 
	tp->snd_up = tp->snd_una + t_wcc(tdp) + 1;
	tdp->td_wcount += msgdsize(bp);
	
	/**
	 ** call tcp_output to send expedited data along with 
	 ** normal data through putq
	 **/
	bp1 = bp->b_cont;
	bp->b_cont = NULL;
	freeb(bp);
	bp =bp1;
/*	tp->t_flags |= TF_FORCE;*/
	STRlog(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_te, "tu_exdata:snd_up = %x\n",tp->snd_up);
	STRlog(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_te, "tu_exdata:snd_nxt snd_una = %x %x\n",tp->snd_nxt,tp->snd_una);

	putq(q,bp);
	qenable(q);
}	/*  End of tu_exdata()  */

/*
 * Find the next available port if one
 * is not provided. If one is provided
 * Then don't check the source address.
 */

static ushort tcp_bindaddr = TCPPORT_RESERVED;


static long
tcp_bind(ta, tdp)
     register struct tsap *ta;
     register struct tcpd *tdp;
{
	/*
	 * We have to look up the TCPCB structures
	 * and find if this port is already in use.
	 * But the port numbers may never recycle
	 */
	register ushort port, nxt;
	register struct tcpcb *tp;
	register mblk_t *bp;

	port = ntohs(ta->ta_port);

	/*
	 * Don't allow a mere mortal to bind to a
	 * privileged port.
	 */
	if (port && (port < TCPPORT_RESERVED) && ((tdp->td_flags & TD_SUPERUSER) == 0))
		if (port != 20)
			return TACCES;

	if (++tcp_bindaddr == 0xffff)
		tcp_bindaddr = TCPPORT_RESERVED;

	nxt = htons(tcp_bindaddr);
	port = htons(port);
	for (bp = ((mblk_t *)&tcpcb)->b_next; bp != (mblk_t *)&tcpcb;
		bp = bp->b_next)
	{
		tp = (struct tcpcb *)bp->b_rptr;
		/*
		 * If there is already an active port, but it
		 * is connected to some one. (Not inaddr_any) then
		 * it is OK for a person to bind to this addr.
		 * However if he has requested a new port, then
		 * if there is already one such local port, then
		 * it is not OK.
		 */
		if (port && tp->t_lport == port) {
			if (tdp->td_flags & TD_NEWPORT)
				return TNOADDR;
		        else if ((tp->t_laddr == ta->ta_addr) &&
				    (!(tdp->td_sockflags & SO_REUSEADDR)))
				return TNOADDR;
		}

		if (nxt == tp->t_lport) {
		    if (++tcp_bindaddr == 0xffff)
			tcp_bindaddr = TCPPORT_RESERVED;
		    nxt = htons(tcp_bindaddr);
		    continue;
		}
	}

	if (port)
		ta->ta_port = port;
	else
		ta->ta_port = nxt;

	return 0;
		
}	/*  End of tcp_bind()  */
