/*
 * @(#) Copyright 1986.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)tcp_timer.c (TWG)  1.4     89/08/09 "

#include "sys/param.h"
#include "sys/types.h"
#ifndef XENIX
#include "sys/inline.h"
#endif
#include "sys/stream.h"
#include "sys/stropts.h"
#include "sys/tihdr.h"
#include "sys/strlog.h"
#include "sys/debug.h"
#ifdef XENIX
#include "sys/assert.h"
#endif
#include "sys/inet.h"
#include "sys/socket.h"
#include "sys/if.h"
#include "sys/in.h"
#include "sys/in_var.h"
#include "sys/ip.h"
#include "sys/tcp.h"
#include "sys/errno.h"


/***********************
 *  Table of Contents  *
 ***********************/

void		tcptimeout();
void		tcpuntimeout();
void		tcp_canceltimers();
void		tcp_timers();
void		tw_setpersist();

/***** End of TOC *****/

int	tcp_keepidle = TCPTV_KEEP_IDLE;
int	tcp_keepintvl = TCPTV_KEEPINTVL;
int	tcp_maxidle;

int tcp_need_timeout, tcp_run_serv;
static int tcptimeID = 0;
extern void tw_clean();
/*
 * tcp protocol timeout routine called every 200 Ms
 * update the timers in all active tcpcb
 */
void
tcptimeout()
{
    /*
     * KERNEL BUG>>>
     * THE LOCAL VARIABLES OVERFLOW THE INTERRUPT STACK
     * AND RESET THE TIME VARIABLE.
     * They need to make a larger interrupt stack.
     ** If you are porting to a Multi-processor machine
     * Please beware...
     */
#if u3b2
    static struct tcpcb *tp;
    static mblk_t *bp;
#else
    register struct tcpcb *tp;
    register mblk_t *bp;
#endif
    int s;
    static short ntime = 1;
    void tcp_slowtimo();

    s = spltime();
    /*
     * If we are processing a service routine, then don't run.
     */
    if (tcp_run_serv) {
	tcp_need_timeout = 1;
	splx (s);
	return;
    } else
	tcp_need_timeout = 0;
    /*
     * What happens when tcp_timers frees up stuff out of
     * reassembly queue. ???
     */
    for (bp=((mblk_t *)&tcpcb)->b_next; bp != (mblk_t *)&tcpcb;){
	tp = (struct tcpcb *)bp->b_rptr;
	bp = bp->b_next;
	/*
	 * First the 250 MS timer for Delayed Ack's
	 */
	if (tp->t_flags & TF_DELACK){
	    tp->t_flags &= ~TF_DELACK;
	    tp->t_flags |= TF_ACKNOW;
	    tcpstat.tcps_delack++;
	    (void)tcp_output(tp);
	    }
	/*
	 * Every 500 ms call the Retransmission timers
	 */
	if (ntime % (PR_FASTHZ/PR_SLOWHZ) == 0)
	    tcp_slowtimo(tp);
	}
    ntime++;
    splx(s);
    tcptimeID = timeout(tcptimeout, (caddr_t)0, HZ/PR_FASTHZ);
    return;

} /*  End of tcptimeout()  */


void
tcp_slowtimo(tp)
     register struct tcpcb * tp;
{
#if u3b2
    static mblk_t *head, *bp1, *bp2;
#else
    register mblk_t *head, *bp1, *bp2;
#endif
    register int i;

    tcp_maxidle = TCPTV_KEEPCNT * tcp_keepintvl;

    /*
     * If this is the listening channel, and the queue-length
     * is non-zero then start killing the unaccepted connections.
     */
    if (tp->t_cqlen > 0) {
	register struct tcpcb *tp2;
	
	head = (mblk_t *)&tp->t_conq;
	bp1 = head->b_next;
	while (bp1 != head) {
	    bp2 = bp1;
	    bp1 = bp1->b_next;
	    tp2 = mtod (bp2, struct tcpcb *);
	    if (--tp2->t_timer[TCPT_KEEP] > 0)
		continue;
	    (void) tv_abort (tp2);
	    (void) tw_clean (tp2);
	    if (--tp->t_cqlen == 0)
		tp->t_tdp->td_tstate = TS_IDLE;
	    c_deq(bp2);
	    freeb(bp2);
	    }
	} else for (i = 0; i < TCPT_NTIMERS; i++) {
	    if (tp->t_timer[i] && --tp->t_timer[i] == 0) {
		tcp_timers(tp, i);
		}
	    }

    tp->t_idle++;
    if (tp->t_rtt)
	tp->t_rtt++;
    
    tcp_iss += TCP_ISSINCR/PR_SLOWHZ;

} /* end tcp_slowtimo */

/*
 * Cancel TCP timer in kernel
 */

void
tcpuntimeout()
{
    if (tcptimeID) {
	(void) untimeout(tcptimeID);
	tcptimeID = 0;
	}
    return;

}	/*  End of tcputimeout()  */


/*
 * Cancel all timers for TCP tp.
 */
void
tcp_canceltimers(tp)
     register struct tcpcb *tp;
{
    register int i;

    for (i = 0; i < TCPT_NTIMERS; i++)
	tp->t_timer[i] = 0;
    return;

}	/*  End of tcp_canceltimers()  */

int	tcp_backoff[TCP_MAXRXTSHIFT + 1] =
    { 1, 2, 4, 8, 16, 32, 64, 64, 64, 64, 64, 64, 64 };

/*
 * TCP timer processing.
 */
void
tcp_timers(tp, timer)
     register struct tcpcb *tp;
     int timer;
{
    extern void tn_usrmsg();
    extern void tw_clean();
    extern void tn_disconnected();
    extern void tw_rmtcb();
    register int rexmt;
    
    switch(timer) {
	
      case TCPT_REXMT:
	/*
	 * retransmission timer went off.
	 * message has not been acked within retransmit interval.
	 * Reset snd_nxt to all unacked data so that
	 * tcp_output will retransmit.
	 */
	STRlog( TCP_ID, -1, DPRI_LO, SL_te,
	       "tcp_timer: RXT Number: %d tp=%x una=%d",tp->t_rxtshift,
		tp, tp->snd_una);  /* JTH */
	if (++tp->t_rxtshift > TCP_MAXRXTSHIFT){
	    tp->t_rxtshift = TCP_MAXRXTSHIFT;
	    tcpstat.tcps_timeoutdrop++;
	    tn_usrmsg(tp, ETIMEDOUT);
	    goto reset;
	    }
	tcpstat.tcps_rexmttimeo++;
	rexmt = ((tp->t_srtt >> 2) + tp->t_rttvar) >> 1;
	rexmt *= tcp_backoff[tp->t_rxtshift];
	TCPT_RANGESET(tp->t_rxtcur, rexmt, TCPTV_MIN, TCPTV_REXMTMAX);
	tp->t_timer[TCPT_REXMT] = tp->t_rxtcur;
	/*
	 * If losing, let the lower level know and try for
	 * a better route.  Also, if we backed off this far,
	 * our srtt estimate is probably bogus.  Clobber it
	 * so we'll take the next rtt measurement as our srtt;
	 * move the current srtt into rttvar to keep the current
	 * retransmit times until then.
	 */
	if (tp->t_rxtshift > TCP_MAXRXTSHIFT / 4) {
	    tp->t_rttvar += (tp->t_srtt >> 2);
	    tp->t_srtt = 0;
	    }
	tp->snd_nxt = tp->snd_una;
	/*
	 * If timing a segment in this window, stop the timer.
	 */
	tp->t_rtt = 0;
	/*
	 * Close the congestion window down to one segment
	 * (we'll open it by one segment for each ack we get).
	 * Since we probably have a window's worth of unacked
	 * data accumulated, this "slow start" keeps us from
	 * dumping all that data as back-to-back packets (which
	 * might overwhelm an intermediate gateway).
	 *
	 * There are two phases to the opening: Initially we
	 * open by one mss on each ack.  This makes the window
	 * size increase exponentially with time.  If the
	 * window is larger than the path can handle, this
	 * exponential growth results in dropped packet(s)
	 * almost immediately.  To get more time between 
	 * drops but still "push" the network to take advantage
	 * of improving conditions, we switch from exponential
	 * to linear window opening at some threshhold size.
	 * For a threshhold, we use half the current window
	 * size, truncated to a multiple of the mss.
	 *
	 * (the minimum cwnd that will give us exponential
	 * growth is 2 mss.  We don't allow the threshhold
	 * to go below this.)
	 */
	{
	u_int win = min(tp->snd_wnd, tp->snd_cwnd) / 2 / tp->t_maxseg;
	if (win < 2)
	    win = 2;
	tp->snd_cwnd = tp->t_maxseg;
	tp->snd_ssthresh = win * tp->t_maxseg;
	}
	(void) tcp_output(tp);
	break;
	
      case TCPT_PERSIST:
	/*
	 * Persistance timer into zero window.
	 * Force a byte to be output, if possible.
	 */
	tcpstat.tcps_persisttimeo++;
	tw_setpersist(tp);
	tp->t_force = 1;
	(void) tcp_output(tp);
	tp->t_force = 0;
	break;
	
      case TCPT_KEEP:
	/*
	 * Keep-alive timer went off; send something
	 * or drop connection if idle for too long.
	 */
	
	tcpstat.tcps_keeptimeo++;
	if (tp->t_state < TV_ESTABLISHED) {
	    tcpstat.tcps_keepdrops++;
	STRlog( TCP_ID, -1, DPRI_LO, SL_te,
	       "tcp_timer: keepalive : t_state=%d",tp->t_state); /* JTH */
	    goto reset;
	    }
	if ((tp->t_tdp->td_sockflags & SO_KEEPALIVE)
	    && (tp->t_state <= TV_CLOSE_WAIT)) {
	    if (tp->t_idle >= tcp_keepidle + tcp_maxidle) {
		tcpstat.tcps_keepdrops++;
	STRlog( TCP_ID, -1, DPRI_LO, SL_te,
	       "tcp_timer: keepalive : t_idle=%d, t_state=%d",tp->t_idle,
		tp->t_state); /*JTH*/
		goto reset;
		}
	    /*
	     * Saying tp->rcv_nxt-1 lies about what
	     * we have received, and by the protocol spec
	     * requires the correspondent TCP to respond.
	     * Saying tp->snd_una-1 causes the transmitted
	     * byte to lie outside the receive window; this
	     * is important because we don't necessarily
	     * have a byte in the window to send (consider
	     * a one-way stream!)
	     */
	    tcpstat.tcps_keepprobe++;
	    (void) tw_send(tp, NULL, NULL, tp->snd_una-1, tp->rcv_nxt-1, 0);
	    tp->t_timer[TCPT_KEEP] = tcp_keepintvl;
	} else
	    tp->t_timer[TCPT_KEEP] = tcp_keepidle;
	break;
	
      case TCPT_2MSL:
	/*
	 * 2 MSL timeout in shutdown went off.  If we're closed but
	 * still waiting for peer to close and connection has been idle
	 * too long, or if 2MSL time is up from TIME_WAIT, delete connection
	 * control block.  Otherwise, check again in a bit.
	 */
#ifdef	notdef	/* WHY WAS THIS HERE ??? IT CAUSES EXTREME DELAY ON CLOSE */
	if (tp->t_state != TV_TIME_WAIT &&
	    tp->t_idle <= tcp_maxidle) {
	    tp->t_timer[TCPT_2MSL] = tcp_keepintvl;
	    break;
	    }
#endif
      reset:
	STRLOG(TCP_ID, TCPCHAN(tp->t_tdp), DPRI_LO, SL_TRACE,
	       "tcp_timers: disc tcpcb, tp=%x timer = %d\n",tp,timer);
	
	tp->t_state = TV_CLOSED;
	(void) tw_clean(tp);
	(void) tn_disconnected(tp);
	(void) tw_rmtcb(tp);
	break;
	}
    
    return;
    
    }	/*  End of tcp_timers()  */
