/*	vi: set ts=4 sw=4 : <- vi modeline
 * @(#) Copyright 1986.  The Wollongong Group, Inc.  All Rights Reserved.
 *
 * Fixes: ID	SPR#	Description
 *  	  nw2	4696	rlogin to multiple IP interfaces("Connection reset").
 *
 *	  TWU  		Change KEEPTCP timer to 75 secs if KEEPALIVE flag
 *			is on.
 *	  cah1  	Don't set t_maxseg in tcp_bestmss. It wipes out what
 *			was sent by the initiator.
 */

#ident "@(#)tcp_input.c (TWG)  1.7     89/08/30 "

#include "sys/param.h"
#include "sys/types.h"

#ifndef XENIX
#include "sys/inline.h"
#endif /* XENIX */
#include "sys/stream.h"
#include "sys/strlog.h"
#include "sys/tihdr.h"
#include "sys/debug.h"
#ifdef XENIX
#include "sys/assert.h"
#endif
#include "sys/inet.h"
#include "sys/socket.h"
#include "sys/if.h"
#include "sys/in.h"
#include "sys/in_var.h"
#include "sys/ip.h"
#include "sys/ip_var.h"
#include "sys/tcp.h"
#include "sys/errno.h"


#define T_ADDR(tp)	(ntohl((tp)->t_laddr)), (ntohs((tp)->t_lport))
#define T_FADDR(tp)	(ntohl((tp)->t_faddr)), (ntohs((tp)->t_fport))

int	tcprexmtthresh = 3;

struct tcpstat	tcpstat;
extern	int	tcp_maxrcvwin;


/***********************
 *  Table of Contents  *
 ***********************/

mblk_t		*tiu_lookup();
void		tcp_input();
void		tiu_resp();
void		tiu_dooptions();
void		tiu_ipoptions();
int	        tiu_reass();
void		tiu_dropit();
int	        tiu_adjmsg();
#ifdef NOTDEF
void		tiu_pulloutofband();
#endif
void 		tcp_bestmss();

/***** End of TOC *****/


/*
 * tiu_*: tcp_input internal utilities
 */


/*
 * find the tcpcb which matches the address and port
 * 'head' links (circular queues) all the concerned tcpcb
 * 'wildok' being 1 allows wildcard match
 * return the block containing the matched tp if one is found,
 * or NULL if failed
 */
mblk_t *
tiu_lookup(head, faddr, fport, laddr, lport, wildok)

	mblk_t *head;
	register ulong faddr, laddr;
	register ushort fport, lport;
	int wildok;
{
	register mblk_t *bp, *match = NULL;
	register struct tcpcb *tp;
	int matchwild = 3, wildcard;

	for (bp = head->b_next ; bp != head; bp = bp->b_next) {
		tp = (struct tcpcb *)bp->b_rptr;

		/*
		 * match local port: must be exact
		 */
		if (tp->t_lport != lport)
			continue;
		wildcard = 0;

		if (tp->t_laddr != INADDR_ANY) {
			if (laddr == INADDR_ANY)
				wildcard++;
			else if (tp->t_laddr != laddr)
				continue;
		} else {
			if (laddr != INADDR_ANY)
				wildcard++;
		}

		/*
		 * match foreign addrss, which can also be INADDR_ANY
		 */
		if (tp->t_faddr != INADDR_ANY) {
			if (faddr == INADDR_ANY)
				wildcard++;
			else if (tp->t_faddr != faddr || tp->t_fport != fport)
				continue;
		} else {
			if (faddr != INADDR_ANY)
				wildcard++;
		}

		/*
		 * see if wildcard is allowed
		 */
		if (wildcard && wildok != 1)
			continue;

		if (wildcard < matchwild) {
			match = bp;
			matchwild = wildcard;
			if (matchwild == 0)
				break;
		}
	}
	return(match);

}	/*  End of tiu_lookup()  */

/*
 * Insert segment ti into reassembly queue of tcp with
 * control block tp.  Return TH_FIN if reassembly now includes
 * a segment with FIN.  The macro form does the common case inline
 * (segment is the next to be received on an established connection,
 * and the queue is empty), avoiding linkage into and removal
 * from the queue and repetition of various conversions.
 * Set DELACK for segments received in order, but ack immediately
 * when segments are out of order (so fast retransmit can work).
 */
#define	TCP_REASS(tp, ti, m, flags) { \
    extern void tiu_data();\
	if ((ti)->ti_seq == (tp)->rcv_nxt && \
	    (tp)->t_rsq.b_next == (struct qhead *)&(tp)->t_rsq && \
	    (tp)->t_state == TV_ESTABLISHED) { \
		tp->t_flags |= TF_DELACK; \
		(tp)->rcv_nxt += (ti)->ti_len; \
		flags = (ti)->ti_flags & TH_FIN; \
		tcpstat.tcps_rcvpack++;\
		tcpstat.tcps_rcvbyte += (ti)->ti_len;\
		tiu_data((tp)->t_tdp, (m)); \
	} else { \
		(flags) = tiu_reass((tp), (m)); \
		tp->t_flags |= TF_ACKNOW; \
	} \
}

int tcppreddat,
    tcppredack;

int tcp_cache_enable;
mblk_t *tcp_pcbcache;
int tcp_pcbhit, tcp_pcbmiss;

/*
 * TCP input routine, follows pages 65-76 of the
 * protocol specification dated September, 1981 very closely.
 */
void
tcp_input(ibp)
     mblk_t *ibp;
{
    register struct tcpiphdr *ti;
    register mblk_t          *bp;
    register struct tcpcb    *tp = NULL;
    register struct tcpd     *tdp = NULL;
    register int             tiflags;
    mblk_t                   *optbp = NULL;			/* option block */
    int len, tlen, off, iss = 0, broadcast = 0;
    int todrop, acked, needoutput = 0, ourfinisacked, oursynisacked = 0;
    mblk_t                   *bp1;
    extern void tw_acked();
    extern void tn_orddisconn();
    extern void tw_drop_floater();
    extern void tw_close();
    extern void tn_connected();
    extern void tn_usrmsg();
    extern void tn_exdata();
    extern void tn_exdata1();
    void tiu_data();

    tcpstat.tcpInSegs++;

    /*
     * pull IP and TCP header together in first block
     */
    bp = (MTYPE(ibp) == M_PROTO) ? ibp->b_cont : ibp;
    ti = (struct tcpiphdr *)(bp->b_rptr);
	if (BLEN(bp) < TCPIPHLEN) {
		if ((pullupmsg(bp, TCPIPHLEN)) == 0) {
			STRlog(TCP_ID, 101, DPRI_LO, SL_te,
			   "tcp_input: too short(%d)\n",BLEN(bp));
			tcpstat.tcps_rcvshort++;
			goto drop;
		}
		ti = (struct tcpiphdr *)bp->b_rptr;
	}

    /*
     * checksum extended TCP header and data.
     */
    STRLOG(TCP_ID, 101, DPRI_LO, SL_TRACE,
	   "tcp_input: src = %x.%d", ntohl(ti->ti_src), ntohs(ti->ti_sport));
    STRLOG(TCP_ID, 101, DPRI_LO, SL_TRACE,
	   "tcp_input: dst = %x.%d", ntohl(ti->ti_dst), ntohs(ti->ti_dport));
    STRLOG(TCP_ID, 101, DPRI_LO, SL_TRACE,
	   "tcp_input: flags = %x, cksum = %x\n",ti->ti_flags,ti->ti_sum);

    tlen = ((struct ip *)ti)->ip_len;	/* tlen: tcp header + data */
    len = sizeof(struct ip) + tlen;
    /*
     * the spec says the len should be tlen + 12 (pseudo header) ???
     * but we use the whole ip header (20 bytes) here for checksum
     * Zero out the first 8 bytes.
     */
    broadcast = ti->ti_x1 & IPO_BROADCAST;
    ti->ti_next = ti->ti_prev = 0;
    ti->ti_x1 = 0;
    ti->ti_len = htons((ushort) tlen);
    if (ti->ti_sum = (ushort)in_cksum(bp, len)) {
		tcpstat.tcps_rcvbadsum++;
		STRlog(TCP_ID, 101, DPRI_LO, SL_te,
			   "tcp_input: bad checksum %x\n", ti->ti_sum);
		goto drop;
	}
    /*
     * Check that TCP offset makes sense,
     * pull out TCP options and adjust length.
     */
    off = ti->ti_off << 2;
    if (off < TCPHLEN || off > tlen){
		STRlog(TCP_ID, 101, DPRI_LO, SL_te,
			   "tcp_input: bad offset: off=%d, tlen=%d\n",off,tlen);
		tcpstat.tcps_rcvbadoff++;
		goto drop;
	}
    tlen -= off;
    ti->ti_len = tlen;
	if (off > TCPHLEN) {
		register int optlen;
		register char *cp;

		if (BLEN(bp) < (IPHLEN + off)) {
			if (pullupmsg(bp, IPHLEN + off) == 0) {
				tcpstat.tcps_rcvshort++;
				STRlog(TCP_ID, 101, DPRI_LO, SL_te, "tcp_input: Bad header\n");
				goto drop;
			}
		}
		ti = (struct tcpiphdr *)bp->b_rptr;
		optlen = off - TCPHLEN;
		if ((optbp = allocb(optlen, BPRI_LO)) == NULL) {
			STRlog(TCP_ID, 101, DPRI_LO, SL_te, "tcp_input: No buffers\n");
			goto drop;
		}
		cp = (char *)bp->b_rptr + TCPIPHLEN;	/* option addr */
		/*
		 * Copy the Options to a new mblk
		 */
		bcopy(cp, (char *)optbp->b_wptr, optlen);
		optbp->b_wptr += optlen;
		/*
		 * moving data back optlen position
		 * Compress the data so that
		 * we can unveil the tcpiphdr by moving backward.
		 */
		bcopy(cp+optlen, cp, BLEN(bp)-TCPIPHLEN-optlen);
		bp->b_wptr -= optlen;
		off = TCPHLEN;
	}
    tiflags = ti->ti_flags;

    /* drop tcp & ip header */
    bp->b_rptr += (off + IPHLEN);

    /*
     * convert TCP protocol specific fields to host format.
     */
    ti->ti_seq = ntohl(ti->ti_seq);
    ti->ti_ack = ntohl(ti->ti_ack);
    ti->ti_win = ntohs(ti->ti_win);
    ti->ti_urp = ntohs(ti->ti_urp);
    /*
     * locate tcpcb for segment (allow wildcard match)
     */
findpcb:

    if (tcp_cache_enable && tcp_pcbcache) {
		tp = (struct tcpcb *)tcp_pcbcache->b_rptr;
		if (tp->t_faddr != ti->ti_src
		|| tp->t_fport != ti->ti_sport
		|| tp->t_lport != ti->ti_dport
		|| tp->t_laddr != ti->ti_dst) {
			tcp_pcbcache = tiu_lookup(&tcpcb, ti->ti_src, ti->ti_sport,
					 ti->ti_dst, ti->ti_dport, INPLOOKUP_WILDCARD);
			tcp_pcbmiss++;
		} else
			tcp_pcbhit++;
    } else
	    tcp_pcbcache = tiu_lookup(&tcpcb, ti->ti_src, ti->ti_sport,
			     ti->ti_dst, ti->ti_dport, INPLOOKUP_WILDCARD);

    bp1 = tcp_pcbcache;
    /*
     * If the state is CLOSED (i.e., TCB does not exist) then
     * all data in the incoming segment is discarded.
     * If the TCB exists but is in CLOSED state, it is embryonic,
     * but should either do a listen or a connect soon.
     */
    if (bp1 == NULL) {
		STRlog(TCP_ID, 101, DPRI_LO, SL_te,
			   "tcp_input: No match for %x.%d\n",
			   ntohl(ti->ti_dst), ntohs(ti->ti_dport));
		goto dropwithreset;
	}	
    tp = (struct tcpcb *)bp1->b_rptr;
    tdp = tp->t_tdp;
	if (tdp) {
		STRlog(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_te,
		   "tcp_input: tp->t_state = %d tdp->td_tstate %d",
		   tp->t_state, tdp->td_tstate);
	}

    /*
     * Segment received on connection.
     * Reset idle time and keep-alive timer.
     */
    if (tp->t_state != TV_LISTEN) {
		tp->t_idle = 0;
/* TWU 05/03/91 using 75 secs instead of 2 hrs if SO_KEEPALIVE is set */
		if (tdp != NULL && (tdp->td_sockflags & SO_KEEPALIVE))
			tp->t_timer[TCPT_KEEP] = tcp_keepintvl;
		else
			tp->t_timer[TCPT_KEEP] = tcp_keepidle;
	}
    /*
     * Header prediction: check for the two common cases
     * of a uni-directional data xfer.  If the packet has
     * no control flags, is in-sequence, the window didn't
     * change and we're not retransmitting, it's a
     * candidate.  If the length is zero and the ack moved
     * forward, we're the sender side of the xfer.  Just
     * free the data acked & wake any higher level process
     * that was blocked waiting for space.  If the length
     * is non-zero and the ack didn't move, we're the
     * receiver side.  If we're getting packets in-order
     * (the reassembly queue is empty), add the data to
     * the socket buffer and note that we need a delayed ack.
     */
    if (tp->t_state == TV_ESTABLISHED 
	&& (tiflags & (TH_SYN|TH_FIN|TH_RST|TH_URG|TH_ACK)) == TH_ACK 
	&& ti->ti_seq == tp->rcv_nxt 
	&& ti->ti_win == tp->snd_wnd 
	&& tp->snd_nxt == tp->snd_max) {
	    if (ti->ti_len == 0) {
		    if (SEQ_GT(ti->ti_ack, tp->snd_una) 
			&& SEQ_LEQ(ti->ti_ack, tp->snd_max) 
			&& tp->snd_cwnd >= tp->snd_wnd) {
			    /*
			     * this is a pure ack for outstanding data
			     * and we're not in the middle of slow-start
			     * or congestion avoidance.
			     */
			    ++tcppredack;
#ifdef notdef
			    if (tp->t_rtt && SEQ_GT(ti->ti_ack,tp->t_rtseq))
				    tcp_xmit_timer (tp);
#endif
			    tw_acked(tp->t_tdp, ti->ti_ack - tp->snd_una);
			    tp->snd_una = ti->ti_ack;
			    freemsg (bp);
			    /*
			     * If all outstanding data is acked, stop the
			     * retransmit timer.  If there's no one
			     * waiting to output, let tcp_output decide
			     * between more output or persist.  Otherwise
			     * give the user a crack at the new space,
			     * assuming he'll call tcp_output when it's
			     * filled.  If there is more data to be acked,
			     * restart retransmit timer, using current
			     * (possibly backed-off) value.
			     */
			    if (tp->snd_una == tp->snd_max)
			    {
			     tp->t_timer[TCPT_REXMT] = 0;
			     tp->t_rxtshift = 0;		/* JTH */
			    }
			    else if (tp->t_timer[TCPT_PERSIST] == 0) 
				    tp->t_timer[TCPT_REXMT] = tp->t_rxtcur;

#ifdef notdef
			    if ((so->so_snd.sb_flags & SB_WAIT) ||
				so->so_snd.sb_sel)
				    sowwakeup(so);
			    else if (so->so_snd.sb_cc)
				    (void) tcp_output (tp);
#endif
			    if (t_wcc(tp->t_tdp)) {
				STRlog(TCP_ID, -1, DPRI_LO, SL_te, 
					"tcp_input: call tcp_output to send data on wq tp=%x",tp);
				(void) tcp_output(tp);
				}
			    return;
		    }
	    }
	    else if (ti->ti_ack == tp->snd_una 
		&& tp->t_rsq.b_next == (struct qhead *)&tp->t_rsq 
		&& ti->ti_len <= t_rwin(tp->t_tdp)) {
		    /*
		     * this is a pure, in-sequence data packet
		     * with nothing on the reassembly queue and
		     * we have enough buffer space to take it.
		     */
		    ++tcppreddat;
		    tp->rcv_nxt += ti->ti_len;
		    /*
		     * Drop TCP and IP headers then add data
		     * to socket buffer
		     */
		    tiu_data(tp->t_tdp, bp);
		    tp->t_flags |= TF_DELACK;
		    return;
	    }
    }

    /*
     * Prediction failed:  do things one step at a time.
     ...
    /*
     * Process options if not in LISTEN state,
     * else do it below (after getting remote address).
     */
    if (optbp && tp->t_state != TV_LISTEN && tp->t_state != TV_TIME_WAIT) {
		tiu_dooptions(tp, optbp, ti);
		optbp = NULL;	/* optbp is freed in 'dooptions' */
	}

    /*
     * Calculate amount of space in receive window,
     * and then do TCP input processing.
     * Receive window is amount of space in rcv queue,
     * but not less than advertised window.
     */
    { 
		int win;

		win = t_rwin(tdp);
		if ( win < 0)
			win = 0;
		tp->rcv_wnd = max(win, (int)(tp->rcv_adv - tp->rcv_nxt));
    }

	switch (tp->t_state) {

		/*
		 * If the state is LISTEN then ignore segment if it contains an RST.
		 * If the segment contains an ACK then it is bad and send a RST.
		 * If it does not contain a SYN then it is not interesting; drop it.
		 * Don't bother responding if the destination was a broadcast.
		 * Otherwise initialize tp->rcv_nxt, and tp->irs, select an initial
		 * tp->iss, and send a segment:
		 *     <SEQ=ISS><ACK=RCV_NXT><CTL=SYN,ACK>
		 * Also initialize tp->snd_nxt to tp->iss+1 and tp->snd_una to tp->iss.
		 * Fill in remote peer address fields if not previously specified.
		 * Enter SYN_RECEIVED state, and process any other fields of this
		 * segment in this state.
		 */
	case TV_LISTEN:
		/*
		 * Bind but no t_listen call.
		 */
		if (tp->t_maxcon <= 0) {
			STRlog(TCP_ID, 101, DPRI_LO, SL_te,
			   "tcp_input: listen, maxcon <= 0 %x.%d\n", T_ADDR(tp), 0);
			goto dropwithreset;
			}
		/*
		 * See if the user has done a t_accept on a
		 * new file descriptor. Establish on this.
		 */
		bp1 = tiu_lookup(&tp->t_conq, ti->ti_src, ti->ti_sport,
				 ti->ti_dst, ti->ti_dport, 0);	/* exact match */

		if (tiflags & TH_RST) {
			if (bp1)
				(void)tw_drop_floater(bp1);
			goto drop;
			}
		if (tiflags & TH_ACK)
			goto dropwithreset;
		if ((tiflags & TH_SYN) == 0)
			goto drop;
		if (broadcast)
			goto drop;
		if (bp1 != NULL) {
			/*
			 * this matches a floating tcpcb
			 * note: our rcv-window is 0, else we won't be on t_conq
			 */
			tp = (struct tcpcb *)bp1->b_rptr;
			STRlog(TCP_ID, 101, DPRI_LO, SL_te,
			   "tcp_input: Found a set up TCPCB(%x:%d)\n", T_ADDR(tp), 0);
			tdp = tp->t_tdp;
			tp->t_idle = 0;
			/* TWU 05/03/91 using 75 ms instead of 2 hrs if SO_KEEPALIVE is set */
			if (tdp != NULL && (tdp->td_sockflags & SO_KEEPALIVE))
				tp->t_timer[TCPT_KEEP] = tcp_keepintvl;
			else
				tp->t_timer[TCPT_KEEP] = tcp_keepidle;
			goto trimthenstep6;
			/* break; */
		}

		/* new request: alloc floating tcpcb */
		/*
		 * call tv_connreq to alloc tcpcb, template, initialize
		 * and to queue the floating tp on
		 * listener's t_conq and tell user via T_conn_ind
		 */
		if ((tp = tn_connreq(tdp, ti, iss)) == NULL) {
			/*
			 * e.g. q length > t_maxcon
			 * free tcpcb and RST (with no ACK ??) peer
			 * call tcp_cleanup() ???
			 */
			tiu_resp((struct tcpcb *)NULL, bp, 0, 0, TH_RST|TH_ACK);
			if (optbp)
				freeb(optbp);
			return;
		}

		/*
		 * process options. TCP and IP.
		 */
		if (optbp) {
			tiu_dooptions(tp, optbp, ti);
			optbp = NULL;	/* optbp is freed in 'dooptions' */
		}

		tcpstat.tcpPassiveOpens++;

		/*
		 * Don't send an SYN ACK now.  Let the user's tu_connres()
		 * routine do it. Otherwise there will be a race condition, if
		 * the user takes a while to do an accept, and a FIN arrives.
		 tw_send(tp, NULL, NULL, tp->snd_nxt, tp->rcv_nxt, TH_SYN|TH_ACK);
		 */
		goto trimthenstep6;

		/*
		 * If the state is SYN_SENT:
		 *	if seg contains an ACK, but not for our SYN, drop the input.
		 *	if seg contains a RST, then drop the connection.
		 *	if seg does not contain SYN, then drop it.
		 * Otherwise this is an acceptable SYN segment
		 *	initialize tp->rcv_nxt and tp->irs
		 *	if seg contains ack then advance tp->snd_una
		 *	if SYN has been acked change to ESTABLISHED else SYN_RCVD state
		 *	arrange for segment to be acked (eventually)
		 *	continue processing rest of data/controls, beginning with URG
		 */
	case TV_SYN_SENT:
		if ((tiflags & TH_ACK) 
		&& (SEQ_LEQ(ti->ti_ack, tp->iss) 
			|| SEQ_GT(ti->ti_ack, tp->snd_max)))
			goto dropwithreset;
		if (tiflags & TH_RST) {
			if (tiflags & TH_ACK)
			tiu_dropit(tp, ECONNREFUSED);
			goto drop;
		}
		if ((tiflags & TH_SYN) == 0)
			goto drop;
		if (tiflags & TH_ACK) {
			tp->snd_una = ti->ti_ack;
			if (SEQ_LT(tp->snd_nxt, tp->snd_una))
			tp->snd_nxt = tp->snd_una;
		}
		tp->t_timer[TCPT_REXMT] = 0;
		tp->irs = ti->ti_seq;
		tcp_rcvseqinit(tp);
		tp->t_flags |= TF_ACKNOW;
		if (tiflags & TH_ACK && SEQ_GT(tp->snd_una, tp->iss)) {
			STRLOG(TCP_ID, 101, DPRI_LO, SL_TRACE,
			   "tcp_input: ESTABLISHED %x.%d", T_ADDR(tp), 0);
			tcpstat.tcps_connects++;
			tn_connected(tdp, ti->ti_src, ti->ti_sport);

			tp->t_state = TV_ESTABLISHED;
			tcp_bestmss(tp);
			(void) tiu_reass(tp, (mblk_t *)NULL);
			/*
			 * if we didn't have to retransmit the SYN,
			 * use its rtt as our initial srtt & rtt var.
			 */
			if (tp->t_rtt) {
			tp->t_srtt = tp->t_rtt << 3;
			tp->t_rttvar = tp->t_rtt << 1;
			TCPT_RANGESET(tp->t_rxtcur, 
					  ((tp->t_srtt >> 2) + tp->t_rttvar) >> 1,
					  TCPTV_MIN, TCPTV_REXMTMAX);
			tp->t_rtt = 0;
			}
		} else
			tp->t_state = TV_SYN_RCVD;

trimthenstep6:
		/*
		 * Advance ti->ti_seq to correspond to first data byte.
		 * If data, trim to stay within window,
		 * dropping FIN if necessary.
		 */
		ti->ti_seq++;
		if (ti->ti_len > tp->rcv_wnd) {
			todrop = ti->ti_len - tp->rcv_wnd;
			if (adjmsg(bp, -todrop) == 0)
				goto drop;	/* implementation error */
			ti->ti_len = tp->rcv_wnd;
			tiflags &= ~TH_FIN;
			tcpstat.tcps_rcvpackafterwin++;
			tcpstat.tcps_rcvbyteafterwin += todrop;
		}
		tp->snd_wl1 = ti->ti_seq - 1;
		tp->rcv_up = ti->ti_seq;
		goto step6;
	}
    /*
     * States other than LISTEN or SYN_SENT.
     * First check that at least some bytes of segment are within 
     * receive window.  If segment begins before rcv_nxt,
     * drop leading data (and SYN); if nothing left, just ack.
     */
    todrop = tp->rcv_nxt - ti->ti_seq;
	if (todrop > 0) {
		if (tiflags & TH_SYN) {
			tiflags &= ~TH_SYN;
			ti->ti_seq++;
			if (ti->ti_urp > 1) 
				ti->ti_urp--;
			else
				tiflags &= ~TH_URG;
			todrop--;
		}
		if (todrop > ti->ti_len 
		|| todrop == ti->ti_len && (tiflags&TH_FIN) == 0) {
			tcpstat.tcps_rcvduppack++;
			tcpstat.tcps_rcvdupbyte += ti->ti_len;
			/*
			 * If segment is just one to the left of the window,
			 * check two special cases:
			 * 1. Don't toss RST in response to 4.2-style keepalive.
			 * 2. If the only thing to drop is a FIN, we can drop
			 *    it, but check the ACK or we will get into FIN
			 *    wars if our FINs crossed (both CLOSING).
			 * In either case, send ACK to resynchronize,
			 * but keep on processing for RST or ACK.
			 */
			if ((tiflags & TH_FIN && todrop == ti->ti_len + 1)
#ifdef TCP_COMPAT_42
			|| (tiflags & TH_RST && ti->ti_seq == tp->rcv_nxt - 1)
#endif
			) {
				todrop = ti->ti_len;
				tiflags &= ~TH_FIN;
				tp->t_flags |= TF_ACKNOW;
			} else
				goto dropafterack;
		} else {
			tcpstat.tcps_rcvpartduppack++;
			tcpstat.tcps_rcvpartdupbyte += todrop;
		}
		adjmsg(bp, todrop);
		ti->ti_seq += todrop;
		ti->ti_len -= todrop;
		if (ti->ti_urp > todrop)
			ti->ti_urp -= todrop;
		else {
			tiflags &= ~TH_URG;
			ti->ti_urp = 0;
		}
	}
    /*
     * If new data are received on a connection after the
     * user processes are gone, then RST the other end.
     */
    if (tp->t_tdp == NULL 
	&& tp->t_state > TV_CLOSE_WAIT 
	&& ti->ti_len) {
		/*	tp = tcp_close(tp);*/
		tcpstat.tcps_rcvafterclose++;
		STRLOG(TCP_ID, 101, DPRI_LO, SL_TRACE,
			"tcp_input: Data after user gone");
		goto dropwithreset;
	}

    /*
     * If segment ends after window, drop trailing data
     * (and PUSH and FIN); if nothing left, just ACK.
     */
    todrop = (ti->ti_seq+ti->ti_len) - (tp->rcv_nxt+tp->rcv_wnd);
	if (todrop > 0) {
		tcpstat.tcps_rcvpackafterwin++;
		if (todrop >= ti->ti_len) {
			tcpstat.tcps_rcvbyteafterwin += ti->ti_len;
			/*
			 * If a new connection request is received
			 * while in TIME_WAIT, drop the old connection
			 * and start over if the sequence numbers
			 * are above the previous ones.
			 */
			if (tiflags & TH_SYN 
			&& tp->t_state == TV_TIME_WAIT 
			&& SEQ_GT(ti->ti_seq, tp->rcv_nxt)) {
				iss = tp->rcv_nxt + TCP_ISSINCR;
				(void) tw_close(tp);
				goto findpcb;
			}
			/*
			 * If window is closed can only take segments at
			 * window edge, and have to drop data and PUSH from
			 * incoming segments.  Continue processing, but
			 * remember to ack.  Otherwise, drop segment
			 * and ack.
			 */
			if (tp->rcv_wnd == 0 && ti->ti_seq == tp->rcv_nxt) {
				tp->t_flags |= TF_ACKNOW;
				tcpstat.tcps_rcvwinprobe++;
			} else
				goto dropafterack;
		} else
			tcpstat.tcps_rcvbyteafterwin += todrop;
		adjmsg(bp, -todrop);
		ti->ti_len -= todrop;
		tiflags &= ~(TH_PUSH|TH_FIN);
	}

    /*
     * If the RST bit is set examine the state:
     *    SYN_RECEIVED STATE:
     *	If passive open, return to LISTEN state.
     *	If active open, inform user that connection was refused.
     *    ESTABLISHED, FIN_WAIT_1, FIN_WAIT2, CLOSE_WAIT STATES:
     *	Inform user that connection was reset, and close tcb.
     *    CLOSING, LAST_ACK, TIME_WAIT STATES
     *	Close the tcb.
     */
    if (tiflags&TH_RST) {
		STRlog(TCP_ID, 101, DPRI_LO, SL_te,"tcp_input: RST is on tiflags = %x",
			tiflags);
		switch (tp->t_state) {

		case TV_SYN_RCVD:
			tiu_dropit(tp, ECONNREFUSED);
			tcpstat.tcpAttemptFails++;
			goto drop;

		case TV_ESTABLISHED:
		case TV_CLOSE_WAIT:
			tcpstat.tcpEstabResets++;
		case TV_FIN_W1:
		case TV_FIN_W2:
		case TV_CLOSED:
			STRlog(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_te,
				   "tcp_input: doing tiu_dropit tp->t_state = %d tdp->td_tstate %d", tp->t_state, tdp->td_tstate);
			tiu_dropit(tp, ECONNRESET);
			tcpstat.tcps_drops++;
			goto drop;

		case TV_CLOSING:
		case TV_LAST_ACK:
		case TV_TIME_WAIT:
			tw_close(tp);
			goto drop;
		}
	}

    /*
     * If a SYN is in the window, then this is an
     * error and we send an RST and drop the connection.
     */
    if (tiflags & TH_SYN) {
		switch(tp->t_state) {
		case TV_SYN_RCVD:
			tcpstat.tcpAttemptFails++;
			break;
		case TV_ESTABLISHED:
		case TV_CLOSE_WAIT:
		case TV_FIN_W1:
		case TV_FIN_W2:
			tcpstat.tcpEstabResets++;
		}
		STRlog(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_te,
			   "tcp_input: SYN is in window reset t_state = %d td_tstate %d",				 	tp->t_state, tdp->td_tstate);
		tiu_dropit(tp, ECONNRESET);
		goto dropwithreset;
	}

    /*
     * If the ACK bit is off we drop the segment and return.
     */
    if ((tiflags & TH_ACK) == 0)
		goto drop;

    /*
     * Ack processing.
     */
	STRlog(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_te,
		   "tcp_input: doing ACK processing t_state = %d td_tstate %d",
		   (tp ? tp->t_state : -1) , (tdp ? tdp->td_tstate : -1));

	switch (tp->t_state) {
		/*
		 * In SYN_RECEIVED state if the ack ACKs our SYN then enter
		 * ESTABLISHED state and continue processing, otherwise
		 * send an RST.
		 */
	case TV_SYN_RCVD:
		if (SEQ_GT(tp->snd_una, ti->ti_ack) ||
			SEQ_GT(ti->ti_ack, tp->snd_max)) {
			STRlog(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_te,
			   "tcp_input: bad ACK sending RST");
			goto dropwithreset;
		}
		tcpstat.tcps_connects++;
		/*
		 * we already told user the connection was up, via:
		 * tn_connreq(tdp, ti, iss)
		 * if tli in TS_DATA_XFER already, enable write q and set
		 * rcv window size
		 */
		if (tdp->td_tstate >= TS_DATA_XFER) {
			enableok(  WR(tdp->td_rdq));
			if (qsize( WR(tdp->td_rdq)))
				qenable( WR(tdp->td_rdq));
			tp->rcv_wnd = t_rwin(tdp);
		}
		oursynisacked = 1;
		tp->t_state = TV_ESTABLISHED;
		tcp_bestmss(tp);
		(void) tiu_reass(tp, (mblk_t *)0);
		tp->snd_wl1 = ti->ti_seq - 1;
		/* fall into ... */

		/*
		 * In ESTABLISHED state: drop duplicate ACKs; ACK out of range
		 * ACKs.  If the ack is in the range
		 *	tp->snd_una < ti->ti_ack <= tp->snd_max
		 * then advance tp->snd_una to ti->ti_ack and drop
		 * data from the retransmission queue.  If this ACK reflects
		 * more up to date window information we update our window information.
		 */
	case TV_ESTABLISHED:
	case TV_FIN_W1:
	case TV_FIN_W2:
	case TV_CLOSE_WAIT:
	case TV_CLOSING:
	case TV_LAST_ACK:

		if (SEQ_LEQ(ti->ti_ack, tp->snd_una)) {
			if (ti->ti_len == 0 && ti->ti_win == tp->snd_wnd) {
				tcpstat.tcps_rcvdupack++;
				/*
				 * If we have outstanding data (not a
				 * window probe), this is a completely
				 * duplicate ack (ie, window info didn't
				 * change), the ack is the biggest we've
				 * seen and we've seen exactly our rexmt
				 * threshhold of them, assume a packet
				 * has been dropped and retransmit it.
				 * Kludge snd_nxt & the congestion
				 * window so we send only this one
				 * packet.  If this packet fills the
				 * only hole in the receiver's seq.
				 * space, the next real ack will fully
				 * open our window.  This means we
				 * have to do the usual slow-start to
				 * not overwhelm an intermediate gateway
				 * with a burst of packets.  Leave
				 * here with the congestion window set
				 * to allow 2 packets on the next real
				 * ack and the exp-to-linear thresh
				 * set for half the current window
				 * size (since we know we're losing at
				 * the current window size).
				 */
				if (tp->t_timer[TCPT_REXMT] == 0 
				|| ti->ti_ack != tp->snd_una)
					tp->t_dupacks = 0;
				else if (++tp->t_dupacks == tcprexmtthresh) {
					tcp_seq onxt = tp->snd_nxt;
					u_int win =
					min(tp->snd_wnd, tp->snd_cwnd) / 2 /
						tp->t_maxseg;

					if (win < 2)
					win = 2;
					tp->snd_ssthresh = win * tp->t_maxseg;

					tp->t_timer[TCPT_REXMT] = 0;
					tp->t_rtt = 0;
					tp->snd_nxt = ti->ti_ack;
					tp->snd_cwnd = tp->t_maxseg;
					STRlog(TCP_ID, -1, DPRI_LO, SL_te, 
						"tcp_input: calling tcp_output after setting snd_cwnd tp = %x", tp);
					(void) tcp_output(tp);

					if (SEQ_GT(onxt, tp->snd_nxt))
					tp->snd_nxt = onxt;
					goto drop;
				}
			} else
				tp->t_dupacks = 0;
			break;
		}
		tp->t_dupacks = 0;
		if (SEQ_GT(ti->ti_ack, tp->snd_max)) {
			tcpstat.tcps_rcvacktoomuch++;
			goto dropafterack;
		}
		acked = ti->ti_ack - tp->snd_una - oursynisacked;
		tcpstat.tcps_rcvackpack++;
		tcpstat.tcps_rcvackbyte += acked;

		/*
		 * If transmit timer is running and timed sequence
		 * number was acked, update smoothed round trip time.
		 * Since we now have an rtt measurement, cancel the
		 * timer backoff (cf., Phil Karn's retransmit alg.).
		 * Recompute the initial retransmit timer.
		 */
		if (tp->t_rtt && SEQ_GT(ti->ti_ack, tp->t_rtseq)) {
			tcpstat.tcps_rttupdated++;
			if (tp->t_srtt != 0) {
				register short delta;

				/*
				 * srtt is stored as fixed point with 3 bits
				 * after the binary point (i.e., scaled by 8).
				 * The following magic is equivalent
				 * to the smoothing algorithm in rfc793
				 * with an alpha of .875
				 * (srtt = rtt/8 + srtt*7/8 in fixed point).
				 * Adjust t_rtt to origin 0.
				 */
				delta = tp->t_rtt - 1 - (tp->t_srtt >> 3);
				if ((tp->t_srtt += delta) <= 0)
					tp->t_srtt = 1;
				/*
				 * We accumulate a smoothed rtt variance
				 * (actually, a smoothed mean difference),
				 * then set the retransmit timer to smoothed
				 * rtt + 2 times the smoothed variance.
				 * rttvar is stored as fixed point
				 * with 2 bits after the binary point
				 * (scaled by 4).  The following is equivalent
				 * to rfc793 smoothing with an alpha of .75
				 * (rttvar = rttvar*3/4 + |delta| / 4).
				 * This replaces rfc793's wired-in beta.
				 */
				if (delta < 0)
					delta = -delta;
				delta -= (tp->t_rttvar >> 2);
				if ((tp->t_rttvar += delta) <= 0)
					tp->t_rttvar = 1;
			} else {
				/* 
				 * No rtt measurement yet - use the
				 * unsmoothed rtt.  Set the variance
				 * to half the rtt (so our first
				 * retransmit happens at 2*rtt)
				 */
				tp->t_srtt = tp->t_rtt << 3;
				tp->t_rttvar = tp->t_rtt << 1;
			}
			tp->t_rtt = 0;
			tp->t_rxtshift = 0;
			TCPT_RANGESET(tp->t_rxtcur, 
				  ((tp->t_srtt >> 2) + tp->t_rttvar) >> 1,
				  TCPTV_MIN, TCPTV_REXMTMAX);
		}

		/*
		 * If all outstanding data is acked, stop retransmit
		 * timer and remember to restart (more output or persist).
		 * If there is more data to be acked, restart retransmit
		 * timer, using current (possibly backed-off) value.
		 */
		if (ti->ti_ack == tp->snd_max) {
			tp->t_timer[TCPT_REXMT] = 0;
			tp->t_rxtshift = 0;		/* JTH */
			needoutput = 1;
		} else if (tp->t_timer[TCPT_PERSIST] == 0) 
			tp->t_timer[TCPT_REXMT] = tp->t_rxtcur;
		/*
		 * When new data is acked, open the congestion window.
		 * If the window gives us less than ssthresh packets
		 * in flight, open exponentially (maxseg per packet).
		 * Otherwise open linearly (maxseg per window,
		 * or maxseg^2 / cwnd per packet).
		 */
		{
			u_int incr = tp->t_maxseg;

			if (tp->snd_cwnd > tp->snd_ssthresh)
				incr = max(incr * incr / tp->snd_cwnd, 1);

			tp->snd_cwnd = min(tp->snd_cwnd + incr, IP_MAXPACKET); /* XXX */
		}
		todrop = t_wcc(tdp);
		if (acked > todrop) {
			tp->snd_wnd -= todrop;
			tw_acked(tdp, todrop);
			ourfinisacked = 1;
		} else {
			tw_acked(tdp, acked);
			tp->snd_wnd -= acked;
			ourfinisacked = 0;
		}
		tp->snd_una = ti->ti_ack;
		if (SEQ_LT(tp->snd_nxt, tp->snd_una))
			tp->snd_nxt = tp->snd_una;

	case TV_TIME_WAIT:
		switch (tp->t_state) {

			/*
			 * In FIN_WAIT_1 STATE in addition to the processing
			 * for the ESTABLISHED state if our FIN is now acknowledged
			 * then enter FIN_WAIT_2.
			 */
		case TV_FIN_W1:
			if (ourfinisacked) {
				tp->t_state = TV_FIN_W2;
				tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;	/* PDF TIMERS */
			}
			break;

			/*
			 * In CLOSING STATE in addition to the processing for
			 * the ESTABLISHED state if the ACK acknowledges our FIN
			 * then enter the TIME-WAIT state, otherwise ignore
			 * the segment.
			 */
		case TV_CLOSING:
			if (ourfinisacked) {
				tp->t_state = TV_TIME_WAIT;
				tcp_canceltimers(tp);
				tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;
				STRLOG(TCP_ID, 101, DPRI_LO, SL_TRACE,
					   "tcp_input: entering TIME_WAIT from CLOSING");
				if (tp->t_tdp)
					wakeup(tp->t_tdp);	/* Wake up anyone sleeping on close */
			}
			break;

			/*
			 * In LAST_ACK, we may still be waiting for data to drain
			 * and/or to be acked, as well as for the ack of our FIN.
			 * If our FIN is now acknowledged, delete the TCB,
			 * enter the closed state and return.
			 */
		case TV_LAST_ACK:
			if (ourfinisacked) {
				tp->t_state = TV_CLOSED;
				tw_close(tp);
				goto drop;
			}
			break;

			/*
			 * In TIME_WAIT state the only thing that should arrive
			 * is a retransmission of the remote FIN.  Acknowledge
			 * it and restart the finack timer.
			 */
		case TV_TIME_WAIT:
			tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;
			goto dropafterack;
		}
	}

step6:
    /*
     * Update window information.
     * Don't look at window if no ACK: TAC's send garbage on first SYN.
     */
    if ((tiflags & TH_ACK) 
	&& (SEQ_LT(tp->snd_wl1, ti->ti_seq) 
		|| tp->snd_wl1 == ti->ti_seq && (SEQ_LT(tp->snd_wl2, ti->ti_ack) 
		|| tp->snd_wl2 == ti->ti_ack && ti->ti_win > tp->snd_wnd)
	   )) {
		/* keep track of pure window updates */
		if (ti->ti_len == 0 
		&& tp->snd_wl2 == ti->ti_ack 
		&& ti->ti_win > tp->snd_wnd)
			tcpstat.tcps_rcvwinupd++;
		tp->snd_wnd = ti->ti_win;
		tp->snd_wl1 = ti->ti_seq;
		tp->snd_wl2 = ti->ti_ack;
		if (tp->snd_wnd != 0)
			tp->t_timer[TCPT_PERSIST] = 0;	/* PDF TIMERS */
		if (tp->snd_wnd > tp->max_sndwnd)
			tp->max_sndwnd = tp->snd_wnd;
		needoutput = 1;
	}

    /*
     * Process segments with URG.
     */
    if ((tiflags & TH_URG) && ti->ti_urp 
	&& TV_HAVERCVDFIN(tp->t_state) == 0) {
		/*
		 * If this segment advances the known urgent pointer,
		 * then mark the data stream.  This should not happen
		 * in CLOSE_WAIT, CLOSING, LAST_ACK or TIME_WAIT STATES since
		 * a FIN has been received from the remote side. 
		 * In these states we ignore the URG.
		 *
		 * According to RFC961 (Assigned Protocols),
		 * the urgent pointer points to the last octet
		 * of urgent data.  We continue, however,
		 * to consider it to indicate the first octet
		 * of data past the urgent section
		 * as the original spec states.
		 */
		if (SEQ_GT(ti->ti_seq+ti->ti_urp, tp->rcv_up))
			tp->rcv_up = ti->ti_seq + ti->ti_urp;
		/*
		 * Remove out of band data so doesn't get presented to user.
		 * This can happen independent of advancing the URG pointer,
		 * but if two URG's are pending at once, some out-of-band
		 * data may creep in... ick.
		 */
		if (ti->ti_urp <= ti->ti_len)
			tn_exdata(tdp);
	} 
	/*
	 * If no out of band data is expected,
	 * pull receive urgent pointer along
	 * with the receive window.
	 */
	else if (SEQ_GT(tp->rcv_nxt, tp->rcv_up))
	    tp->rcv_up = tp->rcv_nxt;

    /*
     * Process the segment text, merging it into the TCP sequencing queue,
     * and arranging for acknowledgment of receipt if necessary.
     * This process logically involves adjusting tp->rcv_wnd as data
     * is presented to the user (this happens in tcp_usrreq.c,
     * case PRU_RCVD).  If a FIN has already been received on this
     * connection then we just ignore the text.
     */
    if ((ti->ti_len 
	|| (tiflags&TH_FIN)) && TV_HAVERCVDFIN(tp->t_state) == 0) {
		TCP_REASS(tp,ti,bp,tiflags);
		/*
		 * Note the amount of data that peer has sent into
		 * our window, in order to estimate the sender's
		 * buffer size.
		 */
		len = t_rhiw(tp->t_tdp) - (tp->rcv_adv - tp->rcv_nxt);
		if (len > tp->max_rcvd)
			tp->max_rcvd = len;
	} else {
		freemsg(bp);
		tiflags &= ~TH_FIN;
	}

    /*
     * If FIN is received ACK the FIN and let the user know
     * that the connection is closing.
     */
	if (tiflags & TH_FIN) {
		if (TV_HAVERCVDFIN(tp->t_state) == 0) {
			tp->t_flags |= TF_ACKNOW;
			tp->rcv_nxt++;
		}
		switch (tp->t_state) {

			/*
			 * In SYN_RECEIVED and ESTABLISHED STATES
			 * enter the CLOSE_WAIT state.
			 */
		case TV_SYN_RCVD:
		case TV_ESTABLISHED:
			tp->t_state = TV_CLOSE_WAIT;
			tn_orddisconn(tdp, (mblk_t *)NULL);
			break;

			/*
			 * If still in FIN_WAIT_1 STATE FIN has not been acked so
			 * enter the CLOSING state.
			 */
		case TV_FIN_W1:
			tp->t_state = TV_CLOSING;
			break;

			/*
			 * In FIN_WAIT_2 state enter the TIME_WAIT state,
			 * starting the time-wait timer, turning off the other 
			 * standard timers.
			 */
		case TV_FIN_W2:
			if (tdp && (tdp->td_flags & TD_CLOSING) == 0)
				tn_orddisconn(tdp,(mblk_t *) NULL);
			tp->t_state = TV_TIME_WAIT;
			tcp_canceltimers(tp);
			tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;
			STRLOG(TCP_ID, 101, DPRI_LO, SL_TRACE,
				"tcp_input: entering TIME_WAIT from FIN_WAIT_2");
			if (tp->t_tdp)
				wakeup(tp->t_tdp);	/* Wake up anyone sleeping on close */
			break;

			/*
			 * In TIME_WAIT state restart the 2 MSL time_wait timer.
			 */
		case TV_TIME_WAIT:
			tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;
			break;
		}
	}

    /*
     * Return any desired output.
     */
    if (needoutput || (tp->t_flags & TF_ACKNOW)) {
		STRlog(TCP_ID, -1, DPRI_LO, SL_te, 
			"tcp_input: calling tcp_output tp = %x needoutput = %d t_flags = %x"			,tp, needoutput, tp->t_flags);
		(void) tcp_output(tp);
	}
    return;

dropafterack:
    /*
     * Generate an ACK dropping incoming segment if it occupies
     * sequence space, where the ACK reflects our state.
     */
    if (tiflags & TH_RST)
		goto drop;
    freemsg(bp);
    tp->t_flags |= TF_ACKNOW;
	STRlog(TCP_ID, -1, DPRI_LO, SL_te, 
		"tcp_input: calling tcp_output in dropafterack tp = %x", tp);
    (void) tcp_output(tp);
    return;

dropwithreset:
    if (optbp) {
		(void) freemsg(optbp);
		optbp = 0;
	}
    /*
     * Generate a RST, dropping incoming segment.
     * Make ACK acceptable to originator of segment.
     * Don't bother to respond if destination was broadcast.
     */
	STRlog(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_te,
	   "tcp_input: droppwithreset: tiflags = %x", tiflags);
    if ((tiflags & TH_RST) || broadcast)
		goto drop;
    if (tiflags & TH_ACK)
		tiu_resp(tp, bp, (tcp_seq)0, ti->ti_ack, TH_RST);
    else {
		if (tiflags & TH_SYN)
			ti->ti_len++;
		tiu_resp(tp, bp, ti->ti_seq+ti->ti_len, (tcp_seq)0, TH_RST|TH_ACK);
	}
    goto done;

drop:
    freemsg(bp);
done:
    if (optbp)
		(void) freemsg(optbp);
    /*
     * Drop space held by incoming segment and return.
     */
    return;

}	/*  End of tcp_input()  */


/*
 * Send a single message to the TCP at address specified by
 * the given TCP/IP header.  If flags==0, then we make a copy
 * of the tcpiphdr at ti and send directly to the addressed host.
 * This is used to force keep alive messages out using the TCP
 * template for a connection tp->t_template.  If flags are given
 * then we send a message back to the TCP which originated the
 * segment ti, and discard the block containing it and any other
 * attached blocks.
 *
 * In any case the ack and sequence number of the transmitted
 * segment are as specified by the parameters.
 */
void
tiu_resp(tp, bp, ack, seq, flags)
     struct tcpcb *tp;
     register mblk_t *bp;
     tcp_seq ack, seq;
     unchar flags;
{
    register struct tcpiphdr *ti;

    STRlog(TCP_ID, 101, DPRI_LO, SL_te,"tiu_resp: flags = %x", flags);

    bp->b_rptr -= sizeof(struct tcpiphdr);	/* make header visible */
    /*
     * retain the first block and discard the rest
     */
    if (bp->b_cont) {
		freemsg(bp->b_cont);
		bp->b_cont = NULL;
	}
    ti = (struct tcpiphdr *)bp->b_rptr;
    bp->b_wptr = bp->b_rptr + TCPIPHLEN;
#define xchg(a,b,type) { type t; t=a; a=b; b=t; }
    xchg(ti->ti_dst, ti->ti_src, ulong);
    xchg(ti->ti_dport, ti->ti_sport, ushort);
#undef xchg
    (void) tw_send(tp, bp, NULL, seq, ack, flags);
    return;

}	/*  End of tiu_resp()  */


void
tiu_dooptions(tp, om, ti)
     struct tcpcb *tp;
     mblk_t *om;
     struct tcpiphdr *ti;
{
    register unchar *cp;
    int opt, optlen, cnt;

    cp = (unchar *) om->b_rptr;
    cnt = BLEN(om);
	for (; cnt > 0; cnt -= optlen, cp += optlen) {
		opt = cp[0];
		if (opt == TCPOPT_EOL)
			break;
		if (opt == TCPOPT_NOP)
			optlen = 1;
		else {
			optlen = cp[1];
			if (optlen <= 0)
				break;
		}
		switch (opt) {

		default:
			break;

		case TCPOPT_MAXSEG:
			if (optlen != 4)
				continue;
			if (!(ti->ti_flags & TH_SYN))
				continue;
			tp->t_maxseg = *(u_short *)(cp + 2);
			tp->t_maxseg = ntohs((u_short)tp->t_maxseg);
			tcp_bestmss(tp);
			STRLOG(TCP_ID, TCPCHAN(tp->t_tdp), DPRI_LO, SL_TRACE,
			   "max seg set to %x", tp->t_maxseg);
			break;
		}
	}
    (void) freemsg(om);
    return;

}	/*  End of tiu_dooptions()  */


/*
 * Find the Loose or Strict route.
 * Build reverse route and store into tcpd
 */

void
tiu_ipoptions(td, mopt)
	struct tcpd *td;
	mblk_t *mopt;

{
	register unchar *cp;
	int opt, optlen, cnt, inc;
	register struct in_addr *p, *q;
	register mblk_t *bp;

	cp = (unchar *) mopt->b_rptr;
	cnt = BLEN(mopt);
	for (; cnt > 0; cnt -= optlen, cp += optlen) {
		opt = cp[IPOPT_OPTVAL];
		if (opt == IPOPT_EOL)
			break;
		if (opt == IPOPT_NOP)
			optlen = 1;
		else {
			optlen = cp[IPOPT_OLEN];
			if (optlen <= 0)
				break;
		}
		switch (opt) {

		default:
			break;

		case IPOPT_SECURITY:
		case IPOPT_EXTSECURITY:
		case IPOPT_SATID:
		case IPOPT_TS:
		case IPOPT_RR:
			break;

		case IPOPT_LSRR:
		case IPOPT_SSRR:

			if ((bp = allocb(MAX_IPOPTLEN, BPRI_LO)) == NULL) {
				cnt = 0;
				break;
			}

			/* 
			 * Move over and initialize new option header
			 */
			*(bp->b_rptr) = *cp;
			*(bp->b_rptr + 1) = cp[IPOPT_OLEN];
			*(bp->b_rptr + 2) = IPOPT_MINOFF;

			/*
			 * Initialize from and to pointers
			 */	
			p = (struct in_addr *)(cp + cp[IPOPT_OLEN]);
			q = (struct in_addr *)(mtod(bp,  char *) + IPOPT_MINOFF - 1);
			p--;

			for(; (unchar *)p > cp; p--,q++)
				bcopy(p, q, sizeof( struct in_addr));

			bp->b_wptr = (unchar *)q;

			if( inc = (BLEN( bp ) % 4) )
				for( ; inc < 4; inc++ )
					*bp->b_wptr++ = IPOPT_EOL;

			/*
			 * For use on outgoing tcp segments
			 */
			if ( td->td_ipopt )
				freemsg( td->td_ipopt );
			td->td_ipopt = bp;
			TRACE(1<<10,"tiu_ipoptions: tcpd->td_ipopt = 0x%x\n",bp)
			break;
		}
	}
	return;

}	/*  End of tiu_ipoptions()  */


void
tiu_data(tdp, bp)
     register struct tcpd *tdp;
     register mblk_t *bp;
{
    extern void tn_data();
    mblk_t **bpx, *bp_hd, *bp_free;

    /*
     * Dump the empty buffers.
     */
    bp_hd = bp;
	for (bpx = &bp_hd; *bpx != (mblk_t *)NULL;) {
		if (BLEN(*bpx) == 0) {
			bp_free = *bpx;
			*bpx = bp_free->b_cont;
			bp_free->b_cont = (mblk_t *) NULL;
			freeb(bp_free);
		} else
			bpx = &((*bpx)->b_cont);
	}
    bp = bp_hd;
	if (bp) {
#ifdef notdef
		if ( (ti->ti_flags & TH_URG) && (BLEN(bp))){
			register mblk_t  *bp2;
			if ((bp2 = allocb(ti->ti_len, BPRI_LO)) == NULL) {
				STRlog(TCP_ID, 101, DPRI_LO, SL_te,"tcp_input: No buffers\n");
				goto dropall;
			}

			MTYPE(bp) = M_DATA;
			bcopy(bp->b_rptr,bp2->b_rptr,ti->ti_urp);
			bp2->b_wptr = bp2->b_wptr + ti->ti_urp;
			bp->b_rptr = bp->b_rptr + ti->ti_urp;
			tn_exdata1(tdp,bp2);
		}
#endif
		tn_data(tdp, bp);
	}
} /* end tiu_data */

/*
 * the mesg "bp" has rptr points to data and rptr-40 points to tcpiphdr
 * insert bp into reassembly queue for the control block tp.
 * return TH_FIN if reassembly now includes a segment with FIN.
 */
static int
tiu_reass(tp, bp)
     register struct tcpcb *tp;
     mblk_t *bp;
{
    register struct tcpiphdr *ti, *ti1;
    register mblk_t          *bp1, *head;
    register int             i;
    struct tcpd              *tdp;

    ASSERT(tp);
    tdp = tp->t_tdp;
    head = (mblk_t *)&tp->t_rsq;
    /*
     * call with bp==NULL after become established to
     * force pre-ESTABLISHED data up to user tcpd.
     */
    if (bp == NULL)
		goto present;

    /* XXX can't be MBASE(bp), ethernet hdr in front */
    ti = (struct tcpiphdr *)(bp->b_rptr - sizeof(*ti));
    for (bp1 = head->b_next; bp1 != head; bp1 = bp1->b_next) {
		ti1 = (struct tcpiphdr *)(bp1->b_rptr - sizeof(*ti));
		if (SEQ_GT(ti1->ti_seq, ti->ti_seq))
			break;
	}
    /*
     * If there is a preceding segment, it may provide some of
     * our data already.  If so, drop the data from the incoming
     * segment.  If it provides all of our data, drop us.
     */
	if (bp1->b_prev != head) {
		bp1 = bp1->b_prev;
		ti1 = (struct tcpiphdr *)(bp1->b_rptr - sizeof(*ti));
		i = ti1->ti_seq + ti1->ti_len - ti->ti_seq;
		if (i > 0) {
			/* drop i bytes from incoming seg */
			if (i >= ti->ti_len) {
				tcpstat.tcps_rcvduppack++;
				tcpstat.tcps_rcvdupbyte += ti->ti_len;
				goto dropall;
			}
			if (tiu_adjmsg(bp, i) == 0)
				goto adjerr;
			ti->ti_len -= i;
			ti->ti_seq += i;
		}
		bp1 = bp1->b_next;
	}

    tcpstat.tcps_rcvoopack++;
    tcpstat.tcps_rcvoobyte += ti->ti_len;

    /*
     * While we overlap succeeding segments trim them or,
     * if they are completely covered, dequeue them.
     */
	while (bp1 != head) {
		register mblk_t *bp2;

		ti1 = (struct tcpiphdr *)(bp1->b_rptr - sizeof(*ti));
		i = (ti->ti_seq + ti->ti_len) - ti1->ti_seq;
		if (i <= 0)
			break;
		if (i < ti1->ti_len) {
			ti1->ti_seq += i;
			ti1->ti_len -= i;
			if (tiu_adjmsg(bp1, i) == 0)
				goto adjerr;
			break;
			}
		bp2 = bp1->b_next;
		c_deq(bp1);
		freemsg(bp1);
		bp1 = bp2;
	}
    c_enq(bp, bp1->b_prev);

present:
    /*
     * present data to user, advancing rcv_nxt through
     * completed sequence space.
     */
    if (TV_HAVERCVDSYN(tp->t_state) == 0)
		return 0;

    bp = head->b_next;
    ti = (struct tcpiphdr *)(bp->b_rptr - sizeof(*ti));
    if ((bp == (mblk_t *)head) || ti->ti_seq != tp->rcv_nxt)
		return 0;
    if (tp->t_state == TV_SYN_RCVD && ti->ti_len)
		return 0;

    i = 0;
	while (bp != head && ti->ti_seq == tp->rcv_nxt) {
		tp->rcv_nxt += ti->ti_len;
		i = ti->ti_flags & TH_FIN;	/* what ?? */
		bp1 = bp->b_next;
		c_deq(bp);
		tiu_data(tdp, bp);
		bp = bp1;
		ti = (struct tcpiphdr *)(bp->b_rptr - sizeof(*ti));
	}
    return(i);

adjerr:
    STRlog(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_te,
	   "tiu_reass: adjerr %x\n", bp);
dropall:
    STRLOG(TCP_ID, TCPCHAN(tdp), DPRI_LO, SL_TRACE,
	   "tiu_reass: dropall\n");
    freemsg(bp);
    return(0);

}	/*  End of tiu_reass()  */


void
tiu_dropit(tp, error)
	struct tcpcb *tp;

{
	extern void tn_usrmsg();
	extern void tn_disconnected();
	extern void tw_drop();

	tn_usrmsg(tp, error);
	tn_disconnected(tp); 		/* Send up SIGHUP */
	tw_drop(tp);
	return;

}	/*  End of tiu_dropit()  */


extern mblk_t *bp_copy();

/*
 * drop "todrop" bytes from bp, which has a hidden tcpiphdr in front
 * like "adjmsg()", return 0 on failer, 1 on success.
 */
static int
tiu_adjmsg(bp, todrop)

	register mblk_t *bp;
	int todrop;
{
	register struct tcpiphdr *ti;
	register mblk_t *nbp;

	ASSERT(bp->b_rptr - bp->b_datap->db_base >= sizeof(*ti));

	if ((nbp = bp_copy(bp, todrop, msgdsize(bp) - todrop)) == NULL)
		return(0);
	ASSERT(bp != nbp);
	if (bp->b_cont)
		freemsg(bp->b_cont);
	bp->b_cont = nbp;
	bp->b_wptr = bp->b_rptr;	/* 1st block has no data */
	ti = (struct tcpiphdr *)(bp->b_rptr - sizeof(*ti));
	ti->ti_seq += todrop;
	ti->ti_len -= todrop;
	STRLOG(TCP_ID, 101, DPRI_LO, SL_TRACE,
		"tiu_adjmsg: adjusted %x by %x\n", bp, todrop);
	return(1);

}	/*  End of tiu_adjmsg()  */


#ifdef NOTDEF
/*
 * Remove all the out of band data
 * Present it to the user
 * Leave the sequence number and ti_len alone.
 */
void
tiu_pulloutofband(tdp, ti, bp)

	register struct tcpd *tdp;
	register struct tcpiphdr *ti;
	register mblk_t *bp;
{

	register mblk_t *nbp, *bp_oob = (mblk_t *)NULL;
	register int todrop, i;
	int first = 1;
	extern void tn_exdata();

	/*
	 * Skip past all the buffers
	 * that dont have any data.
	 */
	while ((bp->b_wptr == bp->b_rptr) && bp)
		bp = bp->b_cont;
Nodata:
	if (bp == (mblk_t *)NULL){
		printf("tiu_pulloutofband: Out of OOB data..\n");
		return;
	}
	todrop = ti->ti_urp;
	while (todrop > 0){
		if (!bp)
			goto Nodata;
		/*
		 * Just make a referece to
		 * the urgent data and pass it
		 * up. Drop it from
		 * the regular data
		 */
		if ((nbp = dupb(bp)) == NULL){
			/*
			 * Should I signal error so that
			 * an ACK is not sent to the sender..
			 */
			TRACE(1<<12, "tiu_pulloutofband: can't recv oob..\n",0);
			return;
		}
		i = min(todrop, (bp->b_wptr-bp->b_rptr));
		nbp->b_wptr = i + nbp->b_rptr;
		if (first) {
			first = 0;
			ti->ti_urp = i;
		}
		else
			bp->b_rptr += i;
		if (bp_oob)
			bp_oob->b_cont = nbp;
		else 
			bp_oob = nbp;
		todrop -= i;
		bp = bp->b_cont;
	}
	bp_oob->b_cont = (mblk_t *)NULL;
	/*
	 * Present it to the user.
	 */
	tn_exdata(tdp, bp_oob);
	return;

}	/*  End of tiu_pulloutofband()  */

#endif /* NOTDEF */


/*
 * Determine the value for maxseg size based on the mtu size.
 * A control packet of type M_CTL is sent to IP. IP fills in
 * the information and returns to us. Use a default value of 512
 * if anything goes wrong.
 */

void
tcp_bestmss(tp)
     struct tcpcb *tp;
{
    register mblk_t *bp;
    register struct ip_if *tpf;
    extern unchar my_hosts_class;

    STRLOG(TCP_ID, 101, DPRI_LO, SL_TRACE, "tcp_bestmss: ");
    /* tp->t_laddr = 0;					nw2 */
    /* tp->t_maxseg = min(tp->t_maxseg, 512);		cah1 */
    if ((bp = allocb(sizeof(struct ip_if), BPRI_MED))==NULL)
		return;
    tpf = (struct ip_if *)bp->b_wptr;
    bp->b_wptr += sizeof(struct ip_if);
    MTYPE(bp) = M_CTL;
    tpf->ipf_type = IPIF_NGETMYADDR;
    tpf->ipf_addr = tp->t_faddr;
    tpf->ipf_priv = (char *)tp;
    tpf->ipf_tos = tp->t_tdp->td_tos;
    tcpsend(bp);
    return;

}	/*  End of tcp_bestmss()  */
