/*
 * @(#) Copyright 1989.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)so_main.c        1.5      05:11:50 - 89/09/18 "

/*
 * Socket Library cooperating module
 *
 * Author: Mohsen Mortazavi
 * 	   The Wollongong Group
 * 	   Palo Alto, CA
 *
 * Date:   January 1989
 */

#include "sys/types.h"
#include "sys/param.h"
#include "sys/stream.h"
#include "sys/inet.h"
#include "sys/inetioctl.h"
#include "sys/socket.h"
#include "sys/in.h"
#include "sys/stropts.h"
#include "sys/tihdr.h"
#ifdef u3b2
#include "sys/psw.h"
#include "sys/pcb.h"
#include "sys/immu.h"
#include "sys/region.h"
#include "sys/proc.h"
#endif
#ifndef	XENIX
#include "sys/fs/s5dir.h"
#else
#include "sys/dir.h"
#endif
#include "sys/signal.h"
#include "sys/user.h"
#include "fcntl.h"
#include "sys/strlog.h"
#include "sys/debug.h"
#include "sys/errno.h"
#include "sys/somod.h"

void so_ioctl();

/* stream data structure definitions */
int somodopen(), somodclose(), somodrput(), sorsrv(), somodwput();

static struct module_info somod_info =
    { SOMOD_ID, "somod", 0, INFPSZ, 4096, 1024 };
static struct qinit somodrinit =
    { somodrput, sorsrv, somodopen, somodclose, NULL, &somod_info, NULL };
static struct qinit somodwinit =
    { somodwput, NULL, somodopen, somodclose, NULL, &somod_info, NULL };
struct streamtab somodinfo =
    { &somodrinit, &somodwinit, NULL, NULL };

/*
 * somodopen - open routine gets called when the
 *	       module gets pushed onto the stream.
 */
somodopen(q, dev, flag, sflag)
	register queue_t *q;
{
	register struct so_socket *so;
	register struct proc *pp;

	ASSERT(q != NULL);
	/* check if already open */
	if (q->q_ptr)
		return(1);

	/* find free data structure */
	for (so = so_socket; so < &so_socket[so_cnt]; so++)
		if (!(so->so_flags & USED))
			break;
	if (so >= &so_socket[so_cnt]) {
		u.u_error = ENOBUFS;
		return(OPENFAIL);
	}

	/* initialize data structure */
	so->so_flags = USED;
	so->so_rdq = q;
	q->q_ptr = (caddr_t)so;
	WR(q)->q_ptr = (caddr_t)so;

	/* associate a process group with the stream */
	pp = u.u_procp;

#ifdef NOTFORNOW
	if (pp->p_pid == pp->p_pgrp && u.u_ttyp == NULL) {
		u.u_ttyp = &so->so_pgrp;
		so->so_pgrp = pp->p_pgrp;
	}
#endif

	WR(q)->q_maxpsz = WR(q)->q_next->q_maxpsz;
	q->q_maxpsz = q->q_next->q_maxpsz;
	sowriteblock(WR(q));

	return(1);
}


/*
 * somodclose - This routine gets called when the module
 *              gets popped off of the stream.
 */
somodclose(q)
	register queue_t *q;
{
	register struct so_socket *so;

	ASSERT(q && q->q_ptr);
	so = (struct so_socket *)q->q_ptr;

	if (so->so_ctlbuf)
		freemsg(so->so_ctlbuf);
	if (so->so_iocsave)
		freemsg(so->so_iocsave);
	if (so->so_oob)
		so_clean(&so->so_oob);

	so->so_prim = 0;
	so->so_iocsave = NULL;
	so->so_ctlbuf = NULL;
	so->so_flags = 0;
	so->so_conq = 0;
	so->so_oob = 0;
	so->so_pgrp = 0;
}


/*
 * somodrput - Module read queue put procedure.
 *             This is called from the module or
 *	       driver downstream.
 */
somodrput(q, mp)
	register queue_t *q;
	register mblk_t *mp;
{
	register struct so_socket *so;

	ASSERT(q && q->q_ptr && mp);
	so = (struct so_socket *)q->q_ptr;

	switch(MTYPE(mp)) {

	case M_ERROR:
		so_merror(q, mp);
		return;

	case M_DATA:
		if (msgdsize(mp) == 0) {
			STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
			"somodrput: got 0 len M_DATA, dropping\n");
			freemsg(mp);
			return;
		}

		if (so->so_state & SS_CANTRCVMORE) {
			STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_TRACE,
			    "somodrput: CANTRCVMORE - dropping");
			freemsg(mp);
		} else {
			if (so->so_type == SOCK_STREAM)
				(void) so_data_ind(q, mp, DO_Q);
			else {
				STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_TRACE,
				"somodrput:freeing M_DATA on nonSOCK_STREAM");
				freemsg(mp);
			}
		}
		return;

	case M_PROTO:
	case M_PCPROTO:
		so_proto(q, mp);
		return;

	case M_IOCACK:
		so_iocack(q, mp);
		return;

	case M_IOCNAK:
		so_iocnak(q, mp);
		return;

	case M_FLUSH:
		if (so->so_type != SOCK_STREAM) 
			flushq(q, FLUSHALL);
		putnext(q, mp);
		return;
		
	case M_SETOPTS:
		putnext(q, mp);
		return;

	default:
		/* More checking for type is needed XXX */
		STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
		    "somodrput: dropping message type =%d", MTYPE(mp));
		freemsg(mp);
		return;
	}
}

/*
 * somodwput - Module write queue put procedure.
 *             This is called from the module or
 *	       stream head upstream.
 */
somodwput(q, mp)
	register queue_t *q;
	register mblk_t *mp;
{
	ASSERT(q && q->q_ptr && mp);

/* DEBUG !!! */	STRlog(SOMOD_ID, -1, DPRI_LO, SL_te,
	"somodwput: Entered T=%d",MTYPE(mp));
	switch(MTYPE(mp)) {

	case M_DATA:
/* DEBUG !!! */	STRlog(SOMOD_ID, -1, DPRI_LO, SL_te,
		"somodwput: M_DATA putnext");
		putnext(q, mp);
		return;

	case M_IOCTL:
		so_ioctl(q, mp);
		return;
	
	default:
		/* XXX */
		putnext(q, mp);
		return;
	}
}

/*
 * For SOCK_STREAM sockets passes the data upward if OOB flag is off.
 * messages on the queue are type M_DATA, T_DATA_IND, or T_EXDATA_IND.
 * 
 * For NON-SOCK_STREAM sockets this service routine will be disabled. 
 * The only message types put on this queue are T_UNITDATA_IND's which 
 * are queued by so_ud_ind routine and are removed from the queue by 
 * so_sync_addr routine.
 */
sorsrv(q)
	register queue_t *q;
{
	register struct so_socket *so;
	register mblk_t *bp, *bp1;
	register union T_primitives *Tp;

	so = (struct so_socket *)q->q_ptr;
	if ((so->so_flags & OOB) || !q->q_first)
		return 0;

	while ((bp = getq(q)) != NULL) {
		switch (MTYPE(bp)) {
		
		case M_DATA:
			goto data;
		
		case M_PROTO:
			Tp = (union T_primitives *)bp->b_rptr;
			switch (Tp->type) {
			
			case T_DATA_IND:
data:
				if (so_data_ind(q, bp, DONT_Q) < 0) {
					putbq(q, bp);
					return 0;
				}
				break;

			case T_EXDATA_IND:
				if (so_exdata_ind(q, bp, DONT_Q) < 0) {
					putbq(q, bp);
					return 0;
				}
				break;

			default:
				STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
				"sorsrv: freeing primitive type %d found on the queue",
				Tp->type);
				freemsg(bp);
			}
			break;

		default:
			STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
			"sorsrv: freeing message type %d found on the queue",
			MTYPE(bp));
			freemsg(bp);
			break;
		}	
	}
	return 0;
}

/*
 * so_ioctl - ioctls to somod, q is the write queue
 */
void
so_ioctl(q, mp)
	queue_t *q;
	mblk_t *mp;
{
	register struct so_socket *so;
	register struct iocblk *ip;
	int error;

	ASSERT((mp->b_wptr - mp->b_rptr) == sizeof(struct iocblk));
	so = (struct so_socket *)q->q_ptr;
	ip = (struct iocblk *)mp->b_rptr;

	if (so->so_flags & WAITIOCACK) {
		/*
		 * The outstanding ioctl as seen by somod has
		 * immaturely completed upon receiving a signal.
		 * clean the socket.
		 */
		STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_TRACE,
		    "so_ioctl: re-entry, cleaning sioc_cmd=%x prim=%x",
		    SIOC_CMD(so), so->so_prim);
		freemsg(so->so_iocsave);
		so->so_iocsave = NULL;
		so->so_flags &= ~WAITIOCACK;
		so->so_prim = 0;
	}

/* DEBUG !!! */	STRlog(SOMOD_ID, -1, DPRI_LO, SL_te,
	"so_ioctl: cmd=%d",ip->ioc_cmd);
	switch (ip->ioc_cmd) {

	case SIOC_SOCKET:
		error = so_create(q, mp);
		if (error)
			goto nack;
		goto ack;
			
	case SIOC_BIND:
		error = so_bind(q, mp);
		if (error) 
			goto nack;
		return;
		
	case SIOC_LISTEN:
		error = so_listen(q, mp);
		if (error)
			goto nack;
		return;

	case SIOC_CONNECT:
		error = so_connect(q, mp);
		if (error)
			goto nack;
		return;

	case SIOC_GETINFO:
		error = so_getinfo(q, mp);
		if (error)
			goto nack;
		goto ack;

	case SIOC_BINDQPTR:
		error = so_bindqptr(q, mp);
		if (error)
			goto nack;
		return;
	
	case SIOC_ACCEPT:
		error = so_accept(q, mp);
		if (error)
			goto nack;
		return;

	case SIOC_SEND_S:
	case SIOC_SEND_SF:
	case SIOC_SEND_L:
	case SIOC_SEND_LF:
		error = so_send(q, mp);
		if (error)
			goto nack;
		goto ack;

	case SIOC_SHUTDOWN:
		error = so_shutdown(q, mp);
		if (error)
			goto nack;
		goto ack;

	case SIOC_READOOB:
		error = so_readoob(RD(q), mp);
		if (error)
			goto nack;
		goto ack;

	case SIOC_PEEKOOB:
		error = so_oob(q, mp);
		if (error)
			goto nack;
		goto ack;

	case SIOC_ATMARK:
		error = so_oob(q, mp);
		if (error)
			goto nack;
		goto ack;

	case SIOC_UNBLOCK:
		sowriteunblock(q);
		goto ack;
		
	case SIOC_BLOCK:
		sowriteblock(q);
		goto ack;

	case SIOC_GETMSGSZ:
		if ((mp->b_cont == NULL) ||
		    (ip->ioc_count < sizeof(int))) {
			error = EINVAL;
			goto nack;
		}
		*(int *)mp->b_cont->b_rptr = somsgsz;	
		ip->ioc_count = sizeof(int);
		goto ack;

	case SIOC_NBIO:
		if ((mp->b_cont == NULL) ||
		    (ip->ioc_count < sizeof(int))) {
			error = EINVAL;
			goto nack;
		}
		if (so->so_flags & WAITIOCACK) {
			error = EINPROGRESS;
			goto nack;	
		}
		if (*(int *)mp->b_cont->b_rptr & O_NDELAY) 
			so->so_state |= SS_NBIO;
		else
			so->so_state &= ~SS_NBIO;
		ip->ioc_count = 0;
		goto ack;	
		
	/*
	 * SOCK_DGRAM specific ioctls
	 */
	case SIOC_SENDTO:
	case SIOC_SENDTO2:
		error = so_sendto(q, mp);
		if (error)
			goto nack;
		return;

	case SIOC_RECVFROM:
		error = so_recvfrom(q, mp);
		if (error)
			goto nack;
		return;

	default:
		putnext(q, mp);
		return;

	}

ack:
	MTYPE(mp) = M_IOCACK;
	ip->ioc_error = 0;
	ip->ioc_rval = 0;
	/*
	 * clear these in case the ioctl went as  
	 * far as setting them and then failed.
	 */
	so->so_iocsave = NULL;
	so->so_flags &= ~WAITIOCACK;
	qreply(q, mp);
	return;

nack:
	MTYPE(mp) = M_IOCACK;
	ip->ioc_error = 0;
	ip->ioc_count = 0;
	ip->ioc_rval = error;
	/*
	 * clear these in case the ioctl went as  
	 * far as setting them and then failed.
	 */
	so->so_iocsave = NULL;
	so->so_flags &= ~WAITIOCACK;
	qreply(q, mp);
	return;
}
