/*
 * @(#) Copyright 1986.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)ip_twg.c (TWG)  1.2     89/07/10 "

#include "sys/param.h"
#include "sys/types.h"
#ifndef XENIX
#include "sys/inline.h"
#endif
#include "sys/stream.h"
#include "sys/debug.h"
#ifdef XENIX
#include "sys/assert.h"
#endif
#include "sys/strlog.h"
#include "sys/errno.h"
#include "sys/tiuser.h"

#include "sys/inet.h"
#include "sys/socket.h"
#include "sys/if.h"
#include "sys/in.h"
#include "sys/in_var.h"
#include "sys/ip.h"
#include "sys/ip_var.h"
#include "sys/inetioctl.h"
#include "sys/route.h"

extern struct ipstat ipstat;
extern char ipopt_out;

/***********************
 *  Table of Contents  *
 **********************/

int		ip_security_init();
mblk_t	 	*add_ipopt_security();
int		check_ip_security();
unchar 		*check_opt();
int		delete_opt();
int		find_opt_len();
int		check_len();
int		ip_ctloutput();
int		ip_install_route();

/***** End of TOC *****/

/*
 * All of the code added for the Blacker support was written with the
 * understanding that the host was running dedicated, system high.
 * If/when we get to a Multi-Level Secure (MLS) host, everything changes.
 * Security will be on or off depending on the VALUE of my_hosts_class
 * set in the master.d file.
 */
extern u_char my_hosts_class, my_hosts_authority;
static struct ipoption_security _my_ip_security;
static int _ip_security_init = 1;

ip_security_init()
{

	/*
	 * Check to see if we need security
	 */
	if( my_hosts_class == 0 ) {
		_ip_security_init = 0;
		return;
	}

	/*
	 * set up my IP Security Option.
	 * NOTE that the authority field is not checked.
	 */
	_my_ip_security.ipsec_type = IPOPT_SECURITY;
	_my_ip_security.ipsec_length = IPOPT_SECUR_LENGTH;
	_my_ip_security.ipsec_class = my_hosts_class;

	/*
	 * if/when the authority field gets bigger 
	 * bcopy(my_hosts_authority, _my_ip_security.ipsec_authority,
	 *	IPOPT_SECUR_LENGTH - 3);
	 */

	/* For now make sure the authority field has a chance. */
	_my_ip_security.ipsec_authority[0] = (my_hosts_authority &
		IPOPT_SECUR_MASK);
	_ip_security_init = 0;

}	/*  End of ip_security_init()  */


mblk_t *
add_ipopt_security(bp)
	register mblk_t *bp;
{
	register struct ip *ip = mtod(bp, struct ip *);
	register mblk_t *bpn;
	int optlen, iphlen;
	extern unchar my_hosts_class, my_hosts_authority;

	if ( _ip_security_init )
		ip_security_init();

	/*
	 ********************  NOTE  *********************
	 * If IPOPT_SECUR_LENGTH is NOT a multiple of 4 bytes in length,
	 * then it may become necessary to add or delete IPOPT_EOLs.
	 * If there are no other IP options, then (IPOPT_SECUR_LENGTH % 4)
	 * IPOPT_EOLs must be inserted after the Security option.
	 * If there are other IP options present, we could kludge by
	 * appending (IPOPT_SECUR_LENGTH % 4) IPOPT_NOPs after the security
	 * option OR we must copy those options necessary (IPOPT_COPIED()),
	 * then pad with IPOPT_EOLs as required.
	 */
	optlen = IPOPT_SECUR_LENGTH;
	ip = mtod(bp, struct ip *);
	/* 
	 * if there is not room in the buffer for the security option or
	 * if the ip header lenght > 5
	 * or data is contiguous
	 * get a new buffer to insert option
	 */
	if (((bp->b_datap->db_lim - bp->b_wptr) < optlen) ||
	    (ip->ip_hl > 5) || (BLEN(bp) > IPHLEN)) {
		if ((bpn = allocb(BLEN(bp) + optlen, BPRI_LO)) == NULL) {
			/*
			 * if we can't get the stream buf, 
			 * no need to send datagram
			 */
			freemsg( bp );
			return (bpn);
		}
		/*
		 * Copy the 20 byte ip header
		 * append the security header
		 * append the rest of the buffer
		 */
		bcopy((caddr_t)bp->b_rptr,(caddr_t)bpn->b_wptr,IPHLEN);
		ip = mtod(bpn, struct ip *);
		bcopy((caddr_t)&_my_ip_security, (caddr_t)(ip + 1), optlen);
		bpn->b_wptr += optlen + IPHLEN;
		bcopy((unchar *)(bp->b_rptr + IPHLEN),(unchar *)bpn->b_wptr,
			BLEN( bp ) - IPHLEN);
		bpn->b_wptr += BLEN( bp ) - IPHLEN;
		bpn->b_cont = bp->b_cont;
		bp->b_cont = 0;
		freemsg( bp );
		bp = bpn;
	} else {
		ip = mtod(bp, struct ip *);
		bcopy((caddr_t)&_my_ip_security, (caddr_t)(ip + 1), optlen);
	}
	ip->ip_len += optlen;
	iphlen = optlen + (ip->ip_hl << 2);
	if ((iphlen < IPHLEN) || (iphlen > 60)) {
		freemsg( bp );
		bp = NULL;
	}
	ip->ip_hl += (optlen >> 2);
	return (bp);

}	/*  End of add_ipopt_security()  */


/*
 * Since _my_ip_security is a static, check_ip_security must be in this file.
 */
check_ip_security(ip_sec)
	register struct ipoption_security *ip_sec;
{
	if ( _ip_security_init )
		ip_security_init();

	if( my_hosts_class )
		return( bcmp((caddr_t)&_my_ip_security, (caddr_t) ip_sec,
			IPOPT_SECUR_LENGTH) );
	else {
		if( ip_sec->ipsec_length == IPOPT_SECUR_LENGTH ) {
			if( ip_sec->ipsec_class == IPOPT_SECUR_UNCLASS )
				return( 0 );
		}
	return( 1 );
	}

}	/*  End of check_ip_security()  */

/*
 * IP socket option processing.
 *
 * If setting options bp will contain value of new options.
 * If getting options bp will be filled with return data which will
 * ultimately be sent back to the user.
 *
 * Bp's continuation block points to the options buffer; it will be
 * NULL if no options are currently set.
 */
int
ip_ctloutput(op, level, optname, bp)
	int op;			/* operation - set or get */
	int level;		/* protocol level */
	int optname;		/* option name */
	register mblk_t *bp;
{
	register mblk_t *obp = bp->b_cont; /* block containing options */
	register int inc, inc1, length, error = EINVAL;
	struct ip_timestamp ipt;
	struct ip_satid satid;

	STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
		"ip_ctloutput: entered - op=%d, optname=%d", op, optname);

	if (level != IPPROTO_IP)
		;
	else switch (op) {

	case T_NEGOTIATE:	/* set options */
		if (optname != IP_OPTIONS) {
			if (BLEN(bp) == 0)
				if (obp && BLEN(obp)) {
					error = delete_opt(obp, optname);
					break;
				}

			if (obp)
				if (check_opt(obp->b_rptr, optname) != 0)
					(void)delete_opt(obp, optname);
		}

		switch (optname) {

		case IP_OPTIONS:
		{
			/*
			 * Allow the user to set all options, except
			 * security, in one go.
			 */
			int len = BLEN(bp);

			/* delete all options if none sent to us */
			if (len == 0) {
				if (obp) {
					bzero(obp->b_rptr, MAX_IPOPTLEN);
					obp->b_wptr = obp->b_rptr;
				}
				error = 0;
				break;
			}

			/* ensure sent options not too big */
			if (len > IPOPT_MINOFF)
				if (error = check_len(len))
					break;

			/* don't let user set security options */
			if (check_opt(bp->b_rptr, IPOPT_SECURITY)
				 || check_opt(bp->b_rptr, IPOPT_EXTSECURITY))
				break;

			if (obp == NULL) {
				if ((obp = allocb(MAX_IPOPTLEN, BPRI_LO))
								== NULL) {
					error = ENOBUFS;
					break;
				}
			}

			bzero(obp->b_rptr, MAX_IPOPTLEN);
			bcopy(bp->b_rptr, obp->b_rptr, len);
			obp->b_wptr = obp->b_rptr + len;

			error = 0;
			break;
		}

		case IPOPT_RR:
		{
			short *size;

			if (BLEN(bp) != sizeof(short))
				break;

			size = (short *)bp->b_rptr; /* # of routes to record */

			length = (*size * sizeof(struct in_addr))
							+ IPOPT_MINOFF - 1;

			if (length <= 1)
				break;

			if (error = check_len(length))
				break;

			if (obp == NULL) {
				if ((obp = allocb(MAX_IPOPTLEN, BPRI_LO))
								== NULL) {
					error = ENOBUFS;
					break;
				}
				bzero(obp->b_wptr, MAX_IPOPTLEN);
			} else {
				int len;
				len = find_opt_len(obp);
				if (error = check_len(length + len)) {
					break;
				}
				bzero(obp->b_wptr, length);
				obp->b_wptr = obp->b_rptr + len;
			}

			obp->b_wptr[IPOPT_OPTVAL] = IPOPT_RR;
			obp->b_wptr[IPOPT_OLEN]   = length;
			obp->b_wptr[IPOPT_OFFSET] = IPOPT_MINOFF;
			obp->b_wptr += length;
			error = 0;
			break;
		}

		case IPOPT_TS:
		{
			unchar *mpdata;

			if (BLEN(bp) < 2)
				break;

			ipt.ipt_flg = *(char *)bp->b_rptr & 0x0f;
			inc = length = *(short *)bp->b_rptr & 0x00ff;

			if (!length)
				break;

			switch (ipt.ipt_flg) {

			case IPOPT_TS_TSONLY:
				length = length * sizeof(struct in_addr);
				break;

			case IPOPT_TS_PRESPEC:
			case IPOPT_TS_TSANDADDR:
				length = length *
				   (sizeof(struct ip_timestamp) - IPOPT_MINOFF);
				break;

			default:
				length = 41;  /* invalid to cause error/exit */

			}

			length += IPOPT_MINOFF;

			if (error = check_len(length))
				break;

			if (obp == NULL) {
				if ((obp = allocb(MAX_IPOPTLEN, BPRI_LO))
								== NULL) {
					error = ENOBUFS;
					break;
				}
				bzero(obp->b_wptr, MAX_IPOPTLEN);
			} else {
				int len;

				len = find_opt_len(obp);

				if (error = check_len(length + len))
					break;

				obp->b_wptr = obp->b_rptr + len;
				bzero(obp->b_wptr, length);
			}

			ipt.ipt_code = IPOPT_TS;
			ipt.ipt_len = length;
			ipt.ipt_ptr = 5;
			bcopy((unchar *)&ipt, obp->b_wptr, IPOPT_MINOFF);
			mpdata = obp->b_wptr + IPOPT_MINOFF;

			if (ipt.ipt_flg == IPOPT_TS_PRESPEC) {
				bp->b_rptr += sizeof(short);
				inc *= sizeof(struct in_addr);
				for (inc1 = 0; inc1 < inc;) {
					bcopy(&bp->b_rptr[inc1], mpdata,
						sizeof(struct in_addr));
					mpdata += 2*sizeof(struct in_addr);
					inc1 += sizeof(struct in_addr);
				}
			}

			obp->b_wptr += length;
			error = 0;
			break;
		}

		case IPOPT_SATID:
		{
			/* use value sent by user */
			ushort stream_id;

			if (BLEN(bp) != sizeof(ushort))
				break;

			stream_id = *(ushort *)bp->b_rptr;
			length = sizeof(struct ip_satid);

			if (obp == NULL) {
				if ((obp = allocb(MAX_IPOPTLEN, BPRI_LO))
								== NULL) {
					error = ENOBUFS;
					break;
				}
				bzero(obp->b_wptr, MAX_IPOPTLEN);
			} else {
				int len;
				len = find_opt_len(obp);
				if (error = check_len(length + len))
					break;
				obp->b_wptr = obp->b_rptr + len;
			}

			satid.ipsat_code = (unchar)IPOPT_SATID;
			satid.ipsat_len = (unchar)length;
			satid.ipsat_type = stream_id;
			bcopy((caddr_t)&satid, (caddr_t)obp->b_wptr, length);
			obp->b_wptr += length;
			error = 0;
			break;
		}

		case IPOPT_SSRR:
		case IPOPT_LSRR:
		{
			int size = BLEN(bp);

			if (size % sizeof(struct in_addr) != 0
					|| size < sizeof(struct in_addr))
				break;

			length = size + IPOPT_MINOFF - 1;

			if (error = check_len(length))
				break;

			if (obp == NULL ) {
				if ((obp = allocb(MAX_IPOPTLEN, BPRI_LO))
								== NULL) {
					error = ENOBUFS;
					break;
				}
				bzero(obp->b_wptr, MAX_IPOPTLEN);
			} else {
				int len = find_opt_len(obp);
				if (error = check_len(length + len))
					break;
				obp->b_wptr = obp->b_rptr + len;
				bzero(obp->b_wptr, length); /* this needed ? */
			}

			/* use the ipt structure, same first 3 bytes */
			ipt.ipt_code = optname;
			ipt.ipt_len = length ;
			ipt.ipt_ptr = IPOPT_MINOFF;

			/* copy first 3 bytes of ipt structure to options buf */
			bcopy((caddr_t)&ipt, obp->b_wptr, IPOPT_MINOFF - 1);

			obp->b_wptr += IPOPT_MINOFF - 1;

			/* copy source route list to options buffer */
			bcopy(bp->b_rptr, obp->b_wptr, size);

			obp->b_wptr += size;
			error = 0;
			break;
		}

		default:
			STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
				"ip_ctloutput: HIT DEFAULT:SETOPT");
			break;
		}
		break;

	case T_DEFAULT:		/* get options */
		switch (optname) {

		case IP_OPTIONS:
			error = ENOMSG;
			if (obp) {
			    /*
			     * XXXX - We assume obp->b_rptr
			     * always points to start of options.
			     * But if not we'll need to do:
			     * obp->b_rptr = obp->b_datap->db_base;
			     */

			    /* check user has sent enough space */
			    if (BLEN(bp) >= MAX_IPOPTLEN) {
				int optlen;
				extern char ipopt_out; /* see master.d/ip */

				if (ipopt_out)
					optlen = MAX_IPOPTLEN;
				else
					optlen = BLEN(obp);

				bcopy(obp->b_rptr, bp->b_rptr, optlen);
				bp->b_wptr = bp->b_rptr + optlen;
				error = 0;
			    }
			}
			break;

		case IPOPT_RR:
		case IPOPT_TS:
		case IPOPT_LSRR:
		case IPOPT_SATID:
		case IPOPT_SSRR:
			error = ENOMSG;
			if (obp) {
				unchar *mpdata;
				mpdata = check_opt(obp->b_rptr, optname);

				if (BLEN(bp) < mpdata[IPOPT_OLEN]) {
					/* user hasn't sent us enough space */
					error = EINVAL;
					break;
				}

				if (mpdata == 0)
					break;

				bcopy(mpdata, bp->b_rptr, mpdata[IPOPT_OLEN]);
				bp->b_wptr = bp->b_rptr + mpdata[IPOPT_OLEN];
				error = 0;
			}
			break;

		default:
			STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
				"ip_ctloutput: HIT DEFAULT:GETOPT");
			break;
		}

		break;
	}

	/*
	 * When setting options pad end of option list with EOL if necessary.
	 */
	if ((op == T_NEGOTIATE) && obp && (inc = (BLEN(obp) % 4)))
		for (; inc < 4 ; inc++) {
			*obp->b_wptr = IPOPT_EOL;
			obp->b_wptr++;
		}

	if (error) {
		STRlog(IP_ID, -1, DPRI_HI, SL_te,
			"ip_ctlouput: returning error 0x%x", error);
	}

	if (op == T_NEGOTIATE)
		bp->b_cont = obp;
	else
		bp->b_cont = NULL;

	return error;

}	/*  End of ip_ctloutput() */

/*
 * Check options buffer to make sure no duplicate 
 * option requests are inserted. If there is a specific
 * already in the buffer return a pointer to it.
 */
unchar *
check_opt (optptr, op)
	register unchar *optptr;
	register int op;
{
	register int inc, len = MAX_IPOPTLEN;
	register unchar *retval = 0;

	for(inc = 0; inc <= len; ) {
		if( optptr[IPOPT_OPTVAL] == IPOPT_EOL )
			break;
		if( optptr[IPOPT_OPTVAL] == IPOPT_NOP ) {
			inc++;
			optptr++;
			continue;
		}
		if( optptr[IPOPT_OPTVAL] == op ) {
			retval = optptr;
			break;
		}
		inc += optptr[IPOPT_OLEN];
		optptr += optptr[IPOPT_OLEN];
		if (optptr[IPOPT_OLEN] < IPOPT_MINOFF)
			break;
	}

	return( retval );

}	/*  End of check_opt()  */


/*
 *  Delete option:
 *  Find it and then compact the stream buf, deleting if last one 
 *  Return EINVAL if error.
 */
int
delete_opt( bp, op )
	mblk_t *bp;
	register int op;
{
	register unchar *optptr;
	register unchar *mpdata;
	register int length1, length;
	register int total = BLEN(bp);

	if (total == 0)
		return EINVAL;	/* no options to delete */

	optptr = check_opt( bp->b_rptr, op );

	if( optptr ) {
		length = optptr[IPOPT_OLEN];

		if( length == total ) {
			bp->b_wptr = bp->b_rptr;
			return 0; /* no options left in buffer */
		}

		mpdata = mtod( bp, unchar * );

		/* if the option is first */
		if( optptr == mpdata )
			bcopy(optptr + length, mpdata, total - length);
		else {
			length1 = (optptr + length) - mpdata;
			if( total != length1 )
				bcopy(optptr + length,optptr,(total - length1));
		}

		bp->b_wptr -= length;

		/*
		 * Check for IPOPT_NOP or IPOPT_EOL and no other options.
		 * Zap them if that's all we have left.
		 */
		if(BLEN(bp) < IPOPT_MINOFF || mpdata[IPOPT_OPTVAL]==IPOPT_EOL)
			bp->b_wptr = bp->b_rptr;

		return(0);
	} else
		return( EINVAL ); /* tried to delete non-existant option */

}	/*  End of delete_opt()  */


/*
 * Find true length of options
 */
int
find_opt_len( bp )
	register mblk_t *bp;
{
	register unchar value = 0;
	register unchar *optptr = mtod( bp, unchar * );

	for (;;) {
		if( value >= BLEN( bp ) )
			break;
		if( optptr[IPOPT_OPTVAL] == IPOPT_EOL )
			break;
		if (optptr[IPOPT_OPTVAL] == IPOPT_NOP) {
			value++;
			optptr++;
		} else {
			if( optptr[IPOPT_OLEN] >= IPOPT_MINOFF ) {
				value += optptr[IPOPT_OLEN];
				optptr += optptr[IPOPT_OLEN];
			}
		}
	}

	return( value );

}	/*  End of find_opt_len()  */


/*
 * Check that a new option will fit
 * my_hosts_class determines if security is on
 */
int
check_len( length )
	register int length;
{
	register int error = 0;

	if( my_hosts_class == 0 ) {
		if(length > MAX_IPOPTLEN || length <= IPOPT_MINOFF)
			error = EINVAL;
	} else {
		if((length + IPOPT_SECUR_LENGTH)>MAX_IPOPTLEN ||
		   (length + IPOPT_SECUR_LENGTH)<=IPOPT_MINOFF)
			error = EINVAL;
	}
	return( error );

}	/*  End of check_len()  */

/* 
 * Install received route from input packet for outgoing packet.
 * bopt = existing options.
 * bp = output from ip_srcroute call.
 * Format from ip_srcroute:
 *     dst addr - 4 bytes  not in use at moment
 *  b_rptr starts here!!
 *     nop      - 1 byte
 *     header   - 3 bytes ( opcode, offset, length )
 *     addr     - 4 bytes * number of hops
 *     nhops    - 1 byte  ( located in ip_input )
 */
int
ip_install_route( bp, bopt )
	register mblk_t *bp;
	register mblk_t *bopt;
{
	register unchar *mpdata;
	register int length = 0;

	TRACE(1<<10,"ip_install_route-entered\n","")
	(void)delete_opt( bopt, IPOPT_LSRR );
	(void)delete_opt( bopt, IPOPT_SSRR );

	if( BLEN( bopt ) ) {
		length = find_opt_len( bopt );
		if( check_len( length + BLEN(bopt)))
			return;
	}

	mpdata = mtod( bopt, unchar * );

	if( length )
		mpdata += length;

	/* don't copy the NOP */
	bcopy((caddr_t)(bp->b_rptr + 1),(caddr_t)mpdata, BLEN(bp) - 1);
	bopt->b_wptr += (BLEN( bp ) - 1);

	if( length = (BLEN( bopt ) % 4) ) {
		for( ; length < 4 ; length++ ) {
			*bopt->b_wptr = IPOPT_EOL;
			bopt->b_wptr++;
		}
	}

	mpdata[IPOPT_OFFSET] = IPOPT_MINOFF;
	TRACE(1<<10,"ip_install_route-exit\n","")
	return;

}	/*  End of ip_install_route()  */
