/*
 * @(#) Copyright 1989.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)ip_route.c (TWG)  1.5     89/07/31 "

/*
 * Copyright (c) 1980, 1986 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of California at Berkeley. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 *	@(#)route.c	7.3 (Berkeley) 12/30/87
 */

#include "sys/param.h"
#include "sys/types.h"
#ifndef XENIX
#include "sys/inline.h"
#endif
#include "sys/stream.h"
#include "sys/strlog.h"

#include "sys/inet.h"
#include "sys/socket.h"
#include "sys/if.h"
#include "sys/in.h"
#include "sys/in_var.h"
#include "sys/ip.h"
#include "sys/ip_var.h"
#include "sys/inetioctl.h"
#include "sys/route.h"
#include "sys/errno.h"

struct	in_addr wildcard;	/* zero valued cookie for wildcard searches */
int	rthashsize = RTHASHSIZ;	/* for netstat, etc. */
time_t	rtlastchange;		/* time of last route change */
extern	time_t	time;		/* UNIX System clock */

/* private data structures */
static	struct rtentry *rtnet[RTHASHSIZ];
static	struct rtentry *rthost[RTHASHSIZ];
static	struct rtentry *rtfreelist;

#ifdef notdef
/*
 * Hash function for 64 entry hash table.  (update math if table size changes)
 *
 * This hashing function takes the next to lower three bits of the network
 * address and the lower three bits of the host address.  It will help to
 * spread out host/network hash lookups.  Please replace if you find a better
 * faster lookup function for a combined table.
 */
#define RTHASHMOD(x)	rthashmod(x)
rthashmod(in)
	struct in_addr in;
{
	register u_long i = ntohl(in.s_addr);

	if (IN_CLASSA(i)) {
		return (((i >> IN_CLASSA_NSHIFT) & 070) | (i & 07));
	} else if (IN_CLASSB(i)) {
		return (((i >> IN_CLASSB_NSHIFT) & 070) | (i & 07));
	} else if (IN_CLASSC(i)) {
		return (((i >> IN_CLASSC_NSHIFT) & 070) | (i & 07));
	} else
		return (i & 077);
}
#endif

/*
 * Packet routing routines.
 *
 * Route to destination with type-of-service requested (can be wildcard).
 */
rtlookup(rd, n, tos)
	register struct route_dst *rd;
	int n;
	unsigned char tos;
{
	register struct rtentry *rt;
	register unsigned long *dst, dstnetof, hash;
	struct in_addr rdst;
	int doinghost, count = 0;
	struct rtentry **table;

	rdst = rd->rtd_dst;
	dstnetof = in_netof(rdst);
	hash = ntohl(rdst.s_addr);
	dst = &rdst.s_addr, table = rthost, doinghost = 1;
	STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
		"rtlookup: looking for dst %x, tos %x", hash, tos);
again:
	for (rt = table[RTHASHMOD(hash)]; rt; rt = rt->rt_next) {
		if (rt->rt_hash != hash)
			continue;
		if ((rt->rt_flags & RTF_UP) == 0 ||
		    (rt->rt_ifp->if_flags & IFF_UP) == 0)
			continue;
		if (doinghost) {
			if (rt->rt_dst.s_addr != *dst)
				continue;
		} else {
			if (rt->rt_net != dstnetof)
				continue;
		}
		if ((tos != -1) && (tos != rt->rt_tos))
			continue;
		if (dst == &wildcard.s_addr)
			++rtstat.rts_wildcard;
		++rt->rt_use;
		if (rt->rt_flags & RTF_GATEWAY)
			rd->rtd_dst = rt->rt_gateway;
		else
			rd->rtd_dst = rdst;
		rd->rtd_flags = rt->rt_flags;
		rd->rtd_ifp = rt->rt_ifp;
		STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
			"rtlookup: found ifp %x, flags %x, gateway %x",
			rd->rtd_ifp, rd->rtd_flags, ntohl(rd->rtd_dst));
		++rd;
		if (++count < n)
			continue;
		return (count);
	}

	/*
	 * Search gateway routes next
	 */
	if (doinghost) {
		doinghost = 0;

		hash = in_netof(rdst);
		if (hash)
			while ((hash & 0xff) == 0)
				hash >>= 8;
		table = rtnet;
		goto again;
	}

	/*
	 * Check for wildcard gateway, by convention network 0.
	 */
	if (dst != &wildcard.s_addr) {
		dst = &wildcard.s_addr;
		hash = dstnetof = 0;
		goto again;
	}

	STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE, "rtlookup: failed");
	rtstat.rts_unreach++;
	return (0);
}

static struct rtentry *
rtfind(dst, tos)
	register struct sockaddr_in *dst;
	unsigned char tos;
{
	register struct rtentry *rt;
	u_long hash, dstnetof;
	int doinghost;
	struct rtentry **table;

	hash = ntohl(dst->sin_addr.s_addr), table = rthost, doinghost = 1;
	dstnetof = in_netof(dst->sin_addr);
	STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
		"rtfind: looking for dst %x, tos %x", hash, tos);
again:
	for (rt = table[RTHASHMOD(hash)]; rt; rt = rt->rt_next) {
		if (rt->rt_hash != hash)
			continue;
		if ((rt->rt_flags & RTF_UP) == 0 ||
		    (rt->rt_ifp->if_flags & IFF_UP) == 0)
			continue;
		if (doinghost) {
			if (rt->rt_dst.s_addr != dst->sin_addr.s_addr)
				continue;
		} else {
			if (rt->rt_net != dstnetof)
				continue;
		}
		if (rt->rt_tos != tos)
			continue;
		return (rt);
	}

	/*
	 * Search gateway routes next
	 */
	if (doinghost) {
		doinghost = 0;

		hash = in_netof(dst->sin_addr);
		if (hash)
			while ((hash & 0xff) == 0)
				hash >>= 8;
		table = rtnet;
		goto again;
	}

	return (0);
}

rtfree(rt)
	register struct rtentry *rt;
{

	if (rt == 0)
		panic("rtfree");
	if ((rt->rt_flags & RTF_UP) == 0) {
		rtlastchange = time;
		rt->rt_flags = 0;
		rt->rt_next = rtfreelist;
		rtfreelist = rt;
	}
}

rtflush()
{
	register struct rtentry *rt, *t;
	register u_long hash;
	int doinghost, s;
	struct rtentry **table;

	table = rthost, doinghost = 1;
	s = splni();
again:
	for (hash = 0; hash < RTHASHSIZ; ++hash) {
	    rt = table[hash];
	    while (rt != NULL) {
		t = rt->rt_next;
		rt->rt_flags = 0;
		rt->rt_next = rtfreelist;
		rtfreelist = rt;
		rt = t;
	    }
	    table[hash] = NULL;
	}
	if (doinghost) {
		doinghost = 0;
		table = rtnet;
		goto again;
	}
	(void) splx(s);
}

/*
 * Called by IP to initialize routing code
 */
ip_rtinit_list()
{
	register struct rtentry *rt;
	register int i;
	extern struct rtentry route[];
	extern int route_cnt;

	for (i = 0; i < route_cnt; ++i)
	{
		rt = &route[i];
		bzero((char *)rt, sizeof(struct rtentry));
		rt->rt_next = rtfreelist;
		rtfreelist = rt;
	}
}

/*
 * Force a routing table entry to the specified
 * destination to go through the given gateway.
 * Normally called as a result of a routing redirect
 * message from the network layer.
 *
 * N.B.: must be called at splnet or higher
 */
rtredirect(dst, gateway, flags, src, tos)
	struct sockaddr *dst, *gateway, *src;
	int flags;
	unsigned char tos;
{
	register struct rtentry *rt;
	int s;

	/* verify the gateway is directly reachable */
	if (ifa_ifwithnet(gateway) == 0) {
		rtstat.rts_badredirect++;
		return;
	}
	s = splstr();
	rt = rtfind((struct sockaddr_in *)dst, tos);
#define	equal(a1, a2) \
	(bcmp((caddr_t)(a1), (caddr_t)(a2), sizeof(struct in_addr)) == 0)
	/*
	 * If the redirect isn't from our current router for this dst,
	 * it's either old or wrong.  If it redirects us to ourselves,
	 * we have a routing loop, perhaps as a result of an interface
	 * going down recently.
	 */
	if ((rt && !equal(&((struct sockaddr_in *)src)->sin_addr,
			  &rt->rt_gateway)) || ifa_ifwithaddr(gateway)) {
		rtstat.rts_badredirect++;
		(void) splx(s);
		return;
	}

	/*
	 * Create a new entry if we just got back a wildcard entry
	 * or the the lookup failed.  This is necessary for hosts
	 * which use routing redirects generated by smart gateways
	 * to dynamically build the routing tables.
	 */
	if (rt == 0) {
		ip_rtinit(dst, gateway, (int)IPIOC_ADDROUTE,
		    (flags & RTF_HOST) | RTF_GATEWAY | RTF_DYNAMIC,
		    RTR_ICMP, tos);
		rtstat.rts_dynamic++;
		(void) splx(s);
		return;
	}

	/*
	 * Don't listen to the redirect if it's
	 * for a route to an interface. 
	 */
	if (rt->rt_flags & RTF_GATEWAY) {
		if (((rt->rt_flags & RTF_HOST) == 0) && (flags & RTF_HOST)) {
			/*
			 * Changing from route to net => route to host.
			 * Create new route, rather than smashing route to net.
			 */
			ip_rtinit(dst, gateway, (int)IPIOC_ADDROUTE,
			    flags | RTF_DYNAMIC, RTR_ICMP, tos);
			rtstat.rts_dynamic++;
		} else {
			/*
			 * Smash the current notion of the gateway to
			 * this destination.
			 */
			rt->rt_gateway =
			    ((struct sockaddr_in *)gateway)->sin_addr;
			rt->rt_flags |= RTF_MODIFIED;
			rt->rt_time = time;
			rtlastchange = time;
			rtstat.rts_newgateway++;
		}
	} else
		rtstat.rts_badredirect++;
	(void) splx(s);
	return;
}

/*
 * Routing table ioctl interface.
 */
rtioctl(cmd, data)
	int cmd;
	caddr_t data;
{

	if (cmd != IPIOC_ADDROUTE && cmd != IPIOC_DELETEROUTE)
		return (EINVAL);
	return (rtrequest(cmd, (struct rtentry *)data));
}

static struct sockaddr rtso = { AF_INET, };

/*
 * Carry out a request to change the routing table.  Called by
 * interfaces at boot time to make their ``local routes'' known,
 * for ioctl's, and as the result of routing redirects.
 */
rtrequest(req, entry)
	int req;
	register struct rtentry *entry;
{
	register struct rtentry **rprev;
	struct rtentry **rfirst;
	register struct rtentry *rt;
	int s, error = 0;
	u_long hash;
	struct ifaddr *ifa;
	struct ifaddr *ifa_ifwithdstaddr();

	STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
	       "rtreq: req=%x gate=%x, dest=%x", req,
	       ntohl(entry->rt_gateway.s_addr), ntohl(entry->rt_dst.s_addr));

	if (entry->rt_flags & RTF_HOST) {
		hash = ntohl(entry->rt_dst.s_addr);
		rprev = &rthost[RTHASHMOD(hash)];
	} else {
		hash = in_netof(entry->rt_dst);
		if (hash)
			while ((hash & 0xff) == 0)
				hash >>= 8;
		rprev = &rtnet[RTHASHMOD(hash)];
	}
	s = splni();
	for (rfirst = rprev; rt = *rprev; rprev = &rt->rt_next) {
		if (rt->rt_hash != hash)
			continue;
		if (rt->rt_tos != entry->rt_tos)
			continue;
		if (entry->rt_flags & RTF_HOST) {
			STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
			    "rtreq: comparing rt_dst=%x with entry=%x",
			    ntohl(rt->rt_dst.s_addr),
			    ntohl(entry->rt_dst.s_addr));
			if (!equal(&rt->rt_dst, &entry->rt_dst))
				continue;
		} else {
			STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
			    "rtreq: comparing in_netof rt=%x with entry=%x",
			    in_netof(rt->rt_dst), in_netof(entry->rt_dst));
			if (rtnetmatch(&rt->rt_dst, &entry->rt_dst) == 0)
				continue;
		}
		STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
		    "rtreq: comparing rt->rt_gateway=%x with gate=%x",
		    ntohl(rt->rt_gateway.s_addr),
		    ntohl(entry->rt_gateway.s_addr));
		if (equal(&rt->rt_gateway, &entry->rt_gateway))
			break;
	}
	STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
		"rtreq: after the loop rt=%x", rt);
	switch (req) {

	case IPIOC_DELETEROUTE:
		STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
			"rtreq: deleting rt=%x", rt);
		if (rt == 0) {
			error = ESRCH;
			goto bad;
		}
		*rprev = rt->rt_next;
		rt->rt_flags = 0;
		rt->rt_next = rtfreelist;
		rtfreelist = rt;
		rtlastchange = time;
		break;

	case IPIOC_ADDROUTE:
		if (rt) {
			error = EEXIST;
			goto bad;
		}
		if ((entry->rt_flags & RTF_GATEWAY) == 0) {
			/*
			 * If we are adding a route to an interface,
			 * and the interface is a pt to pt link
			 * we should search for the destination
			 * as our clue to the interface.  Otherwise
			 * we can use the local address.
			 */
			ifa = 0;
			if (entry->rt_flags & RTF_HOST) 
			{
				satosaddr(rtso) = entry->rt_dst.s_addr;
				ifa = ifa_ifwithdstaddr(&rtso);
			}
			if (ifa == 0)
			{
				satosaddr(rtso) = entry->rt_gateway.s_addr;
				ifa = ifa_ifwithaddr(&rtso);
			}
		} else {
			/*
			 * If we are adding a route to a remote net
			 * or host, the gateway may still be on the
			 * other end of a pt to pt link.
			 */
			satosaddr(rtso) = entry->rt_gateway.s_addr;
			ifa = ifa_ifwithdstaddr(&rtso);
		}
		if (ifa == 0) {
			satosaddr(rtso) = entry->rt_gateway.s_addr;
			ifa = ifa_ifwithnet(&rtso);
			if (ifa == 0) {
				error = ENETUNREACH;
				goto bad;
			}
		}
		rt = rtfreelist;
		if (rt == 0) {
			error = ENOBUFS;
			goto bad;
		}
		rtfreelist = rt->rt_next;
		rt->rt_next = *rfirst;
		*rfirst = rt;
		rt->rt_hash = hash;
		rt->rt_dst = entry->rt_dst;
		rt->rt_net = in_netof(rt->rt_dst);
		rt->rt_gateway = entry->rt_gateway;
		rt->rt_flags = RTF_UP | RTF_INUSE |
		    (entry->rt_flags & (RTF_HOST|RTF_GATEWAY|RTF_DYNAMIC));
		rt->rt_use = 0;
		rt->rt_priority = 0;		/* XXX */
		rt->rt_ifp = ifa->ifa_ifp;
		rt->rt_time = time;
		rt->rt_tos = entry->rt_tos;
		rt->rt_protocol = entry->rt_protocol;
		STRLOG(IP_ID, IPBCHAN(ifa->ifa_ifp), DPRI_LO, SL_TRACE,
			"rtreq: added %x %x", ntohl(rt->rt_dst.s_addr),
			ntohl(rt->rt_gateway.s_addr));
		break;
	}
bad:
	(void) splx(s);
	return (error);
}

/*
 * Set up a routing table entry, normally
 * for an interface.
 */
ip_rtinit(dst, gateway, cmd, flags, protocol, tos)
	struct sockaddr *dst, *gateway;
	int cmd, flags, protocol;
	unsigned char tos;
{
	struct rtentry rt;

	bzero((caddr_t)&rt, sizeof (rt));
	rt.rt_dst = ((struct sockaddr_in *)dst)->sin_addr;
	rt.rt_gateway = ((struct sockaddr_in *)gateway)->sin_addr;
	rt.rt_flags = flags;
	rt.rt_protocol = protocol;
	rt.rt_tos = tos;
	(void) rtrequest(cmd, &rt);
}

rtnetmatch(sin1, sin2)
	struct in_addr *sin1, *sin2;
{
	return (in_netof(*sin1) == in_netof(*sin2));
}
