/*
 * @(#) Copyright 1989.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)ip_input.c (TWG)  1.7     89/09/08 "

/*
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of California at Berkeley. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 * Aug-18-92 Check for NULL return from ip_srcroute in ip_input. CAH
 #
 *	@(#)ip_input.c	7.9 (Berkeley) 3/15/88
 */

#include "sys/param.h"
#include "sys/types.h"
#ifndef XENIX
#include "sys/inline.h"
#endif
#include "sys/stream.h"
#include "sys/strlog.h"
#include "sys/debug.h"
#ifdef XENIX
#include "sys/assert.h"
#endif

#include "sys/inet.h"
#include "sys/socket.h"
#include "sys/if.h"
#include "sys/in.h"
#include "sys/in_var.h"
#include "sys/ip.h"
#include "sys/ip_var.h"
#include "sys/ip_icmp.h"
#include "sys/errno.h"
#include "sys/inetioctl.h"
#include "sys/route.h"

extern	u_char my_hosts_class;
extern 	void ipdrint();
struct	in_ifaddr *in_ifaddr;			/* first inet address */
struct	ipq iprsq = { &iprsq, &iprsq, };

/*
 * We need to save the IP options in case a protocol wants to respond
 * to an incoming packet over the same route if the packet got here
 * using IP source routing.  This allows connection establishment and
 * maintenance when the remote end is on a network that is not known
 * to us.
 */
int	ip_nhops = 0;
static	struct ip_srcrt {
	char	nop;				/* one NOP to align */
	char	srcopt[IPOPT_OFFSET + 1];	/* OPTVAL, OLEN and OFFSET */
	struct	in_addr route[MAX_IPOPTLEN / sizeof (struct in_addr)];
} ip_srcrt;

u_short	ip_id;
u_char	ipcksum = 1;
mblk_t	*ip_reass(), *bp_copy();
struct	sockaddr_in ipaddr = { AF_INET };

/*
 * Ip input routine.  Checksum and byte swap header.  If fragmented
 * try to reassamble.  If complete and fragment queue exists, discard.
 * Process options.  Pass to next level.
 */
void
ip_input(m, ifp)
	register mblk_t *m;
	struct ifnet *ifp;
{
	register struct ip *ip;
	register int i;
	register struct ipq *fp;
	register struct in_ifaddr *ia;
	register mblk_t *mo;
	int hlen, s;

	/*
	 * If no IP addresses have been set yet but the interfaces
	 * are receiving, can't do anything with incoming packets yet.
	 */
	if (in_ifaddr == NULL)
		goto bad;
	if (m == NULL)
		return;
	ipstat.ips_total++;
	ipstat.ipInReceives++;
	/*
	 * if the first block contains partial header, pull the next one up
	 */
	if ((MLEN(m) < IPHLEN) && (pullupmsg(m, IPHLEN) == 0)) {
		ipstat.ips_toosmall++;
		ipstat.ipInHdrErrors++;
		goto bad;
	}

#if !(vax || ns32000 || i386 || PSE)
	/*
	 * check for word alignment
	 */
	if ((int)(m->b_rptr) & ~3) {
		mblk_t *np;
		extern mblk_t *ip_alignmsg();

		/*
		 * Try to get 40 bytes together, word-aligned.
		 * If unsuccessful, free the buffer as we weren't 
		 * able to allocate a 40 byte buffer.
		 */
		if ((np = ip_alignmsg(m, 40)) == NULL) {
			ipstat.ipInHdrErrors++;
			goto bad;
		} else
			m = np;
	}
#endif

	ip = mtod(m, struct ip *);
	hlen = ip->ip_hl << 2;
	/*
	 * minimum header length
	 */
	if (hlen < IPHLEN) {
		ipstat.ips_badhlen++;
		ipstat.ipInHdrErrors++;
		goto bad;
	}
	/*
	 * if option, also pull it up if not already in the 1st block
	 */
	if (hlen > MLEN(m)) {
		if (pullupmsg(m, hlen) == 0) {
			ipstat.ips_badhlen++;
			ipstat.ipInHdrErrors++;
			goto bad;
		}
		ip = mtod(m, struct ip *);
	}
	STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
		"ip_input: hlen=%d src=%x dst=%x", hlen,
		ntohl(ip->ip_src.s_addr), ntohl(ip->ip_dst.s_addr));

	/*
	 * checksum the header which includes the option field
	 */
	if (ipcksum) {
#ifdef i386
		if (ip->ip_sum = ip_cksum(ip, hlen)) {
#else
		if (ip->ip_sum = in_cksum(m, hlen)) {
#endif
			ipstat.ips_badsum++;
			ipstat.ipInHdrErrors++;
			goto bad;
		}
	}

	/*
	 * Convert fields to host representation.
	 */
	ip->ip_len = ntohs((u_short)ip->ip_len);
	if (ip->ip_len < hlen) {
		ipstat.ips_badlen++;
		ipstat.ipInHdrErrors++;
		goto bad;
	}
	ip->ip_id = ntohs(ip->ip_id);
	ip->ip_off = ntohs((u_short)ip->ip_off);
	STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
		"ip_input: protocol=%d len=%d offset=%x", ip->ip_p,
		ip->ip_len, ip->ip_off);

	/*
	 * Check that the amount of data in the buffers
	 * is as at least much as the IP header would have us expect.
	 * Trim mbufs if longer than we expect.
	 * Drop packet if shorter than we expect.
	 */
	i = msgdsize(m) - (u_short)ip->ip_len;
	if (i != 0) {
		if (i < 0) {
			ipstat.ips_tooshort++;
			ipstat.ipInHdrErrors++;
			goto bad;
		}
		if (ipadjmsg(m, -i) == 0) {
			printf("ip_input: adjmsg screwed up\n");
			ipstat.ipInHdrErrors++;
			goto bad;
		}
	}

	/*
	 * Process options and, if not destined for us,
	 * ship it on.  ip_dooptions returns 1 when an
	 * error was detected (causing an icmp message
	 * to be sent and the original packet to be freed).
	 */
	ip_nhops = 0;		/* for source routed packets */
	if ((my_hosts_class > IPOPT_SECUR_UNCLASS) ||
	    (hlen > sizeof (struct ip))) {
		if (ip_dooptions(m, ifp))
			return;
	}

	/*
	 * Check our list of addresses, to see if the packet is for us.
	 */
	for (ia = in_ifaddr; ia; ia = ia->ia_next) {
#define	satosin(sa)	((struct sockaddr_in *)(sa))

		if (IA_SIN(ia)->sin_addr.s_addr == ip->ip_dst.s_addr)
			goto ours;
		if (
#ifdef	DIRECTED_BROADCAST
		    ia->ia_ifp == ifp &&
#endif
		    (ia->ia_ifp->if_flags & IFF_BROADCAST)) {
			u_long t;

			if (satosin(&ia->ia_broadaddr)->sin_addr.s_addr ==
			    ip->ip_dst.s_addr)
				goto ours;
			if (ip->ip_dst.s_addr == ia->ia_netbroadcast.s_addr)
				goto ours;
			/*
			 * Look for all-0's host part (old broadcast addr),
			 * either for subnet or net.
			 */
			t = ntohl(ip->ip_dst.s_addr);
			if (t == ia->ia_subnet)
				goto ours;
			if (t == ia->ia_net)
				goto ours;
			/*
			 * Also look for all -1's host part, either
			 * for subnet or net.
			 */
			if (t == (ia->ia_subnet | ~ia->ia_subnetmask))
				goto ours;
			if (t == (ia->ia_net | ~ia->ia_netmask))
				goto ours;
		}
	}
	if (ip->ip_dst.s_addr == (u_long)INADDR_BROADCAST)
		goto ours;
	if (ip->ip_dst.s_addr == INADDR_ANY)
		goto ours;

	/*
	 * Not for us; forward if possible and desirable.
	 */
	STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
		"calling ip_forward to %x", ntohl(ip->ip_dst.s_addr), 0);
	ip_forward(m, ifp);
	return;

ours:
	/*
	 * If offset or IP_MF are set, must reassemble.
	 * Otherwise, nothing need be done.
	 * (We could look in the reassembly queue to see
	 * if the packet was previously fragmented,
	 * but it's not worth the time; just let them time out.)
	 */
	if (ip->ip_off &~ IP_DF) {
		s = splni();
		/*
		 * Look for queue of fragments
		 * of this datagram.
		 */
		for (fp = iprsq.next; fp != &iprsq; fp = fp->next)
			if (ip->ip_id == fp->ipq_id &&
			    ip->ip_src.s_addr == fp->ipq_src.s_addr &&
			    ip->ip_dst.s_addr == fp->ipq_dst.s_addr &&
			    ip->ip_p == fp->ipq_p)
				goto found;
		fp = 0;
found:

		/*
		 * Adjust ip_len to not reflect header,
		 * set ip_mff if more fragments are expected,
		 * convert offset of this to bytes.
		 */
		ip->ip_len -= hlen;
		((struct ipasfrag *)ip)->ipf_mff = 0;
		if (ip->ip_off & IP_MF)
			((struct ipasfrag *)ip)->ipf_mff = 1;
		ip->ip_off <<= 3;

		/*
		 * If datagram marked as having more fragments
		 * or if this is not the first fragment,
		 * attempt reassembly; if it succeeds, proceed.
		 */
		if (((struct ipasfrag *)ip)->ipf_mff || ip->ip_off) {
			ipstat.ips_fragments++;
			ipstat.ipReasmReqds++;
			m = ip_reass(m, fp);
			if (m == 0) {
				(void) splx(s);
				return;
			}
			ip = mtod(m, struct ip *);
		} else {
			if (fp)
				ip_freef(fp);
		}
		(void) splx(s);
	} else
		ip->ip_len -= hlen;

	/*
	 * Switch out to protocol's input routine.
	 */
	ipstat.ipInDelivers++;
	if (ip->ip_p == IPPROTO_ICMP)
		icmp_input(m, ifp);
	else
	{
		/* strip IP options if present (send up srcrt in M_PROTO) */
		hlen = ip->ip_hl << 2;
		if (hlen > sizeof (struct ip))
		{
			if( (mo = ip_srcroute()) != (mblk_t *)0) {
			  mo->b_datap->db_type = M_PROTO;
			  ip->ip_len -= hlen - sizeof (struct ip);
			  (void) ip_dropoptions(m);
			  linkb(mo, m);
			  mo = m;
			}
		}
		(void) ipdrint(m, ip->ip_p);
	}
	return;
bad:
	STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE, "ip_input: bad packet");
	freemsg(m);
	return;
}

/*
 * Take incoming datagram fragment and try to
 * reassemble it into whole datagram.  If a chain for
 * reassembly of this datagram already exists, then it
 * is given as fp; otherwise have to make a chain.
 */
mblk_t *
ip_reass(m, fp)
	register mblk_t *m;
	register struct ipq *fp;
{
	register struct ipasfrag *ip;
	register struct ipasfrag *q;
	mblk_t *t;
	int hlen, i, next;

	ASSERT(m);

	ip = mtod(m, struct ipasfrag *);
	hlen = ip->ipf_hl << 2;

	/*
	 * Presence of header sizes in mbufs
	 * would confuse code below.
	 */
	m->b_rptr += hlen;

	/*
	 * If first fragment to arrive, create a reassembly queue.
	 */
	if (fp == 0) {
		if ((t = allocb(sizeof (struct ipq), BPRI_LO)) == NULL)
			goto dropfrag;
		fp = mtod(t, struct ipq *);
		c_enq(fp, &iprsq);
		fp->ipq_dtom = t;
		fp->ipq_ttl = IPFRAGTTL;
		fp->ipq_p = ((struct ip *)ip)->ip_p;
		fp->ipq_id = ((struct ip *)ip)->ip_id;
		fp->ipq_next = fp->ipq_prev = (struct ipasfrag *)fp;
		fp->ipq_src = ((struct ip *)ip)->ip_src;
		fp->ipq_dst = ((struct ip *)ip)->ip_dst;
		q = (struct ipasfrag *)fp;
		goto insert;
	}

	/*
	 * Find a segment which begins after this one does.
	 */
	for (q = fp->ipq_next; q != (struct ipasfrag *)fp; q = q->ipf_next)
		if (q->ip_off > ip->ip_off)
			break;

	/*
	 * If there is a preceding segment, it may provide some of
	 * our data already.  If so, drop the data from the incoming
	 * segment.  If it provides all of our data, drop us.
	 */
	if (q->ipf_prev != (struct ipasfrag *)fp) {
		i = q->ipf_prev->ip_off + q->ipf_prev->ip_len - ip->ip_off;
		if (i > 0) {
			if (i >= ip->ip_len)
				goto dropfrag;
			if (ipadjmsg(m, i) == 0)
				goto dropfrag;
			ip->ip_off += i;
			ip->ip_len -= i;
		}
	}

	/*
	 * While we overlap succeeding segments trim them or,
	 * if they are completely covered, dequeue them.
	 */
	while (q != (struct ipasfrag *)fp && ip->ip_off + ip->ip_len > q->ip_off) {
		i = (ip->ip_off + ip->ip_len) - q->ip_off;
		if (i < q->ip_len) {
			q->ip_len -= i;
			q->ip_off += i;
			if (ipadjmsg(q->ipf_dtom, i) == 0)
				goto dropfrag;
			break;
		}
		t = q->ipf_dtom;
		q = q->ipf_next;
		ip_deq(q->ipf_prev);
		freemsg(t);
	}

insert:
	/*
	 * Stick new segment in its place;
	 * check for complete reassembly.
	 */
	ip->ipf_dtom = m;
	ip_enq(ip, q->ipf_prev);
	next = 0;
	for (q = fp->ipq_next; q != (struct ipasfrag *)fp; q = q->ipf_next) {
		if (q->ip_off != next)
			return (0);
		next += q->ip_len;
	}
	if (q->ipf_prev->ipf_mff)
		return (0);

	/*
	 * Reassembly is complete; concatenate fragments.
	 */
	q = fp->ipq_next;
	m = q->ipf_dtom;		/* 1st block */
	for (q = q->ipf_next; q != (struct ipasfrag *)fp; q = q->ipf_next)
		linkb(m, q->ipf_dtom);

	/*
	 * Create header for new ip packet by
	 * modifying header of first packet;
	 * dequeue and discard fragment reassembly header.
	 * Make header visible.
	 */
	ip = fp->ipq_next;
	m = ip->ipf_dtom;
	ip->ip_len = next;
	((struct ip *)ip)->ip_src = fp->ipq_src;
	((struct ip *)ip)->ip_dst = fp->ipq_dst;
	c_deq(fp);
	((struct ip *)ip)->ip_p = fp->ipq_p;
	((struct ip *)ip)->ip_id = fp->ipq_id;
	(void) freeb(fp->ipq_dtom);
	ipstat.ipReasmOKs++;
	m->b_rptr -= ip->ipf_hl << 2;
	return (m);

dropfrag:
	ipstat.ips_fragdropped++;
	(void) freemsg(m);
	return (NULL);
}

/*
 * Free a fragment reassembly header and all
 * associated datagrams.
 */
ip_freef(fp)
	struct ipq *fp;
{
	register struct ipasfrag *q, *p;

	for (q = fp->ipq_next; q != (struct ipasfrag *)fp; q = p) {
		p = q->ipf_next;
		ip_deq(q);
		freemsg(q->ipf_dtom);
	}
	c_deq(fp);
	(void) freeb(fp->ipq_dtom);
}

/*
 * Put an ip fragment on a reassembly chain.
 * Like insque, but pointers in middle of structure.
 */
ip_enq(p, prev)
	register struct ipasfrag *p, *prev;
{

	p->ipf_prev = prev;
	p->ipf_next = prev->ipf_next;
	prev->ipf_next->ipf_prev = p;
	prev->ipf_next = p;
}

/*
 * To ip_enq as remque is to insque.
 */
ip_deq(p)
	register struct ipasfrag *p;
{

	p->ipf_prev->ipf_next = p->ipf_next;
	p->ipf_next->ipf_prev = p->ipf_prev;
}

static int iptimeID;
/*
 * IP timer processing;
 * if a timer expires on a reassembly
 * queue, discard it.
 */
iptimeout()
{
	register struct ipq *fp;
	int s = splni();

	fp = iprsq.next;
	if (fp == 0) {
		(void) splx(s);
		return;
	}
	while (fp != &iprsq) {
		--fp->ipq_ttl;
		fp = fp->next;
		if (fp->prev->ipq_ttl == 0) {
			ipstat.ips_fragtimeout++;
			ip_freef(fp->prev);
			ipstat.ipReasmFails++;
		}
	}
	iptimeID = timeout(iptimeout, (caddr_t)NULL, HZ);
	(void) splx(s);
}

/*
 * Cancel IP timer
 */
void
ipuntimeout()
{
	if (iptimeID) {
		(void) untimeout(iptimeID);
		iptimeID = 0;
	}

}

/*
 * Drain off all datagram fragments.
 */
ip_drain()
{
	register int s;

	s = splni();
	while (iprsq.next != &iprsq) {
		ipstat.ips_fragdropped++;
		ip_freef(iprsq.next);
		ipstat.ipReasmFails++;
	}
	(void) splx(s);
}

extern struct in_ifaddr *ifptoia();
struct in_ifaddr *ip_rtaddr();

/*
 * Do option processing on a datagram,
 * possibly discarding it if bad options
 * are encountered.
 *
 *	Options currently recognized:
 *		End of option list
 *		No operation
 *		Security
 *		Extended security
 *		Loose source routing
 *		Strict source routing
 *		Record Route
 *		Timestamp
 *		Stream ID
 *
 *	Options currently acted upon:
 *		End of option list
 *		No operation
 *		Security
 *		Loose source routing
 *		Strict source routing
 *		Record Route
 *		Timestamp
 *		Stream ID	( No action specified at this time )
 */
ip_dooptions(m, ifp)
	mblk_t *m;
	struct ifnet *ifp;
{
	register struct ip *ip;
	register u_char *cp;
	int opt, optlen, cnt, off, code, type = ICMP_PARAMPROB, tos;
	register struct ip_timestamp *ipt;
	register struct in_ifaddr *ia;
	struct in_addr *sin;
	n_time gm_time;

	ip = mtod(m, struct ip *);
	cp = (u_char *)(ip + 1);
	tos = ip->ip_tos;

	/*
	 * If security is on:
	 *	1. it must the first option
	 *	2. it must be in the packet.
	 */
	if (my_hosts_class > IPOPT_SECUR_UNCLASS) {
		if (ip->ip_hl <= (IPHLEN >> 2) || cp[0] != IPOPT_SECURITY) {
			type = ICMP_ADMINPROB;
			code = cp - (u_char *)ip;
			goto bad;
		}
	}
	cnt = (ip->ip_hl << 2) - sizeof (struct ip);
	for (; cnt > 0; cnt -= optlen, cp += optlen) {
		opt = cp[IPOPT_OPTVAL];
		if (opt == IPOPT_EOL)
			break;
		if (opt == IPOPT_NOP)
			optlen = 1;
		else {
			optlen = cp[IPOPT_OLEN];
			if (optlen <= 0 || optlen > cnt) {
				code = &cp[IPOPT_OLEN] - (u_char *)ip;
				goto bad;
			}
		}
		switch (opt) {

		default:
			break;

		case IPOPT_SECURITY:
		case IPOPT_EXTSECURITY:
		/*
		 *  Check that security is enforced
		 *  check_ip_security returns 1 on failure
		 *  Extended security not "supposed" to be used yet
		 *  Security failure means the data must be overwritten
		 */
			TRACE(1<<10,"IPOPT_SECURITY:entry\n","")
			if (check_ip_security((struct ipoption_security *)cp) ||
			    (opt == IPOPT_EXTSECURITY)) {
				register mblk_t *bpn = m;
				register int len;

				TRACE(1<<10,"IPOPT_SECURITY:no match\n","")
				off = (ip->ip_hl << 2);
				len = BLEN(bpn);
				if (len > off) {
					cp = bpn->b_rptr + off;
					len -= off;
					while (len--)
						*cp++ = 0xff;
				}
				for(;;) {
					if ((bpn = bpn->b_cont) == 0)
						break;
					cp = bpn->b_rptr;
					len = BLEN(bpn);
					while (len--)
						*cp++ = 0xff;
				}
				if (opt == IPOPT_EXTSECURITY)
					type = ICMP_ADMINPROB;
				else
					type = ICMP_PARAMPROB;
				code = cp - (u_char *)ip;
				goto bad;
			}
			break;

		/*
		 * Source routing with record.
		 * Find interface with current destination address.
		 * If none on this machine then drop if strictly routed,
		 * or do nothing if loosely routed.
		 * Record interface address and bring up next address
		 * component.  If strictly routed make sure next
		 * address on directly accessible net.
		 */
		case IPOPT_LSRR:
		case IPOPT_SSRR:
			if ((off = cp[IPOPT_OFFSET]) < IPOPT_MINOFF) {
				code = &cp[IPOPT_OFFSET] - (u_char *)ip;
				goto bad;
			}
			ipaddr.sin_addr = ip->ip_dst;
			ia = (struct in_ifaddr *)
				ifa_ifwithaddr((struct sockaddr *)&ipaddr);
			if (ia == 0) {
				if (opt == IPOPT_SSRR) {
					type = ICMP_UNREACH;
					code = ICMP_UNREACH_SRCFAIL;
					goto bad;
				}
				/*
				 * Loose routing, and not at next destination
				 * yet; nothing to do except forward.
				 */
				break;
			}
			off--;			/* 0 origin */
			if (off > optlen - sizeof(struct in_addr)) {
				/*
				 * End of source route.  Should be for us.
				 */
				save_rte(cp, ip->ip_src);
				break;
			}
			/*
			 * locate outgoing interface
			 */
			bcopy((caddr_t)(cp + off), (caddr_t)&ipaddr.sin_addr,
			    sizeof(ipaddr.sin_addr));
			if ((opt == IPOPT_SSRR &&
			    in_iaonnetof(in_netof(ipaddr.sin_addr)) == 0) ||
			    (ia = ip_rtaddr(ipaddr.sin_addr, tos)) == 0) {
				type = ICMP_UNREACH;
				code = ICMP_UNREACH_SRCFAIL;
				goto bad;
			}
			ip->ip_dst = ipaddr.sin_addr;
			bcopy((caddr_t)&(IA_SIN(ia)->sin_addr),
			    (caddr_t)(cp + off), sizeof(struct in_addr));
			cp[IPOPT_OFFSET] += sizeof(struct in_addr);
			break;

		case IPOPT_RR:
			if ((off = cp[IPOPT_OFFSET]) < IPOPT_MINOFF) {
				code = &cp[IPOPT_OFFSET] - (u_char *)ip;
				goto bad;
			}
			/*
			 * If no space remains, ignore.
			 */
			off--;			/* 0 origin */
			if (off > optlen - sizeof(struct in_addr))
				break;
			bcopy((caddr_t)(&ip->ip_dst), (caddr_t)&ipaddr.sin_addr,
			    sizeof(ipaddr.sin_addr));
			/*
			 * locate outgoing interface
			 */
			if ((ia = ip_rtaddr(ipaddr.sin_addr, tos)) == 0) {
				type = ICMP_UNREACH;
				code = ICMP_UNREACH_HOST;
				goto bad;
			}
			bcopy((caddr_t)&(IA_SIN(ia)->sin_addr),
			    (caddr_t)(cp + off), sizeof(struct in_addr));
			cp[IPOPT_OFFSET] += sizeof(struct in_addr);
			break;

		case IPOPT_TS:
			code = cp - (u_char *)ip;
			type = ICMP_PARAMPROB;
			ipt = (struct ip_timestamp *)cp;
			if (ipt->ipt_len < 5)
				goto bad;
			if (ipt->ipt_ptr > ipt->ipt_len - sizeof (long)) {
				if (++ipt->ipt_oflw == 0)
					goto bad;
				break;
			}
			sin = (struct in_addr *)(cp + ipt->ipt_ptr - 1);
			switch (ipt->ipt_flg) {

			case IPOPT_TS_TSONLY:
				break;

			case IPOPT_TS_TSANDADDR:
				if (ipt->ipt_ptr + sizeof(n_time) +
				    sizeof(struct in_addr) > ipt->ipt_len)
					goto bad;
				ia = ifptoia(ifp);
				bcopy((caddr_t)&IA_SIN(ia)->sin_addr,
				    (caddr_t)sin, sizeof(struct in_addr));
				ipt->ipt_ptr += sizeof(struct in_addr);
				break;

			case IPOPT_TS_PRESPEC:
				if (ipt->ipt_ptr + sizeof(n_time) +
				    sizeof(struct in_addr) > ipt->ipt_len)
					goto bad;
				bcopy((caddr_t)sin, (caddr_t)&ipaddr.sin_addr,
				    sizeof(struct in_addr));
				if (ifa_ifwithaddr((struct sockaddr *)&ipaddr) == 0)
					continue;
				ipt->ipt_ptr += sizeof(struct in_addr);
				break;

			default:
				goto bad;
			}
			gm_time = ip_time();
			bcopy((caddr_t)&gm_time, (caddr_t)cp + ipt->ipt_ptr - 1,
			    sizeof(gm_time));
			ipt->ipt_ptr += sizeof(gm_time);
			break;

		case IPOPT_SATID:
			/*
			 * No action taken on input as yet.
			 * Must be on a network that requires SATNETID to
			 * determine action if any.
			 */
			break;	
		}
	}
	return (0);
bad:
	(void) icmp_error(m, type, code, ifp);
	return (1);
}

/*
 * Given address of next destination (final or next hop),
 * return internet address info of interface to be used to get there.
 */
struct in_ifaddr *
ip_rtaddr(nhop, tos)
	struct in_addr nhop;
{
	register struct in_ifaddr *ia;
	register struct ifnet *ifp;
	struct route_dst rtd;

	/*
	 * Lookup route to next hop.
	 */
	rtd.rtd_dst = nhop;
	if (rtlookup(&rtd, 1, tos) == 0) {
		return ((struct in_ifaddr *)0);
	}

	/*
	 * Find address associated with outgoing interface.
	 */
	for (ia = in_ifaddr; ia; ia = ia->ia_next)
		if (ia->ia_ifp == rtd.rtd_ifp)
			break;
	return (ia);
}

/*
 * Save incoming source route for use in replies,
 * to be picked up later by ip_srcroute if the receiver is interested.
 */
save_rte(option, dst)
	u_char *option;
	struct in_addr dst;
{
	unsigned olen;
	extern ipprintfs;

	olen = option[IPOPT_OLEN];
	if (olen > sizeof(ip_srcrt) - 1) {
		STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
			"save_rte: olen %d", olen);
		return;
	}
	bcopy((caddr_t)option, (caddr_t)ip_srcrt.srcopt, olen);
	ip_nhops = (olen - IPOPT_OFFSET - 1) / sizeof(struct in_addr);
	ip_srcrt.route[ip_nhops++] = dst;
}

/*
 * Retrieve incoming source route for use in replies,
 * in the same form used by setsockopt.
 * The first hop is placed before the options, will be removed later.
 */
mblk_t *
ip_srcroute()
{
	register struct in_addr *p, *q;
	register mblk_t *m;
	register int count;

	if (ip_nhops == 0)
		return ((mblk_t *)0);
	count = ip_nhops * sizeof(struct in_addr) + IPOPT_OFFSET + 1 + 1;
	if ((m = allocb(count, BPRI_LO)) == NULL)
		return((mblk_t *)0);

	/*
	 * First save first hop for return route
	 */
	p = &ip_srcrt.route[ip_nhops - 1];
	*(mtod(m, struct in_addr *)) = *p--;

	/*
	 * Copy option fields and padding (nop) to mbuf.
	 */
	ip_srcrt.nop = IPOPT_NOP;
	bcopy((caddr_t)&ip_srcrt, mtod(m, caddr_t) + sizeof(struct in_addr),
	    IPOPT_OFFSET + 1 + 1);
	q = (struct in_addr *)(mtod(m, caddr_t) +
	    sizeof(struct in_addr) + IPOPT_OFFSET + 1 + 1);
	m->b_wptr += sizeof(struct in_addr) + IPOPT_OFFSET + 1 + 1;

	/*
	 * Record return path as an IP source route,
	 * reversing the path (pointers are now aligned).
	 */
	while (p >= ip_srcrt.route) {
		*q++ = *p--;
		m->b_wptr += sizeof (struct in_addr);
	}
	m->b_rptr += sizeof (struct in_addr);
	return (m);
}

#ifndef	IPFORWARDING
#define	IPFORWARDING	1
#endif
#ifndef	IPSENDREDIRECTS
#define	IPSENDREDIRECTS	1
#endif
int	ipprintfs = 0;
extern	int ipforwarding;
extern	int in_interfaces;
extern	int ipsendredirects;
extern	int ipgateway_icmp;

/*
 * Forward a packet.  If some error occurs return the sender
 * an icmp packet.  Note we can't always generate a meaningful
 * icmp message because icmp doesn't have a large enough repertoire
 * of codes and types.
 *
 * If not forwarding (possibly because we have only a single external
 * network), just drop the packet.  This could be confusing if ipforwarding
 * was zero but some routing protocol was advancing us as a gateway
 * to somewhere.  However, we must let the routing protocol deal with that.
 */
ip_forward(m, ifp)
	mblk_t *m;
	struct ifnet *ifp;
{
	register int error, type = 0, code;
	mblk_t *mcopy = m;
	struct in_addr dest;
	register struct ip *ip;
	struct route_dst rtd;

	ip = mtod(m, struct ip *);
	dest.s_addr = 0;
	STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
	    "ip_forward: src %x dst %x ttl %x", ntohl(ip->ip_src.s_addr), 
	    ntohl(ip->ip_dst.s_addr), ip->ip_ttl);
	ip->ip_id = htons(ip->ip_id);
	if (ipforwarding == 0 || in_interfaces <= 1) {
		ipstat.ips_cantforward++;
		if (ipgateway_icmp) {
			type = ICMP_UNREACH, code = ICMP_UNREACH_NET;
			goto sendicmp;
		} else {
			ipstat.ipInAddrErrors++;
			freemsg(m);
			return;
		}
	}
	if (in_canforward(ip->ip_dst) == 0) {
		ipstat.ipInAddrErrors++;
		freemsg(m);
		return;
	}
	if (ip->ip_ttl <= IPTTLDEC) {
		type = ICMP_TIMXCEED, code = ICMP_TIMXCEED_INTRANS;
		TRACE(1<<11,"ip_forward: ip_ttl expired\n","")
		ipstat.ipInHdrErrors++;
		goto sendicmp;
	}
	ip->ip_ttl -= IPTTLDEC;

	/*
	 * Save at most 64 bytes of the packet in case
	 * we need to generate an ICMP message to the src.
	 */
	mcopy = bp_copy(m, 0, MIN((int)ip->ip_len, 64));

	/*
	 * Lookup forwarding route.
	 */
	rtd.rtd_dst = ip->ip_dst;
	if (rtlookup(&rtd, 1, ip->ip_tos) == 0) {
		error = ENETUNREACH;
		freemsg(m);		/* NBB 02-01-90 */
		goto failed;
	}

	/*
	 * If forwarding packet using same interface that it came in on,
	 * perhaps should send a redirect to sender to shortcut a hop.
	 * Only send redirect if source is sending directly to us,
	 * and if packet was not source routed (or has any options).
	 * Also, don't send redirect if forwarding using a default route
	 * or a route modfied by a redirect.
	 */
#define	satosin(sa)	((struct sockaddr_in *)(sa))
	if ((rtd.rtd_ifp == ifp) &&
	    (rtd.rtd_flags & (RTF_DYNAMIC|RTF_MODIFIED)) == 0 &&
	    (rtd.rtd_dst.s_addr != 0) && ipsendredirects &&
	    ip->ip_hl == (sizeof(struct ip) >> 2))
	{
		struct in_ifaddr *ia;
		u_long src = ntohl(ip->ip_src.s_addr);
		u_long dst = ntohl(ip->ip_dst.s_addr);

		if ((ia = ifptoia(ifp)) &&
		   (src & ia->ia_subnetmask) == ia->ia_subnet)
		{
		    /*
		     * If the destination is reached by a route to host,
		     * is on a subnet of a local net, or is directly
		     * on the attached net (!), use host redirect.
		     * (We may be the correct first hop for other subnets.)
		     */
		    dest = rtd.rtd_dst;
		    type = ICMP_REDIRECT;
		    code = ICMP_REDIRECT_NET;
		    if ((rtd.rtd_flags & RTF_HOST) ||
			(rtd.rtd_flags & RTF_GATEWAY) == 0)
			code = ICMP_REDIRECT_HOST;
		    else for (ia = in_ifaddr; ia = ia->ia_next; ) {
			if ((dst & ia->ia_netmask) == ia->ia_net) {
			    if (ia->ia_subnetmask != ia->ia_netmask)
				    code = ICMP_REDIRECT_HOST;
			    break;
			}
		    }
		    STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
			   "redirect (%x) to %x", code, dest);
		}
	}

	error = ip_output(m, (mblk_t *)0, IP_FORWARDING);
failed:
	if (error)
		ipstat.ips_cantforward++;
	else if (type)
		ipstat.ips_redirectsent++;
	else {
		if (mcopy)
			freemsg(mcopy);
		ipstat.ips_forward++;
		ipstat.ipForwDatagrams++;
		return;
	}
	if (mcopy == NULL)
		return;
	if (! ipgateway_icmp) {
		if (mcopy)
			freemsg(mcopy);
		return;
	}
	ip = mtod(mcopy, struct ip *);
	type = ICMP_UNREACH;
	switch (error) {

	case 0:				/* forwarded, but need redirect */
		type = ICMP_REDIRECT;
		/* code set above */
		break;

	case ENETUNREACH:
	case ENETDOWN:
		if (in_localaddr(ip->ip_dst))
			code = ICMP_UNREACH_HOST;
		else
			code = ICMP_UNREACH_NET;
		break;

	case EMSGSIZE:
		code = ICMP_UNREACH_NEEDFRAG;
		break;

	case EPERM:
		code = ICMP_UNREACH_PORT;
		break;

	case ENOBUFS:
		type = ICMP_SOURCEQUENCH;
		break;

	case EHOSTDOWN:
	case EHOSTUNREACH:
		code = ICMP_UNREACH_HOST;
		break;
	}
sendicmp:
	icmp_error(mcopy, type, code, ifp, dest);
}

/*
 * Strip the IP options out of the datagram.
 */
ip_dropoptions(bp)
	register mblk_t *bp;
{
	register struct ip *ip;
	register int tlen, hlen;

	/* Don't do anything if no options present */
	ip = (struct ip *)bp->b_rptr;
	if ((hlen = (ip->ip_hl << 2)) <= sizeof (struct ip))
		return;

	/* Pull up the trailing data and remove the hole */
	tlen = BLEN(bp) - hlen;
	if (tlen > 0)
	{
		bcopy(bp->b_rptr + hlen, bp->b_rptr + IPHLEN, tlen);
		bp->b_wptr = bp->b_rptr + sizeof (struct ip) + tlen;
	}

	/* Adjust IP header to remove the hole */
	ip->ip_hl = sizeof (struct ip) >> 2;

	return;
}

#if !(vax || ns32000 || i386 || PSE)

/*
 * Align buffer (arg "bp") to arg "count" bytes.
 */

mblk_t *
ip_alignmsg(bp, count)

	register mblk_t *bp;
{
	register mblk_t *nbp, *p;
	register int size;

	if ((nbp = allocb(count, BPRI_LO)) == NULL)
		return NULL;
	/*
	 * Copy the required number of bytes
	 * to the new buffer.
	 */
	for (;  (count > 0) && (bp != NULL); ){
		size = min(count, (bp->b_wptr - bp->b_rptr));
		if (size){
			bcopy(bp->b_rptr, nbp->b_wptr, size);
			nbp->b_wptr += size;
			bp->b_rptr += size;
			count -= size;
		}
		if ((bp->b_wptr - bp->b_rptr) == 0){
			/*
			 * Free up this baby.
			 */
			p = bp->b_cont;
			bp->b_cont = NULL;
			freemsg(bp);
			bp = p;
		}
		else 
			break;
	}
	nbp->b_cont = bp;
	return  nbp;

}	/*  End of ip_alignmsg()  */
#endif

/* end */
