/*
 * @(#) Copyright 1989.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)ip_icmp.c (TWG)  1.4     89/07/30 "

/*
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of California at Berkeley. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 *	@(#)ip_icmp.c	7.7 (Berkeley) 12/7/87
 */

#include "sys/param.h"
#include "sys/types.h"
#ifndef XENIX
#include "sys/inline.h"
#endif
#include "sys/stream.h"
#include "sys/strlog.h"
#include "sys/debug.h"
#ifdef XENIX
#include "sys/assert.h"
#endif
#include "sys/errno.h"
#include "sys/time.h"

#include "sys/inet.h"
#include "sys/socket.h"
#include "sys/if.h"
#include "sys/in.h"
#include "sys/in_var.h"
#include "sys/ip.h"
#include "sys/ip_var.h"
#include "sys/ip_icmp.h"
#include "sys/icmp_var.h"
#include "sys/inetioctl.h"
#include "sys/route.h"

void ip_ctlmsg();
void ipdrint();

/*
 * Generate an error packet of type error
 * in response to bad packet ip.
 */
/*VARARGS4*/
icmp_error(omp, type, code, ifp, dest)
	mblk_t *omp;
	int type, code;
	struct ifnet *ifp;
	struct in_addr dest;
{
	register unsigned oiplen;
	register struct icmp *icp;
	mblk_t *m;
	struct ip *oip, *nip;
	unsigned icmplen;

	oip = mtod(omp, struct ip *);
	oiplen = oip->ip_hl << 2;
	STRlog(IP_ID, -1, DPRI_LO, SL_te,
		"icmp_error(%x, %d, %d, %x, %x)",
		omp, type, code, ifp, dest.s_addr);
	if (type != ICMP_REDIRECT)
		icmpstat.icps_error++;
	/*
	 * Don't send error if not the first fragment of message.
	 * Don't error if the old packet protocol was ICMP
	 * error message, only known informational types.
	 */
	if (oip->ip_off &~ (IP_MF|IP_DF))
		goto free;
	if (oip->ip_p == IPPROTO_ICMP && type != ICMP_REDIRECT &&
	  !ICMP_INFOTYPE(((struct icmp *)((caddr_t)oip + oiplen))->icmp_type)) {
		icmpstat.icps_oldicmp++;
		goto free;
	}
	if ((unsigned)type > ICMP_MAXTYPE)
		goto free;

	/*
	 * First, formulate icmp message
	 */
	icmplen = oiplen + MIN(8, oip->ip_len);
	m = allocb(oiplen + icmplen + ICMP_MINLEN, BPRI_MED);
	if (m == NULL)
		goto free;
	m->b_wptr = m->b_datap->db_lim;
	m->b_rptr = m->b_wptr - (icmplen + ICMP_MINLEN);
	icp = mtod(m, struct icmp *);
	icmpstat.icps_outhist[type]++;
	icp->icmp_type = type;
	if (type == ICMP_REDIRECT)
		icp->icmp_gwaddr = dest;
	else
		icp->icmp_void = 0;
	if (type == ICMP_PARAMPROB) {
		icp->icmp_pptr = code;
		code = 0;
	}
	icp->icmp_code = code;
	bcopy((caddr_t)oip, (caddr_t)&icp->icmp_ip, icmplen);
	nip = &icp->icmp_ip;
	nip->ip_len += oiplen;
	nip->ip_len = htons((ushort)nip->ip_len);

	/*
	 * Now, copy old ip header in front of icmp message.
	 */
	if (MLEN(m) + oiplen > DLEN(m))
		oiplen = sizeof(struct ip);
	if (MLEN(m) + oiplen > DLEN(m))
		panic("icmp len");
	m->b_rptr -= oiplen;
	nip = mtod(m, struct ip *);
	bcopy((caddr_t)oip, (caddr_t)nip, oiplen);
	nip->ip_len = MLEN(m);
	nip->ip_p = IPPROTO_ICMP;
	icmp_reflect(m, ifp);

free:
	freemsg(omp);
}

static struct sockaddr_in icmpsrc = { AF_INET };
static struct sockaddr_in icmpdst = { AF_INET };
static struct sockaddr_in icmpgw = { AF_INET };
struct in_ifaddr *ifptoia();

/*
 * Process a received ICMP message.
 */
icmp_input(m, ifp)
	register mblk_t *m;
	struct ifnet *ifp;
{
	register struct icmp *icp;
	register struct ip *ip;
	int icmplen, hlen;
	register int i;
	struct in_ifaddr *ia;
	int code;
	extern struct in_addr in_makeaddr();

	ip = mtod(m, struct ip *);
	icmplen = ip->ip_len;
	hlen = ip->ip_hl << 2;
	/*
	 * Locate icmp structure in mbuf, and check
	 * that not corrupted and of at least minimum length.
	 */
	STRlog(IP_ID, -1, DPRI_LO, SL_te,
		"icmp_input from %x, len %d",
		ntohl(ip->ip_src.s_addr), icmplen);
	if (icmplen < ICMP_MINLEN) {
		icmpstat.icps_tooshort++;
		goto free;
	}
	i = hlen + MIN(icmplen, ICMP_ADVLENMIN);
 	if ((MLEN(m) < i) && (pullupmsg(m, i) == 0)) {
		icmpstat.icps_tooshort++;
		freemsg(m);
		return;
	}
 	ip = mtod(m, struct ip *);
	m->b_rptr += hlen;
	icp = mtod(m, struct icmp *);
	if (in_cksum(m, icmplen)) {
		icmpstat.icps_checksum++;
		goto free;
	}
	m->b_rptr -= hlen;

	STRlog(IP_ID, -1, DPRI_LO, SL_te,
		"icmp_input type %d code %d", icp->icmp_type, icp->icmp_code);
	if (icp->icmp_type > ICMP_MAXTYPE)
		goto raw;
	icmpstat.icps_inhist[icp->icmp_type]++;
	code = icp->icmp_code;
	switch (icp->icmp_type) {

	case ICMP_UNREACH:
		if (code > 5)
			goto badcode;
		code += PRC_UNREACH_NET;
		goto deliver;

	case ICMP_TIMXCEED:
		if (code > 1)
			goto badcode;
		code += PRC_TIMXCEED_INTRANS;
		goto deliver;

	case ICMP_PARAMPROB:
		if (code)
			goto badcode;
		code = PRC_PARAMPROB;
		goto deliver;

	case ICMP_SOURCEQUENCH:
		if (code)
			goto badcode;
		code = PRC_QUENCH;
	deliver:
		/*
		 * Problem with datagram; advise higher level routines.
		 */
		icp->icmp_ip.ip_len = ntohs((u_short)icp->icmp_ip.ip_len);
		if (icmplen < ICMP_ADVLENMIN || icmplen < ICMP_ADVLEN(icp)) {
			icmpstat.icps_badlen++;
			goto free;
		}
		STRlog(IP_ID, -1, DPRI_LO, SL_te,
			"deliver to protocol %d", icp->icmp_ip.ip_p);
		icmpsrc.sin_addr = icp->icmp_ip.ip_dst;
		ip_ctlmsg(code, icp->icmp_ip.ip_p,
			(struct sockaddr_in *)&icmpsrc);
		break;

	badcode:
		icmpstat.icps_badcode++;
		break;

	case ICMP_ECHO:
		icp->icmp_type = ICMP_ECHOREPLY;
		goto reflect;

	case ICMP_TSTAMP:
		if (icmplen < ICMP_TSLEN) {
			icmpstat.icps_badlen++;
			break;
		}
		icp->icmp_type = ICMP_TSTAMPREPLY;
		icp->icmp_rtime = ip_time();
		icp->icmp_ttime = icp->icmp_rtime;	/* bogus, do later! */
		goto reflect;
		
	case ICMP_IREQ:
#define	satosin(sa)	((struct sockaddr_in *)(sa))
		if (in_netof(ip->ip_src) == 0 && (ia = ifptoia(ifp)))
			ip->ip_src = in_makeaddr(in_netof(IA_SIN(ia)->sin_addr),
			    in_lnaof(ip->ip_src));
		icp->icmp_type = ICMP_IREQREPLY;
		goto reflect;

	case ICMP_MASKREQ:
		if (icmplen < ICMP_MASKLEN || (ia = ifptoia(ifp)) == 0)
			break;
		icp->icmp_type = ICMP_MASKREPLY;
		icp->icmp_mask = htonl(ia->ia_subnetmask);
		if (ip->ip_src.s_addr == 0) {
			if (ia->ia_ifp->if_flags & IFF_BROADCAST)
			    ip->ip_src = satosin(&ia->ia_broadaddr)->sin_addr;
			else if (ia->ia_ifp->if_flags & IFF_POINTOPOINT)
			    ip->ip_src = satosin(&ia->ia_dstaddr)->sin_addr;
		}
reflect:
		ip->ip_len += hlen;	/* since ip_input deducts this */
		icmpstat.icps_reflect++;
		icmpstat.icps_outhist[icp->icmp_type]++;
		icmp_reflect(m, ifp);
		return;

	case ICMP_REDIRECT:
		if (icmplen < ICMP_ADVLENMIN || icmplen < ICMP_ADVLEN(icp)) {
			icmpstat.icps_badlen++;
			break;
		}
		/*
		 * Short circuit routing redirects to force
		 * immediate change in the kernel's routing
		 * tables.  The message is also handed to anyone
		 * listening on a raw socket (e.g. the routing
		 * daemon for use in updating its tables).
		 */
		icmpgw.sin_addr = ip->ip_src;
		icmpdst.sin_addr = icp->icmp_gwaddr;
		STRlog(IP_ID, -1, DPRI_LO, SL_te,
			"redirect dst %x to %x",
			icp->icmp_ip.ip_dst, icp->icmp_gwaddr);
		if (code == ICMP_REDIRECT_NET || code == ICMP_REDIRECT_TOSNET) {
			icmpsrc.sin_addr =
			  in_makeaddr(in_netof(icp->icmp_ip.ip_dst),INADDR_ANY);
			rtredirect((struct sockaddr *)&icmpsrc,
			  (struct sockaddr *)&icmpdst, RTF_GATEWAY,
			  (struct sockaddr *)&icmpgw, ip->ip_tos);
			icmpsrc.sin_addr = icp->icmp_ip.ip_dst;
			ip_ctlmsg(PRC_REDIRECT_NET, icp->icmp_ip.ip_p,
				(struct sockaddr_in *)&icmpsrc);
		} else {
			icmpsrc.sin_addr = icp->icmp_ip.ip_dst;
			rtredirect((struct sockaddr *)&icmpsrc,
			  (struct sockaddr *)&icmpdst, RTF_GATEWAY|RTF_HOST,
			  (struct sockaddr *)&icmpgw, ip->ip_tos);
			ip_ctlmsg(PRC_REDIRECT_HOST, icp->icmp_ip.ip_p,
				(struct sockaddr_in *)&icmpsrc);
		}
		break;

	/*
	 * No kernel processing for the following;
	 * just fall through to send to raw listener.
	 */
	case ICMP_ECHOREPLY:
	case ICMP_TSTAMPREPLY:
	case ICMP_IREQREPLY:
	case ICMP_MASKREPLY:
	default:
		break;
	}

raw:
	STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE, "ip_raw_input: from %x to %x",
		ntohl(ip->ip_src.s_addr), ntohl(ip->ip_dst.s_addr));
	ipdrint(m, IPPROTO_RAW);
	return;

free:
	freemsg(m);
}

/*
 * Reflect the ip packet back to the source
 */
icmp_reflect(m, ifp)
	mblk_t *m;
	struct ifnet *ifp;
{
	register struct ip *ip;
	register struct in_ifaddr *ia;
	struct in_addr t;
	mblk_t *opts = 0, *ip_srcroute();
	int optlen;

	ip = mtod(m, struct ip *);
	optlen = (ip->ip_hl << 2) - sizeof(struct ip);
	t = ip->ip_dst;
	ip->ip_dst = ip->ip_src;
	/*
	 * If the incoming packet was addressed directly to us,
	 * use dst as the src for the reply.  Otherwise (broadcast
	 * or anonymous), use the address which corresponds
	 * to the incoming interface.
	 */
	for (ia = in_ifaddr; ia; ia = ia->ia_next) {
		if (t.s_addr == IA_SIN(ia)->sin_addr.s_addr)
			break;
		if ((ia->ia_ifp->if_flags & IFF_BROADCAST) &&
		    t.s_addr == satosin(&ia->ia_broadaddr)->sin_addr.s_addr)
			break;
	}
	if (ia == (struct in_ifaddr *)0)
		ia = ifptoia(ifp);
	if (ia == (struct in_ifaddr *)0)
		ia = in_ifaddr;
	t = IA_SIN(ia)->sin_addr;
	ip->ip_src = t;
	ip->ip_ttl = MAXTTL;

	if (optlen > 0) {
		/*
		 * Retrieve any source routing from the incoming packet
		 * and strip out other options.  Adjust the IP length.
		 */
		opts = ip_srcroute();
		ip->ip_len -= optlen;
		ip_dropoptions(m);
	}
	icmp_send(m, opts);
	if (opts)
		(void) freemsg(opts);
}

struct in_ifaddr *
ifptoia(ifp)
	struct ifnet *ifp;
{
	register struct in_ifaddr *ia;

	for (ia = in_ifaddr; ia; ia = ia->ia_next)
		if (ia->ia_ifp == ifp)
			return (ia);
	return ((struct in_ifaddr *)0);
}

/*
 * Send an icmp packet back to the ip level,
 * after supplying a checksum.
 */
icmp_send(m, opts)
	register mblk_t *m;
	mblk_t *opts;
{
	register int hlen;
	register struct icmp *icp;
	register struct ip *ip;

	ip = mtod(m, struct ip *);
	hlen = ip->ip_hl << 2;
	m->b_rptr += hlen;
	icp = mtod(m, struct icmp *);
	icp->icmp_cksum = 0;
	icp->icmp_cksum = in_cksum(m, ip->ip_len - hlen);
	m->b_rptr -= hlen;
	STRlog(IP_ID, -1, DPRI_LO, SL_te,
		"icmp_send dst %x src %x",
		ntohl(ip->ip_dst.s_addr), ntohl(ip->ip_src.s_addr));
	(void) ip_output(m, opts, 0);
}

ip_time()
{
	register int s;
	u_long t;
	extern int time;
	extern int lbolt;

	s = spltime();
	t = (time % (24*60*60)) * 1000 + (((lbolt%HZ)*1000)/HZ) / 1000;
	(void) splx(s);

	return (htonl(t));
}
