/*  vi: set ts=4 sw=4 : <- modeline
 * @(#) Copyright 1989.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)arp.c  1.9  15:47:23 - 89/08/28"

/*
 * SNH - 10/16/91
 * Add RARP protocol support.
 */
/*
 * New version of Address resolution code - per RFC 826.
 * tested on i386-AT and 3b2/400.
 *
 * - 7/20/89
 * Corrected M_START/M_STOP operation to work with drivers that support
 * UP/DOWN.  M_HANGUP not supported (ever used?).  Note: by default all
 * drivers marked up for compat with old software.
 *
 * - 3/20/89
 * Added support AT&T Data Link Provider Interface (DLPI).
 *
 * We now follow the DLPI spec when talking to drivers.  Automatic fall
 * back to LLI for old drivers is detected.  If DLPI is detected, then
 * we will ATTACH if the driver is running in STYLE2 mode.  It is upto
 * the user level daemon (inetinit) to download the PPA to us if we need
 * to send down an ATTACH.
 * 
 * - 3/7/89
 * Added support for SNAP encoding, Reverse ARP.
 *
 * Cleaned up top<->bottom multiplexing.  Any variable number N streams
 * can be linked below a single upper stream.  Each substream must have
 * a unique SAP value.  Each upper stream must have a unique IP address.
 * All ethernet mapping (and IEEE802.3/SNAP vs. OLD ethernet selection)
 * is hidden in the ARP module.  All IP addresses are now kept in network
 * byte order!
 *
 * Notes:
 *
 * This ARP no longer passes thru IP bind/unbind/inforeq messages.  These
 * are processed by the ARP module directly.  LLI bind/unbind/inforeq
 * messages for the lower streams are generated by the ARP module itself.
 * Since the MTU value in the IP module above depends on the MTU values
 * from the N substreams attached below it, we wait for the lower streams
 * to be completely attached before IP can finishing connecting.
 *
 * Note: set tabstop=4
 */

#include "sys/param.h"
#include "sys/types.h"
#ifndef XENIX
#include "sys/inline.h"
#endif
#include "sys/systm.h"
#include "sys/stream.h"
#include "sys/stropts.h"
#include "sys/log.h"
#include "sys/strlog.h"
#include "sys/debug.h"
#ifdef XENIX
#include "sys/assert.h"
#endif
#include "sys/errno.h"

#include "sys/dlpi.h"
#include "sys/inet.h"
#include "sys/socket.h"
#include "sys/if.h"
#include "sys/in.h"
#include "sys/in_var.h"
#include "sys/ip.h"
#include "sys/arp.h"
#include "sys/inetioctl.h"

#undef LC_REQSIZE
#undef LC_INDSIZE

extern struct arp    arp[];
extern struct arpb   arpb[];
extern struct arptab arptab[];
extern int           arp_cnt;
extern int           arpb_cnt;
extern int           arptab_cnt;
extern unchar        Bcastnid[];

int canput_dlarpsnd, arpcalls, arpcache;
struct arptab *arp_bhash[NARP_BUCKETS];

static int arpopen(), arpclose();
extern int arpbiput(), arpoput();

static struct module_info arp_info =
	{ ARP_ID, "arp", 0, INFPSZ, 0, 0 };

static struct qinit arprinit =
	{ NULL, NULL, arpopen, arpclose, NULL, &arp_info, NULL };
static struct qinit arpwinit =
	{ arpoput, NULL, NULL, NULL, NULL, &arp_info, NULL };

static struct qinit arpmrinit =
	{ arpbiput, NULL, NULL, NULL, NULL, &arp_info, NULL };
static struct qinit arpmwinit =
	{ NULL, NULL, NULL, NULL, NULL, &arp_info, NULL };

struct streamtab arpinfo =
	{ &arprinit, &arpwinit, &arpmrinit, &arpmwinit };

static char *arp_etherprf();
static void arp_ackunlnk();

/*
 * initialization (nothing yet)
 */
arpinit()
{
	;
}

static
arpopen(q, dev, flag, sflag)
	queue_t *q;
{
	register struct arp *ap;
	register int mdev;

	mdev = minor(dev);
	STRLOG(ARP_ID, mdev, DPRI_LO, SL_TRACE, "arpopen", 0, 0);

	if (sflag == CLONEOPEN) {
		for (ap = &arp[1]; ap < &arp[arp_cnt]; ap++) {
			if (ap->ar_rdq == (queue_t *)NULL)
				break;
		}
		if ((mdev = ap - &arp[0]) >= arp_cnt) {
			goto fail;
		}
	} else if (ap = (struct arp *)q->q_ptr) {
		return(mdev);
	} else {
		if (mdev < 0 || mdev >= arp_cnt) {
			goto fail;
		}
		ap = &arp[mdev];
	}

	/* need valid q_ptr before sending 'inforeq' */
	q->q_ptr = WR(q)->q_ptr = (caddr_t)ap;
	ap->ar_rdq = q;
	ap->ar_inaddr = 0;  /* invalid */
	ap->ar_first = 0;
	ap->ar_state = DL_UNBND;
	return (mdev);

fail:
	STRlog(ARP_ID, mdev, DPRI_LO, SL_te, "arpopen: failed dev %x", mdev);
	return (OPENFAIL);
}

/*
 * close the top arp data structure
 * called by STREAMS internally
 */
static
arpclose(q)
	queue_t *q;
{
	struct arp *ap = (struct arp *)q->q_ptr;

	ASSERT(q);

	flushq(q, FLUSHALL);
	flushq(OTHERQ(q), FLUSHALL);
	bzero((caddr_t)ap, sizeof(struct arp));
	return (0);
}

/*
 * close the bottom arp data structure
 * called by the UNLINK code or from an error received from lower stream
 */
static
arpbclose(q)
	queue_t *q;
{
	struct arpb **abp, *ab = (struct arpb *)q->q_ptr;
	struct arp *ar;

	ASSERT(q && ab);

	/* delink from top queue */
	ar = ab->ab_ar;
	if (ar != NULL)
	{
		abp = &ar->ar_first;
		while (*abp != NULL)
		{
			if (*abp == ab)
			{
				*abp = ab->ab_next;
				break;
			}

			abp = &((*abp)->ab_next);
		}
	}

	/* flush and clear bottom queue */
	flushq(q, FLUSHALL);
	flushq(OTHERQ(q), FLUSHALL);
	/* 
	 * Now ack any pending I_UNLINK ioctl
	 */
	arp_ackunlnk(ab);
	bzero((caddr_t)ab, sizeof(struct arpb));
	return (0);
}

/*
 * prohibit the bottom arp data structure from being able to send data
 * called by the M_HANGUP code from a message received from lower stream
 */
static
arpbhangup(ab)
	struct arpb *ab;
{
	arp_ackunlnk(ab);
	return (0);
}

/*
 * given a pointer to a arp bottom structure, find the arp top data
 * structure with the same internet entry
 */
struct arp *
findtopq(ptr)
	struct arpb *ptr;
{
	struct arp *ap;

	ASSERT(ptr);

	for (ap = &arp[0]; ap < &arp[arp_cnt]; ap++)
		if (ptr->ab_inaddr == ap->ar_inaddr)
			return (ap);

	return ((struct arp *)NULL);
}

/*
 * given a cookie value, which is a unique integer assigned to a stream after
 * a I_LINK has been done, find a bottom structure with the same cookie.
 */
struct arpb *
arpmuxq(cookie)
	int cookie;
{
	struct arpb *ab;

	for (ab = &arpb[0]; ab < &arpb[arpb_cnt]; ab++)
		if (cookie == ab->ab_cookie)
			return (ab);

	return ((struct arpb *)NULL);
}

void
arpoput_mproto(q, bp)
	register queue_t    *q;
	register mblk_t     *bp;
{
	register struct arp        *ap = (struct arp *)q->q_ptr;
	register dl_unitdata_req_t *dp;

	dp = (dl_unitdata_req_t *)bp->b_rptr;
	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE,
		"arpoput: M_PROTO prim %x", dp->dl_primitive);
	switch (dp->dl_primitive)
	{
	/* ip requests to resolve some internet address */
	case DL_UNITDATA_REQ:
		(void) arp_resolve(ap, bp);
		break;

	/* ip requests to bind to top level queue */
	case DL_BIND_REQ:
		arp_bind_req(ap, bp);
		break;

	/* ip requests to unbind a top level queue */
	case DL_UNBIND_REQ:
		freemsg(bp);
		arp_unbind(ap);
		flushq(q, FLUSHALL);
		flushq(OTHERQ(q), FLUSHALL);
		break;

	/* ip requests to DL info for lower queues */
	case DL_INFO_REQ:
	{
		int s;

		freemsg(bp);
		s = splni();
		if (arp_info_req(ap) < 0)
			ap->ar_flags |= ARP_NEEDINFO;
		(void) splx(s);
		break;
	}

	default:
		freemsg(bp);
		break;
	}
	return;

}

void
arpoput_mioctl(q, bp)
	queue_t    *q;
	mblk_t     *bp;
{
	register struct arp     *ap = (struct arp *)q->q_ptr;
	register arpb_t         *arbp;
	register struct iocblk  *iop;
	register arpioctl       *iop1;
	register mblk_t         *bp1;
	register struct linkblk *lp;

	iop = (struct iocblk *)bp->b_rptr;
	if (ap->ar_rdq == NULL 
	|| (bp1 = bp->b_cont) == NULL) {
		iop->ioc_error = EIO;
nack:
		MTYPE(bp) = M_IOCNAK;
		qreply(q, bp);
		return;
	}
	iop1 = (arpioctl *)bp1->b_rptr;
	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE,
		"arpoput: M_IOCTL cmd %x user %d", iop->ioc_cmd, iop->ioc_uid);

	/*
	 * Must be root to perform some operations
	 */
	switch (iop->ioc_cmd)
	{
	case ARP_GET:
	case ARP_GETTABLEN:
	case ARP_GETARPTAB:
	case ARP_GETBOTQUE:
		break;

	default:
		if (iop->ioc_uid)
		{
			iop->ioc_error = EPERM;
			goto nack;
		}
	}

	switch (iop->ioc_cmd)
	{
	/* set TOP queue address and link up children */
	case ARP_INTOP:
	{
		int s;

		ap->ar_inaddr = iop1->at_in;
		/* Link up bottom queues to parent */
		s = splni();
		if (ap->ar_first == NULL)
		{
			arpb_t **abp;

			abp = &ap->ar_first;
			for (arbp = &arpb[0]; arbp < &arpb[arpb_cnt]; arbp++)
			{
				if (arbp->ab_inaddr == ap->ar_inaddr)
				{
					*abp = arbp;
					abp = &arbp->ab_next;
					arbp->ab_ar = ap;

					STRLOG(ARP_ID, -1, DPRI_HI, SL_TRACE, 
						"arpoput: linked arpb to ap for %x sap=%x",
						ap->ar_inaddr, arbp->ab_sap);
				}
			}
		}
		if (ap->ar_flags & ARP_NEEDINFO)
			(void) arp_info_req(ap);
		(void) splx(s);
		iop->ioc_count = 0;
		goto reply;
	}

	/* set my SAP type (ARP/IP/SNAP) */
	case ARP_INSAP:
		/*
		 * select the lower stream using the cookie hidden in
		 * the at_time field of the IOCTL data structure.
		 */
		arbp = arpmuxq((int)iop1->at_time);
		if (arbp == NULL)
			goto nack;
		arbp->ab_sap = iop1->at_in;
		arbp->ab_state = DL_UNBND;
		iop->ioc_count = 0;
		arpdl_info_req(arbp);
		goto reply;

	/* set my bottom queue internet address */
	case ARP_INSRC:
		/*
		 * select the lower stream using the cookie hidden in
		 * the at_time field of the IOCTL data structure.
		 */
		arbp = arpmuxq((int)iop1->at_time);
		if (arbp == NULL)
			goto nack;
		arbp->ab_inaddr = iop1->at_in;
		iop->ioc_count = 0;
		goto reply;

	/* set my bottom queue DLS provider PPA */
	case ARP_INPPA:
		/*
		 * select the lower stream using the cookie hidden in
		 * the at_time field of the IOCTL data structure.
		 */
		arbp = arpmuxq((int)iop1->at_time);
		if (arbp == NULL)
			goto nack;
		arbp->ab_ppa = iop1->at_in;
		iop->ioc_count = 0;
		goto reply;

	/* get ethernet address */
	case ARP_GET:
	{
		struct arptab *at;

		ARPTAB_LOOK(at, iop1->at_in, iop1->at_type);
		if (at == NULL) {
			iop->ioc_error = ENOENT;
			goto nack;
		}
		iop1->at_flags = at->at_flags;
		bcopy((char *)at->at_mac, (char *)iop1->at_mac, MACSIZE);
		/* set return value count for the ioctl */
		iop->ioc_count = sizeof(arpioctl);
		goto reply;
	}
		
	/* get number of ARP table entries */
	case ARP_GETTABLEN:
		iop->ioc_error = EIO;
		if (iop->ioc_count < sizeof arptab_cnt) 
			goto nack;
		iop->ioc_count = sizeof arptab_cnt;
		bp1->b_wptr    = bp1->b_rptr + sizeof arptab_cnt;
		*(int *)iop1   = arptab_cnt;
		goto reply;

	/* get an entry from the bottom queue table */
	case ARP_GETBOTQUE:
	{
		arpb_t abp;

		iop->ioc_error = EIO;
		if (iop->ioc_count < sizeof (arpb_t)) 
			goto nack;
		bcopy(bp1->b_rptr, (caddr_t)&abp, sizeof (arpb_t));
		if ((abp.ab_next < arpb) || (abp.ab_next >= &arpb[arpb_cnt]))
			goto nack;
		iop->ioc_count = sizeof (arpb_t);
		bcopy((caddr_t)abp.ab_next, (caddr_t)bp1->b_rptr, sizeof (arpb_t));
		bp1->b_wptr    = bp1->b_rptr + sizeof (arpb_t);
		goto reply;
	}

	/* get a section of the arp table */
	case ARP_GETARPTAB:
	{
		char    *at;
		int     bytes_left;
		int     bytes_to_copy;
		int     offset, number;

		offset = *(int *)bp1->b_rptr;
		number = *(int *)(bp1->b_rptr + sizeof (int));
		iop->ioc_error = EIO;
		if ((offset + number) > arptab_cnt)
			goto nack;
		iop->ioc_count = bytes_left = 
			min( iop->ioc_count, number * sizeof( struct arptab ) );
		at = (char *)&arptab[offset];
		while ( bytes_left > 0 && bp1 ) {
			bytes_to_copy = min( bytes_left, bp1->b_wptr - bp1->b_rptr );
			bcopy( at, bp1->b_rptr, bytes_to_copy );
			bytes_left -= bytes_to_copy;
			at         += bytes_to_copy;
			bp1->b_wptr = bytes_to_copy + bp1->b_rptr;
			bp1 = bp1->b_cont;
		}
		goto reply;
	}

	/* set record in the kernel table */
	case ARP_RESOLVE:
		if (iop->ioc_error = arp_install(iop1->at_in, iop1->at_mac,
				iop1->at_type, iop1->at_flags))
			goto nack;

		/* set return value count for the ioctl */
		iop->ioc_count = 0;
reply:
		MTYPE(bp) = M_IOCACK;
		iop->ioc_error = 0;
		qreply(q, bp);
		return;

	case ARP_DELETE:
		/*
		 * Delete an ARP table entry.
		 */
		if (iop->ioc_error = arp_delete((struct arptab *)iop1))
			goto nack;

		/* set return value count for the ioctl */
		iop->ioc_count = sizeof(arpioctl);
		goto reply;

	case I_LINK: 
	{
		queue_t        *bq;

		lp = (struct linkblk *)bp->b_cont->b_rptr;
		/*
		 * l_qbot is a pointer set up by STREAMS that points to
		 * the lower queue of the driver you just linked to
		 */
		bq = lp->l_qbot;
		/*
		 * internal open of an unused bottom q
		 */
		for (arbp = &arpb[0]; arbp < &arpb[arpb_cnt]; arbp++) 
		{
			if (arbp->ab_wrq == NULL)
			{
				bq->q_ptr = RD(bq)->q_ptr = (char *)arbp;
				arbp->ab_wrq = bq;
				/* save cookie for later arpmuxq calls */
				arbp->ab_cookie = lp->l_index;
				/* default everything to IP for now */
				arbp->ab_sap = IP_SAP;
				arbp->ab_state = DL_UNBND;
				/* set return value count for the ioctl */
				iop->ioc_count = 0;
				goto reply;
			}
		}
		goto nack;
	}

	case I_UNLINK:
		lp = (struct linkblk *)bp->b_cont->b_rptr;
		if ((arbp = arpmuxq(lp->l_index)) == NULL)
			goto nack;
		if (arbp->ab_wrq == NULL)
			goto nack;
		/* We must unbind everything */
		arbp->ab_flags &= ~ARPB_HANGUP;
		/*
		 * Delay the acking until all hadshakes with the
		 * lower streams are done, and all lower streams
		 * are closed. 
		 */
		arbp->ab_iocblk = bp;
		arpdl_unbind_req(arbp);
		return;

	default:  
		goto nack;
	}
}

/*
 * Arp output routine
 * - called when messages are put on arp's queue from upper streams
 * - handle all LLC requests from upper streams and pass on to lower streams
 * - LLC request DL_SENDDATA is ip asking for ARP resolution
 * - handle all request from user level arpbypass
 * - handle internal stream control messages M_CTL
 */
arpoput(q, bp)
	queue_t *q;
	mblk_t *bp;
{
	register struct arp     *ap = (struct arp *)q->q_ptr;
	register arpb_t         *arbp;

	ASSERT(ap && bp);
	ASSERT(bp != bp->b_cont);

	switch (MTYPE(bp)) {

	case M_PROTO:
	case M_PCPROTO:
		arpoput_mproto(q, bp);
		break;

	case M_IOCTL:
		arpoput_mioctl(q, bp);
		break;

	case M_CTL: 
	{
		ulong *myinaddr;
		ulong oldaddr;
		
		/*
		 * set internet address in arp top structure
		 * matches ip bottom structure internet address entries 1 to 1
		 */
		myinaddr = (ulong *)bp->b_rptr;
		ap->ar_inaddr = *myinaddr;
		STRLOG(ARP_ID, -1, DPRI_HI, SL_TRACE, 
			"arpoput: changed arp addr to %x", ap->ar_inaddr);
		
		/*
		 * If there was an old address, update existing table.
		 */
		if (bp->b_cont != NULL)
		{
			oldaddr = *(ulong *)bp->b_cont->b_rptr;
			for (arbp = ap->ar_first; arbp; arbp = arbp->ab_next)
			{
				if (arbp->ab_inaddr == oldaddr)
				{
					arbp->ab_inaddr = *myinaddr;

					STRLOG(ARP_ID, -1, DPRI_HI, SL_TRACE, 
						"arpoput: changed arbp=%x addr from %x to %x",
						arbp, oldaddr, arbp->ab_inaddr);
				}
			}
		}
		freemsg(bp);
		break;
	}

	default:
		STRlog(ARP_ID, -1, DPRI_LO, SL_te, "arpoput: type %x", MTYPE(bp));
		freemsg(bp);
	}
}

void
arpb_putnext(ab, bp)
register struct arpb *ab;
register mblk_t *bp;
{
	register struct arp *ap = ab->ab_ar;

	if (ap && ap->ar_state == DL_IDLE)
		putnext(ap->ar_rdq, bp);
	else
		freemsg(bp);
}

/*
 * Arp input routine
 * - called when lower streams puts messages on Arp's bottom input queue
 * - handles data for Arp packets as well as IP packets
 * - checks the sap value returned in the q pointers private data structure
 *   to tell who the packet is for.
 * - handles all LLC acks and indications
 */
arpbiput(q, bp)
	register queue_t *q;
	register mblk_t *bp;
{
	register struct arpb *ab = (struct arpb *)q->q_ptr;
	register struct arp *ar;
	register union DL_primitives *dp;

	ASSERT(q && bp && ab);

	switch (MTYPE(bp))
	{
	case M_PROTO:
	case M_PCPROTO:
		dp = (union DL_primitives *)bp->b_rptr;
		STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE,
			"arpbiput: get M_PROTO prim %x", dp->dl_primitive);

		switch (dp->dl_primitive)
		{
		case DL_UNITDATA_IND:
			switch (ab->ab_sap)
			{
			/* SNAP packet, decode and give to correct dest */
			case SNAP_SAP:
			{
				register struct llc_aemd *lp;

				lp = (struct llc_aemd *)
					((char *)dp + dp->unitdata_ind.dl_dest_addr_offset);
				lp->l_prot = ntohs(lp->l_prot);
				STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE,
					"arpbiput: SNAP proto=0x%x", lp->l_prot);
				if (lp->l_prot == ETHERPUP_ARPTYPE)
					(void) arp_rcvd(q, bp);
				else 
					arpb_putnext(ab, bp);
				break;
			}

			/* ip packet from the net, pass it on to ip */
			case ETHERPUP_IPTYPE:
			case IP_SAP:
				arpb_putnext(ab, bp);
				break;

			/* Arp packet from the net, process it */
			case ETHERPUP_ARPTYPE:
			case ETHERPUP_RARPTYPE:	/* SNH */
				(void) arp_rcvd(q, bp);
				break;

			default:
				freemsg(bp);
				break;
			}
			break;
			
		case DL_INFO_ACK:
			arpdl_info_ack(ab, bp);
			return;

		case DL_BIND_ACK:
			arpdl_bind_ack(ab, bp);
			return;

		case DL_OK_ACK:
			arpdl_ok_ack(ab, bp);
			return;

		case DL_ERROR_ACK:
		{
			dl_error_ack_t *ep = (dl_error_ack_t *)dp;

			STRlog(ARP_ID, -1, DPRI_LO, SL_te,
				"arpbiput: ERR_ACK, prim %x, LLC err %x, unix err %x",
				ep->dl_error_primitive, ep->dl_errno, ep->dl_unix_error);
			freemsg(bp);
			break;
		}

		case DL_UDERROR_IND:
			STRlog(ARP_ID, -1, DPRI_LO, SL_te,
				"arpbiput: UDERROR, type %x",
				dp->uderror_ind.dl_errno);
			freemsg(bp);
			break;

		default:
			STRlog(ARP_ID, -1, DPRI_LO, SL_te,
				"arpbiput: unknown DL_prim %x", dp->dl_primitive);
			freemsg(bp);
			break;
		}
		return;

	case M_ERROR:   
		arpdl_unbind_req(ab);
		if ((ar = findtopq(ab)) != NULL)
			(void) putctl1(ar->ar_rdq->q_next, M_ERROR, EPROTO);
		freemsg(bp);
		return;

	case M_HANGUP: 
		ab->ab_flags |= ARPB_HANGUP;
		arpdl_unbind_req(ab);
		if ((ar = findtopq(ab)) != NULL)
		{
			if (arpb_status(ar) == 2)
			{
				putnext(ar->ar_rdq, bp);
				return;
			}
		}
		break;

	case M_START:   
		if (ab->ab_flags & ARPB_HANGUP)
				arpdl_info_req(ab);
		else
		{
			ab->ab_flags &= ~(ARPB_HANGUP|ARPB_STOP);
			if ((ar = findtopq(ab)) != NULL)
			{
				if (arpb_status(ar) == 1)
				{
					putnext(ar->ar_rdq, bp);
					return;
				}
			}
		}
		break;

	case M_STOP:
	ab->ab_flags |= ARPB_STOP;
		if ((ar = findtopq(ab)) != NULL)
		{
			if (arpb_status(ar) == 2)
			{
				putnext(ar->ar_rdq, bp);
				return;
			}
		}
		break;

	default:        /* e.g. M_IOCACK, etc. */
		if ((ar = findtopq(ab)) != NULL)
			putnext(ar->ar_rdq, bp);
		return;
	}
	freemsg(bp);
}

/*
 * Check status of all sub-streams
 * returns:  0 - status unknown, 1 - all streams up, 2 - all streams down.
 */
arpb_status(ar)
	register struct arp *ar;
{
	register struct arpb *ab;
	int total, upcount;

	/* Search all streams */
	for (ab = ar->ar_first; ab; ab = ab->ab_next)
	{
		++total;
		if ((ab->ab_flags & (ARPB_HANGUP|ARPB_STOP)) == 0)
			++upcount;
	}

	/* Return status */
	if (total == upcount)
		return (1);
	if (upcount == 0)
		return (2);
	return (0);
}

/*
 * Timeout routine. Age arp_tab entries once a minute
 */
int arpfirst = 1;

arptimer()
{
	register struct arptab *at;
	register int i;
	int val;

	if (arpb[0].ab_wrq == NULL)
	{
		arpfirst = 1;
		return;
	}
	timeout(arptimer, (caddr_t)0, HZ*ARPT_AGE);

	at = &arptab[0];
	for (i = 0; i < arptab_cnt; i++, at++)
	{
		/* Do nothing to free or permanent entries */
		if ((at->at_in == 0) || (at->at_flags & ATF_PERM))
			continue;

		/* Decay entry */
		++at->at_time;
		if (at->at_flags & ATF_COM)
			val = ARPT_KILLC;
		else
			val = ARPT_KILLI;
		if (at->at_time < val)
			continue;

		/* Too old, delete it */
		(void) arp_delete(at);
	}
}

/*
 * receive an ARP data packet
 * q is the read q, hbp is a DL_unitdata_ind followed by ether_arp
 */
arp_rcvd(q, hbp)
	queue_t *q;
	mblk_t *hbp;
{
	register mblk_t *bp;
	register struct ether_arp *ea;
	register struct arpb *ap;
	register struct arptab *at;
	ulong spa, tpa;
	ushort format, pro;
	short sap;
	int s;

	ASSERT(q && hbp);

	s = splstr();	
	ap = (struct arpb *)q->q_ptr;
	bp = unlinkb(hbp);
	(void) freeb(hbp);
	if (pullupmsg(bp, sizeof(*ea)) == 0)
	{
		STRlog(ARP_ID, -1, DPRI_LO, SL_te,
			"arp_rcvd: pullupmsg failed. bp = 0x%x", bp);
		goto free;
	}
	ea = (struct ether_arp *)bp->b_rptr;
	STRlog(ARP_ID, -1, DPRI_LO, SL_te,
		"arp_rcvd: Entered. ea->arp_op=0x%x", htons(ea->arp_op));

	switch (ap->ab_type) {

	case DL_CSMACD:
		format = ARPHRD_IEEE;
		break;

	case DL_ETHER:
		format = ARPHRD_ETHER;
		break;

	default:
		goto bad;
	}
	sap = (short)ap->ab_sap;    /* 2 byte protocol type */
	pro = ap->ab_pro;           /* 2 byte protocol type */
	if (ntohs(ea->arp_pro) != pro)  {
		STRlog(ARP_ID, -1, DPRI_LO, SL_te,
			" bad  exp  %x   got  %x  ",pro, ntohs(ea->arp_pro));
		goto bad;
	}

	STRlog(ARP_ID, -1, DPRI_LO, SL_te, "arp_rcvd: sap=0x%x", sap);
	bcopy((caddr_t)ea->arp_spa, (caddr_t)&spa, sizeof(spa));
	bcopy((caddr_t)ea->arp_tpa, (caddr_t)&tpa, sizeof(tpa));

	switch(ntohs(ea->arp_op)) {

	case RARPOP_REPLY:
		break;

	case RARPOP_REQUEST:
		/*
		 * this is a reverse ARP request
		 */
		STRlog(ARP_ID, -1, DPRI_LO, SL_te,
			"arp_rcvd: RARP ap->ab_inaddr=0x%x", ntohl(ap->ab_inaddr));
		ARPTAB_ENLOOK(at, ea->arp_tha, ap->ab_type);
		if ((at == (struct arptab *)NULL) || !(at->at_flags & ATF_PUBL)) 
			break;

		/*
		 * generate response
		 */
		ea->arp_hrd = htons(format);
		ea->arp_pro = htons(pro);
		ea->arp_op = htons(RARPOP_REPLY);	/* SNH */
		tpa = at->at_in;        /* requester  */
		spa = ap->ab_inaddr;    /* my ip addr */
		bcopy((caddr_t)&tpa, (caddr_t)ea->arp_tpa, 4);
		bcopy((caddr_t)&spa, (caddr_t)ea->arp_spa, 4);
		bcopy((caddr_t)ea->arp_sha, (caddr_t)ea->arp_tha, MACSIZE);
		bcopy((caddr_t)ap->ab_sha, (caddr_t)ea->arp_sha, MACSIZE);
		sap = ETHERPUP_RARPTYPE;
		if ( arpdl_send( ap->ab_wrq, ap->ab_dltype, sap, ea->arp_tha, bp) )
			freemsg( bp );
		(void) splx(s);
		return;

	case ARPOP_REPLY:
		/*
		 * this is a reply, update my arptab
		 */
		STRlog(ARP_ID, -1, DPRI_LO, SL_te,
			"arp_rcvd: tpa=0x%x, ap->ab_inaddr=0x%x",
			tpa, ntohl(ap->ab_inaddr));

		/*
		 * This really is my reply?
		 */
		if (tpa != ap->ab_inaddr) {
			goto free;
		}

		/*
		 * See if this completes are partial REQUEST (when won't this be true?)
		 */
		ARPTAB_LOOK(at, spa, DL_CSMACD);

		/*
		 * Entry found, update/complete it
		 */
		if (at && !(at->at_flags & ATF_COM))
		{
			/* Complete the entry */
			bcopy((char *)ea->arp_sha, (char *)at->at_mac, MACSIZE);
			(void) arp_partial(ap, at);
		}
		else
		{
			(void) arp_install(spa, ea->arp_sha, ap->ab_type, ATF_COM);
		}
		break;

	case ARPOP_REQUEST:
		/*
		 * if spa is my internet address and hardware
		 * address is not the same, it is a forgery!!!
		 */
		if ((spa == ap->ab_inaddr) && bcmp(ea->arp_sha, ap->ab_sha, MACSIZE))
		{
			STRlog(ARP_ID, -1, DPRI_LO, SL_te,
				"arp_rcvd: forgery inet %x", spa);
			goto free;
		}

		/*
		 * Check to see if this is a proxy arp request. 
		 */
		ARPTAB_LOOK(at, spa, ap->ab_type);

		/* Entry found, update/complete it */
		if (at)
		{
			/* Fill in type and ethernet address (set to complete) */
			bcopy((char *)ea->arp_sha, (char *)at->at_mac, MACSIZE);
			(void) arp_partial(ap, at);
		}
		else if (tpa == ap->ab_inaddr)
		{
			(void) arp_install(spa, ea->arp_sha, ap->ab_type, ATF_COM);
		}

		if (tpa != ap->ab_inaddr) {
			if ((at == (struct arptab *)NULL) ||
				!(at->at_flags & ATF_PUBL) )
				break;
		}

		/*
		 * assemble a reply ether_arp message and send it out
		 */
		ea->arp_hrd = htons(format);
		ea->arp_pro = htons(pro);
		ea->arp_op = htons(ARPOP_REPLY);
		tpa = spa;              /* requester  */
		spa = ap->ab_inaddr;    /* my ip addr */
		bcopy((caddr_t)&tpa, (caddr_t)ea->arp_tpa, 4);
		bcopy((caddr_t)&spa, (caddr_t)ea->arp_spa, 4);
		bcopy((caddr_t)ea->arp_sha, (caddr_t)ea->arp_tha, MACSIZE);
		bcopy((caddr_t)ap->ab_sha, (caddr_t)ea->arp_sha, MACSIZE);
		sap = ETHERPUP_ARPTYPE;
		if ( arpdl_send( ap->ab_wrq, ap->ab_dltype, sap, ea->arp_tha, bp) )
			freemsg( bp );
		(void) splx(s);
		return;

	default:
bad:
		STRlog(ARP_ID, -1, DPRI_LO, SL_te,
			"arp_rcvd:bad arp, type %x, bp %x", ea->arp_op, bp);
	}

free:
	(void)freemsg(bp);
	(void) splx(s);
	return;
}

/*
 * update arptab: in/en in host representation
 * return -1 if failed, 0 if o.k.
 */
arp_install(in, en, type, flags)
	register ulong in;
	unchar *en;
	int type;
{
	register struct arptab *at, *oat;
	register int bkt;
	struct arptab *currat, *lastat;
	int sap, bkt_loop, s;

	ASSERT(en && in);

	STRlog(ARP_ID, -1, DPRI_LO, SL_te,
		"arp_install: Entered. in=0x%x, type=%d", ntohl(in), type);

	/*
	 * Protect table mods with spl.
	 */
	s = splstr();
	if (in == 0) {
bad:
		STRlog(ARP_ID, -1, DPRI_LO, SL_te,
			"arp_install: fail: inaddr %x, type %x", ntohl(in), type);
		(void) splx(s);
		return (EINVAL);
	}

	/*
	 * Kick the timer if this is the first time.    -ac 8/12/87
	 */
	if (arpfirst && (arpb[0].ab_wrq != NULL))
	{
		arpfirst = 0;
		timeout(arptimer, (caddr_t)0, HZ);
	}

	/*
	 * If we already have an entry then, update the entry.    XXXX
	 */
	bkt = ARP_HASH(in);
	for (at = arp_bhash[bkt]; at; at = at->at_next)
	{
		if ((at->at_in != in) || (at->at_type != type))
			continue;

		/* Record ethernet address */
		bcopy((char *)en, (char *)at->at_mac, MACSIZE);

		(void) splx(s);
		return (0);
	}

	/*
	 * Search for an empty spot where we can put this one.
	 */
	for (at = oat = &arptab[0] ; at < &arptab[arptab_cnt]; at++)
	{
		if (at->at_in == 0)
			goto Install;
		/*
		 * Keep track of the oldest unused one.
		 */
		if (oat->at_time < at->at_time)
			oat = at;
	}

	/*
	 * We could not find any empty buckets.
	 * Kick out the oldest entry.
	 */
	at = oat;

Install:
	/*
	 * Break old links if reusing existing entry
	 */
	if (at->at_in != 0)
		arp_unlink(at);

	/*
	 * Fill in arp entry with addresses, flags and interface type.
	 */
	bcopy((char *)en, (char *)at->at_mac, MACSIZE);
	at->at_in = in;
	at->at_type = type;
	at->at_flags = flags;
	at->at_next = NULL;
	at->at_time = 0;
	at->at_last = NULL;

	/*
	 * Sort based on ethernet/csmacd (in case we get both)
	 */
	arp_insert(at);
	
	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE,
		"arp_install: ok:in %x, type %x", ntohl(in), type);
	(void) splx(s);
	return (0);
	
}

/*
 * Insert arp entry into table (ordered based on CSMA/ETHER type)
 * Note: must be called with interrupts blocked!
 */
arp_insert(at)
	register struct arptab *at;
{
	register int bkt;

	ASSERT(at);

	/*
	 * Sort based on ethernet/csmacd (in case we get both)
	 */
	bkt = ARP_HASH(at->at_in);
	if (at->at_type == DL_CSMACD)
	{
		struct arptab **iap;

		for (iap = &arp_bhash[bkt]; *iap; iap = &((*iap)->at_next))
			;
		*iap = at;
	}
	else
	{
		at->at_next = arp_bhash[bkt];
		arp_bhash[bkt] = at;
	}

	return;
}
	
/*
 * Remove from hash table (but don't free) arp entry
 * Note: must be called with interrupts blocked!
 */
arp_unlink(at)
	register struct arptab *at;
{
	register int bkt;
	struct arptab *currat, *lastat;

	ASSERT(at);

	bkt = ARP_HASH(at->at_in);
	lastat = NULL;
	for (currat = arp_bhash[bkt]; currat; currat = currat->at_next)
	{
		if (currat == at)
		{
			if (lastat)
				lastat->at_next = at->at_next;
			else
				arp_bhash[bkt] = at->at_next;
			break;
		}
		lastat = currat;
	}

	return;
}

/*
 * Complete partial entry correctly
 * Note: must be called with interrupts blocked!  (B.S. !!  NBB )
 */
arp_partial(ab, at)
	struct arpb *ab;
	register struct arptab *at;
{
	unsigned short sap;

	ASSERT(ab && at);

	/* If packet waiting for resolution, then send it out */
	if (at->at_last) {
		if (ab->ab_dltype == DL_ETHER)
			sap = ETHERPUP_IPTYPE;
		else
			sap = IP_SAP;
		if (arpdl_send(ab->ab_wrq, ab->ab_dltype, sap, at->at_mac, at->at_last))
			freemsg(at->at_last);
		at->at_last = NULL;
	}

	/* Update entry */
	at->at_flags |= ATF_COM;
	at->at_time = 0;
	at->at_last = NULL;
	at->at_type = ab->ab_type;

	/* Move entry to correct place in table based on dltype */
	arp_unlink(at);
	arp_insert(at);

	return;
}

int
arp_uderror( ar, inaddr )
	register struct arp       *ar;
	ulong                     inaddr;
{
	register mblk_t           *dmp = NULL;
	register dl_uderror_ind_t *dep;

	/*
	 * Send up a DL_unitdata_req error indication (used for if_oerrors!)
	 */
	if ((dmp = allocb(DL_UDERROR_IND_SIZE + sizeof (long), BPRI_MED)) == NULL)
		return (-1);
	dmp->b_datap->db_type = M_PCPROTO;
	dmp->b_wptr = dmp->b_rptr + DL_UDERROR_IND_SIZE + sizeof (long);
	dep = (dl_uderror_ind_t *)dmp->b_rptr;
	dep->dl_primitive        = DL_UDERROR_IND;
	dep->dl_dest_addr_length = sizeof (long);
	dep->dl_dest_addr_offset = DL_UDERROR_IND_SIZE;
	bcopy(&inaddr, (caddr_t)dep + dep->dl_dest_addr_offset, sizeof (long));
	dep->dl_errno = ENOMEM;         /* XXXXX - really q full */
	putnext(ar->ar_rdq, dmp);
	return (-1);
}

/*
 * resolve an internet address (inaddr) of subnetwork's type "type"
 * if can't resolve, broadcast a ARP request
 * This routine is too internet dependent. 
 * It decides that a network address is 32 bits (XXXX).
 */
arp_resolve(ar, mp)
	register struct arp       *ar;
	register mblk_t           *mp;
{
	register struct arptab    *at;
	register struct ether_arp *ep;
	register arpb_t           *ab;
	ushort format, len;
	dl_unitdata_req_t         *dp;
	struct llc_ip             *llc;
	mblk_t                    *dmp = NULL;
	ulong                     inaddr;
	int                       bkt;
	int                       sap;
	int                       s;

	static struct arptab      *atcache;
	static struct arpb        *abcache;

	ASSERT(ar && mp);

	dp = (dl_unitdata_req_t *)mp->b_rptr;
	llc = (struct llc_ip *)(((caddr_t)dp) + dp->dl_dest_addr_offset);
	bcopy((caddr_t)&llc->l_addr, (caddr_t)&inaddr, sizeof (ulong));

	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE,
		"arp_resolve: inaddr %x", ntohl(llc->l_addr));
	if (llc->l_flag & IFF_BROADCAST)
	{
		/*
		 * Send a copy of broadcast message out all IP type bottom streams.
		 */
		for (sap = -1, ab = ar->ar_first; ab; ab = ab->ab_next)
			if ((ab->ab_wrq) 
			&&  ((ab->ab_sap == ETHERPUP_IPTYPE) || (ab->ab_sap == IP_SAP))
			&&  (ab->ab_inaddr == ar->ar_inaddr))  
			{

				/*
				 * Set sap based on DL type (SNAP?)
				 */
				if (ab->ab_dltype == DL_ETHER)
					sap = ETHERPUP_IPTYPE;
				else
					sap = IP_SAP;
				dmp = copymsg(mp);
				if (arpdl_send(ab->ab_wrq, ab->ab_dltype, sap, Bcastnid, mp)) {
					freemsg( mp );
					(void) arp_uderror( ar, inaddr );
				}
				if ((mp = dmp) == NULL)
					break;
			}
#if defined(NCR) || defined(ARIX)
		if (mp != NULL)
			arpdl_loop(ar, mp);
#else
		freemsg(mp);
#endif
		return (0);
	}

	/*
	 * Check CACHE entry: if still valid and same as last time, use it.
	 */
	s = splni();
	++arpcalls;
	if (abcache && (abcache->ab_wrq != NULL) &&
		atcache && (atcache->at_in == inaddr))
	{
		++arpcache;
		STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE,
			"arp_resolve: cache found, arpdl_send sap=0x%x", abcache->ab_sap);
		if (abcache->ab_dltype == DL_ETHER)
			sap = ETHERPUP_IPTYPE;
		else
			sap = IP_SAP;
		if (arpdl_send(abcache->ab_wrq, abcache->ab_dltype, sap,
					   atcache->at_mac, mp))
		{
			freemsg( mp );
			abcache = NULL;
			atcache = NULL;
			(void) splx(s);
			return ( arp_uderror( ar, inaddr ) );
		}
		(void) splx(s);
		return(0);
	}

	/*
	 * Search for destination in ARP address table.
	 */
	for (at = arp_bhash[ARP_HASH(inaddr)]; at; at = at->at_next) 
	{
		/*
		 * Should change the matching scheme to be network independent.
		 */
		if (at->at_in != inaddr)
			continue;

		/*
		 * Watch out for partial entry, must wait (can't do anything now)
		 */
		if (!(at->at_flags & ATF_COM))
		{
			if (at->at_last)
				freemsg(at->at_last);
			at->at_last = mp;
			mp = NULL;
			break;
		}

		/*
		 * reset arp timer (entry has been used)
		 */
		at->at_time = 0;

		/*
		 * send packet out IP/SNAP bottom structure.  Note that SNAP
		 * uses OLD ETHER type DL_UNITDATA_REQ messages!
		 */
		for (ab = ar->ar_first; ab; ab = ab->ab_next)
		{
			if ((ab->ab_flags & ARPB_IP) && (ab->ab_type == at->at_type))
			{
				if (ab->ab_dltype == DL_ETHER)
					sap = ETHERPUP_IPTYPE;
				else
					sap = IP_SAP;

				/* new cache entry */
				abcache = ab;
				atcache = at;

				if (arpdl_send(ab->ab_wrq, ab->ab_dltype, sap, at->at_mac, mp))
				{
					freemsg( mp );
					(void) splx(s);
					return ( arp_uderror( ar, inaddr ) );
				}
				(void) splx(s);
				return (0);
			}
		}
	}

	/*
	 * internet address is not found in the arp table, discard ip packet,
	 * and send ARP request packets down all arp streams.
	 */
	for (ab = ar->ar_first; ab; ab = ab->ab_next)
	{
		if (    (ab->ab_wrq ) 
#ifdef u3b2
		&& ((ab->ab_sap == ETHERPUP_IPTYPE) || (ab->ab_sap == IP_SAP) )
#else
		&&  (ab->ab_sap == ETHERPUP_ARPTYPE) 
#endif
		&&  (ab->ab_inaddr == ar->ar_inaddr))
		{

			switch (ab->ab_type)
			{
				case DL_CSMACD:
					format = ARPHRD_IEEE;
					break;

				case DL_ETHER:
					format = ARPHRD_ETHER;
					break;

				default:
					/* not supported yet */
					continue;
			}

			/*
			 * Construct ARP REQUEST packet
			 */
			len = max(ab->ab_minpsz, sizeof (struct ether_arp));
			if ((dmp = allocb(len, BPRI_MED)) == NULL)
			{
				freemsg(mp);
				(void) splx(s);
				return (-1);
			}
			dmp->b_wptr = dmp->b_rptr + len;
			ep = (struct ether_arp *)dmp->b_rptr;
			ep->arp_hrd = htons(format);
			ep->arp_pro = htons((ushort)ab->ab_pro);
			ep->arp_hln = sizeof(ep->arp_sha);
			ep->arp_pln = sizeof(ep->arp_spa);
			ep->arp_op = htons(ARPOP_REQUEST);
			bcopy(ab->ab_sha, ep->arp_sha, sizeof(ep->arp_sha));
			bcopy((caddr_t)&ab->ab_inaddr, ep->arp_spa, sizeof(ep->arp_spa));
			bcopy((caddr_t)&inaddr, ep->arp_tpa, sizeof(ep->arp_tpa));

			/*
			 * ETHERPUP_ARPTYPE tells NI to respond with REPLY's on
			 * the arp channel
			 */
			sap = ETHERPUP_ARPTYPE;
			STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE,
				"arp_resolve: request for addr=%x sap=%x dltype=%d",
				ntohl(inaddr), ab->ab_sap, ab->ab_dltype);
			if ( arpdl_send(ab->ab_wrq, ab->ab_dltype, sap, Bcastnid, dmp) )
				freemsg( dmp );
		}
	}

	/*
	 * Search for an empty ARP table slot to record partial entry.
	 * mp == NULL if partial ARP entry already exists.
	 */
	if (mp != NULL) {
		for (at = &arptab[0]; at < &arptab[arptab_cnt]; at++)
		{
			if (at->at_in == 0)
				goto install;
		}
	}
	(void) splx(s);
	freemsg(mp);
	return (-1);

install:
	/*
	 * We setup a temporary entry and save the message waiting to go out.
	 */
	ASSERT(mp);
	bkt = ARP_HASH(inaddr);
	at->at_in = inaddr;
	at->at_type = DL_CSMACD;
	at->at_flags = 0;
	at->at_last = mp;
	bzero((caddr_t)at->at_mac, MACSIZE);
	at->at_next = arp_bhash[bkt];
	arp_bhash[bkt] = at;
	(void) splx(s);
	return (-1);

}

/*
 * This routine deletes an arptab entry.
 */
arp_delete(at)
	register struct arptab *at;
{
	register struct arptab **p;
	register int s = splstr();

	ASSERT(at);

	for (p = &arp_bhash[ARP_HASH(at->at_in)]; *p ; p = &((*p)->at_next))
	{
		if ((at->at_in == (*p)->at_in) && (at->at_type == (*p)->at_type))
		{
			bcopy((char *)((*p)->at_mac), (char *)at->at_mac, MACSIZE);
			at->at_flags = (*p)->at_flags;
			if ((*p)->at_last)
				freemsg((*p)->at_last);
			(*p)->at_last = NULL;
			(*p)->at_in = 0;
			*p = (*p)->at_next;
			(void) splx(s);
			return (0);
		}
	}
	(void) splx(s);
	return (ENOENT);
}

/*
 * send unitdata to the ethernet, 
 * return 0 if o.k., -1 if failure
 * NB: if failure, it is the caller's responsibility to free ubp !!
 */
arpdl_send(q, type, sap, macaddr, ubp)
	register queue_t *q;        /* lower side write queue    */
			 short    type;     /* subnet type               */
			 short    sap;      /* my lsap number            */
			 unchar  *macaddr;  /* remote address (6 octets) */
			 mblk_t  *ubp;      /* the DL_UNITDATA_REQ       */
{
	register dl_unitdata_req_t *dp;
			 mblk_t  *bp;

	ASSERT(q && macaddr && ubp);

	/*
	 * make flow control work for IP
	 */
	if (!canput(q->q_next)) {
		++canput_dlarpsnd;
		STRlog(-1, -1, DPRI_LO, SL_te,
			"arpdl_send: canput_dlarpsnd=%d", canput_dlarpsnd);
		return (-1);
	}
#ifdef notdef
	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE,
		"arpdl_send: type=%d sap=%x mac=%x", type, sap, arp_etherprf(macaddr));
#endif

	/*
	 * Might need to allocate UNITDATA_REQ header?
	 */
	if (ubp->b_datap->db_type == M_DATA) {
#define DLP_REQSIZE     (DL_UNITDATA_REQ_SIZE + sizeof(struct llc_aemd))
		if ((bp = allocb(DLP_REQSIZE, BPRI_MED)) == NULL) {
			return (-1);
		}
		bp->b_datap->db_type = M_PROTO;
		linkb(bp, ubp);
		ubp = bp;
	}
	ubp->b_datap->db_type = M_PROTO;
	dp = (dl_unitdata_req_t *)ubp->b_rptr;
	dp->dl_primitive = DL_UNITDATA_REQ;
	dp->dl_dest_addr_offset = sizeof(dl_unitdata_req_t);

	switch(type) {

		case DL_CSMACD: {
			register struct llc_a *ap;

			ap = (struct llc_a *)((char *)dp + dp->dl_dest_addr_offset);
			bcopy(macaddr, ap->l_addr, MACSIZE);
			dp->dl_dest_addr_length = LSAP_SIZE;
			ap->l_sap = sap;
			break;
		}

		case DL_ETHER: {
			register struct llc_aemd *ap;

			ap = (struct llc_aemd *)((char *)dp + dp->dl_dest_addr_offset);
			bcopy(macaddr, ap->l_addr, MACSIZE);
			dp->dl_dest_addr_length = EMDSAP_SIZE;
			ap->l_prot = htons((ushort)sap);
			break;
		}

		default:
			STRlog(-1, -1, DPRI_LO, SL_te,
				"arpdl_send: unsupport type %x", type);
			return (-1);
	}
	
	ubp->b_wptr = ubp->b_rptr + DL_UNITDATA_REQ_SIZE + dp->dl_dest_addr_length;
	putnext(q, ubp);
	return (0);
}


#if defined(NCR) || defined(ARIX)
/*
 * Convert into a unitdata_ind and send back upstream
 */
arpdl_loop(ar, mp)
	register struct arp *ar;
	register mblk_t     *mp;
{
	register dl_unitdata_ind_t *dlp;
	register mblk_t *bp;

	ASSERT(ar && mp);

	/* Remove old unitdata_req header */
	bp = unlinkb(mp);
	freeb(mp);

	/* Allocate a new header */
	if ((mp = allocb(DL_UNITDATA_IND_SIZE, BPRI_MED)) == NULL)
	{
		freemsg(bp);
		return (-1);
	}
	mp->b_datap->db_type = M_PROTO;
	mp->b_wptr = mp->b_rptr + DL_UNITDATA_IND_SIZE;
	linkb(mp, bp);

	/* fill in the unitdata_ind header */
	dlp = (dl_unitdata_ind_t *)mp->b_rptr;
	dlp->dl_primitive = DL_UNITDATA_IND;
	dlp->dl_dest_addr_offset = 0;
	dlp->dl_dest_addr_length = 0;
	dlp->dl_src_addr_offset = 0;
	dlp->dl_src_addr_length = 0;

	/* send it up */
	putnext(ar->ar_rdq, mp);

	return (0);
}
#endif /* ARIX || NCR */

/*
 * send info_req to ETHERNET, 'q' is write q
 * return 0 if o.k., -1 if failure
 */
arpdl_info_req(ab)
	register struct arpb *ab;
{
	register mblk_t *bp;
	register dl_info_req_t *dp;

	ASSERT(ab);

	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE, "arpdl_info_req(ab=%x)", ab);

	/* allocate message */
	if ((bp = allocb(DL_INFO_REQ_SIZE, BPRI_MED)) == NULL)
		return(-1);

	/* build the request */
	MTYPE(bp) = M_PCPROTO;
	dp = (dl_info_req_t *)bp->b_rptr;
	dp->dl_primitive = DL_INFO_REQ;
	bp->b_wptr += DL_INFO_REQ_SIZE;
	putnext(ab->ab_wrq, bp);
	return(0);
}

/*
 * response from ETHERNET to info_req (switch LLI or DLPI)
 * return 0 if o.k., -1 if failure
 */
arpdl_info_ack(ab, bp)
	register struct arpb *ab;
	register mblk_t *bp;
{
	register dl_info_ack_t *dp;

	ASSERT(ab && bp);

	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE,
		"arpdl_info_ack(ab=%x,bp=%x)", ab, bp);

	/* update interface info */
	dp = (dl_info_ack_t *)bp->b_rptr;
	ab->ab_dltype = ab->ab_type = dp->dl_mac_type;
	ab->ab_maxpsz = dp->dl_max_sdu;
	ab->ab_minpsz = dp->dl_min_sdu;

	/* record protocol type for ARP messages */
	switch (dp->dl_mac_type)
	{
		case DL_CSMACD:
			ab->ab_pro = IP_SAP;
			break;
		case DL_ETHER:
			ab->ab_pro = ETHERPUP_IPTYPE;
			break;
		default:
			STRlog(ARP_ID, -1, DPRI_LO, SL_te,
				"arpbiput: unknown info_ack type %x", dp->dl_mac_type);
			break;
	}

	/* Update individual stream state */
	switch (ab->ab_sap)
	{
		/* Special case for SNAP!!! It is ETHERTYPE! */
		case SNAP_SAP:
			ab->ab_dltype = DL_ETHER;
			ab->ab_flags |= (ARPB_IP|ARPB_ARP);
			ab->ab_pro = ETHERPUP_IPTYPE;
			break;

		case ETHERPUP_ARPTYPE:
			ab->ab_flags |= ARPB_ARP;
			ab->ab_sap = ETHERPUP_ARPTYPE;
			break;

		case ETHERPUP_IPTYPE:
			ab->ab_flags |= ARPB_IP;
			ab->ab_sap = ETHERPUP_IPTYPE;
			break;

		case IP_SAP:
			ab->ab_flags |= ARPB_IP;
			ab->ab_sap = IP_SAP;
			break;

		default:
			break;
	}
	freemsg(bp);

	/* if unbound then either attach or bind next */
	if (ab->ab_state == DL_UNBND)
	{
		/* DLPI/LLI compatibility */
		if ((bp->b_wptr - bp->b_rptr) >= DL_INFO_ACK_SIZE)
		{
			if (dp->dl_provider_style == DL_STYLE1)
				ab->ab_flags &= ~ARPB_PPA;
			else
				ab->ab_flags |= ARPB_PPA;
		}
		else
		{
			ab->ab_flags &= ~ARPB_PPA;
		}

		/* if request is from a DLPI provider, then issue attach */
		if (ab->ab_flags & ARPB_PPA)
			arpdl_attach_req(ab);
		else
			arpdl_bind_req(ab, ab->ab_sap);
	}
	else
	{
		(void) arp_info_req(ab->ab_ar);
	}

	return;
}

/*
 * send DLPI attach request to DLS provider.
 * return 0 if o.k., -1 if failure
 */
arpdl_attach_req(ab)
	register struct arpb *ab;
{
	register mblk_t *bp;
	register dl_attach_req_t *dp; 

	ASSERT(ab);

	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE, "arpdl_attach_req(ab=%x)", ab);

	/* allocate message */
	if ((bp = allocb(DL_ATTACH_REQ_SIZE, BPRI_MED)) == NULL)
		return (-1);

	/* build the request */
	MTYPE(bp) = M_PROTO;
	dp = (dl_attach_req_t *)bp->b_wptr;
	dp->dl_primitive = DL_ATTACH_REQ;
	dp->dl_ppa = ab->ab_ppa;
	bp->b_wptr += DL_ATTACH_REQ_SIZE;
	putnext(ab->ab_wrq, bp);
	return (0);
}

/*
 * send a bind request to the LLC provider
 * q is my write queue and sap is service acces point I'm binding to
 */
arpdl_bind_req(ab, sap)
	register struct arpb *ab;
	unsigned short sap;
{
	register mblk_t *bp;
	register dl_bind_req_t *dp;

	ASSERT(ab);

	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE, "arpdl_bind_req(ab=%x)", ab);

	/* allocate message */
	if ((bp = allocb(DL_BIND_REQ_SIZE, BPRI_MED)) == NULL)
		return(-1);
	ab->ab_state = DL_WACK_B;

	/* build the request */
	MTYPE(bp) = M_PROTO;
	dp = (dl_bind_req_t *)bp->b_rptr;
	dp->dl_primitive = DL_BIND_REQ;
	dp->dl_service_mode = DL_CLDLS;
	dp->dl_conn_mgmt = 0;
	dp->dl_sap = (unsigned long)sap;
	bp->b_wptr += DL_BIND_REQ_SIZE;
	putnext(ab->ab_wrq, bp);
	return(0);
}

/*
 * bind response
 */
arpdl_bind_ack(ab, bp)
	register struct arpb *ab;
	mblk_t *bp;
{
	register dl_bind_ack_t *dp;
	struct arp *ar;

	ASSERT(ab && bp);

	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE,
		"arpdl_bind_ack(ab=%x,bp=%x)", ab, bp);

	/* stream now ready, save ethernet address and put in IDLE state */
	dp = (dl_bind_ack_t *)bp->b_rptr;
	ab->ab_state = DL_IDLE;
	bcopy((char *)dp + dp->dl_addr_offset, ab->ab_sha, MACSIZE);
	freemsg(bp);

	/* notify upper level that we have restarted */
	if (ab->ab_flags & ARPB_HANGUP)
	{
		ab->ab_flags &= ~(ARPB_HANGUP|ARPB_STOP);
		if ((ar = findtopq(ab)) != NULL)
		{
			if (arpb_status(ar) == 1)
			{
				(void) putctl(ar->ar_rdq, M_START);
				return;
			}
		}
	}

	/* reissue inforeq to update min/max correctly */
	arpdl_info_req(ab);
	return;
}

/*
 * unbind a lsap # to the ETHERNET, 'q' is write q
 * return 0 if o.k., -1 if failure
 */
arpdl_unbind_req(ab)
	register struct arpb *ab;
{
	register mblk_t *bp;
	register dl_unbind_req_t *dp;

	ASSERT(ab);

	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE, "arpdl_unbind_req(ab=%x)", ab);

	/* allocate message */
	if ((bp = allocb(DL_UNBIND_REQ_SIZE, BPRI_MED)) == NULL)
		return (-1);

	/* build the request */
	MTYPE(bp) = M_PROTO;
	dp = (dl_unbind_req_t *)bp->b_rptr;
	dp->dl_primitive = DL_UNBIND_REQ;
	bp->b_wptr += DL_UNBIND_REQ_SIZE;
	putnext(ab->ab_wrq, bp);
	return(0);
}

/*
 * ok_ack: process known packets
 * return 0 if o.k., -1 if failure
 */
arpdl_ok_ack(ab, bp)
	register struct arpb *ab;
	mblk_t *bp;
{
	register dl_ok_ack_t *dp;

	ASSERT(ab && bp);

	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE, "arpdl_ok_ack(ab=%x,bp=%x)", ab, bp);

	/* Switch on type of response */
	dp = (dl_ok_ack_t *)bp->b_rptr;
	switch (dp->dl_correct_primitive)
	{
		case DL_UNBIND_REQ:
			ab->ab_state = DL_UNBND;
			if (ab->ab_flags & ARPB_PPA)
				arpdl_detach_req(ab);
			else
			{
				if (ab->ab_flags & ARPB_HANGUP)
					arpbhangup(ab);
				else
					arpbclose(ab->ab_wrq);
			}
			break;

		case DL_ATTACH_REQ:
			arpdl_bind_req(ab, ab->ab_sap);
			break;

		case DL_DETACH_REQ:
			ab->ab_state = DL_UNBND;
			if (ab->ab_flags & ARPB_HANGUP)
				arpbhangup(ab);
			else
				arpbclose(ab->ab_wrq);
			break;
	}
	freemsg(bp);
	return;
}

/*
 * send DLPI detach request to DLS provider.
 * return 0 if o.k., -1 if failure
 */
arpdl_detach_req(ab)
	register struct arpb *ab;
{
	register mblk_t *bp;
	register dl_detach_req_t *dp;

	ASSERT(ab);

	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE, "arpdl_detach_req(ab=%x)", ab);

	/* allocate message */
	if ((bp = allocb(DL_DETACH_REQ_SIZE, BPRI_MED)) == NULL)
		return (-1);

	/* build the request */
	MTYPE(bp) = M_PROTO;
	dp = (dl_detach_req_t *)bp->b_rptr;
	dp->dl_primitive = DL_DETACH_REQ;
	bp->b_wptr += DL_DETACH_REQ_SIZE;
	putnext(ab->ab_wrq, bp);
	return (0);
}

/*
 * send bind response back up to 'q'.
 * return 0 if o.k., -1 if failure
 */
arp_bind_req(ar, bp)
	register struct arp *ar;
	register mblk_t *bp;
{
	register dl_bind_ack_t *dp;
	unsigned long sap;

	ASSERT(ar && bp);

	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE, "arp_bind_req(ar=%x)", ar);

	/* get requested sap and free message */
	sap = ((dl_bind_req_t *)bp->b_rptr)->dl_sap;
	freemsg(bp);
	if ((bp = allocb(DL_BIND_ACK_SIZE + sizeof(ulong), BPRI_MED)) == NULL)
		return(-1);

	/* build the ack */
	bp->b_datap->db_type = M_PCPROTO;
	dp = (dl_bind_ack_t *)bp->b_rptr;
	dp->dl_primitive = DL_BIND_ACK;
	dp->dl_sap = sap;
	dp->dl_addr_length = sizeof(ulong);
	dp->dl_addr_offset = DL_BIND_ACK_SIZE;
	bcopy((caddr_t)&ar->ar_inaddr,
		((caddr_t)dp + dp->dl_addr_offset), sizeof(long));
	bp->b_wptr = bp->b_rptr + DL_BIND_ACK_SIZE + sizeof(ulong);
	putnext(ar->ar_rdq, bp);
	ar->ar_sap = sap;
	ar->ar_state = DL_IDLE;
	return(0);
}

/*
 * send unbind response back up to 'q'.
 * return 0 if o.k., -1 if failure
 */
arp_unbind(ar)
	register struct arp *ar;
{
	register dl_ok_ack_t *dp;
	register mblk_t *bp;

	ASSERT(ar);

	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE, "arp_unbind_req(ar=%x)", ar);

	/* free request and get response */
	if ((bp = allocb(DL_OK_ACK_SIZE, BPRI_MED)) == NULL)
		return(-1);

	/* build the ack */
	bp->b_datap->db_type = M_PCPROTO;
	dp = (dl_ok_ack_t *)bp->b_rptr;
	dp->dl_primitive = DL_OK_ACK;
	dp->dl_correct_primitive = DL_UNBIND_REQ;
	bp->b_wptr = bp->b_rptr + DL_OK_ACK_SIZE;
	putnext(ar->ar_rdq, bp);
	ar->ar_state = DL_UNBND;
	return(0);
}

/*
 * send info_req response back up to 'q'.
 * return 0 if o.k., -1 if failure
 */
arp_info_req(ar)
	register struct arp *ar;
{
	register dl_info_ack_t *dp;
	register struct arpb *ab;
	register mblk_t *bp;
	int size, pmax = 0xffff, pmin = 0;
	int s;

	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE, "arp_info_req(ar=%x)", ar);

	/* In case we are still setting up links */
	if (ar == NULL)
		return (-1);

	/* search bottom queues for best MIN/MAX values */
	s = splni();
	if (ar->ar_first == NULL)
	{
		STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE, "arp_info_req: !linked");
		(void) splx(s);
		return (-1);
	}
	for (ab = ar->ar_first; ab; ab = ab->ab_next)
	{
		if (ab->ab_state == DL_IDLE)
		{
			pmin = max(pmin, ab->ab_minpsz);
			pmax = min(pmax, ab->ab_maxpsz);
		}
		else
		{
			STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE, "arp_info_req: !DL_IDLE");
			(void) splx(s);
			return (-1);
		}
	}
	ar->ar_flags &= ~ARP_NEEDINFO;
	(void) splx(s);

	/* Conform to DLPI spec, only return address when in unbound state */
	if (ar->ar_state == DL_UNBND)
		size = DL_INFO_ACK_SIZE + sizeof(ulong);
	else
		size = DL_INFO_ACK_SIZE;

	/* free request and allocate response */
	if ((bp = allocb(size, BPRI_MED)) == NULL)
		return (-1);

	/* Make the DL_INFO_ACK message */
	bp->b_datap->db_type = M_PCPROTO;
	dp = (dl_info_ack_t *)bp->b_rptr;
	dp->dl_primitive = DL_INFO_ACK;
	dp->dl_max_sdu = pmax;
	dp->dl_min_sdu = pmin;
	dp->dl_mac_type = DL_ETHER;
	dp->dl_reserved = 0;
	dp->dl_current_state = (ulong)ar->ar_state;
	dp->dl_max_idu = 0;
	dp->dl_service_mode = DL_CLDLS;
	dp->dl_provider_style = DL_STYLE1;
	if (ar->ar_state == DL_UNBND)
	{
		dp->dl_addr_length = sizeof(ulong);
		dp->dl_addr_offset = DL_INFO_ACK_SIZE;
		bcopy((caddr_t)&ar->ar_inaddr,
			((caddr_t)dp + dp->dl_addr_offset), sizeof(ulong));
	}
	else
	{
		dp->dl_addr_length = 0;
		dp->dl_addr_offset = 0;
	}
	bp->b_wptr = bp->b_rptr + size;
	putnext(ar->ar_rdq, bp);
	STRLOG(ARP_ID, -1, DPRI_LO, SL_TRACE, "arp_info_req: ack sent up");
	return (0);
}

static char *
arp_etherprf(addr)
	u_char *addr;
{
	static char string[40];
	static char convert[] = "0123456789ABCDEF";
	char *p;
	int i;

	p = string;
	for (i = 1; i <= MACSIZE; i++)
	{
	p[0] = convert[*addr >> 4]; 
	p[1] = convert[*addr & 0xf]; 
	if (i < MACSIZE)
		p[2] = ':'; 
	p += 3;
	++addr;
	}

	return (string);
}

/* end */

/*
 * Ack any pending I_UNLINK ioctl.
 * ab is the pointer to the private structure 
 * of the arp bottom queue where ioctl message
 * is temporary recorded. 
 */
static void
arp_ackunlnk(ab)
struct arpb *ab;
{
	register mblk_t *bp;
	register struct iocblk *iop;
	register struct linkblk *lp;

	ASSERT (ab);
	bp = ab->ab_iocblk;
	if (bp) {
		iop = (struct iocblk *)bp->b_rptr;	
		lp = (struct linkblk *)bp->b_cont->b_rptr;
		MTYPE(bp) = M_IOCACK;
		iop->ioc_error = 0;
		iop->ioc_count = 0;
		ab->ab_iocblk = NULL;	
		qreply(lp->l_qtop, bp);
	}
}
