/*****************************************************************************
 *
 * ICB --- Interprocessor Communication Bus
 *
 * the arete 1000 architecture includes an interprocessor communication
 * bus (ICB).  the ICB master is able to address any board in the card cage
 * with an ICB interface.
 *****************************************************************************/
#define SLOTSTART (0x200000 + IOSPACE)  /* this is the first slot's icb address */
#define NUMSLOTS 32 /* maximum number of slots possible */
#define SLOTSHFT 16 /* shift value for converting slot address to slot num */

struct icb {
	unsigned char dpmem[0xfffc]; /* ICB dual ported memory */
	unsigned char bdtypreg; /* board type register */
	unsigned char tasreg; /* test and set status register */
	unsigned short intreg; /* interrupt register */
};

/* interrupt register  (write)
	X X X X  X X X X  X X X X  X X X X b
	| | | |  | | | |  | | | |  | | | |
	| | | |  interrupt data  (12 bits)
	| | | halt
	| | interrupt iop
	| nmi to iop
	reset iop
 */
#define IDATMSK   0xff  /* interrupt data mask */
#define HLTIOP    0x1000  /* halt the i/o processor */
#define INTIOP    0x2000  /* interrupt the i/o processor */
#define NMIIOP    0x4000  /* non-maskable interrupt the i/o processor */
#define RSTIOP    0x8000  /* reset the i/o processor */

/* interrupt register  (read)
	X X X X  X X X X  X X X X  X X X X b
	| | | |  | | | |  | | | |  | | | |
	| | | |  interrupt data  (12 bits)
	| | | not used
	interrupt request level

 * note:
 * reading this register clears an ICB interrupt from this card
 */
#define IRQLEVMSK 0xe000 /* interrupt request level mask */
#define IRQLEVSH  13     /* interrupt request level shift */

/* board type register (read only)
	X X X X  X X X X b
	| | | board type
	| | not used
	| iop interrupt pending
	icb interrupt pending
 */
#define BDTYPMSK  0x1f /* board type mask */
#define IOPIPND   0x40 /* i/o processor interrupt pending */
#define ICBIPND   0x80 /* icb interrupt pending */

/* test and set register  (read)
	X X X X  X X X X b
	| | | |  | | | |
	| | | currently not used
	| | IOP ready
	| IOP fault
	test and set status (read 0-yours, 1-taken)

note: a write to this register clears the TAS bit for the master	
 */
#define IOPRDY    0x20 /* IOP ready (master may access ICB dual port mem */
#define IOPFLT    0x40 /* IOP fault */
#define TASBIT    0x80 /* test and set bit */
#define GETIODPMEM(X) {while(((struct icb *)X)->tasreg & TASBIT);}
#define RELIODPMEM(X) {((struct icb *)X)->tasreg = 1;}

/* board type defines */
#define MCPUBDTYP  0x00 /* master cpu */
#define SCPUBDTYP  0x01 /* slave cpu */
#define DMCBDTYP   0x02 /* dma memory controller */
#define DTCBDTYP   0x03 /* disk tape controller */
#define ASBDTYP    0x04 /* asynchronous serial */
#define HSDTBDTYP  0x05 /* 68k based disk tape controller */
#define MEMBDTYP   0x06 /* memory */
#define GCPBDTYP   0x07 /* general purpose communication processor */
#define MACBDTYP   0x08 /* general purpose multibus adapter card */
#define DMC4BDTYP  0x09 /* dmc4 dma memory controller */
#define MCPU32BDTP 0x0a /* master cpu32 (68020) */
#define SCPU32BDTP 0x0b /* slave cpu32 (68020) */
#define EDTBDTYP   0x0c /* enhanced 68k based disk tape controller */
#define DMC42BDTYP 0x0d /* 2 channel dmc4 dma memory controller */
#define SCSIBDTYP  0x10 /* SCSI EDT disk tape controller */

/*
 *	board vectors
 */

#define VECDTC 0
#define VECHSDT 0
#define VECASP 1
#define VECGCP 1
#define VECMAC 1
#define VECDMC 2
#define VECCPU 3


