
/******************************************************************************
 *
 * GCP --- General purpose Communications Processor
 *
 ******************************************************************************/

#define XRAM      0x10000  /* expansion ram starting address */
#define XRAMSIZ   0x20000  /* expansion ram size */
#define EGCRMTST  0xc0000  /* egc expansion ram test address */
#define EGCRAM    0x80000  /* egc expansion ram starting address */
#define SMEGCRAM  0x80000  /* small egc expansion ram starting address */
#define EGCRAMSIZ 0xc0000  /* egc expansion ram size */
#define SMEGCRMSZ 0x40000  /* small egc expansion ram size */
#define FLOATRAM  0x30000  /* floating ram starting address */
#define FLOATSIZ  0xfff0   /* floating ram size */
#define FLIPBAS   0x40000  /* flip ram starting address */
#define FLIPSIZ   0x10000  /* flip ram size */

/*******************************************************************
 *
 *	BOARD CONTROL REGISTER
 *
 *******************************************************************/

#define BDCNTRL ((unsigned char *)0x60001)
/*
	X X X X   X X X X b     (write only)
	| | | |   | | | |
	| | | |   | | led bits
	| | | |   | memory refresh enable*
	| | | |   parity enable
	| | | floating memory enable
	| | iop ready
	| memory configuration
	reset interface card*
 */
#define LEDBITS  3    /* led bits */
#define LEDYEL   0    /* light the yellow led */
#define LEDGRN   1    /* light the green led */
#define LEDRED   2    /* light the red led */
#define NOLED    3    /* all led's dark */
#define _RFRENA  0x04 /* memory refresh enable (negative logic) */
#define PARENA   0x08 /* memory parity enable */
#define FLOATM   0x10 /* floating memory enable */
#define IOPRDY   0x20 /* set the iop ready bit for the icb master */
#define FLIPMEM  0x40 /* if set ram at 0, prom at 4 meg */
#define _IFRESET 0x80 /* interface card reset (negative logic) */

/*******************************************************************
 *
 *	ICB STATUS REGISTER
 *
 *******************************************************************/

#define ICBSTATREG ((unsigned char *)0x60004) /* icb status register (read only) */
/*
	X X X X   X X X X b
	| | | |   | | | |
	| | | |   | unused (always 0)
	| | interface board type
	| iop interrupt pending
	icb interrupt pending
 */
#define IFBTMSK    0x38 /* interface board type mask */
#define IFBTSHFT   3    /* interface board type shift */
#define IOPIPNDG   0x40 /* iop interrupt pending */
#define ICBIPNDG   0x80 /* icb interrupt pending */

/*******************************************************************
 *
 *	TEST AND SET REGISTER
 *
 *******************************************************************/

#define TASREG ((unsigned char *)0x60005) /* TAS status register (read only) */
/*
	X X X X   X X X X b
	| | | |   | | | |
	| | | slot number
	| | memory configuration
	| iop faulted
	test and set

 * note: reading this register with tas == 0 locks the test and set
 * mechanism for you. 1 means it is taken.
 */
#define SLTNUMMSK  0x1f /* slot number mask */
#define MEMCONF    0x20 /* current state of memory configuration */
#define IOPFAULT   0x40 /* iop fault */
#define TASBIT     0x80 /* test and set bit */
#define GETDPMEM   {while(*TASREG & TASBIT);}

/*******************************************************************
 *
 *	ICB INTERRUPT REGISTER
 *
 *******************************************************************/

#define ICBINTREG  ((unsigned short *)0x60006) /* icb interrupt register */
/*
	X X X X   X X X X   X X X X   X X X X b     (read/write)
	| | | |   | | | |   | | | |   | | | |
	| | | |   icb interrupt data area----
	| | | unused
	icb interrupt level

 * writing this register interrupts the master.
 * reading resets an interrupt from the master.
 */
#define IIDATMSK  0x0fff /* icb interrupt data mask */
#define IILEVMSK  0xe000 /* icb interrupt level mask */
#define IILEVSH   13     /* icb interrupt level shift */

/*******************************************************************
 *
 *	OTHER REGISTERS
 *
 *******************************************************************/

#define TASCLR     ((unsigned char *)0x60000)  /* clear the test and set bit (write only) */
#define RELDPMEM   {*TASCLR = 1;}
#define FLOATBAS   ((unsigned char *)0x60003)  /* base address for the floating memory (write only) */
#define SLOTSHFT   16   /* shift value for FLOATBAS */
#define MFLTCLR    ((unsigned char *)0x60002)  /* memory fault clear (write only) */
#define IFCTLREG   ((unsigned short *)0x60004) /* interface control register (write only) */

/*******************************************************************
 *
 *	M E M O R Y    M A P P E D    I / O
 *
 *******************************************************************/

#define SCC0B     ((struct scc *)0x50000)
#define SCC0A     ((struct scc *)0x50020)
#define SCC1B     ((struct scc *)0x52000)
#define SCC1A     ((struct scc *)0x52020)
#define SCC2B     ((struct scc *)0x54000)
#define SCC2A     ((struct scc *)0x54020)
#define SCC3B     ((struct scc *)0x56000)
#define SCC3A     ((struct scc *)0x56020)
#define CIO       ((struct cio *)0x58000)
#define RTC_TSMS  16     /* real time clock time slice in milliseconds */
#define RTC_TC    (RTC_TSMS*2000)  /* real time clock time constant */
#define RTC_RNDUP(x) ((x+(RTC_TSMS-1))/RTC_TSMS) /* round up value */
#define NUMSCCS   4      /* number of SCC's */
#define NUMSCHAN  (NUMSCCS*2) /* number of SCC channels to program */

/*******************************************************************
 *
 *	I N T E R R U P T    V E C T O R S
 *
 *******************************************************************/

#define SCCSTVECT 64   /* starting vector for the SCC's */
#define ADJSCCVECT(I) (SCCSTVECT + ((I & 0xfe) << 3)) /* adjust SCC vector */
#define PPORTAVEC 128
#define PPORTBVEC 144
#define CTVECT    160

/*******************************************************************
 *
 * CPU INTERRUPT PRIORITY LEVELS
 *
 *******************************************************************/

#define NMIPLEV   7  /* ICB NMI */
#define ICBIPLEV  6  /* ICB */
#define MFIPLEV   5  /* memory fault */
#define CIOIPLEV  3  /* counter timer chip */
#define SG0IPLEV  2  /* serial group 0 */
#define SG1IPLEV  1  /* serial group 1 */
