/*****************************************************************************
 *
 * CPU --- Arete 1000 cpu board
 *
 * herein is a port, memory, and bit description of the cpu board
 *
 * the cpu has memory mapped i/o, a four megabyte 'hole' in the 16
 * megabyte address space which contains all of the ports and special
 * memories (tags,maps,etc.)
 * this hole has been moved in the past, so I found it necessary to
 * define everything as an offset from a base -- 'IOSPACE'
 *
 *****************************************************************************/

#include "iospace.h"
#define PROMSIZ 	0x10000		/* size of the cpu prom */
#define MPAGESH		12			/* memory page shift value */
#define MPAGESIZ	(4*1024)	/* memory page size */
#define MAXMPAGE	(4*1024)	/* maximum number of memory pages */

/*****************************************************************************
 *
 * the cpu cache memory is directly accessable in the iospace
 *
 *****************************************************************************/
#define CACHESTART ((unsigned *)(0x00000 + IOSPACE))  /* cache memory */
#define CACHELWORDS (1*1024)  /* 4k bytes */

/*****************************************************************************
 *
 * map ram defines
 *
 *****************************************************************************/
#define MAPSTART ((unsigned *)(0x80e00 + IOSPACE))  /* memory maps */
#define MAPLWORDS (4096)  /* 4k bytes (not byte accessable) */
#define MAPTESTMASK 0x0fff0fff  /* only 12bits/word valid (see below)  */

/* below is a description of one map entry (address and access bits)
	X X X X  X X X X  X X X X  X X X X  X X X X  X X X X  X X X X  X X X X b
	| | | |  | | | |  | | | |  | | | |  | | | |  | | | |  | | | |  | | | |
	| | | |  | | | |  | | | |  | | | |  | | | |  physical address bits 23 - 12 (page number)
	| | | |  | | | |  | | | |  | | | |  not used
	| | | |  | | | |  | process identification
	| | | |  | | | |  write protect
	| | | |  | | | accessed (this memory block has been accessed)
	| | | |  | | memory block has been written
	| | | |  | block is cacheable
	| | | |  code page
	not used
 */
#define MPPAMSK   0x00000fff /* a map entry's physical address mask */
#define MPPIDMSK  0x007f0000 /* a map entry's process id mask */
#define MPPIDSH   16         /* a map entry's process id shift */
#define MPWRPROT  0x00800000 /* a map entry's write protect bit */
#define MPACCSD   0x01000000 /* a map entry's accessed bit */
#define MPWRITN   0x02000000 /* a map entry's written bit */
#define MPCCHBL   0x04000000 /* a map entry's cacheable bit */
#define MPCDPG    0x08000000 /* a map entry's code page bit */

/*****************************************************************************
 *
 * tag ram defines
 *
 *****************************************************************************/
#define TAG1START ((unsigned *)(0xc0000 + IOSPACE))
#define TAG2START ((unsigned *)(0xc1000 + IOSPACE))
#define TAGLWORDS (1*1024)
#define TAGMASK   0x00000fff  /* only 12bits/lword valid */

/* below is a description of one tag entry
	X X X X  X X X X  X X X X  X X X X  X X X X  X X X X  X X X X  X X X X b
	| | | |  | | | |  | | | |  | | | |  | | | |  | | | |  | | | |  | | | |
	| | | |  | | | |  | | | |  | | | |  | | | |  tag data (physical page number)
	unused   unused   unused   unused   unused
 */

/*****************************************************************************
 *
 * there are two peripheral devices on the cpu board, an acia (asynchronous
 * communications interface adapter) and a ptm (programmable timer module)
 * below are their port numbers, bit descriptions can be found in the Motorola
 * microprocessors manual
 *
 *****************************************************************************/
#define ACIASTAT ((char *)(0x20011 + IOSPACE))
#define ACIADATA ((char *)(0x20013 + IOSPACE))
#define ACIACNTL ((char *)(0x20011 + IOSPACE))
#define PTMSTAT ((char *)(0x20003 + IOSPACE))
#define PTMCNT1 ((char *)(0x20001 + IOSPACE))
#define PTMCNT2 ((char *)(0x20003 + IOSPACE))
#define PTMCNT3 ((char *)(0x20001 + IOSPACE))
#define PTMMSB1 ((char *)(0x20005 + IOSPACE))
#define PTMLSB1 ((char *)(0x20007 + IOSPACE))
#define PTMMSB2 ((char *)(0x20009 + IOSPACE))
#define PTMLSB2 ((char *)(0x2000b + IOSPACE))
#define PTMMSB3 ((char *)(0x2000d + IOSPACE))
#define PTMLSB3 ((char *)(0x2000f + IOSPACE))
#define ACIA ((struct acia *)(0x20010 + IOSPACE))
#define PTM ((struct ptm *)(0x20000 + IOSPACE))
#define RTC_TSMS 16 /* time slice in milliseconds */
#define RTC_TC (RTC_TSMS * 1250) /* time constant */
#define RTC_RNDUP(x) ((x + (RTC_TSMS-1))/RTC_TSMS) /* round up value */

/*****************************************************************************
 *
 * cpu system port
 *
 *****************************************************************************/
#define SYSPORT ((char *)(0xe0001 + IOSPACE))
/* the enable port
	X X X X  X X X X b
	| | | |  | | | |
	| | | |  | | | PS.CTRL.0
	| | | |  | | PS.CTRL.1
	| | | |  | alternate register select
	| | | |  prom disable
	| | | cache enable
	| | updater enable
	| ac fail interrupt enable
	serial port handshake enable
 */
#define SPSTWKLO   0x01 /* system power supply tweak lo */
#define SPSTWKHI   0x02 /* system power supply tweak hi */
#define SPSSHTDWN  0x03 /* system power supply shut down */
#define ALTREGSEL  0x04 /* alternate register select */
#define CPROMDIS   0x08 /* cpu prom disable */
#define CCACHENA   0x10 /* cpu cache enable */
#define CUPDTRENA  0x20 /* cpu updater enable */
#define SACFLIENA  0x40 /* system ac fail interrupt enable */
#define CSERHENA   0x80 /* cpu serial port handshake enable */

/*****************************************************************************
 *
 * cpu map port
 *
 *****************************************************************************/
#define MAPPORT ((char *)(0xa000f + IOSPACE))
/* the map enable port
	X X X X  X X X X b
	| | | |  | | | |
	| | | |  | | | map enable
	| | | |  | | page stats enable
	| | | |  | id fault enable
	| | | |  write protect fault enable
	| | | code access fault enable
	| | data access fault enable
	not used
 */
#define MAPENA   0x01 /* enable the maps */
#define PGSTSENA 0x02 /* page stats enable */
#define IDFLTENA 0x04 /* id fault enable */
#define WPFLTENA 0x08 /* write protect fault enable */
#define CAFLTENA 0x10 /* code access fault enable */
#define DAFLTENA 0x20 /* data access fault enable */


/*****************************************************************************
 *
 * cpu memory manager offset registers
 * byte wide map offset registers for system and user
 *
 *****************************************************************************/
#define SYSOFSREG ((char *)(0xa0001 + IOSPACE))
#define USROFSREG ((char *)(0xa0003 + IOSPACE))

/*****************************************************************************
 *
 * sys/usr id registers (7 valid bits)
 *
 *****************************************************************************/
#define SYSIDREG ((char *)(0xa0009 + IOSPACE))
#define USRIDREG ((char *)(0xa000b + IOSPACE))
#define IDMASK   0x7f

/*****************************************************************************
 *
 * pmb segment register extends main memory address space
 * (address bits 24 thru 31)
 *
 *****************************************************************************/
#define PMBSEGREG ((char *)(0xa000d + IOSPACE))

/* the following registers allow master/slave communication over the icb */
#define SLVSTATREG ((char *)(0xe0005 + IOSPACE))  /* slave's access to icb (read clears int from master) */
#define SLVINTREG  ((char *)(0xe0007 + IOSPACE))  /* slave to master interrupt */

/*****************************************************************************
 *
 * bus error fault register
 *
 *****************************************************************************/
#define FLTREG ((char *)(0xe0007 + IOSPACE))
/* bus error fault register (resets when read)
	X X X X  X X X X b
	| | | |  | | | |
	| | | |  | | | process id fault
	| | | |  | | write protect fault
	| | | |  | code access fault
	| | | |  data access fault
	| | | fatal memory error
	| | icb bus error
	| icb time out
	unused
 */
#define PIDFLT    0x01 /* process id fault */
#define WRPFLT    0x02 /* write protect fault */
#define CDACCFLT  0x04 /* code access fault */
#define DTACCFLT  0x08 /* data access fault */
#define FTLMEMERR 0x10 /* fatal memory error */
#define ICBBUSERR 0x20 /* icb bus error */
#define ICBTIMOUT 0x40 /* icb time out error */

/*****************************************************************************
 *
 * system status register
 *
 *****************************************************************************/
#define SYSSTATREG ((char *)(0xe0001 + IOSPACE))
/* system status register (must be polled)
	X X X X  X X X X b
	|              |
	|              master cpu
	| 
	|
	|
	|
	|
	|
	ac failure
 */
#define _MSTRCPU  0x01 /* master cpu (negative logic) */
#define INTACFAIL 0x80 /* ac failure */
 
/*   for ALTREGSEL = 0   ( write bit 2 of the same register ) */

#define PSACFAIL   0x02
#define BPACFAIL   0x04
#define BACKUP     0x08  /* backup present */
#define AUTOSTART  0x10  /* autostart  */
#define VERYHOT    0x20
#define ANYHOT     0x40

/*  for ALTREGSEL = 1   */

#define CRDCGHOT  0x02   /*  card cage hot  */
#define PSHOT     0x04  /*  ps hot   */
#define DISK4HOT  0x08
#define DISK3HOT  0x10
#define DISK2HOT  0x20
#define DISK1HOT  0x40


/*****************************************************************************
 *
 * the cpu boards have an icb interface that can be addressed using
 * struct icb (in icb.h).
 * below are the bit descriptions of the above two registers.
 *
 *****************************************************************************/
/* cpu interrupt register (read/write)
	X X X X   X X X X   X X X X   X X X X b
	| | | |   | | | |   | | | |   | | | |
	| | | |   | | | |   interrupt data (8 bits)
	| | | |   | icb interrupt level
	| | | pmb priority
	| | interrupt the slave
	| halt slave*
	reset*

 * special note:
 * reading this register clears an ICB interrupt from this board
 * the high byte of this register is write only
 * a read of the high byte will not cause an error but will yield garbage
 */
/* use IDATMSK (icb.h) for an interrupt data mask */
#define ICBILMSK   0x0700 /* icb interrupt level mask */
#define ICBILSH    8      /* icb interrupt level shift */
#define PMBPRIMSK  0x1800 /* pmb priority mask */
#define PMBPRISH   11     /* pmb priority shift */
#define INTSLV     0x2000 /* interrupt the slave */
#define _HLTSLV    0x4000 /* halt the slave (negative logic) */
#define _RSTSLV    0x8000 /* reset the slave (negative logic) */


/* cpu board type register (read only)
	X X X X   X X X X b
	| | | board type register
	| | unused
	| slave interrupting master
	master is interrupting slave
 */
/* use BDTYPMSK (icb.h) for a board type mask */
#define SLVIPND   0x40 /* the slave is interrupting the master */
#define MSTRIPND  0x80 /* the master is interrupting the slave */

/*****************************************************************************
 *
 * configuration and LED ports
 *
 *****************************************************************************/
/* the old cpu board has a byte wide led status port */
#define LEDPORT ((char *)(0xe0003 + IOSPACE)) /* negative logic */
/* the new cpu board has a nibble wide led status port */
#define NEWLEDPORT ((char *)(0xa0005 + IOSPACE))

/* NEWLEDPORT
	X X X X  X X X X
	| | | |  | | | |
	| | | |  | | encoded leds
	| | | |  | sw0
	| | | |  sw1
	not used

encoding is as follows:
*/
#define YELLOW 0
#define GREEN  1
#define RED    2
#define NONE   3

/* the user configurable switches (CONFIGPORT) have been removed
   from the new cpu board and are no longer supported on either board */



