#include "hsdtccpu.h"
#include "hsdtdisksect.h"
#include "hsdt.h"
#include "hsdtdevstr.h"
#include "hsdtdevdq.h"
#include "hsdtstruct.h"
#include "hsdtdtc.h"
#include "hsdtptm.h"
#include "hsdterror.h"
#include "hsdtextrn.h"
#include "vreg.h"
#include "data_struct.h"


/*
	interrupts are disabled when entering this routine
	if the req has the wait for completeion option, then interrupts
		are enabled upon return
	insure the unit is selected and ready before seeking

	if unitstatus returns cyl not on, or seek error
		rezero the drive, and set rc1

	if unit status returns device check bit set, send
		fault clear command, and set rc1
	if unit is not ready then set rc1

	Then setup cylinder to seek
	if we are not waiting for seek completion, return 0, 
	    	interrupts are still disabled

	if we are waiting for seek completion, enable interrupts and wait 
		to check for error after the seek
	
 */
aseek (dev, sptr)
register struct devq *dev;
register struct seek_per_disk *sptr;
{
	register retcode; register unsigned short status;
	register struct smdwr *wr; register struct smdrd *rd;
	register struct dtb_que *dptr;

	enter_short_cr;

	*(short *)&dev->rc1 = 0;
	
	wr = (struct smdwr *)DK_BUSW + dev->q_devnum;

	/* insure drive is selected and ready to operate */

	if ( retcode = unitsel (dev->q_devnum )) {

		if ( retcode == UNIT_NOT_RDY )
			dev->rc1 = DER_NRDY;
		else if ( (retcode == CYLNON) || (retcode == SEEKER ) ) {
			if(dkrezero (dev->q_devnum, sptr))
				dev->rc1 = DER_CYLNON;
		}
	
		else if ( retcode == DEVCK )  {
			wr->control = FAULTCLR;
			dev->rc1 = DER_DEVCK;
		}
	}

	/* ******************************************* */
	/* following code should never happen, because the */
	/* unitsel() routine has already checked to insure */
	/* that the disk is ok 				 */
	/* ******************************************* */
	rd =(struct smdrd *)wr;
	status = rd->unitstus & 0x0f;
	if ( (status != 0x03)) {
		printf ("s1");
	}
	if ( (*DDC_SR & 0x80) != 0x80 ) {
		/* ************************ */
		/* disk interface is hung   */
		/* ************************ */
		printf ("s2");
		*DDC_CR = 0;
		*DDC_CR  = ENDKDATAC;
		dev->rc1 = DER_ABNOM;
	}

	if ( dev->rc1 ) {
		printrc (dev);
		/* we need to enable interrupts */
		exit_short_cr;
		return(1);
	}


	/* ready to seek */

	/* ************************************ */
	/* program clock to monitor             */
	/* if there is a timeout, we need to reset disk drive */

	sptr->s_time =  TICK_1SEC;

	/* ************************************ */


	/* seek a cylinder */
	wr->cylinder = dev->q_devun.pdisk.cyl;
	wr->seekend = ENSEEKEND;


	/* only return if we are sending to main memory and we are not waiting
	   for completion */

	seekque.seek_active |= BIT(dev->q_devnum);
	if ( (dev->q_flag & OPEN_DISK) == 0 ){
		exit_short_cr;
		return(0);		/* interrupts are disabled on return */
	}


	/* allow the seek interrupt to happen */
	exit_short_cr;

	/* wait for the seek to complete on a specific drive */
	/* if the seek never completes, a disk timeout will reset 
	    the timer */
	while ( sptr->s_time );

	/* set return code if there is a seek error */

	enter_short_cr;
	retcode = checkseek(dev, sptr);
	sptr->current_cyl = dev->q_devun.pdisk.cyl;
	if((sptr->s_first = dev->q_next) == (struct devq *)0)
		sptr->s_last = (struct devq *)0;
	if(dev->opr > OPR_WRITE){
		exit_short_cr;
		return(retcode);
	}
	dptr = &dtbque;
	if(dptr->d_first)
		dptr->d_last->q_next = dev;
	else
		dptr->d_first = dev;
	dptr->d_last = dev;
	dptr->dtb_active |= DISK;
	exit_short_cr;
	if(dev->opr < OPR_READ)
		retcode = rw_the_id(dev);
	else
		retcode = diskrw(dev);

	return (retcode);	/* interrupts are enabled, return 0 or 1 */
}


/* interrupts are disabled when entering this routine */

checkseek(dev, sptr)
register struct devq *dev;
register struct seek_per_disk *sptr;
{
	register unsigned short status;
	register struct smdwr *wr; register struct smdrd *rd;
	register i;


	/* ********************** */
	/* turn off tick for disk */
	/* ********************** */
	sptr->s_time = 0;

	rd = (struct smdrd *)DK_STSR + dev->q_devnum;

	/* insure seek end , cyl on, drive ready */
	for (i=0; i<300000; i++) {
		if ( (status = rd->unitstus) & SEEKEND ) 
			break;
	}


	/* after seek, unit ready (bit 0) , cyl on  (bit 1)= true, 
			seek error (bit 2), device check (bit 3) = false */
 
	if ( (status & SEEKMASK) == SEEKOK )
		return (0);

	else				/* may not have settled */
		for(i=0;i<513;i++)
		    if(!(i & 077))	/* check every 64 counts */
			if((rd->unitstus & SEEKMASK) == SEEKOK) 
				return(0);


	if ( (status & UNIT_RDY) == 0 ) 
		dev->rc1 = DER_NRDY;
	/* check error after seek */
	else if ( (status & SEEKER) || ((status & CYLNON) == 0) ) {
		dkrezero (dev->q_devnum, sptr);
		dev->rc1 = DER_SEEK;
	}
	else if ( status & DEVCK ) {
		wr = (struct smdwr *)DK_BUSW + dev->q_devnum;
		wr->control = FAULTCLR;
		dev->rc1 = DER_DEVCK;
	}
	else
		/* at this point we are not sure what the error is but
		    we will give it a general seek error return code */

		dev->rc1 = DER_SEEK;

	if ( PRINT1 ){
		printf ("s5(%x)", status);
	 	printrc (dev);
	}
	return (1);
}

/*
	check the status word off the physical drive
 output:
	if unit is selected and ready, return 0
	if unit is not ready , return 1(UNIT_NOT_RDY)
	if we are not on cylinder, return 2(CYLNON)
	if seek errror bit is set, return 4(SEEKER)
	if device check is set, return 8(DEVCK)
	
 */
unitsel (drive)
int drive;
{
	register struct smdrd *p;
	register unsigned short status;

	p = (struct smdrd *)DK_STSR + drive;

	drivein = 0;
	/* read unit status off the physical drive */
	status = p->unitstus;

	/* ********************************* */
	/* check if this causes any bus error */
	/* if a bus error occurs here, the pc   */
	/* will get pushed on the stack      */
	/* ********************************* */
	if ( drivein || ((status&0x8000) == 0)) { /* finer test(drive enable) */
		drivein = 0;
		return (UNIT_NOT_RDY);
	}

	if ( (status & SEEKMASK) == SEEKOK )
		return (0);

	if ( (status & UNIT_RDY) == 0 )
		return (UNIT_NOT_RDY);
	if ( (status & CYLNON) == 0 )
		return (CYLNON);
	if ( status & SEEKER )
		return (SEEKER);
	if ( status & DEVCK )
		return (DEVCK);
	printf ("s3(%x)",status);
	return (-1);
}

/*
	setup DDC timing ID register
	input:	pointer to integer array containing timing data
 */
setimid( ptr)
register char *ptr;
{ 
	register char *timingreg;
	register int idcount;

	timingreg = DDC_TR;

	idcount = TIMIDCNT;
	while ( idcount-- ){
		*timingreg = *ptr++;
		timingreg+=2;
	}
}

/* *********************************** */
/* rezero will wait for seek completion */
/* interrupts are disabled */
/* *********************************** */

dkrezero (drive,sptr)
register char drive;
register struct seek_per_disk *sptr;
{
	register struct smdwr *wr; register struct smdrd *rd;
	register i, timeout;
	register unsigned short status;

	sptr->current_cyl = -1;
	sptr->s_time = 0;

	if ( PRINT1 )
		printf ("dkrezD=%x ", drive);


	wr = (struct smdwr *)DK_BUSW + drive;
	wr->control = REZERO | FAULTCLR;
	wr->seekend = ENSEEKEND;

	rd = (struct smdrd *)wr;
	
	/* put in 600ms (600000 us) time out */
	
	timeout = 1;
	for (i=0; i<300000; i++) { 
		if ( (status = rd->unitstus) & SEEKEND ) {
			timeout = 0;
			break;
		}
	}

	/* disable 'enable seek end' interrupt bit */
	wr->seekend = 0;
	seekque.seek_active &= ~(UPPERBIT(sptr->s_first->q_devnum));

	if ( timeout ) {
		printf ("s4");
		return(-1);
	}

	/* insure SEEKEND, and CYLON, and UNITREADY */

	/* put in 1200ms (1200000 us) time out */
	for (i=0; i<600000; i++) {
		if ( (status = rd->unitstus) & SEEKEND )
			break;
	}

	if ( (status & SEEKMASK) == SEEKOK ){
		if(seekque.disk_seek_ptr == (struct seek_per_disk *)0
			&& sptr->s_first) /*sanity check */
			seekque.disk_seek_ptr = sptr;
		return (0);
	}

	printf (" REZer(%x)", status);
	return (-1);
}

setup_and_wait(dev)
register struct devq *dev;
{
	register struct seek_per_disk *sptr;
	register struct seek_que *seek_que_ptr;
	register struct dtb_que *dptr;
	register unsigned int i;
	
	seek_que_ptr = &seekque;
	sptr = &seek_que_ptr->s_p_d[dev->q_devnum];
	if ( sptr->current_cyl != dev->q_devun.pdisk.cyl ) {
		enter_short_cr;
		if(sptr->s_first)
			sptr->s_last->q_next = dev;
		else{
			sptr->s_first = dev;
			sptr->sort_ptr = dev;
		}
		sptr->s_last = dev;
		sptr->sort_count++;
		dev->q_next = (struct devq *)0;
		seek_que_ptr->disk_seek_ptr = sptr;
		seek_que_ptr->seek_active |= UPPERBIT(sptr->s_first->q_devnum);
		exit_short_cr;
		start_seek(sptr);
		enter_short_cr;
		seek_que_ptr->seek_active &= ~(UPPERBIT(dev->q_devnum));
		for(i=0;i<MAX_PDRIVE;i++){
			sptr = sptr->next_seek_per_disk_ptr;
			if(sptr->s_first && !(seek_que_ptr->seek_active & 
		    		UPPERBIT(sptr->s_first->q_devnum))) {
				seek_que_ptr->disk_seek_ptr = sptr;
	    				break;
			}
			seek_que_ptr->disk_seek_ptr = 0;
		}
		exit_short_cr;
	}
	else{
		dptr = &dtbque;
		enter_short_cr;
		if(dptr->d_first)
			dptr->d_last->q_next = dev;
		else
			dptr->d_first = dev;
		dptr->d_last = dev;
		dptr->dtb_active |= DISK;
		seek_que_ptr->seek_active |= UPPERBIT(dev->q_devnum);
		exit_short_cr;
		if(dev->opr < OPR_READ)
			rw_the_id(dev);
		else
			diskrw(dev);
		seek_que_ptr->seek_active &= ~(UPPERBIT(dev->q_devnum));
	}
}

