/*----------------------------------------------------------------------------
/ scsitest.c - SCSI drive test functions
/
/ Craig J. Kim      June 1988
/ (c) Copyright 1988 ARIX Corp.  San Jose, CA
/---------------------------------------------------------------------------*/

#include "scsidt.h"

int scsi_test_drive();                  /* main test routine */
int ecc();                              /* set/clear ECC bit */
int read_blocks();                      /* read disk blocks */
int write_blocks();                     /* write disk blocks */
static int  reassign_blocks();          /* reassign bad blocks */
static int  add_bad();                  /* add a block to bad block list */
static void patinit();                  /* initialize a buffer */
static int compare_bufs();             /* compare two buffers */
static int dumpbuf();                  /* hex dump a buffer */
static void showstat();                 /* show current disk test status */
static void edit_buf();                 /* edit a buffer */

static char *str_dt_term = "Do you wish to terminate the disk test";
static char *str_toomany = "Too many blocks.  Must be less than $%x.\n";
static char *str_startbk = "Enter starting block (in hex)";
static char *str_toobigb = "Block number exceeds disk capacity.  Must be less than $%x.\n";
static char *str_pattern = "Enter buffer pattern (in hex)";
static char *str_patnblk = "Enter number of blocks to initialize (in hex)";


/*---------------------------------------------------- scsi_test_drive() ---*/
int scsi_test_drive(start, numsec, command)
unsigned int start;     /* starting sector number */
unsigned int numsec;    /* number of sectors to test */
unsigned int command;   /* how to test */
{
        static char *rd = "Read";
        static char *wrt = "Write";
        register unsigned int currentsec;
        register int retcode;
        register int retry = 0;
        uint endsec, sectrack = MthrBlk.sechead;
        uint bytetrack;

        ecc(FALSE);
        if (scsi_drv[scsi_drvtype].maxbuf
                && (MthrBlk.sechead > scsi_drv[scsi_drvtype].maxbuf))
            sectrack = scsi_drv[scsi_drvtype].maxbuf;
        else
            sectrack = MthrBlk.sechead;
        bytetrack = sectrack * MthrBlk.bytesec;

        printf("\r%s Pass %d - Pattern = ", command == PREADPS ? rd : wrt, Pass);
        if (DBPat)
            printf("DEBUG\n");
        else {
            printf("$%8x\n", Pattern);
            patinit(CompBuf, bytetrack, 0, Pattern);
        }
        endsec = start + numsec;
        for (currentsec = start; currentsec < endsec; currentsec += sectrack) {
            unsigned int c, opsec, opbytes;

            if (endsec - currentsec > sectrack)
                opsec = sectrack;
            else
                opsec = endsec - currentsec;
            if ((opbytes = opsec * sectsiz) > bytetrack) {
                DBG(printf(">something funny going on here [%d] --> ", opbytes);)
                DBG(pgchar();)
                opbytes = bytetrack;
                opsec = bytetrack / sectsiz;
            }

            if (DBPat)
                patinit(CompBuf, bytetrack, currentsec, 0);
            if ((c = conin()) == 's')
                showstat(currentsec);
            else if (c == 'z')
                flipdebug();
            if (usrabort()) {
                showstat(currentsec);
                if (getyn(str_dt_term)) {
                    reassign_blocks();
                    ecc(TRUE);
                    return(FALSE);
                }
            }
            retry = 0;

#ifndef DEBUG
            xprintf("\r%d", currentsec);
#endif

            if (command == PREADPS)
                retcode = sc_read(TrackBuffer, currentsec, opsec);
            else
                retcode = sc_write(CompBuf, currentsec, opsec);

            if (retcode) {
                uint sec;

                DBG(printf("\n>sc_%s: $%x", command == PREADPS ? rd : wrt, retcode);)
                printf("\nSoft %s Error! - %s block %x(h)\n",
                        command == PREADPS ? rd : wrt, sc_derrmsg(errno),currentsec);
		++SoftDiskErr;
                for (sec = currentsec; sec < currentsec + sectrack; sec++) {
                    retry = 1;
                    while (retcode && retry < RETRYCOUNT) {
                        if (command == PREADPS)
                            retcode = sc_read(TrackBuffer, sec, 1);
                        else
                            retcode = sc_write(CompBuf, sec, 1);

                        if (retcode) {
                            retry++;
                            printf("\n%s %s Error! - %s\n",
                                    (retry >= RETRYCOUNT) ? "Hard" : "Soft",
                                    command == PREADPS ? rd : wrt, sc_derrmsg(errno));
			    if(retry < RETRYCOUNT)  /* soft disk error */
			        ++SoftDiskErr;
			    else {
				SoftDiskErr -= RETRYCOUNT; /* compensate for the hard error */
			        ++HardDiskErr;
			    }
                        }
                        else if (command == PREADPS && Compare) { 
                            if (!compare_bufs(sectsiz, sec)) {
                                if (getyn(str_dt_term)) {
                                    reassign_blocks();
                                    ecc(TRUE);
                                    return(FALSE);
                                }
                            }
                        }
                    }
                    if (retcode)        /* bad block */
                        add_bad(sec);
                }
            }
            else if (command == PREADPS && Compare)
                compare_bufs(opbytes, currentsec);
        }
        ecc(TRUE);
        return(TRUE);
}

/*---------------------------------------------------- ecc() -----------------
/ Turns on/off ECC - for disk test, this bit should be off
/---------------------------------------------------------------------------*/
ecc(set)
int set;
{
        register int retcode;

	if (!scsi_drv[scsi_drvtype].s_select)   /*  skip if no select support */
            return(TRUE);

        xprintf("Reading current error recovery page\n");
        retcode = sc_modesense(SCSI_V_PC_CUR, SCSI_V_MS_ERROR, &merror);
        if (retcode) {
            DBG(printf(">After MOD SEN:  $%x", retcode);            )
            printf("\n%s\n", sc_derrmsg(errno));
            printf("Unable to read error recovery page.");
            return(FALSE);
        }

        merror.datalen = 0;             /* blame micropolis drives! */
        merror.wp = 0;
        merror.bdlen = 8;
        merror.density = 0;
        merror.errec = 0;
        if (set) {
            if (scsi_drv[scsi_drvtype].s_autospare) {
                merror.errec |= SCSI_M_ER_AWRE;
                merror.errec |= SCSI_M_ER_ARRE;
            }
            merror.retry = SCSI_V_MAXRETRY;
        }
        else {
            merror.retry = 1;           /* try only once */
            merror.errec |= SCSI_M_ER_PER;
            merror.errec |= SCSI_M_ER_DCR;
        }
        merror.corrspan = 0;
        merror.headoff = 0;
        merror.strobe = 0;
        merror.tlimit = 0;

        xprintf("Writing error recovery page\n");
        retcode = sc_modeselect(SCSI_V_MS_ERROR, &merror);
        if (retcode) {
            DBG(printf(">After MOD SEL:  $%x", retcode);            )
            printf("\n%s\n", sc_derrmsg(errno));
            printf("Unable to write error recovery page.");
            return(FALSE);
        }

        xprintf("ECC bit is now %s.\n", set ? "SET" : "CLEARED");
        return(TRUE);
}

/*---------------------------------------------------- read_blocks() ---------
/ A debug function - reads blocks of user specified address on demain
/---------------------------------------------------------------------------*/
read_blocks()
{
        register int block = 0, numblks = 1, c, retcode;
        uint pattern;

        while (1) {
redo:
            printf("\nRead: P)READPS, X)tended-read, D)ump, I)nit-buf, Q)uit--> ");
            if ((c = getreply()) < 0)
                return;
            putchar('\n');
            if (isupper(c))
                c = _tolower(c);
            switch (c) {
                case 'd':
                    printf("Dumping %d blocks.\n", numblks);
                    dumpbuf(CompBuf, numblks * BLKSIZE, 0, block);
                    break;
                case 'i':
                    if (get_qunum(&pattern, str_pattern)) {
                        while (1) {
                            retcode = get_qnum(str_patnblk);
                            if (retcode <= 0)
                                goto redo;
                            if (retcode < MAXSECHEAD)
                                break;
                            printf(str_toomany, MAXSECHEAD);
                        }
                        patinit(CompBuf, retcode * BLKSIZE, 0, pattern);
                    }
                    break;
                case 'q':
                    return;
                case 'p':
                case 'x':
                    while (1) {
                        block = get_qnum(str_startbk);
                        if (block < 0)
                            goto redo;
                        if (block < capacity.lbaddr)
                            break;
                        printf(str_toobigb, capacity.lbaddr);
                    }
                    while (1) {
                        numblks = get_qnum("Enter number of blocks to read (in hex)");
                        if (numblks <= 0)
                            goto redo;
                        if (numblks < MAXSECHEAD)
                            break;
                        printf(str_toomany, MAXSECHEAD);
                    }
                    if (c == 'p')
                        retcode = sc_read(CompBuf, block, numblks);
                    else
                        retcode = sc_readxt(CompBuf, block, numblks);
                    if (retcode) {
                        DBG(printf(">sc_read%s: $%x", c == 'e' ? "xt" : "", retcode);)
                        printf("Read%s Error! - %s\n",
                                c == 'e' ? "xt" : "", sc_derrmsg(errno));
                    }
                    break;
                default:
                    printf("Unrecognized command.\n");
            }
        }
}

/*---------------------------------------------------- write_blocks() --------
/ A debug function
/---------------------------------------------------------------------------*/
write_blocks()
{
        register int block = 0, numblks = 1, c, retcode;
        int offset, patok = FALSE;
        uint pattern;

        while (1) {
redo:
            printf("\nWrite: P)WRITPS, X)tended-write, D)ump, E)dit, N)ew-pattern, Q)uit--> ");
            if ((c = getreply()) < 0)
                return;
            putchar('\n');
            if (isupper(c))
                c = _tolower(c);
            switch (c) {
                case 'd':
                    while (1) {
                        offset = get_qnum("Enter dump offset in block counts (in hex)");
                        if (offset < 0)
                            goto redo;
                        if (numblks < MAXSECHEAD)
                            break;
                        printf(str_toobigb, MAXSECHEAD);
                    }
                    while (1) {
                        numblks = get_qnum("Enter number of blocks to dump (in hex)");
                        if (numblks <= 0)
                            goto redo;
                        if (numblks + offset < MAXSECHEAD)
                            break;
                        printf(str_toomany, MAXSECHEAD - offset);
                    }
                    offset *= BLKSIZE;
                    dumpbuf(CompBuf, numblks * BLKSIZE + offset, offset, block);
                    break;
                case 'e':
                    printf("You are editing the buffer ($%x blocks) to be written.\n", numblks);
                    edit_buf(CompBuf, numblks * BLKSIZE);
                    break;
                case 'n':
                    if (patok)
                        printf("Pattern now is $%x.\n", pattern);
                    else
                        printf("No pattern defined.\n");
                    patok = get_qunum(&pattern, str_pattern);
                    break;
                case 'q':
                    return;
                case 'p':
                case 'x':
                    while (1) {
                        block = get_qnum(str_startbk);
                        if (block < 0)
                            goto redo;
                        if (block < capacity.lbaddr)
                            break;
                        printf(str_toobigb, capacity.lbaddr);
                    }
                    while (1) {
                        numblks = get_qnum("Enter number of blocks to write (in hex)");
                        if (numblks <= 0)
                            goto redo;
                        if (numblks < MAXSECHEAD)
                            break;
                        printf(str_toomany, MAXSECHEAD);
                    }
                    if (patok)
                        patinit(CompBuf, numblks * BLKSIZE, 0, pattern);
                    if (c == 'p')
                        retcode = sc_write(CompBuf, block, numblks);
                    else
                        retcode = sc_writext(CompBuf, block, numblks);
                    if (retcode) {
                        DBG(printf(">sc_write%s: $%x", c == 'e' ? "xt" : "", retcode);)
                        printf("Write%s Error! - %s\n",
                                c == 'e' ? "xt" : "", sc_derrmsg(errno));
                    }
                    break;
                default:
                    printf("Unrecognized command.\n");
            }
        }
}

/*---------------------------------------------------- reassign_blocks() -----
/ REASSIGN BLOCK (i.e. mark it unusuable) the passed sector number
/---------------------------------------------------------------------------*/
static int reassign_blocks()
{
        register int numassign, tot = TotEntries, idx = 0;

        if (tot && getyn("Do you wish to reassign the bad blocks found during this test")) {
#ifdef FUTURE
            while (tot) {
                numassign = 1000;       /* do 1000 at max at a time */
                if (tot < numassign)
                    numassign = tot;
                if (retcode = sc_reassign(numassign, &badblock[idx])) {
                    DBG(printf(">After REASSGN:  $%x", retcode);            )
                    printf("\n%s\n", sc_derrmsg(errno));
                    printf("Unable to reassign %d blocks.\n", numassign);
                    break;
                }
                tot -= numassign;
                idx += numassign;
            }
            printf("%d blocks reassigned.\n", idx);
#else
            printf("Reassigning #d blocks. (a fake message)\n", TotEntries);
#endif
        }
}

/*---------------------------------------------------- add_bad() -------------
/ Add the passed sector to the list and check for duplication and overflow
/---------------------------------------------------------------------------*/
static int add_bad(sec)
uint sec;
{
        register int i;

        for (i = 0; i < TotEntries; i++)
            if (badblock[i] == sec)     /* already exists */
                return(TRUE);

        badblock[TotEntries] = sec;     /* add to the list */
        return(++TotEntries < (TBsize-4) / 4);
}

/*---------------------------------------------------- patinit() -------------
/ initialize the buffer content according to pattern or debug
/---------------------------------------------------------------------------*/
static void patinit(target, size, current, pattern)
uchar *target;
uint size;              /* size in bytes */
uint current;           /* debug pattern */
uint pattern;           /* pattern */
{
        register uint *p;
        register uint i;

        i = size >> 2;                  /* divide by 4 */
        p = (uint *) target;
        while (i--)
            if (current)
                *p++ = (Pass << 16) | ((ushort) ((current++ << 8) | i));
            else
                *p++ = pattern;
}

/*---------------------------------------------------- compare_bufs() --------
/ Compares "CompBuf" and "TrackBuffer" and report differences if any
/---------------------------------------------------------------------------*/
static int compare_bufs(bytecount, block)
uint bytecount;
int block;
{
        extern uint comp_bufs();
        uint bspot;

        bspot = comp_bufs(TrackBuffer, CompBuf, bytecount);
        if (bspot < bytecount) {
            printf("\rCompare Error #%d at offset $%x\n",
                    CompErr++, bspot);
            if (DumpIt)
                return(dumpbuf(TrackBuffer, bytecount, bspot, block));
        }
        return(TRUE);
}

/*---------------------------------------------------- dumpbuf() -------------
/ displays the buffer content in long words
/---------------------------------------------------------------------------*/
static int dumpbuf(buf, size, offset, block)
char *buf;
uint size;
uint offset;
int block;
{
        register int i, j, blk;
        register uint *c;

        offset &= ~0x1f;
        size -= offset;
        size >>= 5;                     /* get number of long words */
        c = (uint *) ((uint) buf + offset);
        for (i = 0; i < size; i++) {
            blk = (uint) c - (uint) buf;
            if (block && !(blk & 0x3ff))
                printf("Block $%x\n", (blk >> 10) + block);
            printf("%04x: ", blk);
            for (j = 0; j < 8; j++) {
                printf("%08x ", *c++);
                if (j == 3)
                    putchar(' ');
            }
            printf("\n");
            if ((i & 0xf) == 0xf)
                switch (qc_continue()) {
                    case ESCAPE:
                        return(FALSE);
                    case 'n':
                    case 'N':
                        return(TRUE);
                }
        }
        return(TRUE);
}

/*---------------------------------------------------- showstat() ----------*/
static void showstat(csect)
int csect;
{
        printf("\n\n");
        printf("SCSI Unit Testing: c%dd%d\n", UnitNum >> 4, DriveNum);
        printf("     Current Pass: %d\n", Pass);
        if (csect)
            printf("    Current Block: %d ($%x)\n", csect, csect);
        if (Compare)
            printf("   Compare Errors: %d\n", CompErr);
        printf(" Soft Disk Errors: %d\n", SoftDiskErr);
        printf(" Hard Disk Errors: %d\n", HardDiskErr);
        printf(" Bad Blocks Found: %d\n", TotEntries);
}

/*---------------------------------------------------- edit_buf() ------------
/ A debug function
/---------------------------------------------------------------------------*/
static void edit_buf(mem, limit)
uchar *mem;
uint limit;
{
        int offset, val;

        while (1) {
            offset = get_qnum("Offset (in hex)");
            if (offset < 0)
                break;
            if (offset > limit) {
                printf("Value out of range.  Must be within $%x.\n", limit);
                continue;
            }
            while (1) {
                printf("%x: %x", offset, mem[offset]);
                val = get_qnum(" ");
                if (val < 0)
                    break;
                if (val > 0xff) {
                    printf("Value too large.  Must be less than $FF.\n");
                    continue;
                }
                mem[offset] = (uchar) val;
                offset++;
            }
        }
}

/*------------------------------ End of scsitest.c -------------------------*/
