/*
 *	Macros and definitions for communicating with the Multidrop board
 * with ISO/Ethernet.  Also called the ACPA board and the Geneva board.
 *
 *	This board also uses the command structure for the old Multidrop
 * board (with WorkNet instead of ISO/Ethernet).  The following commands
 * and interrupts are used in the standard Multidrop command structure.
 * They are defined in terms of standard Foreign Code command and interrupts.
 * (See mdc.h)
 */
#ident	"@(#)me.h	1.4	2/19/88 08:39:31"

					/* commands */
#define ENA_ETHERNET	FC_CMD(0)	    /* enable ISO/Ethernet */
#define DIS_ETHERNET	FC_CMD(1)	    /* disable ISO/Ethernet */
#define REQ_ETHERNET	FC_CMD(2)	    /* indicate request ready */

					/* interrupts */
#define ACK_ETHERNET	INT_FC(1)	    /* acknowledge request */
#define IND_ETHERNET	INT_FC(2)	    /* indicate event */

#define ETHERNET_IDCHAR	'e'		/* Ethernet process identifier */

/*
 *	The address of the ISO/Ethernet control structure (ether_ctrl) is
 * passed by the ENA_ETHERNET command.  All requests and events are passed
 * through this structure.  The event queue immediately follows the request
 * queue.  Queue entries are defined in the structure ether_queue.
 *
 *	Arrays of 2 or 4 unsigned char's are two and four byte integers.
 * They are stored LOW BYTE FIRST.
 *
 *	The request queue is written by the CPU, and read by the ISO/Ethernet
 * board.  However, both boards write in the e_used field.  For request queue
 * entries, the CPU sets it and the ISO/Ethernet board clears it.  Control and
 * data portions of requests and events are kept separate so that the address
 * of data buffers (usually stream buffers) can be passed directly to the
 * ISO/Ethernet board without copying.  The value of the input index is saved
 * so that the buffers associated with the requests can be released after
 * the ISO/Ethernet board has acknowledged the requests.
 *
 *	The event queue is handled similarly to the request queue, only in
 * reverse.  However, the CPU must initialize the event queue with address and
 * MAXIMUM lengths for control and data blocks.  The ISO/Ethernet board will
 * fill these blocks and provide the correct length.  The CPU can then use the
 * blocks and allocate new ones for the queue before freeing the entries for
 * the ISO/Ethernet board.  The block addresses can change over time.
 */

struct ether_qinfo  {			/* queue information */
	unsigned char	e_in;		    /* 0: input index */
	unsigned char	e_out;		    /* 1: output index */
	unsigned short	e_size;		    /* 2: queue size */
	unsigned short	e_hiwat;	    /* 4: high water mark */
	unsigned short	e_lowat;	    /* 6: low water mark */
};					    /* 8: total */

struct ether_queue  {			/* queue entry */
	unsigned char	e_type;		    /*  0: request or event type */
	unsigned char	e_used;		    /*  1: non-zero if entry is used */
	unsigned short	e_ctrl_rsrv;	    /*  2: (reserved) */
	unsigned short	e_ctrl_len;	    /*  4: length of control block */ 
	unsigned short	e_data_len;	    /*  6: length of data block */ 
	unsigned long	e_ctrl_addr;	    /*  8: address of control block */
	unsigned long	e_data_addr;	    /* 12: address of data block */
};					    /* 16: total */

					/* general event types */
#define E_OK_RESP	1		    /* OK response */
#define E_ERROR_RESP	2		    /* error response */
#define E_UNKNOWN	3		    /* unknown request type */
#define E_ENABLE_RESP	4		    /* response to enable */
#define E_LINK_STATS 	5		    /* link layer statistics */
#define E_NETWORK_STATS	6		    /* network layer statistics */
#define E_TRANS_STATS	7		    /* transport layer statistics */
#define E_MESSAGE	8		    /* message from ACPA board */
#define E_CURR_STATS	9		    /* current status from ACPA board */

					/* general request types */
#define E_STAT_REQ	192		    /* request statistics */
#define E_STAT_RESET	193		    /* reset statistics */
#define E_NOP		194		    /* do nothing */
#define E_RESTART	195		    /* restart sending events */

					/* protocol-specific type boundaries
					 * (see the header file for each
					 *  protocol for individual types) */
#define E_MIN_GEN_REQ		192	    /* start of general event block */
#define E_MAX_GEN_REQ		207	    /* end of general event block */
#define E_MIN_GEN_EVENT		0	    /* start of general event block */
#define E_MAX_GEN_EVENT		15	    /* end of general event block */
#define E_MIN_TRANS_REQ		160	    /* start of trans request block */
#define E_MAX_TRANS_REQ		175	    /* end of trans request block */
#define E_MIN_TRANS_EVENT	176	    /* start of trans event block */
#define E_MAX_TRANS_EVENT	191	    /* end of trans event block */

#define E_CTRL_SIZE	128		/* max size of control block */
#define E_DATA_SIZE	1024		/* max size of data block */

struct ether_ctrl	{		/* ISO/Ethernet control block */
	struct ether_qinfo e_reqinfo;	   /*  0: request queue info */
	struct ether_qinfo e_eventinfo;	   /*  4: event queue info */
};					   /*  8: total */

					/* information on each queue */
#define e_req_in	e_reqinfo.e_in
#define e_req_out	e_reqinfo.e_out
#define e_req_size	e_reqinfo.e_size
#define e_event_in	e_eventinfo.e_in
#define e_event_out	e_eventinfo.e_out
#define e_event_size	e_eventinfo.e_size

struct ether_resp  {			/* non-protocol-specific event */
	unsigned char	e_type;		    /*  0: event type */
	unsigned char	e_code;		    /*  1: error code */
	unsigned short	e_circ;		    /*  2: circuit id */
	unsigned short	e_mach;		    /*  4: machine id */
	unsigned char	e_rsvd[2];	    /*  6: reserved */
};					    /*  8: total */

#define E_ADDR_SIZE	20
struct ether_addr  {			/* address buffer */
	unsigned char	e_size;		    /*  0: size of address */
	unsigned char	e_rsvd[3];	    /*  1: reserved */
	unsigned char	e_addr[E_ADDR_SIZE];/*  4: address */
};					    /* 24: total */

struct ether_enable	{		/* enable response */
	struct ether_addr  e_lladdr;	    /*  0: LLC address */
	struct ether_addr  e_nladdr;	    /* 24: network layer address */
	short	e_maxbuf;		    /* 48: total buffer count */
	short	e_datasz;		    /* 50: max normal data size */
	short	e_xdatasz;		    /* 52: max expedited data size */
	short	e_udatasz;		    /* 54: max unit data size */
	short	e_cdatasz;		    /* 56: max connect data size */
	short	e_ddatasz;		    /* 58: max disconnect data size */
	short	e_status;		    /* 60: status of enable */
	short	e_curbuf;		    /* 62: current buffer count */
	short	e_maxmach;		    /* 64: max machine structures */
	unsigned short	e_data_high;	    /* 66: data pkt high water mark */
	unsigned short	e_data_low;	    /* 68: data pkt low water mark */
};					    /* 72: total */

/*
 * Error Codes
 */

#define E_NONSPEC	0		/* non-specific error */

/*
 * Local data structures
 */

struct ether_mblk  {
	mblk_t	*e_ctrl;	/* 0: message block for e_ctrl_addr */
	mblk_t	*e_data;	/* 4: message block for e_data_addr */
};				/* 8: total */

#define ETH_PRI		(PZERO + 5)	/* init sleep priority */
#define E_CALL_SIZE	32		/* size of function call list */
#define E_STATQ_SIZE	32		/* size of statistics request queue */

/*
 *	The ether_call structure provides a means for a driver to get notified
 * when request entries have been released.
 */

struct ether_call	{
	int	(*e_func)();
	int	e_arg;
};

/*
 *	The ether_statq structure keeps track of which queues have statistics
 * requests pending.
 */

struct ether_statq	{
	queue_t	*eq_queue;		/* requesting queue */
	mblk_t	*eq_blk;		/* control block */
};

#define INVALID		-1		/* invalid foreign code pid */
#ifdef DEBUG

/*
 * This is extremely gross debug stuff.  The are indexed by board relative
 * zero, so the zero index can be used for the default case.
 */

#define ME_MEM		0x001		/* memory allocation */
#define ME_INIT		0x002		/* initialization */
#define ME_REQ		0x004		/* requests */
#define ME_ACK		0x008		/* acknowledgements */
#define ME_IND		0x010		/* indications */
#define ME_CALL		0x020		/* calls after blocking */
#define ME_STAT		0x040		/* status information */
#define ME_END		0x080		/* disabling */
#define ME_BLK		0x100		/* message blocks */
#define ME_IOCTL	0x200		/* ioctl's */
#define ME_INTR		0x400		/* interrupts */

extern unsigned short	ME_DMASK[MAXMDROPS + 1];
extern unsigned short	ME_VMASK[MAXMDROPS + 1];

#define ME_DEBUG(brd,bit,args) \
	if ((ME_DMASK[(brd) + 1] & (bit)) == (bit)) printf args;else
#define ME_VERBOSE(brd,bit,args) \
	if ((ME_VMASK[(brd) + 1] & (bit)) == (bit)) printf args;else

#else /* DEBUG */

#define ME_DEBUG(brd,bit,args)
#define ME_VERBOSE(brd,bit,args)

#endif /* DEBUG */
