/*
 *	multi-drop (RS232/RS422 comm board) memory driver
 *	for accessing memory on the multi-board (specifically, for
 *	downloading foreign code).
 *	Original author: Mike Thompson
 */
#ident	"@(#)mdcmem.h	1.3	2/19/88 08:39:22"

#define MDMBUFSIZE	512	/* Size of mdc interface buffers	      */
#define	P_IOP		 64	/* MAX possible iop pages		      */
#define	far_t		paddr_t /* This should be in types.h, but it can't be */
#define	P_MAXARGS	20
#define P_STRSIZE	32
#define P_ARGSTRSIZE	MDMBUFSIZE

/* Multi-drop memory control block.					      */
struct	m_desc {
    uchar_t	m_src[4];
    uchar_t	m_dest[4];
    uchar_t     m_count[2];
};

/* Multi-drop control block. Used for process stats, etc. */
struct	m_cntl {
    uchar_t	c_type;		/* control command			      */
    uchar_t	c_addr[4];	/* Physical address of any structure          */
    uchar_t	c_length[2];	/* Length of structure/return status	      */
};

/* Control commands. These are what might be in the 'type' field of
 * a control descriptor. (see above)
 */
#define	M_STAT		0x00	/* Return table of pointers to tables	      */

				/* M_START conflicts with stream.h	      */
#if !defined(INKERNEL) && !defined(KERNEL)
#define	M_INIT		0x01	/* Initialize the foreign code task	      */
#define	M_START		0x02	/* Start the foreign code task		      */
#define	M_EXIT		0x04	/* Stop the foreign code task		      */
#endif /* !INKERNEL or !KERNEL */

#define	M_IDENT		0x08	/* Report foreign code id to kernel	      */

/* The following is related to process managment			      */

/* The following is a table of physical addresses. These address are pointers
 * to tables in the multi-drop kernel's own data space. The host may request
 * this table through a MEM_CNTL command, and may read and write the tables
 * pointed to it using MEM_READ and MEM_WRITE commands.
 *
 * WARNING:
 *    If the size of this table changes, the file mdcmem.c in the kernel
 *    must be re-compiled as well as everything else that uses this file
 *    (mdl, the download code, etc).
 */
struct	m_tabptrs {
    ulong_t	tb_m_UNIX;	/* Multidrop board puts "UNIX" in "long" form */
    ulong_t	tb_h_UNIX;	/* Host puts "UNIX" in "long" form here	      */
    paddr_t	tb_stat;	/* Global memory status table		      */
    paddr_t	tb_proc;	/* Process table pointer		      */
};

#define NULLCHAN	0xFF    /* chan == NULLCHAN means process has no chan */
#define	MAXPROCS	15	/* Maximum number of processes 		      */

struct area {
    paddr_t	start;
    ulong_t	length;
};

struct cron {
    ulong_t	ticks;		/* In units of mt_SYS_HZ (see below)	      */
    ushort	milliticks;	/* Time since last tick. In units of mt_HZ    */
};

struct memtab {
    ulong_t	mt_clock;	/* Current system software clock time	      */
    ushort	mt_SYS_HZ;	/* Software clock rate (ticks per sec)	      */
    ulong_t	mt_HZ;		/* Hardware timer rate (ticks per sec)	      */
    ulong_t	mt_PCLK;	/* Hardware clock rate (cycles per sec)	      */
    ushort	mt_ptabsize;	/* Size of the process table		      */
    ulong_t	mt_memsize;	/* Size of memory (in bytes)		      */
    struct area	mt_foreign;	/* Total area used by foreign code	      */
    struct area	mt_free;	/* Total area currently free		      */
};

/* The following are various process state flags. These flags exist
 * in the field p_status (see the mdc_proc structure)
 */
#define	F_LOADED	0x01	/* Means process downloaded		      */
#define F_INITED	0x02	/* Means process initialized		      */
#define F_STARTED	0x04	/* Means process started		      */
#define F_DMAINTR	0x08	/* Means dma interrupts enabled		      */
#define F_KERNEL	0x10	/* Means this is a kernel process	      */
#define F_EXITED	0x20	/* Means this process has exited	      */

struct mdc_proc {
    uchar_t	p_chan;		   /* Channel associated with this process    */
    uchar_t	p_status;	   /* Various state flags		      */
    uchar_t	p_ident[8];	   /* Foreign code identifier		      */
    uchar_t	p_name[P_STRSIZE]; /* Process name			      */
    far_t	p_Start;	   /* Entry point (far pointer)		      */
    paddr_t     p_textlen;	   /* Text length			      */
    paddr_t     p_datapos;	   /* Address of initial data segment	      */
    ulong_t     p_datalen;	   /* Data length			      */
    paddr_t	p_taskentry;	   /* Physical address of task entry	      */
    paddr_t	p_iopaddr;	   /* Address of iopmem in main memory	      */
    ulong_t	p_ioplen;	   /* Length of iopmem (in Kbytes)	      */
    uchar_t	p_ioplst[P_IOP][4];/* MAX possible iop pages		      */
    struct area	p_image;	   /* Process image (not stack or scratch)    */
    struct area p_stack;	   /* Stack area			      */
    struct area p_istack;	   /* Interrupt stack area		      */
	/* The stack lengths and addresses are filled in by the downloader    */
    struct area p_scratch;	   /* Scratch area (usually for DMA)	      */
	/* The scratch length is filled in by p_Start(), but
	 * the scratch start address is filled in by the multidrop
	 * kernel (at the same time when the stack is set up).
	 */
    ulong_t	p_starttime;	   /* Time (unix format) process was started  */
    struct cron	p_userticks;	   /* Time at user level (non-interrupt)      */
    struct cron p_interticks;	   /* Time at interrupt level		      */

	/* The following are filled in by the multidrop kernel.
	 * The address of the proc table entry for each process is handed
	 * to the foreign code initialization routine. These values are
	 * copied into global foreign code variables for the convenience
	 * of the foreign code.
	 */
    ushort	p_Chan_data;	   /* SCC data port address		      */
    ushort	p_Chan_cntl;	   /* SCC control port address		      */

	/* The following are filled in by p_Start() 			      */
    far_t	p_main;		   /* Pointer to main			      */
    far_t	p_stop;		   /* Pointer to user cleanup function	      */
    far_t	p_cntl;		   /* Pointer to user ioctl-like function     */
    far_t	p_txintr;	   /* Pointer to transmit interrupt handler   */
    far_t	p_mdmintr;	   /* Pointer to modem interrupt handler      */
    far_t	p_recintr;	   /* Pointer to receive interrupt handler    */
    far_t	p_speintr;	   /* Pointer to special interrupt handler    */
    far_t	p_dmaintr;	   /* Pointer to dma interrupt handler	      */
    far_t	p_timeout;	   /* Pointer to timeout interrupt handler    */
};


/* The following is a single entry in the system task table. All values
 * are filled in by AddNewTask() and manipulated by NextTask().
 */
struct taskentry {
    uchar_t	te_priority;	/* Task Priority (0 = inactive)		      */
    uchar_t	te_prcount;	/* Task Priority Counter (internal)	      */
    ushort	te_sp;		/* Suspended Task Stack Pointer		      */
    ushort	te_ss;		/* Suspended Task Stack Segment		      */
};

/* Foreign code identifier structure. There is one of these in the kernel
 * for each possible foreign code. Drivers can access this structure to
 * determine where code associated with them is (e.g. which board and
 * what PID).
 * The m_notify field is filled in by the driver and is called when
 * the foreign code process is removed from the board by the downloader.
 * It is passed the board number and PID of the process being removed.
 */
struct	mdm_id {
    uchar_t	m_ident[sizeof(((struct mdc_proc *)0)->p_ident)];
    void	(*m_notify)();
};
