/*
 *  @(#)et.h	1.8 88/03/19 88/02/19
 *
 *	ISO/Ethernet Transport Layer Interface macros and definitions.
 *
 * This defines data structures needed to communicate with the transport layer
 * facilities on the ISO/Ethernet board.
 *
 * Arrays of 2 or 4 unsigned char's are two and four byte integers.  They are
 * stored LOW BYTE FIRST.
 */
#ident	"@(#)et.h	1.8	2/19/88 08:37:51"
/*
 * Network Layer Service Access Point Address
 */

#define NSAP_SIZE	20		/* maximum size of NSAP address */

struct nsap_addr  {
	unsigned char	nsap_size;		/*  0: size of address */
	unsigned char	nsap_rsrv[3];		/*  1: reserved */
	unsigned char	nsap_addr[NSAP_SIZE];	/*  4: address */
};						/* 24: total */

/*
 * Transport Layer Service Access Point Selector
 */

#define TSAP_SIZE	16		/* maximum size of TSAP selector */

struct tsap_sel  {
	unsigned char	tsap_size;		/*  0: size of selector */
	unsigned char	tsap_rsrv[3];		/*  1: reserved */
	unsigned char	tsap_sel[TSAP_SIZE];	/*  4: selector */
};						/* 20: total */

struct trans_ctrl  {
	unsigned short	t_mach;		/*  0: ISO/Ethernet TSAP ID */
	unsigned char	t_qos;		/*  2: currently not used */
	unsigned char	t_exp;		/*  3: currently always zero */
	struct nsap_addr t_from_nsap;	/*  4: source nsap address */
	struct tsap_sel	 t_from_tsap;	/* 28: source tsap selector */
	struct nsap_addr t_to_nsap;	/* 48: destination nsap address */
	struct tsap_sel	 t_to_tsap;	/* 72: destination tsap selector */
};					/* 92: total */

/*
 * Transport layer requests and events
 */

#define ET_REQ_MIN	160
#define ET_ADD		160		/* add transport service user */
#define ET_REMOVE	161		/* remove transport service user */
#define ET_CONN_REQ	162		/* send connect request */
#define ET_CONN_RESP	163		/* send connect response */
#define ET_DIS_REQ	164		/* send disconnect request */
#define ET_DATA_REQ	165		/* send data */
#define ET_XDATA_REQ	166		/* send expedited data */
#define ET_UDATA_REQ	167		/* send unit data */
#define ET_FLUSH	168		/* flush queues */
#define ET_REQ_MAX	168

#define ET_MIN_IND	176
#define ET_CONN_IND	176		/* indicate connect request */
#define ET_CONN_CON	177		/* confirm connect request */
#define ET_DIS_IND	178		/* indicate disconnect */
#define ET_DATA_IND	179		/* indicate data */
#define ET_XDATA_IND	180		/* indicate expedited data */
#define ET_UDATA_IND	181		/* indicate unit data */
#define ET_MAX_IND	181

					/* disconnect reasons */
#define ET_NORMAL	128		    /* normal disconnect */
#define ET_PROTO	133		    /* protocol failure */

/*
 * Transport Layer Error Codes
 */

#define ET_EDUPADDR	13		/* duplicate address on bind */

/*
 * Data structures used locally
 */
					/* value to return if waiting for board
					 * to enable */
#define ETBRD_WAIT	99

					/* ET_SIZE and ET_DATA_SIZE cannot
					 * exceed E_SIZE and E_DATA_SIZE
					 * in me.h */

#define ET_ADDR_SIZE	(TSAP_SIZE + NSAP_SIZE + 1)	/* adress size */
#define ET_SIZE		1016		/* largest packet size */
#define ET_DATA_SIZE	1016		/* data block size */
#define ET_UDATA_SIZE	1016		/* unit data block size */
#define ET_XDATA_SIZE	16		/* largest expedited data block */
#define ET_CDATA_SIZE	32		/* connect data limit */
#define ET_DDATA_SIZE	-2		/* disconnect data limit */

#define ET_COPY_SIZE	256		/* size to copy to smaller buffer */
#define ET_PTY_SIZE	256		/* buffer size for pty's */

#define ET_HI		4500		/* streams high water mark */
#define ET_LO		1500		/* streams low water mark */

#define ET_HUPWAIT	(2 * HZ)	/* time to wait after hup before kill */

typedef unsigned char	etopt_t;

/*
 * struct etcircuit
 *	This contains information about the circuit.  et_state through et_reason
 * are structures used for the current request.  A circuit can have only one
 * request pending.
 */

struct etcircuit  {
	struct tsap_sel	  et_sel;	/*  0: local TSAP selector */
	struct etconnlist *et_conn;	/* 20: connection list pointer */
	queue_t	*et_queue;		/* 24: write queue pointer (streams) */
	unsigned char	et_brd;		/* 28: board offset */
	unsigned char	et_conmax;	/* 29: maximum number of connect reqs */
	unsigned char	et_cons;	/* 30: number of connect reqs left */
	unsigned char	et_req;		/* 31: request type */
	unsigned short	et_machine;	/* 32: Tmachine index (for board) */
	unsigned char	et_state;	/* 34: TI library state */
	unsigned char	et_prim;	/* 35: TI primitive being processed */
	mblk_t	*et_ctrl;		/* 36: control block */
	mblk_t	*et_data;		/* 40: data block */
	mblk_t	*et_resp;		/* 44: control block */
	int	et_tm;			/* 48: pty timeout */
	unsigned short	et_flags;	/* 52: flags */
	unsigned char	et_reason;	/* 54: disconnect reason */
	unsigned char	et_tlierr;	/* 55: TLI error code */
	unsigned char	et_syserr;	/* 56: UNIX error code */
	unsigned char	et_ptycmd;	/* 57: pty command */
	unsigned short	et_ptycount;	/* 58: pty xmit length */
	char	*et_ptydata;		/* 60: pty data */
	struct tty   	*et_ptydev;	/* 64: pty device pointer */
	mblk_t	*et_ptyblk;		/* 68: pty data block if ET_RPTY set */
	int	et_ndata;		/* 72: data req's not responded yet */
};					/* 76: total */
extern struct etcircuit	etcircuit[];
extern struct etcircuit	*etcircend;
extern int	et_maxcirc;		/* number of circuits */

/*
 * struct etconnlist
 *	This contains a list of connection indications that have not yet
 * been responded to.  If the circuit is NULL, then the connection needs
 * to be refused (with a disconnect request).  If the machine pointer is
 * BADMACH, the entry is not used.
 */

struct etconnlist  {			/* connection list */
	struct nsap_addr  el_addr;	    /*  0: NSAP address of requestor */
	struct tsap_sel   el_sel;	    /* 24: TSAP selector of requestor */
	struct etcircuit  *el_circ;	    /* 44: circuit sent indication */
	union  {			    /* 48: if accepting or refusing: */
		struct etcircuit  *elu_acc;	/* circuit accepting connect */
		mblk_t		  *elu_blk;	/* disconnect control block */
	} elu;
	unsigned short	  el_mach;	    /* 52: machine pointer */
	unsigned char	  el_brd;	    /* 54: board number */
	unsigned char	  el_qos;	    /* 55: quality of service */
	unsigned char	  el_exp;	    /* 56: allow expedited data */
	unsigned char	  el_rsvd[3];	    /* 57: unused */
};					    /* 60: total */
extern struct etconnlist	etconnlist[];
#define el_acc elu.elu_acc
#define el_mblk elu.elu_blk
extern int	et_maxconn;		/* number of connections */
extern struct etconnlist *etconnend;	/* end of connection list array */

struct et_sizes  {			/* supported data sizes */
	int	es_data;		    /*  0: normal data */
	int	es_xdata;		    /*  4: expedited data */
	int	es_udata;		    /*  8: unit data */
	int	es_cdata;		    /* 12: connect request data */
	int	es_ddata;		    /* 16: disconnect request data */
};					    /* 20: total */
extern struct et_sizes	et_sizes[];

					/* flags (et_flags) */
#define ET_USED		0x0001		    /* used */
#define ET_UDATA	0x0002		    /* connectionless mode (unitdata) */
#define ET_QUEUE	0x0004		    /* wait for request queue entry */
#define ET_MBLK		0x0008		    /* wait for write message block */
#define ET_CONNECTED	0x0010		    /* connection open */
#define ET_CLOSE	0x0020		    /* blocking on close */
#define ET_PASS		0x0040		    /* used in passing a connection */
#define ET_ADDR		0x0080		    /* waiting for NSAP address */
#define ET_NOEXP	0x0100		    /* no expedited data allowed */
#define ET_HOOKED	0x0200		    /* hooked into a pseudo TTY */
#define ET_RPTY		0x0400		    /* remote to pseudo TTY */
#define ET_RSTOP	0x0800		    /* remote pty stopped */
#define ET_FORCE	0x1000		    /* force pty to send */
#define ET_WACK		0x2000		    /* waiting for ack from remote */
#define ET_STOP		0x4000		    /* stopped for data resp */
#define ET_ERROR	0x8000		    /* don't accept data */

					    /* these make write queue block */
#define ET_WAIT		(ET_QUEUE|ET_MBLK|ET_PASS|ET_ADDR|ET_WACK|ET_STOP)

struct etswitch  {			/* event switch */
	int	(*et_func)();		    /* function to process event */
	unsigned char	et_event;	    /* TI event code (TE_*) */
};

extern struct nsap_addr	local_nsap[];	/* local nsap addresses for boards */
extern int	et_data_low[];		/* low water marks (per board) */
extern int	et_data_high[];		/* low water marks (per board) */

#define INVALID		0x7F		/* invalid state or event */
#define BADMACH		0		/* invalid machine pointer */
#define FAKEMACH	0xFFFF		/* used to reserve a conn list entry */

#if defined(OSDEBUG) || (defined(i386) && defined(DEBUG))
/*
 * This is extremely gross debug stuff.  There are bit masks on a per-board
 * per-connection basis.  The boards index relative 1 so that 0 can be used
 * in the general case.
 */
#define ED_OPEN		0x0000001	/* open */
#define ED_CLOSE	0x0000002	/* close */
#define ED_CLEAN	0x0000004	/* cleanup */
#define ED_ENABLE	0x0000008	/* enable */

#define ED_REQ		0x0000010	/* requests */
#define ED_RESP		0x0000020	/* responses */
#define ED_IND		0x0000040	/* indications */
#define ED_BLK		0x0000080	/* message blocks */

#define ED_QUEUE	0x0000100	/* streams queueing information */
#define ED_TLI		0x0000200	/* TLI library information */
#define ED_ERR		0x0000400	/* error messages */
#define ED_FATAL	0x0000800	/* fatal errors */

#define ED_BIND		0x0001000	/* bind transactions */
#define ED_UNBIND	0x0002000	/* unbind transactions */
#define ED_INFO		0x0004000	/* info request transactions */
#define ED_OPT		0x0008000	/* option management */

#define ED_CONN		0x0010000	/* connect transactions */
#define ED_PASS		0x0020000	/* passed connect transactions */
#define ED_DISCON	0x0040000	/* disconnect transactions */
#define ED_DATA		0x0080000	/* data transactions */

#define ED_XDATA	0x0100000	/* expedited data transactions */
#define ED_UDATA	0x0200000	/* unit data transactions */

#define ED_ETH		0x1000000	/* Ethernet interface */
#define ED_IOCTL	0x2000000	/* ioctl's */
#define ED_PTY		0x4000000	/* pseudo tty's */

					/* 65 is hard coded, but should be 1
					 * greater than ET_MAXCIRC in the master
					 * file for etdrv.  This is to avoid
					 * putting debug structures in the
					 * master file since they cannot be
					 * #ifdef'ed */
extern unsigned long	ET_DMASK[65];
extern unsigned long	ET_VMASK[65];

#define ET_CIRC(x) ((x)? (((struct etcircuit *)(x)) - etcircuit) + 1: 0)
#define ET_DEBUG(x,bit,args) \
	if ((ET_DMASK[ET_CIRC(x)] & (bit)) == (bit)) printf args;else
#define ET_VERBOSE(x,bit,args) \
	if ((ET_VMASK[ET_CIRC(x)] & (bit)) == (bit)) printf args;else
#else /* DEBUG */
#define ET_DEBUG(x,bit,args)
#define ET_VERBOSE(x,bit,args)
#endif /* DEBUG */
