; --------------------------------------------------------------------------- 
; Pac-Man CPC Code (c) SyX 2011
; --------------------------------------------------------------------------- 

    include 'constantes.i'
    include 'hardware.i'
    include 'macros.i'
    include 'macros_pacman.i'

; ---------------------------------------------------------------------------

    ORG  $8000

; ---------------------------------------------------------------------------
; CPC tiles
; ---------------------------------------------------------------------------
cpc_tiles
    incbin 'recursos/cpc_tiles.bin'

; ---------------------------------------------------------------------------
; Conversion table from Arcade tile numbers to CPC
; ---------------------------------------------------------------------------
arcade2cpc_tiles
    DEFB $10,$11,$12,$13,$14,$15,$16,$17,$18,$19,$1A,$1B,$1C,$1D,$1E,$1F    ; Debug (GAME OVER)
    DEFB $0A,$0A,$0B,$0B,$0C,$0C,$4E,$4E,$4E,$4E,$4E,$4E,$4E,$4E,$4E,$4E    ; Pac-Gums
    DEFB $00,$01,$02,$03,$4E,$0D,$0E,$0E,$38,$39,$3A,$3B,$3C,$3D,$3E,$4E    ; Life Pacman/Namco
    DEFB $10,$11,$12,$13,$14,$15,$16,$17,$18,$19,$4E,$0F,$4E,$4E,$4E,$4E    ; Numbers/-
    DEFB $4E,$1A,$1B,$1C,$1D,$1E,$1F,$20,$21,$22,$23,$24,$25,$26,$27,$28    ; ESP / A-O
    DEFB $29,$2A,$2B,$2C,$2D,$2E,$2F,$30,$31,$32,$33,$34,$35,$36,$37,$4E    ; P-Z!(c)PTS
    DEFB $4F,$50,$51,$52,$53,$54,$55,$56,$57,$54,$58,$59,$5A,$5B,$4E,$4E    ; .| Intermission
    DEFB $4E,$4E,$4E,$4E,$4E,$4E,$4E,$4E,$4E,$4E,$4E,$4E,$4E,$4E,$4E,$4E    ; ESP
    DEFB $4E,$40,$41,$42,$43,$44,$45,$46,$47,$48,$49,$4A,$4B,$4C,$4D,$4E    ; Score
    DEFB $5C,$5C,$5D,$5E,$5F,$60,$61,$62,$63,$64,$65,$66,$67,$67,$68,$68    ; Labyrinth
    DEFB $69,$69,$6A,$6B,$6C,$6D,$6E,$6E,$6F,$70,$71,$72,$73,$74,$75,$76    ; Labyrinth
    DEFB $77,$78,$79,$7A,$7B,$7C,$7D,$7E,$6F,$70,$73,$74,$7F,$80,$4E,$4E    ; Labyrinth
    
; --------------------------------------------------------------------------- 
; The arcade patched rom will write here the datas for the sprite hardware
sprite_buffer
    DEFS 6 * 4

; 0 = Fruit invisible
FRUIT_STATE1            ; First buffer
    DEFB 0

FRUIT_STATE2            ; Second buffer
    DEFB 0

; --------------------------------------------------------------------------- 
; Fruit graphics
fruit_gfxs
    incbin 'recursos/cpc_fruits.bin'

; --------------------------------------------------------------------------- 
; Pac-Man code to get a random number when the Ghosts are blue. 
;
; Use the formula:
;   random_seed = ((5 * random_seed) + 1) & $1FFF;
;   result = rom [random_seed];
; RESULT:
;    A = get_random()
;
; Notes: random_seed is reset to 0 at $2A69 when Pac-Man dies or a level starts
; or a level ends. Removing the reset would make the game harder ;)
; --------------------------------------------------------------------------- 
cpc_get_random
    LD   HL,(RANDOM_SEED)
    LD   D,H
    LD   E,L
    ADD  HL,HL
    ADD  HL,HL
    ADD  HL,DE
    INC  HL
    LD   A,H
    AND  $1F            ; Original Pac-Man code search in ROM between $0000 - $1FFF
    OR   $40            ; New code search in ROM between $4000 - $5FFF
    LD   H,A

    ; Put original Pac-Man ROM in $4000
    LD   D,B            ; Save BC
    LD   E,C
    SET_PAGE RANDOM_TABLE

    LD   A,(HL)         ; Get Random Number

    ; Restore Pac-Man RAM in $4000
    INC  C
    OUT  (C),C
    LD   B,D            ; Restore BC
    LD   C,E

    LD   (RANDOM_SEED),HL
    RET

; --------------------------------------------------------------------------- 
; The ROM uses the most significant bit of the display address to indicate 
; text to the top/bottom lines, which is rotated compared to the main screen.  
; It doesn't clear the bit before using the address, but due to RAM mirroring
; the arcade version still works. Our version requires a fix.
; --------------------------------------------------------------------------- 
cpc_text_fix
    LD   E,(HL)
    INC  HL
    LD   D,(HL)
    RES  7,D        ; Additional code to reset the most significative bit
    RET

; --------------------------------------------------------------------------- 
; Speed-up hack by 40Crisis
; --------------------------------------------------------------------------- 
launch_cpc_emulation_code
    CALL cpc_emulation_code         ; Run CPC emulation code
    LD   HL,($4C82)
    LD   A,(HL)         		
    AND  A               	        	
    JP   M,launch_cpc_emulation_code
    JP   $2395			            ; Back to pacman rom

; --------------------------------------------------------------------------- 
; CPC emulation code
; --------------------------------------------------------------------------- 
cpc_emulation_code
    ; Test for VBlank
    LD   A,>PPI_B
    IN   A,($00)
    RRA
    RET  NC

    ; Only update the CPC side when start the VBlank interrupt
    
    ; Save Pac-Man stack pointer
    LD   (sm_pacman_stack + 1),SP
    LD   SP,CPC_STACK

    SET_PAGE RUN_CODE       ; Select the normal run code banking mode

    ; Show last frame and page in the new one
    CALL do_flip

    ; Only update the fruit, if there is a change of state
    LD   A,IXH              ; BIT  1,IXH
    BIT  1,A
    JR   NZ,.second_buffer
.first_buffer
    LD   HL,FRUIT_STATE1
    JR   .continue_test
.second_buffer
    LD   HL,FRUIT_STATE2
.continue_test
    LD   A,(FRUIT_POS)
    CP   (HL)
    JR   Z,.not_make_changes
    LD   (HL),A
    OR   A
    JR   NZ,.show_fruit
.del_fruit
    LD   A,$21
    LD   (restore_fruit),A
    LD   A,16
    LD   (sm_skip_fruit + 1),A
    JR   .not_make_changes
.show_fruit
    LD   A,$C9
    LD   (restore_fruit),A
    LD   A,0
    LD   (sm_skip_fruit + 1),A
.not_make_changes
    
    IF DEBUG_RASTER
        SET_BORDER AZUL_BRILLANTE
    ENDIF
    CALL do_restore         ; Restore under the old sprites

    IF DEBUG_RASTER
        SET_BORDER ROJO_BRILLANTE
    ENDIF
    CALL do_tiles           ; Update a portion of the background tiles

    IF DEBUG_RASTER
        SET_BORDER MAGENTA_BRILLANTE
    ENDIF
    CALL flash_maze         ; Update maze colour if changed
    
    CALL flash_pills        ; Flash the power pills
    
    IF DEBUG_RASTER
        SET_BORDER VERDE_BRILLANTE
    ENDIF
    CALL do_save            ; Save under the new sprite positions

    IF DEBUG_RASTER
        SET_BORDER CIAN_BRILLANTE
    ENDIF
    CALL do_sprites         ; Draw the 6 new masked sprites
    
    IF DEBUG_RASTER
        SET_BORDER BLANCO_BRILLANTE
    ENDIF
    SET_PAGE RUN_CODE       ; Screen fully update, select the normal run code banking mode
    CALL do_input           ; Scan the joystick and DIP switches
    
    CALL do_sound           ; Convert the sound to the AY chip
    
    IF DEBUG_RASTER
        SET_BORDER NEGRO
    ENDIF
   
    ; Restore Pac-Man stack pointer
sm_pacman_stack
    LD   SP,$0000
    
    ; Pac-Man interrupt handler
sm_pacman_int_handler
    JP   $30B0              ; Pac-Man first interrupt handler, changed after first run

; --------------------------------------------------------------------------- 
; Change the visible buffer and page in the new one for updating
; --------------------------------------------------------------------------- 
do_flip
    ; Show frame finished
    LD   BC,CRTC_SELECT + REG_0C
    OUT  (C),C
.sm_show_frame
    LD   A,VRAM_P1                  ; VRAM en $4000 ... $7FFF ($10)
    INC  B
    OUT  (C),A
    
    ; Update the show_frame variable for the next frame
    XOR  $20                        ; Flip between $10 and $30
    LD   (.sm_show_frame + 1),A
    
    ; Get the page for the new buffer to update
.sm_update_frame
    LD   A,UPDATE_VRAM2
    LD   IXH,A                      ; Make a copy in IXH to fast page in
    ; Update the update_frame variable for the next frame
    XOR  $02
    LD   (.sm_update_frame + 1),A
    RET

; --------------------------------------------------------------------------- 
; Update a portion of the background tiles
; --------------------------------------------------------------------------- 
do_tiles
    ; Check sprite visibility
    LD   A,MIN_SPR_Y        ; Minimum y position to be visible
    EX   AF,AF'
    LD   A,MIN_SPR_X        ; Minimum x position to be visible
    LD   B,6                ; 6 sprites to check
    LD   HL,SPRITE1_XPOS
.loop_check_sprites_visibility
    CP   (HL)               ; X
    JR   C,draw_tile_strips ; If any sprites are visible we'll draw in strips
    EX   AF,AF'
    INC  L
    CP   (HL)               ; Y
    RET  C
    EX   AF,AF'
    INC  L
    DJNZ .loop_check_sprites_visibility

    ; There aren't sprites visibles, we'll analyze the full screen
draw_full_screen
    LD   B,LEN_STRIP0_4     ; Scans to analyze
    LD   DE,PAC_CHARS       ; Pac-Man display
    LD   A,IXH              ; BIT  1,IXH
    BIT  1,A
    JR   NZ,.second_tilemap
    ; First copy of Pac-Man display
.first_tilemap
    LD   HL,CPC_TILEMAP1
    CALL tile_comp
    LD   H,>CPC_TILEMAP1
    CALL do_lives
    LD   H,>(CPC_TILEMAP1 + $0300)
    JP   do_scores

    ; Second copy of Pac-Man display
.second_tilemap
    LD   HL,CPC_TILEMAP2
    CALL tile_comp
    LD   H,>CPC_TILEMAP2
    CALL do_lives
    LD   H,>(CPC_TILEMAP2 + $0300)
    JP   do_scores
    
    ; There are sprites visibles, we'll need to draw the tiles in strips
draw_tile_strips
    LD   A,IXH              ; BIT  1,IXH
    BIT  1,A
    JP   NZ,strip_odd

    ; First copy of Pac-Man display
strip_even
    JP   strip_0

strip_0
    LD   B,LEN_STRIP0
    LD   DE,PAC_CHARS
    LD   HL,CPC_TILEMAP1
    CALL tile_comp
    LD   HL,strip_1
    LD   (strip_even + 1),HL
    RET

strip_1
    LD   B,LEN_STRIP1
    LD   DE,PAC_CHARS + 32 * LEN_STRIP0
    LD   HL,CPC_TILEMAP1 + 32 * LEN_STRIP0
    CALL tile_comp
    LD   H,>(CPC_TILEMAP1 + $0300)
    CALL do_scores
    LD   HL,strip_2
    LD   (strip_even + 1),HL
    RET

strip_2
    LD   B,LEN_STRIP2
    LD   DE,PAC_CHARS + 32 * LEN_STRIP0_1
    LD   HL,CPC_TILEMAP1 + 32 * LEN_STRIP0_1
    CALL tile_comp
    LD   HL,strip_3
    LD   (strip_even + 1),HL
    RET

strip_3
    LD   B,LEN_STRIP3
    LD   DE,PAC_CHARS + 32 * LEN_STRIP0_2
    LD   HL,CPC_TILEMAP1 + 32 * LEN_STRIP0_2
    CALL tile_comp
    LD   H,>CPC_TILEMAP1
    CALL do_lives
    LD   HL,strip_4
    LD   (strip_even + 1),HL
    RET

strip_4
    LD   B,LEN_STRIP4
    LD   DE,PAC_CHARS + 32 * LEN_STRIP0_3
    LD   HL,CPC_TILEMAP1 + 32 * LEN_STRIP0_3
    CALL tile_comp
    LD   HL,strip_0
    LD   (strip_even + 1),HL
    RET

    ; Second copy of Pac-Man display
strip_odd
    JP   strip_0_alt

strip_0_alt
    LD   B,LEN_STRIP0
    LD   DE,PAC_CHARS
    LD   HL,CPC_TILEMAP2
    CALL tile_comp
    LD   HL,strip_1_alt
    LD   (strip_odd + 1),HL
    RET

strip_1_alt
    LD   B,LEN_STRIP1
    LD   DE,PAC_CHARS + 32 * LEN_STRIP0
    LD   HL,CPC_TILEMAP2 + 32 * LEN_STRIP0
    CALL tile_comp
    LD   H,>(CPC_TILEMAP2 + $0300)
    CALL do_scores
    LD   HL,strip_2_alt
    LD   (strip_odd + 1),HL
    RET

strip_2_alt
    LD   B,LEN_STRIP2
    LD   DE,PAC_CHARS + 32 * LEN_STRIP0_1
    LD   HL,CPC_TILEMAP2 + 32 * LEN_STRIP0_1
    CALL tile_comp
    LD   HL,strip_3_alt
    LD   (strip_odd + 1),HL
    RET

strip_3_alt
    LD   B,LEN_STRIP3
    LD   DE,PAC_CHARS + 32 * LEN_STRIP0_2
    LD   HL,CPC_TILEMAP2 + 32 * LEN_STRIP0_2
    CALL tile_comp
    LD   H,>CPC_TILEMAP2
    CALL do_lives
    LD   HL,strip_4_alt
    LD   (strip_odd + 1),HL
    RET

strip_4_alt
    LD   B,LEN_STRIP4
    LD   DE,PAC_CHARS + 32 * LEN_STRIP0_3
    LD   HL,CPC_TILEMAP2 + 32 * LEN_STRIP0_3
    CALL tile_comp
    LD   HL,strip_0_alt
    LD   (strip_odd + 1),HL
    RET

; --------------------------------------------------------------------------- 
tile_comp
    CALL real_find_change   ; Scan block for display changes
    DEC  SP                 ; Restore the same return address to here
    DEC  SP

    LD   (HL),A             ; Update with new tile value
    CP   $40                ; Arcade blank tile
    JP   Z,is_a_blank_tile
    CP   BEFORE_FRUIT_TILE  ; Before fruit tiles?
    JR   C,.get_cpc_tile
    CP   AFTER_GHOST_TILE   ; After ghost tiles?
    JP   C,is_a_ghost_tile

    ; It's a labyrinth tile
    SUB  A,RELOC_LABY_TILE  ; Relocate to account for removed tiles

    ; Get the CPC tile number
.get_cpc_tile
    LD   IXL,B
    LD   B,>arcade2cpc_tiles
    ADD  A,<arcade2cpc_tiles
    LD   C,A
    LD   A,(BC)
    LD   B,IXL

; --------------------------------------------------------------------------- 
tile_mapped
    EX   AF,AF'         ; Save tile for later
    PUSH DE
    EXX                 ; Save to resume find
    POP  HL             ; Pac-Man screen address of changed tile

    LD   A,L
    AND  %00011111      ; Column is in bits 0-4
    LD   B,A            ; Tile Coord Y
    ADD  A,A            ; *2
    ADD  A,A            ; *4
    ADD  A,B            ; *5 (code size to check each byte)
    ADD  A,3            ; Skip ld+cp+ret, so we advance pointers
    LD   E,A
    LD   D,>find_change
    PUSH DE             ; Return address to resume find

    ADD  HL,HL          ; *2
    ADD  HL,HL          ; *4
    ADD  HL,HL          ; *8 (ignore overflow), H is now mirrored column number
    LD   A,28+2         ; 28 columns wide, offset 2 by additional rows
    SUB  H              ; Unmirror the column
    LD   C,A            ; Tile Coord X

; --------------------------------------------------------------------------- 
;  A': Tile
; BC : Tile Coords (Y,X)
draw_tile_xy
    ; Convert from coords to screen address
    LD   D,B                ; D = CY
    XOR  A
    SCF                     ; Carry = 1
    RR   D
    RRA
    RR   D                  ; D = %0100 0yyy
    RRA                     ; A = %yy00 0000
    INC  C                  ; CX++
    ADD  A,C
    ADD  A,C                ; A += CX * 2 | A = %yyxx xxxx
    LD   E,A                ; DE = Screen Address

    ; Get the tile address ($8000 + tile_number << 4) 
    EX   AF,AF'             ; Restore tile number
    LD   H,%00001000
    SLA  A
    RL   H
    RLA
    RL   H
    RLA
    RL   H
    RLA
    RL   H
    LD   L,A                ; HL hold tile address

; DE : Screen address
; HL : Tile address
draw_tile
    SET_VRAM_PAGE           ; Put the VRAM page in $4000 (BC, IXH)

    LD   (.sm_old_stack + 1),SP
    LD   SP,HL
    EX   DE,HL

    ; 1º Scanline
    POP  DE
    LD   (HL),E
    INC  L
    LD   (HL),D
    SET  3,H
    ; 2º Scanline
    POP  DE
    LD   (HL),D
    DEC  L
    LD   (HL),E
    SET  4,H
    ; 4º Scanline
    POP  DE
    LD   (HL),E
    INC  L
    LD   (HL),D
    RES  3,H
    ; 3º Scanline
    POP  DE
    LD   (HL),D
    DEC  L
    LD   (HL),E
    SET  5,H
    ; 7º Scanline
    POP  DE
    LD   (HL),E
    INC  L
    LD   (HL),D
    RES  4,H
    ; 5º Scanline
    POP  DE
    LD   (HL),D
    DEC  L
    LD   (HL),E
    SET  3,H
    ; 6º Scanline
    POP  DE
    LD   (HL),E
    INC  L
    LD   (HL),D
    SET  4,H
    ; 8º Scanline
    POP  DE
    LD   (HL),D
    DEC  L
    LD   (HL),E

.sm_old_stack
    LD   SP,$0000

    SET_PAGE_FAST RUN_CODE

    EXX
    RET

; --------------------------------------------------------------------------- 
; Routine for printing blank tiles FAST!!! :P
is_a_blank_tile
    PUSH DE
    EXX                 ; Save to resume find
    POP  HL             ; Pac-Man screen address of changed tile

    LD   A,L
    AND  %00011111      ; Column is in bits 0-4
    LD   B,A            ; Tile Coord Y
    ADD  A,A            ; *2
    ADD  A,A            ; *4
    ADD  A,B            ; *5 (code size to check each byte)
    ADD  A,3            ; Skip ld+cp+ret, so we advance pointers
    LD   E,A
    LD   D,>find_change
    PUSH DE             ; Return address to resume find

    ADD  HL,HL          ; *2
    ADD  HL,HL          ; *4
    ADD  HL,HL          ; *8 (ignore overflow), H is now mirrored column number
    LD   A,28+2         ; 28 columns wide, offset 2 by additional rows
    SUB  H              ; Unmirror the column
    LD   C,A            ; Tile Coord X

; BC : Tile Coords (Y,X)
draw_tile_xy_blank
    ; Convert from coords to screen address
    LD   D,B                ; D = CY
    XOR  A
    SCF                     ; Carry = 1
    RR   D
    RRA
    RR   D                  ; D = %0100 0yyy
    RRA                     ; A = %yy00 0000
    INC  C                  ; CX++
    ADD  A,C
    ADD  A,C                ; A += CX * 2 | A = %yyxx xxxx
    LD   E,A                ; DE = Screen Address

; DE : Screen address
draw_tile_blank
    SET_VRAM_PAGE           ; Put the VRAM page in $4000 (BC, IXH)
    XOR  A
    ; 1º Scanline
    LD   (DE),A
    INC  E
    LD   (DE),A
    SET  3,D
    ; 2º Scanline
    LD   (DE),A
    DEC  E
    LD   (DE),A
    SET  4,D
    ; 4º Scanline
    LD   (DE),A
    INC  E
    LD   (DE),A
    RES  3,D
    ; 3º Scanline
    LD   (DE),A
    DEC  E
    LD   (DE),A
    SET  5,D
    ; 7º Scanline
    LD   (DE),A
    INC  E
    LD   (DE),A
    RES  4,D
    ; 5º Scanline
    LD   (DE),A
    DEC  E
    LD   (DE),A
    SET  3,D
    ; 6º Scanline
    LD   (DE),A
    INC  E
    LD   (DE),A
    SET  4,D
    ; 8º Scanline
    LD   (DE),A
    DEC  E
    LD   (DE),A

    SET_PAGE_FAST RUN_CODE

    EXX
    RET

; --------------------------------------------------------------------------- 
; When a Ghost is printed as tiles, the order is $B1, $B3, $B5, $B0, $B2 and
; $B4. We print the ghost when $B1 is needed and skip in the others.
is_a_ghost_tile
    CP   FIRST_GHOST_TILE + 1
    JR   NZ,skip_ghost_tile
    SET  2,D            ; Switch to attributes
    LD   A,(DE)         ; Fetch tile attribute
    RES  2,D            ; Switch back to data

    ; Update the return address to resume find after draw the ghost
    PUSH DE
    EXX                 ; Save to resume find
    POP  HL             ; Pac-Man screen address of changed tile

    LD   C,A            ; Save ghost tile attributes

    LD   A,L
    AND  %00011111      ; Column is in bits 0-4
    LD   B,A            ; Tile Coord Y
    ADD  A,A            ; *2
    ADD  A,A            ; *4
    ADD  A,B            ; *5 (code size to check each byte)
    ADD  A,3            ; Skip ld+cp+ret, so we advance pointers
    LD   E,A
    LD   D,>find_change
    PUSH DE             ; Return address to resume find
    
    ; *** Test ghosts ***
.test_blinky                    ; Red ghost
    DEC  C
    JR   NZ,.test_pinky
.is_blinky
    LD   DE,$690C               ; Screen Address
    LD   HL,SPR_GHOST_RED_32    ; Ghost GFX
    JR   .draw_tile_ghost

.test_pinky                     ; Pink ghost
    RR   C
    DEC  C
    JR   NZ,.test_inky
.is_pinky
    LD   DE,$69CC               ; Screen Address
    LD   HL,SPR_GHOST_PINK_32   ; Ghost GFX
    JR   .draw_tile_ghost

.test_inky                      ; Cyan ghost
    DEC  C
    JR   NZ,.is_clyde
.is_inky
    LD   DE,$6A8C               ; Screen Address
    LD   HL,SPR_GHOST_CYAN_32   ; Ghost GFX
    JR   .draw_tile_ghost

.is_clyde                       ; Orange ghost
    LD   DE,$6B4C               ; Screen Address
    LD   HL,SPR_GHOST_ORANGE_32 ; Ghost GFX

; --------------------------------------------------------------------------- 
; DE : Screen address
; HL : Ghost GFX
.draw_tile_ghost
    SET_VRAM_PAGE               ; Put the VRAM page in $4000 (BC, IXH)
    CALL print_scanlines_3_8_3_ghost
    SET_PAGE RUN_CODE
    EXX
    RET

; Update only the return address to resume find
skip_ghost_tile
    PUSH DE
    EXX                         ; Save to resume find
    POP  HL                     ; Pac-Man screen address of changed tile

    LD   A,L
    AND  %00011111              ; Column is in bits 0-4
    LD   B,A                    ; Tile Coord Y
    ADD  A,A                    ; *2
    ADD  A,A                    ; *4
    ADD  A,B                    ; *5 (code size to check each byte)
    ADD  A,3                    ; Skip ld+cp+ret, so we advance pointers
    LD   E,A
    LD   D,>find_change
    PUSH DE                     ; Return address to resume find

    EXX
    RET

; --------------------------------------------------------------------------- 
; Draw changes to the number of remaining lives
; --------------------------------------------------------------------------- 
; TODO: Optimizar -> Mejor caso 115ms (antes 140ms) / Peor caso 770ms (antes 995ms)
;       LD  BC,$00xx -> B es siempre 0; por lo que las convertimos a LD C,$xx
;       * Separar la actualización en 5 partes (23 ms/ ~240ms). 
do_lives
    LD   A,H                ; Save H in IXL
    LD   IXL,A
    LD   L,$1B              ; Offset to first life in display and comparison buffer
    LD   C,($1B + 1) * 2    ; Offset X to the CPC screen
    CALL chk_life

    LD   A,IXL              ; Restore H from IXL
    LD   H,A
    LD   L,$19
    LD   C,($1A + 1) * 2
    CALL chk_life

    LD   A,IXL              ; Restore H from IXL
    LD   H,A
    LD   L,$17
    LD   C,($19 + 1) * 2
    CALL chk_life

    LD   A,IXL              ; Restore H from IXL
    LD   H,A
    LD   L,$15
    LD   C,($18 + 1) * 2
    CALL chk_life

    LD   A,IXL              ; Restore H from IXL
    LD   H,A
    LD   L,$13
    LD   C,($17 + 1) * 2

; --------------------------------------------------------------------------- 
; Draw either a blank or a mini Pac-Man
; ENTRY:
;     L : Offset to the score tilemap
;     C : Offset X to the CPC screen address where print life
; --------------------------------------------------------------------------- 
chk_life
    LD   D,>PAC_FOOTER
    LD   E,L
    LD   A,(DE)
    CP   (HL)
    RET  Z

    LD   (HL),A                 ; Update buffered copy

    CP   $40                    ; blank?
    JR   NZ,.mini_live
    LD   HL,cpc_tiles + CPC_BLANK_TILE * 16 ; HL = Tile Address (Blank = $4E)
    JR   .draw_live
.mini_live
    LD   HL,cpc_tiles + 11 * 16 ; HL = Tile Address (mini Pac-Man = 11)
.draw_live
    LD   D,$40
    LD   E,C                    ; DE = Screen address

    JP   draw_tile7             ; Draw Tile

; --------------------------------------------------------------------------- 
; Draw Scores
; --------------------------------------------------------------------------- 
;    LD   HL,CPC_TILEMAPx + $0300
do_scores
    LD   A,(PAC_NUM_PLAYER)     ; 0 = P1 / 1 = P2
    OR   A
    JR   NZ,.do_score2

.do_score1
    LD   L,$DA
    LD   C,$02
    CALL chk_digit      ; 1

    LD   L,$D8
    LD   C,$03
    CALL chk_digit      ; P

    LD   L,$FC
    LD   C,$04
    CALL chk_digit      ; 100,000s

    LD   L,$FB
    LD   C,$05
    CALL chk_digit      ; 10,000s

    LD   L,$FA
    LD   C,$06
    CALL chk_digit      ; 1,000s

    LD   L,$F9
    LD   C,$07
    CALL chk_digit      ; 100s

    LD   L,$F8
    LD   C,$08
    CALL chk_digit      ; 10s

    LD   L,$F7
    LD   C,$09
    CALL chk_digit      ; 1s
    JR   .do_highscore

.do_score2
    LD   L,$C7
    LD   C,$02
    CALL chk_digit      ; 2

    LD   L,$C5
    LD   C,$03
    CALL chk_digit      ; P

    LD   L,$E9
    LD   C,$04
    CALL chk_digit      ; 100,000s

    LD   L,$E8
    LD   C,$05
    CALL chk_digit      ; 10,000s

    LD   L,$E7
    LD   C,$06
    CALL chk_digit      ; 1,000s

    LD   L,$E6
    LD   C,$07
    CALL chk_digit      ; 100s

    LD   L,$E5
    LD   C,$08
    CALL chk_digit      ; 10s

    LD   L,$E4
    LD   C,$09
    CALL chk_digit      ; 1s

.do_highscore
    ; High Score
    LD   L,$D4
    LD   C,$0B
    CALL chk_digit      ; H

    LD   L,$D3
    LD   C,$0C
    CALL chk_digit      ; I

    LD   L,$F2
    LD   C,$0D
    CALL chk_digit      ; 100,000s

    LD   L,$F1
    LD   C,$0E
    CALL chk_digit      ; 10,000s

    LD   L,$F0
    LD   C,$0F
    CALL chk_digit      ; 1,000s

    LD   L,$EF
    LD   C,$10
    CALL chk_digit      ; 100s

    LD   L,$EE
    LD   C,$11
    CALL chk_digit      ; 10s

    LD   L,$ED
    LD   C,$12
    JR   chk_digit      ; 1s (Nos lo podemos ahorrar)

; --------------------------------------------------------------------------- 
; ENTRY:
;     L : Offset to the score tilemap
;    BC : Tile Coords (Y, X)
; --------------------------------------------------------------------------- 
chk_digit
    LD   D,$43
    LD   E,L
    LD   A,(DE)             ; Score
    CP   (HL)               ; Has it changed?
    RET  Z                  ; Return if not

    LD   (HL),A             ; Update buffered copy

    ; Get the CPC tile address
.get_cpc_tile
    LD   E,C                    ; Save tile Coord X
    LD   B,>arcade2cpc_tiles
    ADD  A,<arcade2cpc_tiles
    LD   C,A
    LD   A,(BC)             ; A = Tile number

    PUSH HL
    LD   H,%00001000
    SLA  A
    RL   H
    RLA
    RL   H
    RLA
    RL   H
    RLA
    RL   H
    LD   L,A                ; HL = Tile Address
    
    ; Get the CPC screen address
    LD   D,$40
    XOR  A
    LD   C,E
    INC  E                  ; CX++
    ADD  A,E
    ADD  A,E                ; A += CX * 2 | A = %yyxx xxxx
    LD   E,A                ; DE = Screen Address

    CALL draw_tile7

    POP  HL

    ; Did it happen a change of player?
    LD   A,L
    CP   $C7
    JR   Z,.update_2up
    CP   $DA
    RET  NZ
    
.update_1up
    LD   L,$F7              ; $F7 - $FC
    JR   .update_score

.update_2up
    LD   L,$E4              ; $E4 - $E9

.update_score
    LD   A,CPC_BLANK_TILE
    LD   (HL),A
    INC  L
    LD   (HL),A
    INC  L
    LD   (HL),A
    INC  L
    LD   (HL),A
    INC  L
    LD   (HL),A
    INC  L
    LD   (HL),A
    
    RET

; --------------------------------------------------------------------------- 
; Draw a tile with only 7 scanlines
; DE : Screen address
; HL : Tile address
; --------------------------------------------------------------------------- 
draw_tile7
    SET_VRAM_PAGE           ; Put the VRAM page in $4000 (BC, IXH)

    LD   (.sm_old_stack + 1),SP
    LD   SP,HL
    EX   DE,HL

    ; 1º Scanline
    POP  DE
    LD   (HL),E
    INC  L
    LD   (HL),D
    SET  3,H
    ; 2º Scanline
    POP  DE
    LD   (HL),D
    DEC  L
    LD   (HL),E
    SET  4,H
    ; 4º Scanline
    POP  DE
    LD   (HL),E
    INC  L
    LD   (HL),D
    RES  3,H
    ; 3º Scanline
    POP  DE
    LD   (HL),D
    DEC  L
    LD   (HL),E
    SET  5,H
    ; 7º Scanline
    POP  DE
    LD   (HL),E
    INC  L
    LD   (HL),D
    RES  4,H
    ; 5º Scanline
    POP  DE
    LD   (HL),D
    DEC  L
    LD   (HL),E
    SET  3,H
    ; 6º Scanline
    POP  DE
    LD   (HL),E
    INC  L
    LD   (HL),D
    SET  4,H
    ; Skip 8º Scanline

.sm_old_stack
    LD   SP,$0000

    SET_PAGE_FAST RUN_CODE

    RET

; --------------------------------------------------------------------------- 
; Set the maze palette colour by detecting the attribute used for the maze 
; white. We also need to remove the crypt, as the real attribute wipe does.
; --------------------------------------------------------------------------- 
flash_maze
    LD   A,($4440)      ; Attribute of maze top-right
.no_intermission    
    CP   $1F
.sm_flash_blue
    LD   A,INK + AZUL_BRILLANTE
    JR   NZ,.set_maze_colour

    LD   A,64           ; Blank tile
    LD   ($420D),A      ; Clear left of crypt door
    LD   ($41ED),A      ; Clear right of crypt door

    LD   A,INK + BLANCO_BRILLANTE
.set_maze_colour
    LD   BC,GATE_ARRAY + PEN_01
    OUT  (C),C
    OUT  (C),A

    ; Restore from the intermission Pink :P
    LD   A,INK + AZUL_BRILLANTE
    LD   (.sm_flash_blue + 1),A

    RET

; --------------------------------------------------------------------------- 
; Set the power pill palette colour to the correct state by reading the 6 
; known pill locations, and reacting to changes in the attribute setting.
; --------------------------------------------------------------------------- 
flash_pills
    LD   A,($4278)              ; Bottom attract screen pill location 
    CP   $14                    ; Power pill?
    JP   Z,.pill_attract_mode   ; If so, we're in attract mode

    LD   A,($4384)              ; Top-left
    CP   $14
    CALL Z,.test_pill_game_01
    SET_PAGE RUN_CODE

    LD   A,($4064)              ; Top right
    CP   $14
    CALL Z,.test_pill_game_02
    SET_PAGE_FAST RUN_CODE

    LD   A,($4398)              ; Bottom-left
    CP   $14
    CALL Z,.test_pill_game_03
    SET_PAGE_FAST RUN_CODE

    LD   A,($4078)              ; Bottom-right
    CP   $14
    RET  NZ
.test_pill_game_04
    LD   A,($4478)
    CP   $10                    ; Test if pill is visible
    SET_VRAM_PAGE_FAST          ; Put the VRAM page in $4000 (BC, IXH)
    JR   NZ,.clear_pill_game_04
    DRAW_PILL_COMPILED $4638    ; X=27, Y=24 --> $4638 (CPC Screen Address)
    SET_PAGE_FAST RUN_CODE
    RET
.clear_pill_game_04
    CLEAR_PILL_COMPILED $4638   ; X=27, Y=24 --> $4638 (CPC Screen Address)
    SET_PAGE_FAST RUN_CODE
    RET

.test_pill_game_01
    LD   A,($4784)
    CP   $10                    ; Test if pill is visible
    SET_VRAM_PAGE               ; Put the VRAM page in $4000 (BC, IXH)
    JR   NZ,.clear_pill_game_01
    DRAW_PILL_COMPILED $4106    ; X=2, Y=4   --> $4106 (CPC Screen Address)
    RET
.clear_pill_game_01
    CLEAR_PILL_COMPILED $4106   ; X=2, Y=4   --> $4106 (CPC Screen Address)
    RET

.test_pill_game_02
    LD   A,($4464)
    CP   $10                    ; Test if pill is visible
    SET_VRAM_PAGE_FAST          ; Put the VRAM page in $4000 (BC, IXH)
    JR   NZ,.clear_pill_game_02
    DRAW_PILL_COMPILED $4138    ; X=27, Y=4  --> $4138 (CPC Screen Address)
    RET
.clear_pill_game_02
    CLEAR_PILL_COMPILED $4138   ; X=27, Y=4  --> $4138 (CPC Screen Address)
    RET

.test_pill_game_03
    LD   A,($4798)
    CP   $10                    ; Test if pill is visible
    SET_VRAM_PAGE_FAST          ; Put the VRAM page in $4000 (BC, IXH)
    JR   NZ,.clear_pill_game_03
    DRAW_PILL_COMPILED $4606    ; X=2, Y=24  --> $4606 (CPC Screen Address)
    RET
.clear_pill_game_03
    CLEAR_PILL_COMPILED $4606   ; X=2, Y=24  --> $4606 (CPC Screen Address)
    RET

.pill_attract_mode
    CALL .test_pill_attract_01
    SET_PAGE_FAST RUN_CODE

    LD   A,($4332)              ; Top attract
    CP   $14
    RET  NZ
.test_pill_attract_02
    LD   A,($4732)
    CP   $10                    ; Test if pill is visible
    SET_VRAM_PAGE_FAST          ; Put the VRAM page in $4000 (BC, IXH)
    JR   NZ,.clear_pill_attract_02
    DRAW_PILL_COMPILED $448C    ; X=5, Y=18  --> $448C (CPC Screen Address)
    SET_PAGE_FAST RUN_CODE
    RET
.clear_pill_attract_02
    CLEAR_PILL_COMPILED $448C   ; X=5, Y=18  --> $448C (CPC Screen Address)
    SET_PAGE_FAST RUN_CODE
    RET

.test_pill_attract_01
    LD   A,($4678)              ; Bottom attract
    CP   $10                    ; Test if pill is visible
    SET_VRAM_PAGE               ; Put the VRAM page in $4000 (BC, IXH)
    JR   NZ,.clear_pill_attract_01
    DRAW_PILL_COMPILED $4618    ; X=11, Y=24 --> $4618 (CPC Screen Address)
    RET
.clear_pill_attract_01
    CLEAR_PILL_COMPILED $4618   ; X=11, Y=24 --> $4618 (CPC Screen Address)
    RET

; --------------------------------------------------------------------------- 
; Scan the input DIP switches for joystick movement and button presses
; --------------------------------------------------------------------------- 
do_input
    LD   DE,$FFFF                               ; Nothing pressed
    ; 1.- Read the CPC Joystick and Keyboard

.read_joystick
    ; Select the PSG IOPORT_A
    LD   BC,PPI_A + PSG_IOPORT_A
    OUT  (C),C

    ; Operation Select PSG Register and Joystick 0 key Line 
    LD   BC,PPI_C + PPI_PSG_SELECT              ; + JOY0 <--- Puto ERROR!!! xDDD
    OUT  (C),C

    ; PSG Inactive (Needed in CPC+, because the PPI emulation do not automatically
    ; resets the Ports A, B and C to $00, when PPI Control changes (bit 7 set)
    DEFB $ED,$71                                ; OUT (C),0 

    ; Set the PPI_A in input mode
    LD   BC,PPI_CONTROL + PPI_A_INPUT
    OUT  (C),C

    ; Read a byte from the joyport
    LD   BC,PPI_C + PPI_PSG_READ + JOY0
    OUT  (C),C
    LD   B,>PPI_A
    IN   A,(C)                                  ; A = byte received

.joy_up
    RRA
    JR   C,.joy_down
    RES  P1_UP,D
.joy_down
    RRA
    JR   C,.joy_left
    RES  P1_DOWN,D
.joy_left
    RRA
    JR   C,.joy_right
    RES  P1_LEFT,D
.joy_right
    RRA
    JR   C,.joy_fire1
    RES  P1_RIGHT,D
.joy_fire1
    RRA
    JR   C,.joy_fire2
    RES  DIP_START1,E
.joy_fire2
    RRA
    JR   C,.joy_fire3
    BIT  DIP_START1,E
    JR   NZ,.insert_coin
.start_2_players
    SET  DIP_START1,E
    RES  DIP_START2,E
    JR   .joy_fire3
.insert_coin
    RES  DIP_COIN1,D
.joy_fire3
    RRA
    JR   C,.end_joy
    ; Fire 3 = TAB
    JP   .launch_menu
.end_joy

.read_cursor_keys
    ; Check Cursors Up, Down and Right
    LD   BC,PPI_C + PPI_PSG_READ + KEY_LINE0
    OUT  (C),C                  
    LD   B,>PPI_A
    IN   A,(C)                                  ; A = byte received
.cursor_up
    RRA
    JR   C,.cursor_right
    RES  P1_UP,D
.cursor_right
    RRA
    JR   C,.cursor_down
    RES  P1_RIGHT,D
.cursor_down
    RRA
    JR   C,.cursor_left
    RES  P1_DOWN,D
.cursor_left

    ; Check Cursor Left
    LD   BC,PPI_C + PPI_PSG_READ + KEY_LINE1
    OUT  (C),C                  
    LD   B,>PPI_A
    IN   A,(C)                                  ; A = byte received
    RRA
    JR   C,.end_cursor
    RES  P1_LEFT,D
.end_cursor

.read_coin_keys
    ; Check insert coin keys (3 and 4)
    LD   BC,PPI_C + PPI_PSG_READ + KEY_LINE7
    OUT  (C),C                  
    LD   B,>PPI_A
    IN   A,(C)                                  ; A = byte received
.coin1
    RRA
    JR   C,.coin2
    RES  DIP_COIN1,D
.coin2
    RRA
    JR   C,.end_coin_keys
    RES  DIP_COIN2,D
.end_coin_keys

.read_start_keys
    ; Check start game and skip level keys (1, 2 and ESC + TAB)
    LD   BC,PPI_C + PPI_PSG_READ + KEY_LINE8
    OUT  (C),C                  
    LD   B,>PPI_A
    IN   A,(C)                                  ; A = byte received
.start1
    RRA
    JR   C,.start2
    RES  DIP_START1,E
.start2
    RRA
    JR   C,.pause_mode
    RES  DIP_START2,E
.pause_mode
    BIT  KEY_TAB - 2,A                          ; TAB
    JR   NZ,.skip_level1
.launch_menu
    PUSH DE
    LD   HL,psg_sound_list                      ; To optimize or not optimize :P
    LD   A,(PAC_LEVEL_NUMBER)                   ; Get level number
    LD   E,A
    LD   A,(DIP_5080)                           ; Get DIP config
    SET_PAGE MENU_CODE
    CALL LAUNCH_MENU
    SET_PAGE RUN_CODE
    POP  DE
    JR   Z,.end_start_keys
    DEC  A
    LD   (PAC_LEVEL_NUMBER),A                   ; Update level number
    JR   .skip_level_please
.skip_level1
    AND  %00100001                              ; ESC + Z   
    JR   NZ,.end_start_keys
.skip_level_please
    RES  DIP_RACK,D
.end_start_keys

    ; Set the PPI_A in output mode
    LD   BC,PPI_CONTROL + PPI_A_OUTPUT
    OUT  (C),C

    ; PSG Inactive (Needed in CPC+)
    DEC  B                                      ; B = PPI_C
    DEFB $ED,$71                                ; OUT (C),0 

    ; 2.- Process input and update DIPs
    
    ; Update the DIPs variables
    LD   A,D                                    ; Dip including controls
    CPL                                         ; Invert so set=pressed
    AND  %00001111                              ; Keep only direction bits
    JR   Z,.joy_done                            ; Skip if nothing pressed
    LD   C,A
    NEG
    AND  C                                      ; Keep least significant set bit
    CP   C                                      ; Was it the only bit?
    JR   Z,.joy_done                            ; Skip if so

.sm_last_controls
    LD   A,$00                                  ; Last valid (single) controls
    XOR  C                                      ; Check for differences
    OR   %11110000                              ; Convert to mask
    LD   C,A
    LD   A,D                                    ; Current controls
    OR   %00001111                              ; Release all directions
    AND  C                                      ; Press the changed key
    JR   .joy_multi                             ; Apply change but don't save

.joy_done
    LD   A,D
    LD   (.sm_last_controls + 1),A              ; update last valid controls
.input_done
    LD   A,D                                    ; use original value
.joy_multi
    LD   (DIP_5000),A
    LD   A,E
    LD   (DIP_5040),A
    
    RET

; --------------------------------------------------------------------------- 
; Map the current voice waveform, frequency and volume to the AY.
; ENTRY
;     A: Voice Waveform
;    HL: Voice Frequency and Volume
; EXIT
;     A: PSG Volume
;    HL: PSG Frequency
; --------------------------------------------------------------------------- 
map_sound
    LD   B,A                                    ; Save waveform

    LD   A,(HL)
    AND  %00001111
    ADD  A,A
    ADD  A,A
    ADD  A,A
    ADD  A,A
    LD   E,A
    INC  L      ; HL
    LD   A,(HL)
    AND  %00001111
    LD   D,A
    INC  L      ; HL
    LD   A,(HL)
    ADD  A,A
    ADD  A,A
    ADD  A,A
    ADD  A,A
    INC  L      ; HL
    INC  L      ; HL
    OR   D
    LD   D,A
    OR   E                                      ; Check for zero frequency
    LD   A,(HL)                                 ; Volume
    EX   DE,HL

    JR   NZ,.not_silent
    XOR  A                                      ; Zero frequency gives silence
.not_silent
    LD   D,A                                    ; Save volume

    LD   A,B
    CP   5                                      ; Waveform used when eating ghost?
    JR   Z,.eat_sound                           ; If so, don't divide freq by 8
    SRL  H
    RR   L
    SRL  H
    RR   L
    SRL  H
    RR   L
.eat_sound
    LD   A,H
    OR   $60                                    ; MSB of sound table
    LD   H,A
    RES  0,L                                    ; LSB is always %xxxxxxx0

    SET_PAGE SOUND_TABLE                        ; Make the sound table visible to the Z80
    
    LD   A,(HL)                                 ; Pick up LSB
    INC  L                                      ; Before "INC HL", but LSB is %xxxxxxx0
    LD   H,(HL)                                 ; Pick up MSB
    LD   L,A

    INC  C                                      ; LD C,RUN_CODE
    OUT  (C),C                                  ; Restore the Pac-Man RAM page
    
    LD   A,D                                    ; Restore Volume
    RET

; --------------------------------------------------------------------------- 
; Map the current sound chip frequencies to the AY and play it.
; --------------------------------------------------------------------------- 
; PSG Frequencies + Volumes for the actual frame
psg_sound_list
    DEFS 9

do_sound
    ; Convert Pac-Man voice 0
    LD   HL,FREQ_VOL_0                          ; Voice 0 freq and volume
    LD   A,(WAVEFORM_0)                         ; Voice 0 waveform
    CALL map_sound
    LD   (psg_sound_list + 2),HL
    LD   (psg_sound_list + 7),A

    ; Convert Pac-Man voice 1
    LD   HL,FREQ_VOL_1                          ; Voice 1 freq and volume
    LD   A,(WAVEFORM_1)                         ; Voice 1 waveform
    CALL map_sound
    LD   (psg_sound_list + 0),HL
    LD   (psg_sound_list + 6),A

    ; Convert Pac-Man voice 2
    LD   HL,FREQ_VOL_2                          ; Voice 2 freq and volume
    LD   A,(WAVEFORM_2)                         ; Voice 2 waveform
    CALL map_sound
    LD   (psg_sound_list + 4),HL
    LD   (psg_sound_list + 8),A

; --------------------------------------------------------------------------- 
; Play sound (unrolled loop)
; --------------------------------------------------------------------------- 
play_sound
    ; PSG Register 0 (Fine Tune Channel A)
    LD   HL,psg_sound_list
    LD   DE,PPI_C + PPI_PSG_SELECT
    LD   BC,PPI_C + PSG_REG_01
    OUT  (C),E                                  ; PPI_C + PPI_PSG_SELECT
    LD   B,>PPI_A
    DEFB $ED,$71                                ; PPI_A + PSG_REG_00
    LD   B,D
    DEFB $ED,$71                                ; PPI_C + PPI_PSG_INACTIVE
    DEC  B
    OUTI                                        ; PPI_A + Value
    EXX
    LD   DE,PPI_PSG_WRITE * 256 + PPI_PSG_SELECT
    LD   B,>PPI_C
    OUT  (C),D                                  ; PPI_C + PPI_PSG_WRITE
    OUT  (C),E                                  ; PPI_C + PPI_PSG_SELECT

    ; PSG Register 1 (Coarse Tune Channel A)
    EXX
    OUT  (C),C                                  ; PPI_A + PSG_REG_01
    INC  C                                      ; Update to the next PSG register
    LD   B,D
    DEFB $ED,$71                                ; PPI_C + PPI_PSG_INACTIVE
    DEC  B
    OUTI                                        ; PPI_A + Value
    EXX
    OUT  (C),D                                  ; PPI_C + PPI_PSG_WRITE
    OUT  (C),E                                  ; PPI_C + PPI_PSG_SELECT

    ; PSG Register 2 (Fine Tune Channel B)
    EXX
    OUT  (C),C                                  ; PPI_A + PSG_REG_02
    INC  C                                      ; Update to the next PSG register
    LD   B,D
    DEFB $ED,$71                                ; PPI_C + PPI_PSG_INACTIVE
    DEC  B
    OUTI                                        ; PPI_A + Value
    EXX
    OUT  (C),D                                  ; PPI_C + PPI_PSG_WRITE
    OUT  (C),E                                  ; PPI_C + PPI_PSG_SELECT

    ; PSG Register 3 (Coarse Tune Channel B)
    EXX
    OUT  (C),C                                  ; PPI_A + PSG_REG_03
    INC  C                                      ; Update to the next PSG register
    LD   B,D
    DEFB $ED,$71                                ; PPI_C + PPI_PSG_INACTIVE
    DEC  B
    OUTI                                        ; PPI_A + Value
    EXX
    OUT  (C),D                                  ; PPI_C + PPI_PSG_WRITE
    OUT  (C),E                                  ; PPI_C + PPI_PSG_SELECT

    ; PSG Register 4 (Fine Tune Channel C)
    EXX
    OUT  (C),C                                  ; PPI_A + PSG_REG_04
    INC  C                                      ; Update to the next PSG register
    LD   B,D
    DEFB $ED,$71                                ; PPI_C + PPI_PSG_INACTIVE
    DEC  B
    OUTI                                        ; PPI_A + Value
    EXX
    OUT  (C),D                                  ; PPI_C + PPI_PSG_WRITE
    OUT  (C),E                                  ; PPI_C + PPI_PSG_SELECT

    ; PSG Register 5 (Coarse Tune Channel C)
    EXX
    OUT  (C),C                                  ; PPI_A + PSG_REG_05
    LD   C,PSG_REG_08                           ; Update to the next PSG register
    LD   B,D
    DEFB $ED,$71                                ; PPI_C + PPI_PSG_INACTIVE
    DEC  B
    OUTI                                        ; PPI_A + Value
    EXX
    OUT  (C),D                                  ; PPI_C + PPI_PSG_WRITE
    OUT  (C),E                                  ; PPI_C + PPI_PSG_SELECT

    ; PSG Register 8 (Volume Channel A)
    EXX
    OUT  (C),C                                  ; PPI_A + PSG_REG_08
    INC  C                                      ; Update to the next PSG register
    LD   B,D
    DEFB $ED,$71                                ; PPI_C + PPI_PSG_INACTIVE
    DEC  B
    OUTI                                        ; PPI_A + Value
    EXX
    OUT  (C),D                                  ; PPI_C + PPI_PSG_WRITE
    OUT  (C),E                                  ; PPI_C + PPI_PSG_SELECT

    ; PSG Register 9 (Volume Channel B)
    EXX
    OUT  (C),C                                  ; PPI_A + PSG_REG_09
    INC  C                                      ; Update to the next PSG register
    LD   B,D
    DEFB $ED,$71                                ; PPI_C + PPI_PSG_INACTIVE
    DEC  B
    OUTI                                        ; PPI_A + Value
    EXX
    OUT  (C),D                                  ; PPI_C + PPI_PSG_WRITE
    OUT  (C),E                                  ; PPI_C + PPI_PSG_SELECT

    ; PSG Register 10 (Volume Channel C)
    EXX
    OUT  (C),C                                  ; PPI_A + PSG_REG_10
    LD   B,D
    DEFB $ED,$71                                ; PPI_C + PPI_PSG_INACTIVE
    DEC  B
    OUTI                                        ; PPI_A + Value
    EXX
    OUT  (C),D                                  ; PPI_C + PPI_PSG_WRITE
    DEFB $ED,$71                                ; PPI_C + PPI_PSG_INACTIVE

    RET

; --------------------------------------------------------------------------- 
; Scan a 32-byte block for changes, used for fast scanning of the Pac-Man 
; display.
; ENTRY:
;     B : Number of Pac-Man scanlines to scan
;    DE : Pointer to the Pac-Man display
;    HL : Pointer to one of the copy buffers of the Pac-Man display 
;
; NOTE: Is aligned on a 256-byte boundary for easy resuming of the scanning.
; --------------------------------------------------------------------------- 
    ALIGN 8             ; Align to the next 256-byte boundary
find_change
    ; The first search is not needed, because first row in the arcade is blank
    DEFS 3              ; 0
real_find_change
    INC  E
    INC  L

    REPT 30             ; 1 - 30
        LD   A,(DE)
        CP   (HL)
        RET  NZ
        INC  E
        INC  L
    ENDR

    LD   A,(DE)         ; 31
    CP   (HL)
    RET  NZ
    INC  DE             ; 16-bit increment as we may be at 256-byte boundary
    INC  HL

    DEC  B
    JP   NZ,real_find_change ; Jump too big for DJNZ

    POP HL              ; Junk return to update
    RET

; --------------------------------------------------------------------------- 
    include 'cpc_sprites_code.s'

; --------------------------------------------------------------------------- 
version_string
    DEFB "Pac-Man v1.0 by TotO and SyX"

; --------------------------------------------------------------------------- 
    END
