/*
 * dconv.c
 * Program to extract files in XDFS format
 * By D. R. Stacey
 * 5th December 1997
 */

#include			<stdio.h>
#include			<fcntl.h>

#define				BUFFER_SIZE		204800
#define				PATH_MAX		1024
#define				CAT_SIZE		62
#define				CAT_NAME		"__CATALOG__"

#ifndef				O_BINARY
#define				O_BINARY		0
#endif

typedef struct
{
	char			filename [ 10 ];
	char			lock_flag;
	unsigned int	load_address;
	unsigned int	exec_address;
	unsigned int	file_length;
	unsigned int	start_sector;
} FileInfo;

FileInfo			catalog [ CAT_SIZE ];
unsigned char		disc_buffer [ BUFFER_SIZE ];
int					catalog_entries = 0;
int					tube;


void construct_catalog ( int offset )
{
	int 			i, files_to_process, the_file;

	/*
	 * Location 0x05 of the first sector contains the number of files
	 * on the disc multiplied by 8.
	 */

	files_to_process = disc_buffer [ offset + 0x105 ];

	/*
	 * If there are no files to process then exit now.
	 */

	if ( ! files_to_process ) return;

	for ( the_file = 8; the_file <= files_to_process; the_file += 8 )
	{
		/*
		 * The directory is stored as the least significant seven
		 * bits of the last character of the filename.
		 */

		catalog [ catalog_entries ].filename [ 0 ] =
			disc_buffer [ offset + the_file + 7 ] & 0x7f;

		catalog [ catalog_entries ].filename [ 1 ] = '.';

		for ( i = 0; i < 7; i++ )
			catalog [ catalog_entries ].filename [ i + 2 ] =
				disc_buffer [ offset + the_file + i ];

		catalog [ catalog_entries ].filename [ 9 ] = 0;

		/*
		 * The locked flag is the most significant bit if the last
		 * character of the filename.
		 */

		catalog [ catalog_entries ].lock_flag =
			( disc_buffer [ offset + the_file + 7 ] & 0x80 ) ? 'L' : ' ';

		/*
		 * Now for the load, execution and length. These are pretty much
		 * all the same, a two byte block plus the most significant two
		 * most significant bits of each all packed into one byte.
		 *
		 * It is theoretically possible therefore to have an address such
		 * as 031900. In practice, however, if the most significant two bits
		 * were set then Acorn DFS mapped this to FFFF1900. This is why we
		 * need a couple of little hacks to sort this out below.
		 */

		catalog [ catalog_entries ].load_address =
			disc_buffer [ offset + 0x100 + the_file ] |
			disc_buffer [ offset + 0x101 + the_file ] << 8;

		tube = ( disc_buffer [ offset + 0x106 + the_file ] & 0xc0) >> 6;
		if ( tube == 3 ) tube = 0xff;
		catalog [ catalog_entries ].load_address |= ( tube << 16 );

		/*
		 * Execution address of file.
		 */

		catalog [ catalog_entries ].exec_address =
			disc_buffer [ offset + 0x102 + the_file ] |
			disc_buffer [ offset + 0x103 + the_file ] << 8;

		tube = ( disc_buffer [ offset + 0x106 + the_file ] & 0x0c ) >> 2;
		if ( tube == 3 ) tube = 0xff;
		catalog [ catalog_entries ].exec_address |= ( tube << 16 );

		/*
		 * Length of file.
		 */

		catalog [ catalog_entries ].file_length =
			( disc_buffer [ offset + 0x104 + the_file ]) |
			( disc_buffer [ offset + 0x105 + the_file ] << 8 ) |
			(( disc_buffer [ offset + 0x106 + the_file ] & 0x30 ) << 12 );

		/*
		 * Start sector of file. We will use this to extract the disc
		 * position of the data later.
		 */

		catalog [ catalog_entries ].start_sector =
			( disc_buffer [ offset + 0x107 + the_file ]) |
			(( disc_buffer [ offset + 0x106 + the_file ] & 0x03 ) << 8 );

		/*
		 * Increment the number of catalog entries that we know about
		 */

		catalog_entries ++;
	}
}


void main ( int argc, char** argv )
{
	char			c, cat_buffer [ PATH_MAX ], watford;
	unsigned char	filebyte;
	int				boot_option, catalog_writes, file_fd, i, j;
	FILE			*fp;

	/*
	 * Check that we have been passed a filename. I'll have to provide
	 * some proper parsing of the passed arguments if I extend the
	 * program to cope with double sided interleaved images.
	 */

	if ( argc != 2 )
	{
		fprintf ( stderr, "Syntax : dconv <filename>\n" );
		exit ( 1 );
	}

	/*
	 * We must clear the disc buffer as the disc image may be shorter
	 * than we first thought.
	 */

	for ( i = 0; i < BUFFER_SIZE; i++ )
		disc_buffer [ i ] = 0;

	/*
	 * Load the disc image into memory. The problem here is that we
	 * don't know how large the file is, so it must be read byte at a
	 * time.
	 */

	if (( file_fd = open ( argv [ 1 ], O_RDONLY | O_BINARY )) < 0 )
	{
		fprintf ( stderr, "dconv: file not found\n" );
		exit ( 1 );
	}

	i = 0;

	while (( i < BUFFER_SIZE ) && ( read ( file_fd, &filebyte, 1 )))
		disc_buffer [ i++ ] = filebyte;

	( void ) close ( file_fd );

	/*
	 * All discs will have their catalogue information stored in the
	 * first two sectors of the disc. We'll start to build up the
	 * catalog now.
	 */

	construct_catalog ( 0 );

	/*
	 * We support the Watford 62 catalog file format here. If the first
	 * 8 bytes of track 0, sector 2 are &AA then the disc is identified
	 * to be of the Watford 62 file format.
	 *
	 * Note that the current version of XDFS (0.90) will only support
	 * 31 files. Any others in the catalog file will be ignored.
	 */

	watford = 1;

	for ( i = 0; i < 8; i++ )
		if ( disc_buffer [ 0x200 + i ] != 0xaa ) watford = 0;

	if ( watford ) construct_catalog ( 0x200 );

	/*
	 * Now to write out the catalog to a file called __CATALOG__ in the
	 * current directory.
	 */

	if (( fp = fopen ( CAT_NAME, "w" )) == 0 )
	{
		fprintf ( stderr, "dconv: cannot create file %s\n", CAT_NAME );
		exit ( 1 );
	}

	/*
	 * The first row contains the number of catalog writes and the boot
	 * option. The number of writes is held at sector 1, offset 4. The
	 * boot option is stored in bits 4 and 5 of offset 6.
	 */

	catalog_writes = disc_buffer [ 0x104 ];
	boot_option = ( disc_buffer [ 0x106 ] & 0x30 ) >> 4;

	sprintf ( cat_buffer, "%2.2X %1.1d\n", catalog_writes, boot_option );

	if ( fputs ( cat_buffer, fp ) < 0 )
	{
		fprintf ( stderr, "dconv: error in writing to %s\n", CAT_NAME );
		fclose ( fp );
		unlink ( CAT_NAME );
		exit ( 1 );
	}

	/*
	 * The remainder of the file contains the actual catalog information.
	 */

	for ( i = 0; i < catalog_entries; i++ )
	{
		sprintf ( cat_buffer, "%-9s  %c  %6.6X %6.6X %6.6X %3.3X\n",
			catalog [ i ].filename, catalog [ i ].lock_flag,
			catalog [ i ].load_address, catalog [ i ].exec_address,
			catalog [ i ].file_length, catalog [ i ].start_sector );

		if ( fputs ( cat_buffer, fp ) < 0 )
		{
			fprintf ( stderr, "dconv: error in writing to %s\n", CAT_NAME );
			fclose ( fp );
			unlink ( CAT_NAME );
			exit ( 1 );
		}
	}

	fclose ( fp );

	/*
	 * Well that's the hard bit over with. We just have to save out the
	 * files now.
	 */

	for ( i = 0; i < catalog_entries; i++ )
	{
		for ( j = 0; j < 9; j++ )
			if ( catalog [ i ].filename [ j ] == 0x20 )
				catalog [ i ].filename [ j ] = 0;

		if (( file_fd = creat ( catalog [ i ].filename, 0644 )) < 0 )
		{
			fprintf ( stderr, "dconv: Error in opening %s\n",
				catalog [ i ].filename );
			exit ( 1 );
		}

		if ( write ( file_fd, &disc_buffer [ catalog [ i ].start_sector << 8 ],
			catalog [ i ].file_length ) < 1 )
		{
			fprintf ( stderr, "dconv: Error in writing %s\n",
				catalog [ i ].filename );
			close ( file_fd );
			unlink ( catalog [ i ].filename );
			exit ( 1 );
		}

		close ( file_fd );
	}
}
