/*
 * This is the ray tracing engine.  The ray is traced along horizontal and
 * vertical walls with separate sets of coordinates and step vectors.
 *
 * It currently handles:
 *		Texture mapped walls
 *		Mirroring walls with bitmaps.
 * 		Walls with transparent bitmaps.
 */

#include <assert.h>
#include "trace.hpp"
#include "div0.h"
#include "player.hpp"
#include "resource.hpp"
#include "wslice.hpp"
#include "vga.hpp"				// screenWidthBytes
#include "zox3d.hpp"			// pixelX, pixelY


#define MAX_TRACELEVEL 32
#define MAX_DEPTHRATIO 0x400000L


int hY, vX;						// Wall tracing position variables:
Fixed hX, vY;                   // (hX,hY) ray intersections with
								//         horizontal grid lines
								// (vX,vY) ray intersections with
								//         vertical grid lines

int dhY, dvX;					// The corresponding delta variables.
Fixed dhX, dvY;					//  (or step vectors, if you prefer.)

int hDominant;					// Flag: 1 if the dominant direction is
								// horizontally, 0 if vertically.

int directTrace;				// Flag: 1 if tracing directly to screen,
								// 0 if to the temporary buffer.
char *vgaBuf, 					// Points to the top pixel of screen column.
	*activeBuf, 				// Points to first pixel in active buffer.
	*tempBuf = 0;				// Points to first pixel in temporary buffer.
unsigned bufStepY;				// Number of bytes to add to activeBuf, to
								//  jump to the next pixel (below).

int traceLevel, 				// Current recursion level.
	mirroredX, mirroredY;		// Toggled each time the ray is mirrored in
								// X, Y directions respectively.

unsigned mirrorDepth;			// Current mirror level.
const unsigned maxMirrorDepth = MAX_TRACELEVEL;
								// maximum number of mirror reflections
								// during one trace

// This small table makes it possible to call the correct tracing function,
// typically by indexing with the value of hDominant.

TraceFunc *trace[2] =
	{
	traceVert,
	traceHorz
	};

// See TRACE.HPP for what SHOWSCAN means.

#ifdef SHOWSCAN

#include "vga.hpp"
#include "player.hpp"

// showMap(x,y,c) puts a pixel in the map window.  Useful for debugging
// only.

void showMap(Fixed x, Fixed y, char c)
	{
	x = (x-guyX>>12)+(screenWidth>>1);
	y = (y-guyY>>12)+(screenHeight>>2);
	putPixel(x, screenHeight-y, c);
	}

#endif


// useTempBuf() switches the tracing/drawing over to the temporary buffer,
// if it is not already using it.

inline void useTempBuf()
	{
	if (directTrace)
		{
		directTrace = 0;
		activeBuf = tempBuf;
		bufStepY = 1;
		}
	}

// flushTempBuf() switches back to drawing directly to the screen if
// prevTrace!=0, after flushing the temporary buffer to the screen.

inline void flushTempBuf(int prevTrace, WallSlice &wSlice)
	{
	if (prevTrace)
		{
		wSlice.drawFrom(tempBuf);
		directTrace = 1;
		activeBuf = vgaBuf;
		bufStepY = screenWidthBytes;
		}
	}


/*
 * wallHorz() is called when an intersection between the ray and a
 * horizontal wall (wall->type!=NOWALL) has been detected.
 */

int wallHorz(Wall *wall, Fixed depthRatio)
	{
	SHOWMAP(hX, INT2FIX(hY), wall->type);

	// Save old direct trace status.

	int directTraceOld = directTrace;

	// We're only interested in the magnitude of the depth ratio.  And
	// make sure it doesn't get too big.

	FixAbsDir(depthRatio);
	if (depthRatio > MAX_DEPTHRATIO)
		depthRatio = MAX_DEPTHRATIO;

	// Init a WallSlice with the bitmap number and intersection point.
	// Calculate wall height and elevation (relative to the player).

	WallSlice wSlice(wall->bmap, hX);
	wSlice.calcPos(depthRatio);

	// Draw the floor/ceiling up to the wall we've hit.

	wSlice.drawFloor();

	switch(wall->type & ~SOLID)
		{
		case OPAQUE:

			// A solid, simple wall - draw a solid, simple bitmap.

			wSlice.drawOpaque();
			break;

		case GLASS:

			// A transparent bitmap.  Switch to a temporary drawing buffer,
			// trace on recursively, put the transparent bitmap on top of the
			// result, then flush the buffer.

			if (wSlice.isSpecial())
				{
				useTempBuf();
				trace[hDominant](hDominant);  // ???
				wSlice.drawTransparent();
				flushTempBuf(directTraceOld, wSlice);
				}
			else
				wSlice.drawOpaque();
			break;

		case MIRROR:

			// A mirroring bitmap.  Switch to a temporary drawing buffer,
			// reflect the ray and find the new virtual positions of the
			// player, trace on recursively, put the transparent bitmap on
			// top of the result, then flush the buffer.

			if (mirrorDepth<maxMirrorDepth
				&& (!wSlice.isBitmapped() || wSlice.isSpecial()))
				{
				useTempBuf();

				mirrorDepth++;
				mirroredY ^= 1;
				dhY = -dhY;
				dvY = -dvY;
				pixelY = -pixelY;
				vY += INT2FIX(hY)-vY << 1;
				virtualGuyY += INT2FIX(hY)-virtualGuyY << 1;
				trace[hDominant](hDominant);
				mirrorDepth--;

				if (wSlice.isBitmapped())
					wSlice.drawTransparent();

				flushTempBuf(directTraceOld, wSlice);
				}
			else
				if (wSlice.isBitmapped())
					wSlice.drawOpaque();
			break;
		}
	return 1;
	}


/*
 * wallVert() handles ray intersections with a vertical wall.
 * The structure is identical to wallHorz(), so look above for comments.
 */

int wallVert(Wall *wall, Fixed depthRatio)
	{
	SHOWMAP(INT2FIX(vX), vY, wall->type);

	int directTraceOld = directTrace;

	FixAbsDir(depthRatio);
	if (depthRatio > MAX_DEPTHRATIO)
		depthRatio = MAX_DEPTHRATIO;

	WallSlice wSlice(wall->bmap, vY);
	wSlice.calcPos(depthRatio);

	wSlice.drawFloor();

	switch(wall->type & ~SOLID)
		{
		case OPAQUE:

			wSlice.drawOpaque();
			break;

		case GLASS:

			if (wSlice.isSpecial())
				{
				useTempBuf();
				trace[hDominant](!hDominant);
				wSlice.drawTransparent();
				flushTempBuf(directTraceOld, wSlice);
				}
			else
				wSlice.drawOpaque();
			break;

		case MIRROR:

			if (mirrorDepth<maxMirrorDepth
				&& (!wSlice.isBitmapped() || wSlice.isSpecial()))
				{
				useTempBuf();

				mirrorDepth++;
				mirroredX ^= 1;
				dhX = -dhX;
				dvX = -dvX;
				pixelX = -pixelX;
				hX += INT2FIX(vX)-hX << 1;
				virtualGuyX += INT2FIX(vX)-virtualGuyX << 1;
				trace[hDominant](!hDominant);
				mirrorDepth--;

				if (wSlice.isBitmapped())
					wSlice.drawTransparent();

				flushTempBuf(directTraceOld, wSlice);
				}
			else
				if (wSlice.isBitmapped())
					wSlice.drawOpaque();

			break;
		}

	return 1;
	}


/*
 * traceHorz() traces the ray when its direction is mostly horizontal,
 * i.e. dhX>dvY.  Most of the intersections will thus be with vertical
 * grid lines, i.e. lines of constant x.
 */

void traceHorz(int skip)
	{
	Wall *wall;
	Fixed depthRatio;

	// (newX,newY) must be declared outside the loop because of the
	// goto below.

	int   newX;
	Fixed newY;

	traceLevel++;

	// skipFirst is set if (vX,vY) should be moved without concern for
	// (hX,hY) in the first iteration.  This typically applies at the
	// very start of the trace, if (hX,hY) is behind the player and
	// (vX,vY) is even further behind.
	//
	// The goto *IS* ugly, but very effective.  Feel free to find a nicer
	// (just as fast) way!

	if (skip)
		{
		vX += dvX;
		vY += dvY;
		goto skip1;
		}

	for (;;)
		{

		// Find the next state of (vX,vY).  The order in which (vX,vY) and
		// (hX,hY) is updated is very important.  It must happen in strict
		// sequence along the ray (because of the mirrors)!  Therefore
		// we use the temporaries (newX,newY).

		newX = vX + dvX;
		newY = vY + dvY;

		if (FIX2INT(vY ^ newY))
			{

			// vY will change, so check the 'exposed' horizontal
			// wall between (vX,vY) and (newX,newY).
			// (Note that since |dvY|<=0x10000, the change must be in bit 16.)

			hX += dhX;
			hY += dhY;

			wall = &map[hY][FIX2INT(hX)].horz;
			if (wall->type)
				{
				FixDiv(pixelX, hX-virtualGuyX, depthRatio)
				if (wallHorz(wall, depthRatio))
					break;
				}
			}

		// Now we're ready to update (vX,vY) and check the new map square.

		vX = newX;
		vY = newY;

	skip1:
		wall = &map[FIX2INT(vY)][vX].vert;
		if (wall->type)
			{
			FixDiv(pixelX, INT2FIX(vX)-virtualGuyX, depthRatio)
			if (wallVert(wall, depthRatio))
				break;
			}
		}

	traceLevel--;
	}

/*
 * traceVert() works mostly vertically.  See traceHorz() for comments.
 */

void traceVert(int skip)
	{
	Wall *wall;
	Fixed depthRatio;
	Fixed newX;
	int   newY;

	traceLevel++;

	if (skip)
		{
		hX += dhX;
		hY += dhY;
		goto skip1;
		}

	for (;;)
		{
		newX = hX + dhX;
		newY = hY + dhY;

		if (FIX2INT(hX ^ newX))
			{
			vX += dvX;
			vY += dvY;

			wall = &map[FIX2INT(vY)][vX].vert;
			if (wall->type)
				{
				FixDiv(pixelY, vY-virtualGuyY, depthRatio);
				if (wallVert(wall, depthRatio))
					break;
				}
			}

		hX = newX;
		hY = newY;

	skip1:
		wall = &map[hY][FIX2INT(hX)].horz;
		if (wall->type)
			{
			FixDiv(pixelY, INT2FIX(hY)-virtualGuyY, depthRatio);
			if (wallHorz(wall, depthRatio))
				break;
			}
		}

	traceLevel--;
	}

