#pragma inline
#include "joystick.hpp"


// Define the global Joystick object which represents joystick A.

Joystick joystickA(0);


// Define the static members of the Gamecard class.

UINT Gamecard::xa, Gamecard::ya, Gamecard::xb, Gamecard::yb;
UCHAR Gamecard::buttons;
BOOL Gamecard::present = 0;


// Gamecard::reset() tries to detect a gamecard by polling it.
// If the state parameter is TRUE, Gamecard is enabled if it was found.
// Otherwise it will be disabled.  In any case a BOOL will be returned
// indicating wether the gamecard was found (TRUE) or not (FALSE).

BOOL Gamecard::reset(BOOL state)
	{
	present = TRUE;				// force poll() to at least TRY detecting
	BOOL temp = poll();
	present = state && temp;	// enable Gamecard if caller wants to, and
								//  if it is there.
	return temp;				// return with found/not found indicator
	}

// Gamecard::poll() triggers the gamecard to start countdown for all the
// axes.  If present is set to FALSE, the entire thing is skipped.

BOOL Gamecard::poll()
	{
	if (!present)
		return FALSE;

	xa=ya=xb=yb=1;				// 1 is the minimum coordinate

	asm	mov		dx, 201h
	asm	out		dx, al			// trigger the game card
	asm	mov		cx, 0xffff		// timeout loop count
	asm	xor		ah, ah			// ah holds which axes have not completed
	asm	xor		bh, bh			// bh==0 all the way through
	asm	cli

joyLoop:
	asm	xor		ah, ah			// ah is still 0 below if all axes are done
	asm	in		al, dx

	asm	shr		al, 1			// bit 0 -> x axis joystick A
	asm	setc	bl
	xa += _BX;
	asm	add		ah, bl

	asm	shr		al, 1			// bit 1 -> y axis joystick A
	asm	setc	bl
	ya += _BX;
	asm	add		ah, bl

// The following sections provide counting for joystick B.  I have not
// been able to test this, so the 2. joystick is not supported yet.
/*
	asm	shr		al, 1			// bit 2 -> x axis joystick B
	asm	setc	bl
	xb += _BX;
	asm	add		ah, bl

	asm	shr		al, 1			// bit 3 -> y axis joystick B
	asm	setc	bl
	yb += _BX;
	asm	add		ah, bl
*/
	asm	or		ah, ah			// have all been triggered?
	asm	loopne	joyLoop			// loop until all triggered or timeout

	asm	sti
	asm	not		al
	asm	and		al, 0fh
	buttons = _AL;				// keep record of which buttons are held down

	return (present = (_CX != 0)); // if we timed out, disable further polling
	}


// Joystick constructor links this instance to the correct variables in the
// Gamecard class.

Joystick::Joystick(int j, UINT res)
	: joyNo(j&1), dx(joyNo?Gamecard::xb:Gamecard::xa),
	  dy(joyNo?Gamecard::yb:Gamecard::ya), resolution(res),
	  button1(joyNo ? 4 : 1)
	{
	if (!Gamecard::isPresent())
		Gamecard::reset();		// try resetting Gamecard if not enabled
	calibrate();				// in any case, calibrate.
	}


// Joystick::getPosition() sets the parameters to the current position of
// the joystick.  If the position exceeds the last extrema in any way,
// the extrema are updated.  Thus, this function also calibrates this
// joystick instance.
//
// NOTE: Gamecard::poll() MUST be called to update the joystick positions
// before reading them with Joystick::getMotion() !!  This is so because
// it shall not be neccessary to poll the gamecard twice if you would want
// to get position on both joysticks.  Polling is slow.

void Joystick::getPosition(int &jx, int &jy)
	{
	if (minX == 0)
		{
		jx = jy = 0;
		return;
		}
	if (dx < minX)
		minX = dx;
	else
		if (dx > maxX)
			maxX = dx;
	if (dy < minY)
		minY = dy;
	else
		if (dy > maxY)
			maxY = dy;
	jx = (maxX == minX) ? jx = 0 :
		long(resolution) * (dx-minX) / (maxX-minX) - (resolution>>1);
	jy = (maxY == minY) ? jy = 0 :
		long(resolution) * (dy-minY) / (maxY-minY) - (resolution>>1);
	}


// Joystick::calibrate() is called once to INITIATE calibration, i.e. it
// resets the joystick position extrema.  The actual calibration takes
// place as the player moves the joystick, and the program calls
// Gamecard::poll() and Joystick::getPosition().

void Joystick::calibrate()
	{
	if (Gamecard::isPresent())
		{
		Gamecard::poll();
		minX = maxX = dx;
		minY = maxY = dy;
		}
	else
		minX = 0;
	}


// Joystick::isButtonPressed() returns TRUE if the indicated button is
// currently pressed, FALSE if not.  The parameter must be either
// 0, for the first button, or 1 for the second.  The return value
// is undefined for other parameters.

BOOL Joystick::isButtonPressed(int b)
	{
	return (Gamecard::buttons & (button1 << b) != 0);
	}
