/***************************************************************************

ILBM.HPP -


		INFORMATION ON THE IFF-LBM GRAPHICS STORAGE FORMAT

						Robert Schmidt,
								June 27. 1991

						Updated May 28, 1992
								June 2, 1992



An IFF file consists of chunks, each in the following general format:

Offset 0:       Chunk ID, 4 characters: A unique 4 letter word (ASCII)
				identifying the following data.  For example, ILBM identifies
				a chunk stored in the LBM graphics format.
Offset 5:       Chunk Size, long int: Number of bytes occupied by the chunk
				data.  The ID and Size is *not* counted herein, i.e.,
				immideately after this long int, the given number of bytes
				is the chunk data.
Offset 9:       Chunk data: Occupies the number of bytes given in the Size
				field above.
Offset n:       Next chunk...

NOTE:   If the size is an odd number, the chunk data must be padded with one
		extra dummy byte, making the total number of bytes even.

Recursive chunks are fully supported, and is really the crucial point of
the IFF file format.  I.e. one or more chunks may be contained in a parent
chunk.

A proper IFF file is identified by verifying that the FORM chunk is the parent
chunk of all other chunks in the file.  The size of the form chunk should
equal the file size minus the ID and Size fields (8 bytes).

The ILBM/PBM chunk does not specify any chunk size, for some unknown reason.
Ask Electronic Arts, they made this standard...

A LBM file may contain the following chunks which must reside inside
the ILBM chunk (which in turn lies inside the FORM chunk):
		CMAP :  The Color MAP, containing information on the palette for
				every color in the picture.  Check BMHD to obtain number of
				colors, planes, size etc.
		BMHD :  BitMap HeaDer, containing vital information on bitmap.
				(Se typedef below)
		BODY :  The actual Bitmap body, which may be compressed.  All size
				and orientation info is contained in BMHD.  (See #definitions
				and typedefs below)
		CRNG :  Defined color cycling ranges and their speeds, directions,
				etc.  I haven't tried to figure out the format of this one.
		TINY :  Iconic representation of the image, used for quick browsing
				in DeLuxePaint II Enhanced and, more extensively, in Gallery.
				Offset 0:   UWORD xSize, ySize
				Offset 4:	Rest of image data.  Follows the same
							compression format as BODY.
		DPPS :  I haven't checked this out yet, probably something like
				Deluxe Paint PreferenceS, like fill styles etc.

Electronic Arts now has two standards for their IFF BitMap files.  The oldest,
ILBM (InterLeaved BitMap), was introduced with DeLuxe Paint II.  The BODY
chunk of this format contained the data planewise for each line.  I.e. for
a 16-color picture this would look like:

	line 1, plane 0 - compressed blue component data
	line 1, plane 1 - green component
	line 1, plane 2 - red component
	line 1, plane 3 - intensity component
	line 2, plane 0 ...
	  :
	  :

This was because the Amiga was where DP had the most impact.  On the Amiga,
the screen is always plane-oriented.  On the PC, this works nicely with
the EGA and VGA 16-color modes, however, for the 256-color modes that is
becoming more popular, the process of translating plane-oriented data to
screen memory is slow and cumbersome.

Therefore the PBM format was introduced with DeLuxe Paint II Enhanced.
This is a logical enhancement/extension of the functionality introduced with
ILBM.  A subchunk TINY was introduced, and the orientation of image data in
this chunk and in the BODY chunk was more closely related to the actual
screen memory orientation.  Simply:

	line 1 - compressed raw data
	line 2 - ...
	  :
	  :

The compression involved is actually quite simple, and is a variant of the
well-known RLE (Run Length Encoding).  What follows is a rude decompression
algorithm:

	1)	read a signed byte, call it count
	2)	if count is < 0, then the following byte is to be repeated
		-count+1 times
	3)  if count is >= 0, then the following count+1 bytes is to be
		stored directly
	4)	repeat from 1 if we've not reached end of chunk


***************************************************************************/

#ifndef _ILBM_HPP
#define _ILBM_HPP

#ifndef __COMPACT__
# ifndef __LARGE__
#  ifndef __HUGE__
#   error Large data model required for ILBM library!
#  endif
# endif
#endif

#include <fstream.h>

// Standard IFF types

typedef unsigned char   UBYTE;  //  8 bits unsigned
typedef short           WORD;   // 16 bits signed
typedef unsigned short  UWORD;  // 16 bits unsigned
typedef long            LONG;   // 32 bits signed

typedef unsigned long	ULONG;

// Id typedef and builder for BC++
typedef LONG    Id;
#define MakeId(a,b,c,d) ((long)(a)<<24 | (long)(b)<<16 | (c)<<8 | (d))

// Various Ids
#define Id_FORM MakeId('F','O','R','M')

// ILBM Ids
#define Id_ILBM MakeId('I','L','B','M')
#define Id_PBM  MakeId('P','B','M',' ')
#define Id_BMHD MakeId('B','M','H','D')
#define Id_CMAP MakeId('C','M','A','P')
#define Id_BODY MakeId('B','O','D','Y')
#define Id_CRNG MakeId('C','R','N','G')
#define Id_TINY MakeId('T','I','N','Y')

enum Compression	{ NOCOMP, BYTERUN1 };
enum Masking		{ NOMASK, MASK, TRANSPARENT, LASSO };

#pragma option -po-

class ColorMap;
class Body;
class IffBitMap;
class IffStream;

// Generic chunk class

class Chunk
	{
	protected:
		Id      		id;
		ULONG			size;
		void			*data;

	public:
		Chunk()			{ id = size = 0; data = 0; }
		~Chunk()		{ delete[] data; }
		ULONG getSize()
			{ return size; }
		void *getData()
			{ return data; }

	friend IffBitMap;
	friend IffStream& operator>>(IffStream&, Chunk&);
	};


struct ChunkHeader
	{
	Id			    id;
	LONG    		size;
	};

struct ColorRegister
	{
	unsigned char	red, green, blue;
	};

class ColorMap : public Chunk
	{
	public:
		enum MapType 	{ C16X4, C256X3 };
		int out(MapType);		//output to VGA DAC
		ColorRegister *getMap()
			{ return (ColorRegister *)data; }
	};

class IffBitMapHeader
	{
	protected:
		UWORD           w, h;			//width and height
		WORD            x, y;			//position
		UBYTE           nPlanes;
		Masking         masking;
		Compression 	compression;
		UBYTE           pad1;
		UWORD           transparentColor;
		UBYTE           xAspect, yAspect;
		WORD            pageWidth, pageHeight;
	public:
		friend IffBitMap;
		friend IffStream& operator>>(IffStream&, IffBitMapHeader&);
	};

class Body : public Chunk
	{
	public:
		int uncompress(Compression, UWORD);
	};


class IffBitMap
	{
	protected:
		Id				iffId;			//to distinguish PBM and ILBM
		void load(const char *);		//load from file
	public:
		IffBitMapHeader bmhd;
		ColorMap		cmap;
		Body			body;

		IffBitMap()	{};
		IffBitMap(const char *fn) { load(fn); } //initialize from file
		~IffBitMap() {};
		UWORD getWidth()	{ return bmhd.w; }
		UWORD getHeight()	{ return bmhd.h; }
		int uncompress();			//uncompress body chunk
		int showMode13h();
	};

class IffStream : ifstream
	{
		char			isForm;
		Id  			type;			//what kind of IFF? ILBM, PBM etc.
		ChunkHeader		activeChunk;
		long			formSize;
		long			offsetNextChunk;
	public:
		IffStream(const char *name);
		~IffStream() {};
		unsigned char getByte();
		unsigned getWord()
			{ return unsigned(getByte())<<8 | unsigned(getByte()); }
		long getLong()
			{ return long(getWord())<<16 | long(getWord()); }
		Id iffType()
			{ return isForm ? type : 0; }
		ChunkHeader nextChunk();

		friend IffStream& operator>>(IffStream&, Chunk&);
	};

#pragma option -po.

#endif
