#ifndef _FIXED_H
#define _FIXED_H

#pragma inline

/*
 * This header file provides a C++ program with the neccessities for
 * handling 16.16 fixed point variables, i.e. variables whose value has
 * been scaled by a constant so that the integer part is in the upper 16
 * bits, and the fractional part is in the lower 16 bits.
 *
 * Note that the definitions in this header file are tailored for 16-bit
 * operation.  Much must be changed in a 32-bit C/C++ environment (where
 * ints will be 32-bit).
 * 
 * IMPORTANT WARNING!!!!
 * 
 * Turning the inline expansion option (-vi) off can break code compiled
 * with these macros.  For example:
 * 
 * FixDiv(a, FixAbs(b), c);
 * 
 * will break, since a is first put into edx, then the FixAbs function is
 * called, which returns its result in dx:ax, thereby destroying edx.
 * 
 */

#ifdef __cplusplus
extern "C" {
#endif


// SIGN is kind of generic, and works for any scalar type.  Formally, it
// returns x/abs(x).

#define SIGN(x) ((x)>0?1:((x)<0?-1:0))


// Here follows some constants which are handy for conversions between
// fixed point variables and other integral types.

#define FIXFRAC			16						// number of bits to shift
#define FIXRATIO		(1L<<FIXFRAC)			// int/Fixed ratio
#define INT2FIX(i)		(Fixed(i) << FIXFRAC)	// use to convert int to Fixed
#define FIX2INT(f)		((f) >> FIXFRAC)		// use to convert Fixed to int
#define FIX2FLOAT(f)	((f)/(0.0+FIXRATIO))	// use to convert Fixed to float
#define MAXFIX			0x7fffffff				// the maximum Fixed value

#define TRUNC_FIXED(x) fixed((x) & 0xffff0000)	// use to mask off fractional
												//  part

//===================================================================
/*
 *  The DEGxxx constants define what word values correspond to which angle.
 * 0 always is the minimum angle.
 * 0xffff always is the maximum angle.
 * (Note that 360 degrees == 0 degrees, however.)
 */

#define DEG0 0
#define DEG45 (0x2000)
#define DEG90 (DEG45*2)
#define DEG180 (DEG90*2)
#define DEG270 (DEG90*3)
#define DEG360 DEG0


//===================================================================
/*
 * Now define the Fixed types themselves.
 * Fixed is simply a long (in BC++, 16-bit).  This should be changed to
 *  int in a 32-bit environment.
 * UFixed is the corresponding unsigned fixed point type.  This is currently
 *  UNUSED in Zox3D, and is NOT supported.
 * FixAngle is a kind of a fixed point angle type, consisting of a 16-bit
 *  word (see above, and FixSin() and FixCos() in FIXED.ASM).
 */

typedef long Fixed;
//typedef unsigned long UFixed;
typedef unsigned int FixAngle;


//===================================================================
/*
 * The following macros are quite useful, as they are guaranteed
 * to be compiled inline, so I avoid the overhead of far CALLs and RETs.
 * None of the macros perform corrective rounding of the result, though.
 *
 * Note that I take advantage of an UNDOCUMENTED feature of BC++ (3.1
 * and 4.0), which makes the 32-bit versions of the general purpose
 * registers available in inline assembly.  This is not mentioned anywhere
 * in the docs.
 *
 * To play it safe, you'll want to avoid handing expressions to these
 * macros which are more complex that a simple variable or constant.
 * In FixMul, for example, the assignment _EBX = b must NOT destroy _EAX!
 * Through experiment, you might find expressions which work OK.
 * See the various Zox3D files for suggestions on that topic.
 */

// c = a*b;

#define FixMul(a,b,c)			\
	{							\
	_EAX = a;					\
	_EBX = b;					\
	asm	imul	ebx;			\
	asm	shld	edx, eax, 16;	\
	c = _EDX;					\
	}

// c += a*b;

#define FixMulAdd(a,b,c)		\
	{							\
	_EAX = a;					\
	_EBX = b;					\
	asm	imul	ebx;			\
	asm	shld	edx, eax, 16;	\
	c += _EDX;					\
	}

// c = a/b;

#define FixDiv(a,b,c)			\
	{							\
	_EDX = a;					\
	_EBX = b;					\
	asm	xor		eax, eax;		\
	asm	shrd	eax, edx, 16;	\
	asm	sar		edx, 16;		\
	asm	idiv	ebx;			\
	c = _EAX;					\
	}


//===================================================================
// Square root and square routines defined in FIXED.ASM.
// Neither is used in Zox3D, but they are provided for your convenience.

extern Fixed FixSqrt(Fixed);
extern Fixed FixSqr(Fixed);
//extern UFixed UFixSqrt(UFixed);
//extern UFixed UFixSqr(UFixed);


//===================================================================
// Trigonometry functions.  Definitely used in Zox3D.

extern Fixed FixCos(FixAngle);
extern Fixed FixSin(FixAngle);


//===================================================================
// Absolute value functions.  FixAbs is for returning the absolute
// value of a value, FixAbsDir is for immediately setting a variable
// to its absolute value.
//
// To avoid nasty surprises in conjuction with the FixMul/FixDiv macros
// above, ensure that inline expansion (-vi) is turned on.

inline Fixed FixAbs(Fixed x)
	{
	return (x<0)?-x:x;
	}

#define FixAbsDir(a) if (a<0) asm neg [dword ptr a]

// New FixAbsDir() with no conditional jumps.  Kinda neat,
// but no speed gain though... :-(

/*
#define FixAbsDir(a)			\
	{							\
	asm mov		eax, [dword ptr a]; \
	asm cdq;					\
	asm xor		eax, edx;		\
	asm sub		eax, edx;		\
	asm mov		[dword ptr a], eax; \
	}
*/

#ifdef __cplusplus
}
#endif

#endif
