/*

  ElectrEm (c) 2000 Thomas Harte - an Acorn Electron Emulator

  This is open software, distributed under the GPL 2, see 'Copying' for details

*/
#include "wd1770.h"
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
FILE *temp;

#include "6502.h"
extern C_6502ULA electron;

C_WD1770::C_WD1770(void)
{
	temp = fopen("disk.log", "wt");
	side1 = side2 = cside = NULL;
}

C_WD1770::~C_WD1770(void)
{
	fclose(temp);
	if(side1)
	{
		side1 = NULL;
		free(side1);
	}
	if(side2)
	{
		side2 = NULL;
		free(side2);
	}
	cside = NULL;
}

void C_WD1770::Close(void)
{
	disk.Close();
}

bool C_WD1770::Open(char *name)
{
	broken_word tval;

	present = false;
	disk.Open(name, "r", 5, 0);
	if(disk.FindId(0x0200))
	{
		disk.ReadSome(&disk_id, 1);

		disk.ReadSome(&tval.b.l, 1);
		disk.ReadSome(&tval.b.h, 1);
		sector_size = tval.a;

		disk.ReadSome(&tval.b.l, 1);
		tval.b.h = 0;
		sectors_in_track = tval.a;

		disk.ReadSome(&tval.b.l, 1);
		tval.b.h = 0;
		tracks = tval.a;

		disk.ReadSome(&tval.b.l, 1);
		if((tval.b.l != 0) && (tval.b.l !=3))
			return true;

		disk.FindId(0x0201);
		disk.ReadSome(&tval.b.l, 1);
		if(tval.b.l&128)
		{
			len2 = disk.GetLength() - 1;
			side2 = (unsigned __int8 *)malloc(len2);
		 	disk.ReadAll(side2);
		}
		else
		{
			len1 = disk.GetLength() - 1;
			side1 = (unsigned __int8 *)malloc(len1);
		 	disk.ReadAll(side1);
		}
		disk.FindId(0x0201);
		disk.ReadSome(&tval.b.l, 1);
		if(tval.b.l&128)
		{
			len2 = disk.GetLength() - 1;
			side2 = (unsigned __int8 *)malloc(len2);
		 	disk.ReadAll(side2);
		}
		else
		{
			len1 = disk.GetLength() - 1;
			side1 = (unsigned __int8 *)malloc(len1);
		 	disk.ReadAll(side1);
		}

		present = true;
	}

	cside = side1;
	datalen = len1;

//	fprintf(temp, "datalen : %d side1 : %d[%d] side2 : %d[%d]\n", datalen, (int)side1, len1, (int)side2, len2);

	return true;
}

void C_WD1770::Reset(void)
{
	regs[STATUS]	= regs[TRACK] = regs[SECTOR] = regs[DATA] = 0;
	regs[CONTROL]	= 1;
	instr			= NONE;
}

void C_WD1770::WriteToRegister(int offset, unsigned __int8 value)
{
	switch(offset)
	{
		case 0 :	//drive control
			regs[CONTROL] = value;
//			fprintf(temp, "Drive control : %d\n", value);
			if(!disk.HasFile())
				regs[CONTROL] = 0xff;

			/* first guess :

					0,1 : drive select
					5 : density select (1 = double
			*/
		break;

		case 4 :	//instruction register
			if(!disk.HasFile())
			{
				regs[STATUS] = WD_R_FLAG;
			}
			else
				switch(instr)
				{
					default :
/*						if((value >> 4) == 13)
							fprintf(temp, "ignoring force interrupt\n");
						else
							fprintf(temp, "what the?\n");*/
					break;

					case NONE :
						switch(value >> 4)
						{
							case 0 : //restore		[type 1]
//								fprintf(temp, "Restore command\n");

								track = regs[TRACK] = 0;
								regs[STATUS] = WD_M_FLAG | WD_S_FLAG;
							break;

							case 1 : //seek			[type 1]
//								fprintf(temp, "Seek command\n");
								track = regs[TRACK] = regs[DATA];
								regs[STATUS] = WD_M_FLAG | WD_S_FLAG;
								regs[STATUS] |= track ? WD_Z_FLAG : 0;
							break;

							case 2 :
							case 3 : //step			[type 1]
//								fprintf(temp, "Step command\n");
							break;

							case 4 :
							case 5 : //step in		[type 1]
//								fprintf(temp, "Step in command\n");
								track++;
								if(value&16)
									regs[TRACK] = track;
								regs[STATUS] = WD_M_FLAG | WD_S_FLAG | WD_Z_FLAG;
							break;

							case 6 :
							case 7 : //step out		[type 1]
//								fprintf(temp, "Step out command\n");
								track--;
								if(value&16)
									regs[TRACK] = track;

								regs[STATUS] = WD_M_FLAG | WD_S_FLAG;
								regs[STATUS] |= track ? WD_Z_FLAG : 0;
							break;

							case 8 :
							case 9 : //read sector		[type 2]
								instr = (value&16) ? READ_SECTOR_TRACK : READ_SECTOR;
								regs[STATUS] = WD_M_FLAG | WD_D_FLAG | WD_B_FLAG;
								sector = regs[SECTOR];
								sector_position = 0;

//								fprintf(temp, "Read sector %c [track %d/%d sector %d abs : %d] command [%d %d]\n", (value&16) ? 'T' : 'N', track, regs[TRACK], regs[SECTOR], ((track*sectors_in_track + sector) * sector_size), sector_size, sectors_in_track);
							break;

							case 10 :
							case 11 : //write sector	[type 2]
//								fprintf(temp, "Write sector command\n");
							break;

							case 12 : //read address	[type 3]
//								fprintf(temp, "Read address command\n");
							break;

							case 14 : //read track		[type 3]
//								fprintf(temp, "Read track command\n");
							break;

							case 15 : //write track		[type 3]
//								fprintf(temp, "Write track command\n");
							break;

							case 13 : //force interrupt	[type 4]
//								fprintf(temp, "Force interrupt command\n");
							break;
						}
					break;
				}
		break;

		case 5 :	//track registers
		case 6 :
		case 7 :	//data & sector register
			regs[offset-4] = value;
//			fprintf(temp, "w to %d\n", offset-4);
		break;
	}
}

unsigned __int8 C_WD1770::ReadFromRegister(int offset)
{
	unsigned __int8 ret;
	int addr;

	switch(offset)
	{
		default :
		break;

		case 0 :
		return regs[CONTROL];

		case 4 :
		return regs[STATUS];

		case 5 :
		case 6 :
		return regs[offset-4];

		case 7 :
			switch(instr)
			{
				default :
				case NONE :
//					fprintf(temp, "reading data?\n");
				return regs[DATA];

				case READ_SECTOR :
					addr = ((track*sectors_in_track + sector) * sector_size) + sector_position;

					if(addr < datalen)
						ret = cside[addr];
					else
					{
//						fprintf(temp, "out of range read\n");
						ret = 255;
					}

					regs[DATA] = ret;

					sector_position++;

					if(sector_position == sector_size)
					{
						instr = NONE;
						regs[STATUS] = WD_M_FLAG;
					}
				return ret;
			}
		break;
	}

	return 0;
}