/*

  ElectrEm (c) 2000 Thomas Harte - an Acorn Electron Emulator

  This is open software, distributed under the GPL 2, see 'Copying' for details

*/
#include "gfx.h"
#include <malloc.h>

#ifdef TARGET_SVGALIB
#include <vgamouse.h>

void C_gfx_api::UpdateMouse(int button, int dx, int dy)
{
	char *temp;
	int yc, xc, y1, y2, x1, x2, ex, ey;

	temp = (char *)malloc((mwidth + abs(dx)) * (mheight + abs(dy)));

	y1 = my;
	y2 = my + (dy >> 5);
	if(y2 < 0)
		y2 = 0;
	if(y2 >= 512)
		y2 = 511;
	ey = y2;
	if(y2 < y1)
	{
		y2 ^= y1; y1 ^= y2; y2 ^= y1;
	}
	y2 += mheight;

	x1 = mx;
	x2 = mx + (dx >> 5);
	if(x2 < 0)
		x2 = 0;
	if(x2 >= 640)
		x2 = 639;
	ex = x2;
	if(x2 < x1)
	{
		x2 ^= x1; x1 ^= x2; x2 ^= x1;
	}
	x2 += mwidth;

	for(yc = y1; yc < y2; yc++)
	{
		vga_getscansegment((unsigned char *)&temp[(yc-y1)*(x2-x1)], tx+x1, ty+yc, x2-x1);
	}

	yc = mheight;
	while(yc--)
	{
		memcpy(&temp[((my+yc-y1)*(x2-x1)) + (mx-x1)], &backup[yc * mwidth], mwidth);
	}

	mx = ex;
	my = ey;

	yc = mheight;
	while(yc--)
	{
		memcpy(&backup[yc *mwidth], &temp[((my+yc-y1)*(x2-x1)) + (mx-x1)], mwidth);

		xc = mwidth;
		while(xc--)
			temp[((my+yc-y1)*(x2-x1)) + (mx-x1) + xc] = cursor[(yc * mwidth) + xc] ? cursor[(yc * mwidth) + xc] : temp[((my+yc-y1)*(x2-x1)) + (mx-x1) + xc];
	}

	for(yc = y1; yc < y2; yc++)
	{
		vga_drawscansegment((unsigned char *)&temp[(yc-y1)*(x2-x1)], tx+x1, ty+yc, x2-x1);
	}

	mbutton = button;

	free(temp);
}

void mouse_handler(int button, int dx, int dy, int dz, int drx, int dry, int drz)
{
	mouse->UpdateMouse(button, dx, dy);
}

void C_gfx_api::EnableMouse(void)
{
	int c, x, y;
	char *mouse_space;

	mouse_space = (char *)malloc(mwidth*mheight);
	y = mheight;
	while(y--)
	{
		vga_getscansegment((unsigned char *)&backup[y * mwidth], tx+mx, my+ty+y, mwidth);

		x = mwidth;
		while(x--)
		{
			mouse_space[(y * mwidth) + x] = cursor[(y * mwidth) + x] ? cursor[(y * mwidth) + x] : backup[(y * mwidth) + x];
		}

		vga_drawscansegment((unsigned char *)&mouse_space[y * mwidth], tx+mx, my+ty+y, mwidth);
	}
	free(mouse_space);
}

void C_gfx_api::DisableMouse(void)
{
	int y;
	y = mheight;
	while(y--)
	{
		vga_drawscansegment((unsigned char *)&backup[y *mwidth], tx+mx, my+ty+y, mwidth);
	}
}
#endif

#ifdef TARGET_WIN32
void C_gfx_api::Restore(void)
{
	if(gui_active)
	{
		SetCursor(LoadCursor(NULL, IDC_ARROW));
		EndRedraw();
	}
}
#endif

void C_gfx_api::BeginRedraw(void)
{
	#ifdef TARGET_ALLEGRO
	show_mouse(NULL);
	blit(screen, backupscr, tx, ty, 0, 0, scw, sch);
	#endif

	#ifdef TARGET_SVGALIB
	int y;
	DisableMouse();
	y = sch;
	while(y--)
		vga_getscansegment((unsigned char *)&backupscr[y*scw], tx, ty+y, scw);
	#endif

	#ifdef TARGET_WIN32
	DDSURFACEDESC2 desc;
	int y;
	char *base;

	DDRAW_INIT_STRUCT(desc);

	if(front4->Lock(NULL, &desc, DDLOCK_WAIT | DDLOCK_READONLY, NULL) == DD_OK)
	{
		base = (char *)desc.lpSurface;
		y = sch;
		while(y--)
		{
			memcpy(&backupscr[y*scw], &base[(y+ty)*desc.lPitch + tx], scw);
		}
		front4->Unlock(NULL);
	}

	#endif
}

void C_gfx_api::EndRedraw(void)
{
	#ifdef TARGET_ALLEGRO
	blit(backupscr, screen, 0, 0, tx, ty, scw, sch);
	show_mouse(screen);
	#endif

	#ifdef TARGET_SVGALIB
	int y;
	y = sch;
	while(y--)
		vga_drawscansegment((unsigned char *)&backupscr[y*scw], tx, ty+y, scw);
	EnableMouse();
	#endif

	#ifdef TARGET_WIN32
	DDSURFACEDESC2 desc;
	int y;
	char *base;
	HRESULT res;

	DDRAW_INIT_STRUCT(desc);

	res = front4->Lock(NULL, &desc, DDLOCK_WAIT | DDLOCK_WRITEONLY, NULL);

	if(res == DDERR_SURFACELOST)
	{
		front4->Restore();
		res = front4->Lock(NULL, &desc, DDLOCK_WAIT | DDLOCK_WRITEONLY, NULL);
	}

	if(res == DD_OK)
	{
		base = (char *)desc.lpSurface;
		y = sch;
		while(y--)
		{
			memcpy(&base[(y+ty)*desc.lPitch + tx], &backupscr[y*scw], scw);
		}
		front4->Unlock(NULL);
	}
	#endif
}

bool C_gfx_api::EnableGUI(void)
{
	oldw = scw;
	oldh = sch;
	SetResolution(640, 512);

	#ifdef TARGET_ALLEGRO
	show_mouse(screen);
	set_mouse_range(tx, ty, tx+scw, ty+sch);
	#endif

	#ifdef TARGET_SVGALIB
	if(!mouse_init("", vga_getmousetype(), MOUSE_DEFAULTSAMPLERATE))
	{
		mouse_seteventhandler(mouse_handler);
		mouse = this;

		backup = (char *)malloc(mwidth*mheight);
		mbutton = 0;

		EnableMouse();
	}
	#endif

	#ifdef TARGET_WIN32
	ShowCursor(TRUE);
	SetCursor(LoadCursor(NULL, IDC_ARROW));

	gui_active = true;
	#endif

	return false;
}

bool C_gfx_api::DisableGUI(void)
{
	#ifdef TARGET_ALLEGRO
	show_mouse(NULL);
	#endif

	#ifdef TARGET_SVGALIB
	DisableMouse();
	mouse_close();
	#endif

	#ifdef TARGET_WIN32
	ShowCursor(FALSE);
	ClipCursor(NULL);
	gui_active = false;
	#endif

	SetResolution(oldw, oldh);
	return false;
}

bool C_gfx_api::GetButtonOneState(void)
{
	#ifdef TARGET_ALLEGRO
	mx = mouse_x - tx;
	my = mouse_y - ty;
	return (mouse_b&1) ? true : false;
	#endif

	#ifdef TARGET_SVGALIB
	mouse_update();
	return (mbutton&MOUSE_LEFTBUTTON) ? true : false;
	#endif

	#ifdef TARGET_WIN32
	POINT details;
	GetCursorPos(&details);
	mx = details.x - tx;
	my = details.y - ty;
	return lbutton;
	#endif
}

void C_gfx_api::DrawLogo(void)
{
	#ifdef TARGET_ALLEGRO
	blit(logo, backupscr, 0, 0, (scw >> 1)-(logo->w >> 1), sch-logo->h, logo->w, logo->h);
	#endif

	#if TARGET_SVGALIB || TARGET_WIN32
	int y, pos, x;
	x = (scw >> 1) - (lwidth >> 1);
	y = 512-lheight;
	pos = 0;
	while(y < 512)
	{
		memcpy(&backupscr[y*scw + x], (unsigned char *)&blogo[pos], lwidth);
		pos += lwidth;
		y++;
	}
	#endif
}

