/*

	ElectrEm (c) 2000 Thomas Harte - an Acorn Electron Emulator

	This is open software, distributed under the GPL 2, see 'Copying' for details

	file.cpp
	========

	Gets directory contents lists
	uses either opendir/readdir/closedir or FindFirstFile/etc in windows

*/
#include "file.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/stat.h>

char *get_ext(char *name)
{
	char *base = name;

	name += strlen(name)-1;
	while(name > base && *name != '.')
	{
		*name = tolower(*name);
		name--;
	}
	name++;
	return name == base ? NULL : name;
}

bool flower(filetree *st1, filetree *st2)
{
	char *t1, *t2;

	if(st1->dir && !st2->dir)
		return true;

	if(!st1->dir && st2->dir)
		return false;

	t1 = st1->fname;
	t2 = st2->fname;

	while(*t1 && *t2 && tolower(*t1) == tolower(*t2))
	{
		t1++;
		t2++;
	}

	return tolower(*t1) < tolower(*t2);
}

filetree *sort_filetree(filetree *in)
{
	filetree *newtree, *s, *l, *next;

	newtree = in;
	in = in->next;
	newtree->next = NULL;

	while(in)
	{
		next = in->next;

		l = NULL;
		s = newtree;
		while(s && flower(s, in))
		{
			l = s;
			s = s->next;
		}

		if(l)
		{
			in->next = l->next;
			l->next = in;
		}
		else
		{
			in->next = newtree;
			newtree = in;
		}
		in = next;
	}

	return newtree;
}

void free_filetree(struct filetree *first)
{
	struct filetree *next;

	while(first)
	{
		next = first->next;
		free(first->fname);
		free(first);
		first = next;
	}
}

bool valid_path(char *dir)
{
	struct stat buf;

	#if TARG_OS_DOS || TARG_OS_WIN32
	dir++;
	#endif

	if(stat(dir, &buf))
		return false;

	#if TARG_OS_DOS || TARG_OS_LINUX
	return S_ISDIR(buf.st_mode);
	#endif

	#ifdef TARG_OS_WIN32
	return (buf.st_mode&_S_IFDIR) ? true : false;
	#endif
}

#ifdef TARG_FILE_DIRENT
#include <dirent.h>

#ifdef TARG_OS_DOS
#include <unistd.h>
#endif

//to be called with no trailing / on path
//
//for DOS/Windows, consider / to be list of drives, and /<character>:/ to be a
//specific drive

filetree *get_filetree(char *path, char **extensions)
{
	DIR *cdir;
	struct dirent *f;
	struct stat s;
	char *name, *ext, **checkext;
	struct filetree *first, *last, *current;
	unsigned int c;
	bool moveon;

	#ifdef TARG_OS_DOS
	if(*path == '/')	//assumptions : drives a and b are present.
	{
		if(strlen(path) == 1)
		{
		name = (char *)malloc(3);
		last = NULL;
		current = first = (struct filetree *)malloc(sizeof(filetree));

		current->fname = strdup("A:");
		current->dir = true;
		current->next = (struct filetree *)malloc(sizeof(filetree));
		last = current;
		current = current->next;

		current->fname = strdup("B:");
		current->dir = true;
		current->next = (struct filetree *)malloc(sizeof(filetree));
		last = current;
		current = current->next;

		for(c=0;c<24;c++)
		{
			sprintf(name, "%c:/", c+'C');
			if(!access(name, R_OK))
			{
				current->fname = (char *)malloc(3);
				sprintf(current->fname, "%c:", c+'C');
				current->dir = true;
				current->next = (struct filetree *)malloc(sizeof(filetree));
				last = current;
				current = current->next;
			}
		}

		if(last)
		{
			free(last->next);
			last->next = NULL;
		}

		free(name);
	}
	else
	{
		path++;
	#endif
		if(cdir = opendir(path))
		{
			last = NULL;
			current = first = (struct filetree *)malloc(sizeof(filetree));

			while(f = readdir(cdir))
			{
				if(strcmp(f->d_name, "."))
				{
					if(!strcmp(f->d_name, "..") || *f->d_name != '.')
					{
						moveon = true;

						name = (char *)malloc(strlen(f->d_name)+2+strlen(path));
						sprintf(name, "%s%s", path, f->d_name);

						stat(name, &s);

						if(S_ISDIR(s.st_mode))
							current->dir = true;
						else
						{
							current->dir = false;

							ext = get_ext(f->d_name);
							moveon = false;

							checkext = extensions;
							while(**checkext)
							{
								if(!strcmp(ext, *checkext))
									moveon = true;

								checkext++;
							}
						}

						free(name);

						if(moveon)
						{
							current->fname = strdup(f->d_name);
							current->next = (struct filetree *)malloc(sizeof(filetree));
							last = current;
							current = current->next;
						}
					}
				}
			}

			closedir(cdir);

			if(last)
			{
				free(last->next);
				last->next = NULL;
			}
		}
		else
		{
			return NULL;
		}
	#ifdef TARG_OS_DOS
	}
	}
	#endif

	return sort_filetree(first);
}
#endif

#ifdef TARG_FILE_WIN32
#include <direct.h>
#include <malloc.h>
#define WIN32_LEAN_AND_MEAN
#include <windows.h>

filetree *get_filetree(char *path, char **extensions)
{
	int drive;
	struct filetree *current, *first, *last;
	HANDLE fsystem;
	WIN32_FIND_DATA entry;
	char *name, *temp, *ext, **checkext;
	bool moveon;

	if(strlen(path) == 1)
	{
		current = first = (filetree *)malloc(sizeof(filetree));
		last = NULL;

		for( drive = 1; drive <= 26; drive++ )
			if( !_chdrive( drive ) )
			{
				current->fname = (char *)malloc(3);
				sprintf(current->fname, "%c:", drive + 'A' - 1 );
				current->dir = true;
				current->next = (filetree *)malloc(sizeof(filetree));
				last = current;
				current = current->next;
			}

		if(last)
		{
			free(last->next);
			last->next = NULL;
		}
	}
	else
	{
		path++;

		temp = name = (char *)malloc(strlen(path)+4);
		sprintf(name, "%s*.*", path);

		last = NULL;
		current = first = (struct filetree *)malloc(sizeof(filetree));

		current->fname = strdup("..");
		current->dir = true;
		current->next = (struct filetree *)malloc(sizeof(filetree));
		last = current;
		current = current->next;

		fsystem = FindFirstFile(name, &entry);
		do
		{
			if(strcmp(entry.cFileName, ".") && strcmp(entry.cFileName, ".."))
			{
				moveon = true;

				if(entry.dwFileAttributes&FILE_ATTRIBUTE_DIRECTORY)
					current->dir = true;
				else
				{
					current->dir = false;

					ext = get_ext(entry.cFileName);
					moveon = false;

					checkext = extensions;
					while(**checkext)
					{
						if(!strcmp(ext, *checkext))
							moveon = true;

						checkext++;
					}
				}

				if(moveon)
				{
					current->fname = strdup(entry.cFileName);
					current->next = (struct filetree *)malloc(sizeof(filetree));
					last = current;
					current = current->next;
				}
			}
		}while(FindNextFile(fsystem, &entry));

		FindClose(fsystem);

		if(last)
		{
			free(last->next);
			last->next = NULL;
		}
		else
			return NULL;
	}

	return sort_filetree(first);
}

#endif