/*

	ElectrEm (c) 2000 Thomas Harte - an Acorn Electron Emulator

	This is open software, distributed under the GPL 2, see 'Copying' for details

	elecscreen.cpp
	==============

	Most of the general case screen stuff.
	Actual pixel drawing is handled elsewhere by some classes with virtual members.

	Contains :

		Electron palette changing routine
		Screen set up (resolution switch and PC palette set)
		Line counter for knowing what to request from virtual member classes 'C_screen_<size>'

*/
#include "keyboard.h"
#include <string.h>
#include <malloc.h>

#include "tape.h"
extern C_Tape tape;

#include "elecscreen.h"
#include "config.h"
extern C_config settings;

#include "timing.h"

C_timer slowdown;

#include "6502.h"
extern C_6502ULA electron;
extern C_screen elec_screen;

C_screen::C_screen(C_gfx_api &api)
{
	gfxptr = &api;
}

C_screen::~C_screen(void)
{
	if(peldraw)
	{
		peldraw->End();
	}
}

void C_screen::Kill(void)
{
	big_pels.img.Destroy();
	small_pels.img.Destroy();
	wide_pels.img.Destroy();
	gfxptr->shutdown();
}

float col_values[] =
{
	1.0f, 0.7f, 0.6f, 0.0f
};

void set_gradient(C_gfx_api *gfxptr, int startloc, float r1, float g1, float b1, float r2, float g2, float b2)
{
	float radd, gadd, badd;
	int c;

	radd = (r2-r1) / 8;
	gadd = (g2-g1) / 8;
	badd = (b2-b1) / 8;
	for(c = startloc; c < startloc+8; c++)
	{
		gfxptr->SetPaletteEntry(c, r1, g1, b1);
		r1 += radd;
		g1 += gadd;
		b1 += badd;
	}
}

void C_screen::SetSize(bool big, int scalelevel, bool scaletype)
{
	int c;
	C_screen_pels *newdraw;

	gfxptr->SetStretchLevel(scalelevel, scaletype);

	if(big)
	{
		newdraw = &big_pels;
		wide = true;

		if(scalelevel>1)
			newdraw = &wide_pels;
		gfxptr->SetNormalResolution(640, (scalelevel>1) ? 256 : 512);
	}
	else
	{
		newdraw = &small_pels;
		wide = false;
		gfxptr->SetNormalResolution(320, 256);
	}

	for(c = 0x8; c <= 0xf; c++)
		SetPalette(c, 0, false);

	if(newdraw != peldraw)
	{
		if(peldraw)
		{
			peldraw->End();
		}

		peldraw = newdraw;
		peldraw->Setup();
	}
}

bool C_screen::GetSize(void)
{
	return wide;
}

bool C_screen::Setup()
{
	int c;
	float r, g, b;
	bool big, scaletype;
	int scalelevel;

	if(!settings.FindSetting("vidmode"))
		big = false;
	else
		big = settings.ReadSetting().result_int ? true : false;

	if(!settings.FindSetting("cleanscale"))
		scaletype = true;
	else
		scaletype = settings.ReadSetting().result_bool;

	if(!settings.FindSetting("scalelevel"))
		scalelevel = 0;
	else
		scalelevel = settings.ReadSetting().result_int;

	if(gfxptr->Setup(big ? 640 : 320, big ? 512 : 256, "Electron Emulator"))
		return true;

	if(gfxptr->LoadFont("data.uef"))
		return true;

	peldraw = NULL;

	SetSize(big, scalelevel, scaletype);

	save = false;
	saved = true;

	c = 16;
	while(c--)
		gfxptr->SetPaletteEntry(c, 0.0f, 0.0f, 0.0f);
	//negative logic r:g:b
	gfxptr->SetPaletteEntry(8, 1.0f, 1.0f, 1.0f);
	gfxptr->SetPaletteEntry(9, 1.0f, 1.0f, 0.0f);
	gfxptr->SetPaletteEntry(10, 1.0f, 0.0f, 1.0f);
	gfxptr->SetPaletteEntry(11, 1.0f, 0.0f, 0.0f);
	gfxptr->SetPaletteEntry(12, 0.0f, 1.0f, 1.0f);
	gfxptr->SetPaletteEntry(13, 0.0f, 1.0f, 0.0f);
	gfxptr->SetPaletteEntry(14, 0.0f, 0.0f, 1.0f);
	gfxptr->SetPaletteEntry(15, 0.0f, 0.0f, 0.0f);

	c = 64;
	while(c--)
	{
		r = col_values[((c&4) >> 2) | ((c&32) >> 4)];
		g = col_values[((c&2) >> 1) | ((c&16) >> 3)];
		b = col_values[(c&1)        | ((c&8)  >> 2)];
		gfxptr->SetPaletteEntry(64|c, r, g, b);
	}

	//GUI colours, same as Windows 95 for now
	gfxptr->SetPaletteEntry(128, 0.0f,  0.0f,  0.0f);	//black
	gfxptr->SetPaletteEntry(129, 0.0f,  0.0f,  0.5f);	//blue
	gfxptr->SetPaletteEntry(130, 0.5f,  0.5f,  0.5f);	//half grey
	gfxptr->SetPaletteEntry(131, 0.75f, 0.75f, 0.75f);	//3/4 grey
	gfxptr->SetPaletteEntry(132, 0.86f, 0.86f, 0.86f);	//near white
	gfxptr->SetPaletteEntry(133, 1.0f,  1.0f,  1.0f);	//white

	set_gradient(gfxptr, 136, 0.75f, 0.75f, 0.75f, 0.0f,  0.0f,  0.0f);
	set_gradient(gfxptr, 144, 0.86f, 0.86f, 0.86f, 0.0f,  0.0f,  1.0f);
	set_gradient(gfxptr, 152, 0.86f, 0.86f, 0.86f, 0.75f, 0.75f, 0.75f);
	set_gradient(gfxptr, 160, 0.0f,  0.0f,  0.0f,  1.0,   1.0,   1.0f);
	set_gradient(gfxptr, 168, 0.5f,  0.5f,  0.5f,  0.0f,  0.0f,  1.0f);
	set_gradient(gfxptr, 176, 0.0f,  0.0f,  0.0f,  0.0,   0.0,   1.0f);
	set_gradient(gfxptr, 184, 0.0f,  0.0f,  0.0f,  1.0,   0.0,   0.0f);

	gfxptr->FinishPalette();

	memset(cols, 8, 16);

	gfx_tables = (conv_table *)malloc(sizeof(conv_table)*7);
	wide_pels.c_table = big_pels.c_table = small_pels.c_table = &gfx_tables[6];

	gfx_tables[0].b_table_wide = cols_2_w;
	gfx_tables[0].b_table_thin = cols_2_m;
	gfx_tables[0].spaced = false;
	gfx_tables[0].wide = true;
	gfx_tables[0].pitch = 20480;

	gfx_tables[1].b_table_wide = cols_4_w;
	gfx_tables[1].b_table_thin = cols_4_m;
	gfx_tables[1].spaced = false;
	gfx_tables[1].wide = true;
	gfx_tables[1].pitch = 20480;

	gfx_tables[2].b_table_wide = cols_16_w;
	gfx_tables[2].b_table_thin = cols_16_m;
	gfx_tables[2].spaced = false;
	gfx_tables[2].wide = true;
	gfx_tables[2].pitch = 20480;

	gfx_tables[3].b_table_wide = cols_2_w;
	gfx_tables[3].b_table_thin = cols_2_m;
	gfx_tables[3].spaced = true;
	gfx_tables[3].wide = true;
	gfx_tables[3].pitch = 16384;

	gfx_tables[4].b_table_wide = cols_2_t;
	gfx_tables[4].b_table_thin = cols_2_w;
	gfx_tables[4].spaced = false;
	gfx_tables[4].wide = false;
	gfx_tables[4].pitch = 10240;

	gfx_tables[5].b_table_wide = cols_4_t;
	gfx_tables[5].b_table_thin = cols_4_w;
	gfx_tables[5].spaced = false;
	gfx_tables[5].wide = false;
	gfx_tables[5].pitch = 10240;

	gfx_tables[6].b_table_wide = cols_2_t;
	gfx_tables[6].b_table_thin = cols_2_w;
	gfx_tables[6].spaced = true;
	gfx_tables[6].wide = false;
	gfx_tables[6].pitch = 8192;

	slowdown.SetBeatsPerSecond(312*50);

	line = 0;
	return 0;
}

#define RED 4
#define GREEN 2
#define BLUE 1

#define red0(x)		((x & 0x01) << 2)
#define red1(x)		((x & 0x02) << 1)
#define red2(x)		 (x & 0x04)
#define red3(x)		((x & 0x08) >> 1)

#define green2(x)	((x & 0x04) >> 1)
#define green3(x)	((x & 0x08) >> 2)
#define green4(x)	((x & 0x10) >> 3)
#define green5(x)	((x & 0x20) >> 4)

#define blue4(x)	((x & 0x10) >> 4)
#define blue5(x)	((x & 0x20) >> 5)
#define blue6(x)	((x & 0x40) >> 6)
#define blue7(x)	((x & 0x80) >> 7)

void C_screen::SetPalette(unsigned __int8 addr, unsigned __int8 val, bool valid)
{
	int c;
	bool lowchange = false;
	unsigned __int8 *col0, *col1, *col2, *col3;

	if(valid)
	{
		c = 256;
		switch(addr)
		{
			case 0x8 :
			case 0x9 :
				col0 = &cols[0];
				col1 = &cols[2];
				col2 = &cols[8];
				col3 = &cols[10];
			break;

			case 0xa :
			case 0xb :
				col0 = &cols[4];
				col1 = &cols[6];
				col2 = &cols[12];
				col3 = &cols[14];
			break;

			case 0xc :
			case 0xd :
				col0 = &cols[5];
				col1 = &cols[7];
				col2 = &cols[13];
				col3 = &cols[15];
			break;

			case 0xe :
			case 0xf :
				col0 = &cols[1];
				col1 = &cols[3];
				col2 = &cols[9];
				col3 = &cols[11];
			break;
		}

		if(addr&1)
		{
			*col0 = ((*col0)&~(GREEN | RED)) | green4(val) | red0(val);
			*col1 = ((*col1)&~(GREEN | RED)) | green5(val) | red1(val);
			*col2 = ((*col2)&~RED) | red2(val);
			*col3 = ((*col3)&~RED) | red3(val);
		}
		else
		{
			*col0 = ((*col0)&~BLUE) | blue4(val);
			*col1 = ((*col1)&~BLUE) | blue5(val);
			*col2 = ((*col2)&~(BLUE | GREEN)) | blue6(val) | green2(val);
			*col3 = ((*col3)&~(BLUE | GREEN)) | blue7(val) | green3(val);
		}
	}

	if(addr <= 9)
	{
		c = 256;
		while(c--)
		{
			//if in wide mode, change cols_[2/4]_t, cols_[2/4]_w
			//else change cols_[2/4]_w, cols[2/4]_m

			//=>w always changes

			cols_2_w[c].l1.b.b0 = cols[(c&0x80) >> 4]; //pixel 0
			cols_2_w[c].l1.b.b1 = cols[(c&0x40) >> 3]; //pixel 1
			cols_2_w[c].l1.b.b2 = cols[(c&0x20) >> 2]; //pixel 2
			cols_2_w[c].l1.b.b3 = cols[(c&0x10) >> 1]; //pixel 3

			cols_2_w[c].h1.b.b0 = cols[(c&0x08) >> 0];
			cols_2_w[c].h1.b.b1 = cols[(c&0x04) << 1];
			cols_2_w[c].h1.b.b2 = cols[(c&0x02) << 2];
			cols_2_w[c].h1.b.b3 = cols[(c&0x01) << 3];

			cols_4_w[c].l1.b.b0 =
			cols_4_w[c].l1.b.b1 = cols[((c&0x80) >> 4) | ((c&0x08) >> 2)];
			cols_4_w[c].l1.b.b2 =
			cols_4_w[c].l1.b.b3 = cols[((c&0x40) >> 3) | ((c&0x04) >> 1)];
			cols_4_w[c].h1.b.b0 =
			cols_4_w[c].h1.b.b1 = cols[((c&0x20) >> 2) | ((c&0x02) >> 0)];
			cols_4_w[c].h1.b.b2 =
			cols_4_w[c].h1.b.b3 = cols[((c&0x10) >> 1) | ((c&0x01) << 1)];

			if(wide)
			{
				cols_2_t[c].l1.b.b0 =
				cols_2_t[c].l1.b.b1 = cols_2_w[c].l1.b.b0;
				cols_2_t[c].l1.b.b2 =
				cols_2_t[c].l1.b.b3 = cols_2_w[c].l1.b.b1;
				cols_2_t[c].l2.b.b0 =
				cols_2_t[c].l2.b.b1 = cols_2_w[c].l1.b.b2;
				cols_2_t[c].l2.b.b2 =
				cols_2_t[c].l2.b.b3 = cols_2_w[c].l1.b.b3;

				cols_2_t[c].h1.b.b0 =
				cols_2_t[c].h1.b.b1 = cols_2_w[c].h1.b.b0;
				cols_2_t[c].h1.b.b2 =
				cols_2_t[c].h1.b.b3 = cols_2_w[c].h1.b.b1;
				cols_2_t[c].h2.b.b0 =
				cols_2_t[c].h2.b.b1 = cols_2_w[c].h1.b.b2;
				cols_2_t[c].h2.b.b2 =
				cols_2_t[c].h2.b.b3 = cols_2_w[c].h1.b.b3;

				cols_4_t[c].l1.b.b0 =
				cols_4_t[c].l1.b.b1 =
				cols_4_t[c].l1.b.b2 =
				cols_4_t[c].l1.b.b3 = cols_4_w[c].l1.b.b0;

				cols_4_t[c].l2.b.b0 =
				cols_4_t[c].l2.b.b1 =
				cols_4_t[c].l2.b.b2 =
				cols_4_t[c].l2.b.b3 = cols_4_w[c].l1.b.b2;

				cols_4_t[c].h1.b.b0 =
				cols_4_t[c].h1.b.b1 =
				cols_4_t[c].h1.b.b2 =
				cols_4_t[c].h1.b.b3 = cols_4_w[c].h1.b.b0;

				cols_4_t[c].h2.b.b0 =
				cols_4_t[c].h2.b.b1 =
				cols_4_t[c].h2.b.b2 =
				cols_4_t[c].h2.b.b3 = cols_4_w[c].h1.b.b2;
			}
			else
			{
				cols_2_m[c].l1.b.b0 = (cols_2_w[c].l1.b.b0&7) | (cols_2_w[c].l1.b.b1 << 3);
				cols_2_m[c].l1.b.b1 = (cols_2_w[c].l1.b.b2&7) | (cols_2_w[c].l1.b.b3 << 3);
				cols_2_m[c].l1.b.b2 = (cols_2_w[c].h1.b.b0&7) | (cols_2_w[c].h1.b.b1 << 3);
				cols_2_m[c].l1.b.b3 = (cols_2_w[c].h1.b.b2&7) | (cols_2_w[c].h1.b.b3 << 3);

				cols_4_m[c].l1.b.b0 = cols_4_w[c].l1.b.b0;
				cols_4_m[c].l1.b.b1 = cols_4_w[c].l1.b.b2;
				cols_4_m[c].l1.b.b2 = cols_4_w[c].h1.b.b0;
				cols_4_m[c].l1.b.b3 = cols_4_w[c].h1.b.b2;
			}
		}
	}

	if(wide)
	{
		c = 256;
		while(c--)
		{
			cols_16_w[c].l1.b.b0 =
			cols_16_w[c].l1.b.b1 =
			cols_16_w[c].l1.b.b2 =
			cols_16_w[c].l1.b.b3 = cols[((c&0x80) >> 4) | ((c&0x20) >> 3) | ((c&0x08) >> 2) | ((c&0x02) >> 1)];

			cols_16_w[c].h1.b.b0 =
			cols_16_w[c].h1.b.b1 =
			cols_16_w[c].h1.b.b2 =
			cols_16_w[c].h1.b.b3 = cols[((c&0x40) >> 3) | ((c&0x10) >> 2) | ((c&0x04) >> 1) | ((c&0x01) >> 0)];
		}
	}
	else
	{
		c = 256;
		while(c--)
		{
			cols_16_m[c].l1.b.b0 =
			cols_16_m[c].l1.b.b1 = cols[((c&0x80) >> 4) | ((c&0x20) >> 3) | ((c&0x08) >> 2) | ((c&0x02) >> 1)];
			cols_16_m[c].l1.b.b2 =
			cols_16_m[c].l1.b.b3 = cols[((c&0x40) >> 3) | ((c&0x10) >> 2) | ((c&0x04) >> 1) | ((c&0x01) >> 0)];
		}
	}
}

#undef blue7
#undef blue6
#undef blue5
#undef blue4

#undef green5
#undef green4
#undef green3
#undef green2

#undef red3
#undef red2
#undef red1
#undef red0

#undef BLUE
#undef GREEN
#undef RED

void C_screen::ChangeGFXMode(unsigned __int8 newmode)
{
	if(newmode != 7)
		wide_pels.c_table = big_pels.c_table = small_pels.c_table = &gfx_tables[newmode];
}

void C_screen::SaveImage(char *name)
{
	peldraw->img.DumpToFile(name);
}

void C_screen::Keepalive(void)
{
	linebig++;
	if(!(linebig&255))
		tape.Keepalive();

	electron.ram_available = electron.no_line;

	if(!line)
	{
		peldraw->InitDisplay();
		peldraw->DrawPelLine();
	}
	else if(peldraw->locked)
		{
			if(line < 256)
			{
				if((line-blank <= 2) || (line >249 && peldraw->c_table->spaced))
					peldraw->DrawBlankLine();
				else
					peldraw->DrawPelLine();
			}
		}
	line++;

//	if(!(tape.motor_on && tape.pause_display))
		slowdown.RegisterBeat();

	if(line == 256)
	{
		peldraw->EndDisplay();
		electron.SetIRQ(DISPLAY_IRQ);
		update_keyboard();
	}
	else if(line == 100)
		{
			electron.SetIRQ(RTC_IRQ);
		}
		else if(line == 312)
			{
				line = 0;
			}
}
