/*

	ElectrEm (c) 2000 Thomas Harte - an Acorn Electron Emulator

	This is open software, distributed under the GPL 2, see 'Copying' for details

	config.cpp
	==========

	This works with ASCII based configuration / options files. It is relatively new code.

*/
#include "config.h"
#include <stdlib.h>
#include <malloc.h>
#include <string.h>

#if TARG_OS_WIN32
#include <io.h>
#endif

/* GENERAL CONFIG FILE READING STUFF */

char *GetLine(FILE *infile)
{
	char *inpbuf, *inpos, t;

	inpbuf = (char *)calloc(256, 1);
	inpos = inpbuf-1;

	do
	{
		inpos++;
		fread(inpos, 1, 1, infile);
	}
	while((inpos < inpbuf+256) && *inpos != '\n' && !feof(infile));

	if(*inpos != '\n')
	{
		do
		{
			fread(&t, 1, 1, infile);
		}
		while(t != '\n' && !feof(infile));
	}

	return inpbuf;
}

char *PastWhiteSpace(char *in)
{
	while(*in ==' ' || *in =='\t')
	{
		in++;
	}

	return in;
}

i_config_setting *new_i_buffer(void)
{
	i_config_setting *neu;

	neu = (i_config_setting *)malloc(sizeof(i_config_setting));
	neu->next = NULL;
	neu->title = NULL;

	neu->written = false;

	return neu;
}

void free_i_buffer(i_config_setting *kill)
{
	if(kill->title)
		free(kill->title);

	free(kill);
}

config_line *ParseNextLine(FILE *in)
{
	char *b, *b2;
	config_line *newline;

	newline = (config_line *)malloc(sizeof(config_line));
	newline->linebuf = GetLine(in);

	b2 = b = PastWhiteSpace(newline->linebuf);
	switch(*b2)
	{
		case '#' :
		case '\n' :
			newline->type = COMMENT;
		break;

		default :
			newline->type = OPTION;

			while(*b2 != '=' && *b2 != ' ' && *b2 != '\t')
				b2++;

			newline->option.title = (char *)calloc(b2-b+1, 1);
			memcpy(newline->option.title, b, b2-b);

			b = PastWhiteSpace(b2) + 1;
			b = PastWhiteSpace(b);

			newline->option.value.result_int = strtol(b, &b2, 0);
			newline->option.value.result_type = INTEGER_RESULT;

			if(b == b2)
			{
				newline->option.value.result_string = strdup(b);
				newline->option.value.result_string[strlen(newline->option.value.result_string)-1] = '\0';

				newline->option.value.result_type = STRING_RESULT;

				if(!strcmp(newline->option.value.result_string, "true") || !strcmp(newline->option.value.result_string, "TRUE") || !strcmp(newline->option.value.result_string, "yes") || !strcmp(newline->option.value.result_string, "YES"))
				{
					free(newline->option.value.result_string);
					newline->option.value.result_bool = true;
					newline->option.value.result_type = BOOLEAN_RESULT;
				}
				else
					if(!strcmp(newline->option.value.result_string, "false") || !strcmp(newline->option.value.result_string, "FALSE") || !strcmp(newline->option.value.result_string, "no") || !strcmp(newline->option.value.result_string, "NO"))
					{
						free(newline->option.value.result_string);
						newline->option.value.result_bool = false;
						newline->option.value.result_type = BOOLEAN_RESULT;
					}
			}
		break;
	}

	return newline;
}

void FreeLine(config_line *killme)
{
	if(killme)
	{
		free(killme->linebuf);

		if(killme->type == OPTION)
		{
			free(killme->option.title);

			if(killme->option.value.result_type == STRING_RESULT)
			{
				free(killme->option.value.result_string);
			}
		}

		free(killme);
	}
}


/* USUAL CONSTRUCTOR / DESTRUCTOR STUFF */
C_config::C_config(void)
{
	infile = NULL;
	changelist = current = NULL;
	lastline = NULL;
	lastname = NULL;
}

C_config::~C_config(void)
{
	Close();
}

/* FILE READ STUFF */

bool C_config::Open(char *name)
{
	filename = strdup(name);
	changelist = current = new_i_buffer();

	if(!(infile = fopen(name, "rt")))
		return false;

	return true;
}

/* FILE WRITE STUFF */

void WriteSettingLine(FILE *outfile, i_config_setting *current)
{
	fprintf(outfile, "%s = ", current->title);

	switch(current->value.result_type)
	{
		case BOOLEAN_RESULT :
			fputs(current->value.result_bool ? "true" : "false" , outfile);
		break;

		case INTEGER_RESULT :
			fprintf(outfile, "%d", current->value.result_int);
		break;

		case STRING_RESULT :
			fputs(current->value.result_string, outfile);
		break;
	}

	fprintf(outfile, "\n");
}

void C_config::Close(void)
{
	FILE *outfile;

	if(infile)
	{
		if(outfile = fopen("tmp.cfg", "wt"))
		{
			fseek(infile, 0, SEEK_SET);
			while(!feof(infile))
			{
				FreeLine(lastline);
				lastline = ParseNextLine(infile);

				if(lastline->type == COMMENT)
				{
					fputs(lastline->linebuf, outfile);
				}
				else
				{
					current = changelist;
					while(current->next)
					{
						if(!strcmp(current->title, lastline->option.title))
							break;
						current = current->next;
					}

					if(current->next)
					{
						current->written = true;
						WriteSettingLine(outfile, current);
					}
					else
						fputs(lastline->linebuf, outfile);
				}
			}

			current = changelist;
			while(current->next)
			{
				if(!current->written)
					WriteSettingLine(outfile, current);

				current = current->next;
			}

			fclose(outfile);
		}

		fclose(infile);
		rename("tmp.cfg", filename);

		//for some reason, Windows takes rename to mean 'rename file a to file b, but leave a
		//file a with no data in it, hence :
		#if TARG_OS_WIN32
		_unlink("tmp.cfg");
		#endif
	}
	else
		if(changelist)
		{
			if(outfile = fopen(filename, "wt"))
			{
				current = changelist;
				while(current->next)
				{
					WriteSettingLine(outfile, current);
					current = current->next;
				}

				fclose(outfile);
			}
		}
	infile = NULL;

	FreeLine(lastline);
	lastline = NULL;

	while(changelist)
	{
		current = changelist->next;
		free_i_buffer(changelist);
		changelist = current;
	}
	changelist = NULL;

	if(lastname)
		free(lastname);
	lastname = NULL;

	if(filename)
		free(filename);
	filename = NULL;
}

/* SETTING QUERY STUFF */

bool C_config::FindSetting(char *name)
{
	if(lastname)
		free(lastname);
	lastname = strdup(name);

	if(infile)
	{
		fseek(infile, 0, SEEK_SET);

		do
		{
			FreeLine(lastline);
			lastline = ParseNextLine(infile);
		}
		while( !feof(infile) && !(lastline->type == OPTION && !strcmp(lastline->option.title, name)) );

		return !feof(infile);
	}
	else
		return false;
}

config_setting C_config::ReadSetting(void)
{
	return lastline->option.value;
}

void C_config::WriteSetting(struct config_setting setting)
{
	i_config_setting *c;

	c = changelist;
	while(c->next)
	{
		if(!strcmp(c->title, lastname))
			break;

		c = c->next;
	}

	if(!c->next)
	{
		current->title = strdup(lastname);
		c = current;
		current->next = new_i_buffer();
		current = current->next;
	}

	c->value = setting;
}
