/****************************************************************************/
/*              Beebem - (c) David Alan Gilbert 1994                        */
/*              ------------------------------------                        */
/* This program may be distributed freely within the following restrictions:*/
/*                                                                          */
/* 1) You may not charge for this program or for any part of it.            */
/* 2) This copyright message must be distributed with all copies.           */
/* 3) This program must be distributed complete with source code.  Binary   */
/*    only distribution is not permitted.                                   */
/* 4) The author offers no warrenties, or guarentees etc. - you use it at   */
/*    your own risk.  If it messes something up or destroys your computer   */
/*    thats YOUR problem.                                                   */
/* 5) You may use small sections of code from this program in your own      */
/*    applications - but you must acknowledge its use.  If you plan to use  */
/*    large sections then please ask the author.                            */
/*                                                                          */
/* If you do not agree with any of the above then please do not use this    */
/* program.                                                                 */
/* Please report any problems to the author at beebem@treblig.org           */
/****************************************************************************/
/* The window which the beeb emulator displays stuff in */
/* David Alan Gilbert 6/11/94 */

#include <string.h>
#include <assert.h>


//
// BeOS stuff
//
#include <View.h>
#include "my_app.h"
#include "my_view.h"

#include <Application.h>
#include <Window.h>
#include <WindowScreen.h>
#include <FilePanel.h>
#include <OS.h>
#include <ScrollBar.h>
#include <ScrollView.h>
#include <MenuItem.h>
#include <MenuBar.h>
#include <StringView.h>
#include <Alert.h>
#include <Path.h>


#include "6502core.h"
#include "beebwin.h"
#include "disc8271.h"
#include "sysvia.h"

typedef enum {
  Same,
  ForceOn,
  ForceOff
} shiftHackType;

struct BeebKeyTrans {
  int /*KeySym*/ sym;
  int row;
  int col;
  shiftHackType shiftHack;
};


#define MSG_ABOUT	'FM-1'
#define MSG_DISK	'FM-2'
#define MSG_QUIT	'FM-3'


enum {
	BEEB_WIN_WIDTH	= 640,
	BEEB_WIN_HEIGHT	= 256,
	MARGIN			= 20
};


// From main.cc
int real_main(int argc,char *argv[]);

extern int new_argc;
extern char **new_argv;



BeebWin::BeebWin(status_t *ret, bool size) : 
	BWindowScreen("My window", B_8_BIT_640x480, ret), fullsize(size)
{
	// Create a view to attach stuff to...
	requested_size = BRect(0, 0, BEEB_WIN_WIDTH-1, BEEB_WIN_HEIGHT-1 + MARGIN);
	requested_depth = 8;
	view = new my_view(requested_size);	
	Lock();
	AddChild(view);
	view->MakeFocus();
	Unlock();

	// BWindowScreen stuff
	thread_is_locked = true;
	tid = 0;
	if (*ret == B_OK) {
		sem		= create_sem(0, "WindowScreen Access");
		area	= create_area("save", (void **) &save_buffer, B_ANY_ADDRESS,
						640*512*2, B_NO_LOCK, B_READ_AREA | B_WRITE_AREA);
		if (sem < B_NO_ERROR) {
			puts("semophore allocation failed");
		}
		else if (area < B_NO_ERROR) {
			puts("area allocation failed");
		}
		else {
			Show();
		}
	}


	// Create rendering buffer
	BBitmap *bitmap = new BBitmap(requested_size, B_COLOR_8_BIT);
	image_data		= (uchar *) bitmap->Bits();
	bytes_per_row	= bitmap->BytesPerRow();

	memset(image_data, 0, bytes_per_row * (BEEB_WIN_HEIGHT -1 + MARGIN));

}


BeebWin::~BeebWin(void)
{

}



void BeebWin::ScreenConnected(bool connected)
{
	printf("ScreenConnected(%s)\n", connected ? "true" : "false");

	if (connected) {
		puts("Creating display");
		if ((SetSpace(B_8_BIT_640x480) < B_OK) ||
				(SetFrameBuffer(640, 512*2) < B_OK)) {
			be_app->PostMessage(B_QUIT_REQUESTED);
			return;
		}

		// get the hardware acceleration hooks. get them each time 
		// the WindowScreen is connected, because of multiple 
		// monitor support 
		blit = (blit_hook) CardHookAt (7);
		sync = (sync_hook) CardHookAt (10);

		// cannot work with no hardware blitting 
		if (blit == NULL) {
			be_app->PostMessage (B_QUIT_REQUESTED);
			return;
		}

		frame_buffer = (uint8 *) (CardInfo ()->frame_buffer);
		line_length = FrameBufferInfo ()->bytes_per_row;
		if (tid == 0) {
			puts("Clear framebuffer");
			tid = 1;
			// Clear the framebuffer 
			memset (frame_buffer, 0, 512*2 * line_length);

			// Sart emulation
			the_thread = spawn_thread(thread_func, "BeebEm Thread",
							B_NORMAL_PRIORITY, this);
			resume_thread(the_thread);

		} else {
			puts("Restoring display");
			for (int y = 0; y < 512; y++) {
				// restore the framebuffer when switching back from 
				// another workspace. 
				memcpy(frame_buffer + y*line_length, save_buffer + 640 * y, 640);
			}
		}

		// set our color list. 
		for (int col=0; col<8; col++) {
			int red		= (col&1) ? 255 : 0;
			int green	= (col&2) ? 255 : 0;
			int blue	= (col&4) ? 255 : 0;
			rgb_color c	= { red, green, blue };
			SetColorList(&c, col, col);
			cols[col] = col;
		}

		// allow the rendering thread to run. 
		thread_is_locked = false;
		release_sem (sem);


	} else {
		puts("Disconnecting display");
		// block the rendering thread. 
		if (!thread_is_locked) {
			puts("About to aquire sem");
			acquire_sem (sem);
			puts("Aquired sem");
			thread_is_locked = true;
		}

		// kill the rendering and clean up when quitting 
		if ((((my_app *) be_app)->is_quitting)) {
			puts("Thread is quitting");
			status_t ret;
			kill_thread(the_thread);
			wait_for_thread (the_thread, &ret);
			delete_sem (sem);
			puts("Freed sem");
			delete_area (area);
			puts("Freed area");
		} else {
			// set the color list black so that the screen doesn't 
			// seem to freeze while saving the framebuffer 
//			rgb_color c = {0, 0, 0};
//			for (int i = 0; i < 256; i++) {
//				printf("%d\n", i);
//				SetColorList (&c, i, i);
//			}
			// save the framebuffer 
			for (int y = 0; y < 480; y++) {
				memcpy (save_buffer + 640 * y, frame_buffer + y * line_length, 640);
			}
		}
		puts("Leaving ScreenConnected()");
	}

}



EightUChars *BeebWin::GetLinePtr(int y)
{
	if (y>255)
		y=255;

	return ((EightUChars *) (image_data + y*bytes_per_row));
};


SixteenUChars *BeebWin::GetLinePtr16(int y)
{
	if (y>255)
		y=255;

	return ((SixteenUChars *) (image_data + y*bytes_per_row));
};


int BeebWin::bytesPerLine(void)
{
	return bytes_per_row;
};


char *BeebWin::imageData(void)
{
	return image_data;
};


//
// Copy from buffer to screen
//
// Not entirely sure the double-buffering works, I'll
//  investigate later (honest Guv).
//
void BeebWin::updateLines(int starty, int nlines)
{
	static int frame_num = 1;

	// Get start of frame...
	int y_origin = frame_num*512;

//if (((my_app *) be_app)->is_quitting) return;

	int y_offset = view->get_y_offset();

	if (starty<256) {
		if ((starty+nlines-1)>255)
			nlines=255-starty;

		acquire_sem(sem);
		if (Lock()) {
			if (fullsize) {

#if 0
				pixel_doubling_blit_8(
							image_data,			// source
							bytes_per_row,		// source width in bytes
							bytes_per_row - 16,	// Source width
							nlines,				// Source height
							frame_buffer+y_origin*BEEB_WIN_WIDTH,
							line_length);
#endif

#if 0
				uint8 *to	= frame_buffer + y_origin*BEEB_WIN_WIDTH;
				uint8 *from	= image_data;

				for (int y=0; y<BEEB_WIN_HEIGHT; y++) {
					long double *p = (long double *) from;
					long double *q = (long double *) to;
					long double *r = (long double *) (to + line_length);

					for (int x=0; x<BEEB_WIN_WIDTH/sizeof(long double); x++) {
						*q++ = *p;
						*r++ = *p++;
					}
					to += 2*line_length;
					from += bytes_per_row;
				}
#endif


#if 1
				uint8 *to	= frame_buffer + y_origin*BEEB_WIN_WIDTH;
				uint8 *from	= image_data + bytes_per_row*y_offset;

				for (int y=0; y<BEEB_WIN_HEIGHT; y++) {
					memcpy(to, from, BEEB_WIN_WIDTH);
					memcpy(to+line_length, from, BEEB_WIN_WIDTH);
					to += 2*line_length;
					from += bytes_per_row;
				}
#endif	// 0
			} else {
				uint8 *to	= frame_buffer + y_origin*BEEB_WIN_WIDTH;
				uint8 *from	= image_data;

				for (int y=0; y<BEEB_WIN_HEIGHT; y++) {
					memcpy(to, from, BEEB_WIN_WIDTH);
					to += line_length;
					from += bytes_per_row;
				}
			}

			sync();
			MoveDisplayArea(0, y_origin);
	
			Unlock();
		}
	
		release_sem(sem);
	}

	frame_num = 1 - frame_num;
};




//
// These two used for Mode 7 display
//
void BeebWin::doHorizLine(unsigned long Col, int y, int sx, int width)
{
	if (y>255)
		return;

	memset(image_data + y*bytes_per_row+sx, Col, width);
};

void BeebWin::doHorizLine(unsigned long Col, int offset, int width)
{
	if ((offset+width)<DataSize)
		return;

	memset(image_data + offset, Col, width);
};



//
//  Jon Watte's super-fast scaling blitter
//
void BeebWin::pixel_doubling_blit_8(
	const void * source_bitmap,
	int          source_rowbytes,
	int          source_width,
	int          source_height,
	void *       dest_bitmap,
	int          dest_rowbytes)
{
	/*	ensure alignment that we depend on	*/
	assert(!(((long)source_bitmap)&3) && !(((long)dest_bitmap)&7));
	double temp[1];
	register double temp2;
	while (--source_height >= 0)
	{
		unsigned long * src = ((unsigned long *)source_bitmap)-1;
		double * dst1 = ((double *)dest_bitmap)-1;
		double * dst2 = dst1+dest_rowbytes/8;
		int w = source_width;
		while ((w-=2 /*4*/) >= 0)
		{

		/* This loop might be optimizable by assembling the */
		/* "temp" double in another way; check profiling and */
		/* disassembly to make sure. I think this is close to */
		/* optimal, though. */

		/* If we're in little-endian mode, the bytes go the other way */
#if __INTEL__
			unsigned long pixx = *(++src);
			unsigned char * mid = (unsigned char *)&temp[0];
			*(mid++) = pixx;
//			*(mid++) = pixx;
			pixx >>= 8;
			*(mid++) = pixx;
//			*(mid++) = pixx;
			pixx >>= 8;
			*(mid++) = pixx;
//			*(mid++) = pixx;
			pixx >>= 8;
			*(mid++) = pixx;
//			*(mid++) = pixx;

			pixx = *(++src);
			*(mid++) = pixx;
			pixx >>= 8;
			*(mid++) = pixx;
			pixx >>= 8;
			*(mid++) = pixx;
			pixx >>= 8;
			*(mid++) = pixx;


#else
			unsigned long pixx = *(++src);
			unsigned char * mid = (unsigned char *)&temp[1];
			*(--mid) = pixx;
			*(--mid) = pixx;
			pixx >>= 8;
			*(--mid) = pixx;
			*(--mid) = pixx;
			pixx >>= 8;
			*(--mid) = pixx;
			*(--mid) = pixx;
			pixx >>= 8;
			*(--mid) = pixx;
			*(--mid) = pixx;
#endif

		/* This is the kick-ass magic part! Don't touch! */

			temp2 = temp[0];
			*(++dst1) = temp2;
			*(++dst2) = temp2;
		} 
		source_bitmap = ((char *)source_bitmap)+source_rowbytes;
		dest_bitmap = ((char *)dest_bitmap)+2*dest_rowbytes;
	}
}



long BeebWin::thread_func(void *obj)
{
	puts("Launching emulator");
	real_main(new_argc, new_argv);
	puts("Finished emulator");
	return 0;
}


