/*
 * The functions to execute for each opcode.
 *
 * If _65C12_ is defined then the extra Master 65C12 processor instructions
 * are added.
 *
 * If _R65C02_ is defined then the extra Rockwell R65C02 processor (as used
 * in the Master Turbo and 6502 2nd processors) instructions are added.
 *
 * Unimplemented instructions are considered as nops, but of the correct
 * length (ie not all invalid instructions are single byte).
 *
 * 4/12/93 jkb	creation
 */

/*
 * FIXME:
 *
 * --- 1 ---
 * According to the Rockwell CMOS 65C00 family data sheets, the R65C02 includes
 * the extra R65C12 instructions.
 * In addition, RMB and SMB instructions are added:
 *
 * 07, 17, .. 77   RMB     Reset memory bit (Bit Manipulation, ZP)
 * 87, 97, .. F7   SMB     Reset memory bit (Bit Manipulation, ZP)
 *
 * They take 5 cycles and 2 bytes. The opcode encodes the bit to set/reset.
 * So set/reset of bit 3 would be opcodes b7/37.
 *
 * --- 2 ---
 * What are the lengths in time and memory of the NOP instructions? Current
 * values are estimates from length groupings found when examining the opcode
 * table. These are particularly estimated in the case of 0x?3 and 0x?B.
 * 
 * See the HACKROM *MEM code to see the real values I derived on the BBC.
 *
 * --- 3 ---
 * Pretty unimportant: the NMOS 6502 chip performs one read and two writes
 * in each of the read/modify/write instructions.
 * Eg INC &70 will read &70 (say 10), write 10 to &70, write 11 to &70.
 * The Rockwell CMOS series instead perform two reads and one write.
 *
 * Also when accessing via an index across a page boundary the address without
 * the top nibble incrememnted in accessed before accessing (and returning)
 * the actual value.
 *
 * --- 4 ---
 * Examine the roll of 'time stretching' when accessing JIM or FRED.
 */

/*
 * Some common things between opcodes - pushing an pulling the stack
 */
#define pushb(val) writebq(SP + 0x100, val); SP--
#define pullb()    (SP++, readbq(SP + 0x100))

#define pushw(val) pushb((val) / 0x100); pushb((val) & 0xff)
/* #define pullw() (pullb() + 0x100 * pullb()) */

/*
 * Defs for each of the valid opcode types
 */
#define LDA(mode, len, time, func) \
{ \
    A=read##func(mode()); \
    reset_ZN(A); \
    inc_PC(len); \
    inc_time(time); \
}

#define STA(mode, len, time, func) \
{ \
    write##func(mode(), A); \
    inc_PC(len); \
    inc_time(time); \
}

#define LDX(mode, len, time, func) \
{ \
    X=read##func(mode()); \
    reset_ZN(X); \
    inc_PC(len); \
    inc_time(time); \
}

#define STX(mode, len, time, func) \
{ \
    write##func(mode(), X); \
    inc_PC(len); \
    inc_time(time); \
}

#define LDY(mode, len, time, func) \
{ \
    Y=read##func(mode()); \
    reset_ZN(Y); \
    inc_PC(len); \
    inc_time(time); \
}

#define STY(mode, len, time, func) \
{ \
    write##func(mode(), Y); \
    inc_PC(len); \
    inc_time(time); \
}

#ifdef _65C12_
#define STZ(mode, len, time) \
{ \
    writebq(mode(), 0); \
    inc_PC(len); \
    inc_time(time); \
}
#endif

#define AND(mode, len, time, func) \
{ \
    A &= read##func(mode()); \
    reset_ZN(A); \
    inc_PC(len); \
    inc_time(time); \
}

#define EOR(mode, len, time, func) \
{ \
    A ^= read##func(mode()); \
    reset_ZN(A); \
    inc_PC(len); \
    inc_time(time); \
}

#define ORA(mode, len, time, func) \
{ \
    A |= read##func(mode()); \
    reset_ZN(A); \
    inc_PC(len); \
    inc_time(time); \
}

#define DEC(mode, len, time, func) \
{ \
    word addr = mode(); \
    byte val = read##func(addr) - 1; \
\
    write##func(addr, val); \
    reset_ZN(val); \
    inc_PC(len); \
    inc_time(time); \
}

#define INC(mode, len, time, func) \
{ \
    word addr = mode(); \
    byte tmp = read##func(addr) + 1; \
\
    write##func(addr, tmp); \
    reset_ZN(tmp); \
    inc_PC(len); \
    inc_time(time); \
}

#define NOP(len) \
{ \
    inc_PC(len); \
    inc_time(2); /* FIXME: true for 2/3 byte NOPs too? */ \
}

#define CLC(len, time) \
{ \
    clr_C(); \
    inc_PC(len); \
    inc_time(time); \
}

#define CLD(len, time) \
{ \
    clr_D(); \
    inc_PC(len); \
    inc_time(time); \
}

#define CLI(len, time) \
{ \
    clr_I(); \
    inc_PC(len); \
    inc_time(time); \
}

#define CLV(len, time) \
{ \
    clr_V(); \
    inc_PC(len); \
    inc_time(time); \
}

#undef SEC
#define SEC(len, time) \
{ \
    set_C(); \
    inc_PC(len); \
    inc_time(time); \
}

#define SED(len, time) \
{ \
    set_D(); \
    inc_PC(len); \
    inc_time(time); \
}

#define SEI(len, time) \
{ \
    set_I(); \
    inc_PC(len); \
    inc_time(time); \
}

#define DEA(len, time) \
{ \
    A--; \
    reset_ZN(A); \
    inc_PC(len); \
    inc_time(time); \
}

#define INA(len, time) \
{ \
    A++; \
    reset_ZN(A); \
    inc_PC(len); \
    inc_time(time); \
}

#define DEX(len, time) \
{ \
    X--; \
    reset_ZN(X); \
    inc_PC(len); \
    inc_time(time); \
}

#define INX(len, time) \
{ \
    X++; \
    reset_ZN(X); \
    inc_PC(len); \
    inc_time(time); \
}

#define DEY(len, time) \
{ \
    Y--; \
    reset_ZN(Y); \
    inc_PC(len); \
    inc_time(time); \
}

#define INY(len, time) \
{ \
    Y++; \
    reset_ZN(Y); \
    inc_PC(len); \
    inc_time(time); \
}

/* FIXME: how does SED changes CMP? At all? It _does_ do a subtraction */
#define CMP(mode, len, time, func) \
{ \
    unsigned int tmp; \
\
    tmp = read##func(mode()); \
    reset_C(A >= tmp); \
/*    reset_ZN(A-tmp); */\
    reset_Z(A == tmp); \
    reset_N(((byte)(A - tmp)) & 0x80); \
    inc_PC(len); \
    inc_time(time); \
}

#define CPX(mode, len, time, func) \
{ \
    unsigned int tmp; \
\
    tmp = read##func(mode()); \
    reset_C(X >= tmp); \
    reset_Z(X == tmp); \
    reset_N(((byte)(X - tmp)) & 0x80); \
    inc_PC(len); \
    inc_time(time); \
}

#define CPY(mode, len, time, func) \
{ \
    unsigned int tmp; \
\
    tmp = read##func(mode()); \
    reset_C(Y >= tmp); \
    reset_Z(Y == tmp); \
    reset_N(((byte)(Y - tmp)) & 0x80); \
    inc_PC(len); \
    inc_time(time); \
}

#define TAX(len, time) \
{ \
    X = A; \
    reset_ZN(X); \
    inc_PC(len); \
    inc_time(time); \
}

#define TAY(len, time) \
{ \
    Y = A; \
    reset_ZN(Y); \
    inc_PC(len); \
    inc_time(time); \
}

#define TSX(len, time) \
{ \
    X = SP; \
    reset_ZN(X); \
    inc_PC(len); \
    inc_time(time); \
}

#define TXA(len, time) \
{ \
    A = X; \
    reset_ZN(A); \
    inc_PC(len); \
    inc_time(time); \
}

#define TXS(len, time) \
{ \
    SP = X; \
    inc_PC(len); \
    inc_time(time); \
}

#define TYA(len, time) \
{ \
    A = Y; \
    reset_ZN(A); \
    inc_PC(len); \
    inc_time(time); \
}

/*
 * By far the most yucky.
 * There's so very odd things here - especially with BCD.
 * Eg: BC + BC == BE (sets C, V, N). Apparently the simplest approach means
 * this is correctly, luckily :-)
 *
 * [ BCD in 6502 apparently leaves N, V and Z in an indeterminant state. The
 *   65C12 (and 65C02?) chips leaves these correct at the expense of one
 *   extra cycle when in BCD. ]
 */
#ifdef _65C12_ /* also R65C02? */
#    define ADC_KLUDGE get_D()
#else
#    define ADC_KLUDGE 0
#endif
#define ADC(mode, len, time, func) \
{ \
    byte M = read##func(mode()); \
    byte A_o = A; \
    signed int A2; \
\
    if (get_D()) { \
	int Alo = A & 0x0f; \
	int Ahi = A / 0x10; \
\
	Alo += (M & 0x0f) + get_C(); \
	if (Alo >= 10) { \
	    Alo -= 10; \
	    Ahi++; \
	} \
	Ahi += M / 0x10; \
	if (Ahi >= 10) { \
	    Ahi -= 10; \
	    set_C(); \
	} else { \
	    clr_C(); \
	} \
	A = Alo + 0x10 * Ahi; \
    } else { \
	A2 = A + M + get_C(); \
        A = A2 & 0xff; \
	reset_C(A2 >= 0x100); \
    } \
    /* Horrid logic: V is set if the sign of the answer is wrong. \
     * Eg: A is negative, M is negative, but A2 is positive. \
     */ \
    reset_ZN(A); \
    reset_V(((A_o & 0x80) == (M & 0x80)) && ((A_o & 0x80) != (A & 0x80)));\
/*    reset_V((A_o & 0x80) != (A & 0x80) && !get_Z()); */ \
\
    inc_PC(len); \
    inc_time(len); \
}

#define SBC(mode, len, time, func) \
{ \
    byte M = read##func(mode()); \
    byte A_o = A; \
    signed int A2; \
\
    if (get_D()) { \
	int Alo = A & 0x0f; \
	int Ahi = A / 0x10; \
\
	Alo -= (M & 0x0f) - !get_C(); \
	if (Alo <= 0) { \
	    Alo += 10; \
	    Ahi--; \
	} \
	Ahi -= M / 0x10; \
	if (Ahi <= 0) { \
	    Ahi += 10; \
	    clr_C(); \
	} else { \
	    set_C(); \
	} \
	A = Alo + 0x10 * Ahi; \
    } else { \
	A2 = A - M - (!get_C()); \
        A = A2 & 0xff; \
	reset_C(!(A2 < 0x00)); \
    } \
    /* Horrid logic: V is set if the sign of the answer is wrong. \
     * Eg: A is negative, M is negative, but A2 is positive. \
     */ \
    reset_ZN(A); \
    reset_V((A_o & 0x80) != (A & 0x80) && !get_Z()); \
\
    inc_PC(len); \
    inc_time(len); \
}

#define JMP(mode, len, time) \
{ \
    PC = mode(); \
    inc_time(time); \
}

#define PHA(len, time) \
{ \
    pushb(A); \
    inc_PC(len); \
    inc_time(time); \
}

#define PHP(len, time) \
{ \
    pushb(P); \
    inc_PC(len); \
    inc_time(time); \
}

#define PLA(len, time) \
{ \
    A=pullb(); \
    reset_ZN(A); \
    inc_PC(len); \
    inc_time(time); \
}

#define PLP(len, time) \
{ \
    P=pullb() | P_U | P_B; \
    inc_PC(len); \
    inc_time(time); \
}

#ifdef _65C12_
#define PHX(len, time) \
{ \
    pushb(X); \
    inc_PC(len); \
    inc_time(time); \
}

#define PHY(len, time) \
{ \
    pushb(Y); \
    inc_PC(len); \
    inc_time(time); \
}

#define PLX(len, time) \
{ \
    X=pullb(); \
    reset_ZN(X); \
    inc_PC(len); \
    inc_time(time); \
}

#define PLY(len, time) \
{ \
    Y=pullb(); \
    reset_ZN(Y); \
    inc_PC(len); \
    inc_time(time); \
}
#endif /* _65C12_ */

#define JSR(mode, len, time) \
{ \
    pushw(PC + len-1); \
    PC = mode(); \
    inc_time(time); \
}

#define RTS(len, time) \
{ \
    PC = readwq(SP + 0x101) + 1; SP += 2; \
    inc_time(time); \
}

#define RTI(len, time) \
{ \
    P = pullb(); \
    PC = readwq(SP + 0x101); SP += 2; \
    inc_time(time); \
}

#define ASL(mode, len, time, func) \
{ \
    word addr = mode(); \
    byte M = read##func(addr); \
    reset_C(M & 0x80); \
    write##func(addr, M <<= 1); \
    reset_ZN(M); \
    inc_PC(len); \
    inc_time(time); \
}

#define ASLA(len, time) \
{ \
    reset_C(A & 0x80); \
    A <<= 1; \
    reset_ZN(A); \
    inc_PC(len); \
    inc_time(time); \
}

#define LSR(mode, len, time, func) \
{ \
    word addr = mode(); \
    byte M = read##func(addr); \
    reset_C(M & 0x01); \
    write##func(addr, M >>= 1); \
    reset_Z(M == 0); \
    clr_N(); \
    inc_PC(len); \
    inc_time(time); \
}

#define LSRA(len, time) \
{ \
    reset_C(A & 0x01); \
    A >>= 1; \
    reset_Z(A == 0); \
    clr_N(); \
    inc_PC(len); \
    inc_time(time); \
}

#define ROL(mode, len, time, func) \
{ \
    word addr = mode(); \
    byte M = read##func(addr); \
    int tmp = get_C(); \
    reset_C(M & 0x80); \
    write##func(addr, M = (M << 1) + tmp); \
    reset_ZN(M); \
    inc_PC(len); \
    inc_time(time); \
}

#define ROLA(len, time) \
{ \
    int tmp = get_C(); \
    reset_C(A & 0x80); \
    A = (A << 1) + tmp; \
    reset_ZN(A); \
    inc_PC(len); \
    inc_time(time); \
}

#define ROR(mode, len, time, func) \
{ \
    word addr = mode(); \
    byte M = read##func(addr); \
    int tmp = get_C(); \
    reset_C(M & 0x01); \
    write##func(addr, M = (M >> 1) + 0x80 * tmp); \
    reset_ZN(M); \
    inc_PC(len); \
    inc_time(time); \
}

#define RORA(len, time) \
{ \
    int tmp = get_C(); \
    reset_C(A & 0x01); \
    A = (A >> 1) + 0x80 * tmp; \
    reset_ZN(A); \
    inc_PC(len); \
    inc_time(time); \
}

#define BCC(len, time) \
{ \
    if (!get_C()) { \
	byte b = readb(imm()); \
	word oPC = PC; \
	inc_PC(len + (b < 0x80 ? b : b - 0x100)); \
	inc_time(time + 1 + ((PC & 0x100) != (oPC & 0x100))); \
    } else { \
	inc_PC(len); \
	inc_time(time); \
    } \
}

#define BCS(len, time) \
{ \
    if (get_C()) { \
	byte b = readb(imm()); \
	word oPC = PC; \
	inc_PC(len + (b < 0x80 ? b : b - 0x100)); \
	inc_time(time + 1 + ((PC & 0x100) != (oPC & 0x100))); \
    } else { \
	inc_PC(len); \
	inc_time(time); \
    } \
}

#define BEQ(len, time) \
{ \
    if (get_Z()) { \
	byte b = readb(imm()); \
	word oPC = PC; \
	inc_PC(len + (b < 0x80 ? b : b - 0x100)); \
	inc_time(time + 1 + ((PC & 0x100) != (oPC & 0x100))); \
    } else { \
	inc_PC(len); \
	inc_time(time); \
    } \
}

#define BNE(len, time) \
{ \
    if (!get_Z()) { \
	byte b = readb(imm()); \
	word oPC = PC; \
	inc_PC(len + (b < 0x80 ? b : b - 0x100)); \
	inc_time(time + 1 + ((PC & 0x100) != (oPC & 0x100))); \
    } else { \
	inc_PC(len); \
	inc_time(time); \
    } \
}

#define BMI(len, time) \
{ \
    if (get_N()) { \
	byte b = readb(imm()); \
	word oPC = PC; \
	inc_PC(len + (b < 0x80 ? b : b - 0x100)); \
	inc_time(time + 1 + ((PC & 0x100) != (oPC & 0x100))); \
    } else { \
	inc_PC(len); \
	inc_time(time); \
    } \
}

#define BPL(len, time) \
{ \
    if (!get_N()) { \
	byte b = readb(imm()); \
	word oPC = PC; \
	inc_PC(len + (b < 0x80 ? b : b - 0x100)); \
	inc_time(time + 1 + ((PC & 0x100) != (oPC & 0x100))); \
    } else { \
	inc_PC(len); \
	inc_time(time); \
    } \
}

#define BVC(len, time) \
{ \
    if (get_V() == 0) { \
	byte b = readb(imm()); \
	word oPC = PC; \
	inc_PC(len + (b < 0x80 ? b : b - 0x100)); \
	inc_time(time + 1 + ((PC & 0x100) != (oPC & 0x100))); \
    } else { \
	inc_PC(len); \
	inc_time(time); \
    } \
}

#define BVS(len, time) \
{ \
    if (get_V()) { \
	byte b = readb(imm()); \
	word oPC = PC; \
	inc_PC(len + (b < 0x80 ? b : b - 0x100)); \
	inc_time(time + 1 + ((PC & 0x100) != (oPC & 0x100))); \
    } else { \
	inc_PC(len); \
	inc_time(time); \
    } \
}

#ifdef _65C12_
#define BRA(len, time) \
{ \
    byte b = readb(imm()); \
    word oPC = PC; \
    inc_PC(len + (b < 0x80 ? b : b - 0x100)); \
    inc_time(time + 1 + ((PC & 0x100) != (oPC & 0x100))); \
}
#endif

#ifdef _R65C02_
#define BBR(bit, len, time) \
{ \
    byte b = readbq(zp); \
    if (!(b & bit)) { \
	byte b - readb(PC + 2); \
	word oPC = PC; \
	inc_PC(len + (b < 0x80 ? b : b - 0x100)); \
	inc_time(time + 1 + ((PC & 0x100) != (oPC & 0x100))); \
    } else { \
	inc_PC(len); \
	inc_PC(time); \
    } \
}

#define BBS(bit, len, time) \
{ \
    byte b = readbq(zp); \
    if (b & bit) { \
	byte b - readb(PC + 2); \
	word oPC = PC; \
	inc_PC(len + (b < 0x80 ? b : b - 0x100)); \
	inc_time(time + 1 + ((PC & 0x100) != (oPC & 0x100))); \
    } else { \
	inc_PC(len); \
	inc_PC(time); \
    } \
}
#endif

#define BIT(mode, len, time, func) \
{ \
    byte M = read##func(mode()); \
    reset_Z((A & M) == 0); \
    reset_N(M & 0x80); \
    reset_V(M & 0x40); \
    inc_PC(len); \
    inc_time(time); \
}

/*
 * special hack for BIT in immediate mode. In this addressing mode the V and
 * N flags are not changed.
 */
#define BITx(mode, len, time) \
{ \
    byte M = readb(mode()); \
    reset_Z((A & M) == 0); \
    inc_PC(len); \
    inc_time(time); \
}

#ifdef _65C12_

#define TRB(mode, len, time, func) \
{ \
    word addr = mode(); \
    byte M = ~A & read##func(addr); \
    write##func(addr, M); \
    reset_Z(M == 0); \
    inc_PC(len); \
    inc_PC(time); \
}

#define TSB(mode, len, time, func) \
{ \
    word addr = mode(); \
    byte M = A | read##func(addr); \
    write##func(addr, M); \
    reset_Z(M == 0); \
    inc_PC(len); \
    inc_PC(time); \
}
#endif

#define BRK(len, time) \
{ \
    pushw(PC+2); \
    set_B(); \
    pushb(P); \
    PC = readwq(0xFFFE); \
    inc_time(time); \
}

/*---------------------------------------------------------------------------*/

/*
 * Now the functions themselves.
 * We have one for each opcode value, even the unimplemented ones
 */

switch (readbq(PC)) {
 case 0x00: /* BRK */
    BRK(1, 7);
    break;

 case 0x01: /* ORA (aa,X) */
    ORA(ind_x, 2, 6, b);
    break;

 case 0x02: /* Unimplemented */
    NOP(2);
    break;

 case 0x03: /* Unimplemented */
    NOP(2);
    break;

 case 0x04: /* TSB aa */ /* 65C12 */
#ifdef _65C12_
    TSB(zp, 2, 5, bq);
#else
    NOP(2);
#endif
    break;

 case 0x05: /* ORA aa */
    ORA(zp, 2, 3, bq);
    break;

 case 0x06: /* ASL aa */
    ASL(zp, 2, 5, bq);
    break;

 case 0x07: /* Unimplemented */
    NOP(2);
    break;

 case 0x08: /* PHP */
    PHP(1, 3);
    break;

 case 0x09: /* ORA #dd */
    ORA(imm, 2, 2, bi);
    break;

 case 0x0a: /* ASL A */
    ASLA(1, 2);
    break;

 case 0x0b: /* Unimplemented */
    NOP(1);
    break;

 case 0x0c: /* TSB aaaa */ /* 65C12 */
#ifdef _65C12_
    TSB(abs, 3, 6, b);
#else
    NOP(3);
#endif
    break;

 case 0x0d: /* ORA aaaa */
    ORA(abs, 3, 4, b);
    break;

 case 0x0e: /* ASL aaaa */
    ASL(abs, 3, 6, b);
    break;

 case 0x0f: /* BBR 0 aa rr */  /* 65C02 */
#ifdef _R65C02_
    BBR(0x01, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0x10: /* BPL rr */
    BPL(2, 2);
    break;

 case 0x11: /* ORA (aa),Y */
    ORA(ind_y, 2, 5 + page_crossed(), b);
    break;

 case 0x12: /* ORA (aa) */  /* 65C12 */
#ifdef _65C12_
    ORA(ind_zp, 2, 5, b);
#else
    NOP(2);
#endif
    break;

 case 0x13: /* Unimplemented */
    NOP(2);
    break;

 case 0x14: /* TRB aa */  /* 65C12 */
#ifdef _65C12_
    TRB(zp, 2, 5, bq);
#else
    NOP(2);
#endif
    break;

 case 0x15: /* ORA aa,X */
    ORA(zp_x, 2, 4, bq);
    break;

 case 0x16: /* ASL aa,X */
    ASL(zp_x, 2, 6, bq);
    break;

 case 0x17: /* Unimplemented */
    NOP(2);
    break;

 case 0x18: /* CLC */
    CLC(1, 2);
    break;

 case 0x19: /* ORA aaaa,Y */
    ORA(abs_y, 3, 4 + page_crossed(), b);
    break;

 case 0x1a: /* INA */ /* 65C12 */
#ifdef _65C12_
    INA(1, 2);
#else
    NOP(1);
#endif
    break;

 case 0x1b: /* Unimplemented */
    NOP(1);
    break;

 case 0x1c: /* TRB aaaa */ /* 65C12 */
#ifdef _65C12_
    TRB(abs, 3, 6, b);
#else
    NOP(3);
#endif
    break;

 case 0x1d: /* ORA aaaa,X */
    ORA(abs_x, 3, 4 + page_crossed(), b);
    break;

 case 0x1e: /* ASL aaaa,X */
    ASL(abs_x, 3, 7, b);
    break;

 case 0x1f: /* BBR 1 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBR(0x02, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0x20: /* JSR aaaa */
    JSR(abs, 3, 6);
    break;

 case 0x21: /* AND (aa,X) */
    AND(ind_x, 2, 6, b);
    break;

 case 0x22: /* Unimplemented */
    NOP(2);
    break;

 case 0x23: /* Unimplemented */
    NOP(2);
    break;

 case 0x24: /* BIT aa */
    BIT(zp, 2, 3, bq);
    break;

 case 0x25: /* AND aa */
    AND(zp, 2, 3, bq);
    break;

 case 0x26: /* ROL aa */
    ROL(zp, 2, 5, bq);
    break;

 case 0x27: /* Unimplemented */
    NOP(2);
    break;

 case 0x28: /* PLP */
    PLP(1, 4);
    break;

 case 0x29: /* AND #dd */
    AND(imm, 2, 2, bi);
    break;

 case 0x2a: /* ROL A */
    ROLA(1, 2);
    break;

 case 0x2b: /* Unimplemented */
    NOP(1);
    break;

 case 0x2c: /* BIT aaaa */
    BIT(abs, 3, 4, b);
    break;

 case 0x2d: /* AND aaaa */
    AND(abs, 3, 4, b);
    break;

 case 0x2e: /* ROL aaaa */
    ROL(abs, 3, 6, b);
    break;

 case 0x2f: /* BBR 2 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBR(0x04, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0x30: /* BMI rr */
    BMI(2, 2);
    break;

 case 0x31: /* AND (aa),Y */
    AND(ind_y, 2, 5 + page_crossed(), b);
    break;

 case 0x32: /* AND (aa) */ /* 65C12 */
#ifdef _65C12_
    AND(ind_zp, 2, 5, b);
#else
    NOP(2);
#endif
    break;

 case 0x33: /* Unimplemented */
    NOP(2);
    break;

 case 0x34: /* BIT aa,X */ /* 65C12 */
#ifdef _65C12_
    BIT(zp_x, 2, 4, bq);
#else
    NOP(2);
#endif
    break;

 case 0x35: /* AND aa,X */
    AND(zp_x, 2, 4, bq);
    break;

 case 0x36: /* ROL aa,X */
    ROL(zp_x, 2, 6, bq);
    break;

 case 0x37: /* Unimplemented */
    NOP(2);
    break;

 case 0x38: /* SEC */
    SEC(1, 2);
    break;

 case 0x39: /* AND aaaa,Y */
    AND(abs_x, 3, 4 + page_crossed(), b);
    break;

 case 0x3a: /* DEA */ /* 65C12 */
#ifdef _65C12_
    DEA(1, 2);
#else
    NOP(1);
#endif
    break;

 case 0x3b: /* Unimplemented */
    NOP(1);
    break;

 case 0x3c: /* BIT aaaa,X */ /* 65C12 */
#ifdef _65C12_
    BIT(abs_x, 3, 4, b);
#else
    NOP(3);
#endif
    break;

 case 0x3d: /* AND aaaa,X */
    AND(abs_x, 3, 4 + page_crossed(), b);
    break;

 case 0x3e: /* ROL aaaa,X */
    ROL(abs_x, 3, 7, b);
    break;

 case 0x3f: /* BBR 3 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBR(0x08, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0x40: /* RTI */
    RTI(1,6);
    break;

 case 0x41: /* EOR (aa,X) */
    EOR(ind_x, 2, 6, b);
    break;

 case 0x42: /* Unimplemented */
    NOP(2);
    break;

 case 0x43: /* Unimplemented */
    NOP(2);
    break;

 case 0x44: /* Unimplemented */
    NOP(2);
    break;

 case 0x45: /* EOR aa */
    EOR(zp, 2, 3, bq);
    break;

 case 0x46: /* LSR aa */
    LSR(zp, 2, 5, bq);
    break;

 case 0x47: /* Unimplemented */
    NOP(2);
    break;

 case 0x48: /* PHA */
    PHA(1, 3);
    break;

 case 0x49: /* EOR #dd */
    EOR(imm, 2, 2, bi);
    break;

 case 0x4a: /* LSR A */
    LSRA(1, 2);
    break;

 case 0x4b: /* Unimplemented */
    NOP(1);
    break;

 case 0x4c: /* JMP aaaa */
    JMP(abs, 3, 3);
    break;

 case 0x4d: /* EOR aaaa */
    EOR(abs, 3, 4, b);
    break;

 case 0x4e: /* LSR aaaa */
    LSR(abs, 3, 6, b);
    break;

 case 0x4f: /* BBR 4 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBR(0x10, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0x50: /* BVC rr */
    BVC(2, 2);
    break;

 case 0x51: /* EOR (aa),Y */
    EOR(ind_y, 2, 5 + page_crossed(), b);
    break;

 case 0x52: /* EOR (aa) */ /* 65C12 */
#ifdef _65C12_
    EOR(ind_zp, 2, 5, b);
#else
    NOP(2);
#endif
    break;

 case 0x53: /* Unimplemented */
    NOP(2);
    break;

 case 0x54: /* Unimplemented */
    NOP(2);
    break;

 case 0x55: /* EOR aa,X */
    EOR(zp_x, 2, 4, bq);
    break;

 case 0x56: /* LSR aa,X */
    LSR(zp_x, 2, 6, bq);
    break;

 case 0x57: /* Unimplemented */
    NOP(2);
    break;

 case 0x58: /* CLI */
    CLI(1, 2);
    break;

 case 0x59: /* EOR aaaa,Y */
    EOR(abs_y, 3, 4 + page_crossed(), b);
    break;

 case 0x5a: /* PHY */ /* 65C12 */
#ifdef _65C12_
    PHY(1, 2);
#else
    NOP(1);
#endif
    break;

 case 0x5b: /* Unimplemented */
    NOP(1);
    break;

 case 0x5c: /* Unimplemented */
    NOP(3);
    break;

 case 0x5d: /* EOR aaaa,X */
    EOR(abs_x, 3, 4 + page_crossed(), b);
    break;

 case 0x5e: /* LSR aaaa,X */
    LSR(abs_x, 3, 7 + page_crossed(), b);
    break;

 case 0x5f: /* BBR 5 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBR(0x20, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0x60: /* RTS */
    RTS(1, 6);
    break;

 case 0x61: /* ADC (aa,X) */
    ADC(ind_x, 2, 6 + ADC_KLUDGE, b);
    break;

 case 0x62: /* Unimplemented */
    NOP(2);
    break;

 case 0x63: /* Unimplemented */
    NOP(2);
    break;

 case 0x64: /* STZ aa */ /* 65C12 */
#ifdef _65C12_
    STZ(zp, 2, 3);
#else
    NOP(2);
#endif
    break;

 case 0x65: /* ADC aa */
    ADC(zp, 2, 3 + ADC_KLUDGE, bq);
    break;

 case 0x66: /* ROR aa */
    ROR(zp, 2, 5, bq);
    break;

 case 0x67: /* Unimplemented */
    NOP(2);
    break;

 case 0x68: /* PLA */
    PLA(1, 4);
    break;

 case 0x69: /* ADC #dd */
    ADC(imm, 2, 2 + ADC_KLUDGE, bi);
    break;

 case 0x6a: /* ROR A */
    RORA(1, 2);
    break;

 case 0x6b: /* Unimplemented */
    NOP(1);
    break;

 case 0x6c: /* JMP (aaaa) */
    JMP(ind, 3, 5);
    break;

 case 0x6d: /* ADC aaaa */
    ADC(abs, 3, 4 + ADC_KLUDGE, b);
    break;

 case 0x6e: /* ROR aaaa */
    ROR(abs, 3, 6, b);
    break;

 case 0x6f: /* BBR 6 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBR(0x40, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0x70: /* BVS rr */
    BVS(2, 2);
    break;

 case 0x71: /* ADC (aa),Y */
    ADC(ind_y, 2, 5 + page_crossed() + ADC_KLUDGE, b);
    break;

 case 0x72: /* ADC (aa) */ /* 65C12 */
#ifdef _65C12_
    ADC(ind_zp, 2, 5 + ADC_KLUDGE, b);
#else
    NOP(2);
#endif
    break;

 case 0x73: /* Unimplemented */
    NOP(2);
    break;

 case 0x74: /* STZ aa,X */ /* 65C12 */
#ifdef _65C12_
    STZ(zp_x, 2, 4, bq);
#else
    NOP(2);
#endif
    break;

 case 0x75: /* ADC aa,X */
    ADC(zp_x, 2, 4 + ADC_KLUDGE, bq);
    break;

 case 0x76: /* ROR aa,X */
    ROR(zp_x, 2, 6, bq);
    break;

 case 0x77: /* Unimplemented */
    NOP(2);
    break;

 case 0x78: /* SEI */
    SEI(1, 2);
    break;

 case 0x79: /* ADC aaaa,Y */
    ADC(abs_y, 3, 4 + page_crossed() + ADC_KLUDGE, b);
    break;

 case 0x7a: /* PLY */ /* 65C12 */
#ifdef _65C12_
    PLY(1, 2);
#else
    NOP(1);
#endif
    break;

 case 0x7b: /* Unimplemented */
    NOP(1);
    break;

 case 0x7c: /* JMP (aaaa,X) */ /* 65C12 */
#ifdef _65C12_
    JMP(absind_x, 3, 6);
#else
    NOP(3);
#endif
    break;

 case 0x7d: /* ADC aaaa,X */
    ADC(abs_x, 3, 4 + page_crossed() + ADC_KLUDGE, b);
    break;

 case 0x7e: /* ROR aaaa,X */
    ROR(abs_x, 3, 7, b);
    break;

 case 0x7f: /* BBR 7 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBR(0x80, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0x80: /* BRA */ /* 65C12 */
#ifdef _65C12_
    BRA(2, 2);
#else
    NOP(2);
#endif
    break;

 case 0x81: /* STA (aa,X) */
    STA(ind_x, 2, 6, b);
    break;

 case 0x82: /* Unimplemented */
    NOP(2);
    break;

 case 0x83: /* Unimplemented */
    NOP(2);
    break;

 case 0x84: /* STY aa */
    STY(zp, 2, 3, bq);
    break;

 case 0x85: /* STA aa */
    STA(zp, 2, 3, bq);
    break;

 case 0x86: /* STX aa */
    STX(zp, 2, 3, bq);
    break;

 case 0x87: /* Unimplemented */
    NOP(2);
    break;

 case 0x88: /* DEY */
    DEY(1, 2);
    break;

 case 0x89: /* BIT #dd */ /* 65C12 */
#ifdef _65C12_
    BITx(imm, 2, 2, bi);
#else
    NOP(2);
#endif
    break;

 case 0x8a: /* TXA */
    TXA(1, 2);
    break;

 case 0x8b: /* Unimplemented */
    NOP(1);
    break;

 case 0x8c: /* STY aaaa */
    STY(abs, 3, 4, b);
    break;

 case 0x8d: /* STA aaaa */
    STA(abs, 3, 4, b);
    break;

 case 0x8e: /* STX aaaa */
    STX(abs, 3, 4, b);
    break;

 case 0x8f: /* BBS 0 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBS(0x01, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0x90: /* BCC rr */
    BCC(2, 2);
    break;

 case 0x91: /* STA (aa),Y */
    STA(ind_y, 2, 6, b);
    break;

 case 0x92: /* STA (aa) */ /* 65C12 */
#ifdef _65C12_
    STA(ind_zp, 2, 6, b);
#else
    NOP(2);
#endif
    break;

 case 0x93: /* Unimplemented */
    NOP(2);
    break;

 case 0x94: /* STY aa,X */
    STY(zp_x, 2, 4, bq);
    break;

 case 0x95: /* STA aa,X */
    STA(zp_x, 2, 4, bq);
    break;

 case 0x96: /* STX aa,Y */
    STX(zp_y, 2, 4, bq);
    break;

 case 0x97: /* Unimplemented */
    NOP(2);
    break;

 case 0x98: /* TYA */
    TYA(1, 2);
    break;

 case 0x99: /* STA aaaa,Y */
    STA(abs_y, 3, 5, b);
    break;

 case 0x9a: /* TXS */
    TXS(1, 2);
    break;

 case 0x9b: /* Unimplemented */
    NOP(1);
    break;

 case 0x9c: /* STZ aaaa */ /* 65C12 */
#ifdef _65C12_
    STZ(abs, 3, 4);
#else
    NOP(3);
#endif
    break;

 case 0x9d: /* STA aaaa,X */
    STA(abs_x, 3, 5, b);
    break;

 case 0x9e: /* STZ aaaa,X */ /* 65C12 */
#ifdef _65C12_
    STZ(abs_x, 3, 5, b);
#else
    NOP(2);
#endif
    break;

 case 0x9f: /* BBS 1 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBS(0x02, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0xa0: /* LDY #dd */
    LDY(imm, 2, 2, bi);
    break;

 case 0xa1: /* LDA (aa,X) */
    LDA(ind_x, 2, 6, b);
    break;

 case 0xa2: /* LDX #dd */
    LDX(imm, 2, 2, bi);
    break;

 case 0xa3: /* Unimplemented */
    NOP(2);
    break;

 case 0xa4: /* LDY aa */
    LDY(zp, 2, 3, bq);
    break;

 case 0xa5: /* LDA aa */
    LDA(zp, 2, 3, bq);
    break;

 case 0xa6: /* LDX aa */
    LDX(zp, 2, 3, bq);
    break;

 case 0xa7: /* Unimplemented */
    NOP(2);
    break;

 case 0xa8: /* TAY */
    TAY(1, 2);
    break;

 case 0xa9: /* LDA #dd */
    LDA(imm, 2, 2, bi);
    break;

 case 0xaa: /* TAX */
    TAX(1, 2);
    break;

 case 0xab: /* Unimplemented */
    NOP(1);
    break;

 case 0xac: /* LDY aaaa */
    LDY(abs, 3, 4, b);
    break;

 case 0xad: /* LDA aaaa */
    LDA(abs, 3, 4, b);
    break;

 case 0xae: /* LDX aaaa */
    LDX(abs, 3, 4, b);
    break;

 case 0xaf: /* BBS 2 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBS(0x04, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0xb0: /* BCS rr */
    BCS(2, 2);
    break;

 case 0xb1: /* LDA (aa),Y */
    LDA(ind_y, 2, 5 + page_crossed(), b);
    break;

 case 0xb2: /* LDA (aa) */ /* 65C12 */
#ifdef _65C12_
    LDA(ind_zp, 2, 5, b);
#else
    NOP(2);
#endif
    break;

 case 0xb3: /* Unimplemented */
    NOP(2);
    break;

 case 0xb4: /* LDY aa,X */
    LDY(zp_x, 2, 4, bq);
    break;

 case 0xb5: /* LDA aa,X */
    LDA(zp_x, 2, 4, bq);
    break;

 case 0xb6: /* LDX aa,Y */
    LDX(zp_y, 2, 4, bq);
    break;

 case 0xb7: /* Unimplemented */
    NOP(2);
    break;

 case 0xb8: /* CLV */
    CLV(1, 2);
    break;

 case 0xb9: /* LDA aaaa,Y */
    LDA(abs_y, 3, 4 + page_crossed(), b);
    break;

 case 0xba: /* TSX */
    TSX(1, 2);
    break;

 case 0xbb: /* Unimplemented */
    NOP(1);
    break;

 case 0xbc: /* LDY aaaa,X */
    LDY(abs_x, 3, 4 + page_crossed(), b);
    break;

 case 0xbd: /* LDA aaaa,X */
    LDA(abs_x, 3, 4 + page_crossed(), b);
    break;

 case 0xbe: /* LDX aaaa,Y */
    LDX(abs_y, 3, 4 + page_crossed(), b);
    break;

 case 0xbf: /* BBS 3 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBS(0x08, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0xc0: /* CPY #dd */
    CPY(imm, 2, 2, bi);
    break;

 case 0xc1: /* CMP (aa,X) */
    CMP(ind_x, 2, 6, b);
    break;

 case 0xc2: /* Unimplemented */
    NOP(2);
    break;

 case 0xc3: /* Unimplemented */
    NOP(2);
    break;

 case 0xc4: /* CPY aa */
    CPY(zp, 2, 3, bq);
    break;

 case 0xc5: /* CMP aa */
    CMP(zp, 2, 3, bq);
    break;

 case 0xc6: /* DEC aa */
    DEC(zp, 2, 5, bq);
    break;

 case 0xc7: /* Unimplemented */
    NOP(2);
    break;

 case 0xc8: /* INY */
    INY(1, 2);
    break;

 case 0xc9: /* CMP #dd */
    CMP(imm, 2, 2, bi);
    break;

 case 0xca: /* DEX */
    DEX(1, 2);
    break;

 case 0xcb: /* Unimplemented */
    NOP(1);
    break;

 case 0xcc: /* CPY aaaa */
    CPY(abs, 3, 4, b);
    break;

 case 0xcd: /* CMP aaaa */
    CMP(abs, 3, 4, b);
    break;

 case 0xce: /* DEC aaaa */
    DEC(abs, 3, 6, b);
    break;

 case 0xcf: /* BBS 4 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBS(0x10, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0xd0: /* BNE rr */
    BNE(2, 2);
    break;

 case 0xd1: /* CMP (aa),y */
    CMP(ind_y, 2, 5 + page_crossed(), b);
    break;

 case 0xd2: /* CMP (aa) */ /* 65C12 */
#ifdef _65C12_
    CMP(ind_zp, 2, 5, b);
#else
    NOP(2);
#endif
    break;

 case 0xd3: /* Unimplemented */
    NOP(2);
    break;

 break;
 case 0xd4: /* UNX #dd */ {
     byte a = A, x = X, y = Y, p = P, sp = SP;
     word pc = PC;
     unix_call(&a, &x, &y, &p, &sp, &pc);
     A = a; X = x; Y = y; P = p; SP = sp; PC = pc;
     break;
 }
 case 0xd5: /* CMP aa,X */
    CMP(zp_x, 2, 4, bq);
    break;

 case 0xd6: /* DEC aa,X */
    DEC(zp_x, 2, 6, bq);
    break;

 case 0xd7: /* Unimplemented */
    NOP(2);
    break;

 case 0xd8: /* CLD */
    CLD(1, 2);
    break;

 case 0xd9: /* CMP aaaa,Y */
    CMP(abs_y, 3, 4 + page_crossed(), b);
    break;

 case 0xda: /* PHX */ /* 65C12 */
#ifdef _65C12_
    PHX(1, 2);
#else
    NOP(1);
#endif
    break;

 case 0xdb: /* Unimplemented */
    NOP(1);
    break;

 case 0xdc: /* Unimplemented */
    NOP(3);
    break;

 case 0xdd: /* CMP aaaa,X */
    CMP(abs_x, 3, 4 + page_crossed(), b);
    break;

 case 0xde: /* DEC aaaa,X */
    DEC(abs_x, 3, 7, b);
    break;

 case 0xdf: /* BBS 5 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBS(0x20, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0xe0: /* CPX #dd */
    CPX(imm, 2, 2, bi);
    break;

 case 0xe1: /* SBC (aa,X) */
    SBC(ind_x, 2, 6 + ADC_KLUDGE, b);
    break;

 case 0xe2: /* Unimplemented */
    NOP(2);
    break;

 case 0xe3: /* Unimplemented */
    NOP(2);
    break;

 case 0xe4: /* CPX aa */
    CPX(zp, 2, 3, bq);
    break;

 case 0xe5: /* SBC aa */
    SBC(zp, 2, 3 + ADC_KLUDGE, bq);
    break;

 case 0xe6: /* INC aa */
    INC(zp, 2, 5, bq);
    break;

 case 0xe7: /* Unimplemented */
    NOP(2);
    break;

 case 0xe8: /* INX */
    INX(1, 2);
    break;

 case 0xe9: /* SBC #dd */
    SBC(imm, 2, 2 + ADC_KLUDGE, bi);
    break;

 case 0xea: /* NOP */
    NOP(1);
    break;

 case 0xeb: /* Unimplemented */
    NOP(1);
    break;

 case 0xec: /* CPX aaaa */
    CPX(abs, 3, 4, b);
    break;

 case 0xed: /* SBC aaaa */
    SBC(abs, 3, 4 + ADC_KLUDGE, b);
    break;

 case 0xee: /* INC aaaa */
    INC(abs, 3, 6, b);
    break;

 case 0xef: /* BBS 6 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBS(0x40, 3, 5);
#else
    NOP(3);
#endif
    break;

 case 0xf0: /* BEQ rr */
    BEQ(2, 2);
    break;

 case 0xf1: /* SBC (aa),Y */
    SBC(ind_y, 2, 5 + page_crossed() + ADC_KLUDGE, b);
    break;

 case 0xf2: /* SBC (aa) */ /* 65C12 */
#ifdef _65C12_
    SBC(zp, 2, 5 + ADC_KLUDGE, b);
#else
    NOP(2);
#endif
    break;

 case 0xf3: /* Unimplemented */
    NOP(2);
    break;

 case 0xf4: /* Unimplemented */
    NOP(2);
    break;

 case 0xf5: /* SBC aa,x */
    SBC(zp_x, 2, 4 + ADC_KLUDGE, bq);
    break;

 case 0xf6: /* INC aa,X */
    INC(zp_x, 2, 6, bq);
    break;

 case 0xf7: /* Unimplemented */
    NOP(2);
    break;

 case 0xf8: /* SED */
    SED(1, 2);
    break;

 case 0xf9: /* SBC aaaa,Y */
    SBC(abs_y, 3, 4 + page_crossed() + ADC_KLUDGE, b);
    break;

 case 0xfa: /* PLX */ /* 65C12 */
#ifdef _65C12_
    PLX(1, 2);
#else
    NOP(1);
#endif
    break;

 case 0xfb: /* Unimplemented */
    NOP(1);
    break;

 case 0xfc: /* Unimplemented */
    NOP(3);
    break;

 case 0xfd: /* SBC aaaa,X */
    SBC(abs_x, 3, 4 + page_crossed() + ADC_KLUDGE, b);
    break;

 case 0xfe: /* INC aaaa,X */
    INC(abs_x, 3, 7, b);
    break;

 case 0xff: /* BBS 7 aa rr */ /* 65C02 */
#ifdef _R65C02_
    BBS(0x80, 3, 5);
#else
    NOP(3);
#endif
    break;
}
