/*
 * ULA module
 *
 * Created 30/03/94 by James Bonfield
 */

#include <stdio.h>
#include <X11/Xlib.h>

#include "../defs.h"
#include "../memory.h"

/* 8 8bit palette registers */
byte palette[8];

unsigned long colours[16];
byte write_fe20(word addr, byte val);
byte write_fe21(word addr, byte val);
byte read_fe20(word addr);
byte read_fe21(word addr);

/*
 * Returns the pixel value of a specified colour.
 */
unsigned long get_colour(Display *d, int scrn, char *col) {
    XColor col1, col2;

    if (!XAllocNamedColor(d, DefaultColormap(d, scrn), col, &col1, &col2)) {
	fprintf(stderr, "Failed to allocated colour \"%s\".\n", col);
	return -1;
    }

    return col1.pixel;
}

/*
 * Initialise the ULA.
 * This consists of allocating the colours.
 */
void init_ula(Display *d, int scrn) {
    /* memory hooks */
    special_mem_r[0xfe20] = read_fe20;
    special_mem_w[0xfe20] = write_fe20;
    special_mem_r[0xfe21] = read_fe21;
    special_mem_w[0xfe21] = write_fe21;
    
    /* colours 8-15 are really flashing, but we cheat for the time being */
    colours[ 8] = colours[0] = BlackPixel(d, scrn);
    colours[ 9] = colours[1] = get_colour(d, scrn, "red");
    colours[10] = colours[2] = get_colour(d, scrn, "green");
    colours[11] = colours[3] = get_colour(d, scrn, "yellow");
    colours[12] = colours[4] = get_colour(d, scrn, "blue");
    colours[13] = colours[5] = get_colour(d, scrn, "magenta");
    colours[14] = colours[6] = get_colour(d, scrn, "cyan");
    colours[15] = colours[7] = WhitePixel(d, scrn);
}

/*

Changing the palette is an operation frequently done several times in
a single raster scan. Hence we need to be able to do this within X,
which implies the use of either more colours, or a colour translation
table.

A general approach is to always know what colour pixels should be
idrawn in. This involves rewriting the col[256][8] array used by the
6845 every palette update. Alternatively we do not use the fast 8 byte
col[][] optimisations and plot bit at a time determining the colour
with a simple 16 deep table (slow!).

Alternatively we could perhaps hack the X palette to make use of the
larger number of colours available (if they are). Consider in mode 1
on a display with 256 colours. That corresponds to 64 possible
'palettes' - one every 4 mode 1 lines. In mode 2 the situation is
worse (16 palettes, one every 16 lines). Obviously not too good.
We could instead use some system of dynamically shuffling the X palette,
to allow (in mode 2) 16 possible palette switches per raster frame.
However this is quite complex.

Hence the decision has been made to implement the first (col array
hacking) system, with the addition of caching to speed up typical usage.

*/

static byte ula_control = 0xff;

byte write_fe20(word addr, byte val) {
    fprintf (stderr, "&FE20 = 0x%02x\n", val);
    ula_control = val;

    return 0;
}

byte write_fe21(word addr, byte val) {
    fprintf (stderr, "&FE21 = 0x%02x\n", val);

    return 0;
}

byte read_fe20(word addr) {
    fprintf (stderr, "?&FE20 = 0x%02x\n", ula_control);
    return 0xff;
}

byte read_fe21(word addr) {
    return 0xff;
}
